/*****************************************************************************
*
* File: OsmoseCore.h
*
* Project: Osmose emulator.
*
* Description: This file contains Osmose main loop, handling keyboard, SDL
* event, and hardware emulation.
*
* Author: Vedder Bruno
* Date: 23/01/2005, 14h13
*
* URL: http://bcz.emu-france.com/
*****************************************************************************/
#ifndef OSMOSE_CORE_H
#define OSMOSE_CORE_H

#include <SDL/SDL.h>
#include <SDL/SDL_keysym.h>
#include <SDL/SDL_audio.h>
#include <GL/gl.h>

/* On Windows : GL_UNSIGNED_SHORT_5_6_5 is defined in as extension in glext.h*/
#ifdef __WIN32__
#include <GL/glext.h>
#endif

#include <iostream>
#include <iomanip>
#include <sstream>

#include "./cpu/Z80.h"
#include "SmsEnvironment.h"
#include "VDP.h"
#include "VDP_GG.h"
#include "SN76489.h"
#include "Options.h"
#include "SmsDebugger.h"
#include "./z80_disasm/Z80Dasm.h"
#include "Bits.h"
#include "WaveWriter.h"
#include "OsmoseConfiguration.h"
#include "TextWriter.h"
#include "Version.h"
#include "InputDevice.h"
#include "PadInputDevice.h"
#include "PaddleInputDevice.h"
#include "JoystickInputDevice.h"
#include "Definitions.h"
#include "DebugEventThrower.h"
#include "SaveState.h"

using namespace std;

/* Use save state slot from 1 to 5. */
enum
{
	minSaveStateSlot = 1,
	maxSaveStateSlot = 5
};

class OsmoseCore : public DebugEventThrower
{

public:

    OsmoseCore(const char *rom_f);	// Constructor.
    void run_frame();				// Run one Frame.
    void save_bbr();				// Save Battery Backed Ram.
    void handleSDLKeyboardEvent(SDL_Event e);
	void Reshape(int x, int y);

private:

    OsmoseConfiguration *oc;	     // OsmoseConfig.
    SDL_AudioSpec        format;     // Format structure to get our 22050hz 16bits sound.
    SDL_Surface         *screen;     // Surface for rendered filtered image 256x192 or 512x384.
    SDL_Surface         *buffer;     // SMS screen, without modification 256x192.
    string               game_name;  // Used as prefix for screenshot, sound shot and save states.
    VDP                 *v;	     	 // Video Display Processor object.
    MemoryMapper        *mem;        // Memory mapper Object.
    SmsEnvironment      *env;        // Z80 Cpu environment.
    IOMapper            *iom;        // Input/Output mapper Object.
    InputDevice         *input;	     // Pad or Paddle or trackball.
    Z80                 *cpu;	     // Our Z80 core.
    WaveWriter          *wavW;       // Sound ripper.
    TextWriter          *tw;         // Text Writer.
    float               gain;        // gain level.

#ifdef BUILT_IN_DEBUGGER
    unsigned int   old_cycles;	    // Used for cycle count.
    Z80Dasm        *dasm;	     	// Z80 disasembler object.
    SmsDebugger    *dbg;	     	// Sms Debugger.
    bool           exec_f;          // Continue cpu exec Flag.
#endif

    bool nmi;					// nmi flag, used with keyboard pause.
    bool sound_shot_toggle;		// Flag, for start/stop recording sound.
    int  screenshotNbr;			// Screenshot number appended to save.
    int  tileshotNbr;			// Tile rip  number appended to save.
    int  soundshotNbr;			// Sound rip  number appended to save.
    int  saveStateSlotNbr;		// Which slot will be use for save state.
    const char *rom_filename;	// Pointer to argv[1].

    bool snd_started;

    void captureScreen();
    void captureTiles(VDP *vd);
    void displayTiles(SDL_Surface *s, VDP *vd, int tile, int x, int y);
    void setupAudioFormat();
    void setupSDLVideo();
    void setupSDLAudio();
    bool saveSaveState();
    bool loadSaveState();

	/* OpenGL Stuff */
	int sdl_window_width;
	int sdl_window_height;
	GLuint textureName[1];
	void initOpenGL(int x, int y);
	void drawGLFrame();
	void prepare2DViewport(int topleft_x, int topleft_y, int bottomrigth_x, int bottomrigth_y);
};

#endif
