/*
 * OpenFileDialog.java
 *
 * Created on 6. August 2002, 21:00
 */

package pauker.program.gui.swing;

import java.awt.CardLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.nio.charset.Charset;
import java.util.SortedMap;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;

/**
 * The dialog used for opening Pauker lessons
 * @author Ronny.Standtke@gmx.net
 */
public class OpenFileDialog extends JDialog implements PropertyChangeListener {

    private final FileFilter xmlFileFilter;
    private final FileFilter csvFileFilter;
    private final PaukerFrame paukerFrame;
    private final CardLayout cardlayout;
    private final FileChooserAccessory fileChooserAccessory;

    /** Creates new form OpenFileDialog
     * @param paukerFrame the Pauker Frame
     * @param xmlFileFilter the XML file filter
     * @param csvFileFilter the CSV file filter
     */
    public OpenFileDialog(PaukerFrame paukerFrame,
            FileFilter xmlFileFilter, FileFilter csvFileFilter) {

        super(paukerFrame, true);

        this.paukerFrame = paukerFrame;
        this.xmlFileFilter = xmlFileFilter;
        this.csvFileFilter = csvFileFilter;

        initComponents();

        cardlayout = (CardLayout) encodingPanel.getLayout();

        fileChooserAccessory = new FileChooserAccessory();
        fileChooser.setAccessory(fileChooserAccessory);

        fileChooser.addChoosableFileFilter(xmlFileFilter);
        fileChooser.addChoosableFileFilter(csvFileFilter);

        fileChooser.addPropertyChangeListener(this);

        Action okAction = new AbstractAction() {

            public void actionPerformed(ActionEvent event) {
                if (!event.getActionCommand().equals(
                        JFileChooser.CANCEL_SELECTION) &&
                        fileChooser.getSelectedFile() != null) {
                    openSelectedFiles();
                }
            }
        };

        fileChooser.addActionListener(okAction);

        SortedMap<String, Charset> charSets = Charset.availableCharsets();
        for (String charsetName : charSets.keySet()) {
            encodingComboBox.addItem(charsetName);
        }
        encodingComboBox.setSelectedItem("UTF-8");

        pack();
    }

    /**
     * sets the current directory
     * @param directory the current directory
     */
    public void setCurrentDirectory(File directory) {
        fileChooser.setCurrentDirectory(directory);
    }

    /**
     * deselects all files and sets the file filter to "XML"
     */
    public void reset() {
        fileChooser.setSelectedFile(null);
        fileChooser.setFileFilter(xmlFileFilter);
    }

    /**
     * shows the open dialog
     * @param parent the parent component used to center the dialog
     */
    public void showOpenDialog(Component parent) {
        mergingCheckBox.setSelected(false);
        fileChooser.rescanCurrentDirectory();
        setLocationRelativeTo(parent);
        setVisible(true);
        dispose();
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Erzeugter Quelltext ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        fileChooser = new javax.swing.JFileChooser();
        buttonPanel = new javax.swing.JPanel();
        encodingPanel = new javax.swing.JPanel();
        comboBoxPanel = new javax.swing.JPanel();
        encodingComboBox = new javax.swing.JComboBox();
        emptyPanel = new javax.swing.JPanel();
        mergingCheckBox = new javax.swing.JCheckBox();
        openButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();

        getContentPane().setLayout(new java.awt.GridBagLayout());

        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("pauker/Strings"); // NOI18N
        setTitle(bundle.getString("Open")); // NOI18N
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
        });

        fileChooser.setControlButtonsAreShown(false);
        fileChooser.setFileFilter(xmlFileFilter);
        fileChooser.setFileView(new PaukerFileView());
        fileChooser.setName("fileChooser");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        getContentPane().add(fileChooser, gridBagConstraints);

        buttonPanel.setLayout(new java.awt.GridBagLayout());

        encodingPanel.setLayout(new java.awt.CardLayout());

        comboBoxPanel.setLayout(new java.awt.GridBagLayout());

        comboBoxPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(bundle.getString("Encoding"))); // NOI18N
        encodingComboBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                encodingComboBoxActionPerformed(evt);
            }
        });

        comboBoxPanel.add(encodingComboBox, new java.awt.GridBagConstraints());

        encodingPanel.add(comboBoxPanel, "comboBoxPanel");

        encodingPanel.add(emptyPanel, "emptyPanel");

        buttonPanel.add(encodingPanel, new java.awt.GridBagConstraints());

        mergingCheckBox.setText(bundle.getString("Merge_Lesson")); // NOI18N
        mergingCheckBox.setName("mergingCheckBox");
        mergingCheckBox.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                mergingCheckBoxItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        buttonPanel.add(mergingCheckBox, gridBagConstraints);

        openButton.setText(bundle.getString("Open")); // NOI18N
        openButton.setEnabled(false);
        openButton.setName("openButton");
        openButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                openButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 0, 0);
        buttonPanel.add(openButton, gridBagConstraints);

        cancelButton.setText(bundle.getString("Cancel")); // NOI18N
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 0);
        buttonPanel.add(cancelButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 10, 10);
        getContentPane().add(buttonPanel, gridBagConstraints);

        pack();
    }// </editor-fold>//GEN-END:initComponents
    private void encodingComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_encodingComboBoxActionPerformed
        String selectedEncoding = (String) encodingComboBox.getSelectedItem();
        fileChooserAccessory.setEncoding(selectedEncoding);
    }//GEN-LAST:event_encodingComboBoxActionPerformed

    private void mergingCheckBoxItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_mergingCheckBoxItemStateChanged
        if (mergingCheckBox.isSelected()) {
            File selectedFile = fileChooser.getSelectedFile();
            fileChooser.setMultiSelectionEnabled(true);
            if (selectedFile != null) {
                fileChooser.setSelectedFiles(new File[]{selectedFile});
            }
        } else {
            File[] selectedFiles = fileChooser.getSelectedFiles();
            fileChooser.setMultiSelectionEnabled(false);
            if (selectedFiles.length > 0) {
                fileChooser.setSelectedFile(selectedFiles[0]);
            }
        }
    }//GEN-LAST:event_mergingCheckBoxItemStateChanged

    private void openButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_openButtonActionPerformed
        openSelectedFiles();
    }//GEN-LAST:event_openButtonActionPerformed

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
        setVisible(false);
    }//GEN-LAST:event_cancelButtonActionPerformed

    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_closeDialog
        setVisible(false);
    }//GEN-LAST:event_closeDialog

    private void openSelectedFiles() {
        File[] selectedFiles = null;
        if (mergingCheckBox.isSelected()) {
            selectedFiles = fileChooser.getSelectedFiles();
        } else {
            File selectedFile = fileChooser.getSelectedFile();
            // dont try if file does not exist
            if (!selectedFile.exists()) {
                System.out.println(selectedFile + " doesn't exist!");
                return;
            }
            selectedFiles = new File[]{selectedFile};
        }
        if (selectedFiles == null || selectedFiles.length == 0) {
            System.out.println("no files selected!");
        } else {
            if (paukerFrame.openFiles(selectedFiles, getEncoding(),
                    mergingCheckBox.isSelected())) {
                setVisible(false);
            }
        }
    }

    private String getEncoding() {
        return (String) encodingComboBox.getSelectedItem();
    }

    public void propertyChange(PropertyChangeEvent propertyChangeEvent) {
        String property = propertyChangeEvent.getPropertyName();

        if (property.equals(JFileChooser.FILE_FILTER_CHANGED_PROPERTY)) {

            FileFilter newFilter = fileChooser.getFileFilter();

            if (newFilter == csvFileFilter) {
                cardlayout.show(encodingPanel, "comboBoxPanel");
            } else {
                cardlayout.show(encodingPanel, "emptyPanel");
            }

        } else if (property.equals(
                JFileChooser.SELECTED_FILE_CHANGED_PROPERTY)) {
            File selectedFile = fileChooser.getSelectedFile();
            if (selectedFile == null || !selectedFile.exists()) {
                openButton.setEnabled(false);
            } else {
                openButton.setEnabled(true);
            }
            fileChooserAccessory.propertyChange(propertyChangeEvent);
        }
    }
    // Variablendeklaration - nicht modifizieren//GEN-BEGIN:variables
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton cancelButton;
    private javax.swing.JPanel comboBoxPanel;
    private javax.swing.JPanel emptyPanel;
    private javax.swing.JComboBox encodingComboBox;
    private javax.swing.JPanel encodingPanel;
    private javax.swing.JFileChooser fileChooser;
    private javax.swing.JCheckBox mergingCheckBox;
    private javax.swing.JButton openButton;
    // Ende der Variablendeklaration//GEN-END:variables
}
