/* 
    TowBowlTactics: Strategy game for X11

    Copyright (C) 2001 Pascal Bourut (toweld@rocketmail.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "Screen.h"


Screen::Screen(SDL_Surface *screen_m){
  this->screen = screen_m;
  screen_w = Config::get_screen_w();
  screen_h = Config::get_screen_h();  
  type = SCREEN_NONE;
  screen_img[SCREEN_TITLE] = NULL;  
  title_left_btn_img = NULL;  
  title_right_btn_img = NULL;  
  screen_img[SCREEN_CREDITS] = NULL;  
  end_btn_img = NULL;  
  screen_img[SCREEN_END] = NULL;  
  end_btn_img = NULL;  
  screen_img[SCREEN_VS] = NULL;  
  screen_img[SCREEN_TEAM_SELECTOR] = NULL;
  screen_img[SCREEN_OPTIONS] = NULL;
  screen_img[SCREEN_NETWORK] = NULL;
  vs_lower_chars = NULL;
  vs_upper_chars = NULL;
  vs_btn = NULL;
  vs_no_btn = NULL;
  vs_instructions_img = NULL;
  load_input_img = NULL;
  credits_names_img = NULL;
  emblems_img[HOME] = NULL;
  emblems_img[VISITOR] = NULL;
  options_img = NULL;
  tick_img = NULL;
  res_tick = -1;
  net_tick = -1;
  sound_tick = Config::is_sound_enabled();
  video_tick = Config::is_video_enabled();
  language_tick = Config::get_language();
  no_emblem_img = NULL;
  ready_img = NULL;
  ready_emblem_img = NULL;
  console = NULL; 
  waiting_for = NONE;
  adversary_is_ready = false;
  console_editable = false;
}

Screen::~Screen(){
	if(console) delete console;
}

int Screen::get_type(){
	return type;
}

#ifndef NOSOUND
void Screen::set_sound_player(SoundPlayer* sp_m){
	if(Config::is_sound_enabled()){
		this->sp = sp_m;
	}
}
#endif

void Screen::load_images(int id){
  switch(id){
  case SCREEN_TITLE:
    if(!screen_img[SCREEN_TITLE]) screen_img[SCREEN_TITLE] = load_image(ADD_IMG_PATH("screens/title_bg.jpg"),NO_TRANSPARENT);  
    if(!title_left_btn_img) title_left_btn_img = load_image(ADD_IMG_PATH("screens/title_left_btn.jpg"),NO_TRANSPARENT);
    if(!title_right_btn_img) title_right_btn_img = load_image(ADD_IMG_PATH("screens/title_right_btn.jpg"),NO_TRANSPARENT);
    break;
  case SCREEN_CREDITS:
    if(!screen_img[SCREEN_CREDITS]) {
      screen_img[SCREEN_CREDITS] = load_image(ADD_IMG_PATH("screens/credits_bg.jpg"),NO_TRANSPARENT);
    }
    if(!end_btn_img) end_btn_img = load_image(ADD_IMG_PATH("screens/end_btn.jpg"),NO_TRANSPARENT);
    if(!credits_names_img) credits_names_img = load_image(ADD_IMG_PATH("screens/credits_names.png"),TRANSPARENT);
	break;
  case SCREEN_OPTIONS:
	if(!screen_img[SCREEN_OPTIONS]) {
      screen_img[SCREEN_OPTIONS] = load_image(ADD_IMG_PATH("screens/credits_bg.jpg"),NO_TRANSPARENT);
    }
    if(!end_btn_img) end_btn_img = load_image(ADD_IMG_PATH("screens/end_btn.jpg"),NO_TRANSPARENT);
    if(!credits_names_img) credits_names_img = load_image(ADD_IMG_PATH("screens/credits_names.png"),TRANSPARENT);
	if(!options_img) options_img = load_image(ADD_IMG_PATH("screens/options.png"),TRANSPARENT);
	if(!tick_img) tick_img = load_image(ADD_IMG_PATH("screens/tick.png"),TRANSPARENT);
	break;
  case SCREEN_END:
    if(!screen_img[SCREEN_END]) screen_img[SCREEN_END] = load_image(ADD_IMG_PATH("screens/end_bg.jpg"),NO_TRANSPARENT);
    if(!end_btn_img) end_btn_img = load_image(ADD_IMG_PATH("screens/end_btn.jpg"),NO_TRANSPARENT);
    break;
  case SCREEN_VS:
    if(!screen_img[SCREEN_VS]) screen_img[SCREEN_VS] = load_image(ADD_IMG_PATH("screens/vs_bg.jpg"),NO_TRANSPARENT);
    if(!vs_lower_chars) vs_lower_chars = load_image(ADD_IMG_PATH("screens/vs_lower_chars.png"),TRANSPARENT);
    if(!vs_upper_chars) vs_upper_chars = load_image(ADD_IMG_PATH("screens/vs_upper_chars.png"),TRANSPARENT);
	break;
  case SCREEN_TEAM_SELECTOR:
	if(!screen_img[SCREEN_TEAM_SELECTOR]) screen_img[SCREEN_TEAM_SELECTOR] = load_image(ADD_IMG_PATH("screens/vs_bg.jpg"),NO_TRANSPARENT);
    if(!vs_lower_chars) vs_lower_chars = load_image(ADD_IMG_PATH("screens/vs_lower_chars.png"),TRANSPARENT);
    if(!vs_upper_chars) vs_upper_chars = load_image(ADD_IMG_PATH("screens/vs_upper_chars.png"),TRANSPARENT);
	if(!vs_btn) vs_btn = load_image(ADD_IMG_PATH("screens/vs.jpg"),NO_TRANSPARENT);
    if(!vs_instructions_img) vs_instructions_img = load_image(ADD_IMG_PATH("screens/vs_instructions.png"),TRANSPARENT);	
	SDL_SetAlpha(vs_instructions_img, ALPHA_FLAGS, 200);
	break;
  case SCREEN_NETWORK:
	if(!screen_img[SCREEN_NETWORK]) screen_img[SCREEN_NETWORK] = load_image(ADD_IMG_PATH("screens/network_bg.jpg"),NO_TRANSPARENT);
    if(!vs_lower_chars) vs_lower_chars = load_image(ADD_IMG_PATH("screens/vs_lower_chars.png"),TRANSPARENT);
    if(!vs_upper_chars) vs_upper_chars = load_image(ADD_IMG_PATH("screens/vs_upper_chars.png"),TRANSPARENT);
	if(!vs_btn) vs_btn = load_image(ADD_IMG_PATH("screens/vs.jpg"),NO_TRANSPARENT);
    if(!vs_no_btn) vs_no_btn = load_image(ADD_IMG_PATH("screens/vs_no.jpg"),NO_TRANSPARENT);
	if(!end_btn_img) end_btn_img = load_image(ADD_IMG_PATH("screens/end_btn.jpg"),NO_TRANSPARENT);
	if(!tick_img) tick_img = load_image(ADD_IMG_PATH("screens/tick.png"),TRANSPARENT);
	if(!no_emblem_img) no_emblem_img  = load_image(ADD_IMG_PATH("screens/no_emblem.png"),NO_TRANSPARENT);
	if(!ready_emblem_img) ready_emblem_img  = load_image(ADD_IMG_PATH("screens/ready_emblem.png"),NO_TRANSPARENT);
	if(!ready_img) ready_img = load_image(ADD_IMG_PATH("screens/ready.jpg"),NO_TRANSPARENT);
	break;
  case SCREEN_LOAD:
	if(!load_input_img) load_input_img = load_image(ADD_IMG_PATH("screens/load_input.jpg"),NO_TRANSPARENT);  
	if(!vs_lower_chars) vs_lower_chars = load_image(ADD_IMG_PATH("screens/vs_lower_chars.png"),TRANSPARENT);
    if(!vs_upper_chars) vs_upper_chars = load_image(ADD_IMG_PATH("screens/vs_upper_chars.png"),TRANSPARENT);
	break;
  default:
    break;
  }
}

void Screen::free_images(int id){
  switch(id){
  case SCREEN_TITLE:
    if(screen_img[SCREEN_TITLE]) {
      SDL_FreeSurface(screen_img[SCREEN_TITLE]);
      screen_img[SCREEN_TITLE] = NULL;
    }
    if(title_left_btn_img) {
      SDL_FreeSurface(title_left_btn_img);
      title_left_btn_img = NULL;
    }
    if(title_right_btn_img) {
      SDL_FreeSurface(title_right_btn_img);
      title_right_btn_img = NULL;
    }
    break; 
  case SCREEN_CREDITS:
    if(screen_img[SCREEN_CREDITS]) {
      SDL_FreeSurface(screen_img[SCREEN_CREDITS]);
      screen_img[SCREEN_CREDITS] = NULL;
    }
    if(end_btn_img) {
      SDL_FreeSurface(end_btn_img);
      end_btn_img = NULL;
    }
	if(credits_names_img){
		SDL_FreeSurface(credits_names_img);
		credits_names_img = NULL;
	}
    break;
	case SCREEN_OPTIONS:
    if(screen_img[SCREEN_OPTIONS]) {
      SDL_FreeSurface(screen_img[SCREEN_OPTIONS]);
      screen_img[SCREEN_OPTIONS] = NULL;
    }
    if(end_btn_img) {
      SDL_FreeSurface(end_btn_img);
      end_btn_img = NULL;
    }
	if(credits_names_img){
		SDL_FreeSurface(credits_names_img);
		credits_names_img = NULL;
	}
	if(options_img){
		SDL_FreeSurface(options_img);
		options_img = NULL;
	}
    if(tick_img){
		SDL_FreeSurface(tick_img);
		tick_img = NULL;
	}
	break;
  case SCREEN_END:
    if(screen_img[SCREEN_END]) {
      SDL_FreeSurface(screen_img[SCREEN_END]);
      screen_img[SCREEN_END] = NULL;
    }
    if(end_btn_img) {
      SDL_FreeSurface(end_btn_img);
      end_btn_img = NULL;
    }
    break;
  case SCREEN_VS:
    if(screen_img[SCREEN_VS]) {
      SDL_FreeSurface(screen_img[SCREEN_VS]);
      screen_img[SCREEN_VS] = NULL;
    }
    if(emblems_img[HOME]) {
      SDL_FreeSurface(emblems_img[HOME]);
      emblems_img[HOME] = NULL;
    }
    if(emblems_img[VISITOR]) {
      SDL_FreeSurface(emblems_img[VISITOR]);
      emblems_img[VISITOR] = NULL;
    }
	if(vs_lower_chars) {
		SDL_FreeSurface(vs_lower_chars);
		vs_lower_chars = NULL;
	}
    if(vs_upper_chars) {
		SDL_FreeSurface(vs_upper_chars);
		vs_upper_chars = NULL;
	}
    break;
  case SCREEN_TEAM_SELECTOR:
    if(screen_img[SCREEN_TEAM_SELECTOR]) {
      SDL_FreeSurface(screen_img[SCREEN_TEAM_SELECTOR]);
      screen_img[SCREEN_TEAM_SELECTOR] = NULL;
    }
	if(emblems_img[HOME]) {
      SDL_FreeSurface(emblems_img[HOME]);
      emblems_img[HOME] = NULL;
    }
    if(emblems_img[VISITOR]) {
      SDL_FreeSurface(emblems_img[VISITOR]);
      emblems_img[VISITOR] = NULL;
    }
	if(vs_lower_chars) {
		SDL_FreeSurface(vs_lower_chars);
		vs_lower_chars = NULL;
	}
    if(vs_upper_chars) {
		SDL_FreeSurface(vs_upper_chars);
		vs_upper_chars = NULL;
	}
	if(vs_btn){
		SDL_FreeSurface(vs_btn);
		vs_btn = NULL;
	}
    if(vs_instructions_img){
		SDL_FreeSurface(vs_instructions_img);
		vs_instructions_img = NULL;
	}	
	break;
  case SCREEN_NETWORK:
    if(screen_img[SCREEN_NETWORK]) {
      SDL_FreeSurface(screen_img[SCREEN_NETWORK]);
      screen_img[SCREEN_NETWORK] = NULL;
    }
	if(emblems_img[HOME]) {
      SDL_FreeSurface(emblems_img[HOME]);
      emblems_img[HOME] = NULL;
    }
    if(emblems_img[VISITOR]) {
      SDL_FreeSurface(emblems_img[VISITOR]);
      emblems_img[VISITOR] = NULL;
    }
	if(vs_lower_chars) {
		SDL_FreeSurface(vs_lower_chars);
		vs_lower_chars = NULL;
	}
    if(vs_upper_chars) {
		SDL_FreeSurface(vs_upper_chars);
		vs_upper_chars = NULL;
	}
	if(vs_btn){
		SDL_FreeSurface(vs_btn);
		vs_btn = NULL;
	}
    if(vs_no_btn){
		SDL_FreeSurface(vs_no_btn);
		vs_no_btn = NULL;
	}	
	if(end_btn_img) {
        SDL_FreeSurface(end_btn_img);
        end_btn_img = NULL;
    }
	if(tick_img){
		SDL_FreeSurface(tick_img);
		tick_img = NULL;
	}
	if(no_emblem_img){
		SDL_FreeSurface(no_emblem_img);
		no_emblem_img = NULL;
	}
	if(ready_emblem_img){
		SDL_FreeSurface(ready_emblem_img);
		ready_emblem_img = NULL;
	}
	if(ready_img){
		SDL_FreeSurface(ready_img);
		ready_img = NULL;
	}
	break;
  case SCREEN_LOAD:
	  if(load_input_img){
		SDL_FreeSurface(load_input_img);
		load_input_img = NULL;
	  }
      if(vs_lower_chars) {
		SDL_FreeSurface(vs_lower_chars);
		vs_lower_chars = NULL;
	  }
      if(vs_upper_chars) {
		SDL_FreeSurface(vs_upper_chars);
		vs_upper_chars = NULL;
	  }
	  break;
  default:
	  if(load_input_img){
	  SDL_FreeSurface(load_input_img);
	  load_input_img = NULL;
	}
    if(screen_img[SCREEN_TITLE]) {
      SDL_FreeSurface(screen_img[SCREEN_TITLE]);
      screen_img[SCREEN_TITLE] = NULL;
    }
    if(title_left_btn_img) {
      SDL_FreeSurface(title_left_btn_img);
      title_left_btn_img = NULL;
    }
    if(title_right_btn_img) {
      SDL_FreeSurface(title_right_btn_img);
      title_right_btn_img = NULL;
    }
    if(screen_img[SCREEN_CREDITS]) {
      SDL_FreeSurface(screen_img[SCREEN_CREDITS]);
      screen_img[SCREEN_CREDITS] = NULL;
    }
    if(screen_img[SCREEN_OPTIONS]) {
      SDL_FreeSurface(screen_img[SCREEN_OPTIONS]);
      screen_img[SCREEN_OPTIONS] = NULL;
    }
    if(screen_img[SCREEN_END]) {
      SDL_FreeSurface(screen_img[SCREEN_END]);
      screen_img[SCREEN_END] = NULL;
    }
    if(end_btn_img) {
      SDL_FreeSurface(end_btn_img);
      end_btn_img = NULL;
    }
    if(screen_img[SCREEN_VS]) {
      SDL_FreeSurface(screen_img[SCREEN_VS]);
      screen_img[SCREEN_VS] = NULL;
    }
    if(emblems_img[HOME]) {
      SDL_FreeSurface(emblems_img[HOME]);
      emblems_img[HOME] = NULL;
    }
    if(emblems_img[VISITOR]) {
      SDL_FreeSurface(emblems_img[VISITOR]);
      emblems_img[VISITOR] = NULL;
    }
	if(screen_img[SCREEN_TEAM_SELECTOR]) {
      SDL_FreeSurface(screen_img[SCREEN_TEAM_SELECTOR]);
      screen_img[SCREEN_TEAM_SELECTOR] = NULL;
    }
	if(screen_img[SCREEN_NETWORK]) {
      SDL_FreeSurface(screen_img[SCREEN_NETWORK]);
      screen_img[SCREEN_NETWORK] = NULL;
    }
	if(vs_lower_chars) {
		SDL_FreeSurface(vs_lower_chars);
		vs_lower_chars = NULL;
	}
    if(vs_upper_chars) {
		SDL_FreeSurface(vs_upper_chars);
		vs_upper_chars = NULL;
	}
	if(vs_btn){
		SDL_FreeSurface(vs_btn);
		vs_btn = NULL;
	}
	if(vs_no_btn){
		SDL_FreeSurface(vs_no_btn);
		vs_no_btn = NULL;
	}
    if(vs_instructions_img){
		SDL_FreeSurface(vs_instructions_img);
		vs_instructions_img = NULL;
	}	
    if(credits_names_img){
		SDL_FreeSurface(credits_names_img);
		credits_names_img = NULL;
	}
	if(options_img){
		SDL_FreeSurface(options_img);
		options_img = NULL;
	}
    if(tick_img){
		SDL_FreeSurface(tick_img);
		tick_img = NULL;
	}
	if(no_emblem_img){
		SDL_FreeSurface(no_emblem_img);
		no_emblem_img = NULL;
	}
	if(ready_emblem_img){
		SDL_FreeSurface(ready_emblem_img);
		ready_emblem_img = NULL;
	}
	if(ready_img){
		SDL_FreeSurface(ready_img);
		ready_img = NULL;
	}
	break;
  }
}

int Screen::open_title(void){
  bool done=false;
  int r;
  type = SCREEN_TITLE;
  
  init();
  
  while(!done){
    draw();
    r = handle_event();
    done = (r!=NOP);   
  }
  visible = false;
  return r; 
}

int Screen::open_end(void){
  bool done=false;
  int r;
  type = SCREEN_END;
  
  init();
  
  while(!done){
    draw();
    r = handle_event();
    done = (r!=NOP);   
  }
  visible = false;
  free_images(type);
  return r; 
}

int Screen::open_credits(void){

#ifndef NOSOUND
	if(Config::is_sound_enabled()){
		sp->loop_sound(SND_TOMB, BGM_CHANNEL);
	}
#endif

  bool done=false;
  int r;
  type = SCREEN_CREDITS;
  credits_names_y = -CREDITS_NAMES_H+80;
  init();
  
  while(!done){
    draw();
    r = handle_event();
    done = (r!=NOP);   
  }
  visible = false;
  free_images(type);

#ifndef NOSOUND
  if(Config::is_sound_enabled()){
	  sp->stop_sound(BGM_CHANNEL);
  }
#endif

  return r; 
}

int Screen::open_options(void){

  bool done=false;
  int r;
  type = SCREEN_OPTIONS;
  if(res_tick==-1){
	if(screen_w==800 && screen_h==600) res_tick = 0;
	else if(screen_w==1024 && screen_h==768) res_tick = 1;
	else if(screen_w==1024 && screen_h==1054) res_tick = 2;
	else res_tick = -1;
  }
  
  init();
  
  do{
    draw();
    r = handle_event();
    done = (r!=NOP);   
  }while(!done);
  visible = false;
  free_images(type);

  return r; 
}

int Screen::open_vs(team_summary_t* home_p,team_summary_t* visitor_p){
  char str[256];
  type = SCREEN_VS;
  
  free_images(SCREEN_TITLE);
  init();
  
  sprintf(str,"%s/emblems/%s",IMG_PATH,home_p->emblem);
  emblems_img[HOME] = load_image(str,NO_TRANSPARENT);
  sprintf(str,"%s/emblems/%s",IMG_PATH,visitor_p->emblem);
  emblems_img[VISITOR] = load_image(str,NO_TRANSPARENT);
  
  this->home = home_p;
  this->visitor = visitor_p;

  draw();
  visible = false;
  free_images(type);
  return 0;
}

char* Screen::open_load_input(char *s){
	bool done=false;
	int r;
	type = SCREEN_LOAD;
	
	init();

	if(s){
		input_content = s;
		draw();
		SDL_Delay(1000);
		free_images(type);
		return NULL;
	}
	else {
		input_content = (char*) malloc( CHAR_16 );
		input_content[0] = '\0';
	}

	SDL_EnableUNICODE(true);
	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}
	SDL_EnableUNICODE(false);

	visible = false;
	free_images(type); 
	if(r==LOAD_GAME) return input_content;
	else {
		free(input_content);
		return NULL;
	}
}

opponent_t* Screen::open_team_selector(team_list_t *team_list_p){
	bool done=false;
	int r;
	type = SCREEN_TEAM_SELECTOR;
	
	free_images(SCREEN_TITLE);
	
	this->team_list = team_list_p;

	init();

	select_team(HOME,0);
	select_team(VISITOR,0);

	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}
	visible = false;
	free_images(type);
	if(r==BACK || r==EXIT) return NULL;
	else {
		opponent_t* opp = (opponent_t*) malloc(sizeof(opponent_t));
		opp->home = team_list->teams[pos[HOME]];
		opp->visitor = team_list->teams[pos[VISITOR]];
		return opp;
	} 
	
}

#ifndef NONET

opponent_t* Screen::open_network(team_list_t *team_list_p, Network * & net){
	bool done=false;
	int r;
	
	this->net = NULL;
	net_tick = -1;

	console = new Console(screen,NET_CONSOLE_X,NET_CONSOLE_Y,NET_CONSOLE_WIDTH,NET_CONSOLE_HEIGHT,CLIENT);

	type = SCREEN_NETWORK;
	
	free_images(SCREEN_TITLE);
	
	this->team_list = team_list_p;

	init();
  
	select_team(HOME,0);
	select_team(VISITOR,0);

    console->set_visible(true);
	console->print(NET_MSG_CLIENT_SERVER_0);
	console->print(NET_MSG_CLIENT_SERVER_1);
	waiting_for = NONE;
	adversary_is_ready = false;
	console_editable = false;

	do{
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}while(!done);
	visible = false;
	free_images(type);

    console->set_visible(false);
	delete console;

	if(r==BACK || r==EXIT ) {
		net = NULL;
		return NULL;
	}
	else {
		net = this->net;
		opponent_t* opp = (opponent_t*) malloc(sizeof(opponent_t));
		opp->home = (team_summary_t*) malloc(sizeof(team_summary_t));
		opp->home = team_list->teams[pos[HOME]];
		opp->visitor = (team_summary_t*) malloc(sizeof(team_summary_t));
		opp->visitor = team_list->teams[pos[VISITOR]];
		return opp;
	} 
	
}

void Screen::set_nickname(char* nick){
	if(nick && strlen(nick)>0 ) {
		console_editable = false;
		console->print(NET_MSG_HI, nick);
		if(net) net->set_nickname(nick);
		if(net && net->get_type()==CLIENT) {
			console->prompt(NET_MSG_SERVER_ADDRESS);
			console_editable = true;
			waiting_for = IP_ADDRESS;
		}
		else if(net && net->get_type()==SERVER){
			// je lance le serveur
			if( ((NetServer* )net)->open() ) {
				console->print(NET_MSG_SERVER_READY);
			}
			else {
				console->print(NET_MSG_SERVER_FAILED);
			}
		}
	}	
	else {
		console->prompt(NET_MSG_NICKNAME);
		console_editable = true;
		waiting_for = NICKNAME;		
	}
}

void Screen::check_for_connection(){
	if(((NetServer* )net)->wait_client()){
		console->print(NET_MSG_CLIENT_CONNECTED);
		net->send_msg("H:%s", net->get_nickname());
		net->send_msg("R:0");
		net->set_ready(false);
		console_editable = true;
	}
}

void Screen::try_to_connect(){
	if(((NetClient* )net)->connect()){
		console->print(NET_MSG_SERVER_CONNECTED);
		net->send_msg("H:%s", net->get_nickname());
		net->send_msg("R:0");
		net->set_ready(false);
		console_editable = true;
	}
	else {
		static int attempt = 0;
		attempt++;
		console->print(NET_MSG_CLIENT_CONNECTION_FAILED, attempt);
	}
}

void Screen::set_ip_address(char* ip){
	if(ip && strlen(ip)>0) {
		console_editable = false;
		if(net) {
			console->print(NET_MSG_TRY_CONNECT, ip);
			((NetClient*)net)->set_server_address(ip);
			if( !((NetClient*)net)->init_connection(ip) ){
				console->print(NET_MSG_RESOLVE_FAILED,ip);
			}
		}
	}	
	else {
		console->prompt(NET_MSG_SERVER_ADDRESS);
		console_editable = true;
		waiting_for = IP_ADDRESS;		
	}
}

void Screen::handle_msg(char* m){
	if( net && m && strlen(m) > 0) {

		if(m[0]=='C'){
			console->print(NET_MSG_CHAT_FORMAT, net->get_adversary_nickname(), m+2);
		}
		
		if(m[0]=='R'){
			adversary_is_ready = (atoi(m+2)==1);
			update = true;
		}

		if(m[0]=='H') {
			net->set_adversary_nickname(m+2);
			console->print(NET_MSG_ADVERSARY_IS, net->get_adversary_nickname());
		}
	}
}

void Screen::net_send_msg(char* m){
	if( net && m && strlen(m) > 0) {
		net->send_msg("C:%s", m);
		console->print("%s: %s",net->get_nickname(),m);
	}
}

#endif

void Screen::init(void){
  load_images(type);
  over_quick_game_btn = false;
  over_exhibition_btn = false;
  over_new_league_btn = false;
  over_new_cup_btn = false;
  over_load_game_btn = false;
  over_options_btn = false;
  over_network_btn = false;
  over_credits_btn = false;
  over_back_btn = false;
  over_exit_btn = false;
  over_credits_back_btn = false;
  over_credits_end_btn = false;
  over_vs_btn = false;
  update = true;
  visible = true;
  pos[HOME]=0;
  pos[VISITOR]=0;
}

void Screen::select_team(int team_id,int direction){
	pos[team_id]+=direction;
	if(pos[team_id]>=team_list->nb) pos[team_id] = 0;
	else if( pos[team_id] < 0 ) pos[team_id] = team_list->nb-1;
	
	char str[256];
	sprintf(str,"%s/emblems/%s",IMG_PATH,team_list->teams[pos[team_id]]->emblem);
	if(emblems_img[team_id]) SDL_FreeSurface(emblems_img[team_id]);
	emblems_img[team_id] = load_image(str,NO_TRANSPARENT);

	update = true;
}

int Screen::handle_event(){
  SDL_Event event;
  
  while(SDL_PollEvent(&event)) { 

#ifndef NONET
	if( console && console->get_focus() && (event.type==SDL_KEYDOWN) ){
		
		if(console_editable) console->handle_event(&event);

		if(event.key.keysym.sym==SDLK_RETURN && console->is_response_available() ){
			if( waiting_for==NICKNAME ) { 
				set_nickname(console->get_response());
			}
			else if(waiting_for==IP_ADDRESS ) { 
				set_ip_address(console->get_response());
			}
		} 
		
		cliprect.x = NET_CONSOLE_X;
		cliprect.y = NET_CONSOLE_Y;
		cliprect.w = NET_CONSOLE_WIDTH;
		cliprect.h = NET_CONSOLE_HEIGHT;
	}
#endif

	  switch (event.type) {
	  case SDL_USEREVENT:
		return NOP;
		break;
	  case SDL_QUIT:
		return EXIT;
      break;
    case SDL_KEYDOWN:
	
		switch(event.key.keysym.sym){

#ifndef NOSOUND
	case SDLK_F3:
		if(Config::is_sound_enabled()){
			sp->dec_volume();
		}
		break;
	case SDLK_F4:
		if(Config::is_sound_enabled()){
			sp->inc_volume();
		}
		break;
#endif

	  case SDLK_RETURN:
		  if(type==SCREEN_LOAD) return LOAD_GAME;
		  break;
      case SDLK_ESCAPE:
		return EXIT;
	break;
      case SDLK_BACKSPACE:
		  if(type==SCREEN_LOAD) erase_input();
			  break;
	  default:
		  if(type==SCREEN_LOAD) handle_input(event.key.keysym.unicode);
	break;
      }
      break; // esac de SDL_KEYDOWN
    case SDL_MOUSEMOTION:
      over_quick_game_btn = false;
      over_exhibition_btn = false;
      over_new_league_btn = false;
      over_new_cup_btn = false;
      over_load_game_btn = false;
      over_options_btn = false;
      over_network_btn = false;
      over_credits_btn = false;
      over_back_btn = false;
      over_exit_btn = false;
      over_credits_back_btn = false;
      over_credits_end_btn = false;
	  over_vs_btn = false;
	  switch(type){
	  case SCREEN_TEAM_SELECTOR:  
	  if( OVER_VS_BTN(event.motion.x,event.motion.y) ){
		over_vs_btn = true;
	  }
		cliprect.x = VS_BTN_X;
		cliprect.y = VS_BTN_Y;
		cliprect.w = VS_BTN_W;
		cliprect.h = VS_BTN_H;
	  break;

#ifndef NONET
	  case SCREEN_NETWORK:  
	  if(OVER_CREDITS_BACK_BTN(event.motion.x,event.motion.y)){
		over_credits_back_btn = true;
	  }
	  else if( net && net->get_type()==SERVER && ((NetServer*)net)->is_client_connected() && net->is_ready() && adversary_is_ready && OVER_VS_BTN(event.motion.x,event.motion.y) ){
		over_vs_btn = true;
	  }
	  update = true;
	  break;
#endif

	  case SCREEN_TITLE:
	if( OVER_QUICK_GAME_BTN(event.motion.x,event.motion.y) ){
	  over_quick_game_btn = true;
	}	
	else if( OVER_EXHIBITION_BTN(event.motion.x,event.motion.y) ){
	  over_exhibition_btn = true;
	}
	else if( OVER_NEW_LEAGUE_BTN(event.motion.x,event.motion.y) ){
	  over_new_league_btn = true;
	}
	else if( OVER_NEW_CUP_BTN(event.motion.x,event.motion.y) ){
	  over_new_cup_btn = true;
	}
	else if( OVER_LOAD_GAME_BTN(event.motion.x,event.motion.y) ){
	  over_load_game_btn = true;
	}
	else if( OVER_OPTIONS_BTN(event.motion.x,event.motion.y) ){
	  over_options_btn = true;
	}
	else if( OVER_NETWORK_BTN(event.motion.x,event.motion.y) ){
	  over_network_btn = true;
	}
	else if( OVER_CREDITS_BTN(event.motion.x,event.motion.y) ){
	  over_credits_btn = true;
	}
	
	cliprect.x = TITLE_LEFT_BTN_X;
	cliprect.y = TITLE_RIGHT_BTN_Y;
	cliprect.w = TITLE_RIGHT_BTN_W+TITLE_RIGHT_BTN_X-TITLE_LEFT_BTN_X;
	cliprect.h = TITLE_LEFT_BTN_H;
	
	break; 
      case SCREEN_CREDITS:
	if(OVER_CREDITS_BACK_BTN(event.motion.x,event.motion.y)){
	  over_credits_back_btn = true;
	}
	else if(OVER_EXIT_BTN(event.motion.x,event.motion.y)){
	  over_credits_end_btn = true;
	}
	
	update = true;
	break;
      case SCREEN_OPTIONS:
	if(OVER_CREDITS_BACK_BTN(event.motion.x,event.motion.y)){
	  over_credits_back_btn = true;
	}
	else if(OVER_EXIT_BTN(event.motion.x,event.motion.y)){
	  over_credits_end_btn = true;
	}
	cliprect.x = CREDITS_BACK_BTN_X;
	cliprect.y = OPTIONS_Y;
	cliprect.w = EXIT_BTN_X - CREDITS_BACK_BTN_X + EXIT_BTN_W;
	cliprect.h = EXIT_BTN_Y - OPTIONS_Y + EXIT_BTN_H;
	
	break;

      case SCREEN_END:
	if( OVER_EXIT_BTN(event.motion.x,event.motion.y) ){
	  over_exit_btn = true;
	}
	else if( OVER_BACK_BTN(event.motion.x,event.motion.y) ){
	  over_back_btn = true;
	}
	cliprect.x = BACK_BTN_X;
	cliprect.y = BACK_BTN_Y;
	cliprect.w = EXIT_BTN_X - BACK_BTN_X + EXIT_BTN_W;
	cliprect.h = BACK_BTN_H;
	
	break;
      default:
	break;
      }
      break; // esac de SDL_MOUSEMOTION
    case SDL_MOUSEBUTTONDOWN:
      if( event.button.button==SDL_BUTTON_LEFT ){
		switch(type){
		case SCREEN_TEAM_SELECTOR:
			if( OVER_HOME_EMBLEM(event.motion.x,event.motion.y) ) select_team(HOME, -1);
			else if( OVER_VISITOR_EMBLEM(event.motion.x,event.motion.y) ) select_team(VISITOR, -1);
			else if(over_vs_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif

				return EXHIBITION;
			}
		break;

#ifndef NONET

		case SCREEN_NETWORK:
			if( net && net->get_type()==SERVER && OVER_HOME_EMBLEM(event.motion.x,event.motion.y) ) select_team(HOME, -1);
			else if( net && net->get_type()==CLIENT && OVER_VISITOR_EMBLEM(event.motion.x,event.motion.y) ) select_team(VISITOR, -1);
			else if(over_vs_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif

				return NETWORK;
			}
			else if(over_credits_back_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif
				return BACK;
			}
			else if(!net && OVER_SERVER(event.motion.x,event.motion.y)){
				net_tick = 0;
				net = new NetServer();
				net->set_ready(false);
				console->print(NET_MSG_SERVER_SELECTED);
				console->prompt(NET_MSG_NICKNAME);
				console_editable = true;
				waiting_for = NICKNAME;
			}
			else if(!net && OVER_CLIENT(event.motion.x,event.motion.y)){
				net_tick = 1;
				net = new NetClient();
				net->set_ready(false);
				console->print(NET_MSG_CLIENT_SELECTED);
				console->prompt(NET_MSG_NICKNAME);
				console_editable = true;
				waiting_for = NICKNAME;
			}
			
			if( net && net->get_type()==SERVER && ((NetServer*)net)->is_client_connected() && OVER_READY_BTN(event.motion.x,event.motion.y) ) {
				if(net->is_ready()){
					net->set_ready(false);
					net->send_msg("R:0");
				}
				else{
					net->set_ready(true);
					net->send_msg("R:1");
				}
				update = true;
			}
			
			if( net && net->get_type()==CLIENT && ((NetClient*)net)->is_connected() && OVER_READY_BTN(event.motion.x,event.motion.y) ) {
				if(net->is_ready()){
					net->set_ready(false);
					net->send_msg("R:0");
				}
				else{
					net->set_ready(true);
					net->send_msg("R:1");
				}	
				update = true;
			}
			
		break;

#endif

		case SCREEN_TITLE:
			if(over_quick_game_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif
				return QUICK_GAME;
			}
			else if(over_new_league_btn ||over_new_cup_btn ){
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_KEYS,FX0_CHANNEL);
				}
#endif
				printf("Not Yet Implemented...\n");
			}
			else if(over_exhibition_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif
				return EXHIBITION;
			}
			//else if(over_new_league_btn) return NEW_LEAGUE;
			//else if(over_new_cup_btn) return NEW_CUP;
			else if(over_load_game_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif
				return LOAD_GAME;
			}
			else if(over_options_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif
				return OPTIONS;
			}
			else if(over_network_btn) {

#ifndef NONET
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif
				return NETWORK;
#else //if
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_KEYS,FX0_CHANNEL);
				}
#endif
				printf("Not Yet Implemented...\n");
#endif
			}
			else if(over_credits_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif
				return CREDITS;
			}
		break; 
		case SCREEN_CREDITS:
			if(over_credits_back_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif
				return BACK;
			}
			else if(over_credits_end_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif 
				return EXIT;
			}
			break;
		case SCREEN_OPTIONS:
			if(over_credits_back_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif
				Config::set_future_resolution(res_tick);
				Config::set_sound_enabled((sound_tick==0)?false:true);
				Config::set_video_enabled((video_tick==0)?false:true);
				Config::set_language(language_tick);
				Config::save();
				
				return BACK;
			}
			else if(over_credits_end_btn) {
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif 
				Config::set_future_resolution(res_tick);
				Config::set_sound_enabled((sound_tick==0)?false:true);
				Config::set_video_enabled((video_tick==0)?false:true);
				Config::set_language(language_tick);
				Config::save();

				return EXIT;
			}
			else if(OVER_RESOLUTION_MIN(event.motion.x,event.motion.y)){
				res_tick = 0;
			}
			else if(OVER_RESOLUTION_MED(event.motion.x,event.motion.y)){
				res_tick = 1;
			}
			else if(OVER_RESOLUTION_MAX(event.motion.x,event.motion.y)){
				res_tick = 2;
			}
			else if(OVER_SOUND_ENABLE(event.motion.x,event.motion.y)){
				sound_tick = 1;
			}
			else if(OVER_SOUND_DISABLE(event.motion.x,event.motion.y)){
				sound_tick = 0;
			}
			else if(OVER_VIDEO_ENABLE(event.motion.x,event.motion.y)){
				video_tick = 1;
			}
			else if(OVER_VIDEO_DISABLE(event.motion.x,event.motion.y)){
				video_tick = 0;
			}
			else if(OVER_LANGUAGE_FR(event.motion.x,event.motion.y)){
				language_tick = 0;
			}
			else if(OVER_LANGUAGE_EN(event.motion.x,event.motion.y)){
				language_tick = 1;
			}
			break;
		case SCREEN_END:
			if(over_back_btn){
#ifndef NOSOUND
				if(Config::is_sound_enabled()){
					sp->play_sound(SND_COINS,FX0_CHANNEL);
				}
#endif 
				return BACK;
			}
			else if (over_exit_btn) return EXIT;
		break;
		default:
		break;
		}
	  }else if(event.button.button==SDL_BUTTON_RIGHT) {
		switch(type){
		case SCREEN_TEAM_SELECTOR:
			if( OVER_HOME_EMBLEM(event.motion.x,event.motion.y) ) select_team(HOME, 1);
			else if( OVER_VISITOR_EMBLEM(event.motion.x,event.motion.y) ) select_team(VISITOR, 1);
		break;
		case SCREEN_NETWORK:
			if( net && net->get_type()==SERVER && OVER_HOME_EMBLEM(event.motion.x,event.motion.y) ) select_team(HOME, 1);
			else if( net && net->get_type()==CLIENT && OVER_VISITOR_EMBLEM(event.motion.x,event.motion.y) ) select_team(VISITOR, 1);
		break;
		default:
		break;
		}
	  }
      break; // esac de SDL_MOUSEBUTTONDOWN
    default:
      break;
    }
  }
  return NOP;
}

void Screen::draw(){

  SDL_Rect s,d;
 
  if(update){
    cliprect.x = 0;
    cliprect.y = 0;
    cliprect.w = screen_w;
    cliprect.h = screen_h;
	SDL_SetClipRect(screen,&cliprect);
  }
  else if(cliprect.w!=0 && cliprect.h!=0) {
    SDL_SetClipRect(screen,&cliprect);
  }
  else if(!update && (type==SCREEN_CREDITS || type==SCREEN_NETWORK) ){
	  if(type==SCREEN_CREDITS){
		cliprect.x = CREDITS_NAMES_X;
		cliprect.y = CREDITS_NAMES_Y;
		cliprect.w = CREDITS_NAMES_W;
		cliprect.h = CREDITS_NAMES_H;
	  }
	  else {
		cliprect.x = NET_CONSOLE_X;
		cliprect.y = NET_CONSOLE_Y;
		cliprect.w = NET_CONSOLE_WIDTH;
		cliprect.h = NET_CONSOLE_HEIGHT;
	  }
	  SDL_SetClipRect(screen,&cliprect);
  }
  else {
    cliprect.x = 0;
    cliprect.y = 0;
    cliprect.w = 0;
    cliprect.h = 0;
    SDL_SetClipRect(screen,&cliprect);
    wait_for_next_frame();
    return;
  }
  
  update = false;
  cliprect.x = 0;
  cliprect.y = 0;
  cliprect.w = 0;
  cliprect.h = 0;
  
  if(type!=SCREEN_LOAD){ 
		SDL_FillRect(screen, NULL,  SDL_MapRGB(screen->format,60,60,60));
		// on draw le bg de la page
		s.x = SCREEN_MARGIN_W;
		s.y = SCREEN_MARGIN_H;
		s.w = SCREEN_WIDTH;
		s.h = SCREEN_HEIGHT;
		BlitSurface(screen_img[type],NULL,screen,&s);
  }
  else {
		s.x = LOAD_GAME_BTN_X;
		s.y = LOAD_GAME_BTN_Y;
		s.w = LOAD_GAME_BTN_W;
		s.h = LOAD_GAME_BTN_H;
		BlitSurface(load_input_img, NULL, screen, &s);
  }

  switch(type){
  case SCREEN_TEAM_SELECTOR:
	if(emblems_img[HOME]){
		d.x = HOME_EMBLEM_X;
		d.y = HOME_EMBLEM_Y;
		d.w = EMBLEM_W;
		d.h = EMBLEM_H;
		BlitSurface(emblems_img[HOME],NULL,screen,&d);
	}
	if(emblems_img[VISITOR]){
		d.x = VISITOR_EMBLEM_X;
		d.y = VISITOR_EMBLEM_Y;
		d.w = EMBLEM_W;
		d.h = EMBLEM_H;
		BlitSurface(emblems_img[VISITOR],NULL,screen,&d);
    }

	if(team_list && team_list->teams[pos[HOME]] && team_list->teams[pos[HOME]]->name){	
		d.x = VS_HOME_X;
		d.y = VS_HOME_Y;
		d.w = VS_HOME_W;
		d.h = VS_HOME_H;
		draw_string(team_list->teams[pos[HOME]]->name,&d,vs_lower_chars,vs_upper_chars,screen);		
	}

	if(team_list && team_list->teams[pos[VISITOR]] && team_list->teams[pos[VISITOR]]->name){
		d.x = VS_VISITOR_X;
		d.y = VS_VISITOR_Y;
		d.w = VS_VISITOR_W;
		d.h = VS_VISITOR_H;
		draw_string(team_list->teams[pos[VISITOR]]->name,&d,vs_lower_chars,vs_upper_chars,screen);
	}
	
	d.x = VS_INSTRUCTIONS_X;
	d.y = VS_INSTRUCTIONS_Y;
	d.w = VS_INSTRUCTIONS_W;
	d.h = VS_INSTRUCTIONS_H;
	BlitSurface(vs_instructions_img,NULL,screen,&d);

	if(over_vs_btn) {
		d.x = VS_BTN_X;
		d.y = VS_BTN_Y;
		d.w = VS_BTN_W;
		d.h = VS_BTN_H;
		BlitSurface(vs_btn,NULL,screen,&d);	
	}

  break;

#ifndef NONET

  case SCREEN_NETWORK:
	if(net && net->get_type()==SERVER && emblems_img[HOME]){
		d.x = HOME_EMBLEM_X;
		d.y = HOME_EMBLEM_Y;
		d.w = EMBLEM_W;
		d.h = EMBLEM_H;
		BlitSurface(emblems_img[HOME],NULL,screen,&d);
	}
	else {
		d.x = HOME_EMBLEM_X;
		d.y = HOME_EMBLEM_Y;
		d.w = EMBLEM_W;
		d.h = EMBLEM_H;
		BlitSurface(no_emblem_img,NULL,screen,&d);
	}
	
	if(net && net->get_type()==CLIENT && emblems_img[VISITOR]){
		d.x = VISITOR_EMBLEM_X;
		d.y = VISITOR_EMBLEM_Y;
		d.w = EMBLEM_W;
		d.h = EMBLEM_H;
		BlitSurface(emblems_img[VISITOR],NULL,screen,&d);
    }
	else {
		d.x = VISITOR_EMBLEM_X;
		d.y = VISITOR_EMBLEM_Y;
		d.w = EMBLEM_W;
		d.h = EMBLEM_H;
		BlitSurface(no_emblem_img,NULL,screen,&d);
	}

	if(adversary_is_ready){
		d.x = (net->get_type()==SERVER)?(VISITOR_EMBLEM_X):(HOME_EMBLEM_X);
		d.y = (net->get_type()==SERVER)?(VISITOR_EMBLEM_Y):(HOME_EMBLEM_Y);
		d.w = EMBLEM_W;
		d.h = EMBLEM_H;
		BlitSurface(ready_emblem_img,NULL,screen,&d);
	}

	if(net && net->get_type()==SERVER && team_list && team_list->teams[pos[HOME]] && team_list->teams[pos[HOME]]->name){	
		d.x = VS_HOME_X;
		d.y = VS_HOME_Y;
		d.w = VS_HOME_W;
		d.h = VS_HOME_H;
		draw_string(team_list->teams[pos[HOME]]->name,&d,vs_lower_chars,vs_upper_chars,screen);		
	}

	if(net && net->get_type()==CLIENT && team_list && team_list->teams[pos[VISITOR]] && team_list->teams[pos[VISITOR]]->name){
		d.x = VS_VISITOR_X;
		d.y = VS_VISITOR_Y;
		d.w = VS_VISITOR_W;
		d.h = VS_VISITOR_H;
		draw_string(team_list->teams[pos[VISITOR]]->name,&d,vs_lower_chars,vs_upper_chars,screen);
	}
	
	if(over_vs_btn) {
		d.x = VS_BTN_X;
		d.y = VS_BTN_Y;
		d.w = VS_BTN_W;
		d.h = VS_BTN_H;
		BlitSurface(vs_btn,NULL,screen,&d);	
	}
	else if(over_credits_back_btn) {
      s.x = 0;
      s.y = 0;
      s.w = BACK_BTN_W;
      s.h = BACK_BTN_H;
      d.x = CREDITS_BACK_BTN_X;
      d.y = CREDITS_BACK_BTN_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(end_btn_img,&s,screen,&d);
    }
    
    if(net_tick>=0){
		d.x = (net_tick==0?(NET_SERVER_X):(NET_CLIENT_X));
		d.y = NET_TYPE_Y - 8;
		d.w = tick_img->w;
		d.h = tick_img->h;
		BlitSurface(tick_img,NULL,screen,&d);    
	}

	if( !(net && net->get_type()==SERVER && ((NetServer*)net)->is_client_connected() && net->is_ready() && adversary_is_ready ) ){
		d.x = VS_BTN_X;
		d.y = VS_BTN_Y;
		d.w = VS_BTN_W;
		d.h = VS_BTN_H;
		BlitSurface(vs_no_btn,NULL,screen,&d);
	}

	if (net){
		s.x = (net->is_ready())?(READY_BTN_W):(0);
		s.y = 0;
		s.w = READY_BTN_W;
		s.h = READY_BTN_H;
		d.x = READY_BTN_X;
		d.y = READY_BTN_Y;
		d.w = READY_BTN_W;
		d.h = READY_BTN_H;
		BlitSurface(ready_img,&s,screen,&d);
	}

	if( net && net->get_type()==SERVER && ((NetServer*)net)->is_opened() && !((NetServer*)net)->is_client_connected() ) {
		check_for_connection();
	}

	if( net && net->get_type()==CLIENT && ((NetClient*)net)->get_server_address() && !((NetClient*)net)->is_connected() ) {
		try_to_connect();
	}

	if (net && net->get_type()==SERVER && ((NetServer*)net)->is_opened() && ((NetServer*)net)->is_client_connected() ) {
		handle_msg(net->rcv_msg(100));
	}

	if( net && net->get_type()==CLIENT && ((NetClient*)net)->is_connected() ) {
		handle_msg(net->rcv_msg(100));
	}

	if(console && console->get_visible()){
		console->draw();
	}

  break;
  
#endif

  case SCREEN_VS:
    d.x = HOME_EMBLEM_X;
    d.y = HOME_EMBLEM_Y;
    d.w = EMBLEM_W;
    d.h = EMBLEM_H;
    BlitSurface(emblems_img[HOME],NULL,screen,&d);
    d.x = VISITOR_EMBLEM_X;
    d.y = VISITOR_EMBLEM_Y;
    d.w = EMBLEM_W;
    d.h = EMBLEM_H;
    BlitSurface(emblems_img[VISITOR],NULL,screen,&d);
    
	if(home && home->name){	
		d.x = VS_HOME_X;
		d.y = VS_HOME_Y;
		d.w = VS_HOME_W;
		d.h = VS_HOME_H;
		draw_string(home->name,&d,vs_lower_chars,vs_upper_chars,screen);
	}

	if(visitor && visitor->name){
		d.x = VS_VISITOR_X;
		d.y = VS_VISITOR_Y;
		d.w = VS_VISITOR_W;
		d.h = VS_VISITOR_H;
		draw_string(visitor->name,&d,vs_lower_chars,vs_upper_chars,screen);
	}
	break;
  case SCREEN_TITLE:
    s.x = 0;
    if(over_quick_game_btn) {
      s.x = QUICK_GAME_BTN_W;
    }
    s.y = 0;
    s.w = QUICK_GAME_BTN_W;
    s.h = QUICK_GAME_BTN_H;
    d.x = QUICK_GAME_BTN_X;
    d.y = QUICK_GAME_BTN_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(title_left_btn_img,&s,screen,&d);
    ///
    s.x = 0;
    if(over_exhibition_btn) {
      s.x = EXHIBITION_BTN_W;
    }
    s.y = QUICK_GAME_BTN_H;
    s.w = EXHIBITION_BTN_W;
    s.h = EXHIBITION_BTN_H;
    d.x = EXHIBITION_BTN_X;
    d.y = EXHIBITION_BTN_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(title_left_btn_img,&s,screen,&d);
     ///
    s.x = 0;
    if(over_new_league_btn) {
      s.x = NEW_LEAGUE_BTN_W;
    }
    s.y = QUICK_GAME_BTN_H+EXHIBITION_BTN_H;
    s.w = NEW_LEAGUE_BTN_W;
    s.h = NEW_LEAGUE_BTN_H;
    d.x = NEW_LEAGUE_BTN_X;
    d.y = NEW_LEAGUE_BTN_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(title_left_btn_img,&s,screen,&d);
    ///
    s.x = 0;
    if(over_new_cup_btn) {
      s.x = NEW_CUP_BTN_W;
    }
    s.y = QUICK_GAME_BTN_H+EXHIBITION_BTN_H+NEW_LEAGUE_BTN_H;
    s.w = NEW_CUP_BTN_W;
    s.h = NEW_CUP_BTN_H;
    d.x = NEW_CUP_BTN_X;
    d.y = NEW_CUP_BTN_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(title_left_btn_img,&s,screen,&d);
    ///
    s.x = 0;
    if(over_load_game_btn) {
      s.x = LOAD_GAME_BTN_W;
    }
    s.y = 0;
    s.w = LOAD_GAME_BTN_W;
    s.h = LOAD_GAME_BTN_H;
    d.x = LOAD_GAME_BTN_X;
    d.y = LOAD_GAME_BTN_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(title_right_btn_img,&s,screen,&d);
    ///
    s.x = 0;
    if(over_options_btn) {
      s.x = OPTIONS_BTN_W;
    }
    s.y = LOAD_GAME_BTN_H;
    s.w = OPTIONS_BTN_W;
    s.h = OPTIONS_BTN_H;
    d.x = OPTIONS_BTN_X;
    d.y = OPTIONS_BTN_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(title_right_btn_img,&s,screen,&d);
     ///
    s.x = 0;
    if(over_network_btn) {
      s.x = NETWORK_BTN_W;
    }
    s.y = LOAD_GAME_BTN_H+OPTIONS_BTN_H;
    s.w = NETWORK_BTN_W;
    s.h = NETWORK_BTN_H;
    d.x = NETWORK_BTN_X;
    d.y = NETWORK_BTN_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(title_right_btn_img,&s,screen,&d);
    ///
    s.x = 0;
    if(over_credits_btn) {
      s.x = CREDITS_BTN_W;
    }
    s.y = LOAD_GAME_BTN_H+OPTIONS_BTN_H+NETWORK_BTN_H;
    s.w = CREDITS_BTN_W;
    s.h = CREDITS_BTN_H;
    d.x = CREDITS_BTN_X;
    d.y = CREDITS_BTN_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(title_right_btn_img,&s,screen,&d);
    break;
  case SCREEN_CREDITS:
    if(over_credits_back_btn) {
      s.x = 0;
      s.y = 0;
      s.w = BACK_BTN_W;
      s.h = BACK_BTN_H;
      d.x = CREDITS_BACK_BTN_X;
      d.y = CREDITS_BACK_BTN_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(end_btn_img,&s,screen,&d);
    }
    if(over_credits_end_btn) {
      s.x = BACK_BTN_W;
      s.y = 0;
      s.w = EXIT_BTN_W;
      s.h = EXIT_BTN_H;
      d.x = EXIT_BTN_X;
      d.y = EXIT_BTN_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(end_btn_img,&s,screen,&d);
    }

	s.x = 0;
    s.y = credits_names_y;
    s.w = credits_names_img->w;
    s.h = MIN(credits_names_img->h, CREDITS_MAX_HIGH);
    d.x = CREDITS_NAMES_X;
    d.y = CREDITS_NAMES_Y;
    d.w = CREDITS_NAMES_W;
    d.h = CREDITS_NAMES_H;
	BlitSurface(credits_names_img,&s,screen,&d);    
    if(credits_names_y < credits_names_img->h-30) {
		credits_names_y+=2;
	}
	else {
#ifndef NOSOUND
	if(Config::is_sound_enabled()){
		sp->stop_sound(BGM_CHANNEL);
	}
#endif
		credits_names_y = credits_names_img->h-30;
	}

	break;
    case SCREEN_OPTIONS:
    if(over_credits_back_btn) {
      s.x = 0;
      s.y = 0;
      s.w = BACK_BTN_W;
      s.h = BACK_BTN_H;
      d.x = CREDITS_BACK_BTN_X;
      d.y = CREDITS_BACK_BTN_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(end_btn_img,&s,screen,&d);
    }
    if(over_credits_end_btn) {
      s.x = BACK_BTN_W;
      s.y = 0;
      s.w = EXIT_BTN_W;
      s.h = EXIT_BTN_H;
      d.x = EXIT_BTN_X;
      d.y = EXIT_BTN_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(end_btn_img,&s,screen,&d);
    }

    d.x = OPTIONS_X;
    d.y = OPTIONS_Y;
    d.w = options_img->w;
    d.h = options_img->h;
	
	BlitSurface(options_img,NULL,screen,&d);    

    if(res_tick>=0){
		d.x = (res_tick==0?(RES_MIN_X):(res_tick==1?(RES_MED_X):(RES_MAX_X)));
		d.y = RES_Y - 8;
		d.w = tick_img->w;
		d.h = tick_img->h;
		BlitSurface(tick_img,NULL,screen,&d);    
	}

	if(sound_tick>=0){
		d.x = (sound_tick==1?(SOUND_ENABLE_X):(SOUND_DISABLE_X));
		d.y = SOUND_Y - 8;
		d.w = tick_img->w;
		d.h = tick_img->h;
		BlitSurface(tick_img,NULL,screen,&d);    
	}

	if(video_tick>=0){
		d.x = (video_tick==1?(VIDEO_ENABLE_X):(VIDEO_DISABLE_X));
		d.y = VIDEO_Y - 8;
		d.w = tick_img->w;
		d.h = tick_img->h;
		BlitSurface(tick_img,NULL,screen,&d);    
	}

	if(language_tick>=0){
		d.x = (language_tick==0?(LANGUAGE_FR_X):(LANGUAGE_EN_X));
		d.y = LANGUAGE_Y - 8;
		d.w = tick_img->w;
		d.h = tick_img->h;
		BlitSurface(tick_img,NULL,screen,&d);    
	}

	  break;
  case SCREEN_END:
    if(over_back_btn) {
      s.x = 0;
      s.y = 0;
      s.w = BACK_BTN_W;
      s.h = BACK_BTN_H;
      d.x = BACK_BTN_X;
      d.y = BACK_BTN_Y;
      d.w = s.w;
      d.h = s.h;
    BlitSurface(end_btn_img,&s,screen,&d);
    }
    else if(over_exit_btn) {
      s.x = BACK_BTN_W;
      s.y = 0;
      s.w = EXIT_BTN_W;
      s.h = EXIT_BTN_H;
      d.x = EXIT_BTN_X;
      d.y = EXIT_BTN_Y;
      d.w = s.w;
      d.h = s.h;
    BlitSurface(end_btn_img,&s,screen,&d);
    }
    break;
  case SCREEN_LOAD:
	  if(strlen(input_content)>0) {
		d.x = LOAD_GAME_BTN_X-20;
		d.y = LOAD_GAME_BTN_Y+3;
		d.w = LOAD_GAME_BTN_W;
		d.h = LOAD_GAME_BTN_H;	  
		draw_string(input_content,&d,vs_lower_chars,vs_upper_chars,screen);  
	  }

	  d.x = LOAD_GAME_BTN_X + LOAD_GAME_BTN_W/2 + strlen(input_content)*vs_lower_chars->w/52 - 18;
	  d.y = LOAD_GAME_BTN_Y + 4;
	  d.w = 2;
	  d.h = LOAD_GAME_BTN_H - 7;

	  SDL_FillRect(screen,&d,0);
	  break;
  default:
    break;
  }
  // et on flippe
  SDL_Flip(screen);
  wait_for_next_frame();   
} 

void Screen::wait_for_next_frame(){
  static Uint32 next_tick = 0;
  Uint32 this_tick;
  
  /* Wait for the next frame */
  this_tick = SDL_GetTicks(); 
  if ( this_tick < next_tick ) {
    SDL_Delay(next_tick-this_tick);
  }
  next_tick = this_tick + (1000/SCREEN_FPS);
}

SDL_Surface* Screen::load_image(char *datafile, int transparent){
  SDL_Surface *image;
  image = IMG_Load(datafile);
  if ( image == NULL ) {
    fprintf(stderr, "Couldn't load image %s: %s\n",datafile, IMG_GetError());
    return NULL;
  }
   
  if ( transparent ) {
    /* Assuming 8-bit BMP image */
    SDL_SetColorKey(image, (SDL_SRCCOLORKEY|SDL_RLEACCEL),*(Uint8 *)image->pixels);
  }
  
  SDL_Surface *r =  SDL_DisplayFormat(image);
  SDL_FreeSurface(image);

  return r;
}

void Screen::erase_input(){
	if(strlen(input_content)>0) {
		input_content[strlen(input_content)-1] = '\0';
		cliprect.x = LOAD_GAME_BTN_X;
		cliprect.y = LOAD_GAME_BTN_Y;
		cliprect.w = LOAD_GAME_BTN_W;
		cliprect.h = LOAD_GAME_BTN_H;
	}
}

void Screen::handle_input(Uint16 key){
	// input_content;
	if(strlen(input_content)>=14) return; 
	sprintf(input_content, "%s%c",input_content,(char) key);
	cliprect.x = LOAD_GAME_BTN_X;
	cliprect.y = LOAD_GAME_BTN_Y;
	cliprect.w = LOAD_GAME_BTN_W;
	cliprect.h = LOAD_GAME_BTN_H;
}

void Screen::draw_string(char* str,SDL_Rect* placement,SDL_Surface* lower_font, SDL_Surface* upper_font, SDL_Surface* where){
  SDL_Rect s,d;
  int str_len = strlen(str);
  int i;

  for(i=0;str[i]!='\0';i++){
    if((str[i]-'a') >= 0){
      s.x = (str[i] - 'a')*lower_font->w/26;
      s.y = 0;
      s.w = lower_font->w/26;
      s.h = lower_font->h;
      d.w = s.w;
      d.h = d.h;
      d.x = placement->x + placement->w/2 - str_len*lower_font->w/52 + i*lower_font->w/26;
      d.y = placement->y + placement->h/2 - lower_font->h/2;
      BlitSurface(lower_font,&s,where,&d);
    }
    else if( (str[i]-'A') >=0 ){
      s.x = (str[i] - 'A')*upper_font->w/26;
      s.y = 0;
      s.w = upper_font->w/26;
      s.h = upper_font->h;
      d.w = s.w;
      d.h = d.h;
      d.x = placement->x + placement->w/2 - str_len*upper_font->w/52 + i*upper_font->w/26;
      d.y = placement->y + placement->h/2 - upper_font->h/2 - 3;
      BlitSurface(upper_font,&s,where,&d);
    }
  }
}

