/* $Id: DEROutputStream.java,v 1.3 2000/02/09 20:29:58 gelderen Exp $
 *
 * Copyright (C) 2000 The Cryptix Foundation Limited.
 * All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library;
 * if not, you can download a copy from http://www.cryptix.org/ .
 */
package cryptix.jce.util.der;


import java.math.BigInteger;

import java.io.DataInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;


/**
 * Stream for writing (DER encoding) ASNObjects.
 *
 * @version $Revision: 1.3 $
 * @author  Jeroen C. van Gelderen (gelderen@cryptix.org)
 */
public final class DEROutputStream
{

// Instance variables
//...........................................................................

    private final OutputStream os;


// Constructors
//...........................................................................

    public DEROutputStream(OutputStream os)
    {
        //ASSERT(os != null);
        this.os = os;
    }


// Stream methods
//...........................................................................

    /**
     * Close this stream.
     *
     * @throws IOException  if the underlying OutputStream throws one.
     */
    public void close()
    throws IOException
    {
        this.os.close();
    }


    /**
     * Flush this stream.
     *
     * @throws IOException  if the underlying OutputStream throws one.
     */
    public void flush()
    throws IOException
    {
        this.os.flush();
    }


    /**
     * Write the given ASNObject to the underlying OutputStream. The ASNObject
     * is responsible for it's own encoding.
     *
     * @param o  ASNObject to be written.
     *
     * @throws IOException           If underlying OutputStream throws one.
     * @throws NullPointerException  When null is passed as argument.
     */
    public void write(ASNObject o)
    throws IOException
    {
        o.write(this);
    }


// Visitor methods
//...........................................................................

    /**
     * Write type byte to the underlying OutputStream.
     */
    /*package*/ void writeType(byte type) throws IOException
    {
        this.os.write( ((int)type) & 0xFF );
    }


    /**
     * Write a length to the underlying OutputStream. The encoding is 
     * variable length, ASN.1.
     */
    /*package*/ void writeLength(int len) throws IOException
    {
        if( len<0 )
            throw new IllegalArgumentException("len: < 0");
            
        int lenOfLen;
        if( len <= 127 )
            lenOfLen = 1;
        else if( len <= 0xFFFF )
            lenOfLen = 2;
        else if( len <= 0xFFFFFF )
            lenOfLen = 3;
        else
            lenOfLen = 4;

        if( lenOfLen != 1 )
            // encode length of length        
            this.os.write( (byte)(lenOfLen | 0x80) );
        
        // write length bytes
        while( lenOfLen-- > 0 )
            this.os.write( (byte)(len >>> (lenOfLen*8)) );
    }


    /**
     * Write a plain array of bytes to the underlying OutputStream.
     */
    /*package*/ void writeBytes(byte[] bytes) throws IOException
    {
        this.os.write(bytes);
    }
}
