/* $Id: PaddingMD.java,v 1.5 2000/01/20 14:59:31 gelderen Exp $
 *
 * Copyright (C) 1995-2000 The Cryptix Foundation Limited.
 * All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library;
 * if not, you can download a copy from http://www.cryptix.org/ .
 */

package cryptix.jce.provider.md;


import java.security.DigestException;
import java.security.MessageDigestSpi;


/**
 * This abstract class implements the MD4-like block/padding structure as it
 * is used by most hashes (MD4, MD5, SHA-0, SHA-1, RIPEMD-128, RIPEMD-160).
 *
 * This class handles the message buffering, bit counting and padding.
 * Subclasses need implement only the three abstract functions to create a
 * working hash.
 *
 * This class can do both little and big endian padding. This applies to
 * the 64-bit length counter that is appendend to each message to be hashed.
 *
 * @version $Revision: 1.5 $
 * @author  Jeroen C. van Gelderen (gelderen@cryptix.org)
 */
abstract class PaddingMD
extends MessageDigestSpi
{

// Constants
//...........................................................................

    private static final int BLOCKSIZE = 64;


// Instance variables
//...........................................................................

    /** Size (in bytes) of the digest */
    private final int hashSize;


    /** 64 byte buffer */
    private final byte[] buf;


    /** Buffer offset */
    private int bufOff;


    /** Number of bytes hashed 'till now. */
    private long byteCount;


    /** Mode */
    private final int mode;
    
    
    /*package*/ static final int 
        MODE_MD    = 0,
        MODE_SHA   = 1,
        MODE_TIGER = 2;


// Constructors
//...........................................................................


    /**
     * Construct a PaddingMD in MD-like, SHA-like or Tiger-like padding mode.
     *
     * The subclass must call this constructor, giving the length of it's hash
     * in bytes.
     *
     * @param hashSize  Length of the hash in bytes.
     */
    protected PaddingMD(int hashSize, int mode) {
        this.hashSize = hashSize;
        buf = new byte[BLOCKSIZE];
        bufOff    = 0;
        byteCount = 0;
        this.mode = mode;
    }


// Implementation
//...........................................................................

    protected int engineGetDigestLength() {
        return this.hashSize;
    }


    protected void engineUpdate(byte input) {

        //#ASSERT(this.bufOff < BLOCKSIZE);

        byteCount += 1;
        buf[bufOff++] = input;
        if( bufOff==BLOCKSIZE ) {
            coreUpdate(buf, 0);
            bufOff = 0;
        }

        //#ASSERT(this.bufOff < BLOCKSIZE);
    }


    protected void engineUpdate(byte[] input, int offset, int length) {

        byteCount += length;

        //#ASSERT(this.bufOff < BLOCKSIZE);

        int todo;
        while( length >= (todo = BLOCKSIZE - this.bufOff) ) {
            System.arraycopy(input, offset, this.buf, this.bufOff, todo);
            coreUpdate(this.buf, 0);
            length -= todo;
            offset += todo;
            this.bufOff = 0;
        }

        //#ASSERT(this.bufOff < BLOCKSIZE);

        System.arraycopy(input, offset, this.buf, this.bufOff, length);
        bufOff += length;
    }


    protected byte[] engineDigest() {
        try {
            byte[] tmp = new byte[hashSize];
            engineDigest(tmp, 0, hashSize);
            return tmp;
        } catch(DigestException e) {
            throw new RuntimeException(
                "PANIC, this should never happen. Giving up...");
        }
    }


    protected int engineDigest(byte[] buf, int offset, int len)
    throws DigestException 
    {
        if(len<hashSize)
            throw new DigestException();

        //#ASSERT(this.bufOff < BLOCKSIZE);

        this.buf[this.bufOff++] = (mode==MODE_TIGER) ? (byte)0x01 : (byte)0x80;

        if(this.bufOff > 56) {
            while(this.bufOff < 64)
                this.buf[this.bufOff++] = (byte)0x00;

            coreUpdate(this.buf, 0);
            this.bufOff = 0;
        }

        while(this.bufOff < 56) {
            this.buf[this.bufOff++] = (byte)0x00;
        }

        byteCount *= 8; // to bitcount
        if(mode==MODE_SHA) {
            // 64-bit length is appended in big endian order
            for(int i=56; i>=0; i-=8)
                this.buf[this.bufOff++] = (byte)(byteCount >>> (i) );
        } else {
            // 64-bit length is appended in little endian order
            for(int i=0; i<64; i+=8)
                this.buf[this.bufOff++] = (byte)(byteCount >>> (i) );
        }

        coreUpdate(this.buf, 0);
        coreDigest(buf, offset);

        engineReset();
        return hashSize;
    }


    protected void engineReset() {
        this.bufOff    = 0;
        this.byteCount = 0;
        coreReset();
    }


    public Object clone()
    throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }


// Delegated methods
//...........................................................................

    /**
     * Return the hash bytes in <code>buf</code>, starting at offset
     * <code>off</code>.
     *
     * The subclass is expected to write exactly <code>hashSize</code> bytes
     * in the given buffer. The buffer is guaranteed to be large enough.
     */
    protected abstract void coreDigest(byte[] buf, int off);


    /**
     * Reset the hash internal structures to initial state.
     */
    protected abstract void coreReset();


    /**
     * Update the internal state with a single block.
     *
     * <code>buf</code> contains a single block (64 bytes, 512 bits) of data,
     * starting at offset <code>off</code>.
     */
    protected abstract void coreUpdate(byte[] buf, int off);
}
