//
// vtNode and subclasses
//
// Encapsulates behavior for scene graph nodes
//

#include "vtlib/vtlib.h"

#include <sgl/sglPerspectiveCamera.hpp>
#include <sgl/sglOrthographicCamera.hpp>


void vtNode::SetEnabled(bool bOn)
{
//	m_pModel->SetActive(bOn);	// TODO
	vtEnabledBase::SetEnabled(bOn);
}

void vtNode::GetBoundBox(FBox3 &box)
{
	// TODO
//	FBox3 b;
//	m_pTransform->GetBound(&b);
//	box = d2v(b);
}

void vtNode::GetBoundSphere(FSphere &sphere)
{
	const sglSphereBoundf &s = m_pNode->getBound();
	s2v(s, sphere);
}

vtNode *vtNode::CreateClone()
{
	// TODO
	return new vtNode();
}

void vtNode::SetName2(const char *name)
{
	m_pNode->setName((char *)name);
}

const char *vtNode::GetName2()
{
	std::string foo1 = m_pNode->getName();
	return foo1.c_str();
}

void vtNode::SetSglNode(sglNode *n)
{
	m_pNode = n;
	if (n)
		n->setUserData((sglUserData *)this);
}

////////////////////////////////////////////////////

vtGroup::vtGroup(bool suppress) : vtNode(), vtGroupBase()
{
	if (suppress)
		m_pGroup = NULL;
	else
		m_pGroup = new sglGroup;
	SetSglNode(m_pGroup);
}

void vtGroup::AddChild(vtNode *pChild)
{
	// assume that this entity is actually a branch node
	m_pGroup->addChild(pChild->GetSglNode());
}

void vtGroup::RemoveChild(vtNode *pChild)
{
	// assume that this entity is actually a branch node
	m_pGroup->removeChild(pChild->GetSglNode());
}

vtNode *vtGroup::GetChild(int num)
{
	sglNode *pChild = m_pGroup->getChild(num);
	if (pChild)
	{
		vtNode *pNode = (vtNode *) pChild->getUserData();
		return pNode;
	}
	return NULL;
}

int vtGroup::GetNumChildren()
{
	return m_pGroup->getNumChildren();
}


////////////////////////////////////////////////////////

vtTransform::vtTransform() : vtTransformBase()
{
	m_pTrans = new sglTransformf;
}

vtTransform::~vtTransform()
{
	delete m_pTrans;
}

void vtTransform::Identity()
{
	sglMat4f xform;
	xform.buildIdentity();
	m_pTrans->setMatrix(xform);
}

FPoint3 vtTransform::GetTrans()
{
	const sglMat4f &xform = m_pTrans->getMatrix();
	FPoint3 p(xform[3][0], xform[3][1], xform[3][2]);
	return p;
}

void vtTransform::SetTrans(const FPoint3 &pos)
{
	sglMat4f xform = m_pTrans->getMatrix();
	xform[3][0] = pos.x;
	xform[3][1] = pos.y;
	xform[3][2] = pos.z;
	m_pTrans->setMatrix(xform);
}

void vtTransform::Translate1(const FPoint3 &pos)
{
	sglMat4f xform = m_pTrans->getMatrix();
	xform[3][0] += pos.x;
	xform[3][1] += pos.y;
	xform[3][2] += pos.z;
	m_pTrans->setMatrix(xform);
}

void vtTransform::TranslateLocal(const FPoint3 &pos)
{
	sglMat4f xform = m_pTrans->getMatrix();

	sglMat4f trans, result;
	trans.buildTranslation(pos.x, pos.y, pos.z);

	result = trans * xform;
	m_pTrans->setMatrix(result);
}

void vtTransform::Rotate2(const FPoint3 &axis, float angle)
{
	sglMat4f xform = m_pTrans->getMatrix();

	sglMat4f rot, result;
	rot.buildRotation(axis.x, axis.y, axis.z, angle);	// radians!  thank goodness

	result = xform * rot;
	m_pTrans->setMatrix(result);
}

void vtTransform::RotateLocal(const FPoint3 &axis, float angle)
{
	sglMat4f xform = m_pTrans->getMatrix();

	sglMat4f rot, result;
	rot.buildRotation(axis.x, axis.y, axis.z, angle);	// radians!  thank goodness

	result = rot * xform;
	m_pTrans->setMatrix(result);
}

void vtTransform::RotateParent(const FPoint3 &axis, float angle)
{
	sglMat4f xform = m_pTrans->getMatrix();

	sglVec3f trans(xform[3][0], xform[3][1], xform[3][2]);

	xform[3][0] = xform[3][1] = xform[3][2] = 0.0f;

	sglMat4f delta;
	delta.buildRotation(axis.x, axis.y, axis.z, -angle);
	xform *= delta;

	xform[3][0] = trans[0];
	xform[3][1] = trans[1];
	xform[3][2] = trans[2];
	m_pTrans->setMatrix(xform);
}

void vtTransform::Scale3(float x, float y, float z)
{
	sglMat4f xform = m_pTrans->getMatrix();

	sglMat4f scale, result;
	scale.buildScale(x, y, z);

	result = xform * scale;
	m_pTrans->setMatrix(result);
}

void vtTransform::SetTransform1(const FMatrix4 &mat)
{
	sglMat4f xform;
	ConvertMatrix4(mat, xform);

	m_pTrans->setMatrix(xform);
}

void vtTransform::GetTransform1(FMatrix4 &mat)
{
	sglMat4f xform = m_pTrans->getMatrix();

	ConvertMatrix4(xform, mat);
}

void vtTransform::PointTowards(const FPoint3 &point)
{
	sglMat4f xform = m_pTrans->getMatrix();

	sglVec3f trans(xform[3][0], xform[3][1], xform[3][2]);

	sglVec3f p;
	v2s(point, p);

	sglVec3f diff = p - trans;
	float dist = diff.length();

	float theta = atan2f(-diff[2], diff[0]) - PID2f;
	float phi = asinf(diff[1] / dist);

	xform.buildIdentity();

	sglMat4f delta;
	delta.buildRotation(0.0f, 1.0f, 0.0f, -theta);
	xform *= delta;

	delta.buildRotation(1.0f, 0.0f, 0.0f, -phi);
	xform *= delta;

	delta.buildTranslation(trans[0], trans[1], trans[2]);
	xform *= delta;

	m_pTrans->setMatrix(xform);
}


//////////////////////////////////////////

vtGeom::vtGeom() : vtGeomBase(), vtNode()
{
	m_pGeode = new sglGeode;
	SetSglNode(m_pGeode);
}

void vtGeom::AddMesh(vtMesh *pMesh, int iMatIdx)
{
	m_pGeode->addGeometry(pMesh->m_pGeoSet);
//	pMesh->m_pVtx->recalcBSphere();

	vtMaterial *pMat = GetMaterial(iMatIdx);
	if (pMat)
	{
		pMesh->m_pGeoSet->setState(0, pMat->m_state, NULL);
	}
	pMesh->SetMatIndex(iMatIdx);
}

void vtGeom::SetMeshMatIndex(vtMesh *pMesh, int iMatIdx)
{
	vtMaterial *pMat = GetMaterial(iMatIdx);
	if (pMat)
	{
		// TODO?
	}
	pMesh->SetMatIndex(iMatIdx);
}

void vtGeom::RemoveMesh(vtMesh *pMesh)
{
	m_pGeode->removeGeometry(pMesh->m_pGeoSet);
}

int vtGeom::GetNumMeshes()
{
	return m_pGeode->getNumGeometries();
}

vtMesh *vtGeom::GetMesh(int i)
{
	sglGeometry *geom = m_pGeode->getGeometry(i);
	if (geom)
	{
		vtMesh *pMesh = (vtMesh *) geom->getUserData();
		return pMesh;
	}
	return NULL;
}

//////////////////////////////////////////

void vtLOD::SetRanges(float *ranges, int nranges)
{
	for (int i = 0; i < nranges; i++)
		m_pLOD->setRange(i, ranges[i]);
}

void vtLOD::SetCenter(FPoint3 &center)
{
	sglVec3f center2;
	v2s(center, center2);
	m_pLOD->setCenter(center2);
}

//////////////////////////////////////////////

#if 0
void DynLeaf::recalcBSphere()
{
	emptyBSphere();
	bbox.empty();

	FBox3 box;
	m_pDynGeom->DoCalcBoundBox(box);
//	bbox.extend(vertices->get(i));
	sgVec3 corner1, corner2;
	v2s(box.min, corner1);
	v2s(box.max, corner2);
	bbox.extend(corner1);
	bbox.extend(corner2);

	extendBSphere(&bbox);
	dirtyBSphere();  /* Cause parents to redo their bspheres */
	bsphere_is_invalid = FALSE;
}

void DynLeaf::print(FILE *fd, char *indent)
{
	// do nothing
    fprintf(fd, "DynLeaf\n");
}

void DynLeaf::cull(sgFrustum *f, sgMat4 m, int test_needed)
{
	// get camera position
	FPoint3 pos(m[3][0], m[3][1], m[3][2]);

	// get window size
	IPoint2 window_size;
	window_size.x = 640;
	window_size.y = 400;

	// get FOV
	SGfloat hfov, vfov;
	f->getFOV(&hfov, &vfov);
	float fov = hfov;

	FMatrix4 mat;
	ConvertMatrix4(m, &mat);

	// setup the culling planes
	m_pDynGeom->CalcCullPlanes(f, &mat);

	m_pDynGeom->DoCull(pos, window_size, fov);

	// the following will trigger the "draw" step if in view
	ssgLeaf::cull(f, m, test_needed);
}

void DynLeaf::draw()
{
	m_pDynGeom->DoRender();
}
#endif

vtDynGeom::vtDynGeom() : vtGeom()
{
/*	m_pDynMesh = new vtDynMesh();
	m_pDynMesh->m_pDynGeom = this;

	m_pGeode->addGeoSet(m_pDynMesh); */
}

#if 0
///////////////////////////////////////
//
// Calculate the culling planes, for the camera this frame,
// in world coordinates.
//
// This way, each point or sphere doesn't have to be converted
// to Camera coordinates before testing against the view volume
//
// We don't bother to calculate the near and far planes, since they
// aren't important for dynamic terrain.
//
void vtDynGeom::CalcCullPlanes(sgFrustum *pFrust, const FMatrix4* trans)
{
	// TODO
	const Box3 vv = pCam->GetCullVol();

	FPoint3 c_loc (0, 0, 0);
	FPoint3 c_wvpr(0.0f, 0.0f, -vv.min.z);
	FPoint3 c_wvpn (0.0f, 0.0f, 1.0f);
	FPoint3 c_wvpu (0, 1.0f, 0.0f);

	// trans is the transform for world -> camera
	// invert it for camera -> world
	FMatrix4 camworld;
	camworld.Invert(*trans);

	// transform all the camera information into world frame
	FPoint3 loc, wvpr, wvpn, wvpu;
	camworld.Transform(c_loc, loc);
	camworld.TransformVector(c_wvpr, wvpr);
	camworld.TransformVector(c_wvpn, wvpn);
	camworld.TransformVector(c_wvpu, wvpu);

	FPoint3 uxn = wvpu.Cross(wvpn);

	FPoint3 p0, pa, tmp;
	FPoint3 lmv;

	float scll = vv.min.x; //cam.view.left;
	float sclr = vv.max.x; //cam.view.right;

	tmp.x = (wvpr.x + scll * uxn.x);		// vector to left edge
	tmp.y = (wvpr.y + scll * uxn.y);
	tmp.z = (wvpr.z + scll * uxn.z);
	pa =wvpu.Cross(tmp);
	pa.Normalize();
	cullPlanes[0].Set(loc, pa);	// left plane

	tmp.x = (wvpr.x + sclr * uxn.x);		// vector to right edge
	tmp.y = (wvpr.y + sclr * uxn.y);
	tmp.z = (wvpr.z + sclr * uxn.z);
	pa= tmp.Cross(wvpu);
	pa.Normalize();
	cullPlanes[1].Set(loc, pa);	// right plane

	tmp.x = (wvpr.x + vv.max.y * wvpu.x);		// vector to top edge
	tmp.y = (wvpr.y + vv.max.y * wvpu.y);
	tmp.z = (wvpr.z + vv.max.y * wvpu.z);
	pa= uxn.Cross(tmp);
	pa.Normalize( );
	cullPlanes[2].Set(loc, pa);	// top plane

	tmp.x = (wvpr.x + vv.min.y * wvpu.x);	// vector to bot edge
	tmp.y = (wvpr.y + vv.min.y * wvpu.y);
	tmp.z = (wvpr.z + vv.min.y * wvpu.z);
	pa= tmp.Cross(uxn);
	pa.Normalize();
	cullPlanes[3].Set(loc, pa);	// bottom plane
}
#endif


//
// Test a sphere against the view volume.
//
// Result: VT_AllVisible if entirely inside the volume,
//			VT_Visible if partly inside,
//			otherwise 0.
//
int vtDynGeom::IsVisible(const FSphere &sphere) const
{
	unsigned int vis = 0;

	// cull against standard frustum
	int i;
	for (i = 0; i < 4; i++)
	{
		float dist = cullPlanes[i].Distance(sphere.center);
		if (dist >= sphere.radius)
			return 0;
		if ((dist < 0) &&
			(dist <= sphere.radius))
			vis = (vis << 1) | 1;
	}

	// Notify renderer that object is entirely within standard frustum, so
	// no clipping is necessary.
	if (vis == 0x0F)
		return VT_AllVisible;
	return VT_Visible;
}


//
// Test a single point against the view volume.
//
//  Result: true if inside, false if outside.
//
bool vtDynGeom::IsVisible(const FPoint3& point) const
{
	unsigned int vis = 0;

	// cull against standard frustum
	int i;
	for (i = 0; i < 4; i++)
	{
		float dist = cullPlanes[i].Distance(point);
		if (dist > 0.0f)
			return false;
	}
	return true;
}


//
// Test a 3d triangle against the view volume.
//
// Result: VT_AllVisible if entirely inside the volume,
//			VT_Visible if partly intersecting,
//			otherwise 0.
//
int vtDynGeom::IsVisible(const FPoint3& point0,
							const FPoint3& point1,
							const FPoint3& point2,
							const float fTolerance) const
{
	unsigned int outcode0 = 0, outcode1 = 0, outcode2 = 0;
	register float dist;

	// cull against standard frustum
	int i;
	for (i = 0; i < 4; i++)
	{
		dist = cullPlanes[i].Distance(point0);
		if (dist > fTolerance)
			outcode0 |= (1 << i);

		dist = cullPlanes[i].Distance(point1);
		if (dist > fTolerance)
			outcode1 |= (1 << i);

		dist = cullPlanes[i].Distance(point2);
		if (dist > fTolerance)
			outcode2 |= (1 << i);
	}
	if (outcode0 == 0 && outcode1 == 0 && outcode2 == 0)
		return VT_AllVisible;
	if (outcode0 != 0 && outcode1 != 0 && outcode2 != 0)
	{
		if ((outcode0 & outcode1 & outcode2) != 0)
			return 0;
		else
		{
			// tricky case - just be conservative and assume some intersection
			return VT_Visible;
		}
	}
	// not all in, and not all out
	return VT_Visible;
}

int vtDynGeom::IsVisible(FPoint3 point, float radius)
{
	unsigned int incode = 0;

	// cull against standard frustum
	for (int i = 0; i < 4; i++)
	{
		float dist = cullPlanes[i].Distance(point);
		if (dist > radius)
			return 0;			// entirely outside this plane
		if (dist < -radius)
			incode |= (1 << i);	// entirely inside this plane
	}
	if (incode == 0x0f)
		return VT_AllVisible;	// entirely inside all planes
	else
		return VT_Visible;
}


//////////////////////////////////////////////

vtCamera::vtCamera() : vtTransform()
{
	m_psglPerspCamera = new sglPerspectiveCamera();
	m_psglOrthoCamera = NULL;
	m_psglCamera = m_psglPerspCamera;

	m_psglViewPlatform = new sglViewPlatform(*m_psglCamera);
	m_pTrans->addChild(m_psglViewPlatform);
}

void vtCamera::SetHither(float f)
{
	double fovy, aspect, near_clip, far_clip;
	m_psglPerspCamera->getFOV(fovy, aspect, near_clip, far_clip);
	near_clip = f;
	m_psglPerspCamera->setFOV(fovy, aspect, near_clip, far_clip);
}

float vtCamera::GetHither()
{
	double fovy, aspect, near_clip, far_clip;
	m_psglPerspCamera->getFOV(fovy, aspect, near_clip, far_clip);
	return near_clip;
}

void vtCamera::SetYon(float f)
{
	double fovy, aspect, near_clip, far_clip;
	m_psglPerspCamera->getFOV(fovy, aspect, near_clip, far_clip);
	far_clip = f;
	m_psglPerspCamera->setFOV(fovy, aspect, near_clip, far_clip);
}

float vtCamera::GetYon()
{
	double fovy, aspect, near_clip, far_clip;
	m_psglPerspCamera->getFOV(fovy, aspect, near_clip, far_clip);
	return near_clip;
}

void vtCamera::SetFOV(float f)
{
	double fovy, aspect, near_clip, far_clip;
	m_psglPerspCamera->getFOV(fovy, aspect, near_clip, far_clip);
	fovy = f * aspect;
	m_psglPerspCamera->setFOV(fovy, aspect, near_clip, far_clip);
}

float vtCamera::GetFOV()
{
	double fovy, aspect, near_clip, far_clip;
	m_psglPerspCamera->getFOV(fovy, aspect, near_clip, far_clip);
	return fovy / aspect;
}

void vtCamera::ZoomToSphere(const FSphere &sphere)
{
	Identity();
	Translate1(sphere.center);
	Translate1(FPoint3(0.0f, 0.0f, sphere.radius));
}

void vtCamera::SetOrtho(float fWidth)
{
	m_psglOrthoCamera = new sglOrthographicCamera();
	m_psglCamera = m_psglOrthoCamera;
	m_psglViewPlatform->setCamera(*m_psglCamera);
}

//////////////////////////////////////////

vtLight::vtLight()
{
	m_pLight = new sglDirectionalLight;
	m_pLight->setDiffuse(1., 1., 1., 1.);
	m_pLight->setDirection(sglVec3f(0., 0., 1.));
	SetSglNode(m_pLight);
}

void vtLight::SetColor2(RGBf color)
{
	sglVec4f color2;
	v2s(color, color2);
	m_pLight->setDiffuse(color2);
}

void vtLight::SetAmbient2(RGBf &color)
{
	sglVec4f color2;
	v2s(color, color2);
	m_pLight->setAmbient(color2);
}

vtMovLight::vtMovLight(vtLight *pContained) : vtTransform()
{
	m_pLight = pContained;
	AddChild(m_pLight);
}


////////////////////////

// Sprite?
void vtSprite::SetText(const char *msg)
{
}

