//
// LodGrid.cpp
//
// Copyright (c) 2001 Virtual Terrain Project
// Free for all uses, see license.txt for details.
//

#include "vtlib/vtlib.h"
#include "LocalProjection.h"
#include "LodGrid.h"

#define index(a,b) ((a*m_dim)+b)

vtLodGrid::vtLodGrid(FPoint3 &origin, FPoint3 &size,
				 int iDimension, float fLODDistance, vtHeightField *pHF)
{
	m_origin = origin;
	m_size = size;
	m_dim = iDimension;
	m_fLODDistance = fLODDistance;
	m_step = m_size / m_dim;

	// wrap with an array of simple LOD nodes
	m_pCells = (vtLOD **)malloc(m_dim * m_dim * sizeof(vtLOD *));

	int a, b;
	for (a = 0; a < m_dim; a++)
	{
		for (b = 0; b < m_dim; b++)
		{
			m_pCells[index(a,b)] = NULL;
		}
	}
	m_pHeightField = pHF;
}

vtLodGrid::~vtLodGrid()
{
#if 1
	int a, b;
	for (a = 0; a < m_dim; a++)
		for (b = 0; b < m_dim; b++)
		{
			if (m_pCells[index(a,b)] != NULL)
				delete m_pCells[index(a,b)];
		}
#endif
	free(m_pCells);
	m_pCells = NULL;
}


void vtLodGrid::AllocateCell(int a, int b)
{
	int i = index(a,b);

	m_pCells[i] = new vtLOD();

	float ranges[2];
	ranges[0] = 0.0f;
	ranges[1] = m_fLODDistance;
	m_pCells[i]->SetRanges(ranges, 2);

	// determine LOD center
	FPoint3 lod_center;
	lod_center.x = m_origin.x + ((m_size.x / m_dim) * (a + 0.5f));
	lod_center.y = m_origin.y + (m_size.y / 2.0f);
	lod_center.z = m_origin.z + ((m_size.z / m_dim) * (b + 0.5f));
	if (m_pHeightField)
		m_pHeightField->FindAltitudeAtPoint(lod_center, lod_center.y);

	m_pCells[i]->SetCenter(lod_center);
	m_pCells[i]->AddChild(new vtGroup());

	AddChild(m_pCells[i]);
}

void vtLodGrid::DetermineCell(const FPoint3 &pos, int &a, int &b)
{
	a = (int) ((pos.x - m_origin.x) / m_step.x);
	b = (int) ((pos.z - m_origin.z) / m_step.z);
}

void vtLodGrid::AppendToGrid(vtTransform *pTNode)
{
	int a, b;

	DetermineCell(pTNode->GetTrans(), a, b);
	if (a < 0 || a >= m_dim || b < 0 || b >= m_dim)
		return;

	int i = index(a, b);
	if (!m_pCells[i])
		AllocateCell(a, b);

	vtGroup *pGroup = (vtGroup *)m_pCells[i]->GetChild(0);
	pGroup->AddChild(pTNode);
}

void vtLodGrid::AppendToGrid(vtGeom *pGNode)
{
	int a, b;

	FSphere sph;
	pGNode->GetBoundSphere(sph);

	DetermineCell(sph.center, a, b);
	if (a < 0 || a >= m_dim || b < 0 || b >= m_dim)
		return;

	int i = index(a, b);
	if (!m_pCells[i])
		AllocateCell(a, b);

//	m_pCells[i]->AddChild(pGNode);
	vtGroup *pGroup = (vtGroup *)m_pCells[i]->GetChild(0);
	pGroup->AddChild(pGNode);
}

