
/**************************************************************************/
/*** BEGINNING OF CODE Triangulate.cpp ***/
/**************************************************************************/

#include "Triangulate.h"

static const float EPSILON=0.0000000001f;

float Triangulate_f::Area(const FLine2 &contour)
{
	int n = contour.GetSize();

	float A=0.0f;

	for (int p=n-1,q=0; q<n; p=q++)
	{
		A+= contour[p].x*contour[q].y - contour[q].x*contour[p].y;
	}
	return A*0.5f;
}

/*
 InsideTriangle decides if a point P is Inside of the triangle
 defined by A, B, C.
*/
bool Triangulate_f::InsideTriangle(float Ax, float Ay,
						float Bx, float By,
						float Cx, float Cy,
						float Px, float Py)
{
	float ax, ay, bx, by, cx, cy, apx, apy, bpx, bpy, cpx, cpy;
	float cCROSSap, bCROSScp, aCROSSbp;

	ax = Cx - Bx;  ay = Cy - By;
	bx = Ax - Cx;  by = Ay - Cy;
	cx = Bx - Ax;  cy = By - Ay;
	apx= Px - Ax;  apy= Py - Ay;
	bpx= Px - Bx;  bpy= Py - By;
	cpx= Px - Cx;  cpy= Py - Cy;

	aCROSSbp = ax*bpy - ay*bpx;
	cCROSSap = cx*apy - cy*apx;
	bCROSScp = bx*cpy - by*cpx;

	return ((aCROSSbp >= 0.0f) && (bCROSScp >= 0.0f) && (cCROSSap >= 0.0f));
};

bool Triangulate_f::Snip(const FLine2 &contour,int u,int v,int w,int n,int *V)
{
	int p;
	float Ax, Ay, Bx, By, Cx, Cy, Px, Py;

	Ax = contour[V[u]].x;
	Ay = contour[V[u]].y;

	Bx = contour[V[v]].x;
	By = contour[V[v]].y;

	Cx = contour[V[w]].x;
	Cy = contour[V[w]].y;

	if ( EPSILON > (((Bx-Ax)*(Cy-Ay)) - ((By-Ay)*(Cx-Ax))) ) return false;

	for (p=0;p<n;p++)
	{
		if( (p == u) || (p == v) || (p == w) ) continue;
		Px = contour[V[p]].x;
		Py = contour[V[p]].y;
		if (InsideTriangle(Ax,Ay,Bx,By,Cx,Cy,Px,Py)) return false;
	}

	return true;
}

bool Triangulate_f::Process(const FLine2 &contour,FLine2 &result)
{
	/* allocate and initialize list of Vertices in polygon */

	int n = contour.GetSize();
	if ( n < 3 ) return false;

	int *V = new int[n];

	/* we want a counter-clockwise polygon in V */

	if ( 0.0f < Area(contour) )
		for (int v=0; v<n; v++) V[v] = v;
	else
		for (int v=0; v<n; v++) V[v] = (n-1)-v;

	int nv = n;

	/*  remove nv-2 Vertices, creating 1 triangle every time */
	int count = 2*nv;   /* error detection */

	for (int m=0, v=nv-1; nv>2; )
	{
		/* if we loop, it is probably a non-simple polygon */
		if (0 >= (count--))
		{
			//** Triangulate: ERROR - probable bad polygon!
			return false;
		}

		/* three consecutive vertices in current polygon, <u,v,w> */
		int u = v  ; if (nv <= u) u = 0;     /* previous */
		v = u+1; if (nv <= v) v = 0;     /* new v    */
		int w = v+1; if (nv <= w) w = 0;     /* next     */

		if ( Snip(contour,u,v,w,nv,V) )
		{
			int a,b,c,s,t;

			/* true names of the vertices */
			a = V[u]; b = V[v]; c = V[w];

			/* output Triangle */
			result.Append( contour[a] );
			result.Append( contour[b] );
			result.Append( contour[c] );

			m++;

			/* remove v from remaining polygon */
			for (s=v,t=v+1;t<nv;s++,t++) V[s] = V[t]; nv--;

			/* resest error detection counter */
			count = 2*nv;
		}
	}
	delete V;
	return true;
}



//////////////////////////////////////////////////////////////////////////
// Double-precision version
//

double Triangulate_d::Area(const DLine2 &contour)
{
	int n = contour.GetSize();

	double A=0.0f;

	for (int p=n-1,q=0; q<n; p=q++)
	{
		A+= contour[p].x*contour[q].y - contour[q].x*contour[p].y;
	}
	return A*0.5f;
}

/*
 InsideTriangle decides if a point P is Inside of the triangle
 defined by A, B, C.
*/
bool Triangulate_d::InsideTriangle(double Ax, double Ay,
						double Bx, double By,
						double Cx, double Cy,
						double Px, double Py)
{
	double ax, ay, bx, by, cx, cy, apx, apy, bpx, bpy, cpx, cpy;
	double cCROSSap, bCROSScp, aCROSSbp;

	ax = Cx - Bx;  ay = Cy - By;
	bx = Ax - Cx;  by = Ay - Cy;
	cx = Bx - Ax;  cy = By - Ay;
	apx= Px - Ax;  apy= Py - Ay;
	bpx= Px - Bx;  bpy= Py - By;
	cpx= Px - Cx;  cpy= Py - Cy;

	aCROSSbp = ax*bpy - ay*bpx;
	cCROSSap = cx*apy - cy*apx;
	bCROSScp = bx*cpy - by*cpx;

	return ((aCROSSbp >= 0.0f) && (bCROSScp >= 0.0f) && (cCROSSap >= 0.0f));
};

bool Triangulate_d::Snip(const DLine2 &contour,int u,int v,int w,int n,int *V)
{
	int p;
	double Ax, Ay, Bx, By, Cx, Cy, Px, Py;

	Ax = contour[V[u]].x;
	Ay = contour[V[u]].y;

	Bx = contour[V[v]].x;
	By = contour[V[v]].y;

	Cx = contour[V[w]].x;
	Cy = contour[V[w]].y;

	if ( EPSILON > (((Bx-Ax)*(Cy-Ay)) - ((By-Ay)*(Cx-Ax))) ) return false;

	for (p=0;p<n;p++)
	{
		if( (p == u) || (p == v) || (p == w) ) continue;
		Px = contour[V[p]].x;
		Py = contour[V[p]].y;
		if (InsideTriangle(Ax,Ay,Bx,By,Cx,Cy,Px,Py)) return false;
	}

	return true;
}

bool Triangulate_d::Process(const DLine2 &contour,DLine2 &result)
{
	/* allocate and initialize list of Vertices in polygon */

	int n = contour.GetSize();
	if ( n < 3 ) return false;

	int *V = new int[n];

	/* we want a counter-clockwise polygon in V */

	if ( 0.0f < Area(contour) )
		for (int v=0; v<n; v++) V[v] = v;
	else
		for (int v=0; v<n; v++) V[v] = (n-1)-v;

	int nv = n;

	/*  remove nv-2 Vertices, creating 1 triangle every time */
	int count = 2*nv;   /* error detection */

	for (int m=0, v=nv-1; nv>2; )
	{
		/* if we loop, it is probably a non-simple polygon */
		if (0 >= (count--))
		{
			//** Triangulate: ERROR - probable bad polygon!
			return false;
		}

		/* three consecutive vertices in current polygon, <u,v,w> */
		int u = v  ; if (nv <= u) u = 0;     /* previous */
		v = u+1; if (nv <= v) v = 0;     /* new v    */
		int w = v+1; if (nv <= w) w = 0;     /* next     */

		if ( Snip(contour,u,v,w,nv,V) )
		{
			int a,b,c,s,t;

			/* true names of the vertices */
			a = V[u]; b = V[v]; c = V[w];

			/* output Triangle */
			result.Append( contour[a] );
			result.Append( contour[b] );
			result.Append( contour[c] );

			m++;

			/* remove v from remaining polygon */
			for (s=v,t=v+1;t<nv;s++,t++) V[s] = V[t]; nv--;

			/* resest error detection counter */
			count = 2*nv;
		}
	}
	delete V;
	return true;
}

#if 0

/************************************************************************/
/*** END OF CODE SECTION TRIANGULATE.CPP BEGINNING OF TEST.CPP A SMALL **/
/*** TEST APPLICATION TO DEMONSTRATE THE USAGE OF THE TRIANGULATOR     **/
/************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>


#include "triangulate.h"

void main(int argc,char **argv)
{
	// Small test application demonstrating the usage of the triangulate
	// class.

	// Create a pretty complicated little contour.
	DLine2 a;
	a.Append( DPoint2(0,6));
	a.Append( DPoint2(0,0));
	a.Append( DPoint2(3,0));
	a.Append( DPoint2(4,1));
	a.Append( DPoint2(6,1));
	a.Append( DPoint2(8,0));
	a.Append( DPoint2(12,0));
	a.Append( DPoint2(13,2));
	a.Append( DPoint2(8,2));
	a.Append( DPoint2(8,4));
	a.Append( DPoint2(11,4));
	a.Append( DPoint2(11,6));
	a.Append( DPoint2(6,6));
	a.Append( DPoint2(4,3));
	a.Append( DPoint2(2,6));

	// allocate a polyline to hold the answer.
	DLine2 result;

	//  Invoke the triangulator to triangulate this polygon.
	Triangulate_d::Process(a,result);

	// print out the results.
	int tcount = result.GetSize()/3;

	for (int i=0; i<tcount; i++)
	{
		const DPoint2 &p1 = result[i*3+0];
		const DPoint2 &p2 = result[i*3+1];
		const DPoint2 &p3 = result[i*3+2];
		printf("Triangle %d => (%0.0f,%0.0f) (%0.0f,%0.0f) (%0.0f,%0.0f)\n",
			i+1,
			p1.x,p1.y,p2.x,p2.y,p3.x,p3.y);
	}
}

#endif
