//
// Projections.cpp
//
// Copyright (c) 2001 Virtual Terrain Project
// Free for all uses, see license.txt for details.
//
// Parts of the code are derived from public-domain USGS software.
//

#include "Projections.h"
#include "StatePlane.h"


/////////////////////////////////////////////////////////////////////////////
// Implementation of class vtProjection
//

vtProjection::vtProjection()
{
	m_Datum = NO_DATUM;
}

bool vtProjection::operator==(vtProjection &ref)
{
	// Work around problem in IsSame, by detecting this type of difference
	if( IsProjected() != ref.IsProjected() )
		return false;
	return (IsSame( (OGRSpatialReference *) &ref ) != 0);
}

bool  vtProjection::IsUTM()
{
	const char *proj_string = GetAttrValue("PROJECTION");
	return (proj_string &&
		!strcmp(proj_string, "Transverse_Mercator") &&
		GetUTMZone() > 0);
}

void  vtProjection::SetUTMZone(int iZone)
{
    OGRErr err = OGRSpatialReference::SetUTM( iZone );
}

int	  vtProjection::GetUTMZone()
{
	return OGRSpatialReference::GetUTMZone();
}

void  vtProjection::SetDatum(DATUM datum)
{
	// Convert the DATUM enumeration to a Datum string
    OGR_SRSNode *dnode = GetAttrNode("DATUM");
	if (!dnode)
		return;
	switch (datum)
	{
	case NAD27:		dnode->GetChild(0)->SetValue("North_American_Datum_1927"); break;
	case NAD83:		dnode->GetChild(0)->SetValue("North_American_Datum_1983"); break;
	case WGS_72:	dnode->GetChild(0)->SetValue("WGS_1984"); break;
	case OLD_HAWAIIAN_MEAN:	dnode->GetChild(0)->SetValue("Old_Hawaiian"); break;
	default:
	case WGS_84:	dnode->GetChild(0)->SetValue("WGS_1972"); break;
	}
}

DATUM vtProjection::GetDatum()
{
	// Convert new DATUM string to old Datum enum
	const char *datum_string = GetAttrValue("DATUM");
	if (datum_string)
	{
		if (!strcmp(datum_string, "North_American_Datum_1927"))
			return NAD27;
		if (!strcmp(datum_string, "North_American_Datum_1983"))
			return NAD83;
		if (!strcmp(datum_string, "WGS_1984"))
			return WGS_84;
		if (!strcmp(datum_string, "WGS_1972"))
			return WGS_72;
		if (!strcmp(datum_string, "Old_Hawaiian"))
			return OLD_HAWAIIAN_MEAN;
	}
	return WGS_84;	// default
}


/**
  * Assignment operator.
  */
vtProjection &vtProjection::operator=(vtProjection &ref)
{
	// copy new projection
	if (ref.GetRoot() != NULL)
	{
		OGRSpatialReference *ref_copy = ref.Clone();
		(*(OGRSpatialReference *)this) = *ref_copy;
	}

	// copy old fields
	m_Datum = ref.m_Datum;
	return *this;
}

void vtProjection::SetSpatialReference(OGRSpatialReference *pRef)
{
	*((OGRSpatialReference *)this) = *(pRef->Clone());
}

const char *vtProjection::GetProjectionName()
{
	const char *proj_string = GetAttrValue("PROJECTION");
	if (!proj_string)
		return "Geographic";
	else
		return proj_string;
}

/** Return a very short string describing the type of projection.
 * \par
 * Possible values are "Geo", "UTM", "TM", "Albers", "LCC", "Other", or "Unknown"
 */
const char *vtProjection::GetProjectionNameShort()
{
	if (IsGeographic())
		return "Geo";
	const char *proj_string = GetAttrValue("PROJECTION");
	if (!proj_string)
		return "Unknown";
	if (!strcmp(proj_string, "Transverse_Mercator"))
	{
		if (GetUTMZone() > 0)
			return "UTM";
		else
			return "TM";
	}
	if (proj_string && !strcmp(proj_string, "Albers_Conic_Equal_Area"))
		return "Albers";
	if (proj_string && !strncmp(proj_string, "Lambert_Conformal_Conic", 23))
		return "LCC";
	return "Other";
}

void vtProjection::SetProjectionSimple(int proj_type, int iUTMZone, DATUM eDatum)
{
	switch (eDatum)
	{
	case NAD27:	 SetWellKnownGeogCS( "NAD27" ); break;
	case NAD83:	 SetWellKnownGeogCS( "NAD83" ); break;
	case WGS_72: SetWellKnownGeogCS( "WGS72" ); break;
	case WGS_84: SetWellKnownGeogCS( "WGS84" ); break;
	// We may need to support more datums here!
	default:	 SetWellKnownGeogCS( "WGS84" ); break;
	}
	if (proj_type == 1)	// UTM
	{
		// Must we assume Northern Hemisphere?  Revisit if needed.
		SetUTM( iUTMZone, TRUE );
	}
	switch (eDatum)
	{
	case OLD_HAWAIIAN_MEAN:
		SetDatum(eDatum);
	}
}


/////////////////////////////////////////////////////////////////////////////
// Helper functions


/** Given a non-geographic projection, produce a geographic projection which
  * has the same datum/ellipsoid values.
  */
void CreateSimilarGeographicProjection(vtProjection &source, vtProjection &geo)
{
	geo.SetWellKnownGeogCS("WGS84");

	// We can't convert datum yet.  Force assumption that source datum
	// is the same as the destination.
	const char *datum_string = source.GetAttrValue("DATUM");
	const char *ellipsoid_string = source.GetAttrValue("SPHEROID");
	geo.SetGeogCS("WGS84", datum_string, ellipsoid_string,
		source.GetSemiMajor(), source.GetInvFlattening());
}


//
// provide access to the State Plane Table
//
StatePlaneInfo *GetStatePlaneTable()
{
	return g_StatePlaneInfo;
}

int GetNumStatePlanes()
{
	return sizeof(g_StatePlaneInfo) /  sizeof(StatePlaneInfo);
}

//
// Datum strings
//
const char *datumToString(DATUM d)
{
    switch ( d )
    {
    case ADINDAN:
        return "ADINDAN";
    case ARC1950:
        return "ARC1950";
    case ARC1960:
        return "ARC1960";
    case AUSTRALIAN_GEODETIC_1966:
        return "AUSTRALIAN GEODETIC 1966";
    case AUSTRALIAN_GEODETIC_1984:
        return "AUSTRALIAN GEODETIC 1984";
    case CAMP_AREA_ASTRO:
        return "CAMP AREA ASTRO";
    case CAPE:
        return "CAPE";
    case EUROPEAN_DATUM_1950:
        return "EUROPEAN DATUM 1950";
    case EUROPEAN_DATUM_1979:
        return "EUROPEAN DATUM 1979";
    case GEODETIC_DATUM_1949:
        return "GEODETIC DATUM 1949";
    case HONG_KONG_1963:
        return "HONG KONG 1963";
    case HU_TZU_SHAN:
        return "HU TZU SHAN";
    case INDIAN:
        return "INDIAN";
    case NAD27:
        return "NAD27";
    case NAD83:
        return "NAD83";
    case OLD_HAWAIIAN_MEAN:
        return "OLD HAWAIIAN MEAN";
    case OMAN:
        return "OMAN";
    case ORDNANCE_SURVEY_1936:
        return "ORDNANCE SURVEY 1936";
    case PULKOVO_1942:
        return "PULKOVO 1942";
    case PROVISIONAL_S_AMERICAN_1956:
        return "PROVISIONAL SOUTH AMERICAN 1956";
    case TOKYO:
        return "TOKYO";
    case WGS_72:
        return "WGS72";
    case WGS_84:
        return "WGS84";
    case NO_DATUM:
        return "NO DATUM";
    default:
        return "Unknown Datum";
    }
}

