//
// The vtBuildingArray class is an array of Building objects.
//
// It supports operations including loading and saving to a file
// and picking of building elements.
//
// Copyright (c) 2001 Virtual Terrain Project
// Free for all uses, see license.txt for details.
//

#include <stdlib.h>
#include <string.h>
#include "\APIs\gdal\frmts/shapelib/shapefil.h"
#include "BuildingArray.h"

void vtBuildingArray::DestructItems(int first, int last)
{
	for (int i = first; i <= last; i++)
		delete GetAt(i);
}

bool vtBuildingArray::ReadBCF(const char* pathname)
{
	FILE* fp;
	if ( (fp = fopen(pathname, "rb")) == NULL )
		return false;

	char buf[4];
	fread(buf, 3, 1, fp);
	if (strncmp(buf, "bcf", 3))
	{
		// not current bcf, try reading old format
		rewind(fp);
		return ReadBCF_Old(fp);
	}

	float version;
	fscanf(fp, "%f\n", &version);

	if (version < BCFVERSION_SUPPORTED)
	{
		// too old, unsupported version
		return false;
	}

	int zone = 1;
	if (version == 1.2f)
	{
		fscanf(fp, "utm_zone %d\n", &zone);
	}
	m_proj.SetUTM(true, zone);

	int i, j, count;
	DPoint2 p;
	int points;
	char key[80];
	RGBi color;

	fscanf(fp, "buildings %d\n", &count);
	for (i = 0; i < count; i++)	//for each building
	{
		vtBuilding *bld = NewBuilding();

		BldShape type;
		fscanf(fp, "type %d\n", &type);
		bld->SetShape((BldShape) type);

		int stories = 1;
		while (1)
		{
			long start = ftell(fp);

			int result = fscanf(fp, "%s ", key);
			if (result == -1)
				break;

			if (!strcmp(key, "type"))
			{
				fseek(fp, start, SEEK_SET);
				break;
			}
			if (!strcmp(key, "loc"))
			{
				DPoint2 loc;
				fscanf(fp, "%lf %lf\n", &loc.x, &loc.y);
				bld->SetLocation(loc);
			}
			else if (!strcmp(key, "rot"))
			{
				float rot;
				fscanf(fp, "%f\n", &rot);
				bld->SetRotation(rot);
			}
			else if (!strcmp(key, "stories"))
			{
				fscanf(fp, "%d\n", &stories);

				if (stories < 1 || stories > 10) stories = 1;	// TEMPORARY FIX!
				bld->SetStories(stories);
			}
			else if (!strcmp(key, "color"))
			{
				fscanf(fp, "%hd %hd %hd\n", &color.r, &color.g, &color.b);
				bld->SetColor(BLD_BASIC, color);
			}
			else if (!strcmp(key, "color_roof"))
			{
				fscanf(fp, "%hd %hd %hd\n", &color.r, &color.g, &color.b);
				bld->SetColor(BLD_ROOF, color);
			}
			else if (!strcmp(key, "color_trim"))
			{
				fscanf(fp, "%hd %hd %hd\n", &color.r, &color.g, &color.b);
				bld->SetColor(BLD_MOULDING, color);
			}
			else if (!strcmp(key, "size"))
			{
				float w, d;
				fscanf(fp, "%f %f\n", &w, &d);
				bld->SetRectangle(w, d);
			}
			else if (!strcmp(key, "radius"))
			{
				float rad;
				fscanf(fp, "%f\n", &rad);
				bld->SetRadius(rad);
			}
			else if (!strcmp(key, "footprint"))
			{
				DLine2 dl;
				fscanf(fp, "%d", &points);
				dl.SetSize(points);

				for (j = 0; j < points; j++)
				{
					fscanf(fp, " %lf %lf", &p.x, &p.y);
					dl.SetAt(j, p);
				}
				fscanf(fp, "\n");
				bld->SetFootprint(dl);
			}
			else if (!strcmp(key, "trim"))
			{
				int trim;
				fscanf(fp, "%d\n", &trim);
				bld->m_bMoulding = (trim != 0);
			}
			else if (!strcmp(key, "elev"))
			{
				int elev;
				fscanf(fp, "%d\n", &elev);
				bld->m_bElevated = (elev != 0);
			}
			else if (!strcmp(key, "roof_type"))
			{
				int rt;
				fscanf(fp, "%d\n", &rt);
				bld->m_RoofType = (RoofType) rt;
			}
		}
		AddBuilding(bld);
	}
	fclose(fp);
	return true;
}

bool vtBuildingArray::ReadBCF_Old(FILE *fp)
{
	int ncoords;
	int num = fscanf(fp, "%d\n", &ncoords);
	if (num != 1)
		return false;

	DPoint2 point;
	for (int i = 0; i < ncoords; i++)
	{
		fscanf(fp, "%lf %lf\n", &point.x, &point.y);
		vtBuilding *bld = NewBuilding();
		bld->SetLocation(point);
		Append(bld);
	}

	fclose(fp);
	return true;
}

bool vtBuildingArray::ReadSHP(const char* pathname)
{
	SHPHandle hSHP = SHPOpen(pathname, "rb");
	if (hSHP == NULL)
	{
		// try to read the old format
		return ReadBCF(pathname);
	}

    int		nEntities, nShapeType;
    double 	adfMinBound[4], adfMaxBound[4];
	DPoint2 point;

	SHPGetInfo(hSHP, &nEntities, &nShapeType, adfMinBound, adfMaxBound);
	if (nShapeType != SHPT_POINT)
		return false;

	for (int i = 0; i < nEntities; i++)
	{
		SHPObject *psShape = SHPReadObject(hSHP, i);
		point.x = psShape->padfX[0];
		point.y = psShape->padfY[0];
		vtBuilding *bld = NewBuilding();
		bld->SetLocation(point);
		Append(bld);
		SHPDestroyObject(psShape);
	}
	SHPClose(hSHP);
	return true;
}

bool vtBuildingArray::WriteBCF(const char* pathname)
{
	FILE *fp = fopen(pathname, "wb");
	if (!fp) return false;

	int count = GetSize();
	BldShape type;
	DPoint2 loc;
	vtBuilding *bld;
	RGBi color;
	int i, j, points;

	fprintf(fp, "bcf%1.1f\n", BCFVERSION_CURRENT);
	fprintf(fp, "utm_zone %d\n", m_proj.GetUTMZone());
	fprintf(fp, "buildings %d\n", count);
	for (i = 0; i < count; i++)	//for each coordinate
	{
		bld = GetAt(i);
		type = bld->GetShape();

		fprintf(fp, "type %d\n", type);
		loc = bld->GetLocation();
		fprintf(fp, "\tloc %lf %lf\n", loc.x, loc.y);
		fprintf(fp, "\tstories %d\n", bld->GetStories());

		color = bld->GetColor(BLD_BASIC);
		fprintf(fp, "\tcolor %d %d %d\n", color.r, color.g, color.b);
		color = bld->GetColor(BLD_ROOF);
		fprintf(fp, "\tcolor_roof %d %d %d\n", color.r, color.g, color.b);

		if (bld->m_bMoulding)
		{
			fprintf(fp, "\ttrim 1\n");
			color = bld->GetColor(BLD_MOULDING);
			fprintf(fp, "\tcolor_trim %d %d %d\n", color.r, color.g, color.b);
		}

		if (bld->m_bElevated)
		{
			fprintf(fp, "\telev 1\n");
		}

		// don't write
		if (type == SHAPE_RECTANGLE)
		{
			float w, d, rot;
			bld->GetRectangle(w, d);
			bld->GetRotation(rot);
			fprintf(fp, "\tsize %f %f\n", w, d);
			if (rot != -1.0f)
				fprintf(fp, "\trot %f\n", rot);
			fprintf(fp, "\troof_type %d\n", bld->m_RoofType);
		}
		if (type == SHAPE_CIRCLE)
		{
			float rad = bld->GetRadius();
			fprintf(fp, "\tradius %f\n", rad);
		}
		if (type == SHAPE_POLY)
		{
			DLine2 &dl = bld->GetFootprint();
			points = dl.GetSize();
			fprintf(fp, "\tfootprint %d", points);
			for (j = 0; j < points; j++)
			{
				DPoint2 p = dl.GetAt(j);
				fprintf(fp, " %lf %lf", p.x, p.y);
			}
			fprintf(fp, "\n");
		}
	}
	fclose(fp);
	return true;
}

// int nSHPType, int nShapeId, int nParts,
//	int * panPartStart, int * panPartType,
//	int nVertices, double * padfX, double * padfY,
//	double * padfZ, double * padfM )

bool vtBuildingArray::WriteSHP(const char* pathname)
{
	char *ext = strrchr(pathname, '.');
	if (!strcmp(ext, ".bcf"))
		return WriteBCF(pathname);

    SHPHandle hSHP = SHPCreate ( pathname, SHPT_POINT );
    if (!hSHP)
		return false;

	int count = GetSize();
	DPoint2 temp;
	SHPObject *obj;
	for (int i = 0; i < count; i++)	//for each coordinate
	{
		vtBuilding *ptr = GetAt(i);
		temp = ptr->GetLocation();
		obj = SHPCreateSimpleObject(SHPT_POINT, 1, &temp.x, &temp.y, NULL);
		SHPWriteObject(hSHP, -1, obj);
		SHPDestroyObject(obj);
	}
	SHPClose(hSHP);
	return true;
}

int vtBuildingArray::IsBuildingThere(DPoint2 point, double error)
{
	if (IsEmpty())
		return -1;

	for (int i = 0; i < GetSize(); i++)
	{
		vtBuilding *bld = GetAt(i);
		DPoint2 temp = bld->GetLocation();
		if ((fabs(temp.x-point.x) <= error) && (fabs(temp.y-point.y) <= error)) //yes it is
			return i;
	}
	return -1;
}

//
// find the corner of a building closest to the given point
//
// return true if found
//
bool vtBuildingArray::FindClosestCorner(DPoint2 point, double error,
										int &building, int &corner, double &closest)
{
	if (IsEmpty())
		return false;

	building = -1;
	DPoint2 loc;
	double dist;
	closest = 1E8;

	int i, j;
	for (i = 0; i < GetSize(); i++)
	{
		vtBuilding *bld = GetAt(i);
		switch (bld->GetShape())
		{
		case SHAPE_RECTANGLE:
			if (bld->GetFootprint().GetSize() == 0)
				bld->RectToPoly();

		case SHAPE_POLY:
			{
				DLine2 &dl = bld->GetFootprint();
				for (j = 0; j < dl.GetSize(); j++)
				{
					dist = (dl.GetAt(j) - point).Length();
					if (dist > error)
						continue;
					if (dist < closest)
					{
						building = i;
						corner = j;
						closest = dist;
					}
				}
			}
			break;
		case SHAPE_CIRCLE:
			loc = bld->GetLocation();
			dist = fabs((point - loc).Length() - bld->GetRadius());
			if (dist > error)
				continue;
			if (dist < closest)
			{
				building = i;
				corner = 0;
				closest = dist;
			}
			break;
		}
	}
	return (building != -1);
}

//
// find the corner of a building closest to the given point
//
// return true if found
//
bool vtBuildingArray::FindClosestCenter(DPoint2 point, double error,
										int &building, double &closest)
{
	if (IsEmpty())
		return false;

	building = -1;
	DPoint2 loc;
	double dist;
	closest = 1E8;

	for (int i = 0; i < GetSize(); i++)
	{
		vtBuilding *bld = GetAt(i);
		loc = bld->GetLocation();
		dist = (loc - point).Length();
		if (dist > error)
			continue;
		if (dist < closest)
		{
			building = i;
			closest = dist;
		}
	}
	return (building != -1);
}

void vtBuildingArray::GetExtents(DRECT &rect)
{
	if (GetSize() == 0)
		return;

	rect.SetRect(1E9, -1E9, -1E9, 1E9);

	int i, size = GetSize();
	for (i = 0; i < size; i++)
	{
		vtBuilding *ptr = GetAt(i);
		rect.GrowToContainPoint(ptr->GetLocation());
	}
}

int vtBuildingArray::NumSelected()
{
	int sel = 0;
	for (int i = 0; i < GetSize(); i++)
	{
		if (GetAt(i)->IsSelected()) sel++;
	}
	return sel;
}

void vtBuildingArray::DeselectAll()
{
	for (int i = 0; i < GetSize(); i++)
		GetAt(i)->Select(false);
}

void vtBuildingArray::DeleteSelected()
{
	for (int i = 0; i < GetSize();)
	{
		vtBuilding *bld = GetAt(i);
		if (bld->IsSelected())
		{
			delete bld;
			RemoveAt(i);
		}
		else
			i++;
	}
}


