//
//  The main Frame window of the VTBuilder application
//
// Copyright (c) 2001 Virtual Terrain Project
// Free for all uses, see license.txt for details.
//

// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"
#include "wx/resource.h"

#include "vtdata/vtDIB.h"

#include "Frame.h"
#include "SplitterWindow.h"
#include "Treeview.h"
#include "MenuEnum.h"
#include "App.h"
#include "WaterLayer.h"
#include "RoadLayer.h"
#include "BuildingLayer.h"
#include "TransitLayer.h"
#include "ElevDlg.h"
#include "RawLayer.h"
#include "ExtentDlg.h"
#include "ProjectionDlg.h"
#include "ElevPropDlg.h"
#include "Helper.h"

#if defined(__WXMSW__)
static char *dialog1 = NULL;
static char *dialog2 = NULL;
static char *dialog3 = NULL;
static char *dialog4 = NULL;
static char *dialog5 = NULL;
#else
// Other platforms should have compilers that cope with long strings.
#include "dialog1.wxr"
#include "dialog2.wxr"
#include "dialog3.wxr"
#include "dialog4.wxr"
#include "dialog5.wxr"
#endif


DECLARE_APP(MyApp)

// Window ids
#define WID_SPLITTER	100
#define WID_FRAME		101
#define WID_MAINVIEW	102

BEGIN_EVENT_TABLE(MainFrame, wxFrame)
	EVT_MENU(ID_FILE_NEW,		MainFrame::OnProjectNew)
	EVT_MENU(ID_FILE_OPEN,		MainFrame::OnProjectOpen)
	EVT_MENU(ID_FILE_SAVE,		MainFrame::OnProjectSave)
	EVT_MENU(ID_FILE_EXIT,		MainFrame::OnQuit)

	EVT_MENU(ID_EDIT_DELETE, MainFrame::OnEditDelete)
	EVT_MENU(ID_EDIT_DESELECTALL, MainFrame::OnEditDeselectAll)
	EVT_MENU(ID_EDIT_INVERTSELECTION, MainFrame::OnEditInvertSelection)
	EVT_MENU(ID_EDIT_CROSSINGSELECTION, MainFrame::OnEditCrossingSelection)

	EVT_MENU(ID_LAYER_NEW,			MainFrame::OnLayerNew)
	EVT_MENU(ID_LAYER_OPEN,			MainFrame::OnLayerOpen)
	EVT_MENU(ID_LAYER_SAVE,			MainFrame::OnLayerSave)
	EVT_MENU(ID_LAYER_SAVE_AS,		MainFrame::OnLayerSaveAs)
	EVT_MENU(ID_LAYER_IMPORT,		MainFrame::OnLayerImport)
	EVT_MENU(ID_LAYER_PROPS,		MainFrame::OnLayerProperties)
	EVT_MENU(ID_LAYER_CONVERTPROJ,	MainFrame::OnLayerConvert)
	EVT_MENU(ID_LAYER_SETPROJ,		MainFrame::OnLayerSetProjection)
	EVT_MENU(ID_LAYER_FLATTEN,		MainFrame::OnLayerFlatten)
	EVT_MENU(ID_EDIT_OFFSET,		MainFrame::OnEditOffset)

	EVT_UPDATE_UI(ID_LAYER_SAVE,	MainFrame::OnUpdateLayerSave)
	EVT_UPDATE_UI(ID_LAYER_SAVE_AS,	MainFrame::OnUpdateLayerSaveAs)
	EVT_UPDATE_UI(ID_LAYER_FLATTEN,	MainFrame::OnUpdateLayerFlatten)
	EVT_UPDATE_UI(ID_EDIT_OFFSET,	MainFrame::OnUpdateEditOffset)

	EVT_MENU(ID_VIEW_TOOLBAR,		MainFrame::OnViewToolbar)
	EVT_MENU(ID_VIEW_SHOWLAYER,		MainFrame::OnLayerShow)
	EVT_MENU(ID_VIEW_MAGNIFIER,		MainFrame::OnViewMagnifier)
	EVT_MENU(ID_VIEW_PAN,			MainFrame::OnViewPan)
	EVT_MENU(ID_VIEW_DISTANCE,		MainFrame::OnViewDistance)
	EVT_MENU(ID_VIEW_ZOOMIN,		MainFrame::OnViewZoomIn)
	EVT_MENU(ID_VIEW_ZOOMOUT,		MainFrame::OnViewZoomOut)
	EVT_MENU(ID_VIEW_ZOOMALL,		MainFrame::OnViewZoomAll)
	EVT_MENU(ID_VIEW_FULLVIEW,		MainFrame::OnViewFull)
	EVT_MENU(ID_VIEW_SETAREA,		MainFrame::OnElevBox)
	EVT_MENU(ID_VIEW_WORLDMAP,		MainFrame::OnViewWorldMap)
	EVT_MENU(ID_VIEW_SHOWUTM,		MainFrame::OnViewUTMBounds)
	EVT_MENU(ID_VIEW_SHOWMINUTES,	MainFrame::OnViewMinutes)

	EVT_UPDATE_UI(ID_VIEW_TOOLBAR,		MainFrame::OnUpdateToolbar)
	EVT_UPDATE_UI(ID_VIEW_SHOWLAYER,		MainFrame::OnUpdateLayerShow)
	EVT_UPDATE_UI(ID_VIEW_MAGNIFIER,	MainFrame::OnUpdateMagnifier)
	EVT_UPDATE_UI(ID_VIEW_PAN,			MainFrame::OnUpdatePan)
	EVT_UPDATE_UI(ID_VIEW_DISTANCE,		MainFrame::OnUpdateDistance)
	EVT_UPDATE_UI(ID_VIEW_FULLVIEW,		MainFrame::OnUpdateViewFull)
	EVT_UPDATE_UI(ID_VIEW_SETAREA,		MainFrame::OnUpdateElevBox)
	EVT_UPDATE_UI(ID_VIEW_WORLDMAP,		MainFrame::OnUpdateWorldMap)
	EVT_UPDATE_UI(ID_VIEW_SHOWUTM,		MainFrame::OnUpdateUTMBounds)

	EVT_MENU(ID_ROAD_SELECTROAD,	MainFrame::OnSelectRoad)
	EVT_MENU(ID_ROAD_SELECTNODE,	MainFrame::OnSelectNode)
	EVT_MENU(ID_ROAD_SELECTWHOLE,	MainFrame::OnSelectWhole)
	EVT_MENU(ID_ROAD_DIRECTION,		MainFrame::OnDirection)
	EVT_MENU(ID_ROAD_EDIT,			MainFrame::OnRoadEdit)
	EVT_MENU(ID_ROAD_SHOWNODES,		MainFrame::OnShowNodes)
	EVT_MENU(ID_ROAD_SELECTHWY,		MainFrame::OnSelectHwy)
	EVT_MENU(ID_ROAD_CLEAN,			MainFrame::OnRoadClean)

	EVT_UPDATE_UI(ID_ROAD_SELECTROAD,	MainFrame::OnUpdateSelectRoad)
	EVT_UPDATE_UI(ID_ROAD_SELECTNODE,	MainFrame::OnUpdateSelectNode)
	EVT_UPDATE_UI(ID_ROAD_SELECTWHOLE,	MainFrame::OnUpdateSelectWhole)
	EVT_UPDATE_UI(ID_ROAD_DIRECTION,	MainFrame::OnUpdateDirection)
	EVT_UPDATE_UI(ID_ROAD_EDIT,			MainFrame::OnUpdateRoadEdit)
	EVT_UPDATE_UI(ID_ROAD_SHOWNODES,	MainFrame::OnUpdateShowNodes)

	EVT_MENU(ID_ELEV_SELECT,			MainFrame::OnElevSelect)
	EVT_MENU(ID_ELEV_EXPORT,			MainFrame::OnLayerExport)
	EVT_MENU(ID_ELEV_REMOVEABOVESEA,	MainFrame::OnRemoveAboveSea)
	EVT_MENU(ID_ELEV_FILLIN,			MainFrame::OnFillIn)
	EVT_MENU(ID_ELEV_SCALEELEVATION,	MainFrame::OnScaleElevation)
	EVT_MENU(ID_ELEV_EXPORTTERRAGEN,	MainFrame::OnExportTerragen)
	EVT_MENU(ID_ELEV_SHOW,				MainFrame::OnElevShow)
	EVT_MENU(ID_ELEV_SHADING,			MainFrame::OnElevShading)
	EVT_MENU(ID_ELEV_HIDE,				MainFrame::OnElevHide)

	EVT_UPDATE_UI(ID_ELEV_SELECT,		MainFrame::OnUpdateElevSelect)
	EVT_UPDATE_UI(ID_ELEV_EXPORT,		MainFrame::OnUpdateLayerExport)
	EVT_UPDATE_UI(ID_ELEV_REMOVEABOVESEA, MainFrame::OnUpdateRemoveAboveSea)
	EVT_UPDATE_UI(ID_ELEV_FILLIN,		MainFrame::OnUpdateFillIn)
	EVT_UPDATE_UI(ID_ELEV_SCALEELEVATION, MainFrame::OnUpdateScaleElevation)
	EVT_UPDATE_UI(ID_ELEV_EXPORTTERRAGEN, MainFrame::OnUpdateExportTerragen)
	EVT_UPDATE_UI(ID_ELEV_SHOW,			MainFrame::OnUpdateElevShow)
	EVT_UPDATE_UI(ID_ELEV_SHADING,		MainFrame::OnUpdateElevShading)
	EVT_UPDATE_UI(ID_ELEV_HIDE,			MainFrame::OnUpdateElevHide)

	EVT_MENU(ID_VEG_PLANTS,				MainFrame::OnVegPlants)
	EVT_MENU(ID_VEG_BIOREGIONS,			MainFrame::OnVegBioregions)
	EVT_MENU(ID_VEG_GENERATE,			MainFrame::OnVegGenerate)
	EVT_UPDATE_UI(ID_VEG_GENERATE,		MainFrame::OnUpdateVegGenerate)

	EVT_MENU(ID_FEATURE_SELECT,			MainFrame::OnFeatureSelect)
	EVT_UPDATE_UI(ID_FEATURE_SELECT,	MainFrame::OnUpdateFeatureSelect)
	EVT_MENU(ID_BUILDING_EDIT,			MainFrame::OnBuildingEdit)
	EVT_UPDATE_UI(ID_BUILDING_EDIT,		MainFrame::OnUpdateBuildingEdit)

	EVT_MENU(ID_AREA_STRETCH, MainFrame::OnAreaStretch)
	EVT_MENU(ID_AREA_TYPEIN, MainFrame::OnAreaTypeIn)
	EVT_MENU(ID_AREA_BITMAP, MainFrame::OnAreaExportBitmap)

	EVT_MENU(wxID_HELP, MainFrame::OnHelpAbout)

	EVT_SIZE(MainFrame::OnSize)

	EVT_CHAR(MainFrame::OnChar)

END_EVENT_TABLE()

//////////////////////////////////////////////////////////////////

MainFrame *GetMainFrame()
{
	return (MainFrame *) wxGetApp().GetTopWindow();
}

//////////////////////////////////////////////////////////////////
// Frame constructor
//
MainFrame::MainFrame(wxFrame* frame, const wxString& title,
				 const wxPoint& pos, const wxSize& size) :
	wxFrame(frame, WID_FRAME, title, pos, size)
{
	// init app data
	m_pView = NULL;
	m_pActiveLayer = NULL;
	m_PlantListDlg = NULL;
	m_BioRegionDlg = NULL;

#if defined(__WXMSW__)
	// Load the .wxr 'file' from a .rc resource, under Windows.
	dialog1 = wxLoadUserResource("dialog1", "TEXT");
	dialog2 = wxLoadUserResource("dialog2", "TEXT");
	dialog3 = wxLoadUserResource("dialog3", "TEXT");
	dialog4 = wxLoadUserResource("dialog4", "TEXT");
	dialog5 = wxLoadUserResource("dialog5", "TEXT");

	// prepare resources
	bool b1 = wxResourceParseString(dialog1);
	bool b2 = wxResourceParseString(dialog2);
	bool b3 = wxResourceParseString(dialog3);
	bool b4 = wxResourceParseString(dialog4);
	bool b5 = wxResourceParseString(dialog5);
#else
	// prepare resources
	bool b1 = wxResourceParseData(dialog1);	// merge and resample
	bool b2 = wxResourceParseData(dialog2);	// scale elevation
	bool b3 = wxResourceParseData(dialog3);	// specific projection
	bool b4 = wxResourceParseData(dialog4);	// node properties
	bool b5 = wxResourceParseData(dialog5);	// road properties
#endif

	// frame icon
	SetIcon(wxICON(vtbuilder));

	m_statbar = new MyStatusBar(this);
    SetStatusBar(m_statbar);
    m_statbar->Show();
	m_statbar->SetTexts(this);
    PositionStatusBar();

	CreateMenus();
	CreateToolbar();

	SetDropTarget(new DnDFile());

	// splitter
	m_splitter = new MySplitterWindow(this, WID_SPLITTER);

	m_pTree = new MyTreeCtrl(m_splitter, TreeTest_Ctrl,
		wxPoint(0, 0), wxSize(200, 400),
//		wxTR_HAS_BUTTONS |
//		wxTR_EDIT_LABELS |
#ifndef NO_VARIABLE_HEIGHT
		wxTR_HAS_VARIABLE_ROW_HEIGHT |
#endif
		wxNO_BORDER);
	m_pTree->SetBackgroundColour(*wxLIGHT_GREY);

	m_pView = new BuilderView(m_splitter, WID_MAINVIEW,
		wxPoint(0, 0), wxSize(200, 400), "MainView" );
	m_pView->SetBackgroundColour(*wxLIGHT_GREY);
	m_pView->Show(FALSE);

	// Read INI file after creating the view
	ReadINI();

	m_splitter->Initialize(m_pTree);

	////////////////////////
	m_pTree->Show(TRUE);
	m_pView->Show(TRUE);
	m_splitter->SplitVertically( m_pTree, m_pView, 200);

	m_bShowMinutes = false;

	vtProjection proj;
	proj.SetWellKnownGeogCS("WGS84");
	SetProjection(proj);
}

MainFrame::~MainFrame()
{
	WriteINI();
	DeleteContents();
#if defined(__WXMSW__) && 0
	delete dialog1;		// not safe apparently, to delete memory allocated in the DLL
	delete dialog2;
	delete dialog3;
#endif
}

void MainFrame::DeleteContents()
{
	m_Layers.Empty();
	m_pActiveLayer = NULL;
}

void MainFrame::OnSize(wxSizeEvent& event)
{
	int foo = 1;
	wxFrame::OnSize(event);
}

void MainFrame::CreateMenus()
{
	// File (project) menu
	fileMenu = new wxMenu;
	fileMenu->Append(ID_FILE_NEW, "&New\tCtrl+N", "New Project");
	fileMenu->Append(ID_FILE_OPEN, "Open Project\tCtrl+O", "Open Project");
	fileMenu->Append(ID_FILE_SAVE, "Save Project\tCtrl+S", "Save Project As");
	fileMenu->AppendSeparator();
	fileMenu->Append(ID_FILE_EXIT, "E&xit\tAlt-X", "Exit");

 	// Edit
	editMenu = new wxMenu;
	editMenu->Append(ID_EDIT_DELETE, "Delete\t<Del>", "Delete.");
	editMenu->AppendSeparator();
	editMenu->Append(ID_EDIT_DESELECTALL, "Deselect All", "Clears selection.");
	editMenu->Append(ID_EDIT_INVERTSELECTION, "Invert Selection", "Invert Selection.");
	editMenu->Append(ID_EDIT_CROSSINGSELECTION, "Crossing Selection", "Crossing Selection.", true);

	// Layer
	layerMenu = new wxMenu;
	layerMenu->Append(ID_LAYER_NEW, "&New Layer", "Create New Layer");
	layerMenu->Append(ID_LAYER_OPEN, "Open Layer", "Open Existing Layer");
	layerMenu->Append(ID_LAYER_SAVE, "Save Layer", "Save Active Layer");
	layerMenu->Append(ID_LAYER_SAVE_AS, "Save Layer As...", "Save Active Layer As");
	layerMenu->Append(ID_LAYER_IMPORT, "Import Data\tCtrl+I", "Import Elevation");
	layerMenu->AppendSeparator();
	layerMenu->Append(ID_LAYER_PROPS, "Layer Properties", "Layer Properties");
	layerMenu->AppendSeparator();
	layerMenu->Append(ID_EDIT_OFFSET, "Offset Coordinates", "Offset");
	layerMenu->AppendSeparator();
	layerMenu->Append(ID_LAYER_FLATTEN, "&Flatten Layers", "Flatten");
	layerMenu->AppendSeparator();
	layerMenu->Append(ID_LAYER_CONVERTPROJ, "Convert Projection", "Convert");
	layerMenu->Append(ID_LAYER_SETPROJ, "Set Projection", "Set Projection");

	// View
	viewMenu = new wxMenu;
	viewMenu->Append(ID_VIEW_TOOLBAR, "Toolbar", "Show Toolbar", true);
	viewMenu->Append(ID_VIEW_STATUSBAR, "Status Bar", "Show Status Bar", true);
	viewMenu->Append(ID_VIEW_SHOWLAYER, "Current Layer Visible", "Toggle Visibility of the current Layer", true);
	viewMenu->AppendSeparator();
	viewMenu->Append(ID_VIEW_ZOOMIN, "Zoom In\tCtrl++");
	viewMenu->Append(ID_VIEW_ZOOMOUT, "Zoom Out\tCtrl+-");
	viewMenu->Append(ID_VIEW_ZOOMALL, "Zoom All");
	viewMenu->Append(ID_VIEW_FULLVIEW, "Zoom to Full Res (1:1)");
	viewMenu->AppendSeparator();
	viewMenu->Append(ID_VIEW_MAGNIFIER, "Magnifier\tZ", "Magnifier", true);
	viewMenu->Append(ID_VIEW_PAN, "Pan\tCtrl+<Space>", "Pan", true);
	viewMenu->Append(ID_VIEW_DISTANCE, "Obtain Distance", "Obtain Distance", true);
	viewMenu->Append(ID_VIEW_SETAREA, "Set Export Area", "Set Export Area", true);
	viewMenu->AppendSeparator();
	viewMenu->Append(ID_VIEW_WORLDMAP, "World Map", "Show/Hide World Map", true);
	viewMenu->Append(ID_VIEW_SHOWUTM, "Show UTM Boundaries", "Show UTM Boundaries", true);
//	viewMenu->Append(ID_VIEW_SHOWGRID, "Show 7.5\" Grid", "Show 7.5\" Grid", true);
	viewMenu->Append(ID_VIEW_SHOWMINUTES, "Show minutes and seconds");
	viewMenu->AppendSeparator();
	viewMenu->Append(ID_ELEV_SHOW, "Show Terrain Elevation", "Show Terrain Elevation", true);
	viewMenu->Append(ID_ELEV_SHADING, "Artificial Shading", "Artificial Shading", true);
	viewMenu->Append(ID_ELEV_HIDE, "Hide Unknown Areas", "Hide Unknown Areas", true);

	// Roads
	roadMenu = new wxMenu;
	roadMenu->Append(ID_ROAD_SELECTROAD, "Select/Modify Roads", "Select/Modify Roads", true);
	roadMenu->Append(ID_ROAD_SELECTNODE, "Select/Modify Nodes", "Select/Modify Nodes", true);
	roadMenu->Append(ID_ROAD_SELECTWHOLE, "Select Whole Roads", "Select Whole Roads", true);
	roadMenu->Append(ID_ROAD_DIRECTION, "Set Road Direction", "Set Road Direction", true);
	roadMenu->Append(ID_ROAD_EDIT, "Edit Road Points", "Edit Road Points", true);
	roadMenu->AppendSeparator();
	roadMenu->Append(ID_ROAD_SHOWNODES, "Show Nodes", "Show Nodes", true);
	roadMenu->Append(ID_ROAD_SELECTHWY, "Select by Highway Number", "Select Highway", true);
	roadMenu->AppendSeparator();
	roadMenu->Append(ID_ROAD_CLEAN, "Clean RoadMap", "Clean");

	// Elevation
	elevMenu = new wxMenu;
	elevMenu->Append(ID_ELEV_SELECT, "Select Elevation Layer", "Select Elevation Layer", true);
	elevMenu->AppendSeparator();
	elevMenu->Append(ID_ELEV_REMOVEABOVESEA, "Remove Terrain Above Sea");
	elevMenu->Append(ID_ELEV_FILLIN, "Fill in unknown areas");
	elevMenu->Append(ID_ELEV_SCALEELEVATION, "Scale Elevation");            
	elevMenu->Append(ID_ELEV_EXPORTTERRAGEN, "Export to TerraGen");            
	elevMenu->AppendSeparator();
	elevMenu->Append(ID_ELEV_EXPORT, "&Merge Area and Save to BT", "Merge Area and Save to BT");

	// Vegetation
	vegMenu = new wxMenu;
	vegMenu->Append(ID_VEG_PLANTS, "Plants List", "View/Edit list of available plant species");
	vegMenu->Append(ID_VEG_BIOREGIONS, "BioRegions", "View/Edit list of species & density for each BioRegion");
	vegMenu->Append(ID_VEG_GENERATE, "Generate", "Generate Vegetation File (*.vf) containg plant local & info");

	// Buildings
	bldMenu = new wxMenu;
	bldMenu->Append(ID_FEATURE_SELECT, "Select Buildings", "Select Buildings", true);
	bldMenu->Append(ID_BUILDING_EDIT, "Edit Buildings", "Edit Buildings", true);

	// Area
	areaMenu = new wxMenu;
	areaMenu->Append(ID_AREA_STRETCH, "Set to Extents");
	areaMenu->Append(ID_AREA_TYPEIN, "Numeric Values");
	areaMenu->AppendSeparator();
	areaMenu->Append(ID_AREA_BITMAP, "Export Bitmap");

	// Help
	helpMenu = new wxMenu;
    helpMenu->Append(wxID_HELP, "&About", "About VTBuilder");

	menuBar = new wxMenuBar;
	menuBar->Append(fileMenu, "&Project");
	menuBar->Append(editMenu, "&Edit");
	menuBar->Append(layerMenu, "&Layer");
	menuBar->Append(viewMenu, "&View");
	menuBar->Append(roadMenu, "&Roads");
	menuBar->Append(elevMenu, "Elev&ation");
	menuBar->Append(vegMenu, "Veg&etation");
	menuBar->Append(bldMenu, "&Buildings");
	menuBar->Append(areaMenu, "E&xport Area");
	menuBar->Append(helpMenu, "&Help");
	SetMenuBar(menuBar);

    // Accelerators
    wxAcceleratorEntry entries[5];
    entries[0].Set(wxACCEL_CTRL, (int) 'O', ID_FILE_OPEN);
    entries[1].Set(wxACCEL_CTRL, (int) 'S', ID_FILE_SAVE);
    entries[2].Set(wxACCEL_CTRL, (int) '+', ID_VIEW_ZOOMIN);
    entries[3].Set(wxACCEL_CTRL, (int) '-', ID_VIEW_ZOOMOUT);
    entries[4].Set(wxACCEL_NORMAL,  WXK_DELETE,    ID_EDIT_DELETE);
    wxAcceleratorTable accel(5, entries);
    SetAcceleratorTable(accel);
}

void MainFrame::CreateToolbar()
{
	// tool bar
	toolBar_main = CreateToolBar(wxTB_HORIZONTAL | wxNO_BORDER | wxTB_DOCKABLE);
	toolBar_main->SetMargins(2, 2);
	toolBar_main->SetToolBitmapSize(wxSize(20, 20));

	RefreshToolbar();
}

#define NUM_MAIN_TOOLBARS 21

void MainFrame::RefreshToolbar()
{
	int count = toolBar_main->GetToolsCount();
	// remove any existing buttons
	while (count > NUM_MAIN_TOOLBARS)
	{
		toolBar_main->DeleteToolByPos(NUM_MAIN_TOOLBARS);
		count = toolBar_main->GetToolsCount();
	}

	// now add new buttons
	if (count < NUM_MAIN_TOOLBARS)
	{
		ADD_TOOL(ID_FILE_NEW, wxBITMAP(proj_new), _("New Project"), false);
		ADD_TOOL(ID_FILE_OPEN, wxBITMAP(proj_open), _("Open Project"), false);
		ADD_TOOL(ID_FILE_SAVE, wxBITMAP(proj_save), _("Save Project"), false);
		toolBar_main->AddSeparator();
		ADD_TOOL(ID_LAYER_NEW, wxBITMAP(layer_new), _("New Layer"), false);
		ADD_TOOL(ID_LAYER_OPEN, wxBITMAP(layer_open), _("Open Layer"), false);
		ADD_TOOL(ID_LAYER_SAVE, wxBITMAP(layer_save), _("Save Layer"), false);
		ADD_TOOL(ID_LAYER_IMPORT, wxBITMAP(layer_import), _("Import Data"), false);
		toolBar_main->AddSeparator();
		ADD_TOOL(ID_EDIT_DELETE, wxBITMAP(edit_delete), _("Delete"), false);
		ADD_TOOL(ID_EDIT_OFFSET, wxBITMAP(edit_offset), _("Offset"), false);
		ADD_TOOL(ID_VIEW_SHOWLAYER, wxBITMAP(layer_show), _("Layer Visibility"), true);
		toolBar_main->AddSeparator();
		ADD_TOOL(ID_VIEW_ZOOMIN, wxBITMAP(view_plus), _("Zoom In"), false);
		ADD_TOOL(ID_VIEW_ZOOMOUT, wxBITMAP(view_minus), _("Zoom Out"), false);
		ADD_TOOL(ID_VIEW_ZOOMALL, wxBITMAP(view_zoomall), _("Zoom All"), false);
		toolBar_main->AddSeparator();
		ADD_TOOL(ID_VIEW_MAGNIFIER, wxBITMAP(view_mag), _("Magnifier"), true);
		ADD_TOOL(ID_VIEW_PAN, wxBITMAP(view_hand), _("Pan"), true);
		ADD_TOOL(ID_VIEW_DISTANCE, wxBITMAP(distance), _("Distance"), true);
		ADD_TOOL(ID_VIEW_SETAREA, wxBITMAP(elev_box), _("Set Export Area"), true);
	}

	vtLayer *pLayer = GetActiveLayer();
	LayerType lt = LT_UNKNOWN;
	if (pLayer)
		lt = pLayer->GetType();

	switch (lt)
	{
	case LT_ROAD:
		toolBar_main->AddSeparator();
		ADD_TOOL(ID_ROAD_SELECTROAD, wxBITMAP(rd_select_road), _("Select Roads"), true);
		ADD_TOOL(ID_ROAD_SELECTNODE, wxBITMAP(rd_select_node), _("Select Nodes"), true);
		ADD_TOOL(ID_ROAD_SELECTWHOLE, wxBITMAP(rd_select_whole), _("Select Whole Roads"), true);
		ADD_TOOL(ID_ROAD_DIRECTION, wxBITMAP(rd_direction), _("Set Road Direction"), true);
		ADD_TOOL(ID_ROAD_EDIT, wxBITMAP(rd_edit), _("Edit Road Points"), true);
		ADD_TOOL(ID_ROAD_SHOWNODES, wxBITMAP(rd_shownodes), _("Show Nodes"), true);
		ADD_TOOL(ID_EDIT_CROSSINGSELECTION, wxBITMAP(edit_crossing), _("Crossing Selection"), true);
		break;
	case LT_ELEVATION:
		toolBar_main->AddSeparator();
		ADD_TOOL(ID_ELEV_SELECT, wxBITMAP(select), _("Select Elevation"), true);
		ADD_TOOL(ID_VIEW_FULLVIEW, wxBITMAP(view_zoomexact), _("Zoom to Full Detail"), false);
		ADD_TOOL(ID_ELEV_EXPORT, wxBITMAP(layer_export), _("Export Data"), false);
		break;
	case LT_IMAGE:
		toolBar_main->AddSeparator();
		ADD_TOOL(ID_VIEW_FULLVIEW, wxBITMAP(view_zoomexact), _("Zoom to Full Detail"), false);
		break;
	case LT_BUILDING:
		toolBar_main->AddSeparator();
		ADD_TOOL(ID_FEATURE_SELECT, wxBITMAP(select), _("Select Buildings"), true);
		ADD_TOOL(ID_BUILDING_EDIT, wxBITMAP(bld_edit), _("Edit Buildings"), true);
		break;
	}

	toolBar_main->Realize();
//	toolBar_main->EnableTool(ID_STOP, FALSE);	// must be after Realize() !

	menuBar->EnableTop(4, lt == LT_ROAD);
	menuBar->EnableTop(5, lt == LT_ELEVATION);
//	menuBar->EnableTop(6, lt == LT_VEG);
	menuBar->EnableTop(7, lt == LT_BUILDING);
}

////////////////////////////////////////////////////////////////
// Application Methods

//
// Load a layer from a file without knowing its type
//
void MainFrame::LoadLayer(wxString &fname)
{
	// check file extension
	wxString ext = fname.Right(3);

	bool bFirst = (m_Layers.GetSize() == 0);

	vtLayer *pLayer = NULL;
	if (ext.CmpNoCase("rmf") == 0)
	{
		vtRoadLayer *pRL = new vtRoadLayer();
		if (pRL->Load(fname))
			pLayer = pRL;
	}
	if (ext.CmpNoCase("bcf") == 0)
	{
		vtBuildingLayer *pBL = new vtBuildingLayer();
		if (pBL->Load(fname))
			pLayer = pBL;
	}
	if (ext.CmpNoCase(".bt") == 0)
	{
		vtElevLayer *pEL = new vtElevLayer();
		if (pEL->Load(fname))
			pLayer = pEL;
	}
#if SUPPORT_TRANSIT
	if (ext.CmpNoCase("xml") == 0)
	{
		vtTransitLayer *pTL = new vtTransitLayer();
		if (pTL->Load(fname))
			pLayer = pTL;
	}
#endif
	if (ext.CmpNoCase("shp") == 0)
	{
		vtRawLayer *pRL = new vtRawLayer();
		if (pRL->Load(fname))
			pLayer = pRL;
	}
	if (!pLayer)
	{
//		wxString msg = "Couldn't load layer " + fname;
//		SetStatusText(msg);
		// try importing
		ImportDataFromFile(LT_UNKNOWN, fname, true);
		return;
	}

	bool success = AddLayerWithCheck(pLayer, true);
	if (!success)
		delete pLayer;
}

bool MainFrame::AddLayerWithCheck(vtLayerPtr pLayer, bool bRefresh)
{
	vtProjection proj;
	pLayer->GetProjection(proj);

	bool bFirst = (m_Layers.GetSize() == 0);
	if (bFirst)
	{
		// if this is our first layer, adopt its projection
		SetProjection(proj);
	}
	else
	{
		// check for Projection conflict
		if (!(m_proj == proj))
		{
			char *str1, *str2;
			m_proj.exportToProj4(&str1);
			proj.exportToProj4(&str2);

			bool keep = false;
			wxString msg;
			msg.Printf("The data already loaded is in:\n     %s\n"
				"but the file you are attempting to load:\n     %s\n"
				"is using:\n     %s\n"
				"Would you like to attempt to convert it now to the existing projection?",
				str1,
				pLayer->GetFilename(),
				str2);
			int ret = wxMessageBox(msg, "Warning", wxYES_NO | wxCANCEL);
			if (ret == wxNO)
				keep = true;
			if (ret == wxYES)
			{
				bool success = pLayer->ConvertProjection(m_proj);
				if (success)
					keep = true;
				else
				{
					ret = wxMessageBox("Couldn't convert projection.\nProceed anyway?", "Warning", wxYES_NO);
					if (ret == wxYES)
						keep = true;
				}
			}
			if (!keep)
			{
				return false;
			}
		}
	}
	AddLayer(pLayer);
	SetActiveLayer(pLayer, false);
	if (bRefresh)
	{
		// refresh the view
		m_pView->ZoomAll();
		RefreshToolbar();
		RefreshTreeView();
		RefreshStatusBar();
	}
	return true;
}

void MainFrame::AddLayer(vtLayerPtr lp)
{
	m_Layers.Append(lp);
}

void MainFrame::RemoveLayer(vtLayerPtr lp)
{
	if (!lp)
		return;

	// check the type of the layer we're deleting
	LayerType lt = lp->GetType();

	// remove and delete the layer
	m_Layers.RemoveAt(m_Layers.Find(lp));

	// if it was the active layer, select another layer of the same type
	if (GetActiveLayer() == lp)
	{
		vtLayerPtr lp_new = FindLayerOfType(lt);
		SetActiveLayer(lp_new, true);
	}
	DeleteLayer(lp);
	m_pView->Refresh();
	m_pTree->RefreshTreeItems(this);
	RefreshToolbar();
}

void MainFrame::DeleteLayer(vtLayerPtr lp)
{
	delete lp;
}

void MainFrame::SetActiveLayer(vtLayerPtr lp, bool refresh)
{
	LayerType last = m_pActiveLayer ? m_pActiveLayer->GetType() : LT_UNKNOWN;

	m_pActiveLayer = lp;
	if (refresh)
		m_pTree->RefreshTreeStatus(this);

	// change mouse mode based on layer type
	if (lp == NULL)
		m_pView->SetMode(LB_Mag);

	if (lp != NULL)
	{
		if (lp->GetType() == LT_ELEVATION && last != LT_ELEVATION)
			m_pView->SetMode(LB_TSelect);

		if (lp->GetType() == LT_ROAD && last != LT_ROAD)
			m_pView->SetMode(LB_Road);

		if (lp->GetType() == LT_BUILDING && last != LT_BUILDING)
			m_pView->SetMode(LB_FSelect);
	}
}

int MainFrame::LayersOfType(LayerType lt)
{
	int count = 0;
	int layers = m_Layers.GetSize();
	for (int l = 0; l < layers; l++)
	{
		if (m_Layers.GetAt(l)->GetType() == lt)
			count++;
	}
	return count;
}

vtLayerPtr MainFrame::FindLayerOfType(LayerType lt)
{
	int layers = m_Layers.GetSize();
	for (int l = 0; l < layers; l++)
	{
		vtLayerPtr lp = m_Layers.GetAt(l);
		if (lp->GetType() == lt)
			return lp;
	}
	return NULL;
}

//
// read / write ini file
//
bool MainFrame::ReadINI()
{
	m_fpIni = fopen("VTBuilder.ini", "rb+");

	if (m_fpIni)
	{
		int ShowMap, ShowElev, Shading, DoMask, DoUTM;
		fscanf(m_fpIni, "%d %d %d %d %d", &ShowMap, &ShowElev, &Shading, &DoMask, &DoUTM);

		m_pView->m_bShowMap = (ShowMap != 0);
		vtElevLayer::m_bShowElevation = (ShowElev != 0);
		vtElevLayer::m_bShading = (Shading != 0);
		vtElevLayer::m_bDoMask = (DoMask != 0);
		m_pView->m_bShowUTMBounds = (DoUTM != 0);

		return true;
	}
	m_fpIni = fopen("VTBuilder.ini", "wb");
	return false;
}

bool MainFrame::WriteINI()
{
	if (m_fpIni)
	{
		rewind(m_fpIni);
		fprintf(m_fpIni, "%d %d %d %d %d", m_pView->m_bShowMap, vtElevLayer::m_bShowElevation,
			vtElevLayer::m_bShading, vtElevLayer::m_bDoMask, m_pView->m_bShowUTMBounds);
		fclose(m_fpIni);
		m_fpIni = NULL;
		return true;
	}
	return false;
}

DRECT MainFrame::GetExtents()
{
	DRECT rect(1E9,-1E9,-1E9,1E9);

	bool has_bounds = false;

	// Acculumate the extents of all the layers
	DRECT rect2;
	int iLayers = m_Layers.GetSize();

	for (int i = 0; i < iLayers; i++)
	{
		if (m_Layers.GetAt(i)->GetExtent(rect2))
		{
			rect.GrowToContainRect(rect2);
			has_bounds = true;
		}
	}
	if (has_bounds)
		return rect;
	else
		return DRECT(-180,90,180,-90);	// geo extents of whole planet

}

void MainFrame::StretchArea()
{
	m_area = GetExtents();
}

void MainFrame::RefreshTreeView()
{
	if (m_pTree)
		m_pTree->RefreshTreeItems(this);
}

void MainFrame::RefreshTreeStatus()
{
	if (m_pTree)
		m_pTree->RefreshTreeStatus(this);
}

void MainFrame::RefreshStatusBar()
{
	m_statbar->SetTexts(this);
}

LayerType MainFrame::AskLayerType()
{
	wxString choices[LAYER_TYPES];
	for (int i = 0; i < LAYER_TYPES; i++)
		choices[i] = vtLayer::LayerTypeName[i];

    int n = LAYER_TYPES;
	static int cur_type = 0;	// remember the choice for next time

    wxSingleChoiceDialog dialog(this, "These are your choices",
        "Please indicate layer type", n, (const wxString *)choices);

    dialog.SetSelection(cur_type);

    if (dialog.ShowModal() == wxID_OK)
	{
		cur_type = dialog.GetSelection();
		return (LayerType) cur_type;
	}
	else
		return LT_UNKNOWN;
}

//
// merge all terrain data into this one
//
void MainFrame::SampleCurrentTerrains(vtElevLayer *pTarget)
{
	DRECT area = pTarget->GetExtents();
	DPoint2 step = pTarget->m_pGrid->GetSpacing();

	double x, y;
	int i, j, l, layers = m_Layers.GetSize();
	double fData, fBestData;
	int iColumns, iRows;
	pTarget->m_pGrid->GetDimensions(iColumns, iRows);

	// Create progress dialog for the slow part
	OpenProgressDialog("Merging and Resampling Elevation Layers");

	int num_elev = LayersOfType(LT_ELEVATION);
	vtElevationGrid **grids = new vtElevationGrid *[num_elev];
	int g, num_grids = 0;
	for (l = 0; l < layers; l++)
	{
		vtLayerPtr lp = m_Layers.GetAt(l);
		if (lp->GetType() == LT_ELEVATION)
			grids[num_grids++] = ((vtElevLayer *)lp)->m_pGrid;
	}

	// iterate through the vertices of the new terrain
	for (i = 0; i < iColumns; i++)
	{
		UpdateProgressDialog(i*100/iColumns);
		x = area.left + (i * step.x);
		for (j = 0; j < iRows; j++)
		{
			y = area.bottom + (j * step.y);

			// find some data for this point
			fBestData = INVALID_ELEVATION;
			for (g = 0; g < num_grids; g++)
			{
				vtElevationGrid *grid = grids[g];

				// is our point inside the extents of this terrain?
				if (! grid->ContainsPoint(x, y))
					continue;

				fData = grid->GetFilteredValue(x, y);
				if (fData != INVALID_ELEVATION &&
					(fBestData == INVALID_ELEVATION || fBestData == 0.0f))
					fBestData = fData;
				if (fBestData != INVALID_ELEVATION && fBestData != 0)
					break;
			}
			pTarget->m_pGrid->SetFValue(i, j, fBestData);
		}
	}
	CloseProgressDialog();
}


double MainFrame::GetHeightFromTerrain(double lx, double ly)
{
	double height = INVALID_ELEVATION;

	int layers = m_Layers.GetSize();
	for (int i = 0; i < layers; i++)
	{
		vtLayer *l = m_Layers.GetAt(i);
		if (l->GetType() != LT_ELEVATION) continue;
		vtElevLayer *pTerr = (vtElevLayer *)l;

		height = pTerr->m_pGrid->GetFilteredValue(lx, ly);
		if (height != INVALID_ELEVATION)
			break;
	}
	return height;
}

void MainFrame::SetProjection(vtProjection &p)
{
	m_proj = p;
	GetView()->SetWMProj(p);
}

////////////////////////////////////////////////////////////////
// Project ops

void MainFrame::OnProjectNew(wxCommandEvent &event)
{
	SetActiveLayer(NULL);
	DeleteContents();
	m_area.SetRect(0.0, 0.0, 0.0, 0.0);
	m_pView->Refresh();
	Refresh();

	RefreshTreeView();
	RefreshToolbar();

	vtProjection p;
	SetProjection(p);
}

#define PROJECT_FILTER "VTBuilder Project Files (*.vtb)|*.vtb|"

void MainFrame::OnProjectOpen(wxCommandEvent &event)
{
	wxFileDialog loadFile(NULL, "Load Project", "", "", PROJECT_FILTER, wxOPEN);
	bool bResult = (loadFile.ShowModal() == wxID_OK);
	if (!bResult)
		return;
	LoadProject(loadFile.GetPath());
}

void MainFrame::LoadProject(wxString &strPathName)
{
	// read project file
	wxString str;
	FILE *fp = fopen(strPathName, "rb");
	if (!fp)
	{
		str = wxString::Format("Couldn't open project file: %s", strPathName);
		wxMessageBox(str);
		return;
	}

	// read projection info
	char buf[900];
	fgets(buf, 900, fp);
	char *wkt = buf + 11;
	OGRErr err = m_proj.importFromWkt(&wkt);
	if (err != OGRERR_NONE)
	{
		wxMessageBox("Had trouble parsing the projection information from that file.");
		fclose(fp);
		return;
	}

#if 0
	int a, b, c;
//	fscanf(fp, "utm %d\n", &a);
//	fscanf(fp, "utm_zone %d\n", &b);
//	fscanf(fp, "datum %d\n", &c);
	vtProjection p;
	p.SetUTM(a != 0);
	p.SetUTMZone(b);
	p.SetDatum((DATUM) c);
	SetProjection(p);
#endif

	int count = 0;
	LayerType ltype;

	fscanf(fp, "layers: %d\n", &count);
	if (count < 1)
	{
		fclose(fp);
		str = wxString::Format("Empty or invalid project file: %s", strPathName);
		wxMessageBox(str);
		return;
	}
	for (int i = 0; i < count; i++)
	{
		char buf2[80];
		fscanf(fp, "type %d, %s\n", &ltype, buf2);
		fgets(buf, 160, fp);

		// trim trailing LF character
		int len = strlen(buf);
		if (len && buf[len-1] == 10) buf[len-1] = 0;
		len = strlen(buf);
		if (len && buf[len-1] == 13) buf[len-1] = 0;
		wxString fname = buf;

		if (!strcmp(buf2, "import"))
		{
			ImportDataFromFile(ltype, fname, false);
		}
		else
		{
			vtLayerPtr lp = vtLayer::CreateNewLayer(ltype);
			if (lp->Load(fname))
				AddLayer(lp);
			else
				delete lp;
		}
	}

	fscanf(fp, "%s", buf);
	if (!strcmp(buf, "area"))
	{
		fscanf(fp, "%lf %lf %lf %lf\n", &m_area.left, &m_area.top,
			&m_area.right, &m_area.bottom);
	}

	bool bHasView = false;
	fscanf(fp, "%s", buf);
	if (!strcmp(buf, "view"))
	{
		DRECT rect;
		fscanf(fp, "%lf %lf %lf %lf\n", &rect.left, &rect.top,
			&rect.right, &rect.bottom);
		m_pView->ZoomToRect(rect, 0.0f);
		bHasView = true;
	}
	fclose(fp);

	// refresh the view
	if (!bHasView)
		m_pView->ZoomAll();
	RefreshTreeView();
	RefreshToolbar();
}

void MainFrame::OnProjectSave(wxCommandEvent &event)
{
	wxFileDialog saveFile(NULL, "Save Project", "", "", PROJECT_FILTER,
		wxSAVE | wxOVERWRITE_PROMPT );
	bool bResult = (saveFile.ShowModal() == wxID_OK);
	if (!bResult)
		return;
	wxString strPathName = saveFile.GetPath();

	// write project file
	FILE *fp = fopen(strPathName, "wb");
	if (!fp)
		return;

	// write projection info
	char *wkt;
	m_proj.exportToWkt(&wkt);
	fprintf(fp, "Projection %s\n", wkt);
	OGRFree(wkt);

	// write list of layers
	int iLayers = m_Layers.GetSize();
	fprintf(fp, "layers: %d\n", iLayers);

	vtLayerPtr lp;
	for (int i = 0; i < iLayers; i++)
	{
		lp = m_Layers.GetAt(i);

		fprintf(fp, "type %d, %s\n", lp->GetType(), lp->IsNative() ? "native" : "import");
		fprintf(fp, "%s\n", lp->GetFilename());
	}

	// write area
	fprintf(fp, "area %lf %lf %lf %lf\n", m_area.left, m_area.top,
		m_area.right, m_area.bottom);

	// write view rectangle
	DRECT rect = m_pView->GetWorldRect();
	fprintf(fp, "view %lf %lf %lf %lf\n", rect.left, rect.top,
		rect.right, rect.bottom);

	// done
	fclose(fp);
}


void MainFrame::OnQuit(wxCommandEvent &event)
{
	Close(TRUE);
}


///////////////////////////////////////////////////////////////////////

bool DnDFile::OnDropFiles(wxCoord, wxCoord, const wxArrayString& filenames)
{
    size_t nFiles = filenames.GetCount();
    for ( size_t n = 0; n < nFiles; n++ )
	{
		wxString str = filenames[n];
		if (!str.Right(3).CmpNoCase("vtb"))
			GetMainFrame()->LoadProject(str);
		else
			GetMainFrame()->LoadLayer(str);
    }
    return TRUE;
}

void MainFrame::ExportElevation()
{
	// If any of the input terrain are floats, the output should be float
	bool floatmode = false;

	// sample spacing in meters/heixel or degrees/heixel
	DPoint2 spacing(1.0f, 1.0f);
	for (int i = 0; i < m_Layers.GetSize(); i++)
	{
		vtLayer *l = m_Layers.GetAt(i);
		if (l->GetType() == LT_ELEVATION)
		{
			vtElevLayer *el = (vtElevLayer *)l;
			if (el->m_pGrid->IsFloatMode())
				floatmode = true;
			spacing = el->m_pGrid->GetSpacing();
		}
	}

	// Open the Resample dialog
	ResampleDlg dlg(!m_proj.IsGeographic(), spacing.x);
	dlg.LoadFromResource(this, "dialog1");
	dlg.m_area = m_area;

	if (dlg.ShowModal() == wxID_CANCEL)
		return;

	wxString filter = "All Files|*.*|";
	AddType(filter, FSTRING_BT);

	// ask the user for a filename
	wxFileDialog saveFile(NULL, "Export Elevation", "", "", filter, wxSAVE);
	saveFile.SetFilterIndex(1);
	bool bResult = (saveFile.ShowModal() == wxID_OK);
	if (!bResult)
		return;
	wxString strPathName = saveFile.GetPath();

	// Make new terrain
	vtElevLayer *pBig = new vtElevLayer(dlg.m_area, dlg.m_iXSamples, dlg.m_iYSamples,
		floatmode, m_proj);
	pBig->SetFilename(strPathName);

	// fill in the value for pBig by merging samples from all other terrain
	SampleCurrentTerrains(pBig);
	pBig->FillGaps();

	bool success = pBig->m_pGrid->SaveToBT(strPathName);
	if (!success)
	{
		wxMessageBox("Couldn't open file for writing.");
		delete pBig;
		return;
	}

	wxString str = wxString::Format("Successfully wrote BT file %s", strPathName);
	wxMessageBox(str);
	delete pBig;
}

//////////////////////////////////////////////////
// Edit ops

void MainFrame::OnEditDelete(wxCommandEvent &event)
{
	vtRoadLayer *pRL = GetActiveRoadLayer();
	if (pRL && (pRL->NumSelectedNodes() != 0 || pRL->NumSelectedRoads() != 0))
	{
		m_pView->DeleteSelected(pRL);
		return;
	}
	vtBuildingLayer *pBL = GetActiveBuildingLayer();
	if (pBL && pBL->NumSelected() != 0)
	{
		pBL->DeleteSelected();
		pBL->SetModified(true);
		m_pView->Refresh();
		return;
	}

	vtLayer *pL = GetActiveLayer();
	if (pL)
	{
		int result = wxMessageBox("Are you sure you want to delete the current layer?",
			"Question", wxYES_NO | wxICON_QUESTION, this);
		if (result == wxYES)
			RemoveLayer(pL);
	}
}

void MainFrame::OnEditDeselectAll(wxCommandEvent &event)
{
	m_pView->DeselectAll();
}

void MainFrame::OnEditInvertSelection(wxCommandEvent &event)
{
	vtRoadLayer *pRL = GetActiveRoadLayer();
	if (pRL) {
		pRL->InvertSelection();
		m_pView->Refresh();
	}	
}

void MainFrame::OnEditCrossingSelection(wxCommandEvent &event)
{
	m_pView->m_bCrossSelect = !m_pView->m_bCrossSelect;
}

void MainFrame::OnUpdateCrossingSelection(wxUpdateUIEvent& event)
{
	event.Check(m_pView->m_bCrossSelect);
}

void MainFrame::OnEditOffset(wxCommandEvent &event)
{
	wxTextEntryDialog dlg(this, "Offset", "Please enter horizontal offset X, Y", "0.0, 0.0");
	if (dlg.ShowModal() != wxID_OK)
		return;

	DPoint2 offset;
	wxString str = dlg.GetValue();
	sscanf(str, "%lf, %lf", &offset.x, &offset.y);

	GetActiveLayer()->Offset(offset);
	GetActiveLayer()->SetModified(true);
	m_pView->Refresh();
}

void MainFrame::OnUpdateEditOffset(wxUpdateUIEvent& event)
{
	event.Enable(GetActiveLayer() != NULL);
}


//////////////////////////////////////////////////
// Layer ops

void MainFrame::OnLayerNew(wxCommandEvent &event)
{
	LayerType t = AskLayerType();
	if (t == LT_UNKNOWN)
		return;

	vtLayer *pL = NULL;
	switch (t)
	{
	case LT_ROAD:
		pL = new vtRoadLayer();
		break;
	case LT_BUILDING:
		pL = new vtBuildingLayer();
		break;
	case LT_WATER:
		pL = new vtWaterLayer();
		break;
	case LT_VEG:
		pL = new vtVegLayer();
		break;
	}
	if (pL)
	{
		AddLayer(pL);
		m_pTree->RefreshTreeItems(this);
	}
}

void MainFrame::OnLayerOpen(wxCommandEvent &event)
{
	wxString filter = "Native Layer Formats||";

	AddType(filter, FSTRING_RMF);
	AddType(filter, FSTRING_BCF);
	AddType(filter, FSTRING_BT);
	AddType(filter, FSTRING_SHP);
	AddType(filter, FSTRING_XML);

	// ask the user for a filename
	wxFileDialog loadFile(NULL, "Open Layer", "", "", filter, wxOPEN);
	bool bResult = (loadFile.ShowModal() == wxID_OK);
	if (!bResult)
		return;

	LoadLayer(loadFile.GetPath());
}

void MainFrame::OnLayerSave(wxCommandEvent &event)
{
	vtLayerPtr lp = GetActiveLayer();
	if (lp->GetFilename().Left(9).CmpNoCase("untitled.") == 0)
	{
		if (!lp->AskForSaveFilename())
			return;
	}
	wxString msg = "Saving layer to file " + lp->GetFilename();
	SetStatusText(msg);

	if (lp->Save())
		msg = "Saved layer to file " + lp->GetFilename();
	else
		msg = "Save failed.";
	SetStatusText(msg);
}

void MainFrame::OnUpdateLayerSave(wxUpdateUIEvent& event)
{
	vtLayerPtr lp = GetActiveLayer();
	event.Enable(lp != NULL && lp->GetModified());
}

void MainFrame::OnLayerSaveAs(wxCommandEvent &event)
{
	vtLayerPtr lp = GetActiveLayer();

	if (!lp->AskForSaveFilename())
			return;

	wxString msg = "Saving layer to file as " + lp->GetFilename();
	SetStatusText(msg);

	lp->Save();
	lp->SetModified(false);

	msg = "Saved layer to file as " + lp->GetFilename();
	SetStatusText(msg);
}

void MainFrame::OnUpdateLayerSaveAs(wxUpdateUIEvent& event)
{
	event.Enable(GetActiveLayer() != NULL);
}

void MainFrame::OnLayerImport(wxCommandEvent &event)
{
	// first ask what kind of data layer
	LayerType t = AskLayerType();
	if (t != LT_UNKNOWN)
		ImportData(t);
}

void MainFrame::OnLayerProperties(wxCommandEvent &event)
{
	vtLayerPtr lp = GetActiveLayer();
	if (!lp)
		return;

	//
	// Currently, we only have a dialog for Elevation layer properties
	//
	LayerType ltype = lp->GetType();
	if (ltype == LT_ELEVATION)
	{
		vtElevLayer *pEL = (vtElevLayer *)lp;
		vtElevationGrid *grid = pEL->m_pGrid;

		ElevPropDlg dlg(NULL, -1, "Elevation Layer Properties", wxDefaultPosition);

		// Fill in initial values for the dialog
		DRECT rect, rect2;
		pEL->GetExtent(rect);
		dlg.m_fLeft = rect.left;
		dlg.m_fTop = rect.top;
		dlg.m_fRight = rect.right;
		dlg.m_fBottom = rect.bottom;

		int cols, rows;
		grid->GetDimensions(cols, rows);
		wxString str;
		str.Printf("Grid size: %d x %d\n", cols, rows);
		dlg.m_strText += str;

		str.Printf("Floating point: %s\n", grid->IsFloatMode() ? "Yes" : "No");
		dlg.m_strText += str;

		grid->ComputeHeightExtents();
		float fMin, fMax;
		grid->GetHeightExtents(fMin, fMax);
		str.Printf("Minimum elevation: %f\n", fMin);
		dlg.m_strText += str;
		str.Printf("Maximum elevation: %f\n", fMax);
		dlg.m_strText += str;

		const char *dem_name = grid->GetDEMName();
		if (*dem_name)
		{
			str.Printf("Original DEM name: \"%s\"\n", dem_name);
			dlg.m_strText += str;
		}

		if (dlg.ShowModal() == wxID_OK)
		{
			rect2.left = dlg.m_fLeft;
			rect2.top = dlg.m_fTop;
			rect2.right = dlg.m_fRight;
			rect2.bottom = dlg.m_fBottom;
			if (rect2 != rect)
			{
				pEL->m_pGrid->SetGridExtents(rect2);
				m_pView->Refresh();
			}
		}
	}
}

void MainFrame::OnLayerExport(wxCommandEvent &event)
{
	vtLayerPtr lp = GetActiveLayer();
	LayerType ltype = lp->GetType();
	if (ltype == LT_ELEVATION)
	{
		ExportElevation();
	}
}

void MainFrame::OnUpdateLayerExport(wxUpdateUIEvent& event)
{
	event.Enable(GetActiveLayer() != NULL);
}

void MainFrame::OnLayerConvert(wxCommandEvent &event)
{
	// ask for what projection to convert to
	ProjectionDlg dlg(NULL, -1, "Projection", wxDefaultPosition);
	dlg.m_strCaption = "Convert to what projection?";
	dlg.SetProjection(m_proj);

	// might go from geo to utm, provide a good guess for UTM zone
	if (m_proj.IsGeographic())
	{
		DRECT extents = GetMainFrame()->GetExtents();
		dlg.m_iZone = GuessZoneFromLongitude((extents.left + extents.right) / 2.0);
	}

	if (dlg.ShowModal() == wxID_CANCEL)
		return;
	vtProjection proj;
	dlg.GetProjection(proj);

	// count through the layer array, converting
	int layers = m_Layers.GetSize();
	int succeeded = 0;
	for (int i = 0; i < layers; i++)
	{
		vtLayerPtr lp = m_Layers.GetAt(i);
		bool success = lp->ConvertProjection(proj);
		if (success)
			succeeded++;
	}
	if (succeeded < layers)
	{
		wxString str;
		if (layers == 1)
			str.Printf("Failed to convert.");
		else
			str.Printf("Failed to convert %d of %d layers.", layers-succeeded, layers);
		wxMessageBox(str);
	}

	SetProjection(proj);
	m_pView->ZoomAll();
}

void MainFrame::OnLayerSetProjection(wxCommandEvent &event)
{
	// Allow the user to directly specify the projection for all loaded
	// layers (override it, without reprojecting the layer's data)
	// ask for what projection to convert to
	ProjectionDlg dlg(NULL, -1, "Projection", wxDefaultPosition);
	dlg.m_strCaption = "Set to what projection?";
	dlg.SetProjection(m_proj);

	if (m_proj.IsGeographic())
	{
		// might go from geo to utm, provide a good guess for UTM zone
		DRECT extents = GetMainFrame()->GetExtents();
		dlg.m_iZone = GuessZoneFromLongitude((extents.left + extents.right) / 2.0);
	}

	if (dlg.ShowModal() == wxID_CANCEL)
		return;
	vtProjection proj;
	dlg.GetProjection(proj);

	// count through the layer array, converting
	int layers = m_Layers.GetSize();
	for (int i = 0; i < layers; i++)
	{
		vtLayerPtr lp = m_Layers.GetAt(i);
		lp->SetProjection(proj);
	}

	SetProjection(proj);
	m_pView->ZoomAll();
}

void MainFrame::OnUpdateLayerConvert(wxUpdateUIEvent& event)
{
	event.Enable(m_Layers.GetSize() != 0);
}

void MainFrame::OnLayerFlatten(wxCommandEvent &event)
{
	vtLayer *pActive = GetActiveLayer();
	LayerType t = pActive->GetType();

	// count down through the layer array, flattening
	int layers = m_Layers.GetSize();
	for (int i = layers-1; i >= 0; i--)
	{
		vtLayer *pL = m_Layers.GetAt(i);
		if (pL == pActive) continue;
		if (pL->GetType() != t) continue;

//		TRACE("Merging layer %s/%x with %s/%x\n",
//			pL->GetFilename(), pL, pActive->GetFilename(), pActive);
		pActive->AppendDataFrom(pL);
		RemoveLayer(pL);
	}

	wxString newname = "untitled";
	newname += vtLayer::LayerFileExtension[t];
	pActive->SetFilename(newname);
	pActive->SetModified(true);
}

void MainFrame::OnUpdateLayerFlatten(wxUpdateUIEvent& event)
{
	vtLayerPtr lp = GetActiveLayer();
	event.Enable(lp &&
		(lp->GetType() == LT_ROAD ||
		 lp->GetType() == LT_VEG ||
		 lp->GetType() == LT_BUILDING));
}


////////////////////////////////////////////////////////////
// View menu

void MainFrame::OnViewToolbar(wxCommandEvent &event)
{
	toolBar_main->Show( !toolBar_main->IsShown() );

	wxSizeEvent dummy;
	wxFrame::OnSize(dummy);
}

void MainFrame::OnUpdateToolbar(wxUpdateUIEvent& event)
{
	event.Check( toolBar_main->IsShown() );
}

void MainFrame::OnLayerShow(wxCommandEvent &event)
{
	vtLayerPtr pLayer = GetActiveLayer();
	pLayer->SetVisible(!pLayer->GetVisible());

	DRECT r;
	pLayer->GetExtent(r);
	wxRect sr = m_pView->WorldToWindow(r);
	IncreaseRect(sr, 1);
	m_pView->Refresh(TRUE, &sr);
}

void MainFrame::OnUpdateLayerShow(wxUpdateUIEvent& event)
{
	vtLayerPtr pLayer = GetActiveLayer();

	event.Enable(pLayer != NULL);
	event.Check(pLayer && pLayer->GetVisible());
}

void MainFrame::OnViewMagnifier(wxCommandEvent &event)
{
	m_pView->SetMode(LB_Mag);
	m_pView->SetCorrectCursor();
}

void MainFrame::OnUpdateMagnifier(wxUpdateUIEvent& event)
{
	event.Check( m_pView->GetMode() == LB_Mag );
}

void MainFrame::OnViewPan(wxCommandEvent &event)
{
	m_pView->SetMode(LB_Pan);
	m_pView->SetCorrectCursor();
}

void MainFrame::OnUpdatePan(wxUpdateUIEvent& event)
{
	event.Check( m_pView->GetMode() == LB_Pan );
}

void MainFrame::OnViewDistance(wxCommandEvent &event)
{
	m_pView->SetMode(LB_Dist);
	m_pView->SetCorrectCursor();
}

void MainFrame::OnUpdateDistance(wxUpdateUIEvent& event)
{
	event.Check( m_pView->GetMode() == LB_Dist );
}

void MainFrame::OnViewZoomIn(wxCommandEvent &event)
{
	m_pView->SetScale(m_pView->GetScale() * 1.2);
}

void MainFrame::OnViewZoomOut(wxCommandEvent &event)
{
	m_pView->SetScale(m_pView->GetScale() / 1.2);
}

void MainFrame::OnViewZoomAll(wxCommandEvent &event)
{
	m_pView->ZoomAll();
}

void MainFrame::OnViewFull(wxCommandEvent& event)
{
	vtElevLayer *pEL = GetActiveElevLayer();
	if (pEL)
		m_pView->MatchZoomToElev(pEL);
}

void MainFrame::OnUpdateViewFull(wxUpdateUIEvent& event)
{
	vtLayerPtr lp = GetActiveLayer();
	event.Enable(lp && (lp->GetType() == LT_ELEVATION || lp->GetType() == LT_IMAGE));
}

void MainFrame::OnViewWorldMap()
{
	m_pView->m_bShowMap = !m_pView->m_bShowMap;
	m_pView->Refresh();
}

void MainFrame::OnUpdateWorldMap(wxUpdateUIEvent& event)
{
	event.Check(m_pView->m_bShowMap);
}

void MainFrame::OnViewUTMBounds()
{
	m_pView->m_bShowUTMBounds = !m_pView->m_bShowUTMBounds;
	m_pView->Refresh();
}

void MainFrame::OnUpdateUTMBounds(wxUpdateUIEvent& event)
{
	event.Check(m_pView->m_bShowUTMBounds);
}

void MainFrame::OnViewMinutes()
{
	m_bShowMinutes = !m_bShowMinutes;
	m_statbar->SetShowMinutes(m_bShowMinutes);
}

void MainFrame::OnUpdateMinutes(wxUpdateUIEvent& event)
{
	event.Check(m_bShowMinutes);
}

//////////////////////////
// Road

void MainFrame::OnSelectRoad(wxCommandEvent &event)
{
	m_pView->SetMode(LB_Road);
}

void MainFrame::OnUpdateSelectRoad(wxUpdateUIEvent& event)
{
	event.Check( m_pView->GetMode() == LB_Road );
}

void MainFrame::OnSelectNode(wxCommandEvent &event)
{
	m_pView->SetMode(LB_Node);
}

void MainFrame::OnUpdateSelectNode(wxUpdateUIEvent& event)
{
	event.Check( m_pView->GetMode() == LB_Node );
}

void MainFrame::OnSelectWhole(wxCommandEvent &event)
{
	m_pView->SetMode(LB_RoadExtend);
}

void MainFrame::OnUpdateSelectWhole(wxUpdateUIEvent& event)
{
	event.Check( m_pView->GetMode() == LB_RoadExtend );
}

void MainFrame::OnDirection(wxCommandEvent &event)
{
	m_pView->SetMode(LB_Dir);
}

void MainFrame::OnUpdateDirection(wxUpdateUIEvent& event)
{
	event.Check( m_pView->GetMode() == LB_Dir );
}

void MainFrame::OnRoadEdit(wxCommandEvent &event)
{
	m_pView->SetMode(LB_RoadEdit);
}

void MainFrame::OnUpdateRoadEdit(wxUpdateUIEvent& event)
{
	event.Check( m_pView->GetMode() == LB_RoadEdit );
}

void MainFrame::OnShowNodes(wxCommandEvent &event)
{
	bool state = vtRoadLayer::GetDrawNodes();
	vtRoadLayer::SetDrawNodes(!state);
	m_pView->Refresh(state);
}

void MainFrame::OnUpdateShowNodes(wxUpdateUIEvent& event)
{
	event.Check(vtRoadLayer::GetDrawNodes());
}

void MainFrame::OnSelectHwy(wxCommandEvent &event)
{
	vtRoadLayer *pRL = GetActiveRoadLayer();
	if (!pRL) return;

	wxTextEntryDialog dlg(this, "Please enter highway number", "Select Highway", "");
	if (dlg.ShowModal() == wxID_OK)
	{
		int num;
		wxString str = dlg.GetValue();
		sscanf(str, "%d", &num);
		if (pRL->SelectHwyNum(num))
			m_pView->Refresh();
	}
}

void MainFrame::OnRoadClean(wxCommandEvent &event)
{
	vtRoadLayer *pRL = GetActiveRoadLayer();
	if (!pRL) return;

	int count;
	wxString str;
	OpenProgressDialog("Cleaning RoadMap");

	UpdateProgressDialog(10, "Removing unused nodes");
	count = pRL->RemoveUnusedNodes();
	if (count)
	{
		str = wxString::Format("Removed %i nodes\n", count);
		wxMessageBox(str, "", wxOK);
	}

	UpdateProgressDialog(20, "Merging redundant nodes");
	count = pRL->MergeRedundantNodes(progress_callback); // potentially takes a long time...
	if (count)
	{
		str = wxString::Format("Merged %d redundant roads", count);
		wxMessageBox(str, "", wxOK);
	}

	UpdateProgressDialog(30, "Removing degenerate roads");
	count = pRL->RemoveDegenerateRoads();
	if (count)
	{
		str = wxString::Format("Removed %d degenerate roads", count);
		wxMessageBox(str, "", wxOK);
	}

	UpdateProgressDialog(40, "Removing unnecessary nodes");
	count = pRL->RemoveUnnecessaryNodes();
	if (count)
	{
		str = wxString::Format("Removed %d unnecessary nodes", count);
		wxMessageBox(str, "", wxOK);
	}

	UpdateProgressDialog(50, "Cleaning road points");
	count = pRL->CleanRoadPoints();
	if (count)
	{
		str = wxString::Format("Cleaned %d road points", count);
		wxMessageBox(str, "", wxOK);
	}

	UpdateProgressDialog(60, "Removing dangling roads");
	count = pRL->DeleteDanglingRoads();
	if (count)
	{
		str = wxString::Format("Removed %i dangling roads", count);
		wxMessageBox(str, "", wxOK);
	}

	UpdateProgressDialog(70, "Fixing overlapped roads");
	count = pRL->FixOverlappedRoads();
	if (count)
	{
		str = wxString::Format("Fixed %i overlapped roads", count);
		wxMessageBox(str, "", wxOK);
	}

	UpdateProgressDialog(80, "Fixing extraneous parallels");
	count = pRL->FixExtraneousParallels();
	if (count)
	{
		str = wxString::Format("Fixed %i extraneous parallels", count);
		wxMessageBox(str, "", wxOK);
	}

	UpdateProgressDialog(90, "Splitting looping roads");
	count = pRL->SplitLoopingRoads();
	if (count)
	{
		str = wxString::Format("Split %d looping roads", count);
		wxMessageBox(str, "", wxOK);
	}

	CloseProgressDialog();
	pRL->SetModified(true);
	pRL->ComputeExtents();

	m_pView->Refresh();
}

//////////////////////////
// Elevation

void MainFrame::OnElevSelect(wxCommandEvent& event)
{
	m_pView->SetMode(LB_TSelect);
}

void MainFrame::OnUpdateElevSelect(wxUpdateUIEvent& event)
{
	event.Check(m_pView->GetMode() == LB_TSelect);
}

void MainFrame::OnElevBox(wxCommandEvent& event)
{
	m_pView->SetMode(LB_Box);
}

void MainFrame::OnUpdateElevBox(wxUpdateUIEvent& event)
{
	event.Check(m_pView->GetMode() == LB_Box);
}

void MainFrame::OnRemoveAboveSea(wxCommandEvent &event)
{
	vtElevLayer *t = GetActiveElevLayer();
	if (!t)
		return;

	vtElevationGrid *grid = t->m_pGrid;
	int iColumns, iRows;
	grid->GetDimensions(iColumns, iRows);
	for (int i = 0; i < iColumns; i++)
	{
		for (int j = 0; j < iRows; j++)
		{
			if (grid->GetFValue(i, j) > 0.0f)
				grid->SetFValue(i, j, INVALID_ELEVATION);
		}
	}
	t->ReRender();

	m_pView->Refresh();
}

void MainFrame::OnUpdateRemoveAboveSea(wxUpdateUIEvent& event)
{
	event.Enable(GetActiveElevLayer() != NULL);
}

void MainFrame::OnFillIn(wxCommandEvent &event)
{
	vtElevLayer *el = GetActiveElevLayer();
	el->FillGaps();
	el->ReRender();
	m_pView->Refresh();
}

void MainFrame::OnUpdateFillIn(wxUpdateUIEvent& event)
{
	event.Enable(GetActiveElevLayer() != NULL);
}

void MainFrame::OnScaleElevation(wxCommandEvent &event)
{
	vtElevLayer *el = GetActiveElevLayer();
	if (!el)
		return;

	ScaleDlg dlg;
	dlg.LoadFromResource(this, "dialog2");
	dlg.m_bScaleUp = true;
	dlg.m_strUpBy = "1.0";
	dlg.m_strDownBy = "1.0";
	if (dlg.ShowModal() != wxID_OK)
		return;

	float fScale;
	if (dlg.m_bScaleUp)
		fScale = atof(dlg.m_strUpBy);
	else
		fScale = 1.0f / atof(dlg.m_strDownBy);

	vtElevationGrid *grid = el->m_pGrid;

	int iColumns, iRows;
	grid->GetDimensions(iColumns, iRows);
	for (int i = 0; i < iColumns; i++)
	{
		for (int j = 0; j < iRows; j++)
		{
			float f = grid->GetFValue(i, j);
			if (f != INVALID_ELEVATION)
				grid->SetFValue(i, j, f * fScale);
		}
	}
	el->SetModified(true);
	el->ReRender();

	m_pView->Refresh();
}

void MainFrame::OnUpdateScaleElevation(wxUpdateUIEvent& event)
{
	event.Enable(GetActiveElevLayer() != NULL);
}

void MainFrame::OnExportTerragen(wxCommandEvent &event)
{
	vtElevLayer *el = GetActiveElevLayer();
	if (!el)
		return;

	wxString filter = "All Files|*.*|";
	AddType(filter, FSTRING_TER);

	// ask the user for a filename
	wxFileDialog saveFile(NULL, "Export Elevation", "", "", filter, wxSAVE);
	bool bResult = (saveFile.ShowModal() == wxID_OK);
	if (!bResult)
		return;
	wxString strPathName = saveFile.GetPath();

	bool success = el->m_pGrid->SaveToTerragen(strPathName);
	if (!success)
	{
		wxMessageBox("Couldn't open file for writing.");
		return;
	}

	wxString str = wxString::Format("Successfully wrote TerraGen file %s", strPathName);
	wxMessageBox(str);
}

void MainFrame::OnUpdateExportTerragen(wxUpdateUIEvent& event)
{
	event.Enable(GetActiveElevLayer() != NULL);
}

void MainFrame::OnAreaStretch(wxCommandEvent &event)
{
	m_pView->AreaStretch();
}

void MainFrame::OnAreaTypeIn(wxCommandEvent &event)
{
	ExtentDlg dlg(NULL, -1, "Edit Area", wxDefaultPosition);
	dlg.SetArea(m_area, !m_proj.IsGeographic());
	if (dlg.ShowModal() == wxID_OK)
	{
		m_area = dlg.m_area;
		m_pView->Refresh();
	}
}

void MainFrame::OnAreaExportBitmap(wxCommandEvent& event)
{
	wxTextEntryDialog dlg(this, "Please enter pixel size of bitmap", "Export Bitmap", "");
	if (dlg.ShowModal() != wxID_OK)
		return;

	int size;
	wxString str = dlg.GetValue();
	sscanf(str, "%d", &size);
	if (size < 64 || size > 8192)
		return;

	// Ask for file name
	wxFileDialog loadFile(NULL, "Output filename for bitmap", "", "",
		"Bitmap Files (*.bmp)|*.bmp|", wxSAVE|wxOVERWRITE_PROMPT);

	if (loadFile.ShowModal() != wxID_OK)
		return;

	OpenProgressDialog("Generating Bitmap");

	wxString fname = loadFile.GetPath();
	vtDIB dib(size, size, 24, false);

	for (int l = 0; l < m_Layers.GetSize(); l++)
	{
		vtLayer *pL = m_Layers.GetAt(l);
		if (pL->GetType() == LT_ELEVATION)
		{
			vtElevLayer *pEL = (vtElevLayer *)pL;

			pEL->PaintDibFromElevation(&dib, true);
			break;	// only use first elevation
		}
	}
	UpdateProgressDialog(100, "Writing bitmap to file.");
	bool success = dib.WriteBMP(fname);
	CloseProgressDialog();
}

void MainFrame::OnElevShow(wxCommandEvent &event)
{
	vtElevLayer::m_bShowElevation = !vtElevLayer::m_bShowElevation;
	m_pView->Refresh();
}

void MainFrame::OnUpdateElevShow(wxUpdateUIEvent& event)
{
	event.Check(vtElevLayer::m_bShowElevation);
}

void MainFrame::OnElevShading(wxCommandEvent &event)
{
	vtElevLayer::m_bShading = !vtElevLayer::m_bShading;
}

void MainFrame::OnUpdateElevShading(wxUpdateUIEvent& event)
{
	event.Check(vtElevLayer::m_bShading);
}

void MainFrame::OnElevHide(wxCommandEvent &event)
{
	vtElevLayer::m_bDoMask = !vtElevLayer::m_bDoMask;

	if (!LayersOfType(LT_ELEVATION))
		return;

	// refresh the display
	for (int i = 0; i < m_Layers.GetSize(); i++)
	{
		vtLayerPtr lp = m_Layers.GetAt(i);
		if (lp->GetType() == LT_ELEVATION)
		{
			vtElevLayer *elp = (vtElevLayer *)lp;
			elp->ReRender();
		}
	}
	m_pView->Refresh();
}

void MainFrame::OnUpdateElevHide(wxUpdateUIEvent& event)
{
	event.Check(vtElevLayer::m_bDoMask);
}

//////////////////////////
// Vegetation

void MainFrame::OnVegPlants(wxCommandEvent& event)
{
	// if PlantList has not previously been open, get the data from file first
	if (!m_PlantListDlg)
	{
		// Use file dialog to open plant list text file.
		wxFileDialog loadFile(NULL, "Load Plant Info", "", "", PLANTS_FILTER, wxOPEN);

		if (loadFile.ShowModal() != wxID_OK)
			return;

		// Read Plantlist file.
		if (!GetPlantList()->Read(loadFile.GetPath()))
		{
			wxMessageBox("Couldn't read plant list from that file.", "Error", wxOK, this);
			return;
		}

		// Create new Plant List Dialog
		m_PlantListDlg = new PlantListDlg(this, WID_PLANTS, "Plants List", 
		wxPoint(140, 100), wxSize(950, 400), wxSYSTEM_MENU | wxCAPTION);
	}

	// Display plant list data, calling OnInitDialog.
	m_PlantListDlg->Show(true);
}


void MainFrame::OnVegBioregions(wxCommandEvent& event)
{
	// if data isn't there, get the data first
	if (!m_BioRegionDlg)
	{
		// Use file dialog to open bioregion text file.
		wxFileDialog loadFile(NULL, "Load BioRegion Info", "", "", BIOREGIONS_FILTER, wxOPEN);

		if (loadFile.ShowModal() != wxID_OK)
			return;

		// Read bioregions, data kept on frame with m_pBioRegion.
		if (!GetBioRegion()->Read(loadFile.GetPath()))
		{
			wxMessageBox("Couldn't read bioregion list from that file.", "Error", wxOK, this);
			return;
		}

		// Create new Bioregion Dialog
		m_BioRegionDlg = new BioRegionDlg(this, WID_BIOREGIONS, "BioRegions List", 
			wxPoint(120, 80), wxSize(300, 500), wxSYSTEM_MENU | wxCAPTION);
	}

	// Display bioregion data, calling OnInitDialog.
	m_BioRegionDlg->Show(true);
}


void MainFrame::OnVegGenerate(wxCommandEvent& event)
{
	// Open File Save Dialog
	wxFileDialog saveFile(NULL, "Save Vegetation File", "", "", "Vegetation Files (*.vf)|*.vf|",
		wxSAVE | wxOVERWRITE_PROMPT );

	bool bResult = (saveFile.ShowModal() == wxID_OK);
	if (!bResult)
		return;
	wxString strPathName = saveFile.GetPath();

	// Generate the plants
	vtVegLayer *Density = NULL, *BioMap = NULL;
	FindvtVegLayers(&Density, &BioMap);

	float fTreeSpacing = 40.0f;
	float fTreeScarcity = 0.001f;

	GenerateVegetation(strPathName, m_area, Density, BioMap, fTreeSpacing, fTreeScarcity);
}

void MainFrame::OnUpdateVegGenerate(wxUpdateUIEvent& event)
{
	vtVegLayer *Density = NULL, *BioMap = NULL;

	FindvtVegLayers(&Density, &BioMap);

	event.Enable(m_PlantListDlg && m_BioRegionDlg && Density && BioMap);
}

void MainFrame::FindvtVegLayers(vtVegLayer **Density, vtVegLayer **BioMap)
{
	for (int i = 0; i < m_Layers.GetSize(); i++)
	{
		vtLayer *lp = m_Layers.GetAt(i);
		if (lp->GetType() == LT_VEG)
		{
			vtVegLayer *veg = (vtVegLayer *)lp;
			VegLayerType vltype = veg->m_VLType;

			if (vltype == VLT_Density)
				*Density = veg;
			if (vltype == VLT_BioMap)
				*BioMap = veg;
		}
	}
}

void MainFrame::GenerateVegetation(const char *vf_file, DRECT area, 
								   vtVegLayer *pLandUse, vtVegLayer *pVegType,
								   float fTreeSpacing, float fTreeScarcity)
{
	int i, j;
	DPoint2 p, p2;

	for (i = 0; i < m_BioRegions.m_Types.GetSize(); i++)
		m_PlantList.LookupPlantIndices(m_BioRegions.m_Types[i]);

	OpenProgressDialog("Generating Vegetation");

	int tree_count = 0;

	float x_size = (area.right - area.left);
	float y_size = (area.top - area.bottom);
	int x_trees = (int)(x_size / fTreeSpacing);
	int y_trees = (int)(y_size / fTreeSpacing);

	int attrib, veg_type;

	vtPlantInstanceArray pia;
//	pia.m_proj.SetUTM(true);

	// all vegetation
	for (i = 0; i < x_trees; i ++)
	{
		wxString str;
		str.Printf("plants: %d", pia.GetSize());
		UpdateProgressDialog(i * 100 / x_trees, str);

		p.x = area.left + (i * fTreeSpacing);
		for (j = 0; j < y_trees; j ++)
		{
			p.y = area.bottom + (j * fTreeSpacing);

			// randomize the position slightly
			p2.x = p.x + random_offset(fTreeSpacing * 0.5f);
			p2.y = p.y + random_offset(fTreeSpacing * 0.5f);

			// Get Land Use Attribute
			attrib = pLandUse->FindAttribute(p2);

			if (attrib < 0 || attrib > 76)
				continue;	// off the map, or ocean

			veg_type = pVegType->FindAttribute(p2);
			if (veg_type == -1)
				continue;

			int i;
			bool wild = false;
			float density_scale;

			float square_meters = fTreeSpacing * fTreeSpacing;

			switch (attrib)
			{
				case 42:	// forest
					wild = true;
					density_scale = 1.0f;
					break;
				case 32:
				case 33:
					wild = true;
					density_scale = 0.5;
					break;
				case 22:	// orchards
#if 0
					// kludge crops based on ugly hard-coded areas
					if (utm_y > 2204500 || utm_y < 2133000)
					{
						return GetSpeciesIdByCommonName("MacNut");
					}
					else if (utm_x > 235000)
					{
						return GetSpeciesIdByCommonName("Papaya");
					}
					else
					{
						return GetSpeciesIdByCommonName("Coffee");
					}
#else
					continue;	// no crops for now
#endif
					break;
			}

			// look at veg_type to decide which BioType to use
			vtBioType *bio = m_BioRegions.m_Types[veg_type];

			float factor = density_scale * square_meters * fTreeScarcity;

			for (i = 0; i < bio->m_Densities.GetSize(); i++)
			{
				vtPlantDensity *pd = bio->m_Densities[i];

				pd->m_amount += (pd->m_plant_per_m2 * factor);
			}
			int species_num = -1;
			for (i = 0; i < bio->m_Densities.GetSize(); i++)
			{
				vtPlantDensity *pd = bio->m_Densities[i];
				if (pd->m_amount > 1.0f)	// time to plant
				{
					pd->m_amount -= 1.0f;
					pd->m_iNumPlanted++;
					species_num = pd->m_list_index;
					break;
				}
			}
			if (species_num != -1)
			{
				vtPlantSpecies *ps = GetPlantList()->GetSpecies(species_num);
				pia.AddInstance(p2, random(ps->GetMaxHeight()), species_num);
			}
		}
	}
	pia.WriteVF(vf_file);
	CloseProgressDialog();
}



/////////////////////////////////////
// Buildings / Features

void MainFrame::OnFeatureSelect(wxCommandEvent &event)
{
	m_pView->SetMode(LB_FSelect);
}

void MainFrame::OnUpdateFeatureSelect(wxUpdateUIEvent& event)
{
	event.Check(m_pView->GetMode() == LB_FSelect);
}

void MainFrame::OnBuildingEdit(wxCommandEvent &event)
{
	m_pView->SetMode(LB_BldEdit);
}

void MainFrame::OnUpdateBuildingEdit(wxUpdateUIEvent& event)
{
	event.Check(m_pView->GetMode() == LB_BldEdit);
}


////////////////////
// Help

void MainFrame::OnHelpAbout(wxCommandEvent &event)
{
	wxString str = "Virtual Terrain Builder\nPowerful, easy to use, free!\n\n";
	str += "Please read the HTML documentation and license.\n\n";
	str += "Send feedback to: ben@vterrain.org\n";
	str += "Build date: ";
	str += __DATE__;
	wxMessageBox(str, "About VTBuilder");
}

//////////////////
// Keyboard shortcuts

void MainFrame::OnChar(wxKeyEvent& event)
{
	m_pView->OnChar(event);
}
