
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <time.h>

#include "endian.h"
#include "action.h"

void error(char *msg, ...);

int indent = 1;
int targetIndent = 0;
int frameLoadedActions = -1;

int swfVersion;
char swfHeader[4];

int nDict = 0;

struct _dict
{
  char *value;
  int count;
};

struct _dict dictionary[MAX_CONSTANTS];
char *flasmDict = NULL;

int numTargets = 0;
int *targets = NULL;
int curTarget = 0;

long int conststart = 0, constend = 0;

#define INDENT_LEVEL 2

extern int mode;
extern char wasCompressed;

enum
{
  DMODE_NORMAL,
  DMODE_CLEAN,        /* remove callFunction, pop or getVariable after macros */
  DMODE_ASM           /* write inline flasm strings down */
};

int dmode = DMODE_NORMAL;

enum
{
    eventLoad		= 0x01,
    eventEnterFrame	= 0x02,
    eventUnload		= 0x04,
    eventMouseMove	= 0x08,
    eventMouseDown	= 0x10,
    eventMouseUp	= 0x20,
    eventKeyDown	= 0x40,
    eventKeyUp		= 0x80,
    eventData		= 0x100,
    eventInitialize     = 0x200, // flash 6 only; flash 5 stores smartClip parameters in eventLoad
    eventPress		= 0x400,
    eventRelease	= 0x800,
    eventReleaseOutside = 0x1000,
    eventRollOver	= 0x2000,
    eventRollOut	= 0x4000,
    eventDragOver	= 0x8000,
    eventDragOut	= 0x10000,
    eventKeyPress	= 0x20000
};

enum
{
    IdleToOverUp      = 0x01,
    OverUpToIdle      = 0x02,
    OverUpToOverDown  = 0x04,
    OverDownToOverUp  = 0x08,
    OverDownToOutDown = 0x10,
    OutDownToOverDown = 0x20,
    OutDownToIdle     = 0x40,
    IdleToOverDown    = 0x80,
    OverDownToIdle    = 0x100,
};

enum
{
    keyLeft      = 1,
    keyRight     = 2,
    keyHome      = 3,
    keyEnd       = 4,
    keyIns       = 5,
    keyDel       = 6,
    keyBackspace = 8,
    keyEnter     = 13,
    keyUp        = 14,
    keyDn        = 15,
    keyPgUp      = 16,
    keyPgDn      = 17,
    keyTab       = 18,
    keySpace     = 32
};

int strIcmp (char *s, char *t)
{
  int d = 0;
  do {
    d = toupper(*s) - toupper(*t);
  } while (*s++ && *t++ && !d);
  return (d);
}

char *strIstr(const char *String, const char *Pattern)
{
  char *pptr, *sptr, *start;
  unsigned int slen, plen;

  if (String == NULL)
    return(NULL);

  for (start = (char *)String, pptr  = (char *)Pattern, slen  = strlen(String), plen  = strlen(Pattern);
       slen >= plen; start++, slen--)
  {
    while (toupper(*start) != toupper(*Pattern))
    {
      start++;
      slen--;

      if (slen < plen)
        return(NULL);
    }

    sptr = start;
    pptr = (char *)Pattern;

    while (toupper(*sptr) == toupper(*pptr))
    {
      sptr++;
      pptr++;

      if ('\0' == *pptr)
        return (start);
    }
  }
  return(NULL);
}

void skipBuffered(FILE *f, unsigned long int length) {
  unsigned char skipBuffer[MAX_BUFFER];

  while(length>MAX_BUFFER)
    length -= fread(skipBuffer, 1, MAX_BUFFER, f);

  if (length <= 0)
    return;

  fread(skipBuffer, 1, (unsigned int)length, f);
}

void print (char *s, ...)
{
  va_list ap;
  int n = indent*INDENT_LEVEL;

  while(n-- > 0)
    putchar(' ');

  va_start(ap, s);
  vprintf(s, ap);
  va_end(ap);
}

void println (char *s, ...)
{
  va_list ap;
  int n = indent*INDENT_LEVEL;

  while(n-- > 0)
    putchar(' ');

  va_start(ap, s);
  vprintf(s, ap);
  va_end(ap);

  putchar('\n');
}

void printstr (char *str)
{
  int n = 0;
  
  putchar('\'');
  while(str[n++] != 0) {
    switch(str[n-1]) {
      case 8:
        printf("%s","\\b");
        break;
      case 9:
        printf("%s","\\t");
        break;
      case 10:
        printf("%s","\\n");
        break;
      case 12:
        printf("%s","\\f");
        break;
      case 13:
        printf("%s","\\r");
        break;
      case 39:
        printf("%s","\\'");
        break;
      case 92:
        if (dmode == DMODE_CLEAN)
          printf("%s","\\"); /* no double backslashes in $include filenames */
        else
          printf("%s","\\\\");
        break;
      default:
        putchar(str[n-1]);
    }
  }
  putchar('\'');
}

void printFloat (float f, int intCast)
{
  char s[60];
  char *sp, *xpn;

  if (f != f)
    printf("_NANF");
  else if (f == (1.0f/0.0f))
      printf("POSITIVE_INFINITYF");
  else if (f == (-1.0f/0.0f))
      printf("NEGATIVE_INFINITYF");
  else
  {
    sprintf(s,"%#g",f);
	      
    if ((xpn = strchr(s, 'e')) == NULL)
      sp = s + strlen(s) - 1;
    else
      sp = xpn-2;
	      
    while (*sp == '0') --sp;

    if (intCast)
    {
      if (*sp == '.') --sp;
      *(sp+1) = '\0';
      printf("%s", s);
    }
    else
    {
      if (*sp == '.') *++sp = '0';
      *++sp = '\0';
      printf("%s", s);
      if (xpn!=NULL) printf("%s", xpn);
      putchar('f');
    }
  }
}

signed int bitPos;
unsigned int bitBuf;

void InitBits(FILE *f)
{
  bitPos = 8;
  bitBuf = (unsigned int)fgetc(f);
}

unsigned int getWord (FILE *f) {
  return ((unsigned int)fgetc(f)+((unsigned int)fgetc(f)<<8));
}

unsigned long int getDoubleWord (FILE *f) {
  return ((unsigned long int)getWord(f)+((unsigned long int)getWord(f)<<16));
}

int getBits (FILE *f, unsigned int n)
{
  unsigned long int v = 0;

  while (1)
  {
    int s = n - bitPos;

    if (s > 0)
    {
      v |= bitBuf << s;
      n -= bitPos;
      bitBuf = (unsigned int)fgetc(f);
      bitPos = 8;
    }
    else
    {
      v |= bitBuf >> -s;
      bitPos -= n;
      bitBuf &= 0xff >> (8 - bitPos);
      return v;
    }
  }
}

void printFrameNum(int frameNum)
{
  int n = indent*INDENT_LEVEL;
  
  printf("\n");
  while(n-- > 0)
    putchar(' ');

  printf("frame %i\n", frameNum);
}

void printLabel()
{
  int n = indent*INDENT_LEVEL-1;

  while(n-- > 0)
    putchar(' ');

  if (mode<MODE_UPDATE)
    printf("label%i:\n", ++curTarget);
  else
    printf("lbl%i:\n", ++curTarget); /* make sure generated labels don't match user labels */
}

static void addTarget(int offset)
{
  int i;

  for(i=0; i<numTargets; ++i)
  {
    if(targets[i] == offset)
      break;
  }

  if(i < numTargets)
    return;

  targets = realloc(targets, (numTargets+1) * sizeof(int));
  targets[numTargets++] = offset;
}

static int findLabel(int offset)
{
  int i;

  for(i=0; i<numTargets; ++i)
  {
    if(offset == targets[i])
      return i;
  }

  error("Unexpected target offset: %i", offset);
  return 0;
}

int int_compare(const void *ap, const void *bp)
{
  int a = *(int *)ap;
  int b = *(int *)bp;

  if(a < b)
    return -1;
  else if(a > b)
    return 1;
  else
    return 0;
}

void processASMLine(char *line)
{
  char *ci = strIstr(line, "constants ");

  if ((ci == NULL)||(constend == 0)) /* line contains no 'constants' or prev constant declaration absent */
    printf(line);                    /* dump flasm statement */
  else
    if (flasmDict == NULL)
      flasmDict = strdup(ci+10);     /* add user constants */
    else
    {
      flasmDict = realloc(flasmDict, strlen(flasmDict)+strlen(ci+10)+3);
      strcat(flasmDict, ", ");
      strcat(flasmDict, ci+10);
    }
}

void includeFile(char *ifilename)
{
  FILE *ifile;
  char buf[256];

  if((ifile = fopen(ifilename, "rt")) == NULL)
    error("Couldn't include file: %s", ifilename);

  printf("\n// start of %s\n",ifilename);

  while (fgets(buf, 256, ifile))
    processASMLine(buf);

  printf("\n// end of %s\n",ifilename);

  fclose(ifile);
}

unsigned char *buffer;
unsigned int buflen;

/* worst.. macro name.. ever */
#define S16(p)       ((p)[0] + (((p)[1])<<8))
#define S16signed(p) ((p)[0] + ((signed char)((p)[1])<<8))

void printActionRecord(unsigned char *p, Action type, unsigned int *lenptr);

void printActions(unsigned char *p, unsigned long int length)
{
  unsigned long int i=0;
  unsigned int blocklen;

  while(i<length)
  {
    int type = (unsigned char)p[i];

    if(curTarget < numTargets && p+i-buffer == targets[curTarget])
      printLabel();

    ++i;

    if((type & 0x80) != 0)
    {
      blocklen = S16(p+i);
      i += 2;
    }
    else
      blocklen = 0;

    if (frameLoadedActions>-1)
    {
      if (frameLoadedActions == 0)
      {
	 --indent;
	 println("end");
      }
      --frameLoadedActions;
    }

    printActionRecord(p+i, type, &blocklen);

    i += blocklen;
  }

  if(curTarget < numTargets && p+i-buffer == targets[curTarget])
    printLabel();
}

void printActionRecord(unsigned char *p, Action type, unsigned int *lenptr)
{
  unsigned int len = *lenptr;

  if ((mode >= MODE_UPDATE) && (dmode == DMODE_ASM)) {
    /* processing $flasm */
    if (type == SWFACTION_END)
      error("Unexpected end of action block found while processing flasm macro");
    if (type != SWFACTION_PUSHDATA)
      return; /* skip all actions but push */
    if ((*p!=0)&&(*p!=8)&&(*p!=9))
      return; /* skip all push types but string and dictionary */
  }

  switch(type)
  {
    case SWFACTION_ADD:
      println("oldAdd");
      break;
    case SWFACTION_SUBTRACT:
      println("subtract");
      break;
    case SWFACTION_MULTIPLY:
      println("multiply");
      break;
    case SWFACTION_DIVIDE:
      println("divide");
      break;
    case SWFACTION_EQUALS:
      println("oldEquals");
      break;
    case SWFACTION_LESSTHAN:
      println("oldLessThan");
      break;
    case SWFACTION_LOGICALAND:
      println("and");
      break;
    case SWFACTION_LOGICALOR:
      println("or");
      break;
    case SWFACTION_LOGICALNOT:
      println("not");
      break;
    case SWFACTION_STRINGEQ:
      println("stringEq");
      break;
    case SWFACTION_STRINGLENGTH:
      println("stringLength");
      break;
    case SWFACTION_SUBSTRING:
      println("substring");
      break;
    case SWFACTION_INT:
      println("int");
      break;
    case SWFACTION_POP:
      if ((mode >= MODE_UPDATE) && (dmode == DMODE_CLEAN))
        dmode = DMODE_NORMAL;
      else
        println("pop");
      break;
    case SWFACTION_SWAP:
      println("swap");
      break;
    case SWFACTION_INITOBJECT:
      println("initObject");
      break;
    case SWFACTION_INITARRAY:
      println("initArray");
      break;
    case SWFACTION_GETVARIABLE:
      if (!((mode >= MODE_UPDATE) && (dmode == DMODE_CLEAN)))
        println("getVariable");
      break;
    case SWFACTION_SETVARIABLE:
      println("setVariable");
      break;
    case SWFACTION_SETTARGETEXPRESSION:
      if (targetIndent==1)
      {
        --indent;
        println("end");
        targetIndent = 0;
      }
      println("setTargetExpr");
      ++indent;
      targetIndent = 1;
      break;
    case SWFACTION_STRINGCONCAT:
      println("concat");
      break;
    case SWFACTION_GETPROPERTY:
      println("getProperty");
      break;
    case SWFACTION_SETPROPERTY:
      println("setProperty");
      break;
    case SWFACTION_DUPLICATECLIP:
      println("duplicateClip");
      break;
    case SWFACTION_REMOVECLIP:
      println("removeClip");
      break;
    case SWFACTION_TRACE:
      println("trace");
      break;
    case SWFACTION_STARTDRAGMOVIE:
      println("startDrag");
      break;
    case SWFACTION_STOPDRAGMOVIE:
      println("stopDrag");
      break;
    case SWFACTION_STRINGLESSTHAN:
      println("stringLessThan");
      break;
    case SWFACTION_STRINGGREATERTHAN:
      println("stringGreaterThan");
      break;
    case SWFACTION_RANDOM:
      println("random");
      break;
    case SWFACTION_MBLENGTH:
      println("mbLength");
      break;
    case SWFACTION_ORD:
      println("ord");
      break;
    case SWFACTION_CHR:
      println("chr");
      break;
    case SWFACTION_GETTIMER:
      println("getTimer");
      break;
    case SWFACTION_MBSUBSTRING:
      println("mbSubstring");
      break;
    case SWFACTION_MBORD:
      println("mbOrd");
      break;
    case SWFACTION_MBCHR:
      println("mbChr");
      break;
    case SWFACTION_NEXTFRAME:
      println("nextFrame");
      break;
    case SWFACTION_PREVFRAME:
      println("prevFrame");
      break;
    case SWFACTION_PLAY:
      println("play");
      break;
    case SWFACTION_STOP:
      println("stop");
      break;
    case SWFACTION_TOGGLEQUALITY:
      println("toggleQuality");
      break;
    case SWFACTION_STOPSOUNDS:
      println("stopSounds");
      break;

    /* ops with args */
    case SWFACTION_PUSHDATA:
    {
      int type;
      unsigned char *start = p;
      long int pushstart;
      int n = 0;

      /* may need to go back and erase push while processing flasm macros */
      pushstart = ftell(stdout);

      print("push ");

      while(p < start+len)
      {
        switch(type = *p++)
        {
          case 0: /* string */
            if (mode >= MODE_UPDATE) {
              if (strIcmp(p,"$include")==0) {       /* $include("foo.flm") found              */
                dmode = DMODE_CLEAN;                /* skip callFunction/pop later            */
                fseek(stdout, pushstart, SEEK_SET); /* go back to overwrite 'push'            */
                includeFile(start+1);
                p = start+len;                      /* skip to the end of push statement      */
                break;
              }
              if (strIcmp(p,"$flasm")==0) {         /* $flasm found                           */
                dmode = DMODE_ASM;                  /* dump all strings until $end found      */
                fseek(stdout, pushstart, SEEK_SET); /* go back to overwrite 'push'            */
                printf("// start of inline assembly");
                p = start+len;                      /* skip to the end of push statement      */
                break;
              }
              if (strIcmp(p,"$end")==0) {           /* $end found                             */
                dmode = DMODE_CLEAN;                /* skip getVariable later                 */
                fseek(stdout, pushstart, SEEK_SET); /* go back to overwrite 'push'            */
                printf("// end of inline assembly\n");
                p = start+len;                      /* skip to the end of push statement      */
                break;
              }
              if (dmode == DMODE_ASM) {
                fseek(stdout, pushstart, SEEK_SET); /* go back to overwrite 'push'            */
                processASMLine(p);
                p = start+len;                      /* skip to the end of push statement      */
                break;
              } 
            }

            if (n++>0) printf("%s",", ");
            printstr(p);
            p += strlen(p)+1;
            break;

          case 1: /* float, used by flash for properties only */
          {
            float f;
	    double prop;

            if (byteorder == FLASM_BIG_ENDIAN)
            {
              unsigned char *fp = (unsigned char *)(&f);

              fp[0] = p[3];
              fp[1] = p[2];
              fp[2] = p[1];
              fp[3] = p[0];
            }
            else
              f = *(float *)p;
 
	    printf("%s",(n++>0)?", ":"");
	    
	    if (modf((double)f, &prop) == 0)
	    { /* integer, most likely property */
              switch((int)prop) {
                case 0:
                  printf("X_PROPERTY");
                  break;
                case 1:
                  printf("Y_PROPERTY");
                  break;
                case 2:
                  printf("XSCALE_PROPERTY");
                  break;
                case 3:
                  printf("YSCALE_PROPERTY");
                  break;
                case 4:
                  printf("CURRENTFRAME_PROPERTY");
                  break;
                case 5:
                  printf("TOTALFRAMES_PROPERTY");
                  break;
                case 6:
                  printf("ALPHA_PROPERTY");
                  break;
                case 7:
                  printf("VISIBLE_PROPERTY");
                  break;
                case 8:
                  printf("WIDTH_PROPERTY");
                  break;
                case 9:
                  printf("HEIGHT_PROPERTY");
                  break;
                case 10:
                  printf("ROTATION_PROPERTY");
                  break;
                case 11:
                  printf("TARGET_PROPERTY");
                  break;
                case 12:
                  printf("FRAMESLOADED_PROPERTY");
                  break;
                case 13:
                  printf("NAME_PROPERTY");
                  break;
                case 14:
                  printf("DROPTARGET_PROPERTY");
                  break;
                case 15:
                  printf("URL_PROPERTY");
                  break;
                case 16:
                  printf("HIGHQUALITY_PROPERTY");
                  break;
                case 17:
                  printf("FOCUSRECT_PROPERTY");
                  break;
                case 18:
                  printf("SOUNDBUFTIME_PROPERTY");
                  break;
                case 19:
                  printf("QUALITY_PROPERTY");
                  break;
                case 20:
                  printf("XMOUSE_PROPERTY");
                  break;
                case 21:
                  printf("YMOUSE_PROPERTY");
                  break;
                default:
                  printFloat(f,0);
	      }
	    }
	    else
	      printFloat(f,0);

            p += 4;
            break;
          }

          case 2: /* null */
            printf("%sNULL", (n++>0)?", ":"");
            break;

          case 3: /* undefined */
            printf("%sUNDEF", (n++>0)?", ":"");
            break;

          case 4: /* register */
            printf("%sr:%i", (n++>0)?", ":"", *p++);
            break;

          case 5: /* boolean */
            if(*p++)
              printf("%sTRUE", (n++>0)?", ":"");
            else
              printf("%sFALSE", (n++>0)?", ":"");
            break;

          case 6: /* double */
          {
            double d;
            unsigned char *dp = (unsigned char *)(&d);
            char s[100];
            char *sp, *xpn;

            if(byteorder == FLASM_BIG_ENDIAN)
            {
              dp[0] = p[3];
              dp[1] = p[2];
              dp[2] = p[1];
              dp[3] = p[0];
              dp[4] = p[7];
              dp[5] = p[6];
              dp[6] = p[5];
              dp[7] = p[4];
            }
            else
            {
              dp[0] = p[4];
              dp[1] = p[5];
              dp[2] = p[6];
              dp[3] = p[7];
              dp[4] = p[0];
              dp[5] = p[1];
              dp[6] = p[2];
              dp[7] = p[3];
            }

	     /* the old way without '1.5e-24'-like notation
	        fprec = 15-floor(log10(fabs(d)));
 	        sprintf(s,"%.*f",fprec,d);
	     */

            printf("%s", (n++>0)?", ":"");

	    if (d==0)
	    {
	      if (mode < MODE_UPDATE)
	        printf("0.0");
	      else
		printf("0"); /* save some bytes in update mode - integer instead of double */
	    }
	    else if (d != d)
	      printf("_NAN");
	    else if (d == (1.0/0.0))
	      printf("POSITIVE_INFINITY");
	    else if (d == (-1.0/0.0))
	      printf("NEGATIVE_INFINITY");
	    else {
	      sprintf(s,"%#.*g",16,d);
	      
	      if ((xpn = strchr(s, 'e')) == NULL)
	        sp = s + strlen(s) - 1;
	      else
		sp = xpn-2;	    /* one digit less precision for exp form values
				       preventing MAX_NUMBER rounding to INFINITY */
	      
              while (*sp == '0')    /* delete 0's at the end of the number or mantissa */
	        --sp;

	      if (*sp == '.')       /* expand values like "1." to "1.0" */
	        *++sp = '0';

              *++sp = '\0';	    /* terminate buffer (exponent is cutted off) */

	      printf("%s", s);

	      if (xpn!=NULL)	    /* if exponent here, print it */
	        printf("%s", xpn);
	    }

            p += 8;
            break;
          }

          case 7: /* integer */
          {
            int i;

            if(byteorder == FLASM_BIG_ENDIAN)
            {
              unsigned char *ip = (unsigned char *)(&i);

              ip[0] = p[3];
              ip[1] = p[2];
              ip[2] = p[1];
              ip[3] = p[0];
            }
            else
              i = *(int *)p;

            printf("%s%i", (n++>0)?", ":"", i);
            p += 4;
            break;
          }

          case 8: /* dictionary, 1-byte reference */
          {
            unsigned char *d = dictionary[(unsigned char)*p].value;

            if (mode >= MODE_UPDATE) {
              if (strIcmp(d,"$include")==0) {                           /* $include("foo.flm") found         */
                dmode = DMODE_CLEAN;                                    /* skip callFunction/pop later       */
                fseek(stdout, pushstart, SEEK_SET);                     /* go back to overwrite 'push'       */
                includeFile(dictionary[(unsigned char)*(start+1)].value);
                p = start+len;                                          /* skip to the end of push statement */
                break;
              }
              if (strIcmp(d,"$flasm")==0) {                             /* $flasm found                      */
                dmode = DMODE_ASM;                                      /* dump all strings until $end found */
                fseek(stdout, pushstart, SEEK_SET);                     /* go back to overwrite 'push'       */
                printf("// start of inline assembly");
                p = start+len;                                          /* skip to the end of push statement */
                break;
              }
              if (strIcmp(d,"$end")==0) {                               /* $end found                        */
                dmode = DMODE_CLEAN;                                    /* skip getVariable later            */
                fseek(stdout, pushstart, SEEK_SET);                     /* go back to overwrite 'push'       */
                printf("// end of inline assembly\n");
                p = start+len;                                          /* skip to the end of push statement */
                break;
              }
              if (dmode == DMODE_ASM) {
                fseek(stdout, pushstart, SEEK_SET);                     /* go back to overwrite 'push'       */
                processASMLine(d);
                p = start+len;                                          /* skip to the end of push statement */
                break;
              } 
            }

            if (n++>0) printf("%s",", ");
            printstr(dictionary[(unsigned char)*p].value);
            dictionary[(unsigned char)*p].count++;                      /* constant used one more time */
            p++;
            break;
          }

          case 9: /* dictionary, 2-byte reference */
          {
            unsigned char *d = dictionary[S16(p)].value;
            
            if (mode >= MODE_UPDATE) {
              if (strIcmp(d,"$include")==0) {                           /* $include("foo.flm") found         */
                dmode = DMODE_CLEAN;                                    /* skip callFunction/pop later       */
                fseek(stdout, pushstart, SEEK_SET);                     /* go back to overwrite 'push'       */
                if (((unsigned char)*(start)) == 8)                     /* what push type has file name?     */
                  includeFile(dictionary[(unsigned char)*(start+1)].value); /* include file 1-byte reference */
                else
                  includeFile(dictionary[S16(start+1)].value);          /* include file, 2-byte reference    */

                p = start+len;                                          /* skip to the end of push statement */
                break;
              }
              if (strIcmp(d,"$flasm")==0) {                             /* $flasm found                      */
                dmode = DMODE_ASM;                                      /* dump all strings until $end found */
                fseek(stdout, pushstart, SEEK_SET);                     /* go back to overwrite 'push'       */
                printf("// start of inline assembly");
                p = start+len;                                          /* skip to the end of push statement */
                break;
              }
              if (strIcmp(d,"$end")==0) {                               /* $end found                        */
                dmode = DMODE_CLEAN;                                    /* skip getVariable later            */
                fseek(stdout, pushstart, SEEK_SET);                     /* go back to overwrite 'push'       */
                printf("// end of inline assembly\n");
                p = start+len;                                          /* skip to the end of push statement */
                break;
              }
              if (dmode == DMODE_ASM) {
                fseek(stdout, pushstart, SEEK_SET);                     /* go back to overwrite 'push'       */
                processASMLine(d);
                p = start+len;                                          /* skip to the end of push statement */
                break;
              } 
            }

            if (n++>0) printf("%s",", ");
            printstr(dictionary[S16(p)].value);
            dictionary[S16(p)].count++;                                 /* constant used one more time */
            p+=2;
            break;
          }

          default:
            printf("%s%s // unknown push type %i: the rest of push skipped!", (n++>0)?", ":"", "???", type);
	    p = start+len;
        }
      }

      putchar('\n');

      break;
    }

    case SWFACTION_GOTOFRAME:
      println("gotoFrame %i", S16(p));
      p+=2;
      break;

    case SWFACTION_GETURL:
    {
      char *url = p;
      p += strlen(p) + 1;
      print("getURL ");
      printstr(url);
      putchar(' ');
      printstr(p);
      putchar('\n');
      break;
    }

    case SWFACTION_IFFRAMELOADEDEXPRESSION:
      println("ifFrameLoadedExpr");
      ++indent;
      frameLoadedActions = *p++;
      break;

    case SWFACTION_BRANCHALWAYS:
      if (mode<MODE_UPDATE)
        println("branch label%i", findLabel(p+2+S16signed(p)-buffer)+1);
      else
        println("branch lbl%i", findLabel(p+2+S16signed(p)-buffer)+1);
      break;

    case SWFACTION_BRANCHIFTRUE:
      if (mode<MODE_UPDATE)
        println("branchIfTrue label%i", findLabel(p+2+S16signed(p)-buffer)+1);
      else
        println("branchIfTrue lbl%i", findLabel(p+2+S16signed(p)-buffer)+1);
      break;

    case SWFACTION_GETURL2:
    {
      int flags = *p;

      switch(flags)
      {
        case 0: println("getURL2"); break;
        case 1: println("getURL2 GET"); break;
        case 2: println("getURL2 POST"); break;
        case 0x40: println("loadMovie"); break;
        case 0x41: println("loadMovie GET"); break;
        case 0x42: println("loadMovie POST"); break;
        case 0x80: println("loadVariablesNum"); break;
        case 0x81: println("loadVariablesNum GET"); break;
        case 0x82: println("loadVariablesNum POST"); break;
        case 0xC0: println("loadVariables"); break;
        case 0xC1: println("loadVariables GET"); break;
        case 0xC2: println("loadVariables POST"); break;
        default: println("getURL2 (0x%x)", flags);
      }
      break;
    }

    case SWFACTION_CALLFRAME:
      println("callFrame");
      break;

    case SWFACTION_GOTOEXPRESSION:
      print("goto");
      if (*p == 0)
        puts("AndStop");
      else if (*p == 1)
        puts("AndPlay");
      else if ((*p == 2) && (len == 3)) {
	/* undocumented additional argument - the number of frames in all previous scenes */
        p++;
	printf("AndStop skip %i\n", S16(p));
      }
      else if ((*p == 3) && (len == 3)) {
	/* undocumented additional argument - the number of frames in all previous scenes */
        p++;
	printf("AndPlay skip %i\n", S16(p));
      }
      else {
	/* what the hell is it? assume andPlay, since flag>1 */
        printf("AndPlay // unknown goto flag %i, please report\n", *p);	
      }
      break;

    case SWFACTION_IFFRAMELOADED:
    {
      int frame = S16(p);
      p += 2;
      println("ifFrameLoaded %i", frame);
      ++indent;
      frameLoadedActions = *p++;
      break;
    }

    case SWFACTION_SETTARGET:
    {
      if (targetIndent==1)
      {
        --indent;
        println("end");
        targetIndent = 0;
      }
      if (strlen(p)>0)
      {
        println("setTarget '%s'", p);
        ++indent;
        targetIndent = 1;
      }
      break;
    }

    case SWFACTION_GOTOLABEL:
      println("gotoLabel '%s'",p);
      break;

    case SWFACTION_END:
      break;

    /* f5 ops */
    case SWFACTION_DELETE:
      println("delete");
      break;
    case SWFACTION_DELETE2:
      println("delete2");
      break;
    case SWFACTION_VAR:
      println("var");
      break;
    case SWFACTION_VAREQUALS:
      println("varEquals");
      break;
    case SWFACTION_CALLFUNCTION:
      if (!((mode >= MODE_UPDATE) && (dmode == DMODE_CLEAN)))
        println("callFunction");
      break;
    case SWFACTION_RETURN:
      println("return");
      break;
    case SWFACTION_MODULO:
      println("modulo");
      break;
    case SWFACTION_NEW:
      println("new");
      break;
    case SWFACTION_TYPEOF:
      println("typeof");
      break;
    case SWFACTION_TARGETPATH:
      println("targetPath");
      break;
    case SWFACTION_NEWADD:
      println("add");
      break;
    case SWFACTION_NEWLESSTHAN:
      println("lessThan");
      break;
    case SWFACTION_NEWEQUALS:
      println("equals");
      break;
    case SWFACTION_TONUMBER:
      println("toNumber");
      break;
    case SWFACTION_TOSTRING:
      println("toString");
      break;
    case SWFACTION_DUP:
      println("dup");
      break;
    case SWFACTION_GETMEMBER:
      println("getMember");
      break;
    case SWFACTION_SETMEMBER:
      println("setMember");
      break;
    case SWFACTION_INCREMENT:
      println("increment");
      break;
    case SWFACTION_DECREMENT:
      println("decrement");
      break;
    case SWFACTION_NEWMETHOD:
      println("newMethod");
      break;
    case SWFACTION_CALLMETHOD:
      println("callMethod");
      break;
    case SWFACTION_BITWISEAND:
      println("bitwiseAnd");
      break;
    case SWFACTION_BITWISEOR:
      println("bitwiseOr");
      break;
    case SWFACTION_BITWISEXOR:
      println("bitwiseXor");
      break;
    case SWFACTION_SHIFTLEFT:
      println("shiftLeft");
      break;
    case SWFACTION_SHIFTRIGHT:
      println("shiftRight");
      break;
    case SWFACTION_SHIFTRIGHT2:
      println("shiftRight2");
      break;

    case SWFACTION_CONSTANTPOOL:
    {
      int i, n = S16(p);
      int willInclude = 0;
      unsigned int constLen = 2;

      if (n>MAX_CONSTANTS)
	error("Too many constants");
      
      p += 2;
      /*
	this obfuscation is not really protecting from as viewer,
        so drop it till better times come
      if (mode == MODE_OBFUSCATE)
      {
	srand(time(0));
	println("__obf4__:");
	println("push %i.%if",rand()%22+1,rand()+1);
	println("branchIfTrue __obf2__");
	println("return");
	println("branch __obf1__");
	println("__obf2__:");
      }
      */
      /* may need to go back and rewrite/delete constants while processing flasm macros */
      conststart = ftell(stdout);
      print("constants ");

      nDict = 0;

      for(i=0; i<n; ++i)
      { 
	if (strIcmp(p,"$include")==0)
	  willInclude = 1;

        dictionary[i].value = p;
        dictionary[i].count = 0;
        nDict++;
        
        printstr(p);
        if (i<n-1)
          printf("%s",", ");
        else
          printf("%s","  \n");

	constLen += strlen(p)+1;	
	p += strlen(p)+1;
      }
      
      if (constLen!=len)
        error("swf file is corrupt: constant pool overflow");

      /* put some free space after constant pool to add user constants from included file later */
      if ((mode >= MODE_UPDATE) && (willInclude == 1))
        for(i=0; i<MAX_INCLUDE_POOL; ++i)
          putchar(' ');

      constend = ftell(stdout);
      /*
      if (mode == MODE_OBFUSCATE)
      {
	putchar('\n');
	println("branch __obf3__");
	println("__obf1__:");
	println("function () end");
	println("__obf3__:");
      }
      */
      break;
    }

    case SWFACTION_WITH:
    {
      unsigned int withlen = S16(p);

      println("with");

      ++indent;
      printActions(p+2, withlen);
      --indent;

      *lenptr += withlen;

      println("end");

      break;
    }

    case SWFACTION_DEFINEFUNCTION:
    {
      int n;
      unsigned int funclen;
      char *name = p;

      p += strlen(p)+1;
      n = S16(p);
      p += 2;

      if (mode < MODE_UPDATE)
	print("function %s (", name);
      else
	print("function '%s' (", name);

      if(n>0)
      {
        printf("'%s'", p);
        p += strlen(p)+1;
        --n;
      }

      for(; n>0; --n)
      {
        printf(", '%s'", p);
        p += strlen(p)+1;
      }

      putchar(')');
      putchar('\n');

      funclen = S16(p);
      p += 2;

      ++indent;
      printActions(p, funclen);
      --indent;

      println("end // of function %s\n",name);

      *lenptr += funclen;

      break;
    }

    case SWFACTION_ENUMERATE:
      println("enumerate");
      break;

    case SWFACTION_SETREGISTER:
      println("setRegister r:%i", *p);
      break;

    case SWFACTION_STRICTEQUALS:
      println("strictEquals");
      break;

    case SWFACTION_GREATERTHAN:
      println("greaterThan");
      break;

    case SWFACTION_ENUMERATEVALUE:
      println("enumerateValue");
      break;

    case SWFACTION_INSTANCEOF:
      println("instanceOf");
      break;

    case SWFACTION_STRICTMODE:
      print("strictMode");
      if (*p > 0)
        puts(" ON");
      else
        puts(" OFF");
      break;

    default:
      print("swfAction 0x%02x", type);
      if (len>0) {
        unsigned int i;
        printf(" hexdata ");
        for(i=0; i<*lenptr; ++i)
        {
          printf("0x%02X", *p);
          if (i<*lenptr-1)
            printf("%s",",");
          p++;
        }
      }
      printf(" // unknown action!\n");
  }
  return;
}

void rebuildConstantPool()
{
  long int curpos = ftell(stdout);
  int i;
  char constOK = 0;

  fseek(stdout, conststart, SEEK_SET); /* go back to the last constants declaration */
      
  for(i=0; i<nDict; ++i)
  {
    /* remove constants used less than 2 times, and empty strings */
    if ((dictionary[i].count>1) && (strlen(dictionary[i].value)>0))
    {
      if (constOK == 0)
      {
        print("constants ");
        constOK = 1;
      }

      printstr(dictionary[i].value);
      printf("%s",", ");
    }
  }
  
  if (flasmDict != NULL) {
    if (constOK == 0)
    {
      print("constants ");
      constOK = 1;
    }
    if (ftell(stdout)+strlen(flasmDict)>=constend)
      error("Too many user constants: %s", flasmDict);
    else
      printf(flasmDict);  /* add user constants */
  }
  else if (constOK == 1)
    fseek(stdout, -2, SEEK_CUR); /* remove last ", " */

  putchar('\n');
  
  for(i=ftell(stdout); i<constend-1; ++i) /* fill the rest of former constants with spaces */
    putchar(' ');

  putchar('\n');

  conststart = 0;
  constend = 0;
  flasmDict = NULL;
  fseek(stdout, curpos, SEEK_SET);
}

void readActionBuffer(FILE *f, unsigned long int length)
{
  if (length == 0)
    return;

  buffer = malloc(length);
  if (buffer == NULL)
    error("Not enough memory to process action block");
  buflen = length;
  if(fread(buffer, 1, length, f)!=length)
    error("Attempt to read beyond EOF");
}

void printActionBlock(FILE *f, unsigned long int length)
{
  unsigned long int i, blocklen;
  
  numTargets = 0;
  curTarget = 0;
  nDict = 0;

  ++indent;

  readActionBuffer(f, length);

  /* scan for jump offsets */
  for(i=0; i<length; ++i)
  {
    if((buffer[i] & 0x80) == 0)
      continue;

    blocklen = S16(buffer+i+1);

    if(buffer[i] == SWFACTION_BRANCHALWAYS || buffer[i] == SWFACTION_BRANCHIFTRUE)
      addTarget(i+1 + blocklen+2 + S16signed(buffer+i+3));
 
    i += blocklen+2;
  }

  qsort(targets, numTargets, sizeof(int), int_compare);

  printActions(buffer, length);

  if ((mode >= MODE_UPDATE) && (constend>0))
    rebuildConstantPool();

  if (targetIndent==1)
  {
    --indent;
    println("end");
    targetIndent = 0;
  }

  --indent;

  free(buffer);
}

void skipMatrix(FILE *f)
{
  InitBits(f);    
  if (getBits(f,1)) getBits(f,getBits(f,5)*2);
  if (getBits(f,1)) getBits(f,getBits(f,5)*2);
  getBits(f,getBits(f,5)*2);
}

void skipColorTransform(FILE *f) 
{
  unsigned int needAdd,needMul,nBits;
  InitBits(f);
  needAdd = getBits(f,1);
  needMul = getBits(f,1);
  nBits = getBits(f,4);
  if (needMul) getBits(f,nBits*4);
  if (needAdd) getBits(f,nBits*4);
}

void parseKeyPressEvent (unsigned char onKey)
{
  printf("keyPress ");

  if ((onKey) == keySpace) printf("_SPACE");
  else if ((onKey)>47) printf("'%c'",onKey);
  else switch (onKey)
  {
    case keyLeft:
      printf("_LEFT");
      break;
    case keyRight:
      printf("_RIGHT");
      break;
    case keyHome:
      printf("_HOME");
      break;
    case keyEnd:
      printf("_END");
      break;
    case keyIns:
      printf("_INS");
      break;
    case keyDel:
      printf("_DEL");
      break;
    case keyBackspace:
      printf("_BACKSPACE");
      break;
    case keyEnter:
      printf("_ENTER");
      break;
    case keyUp:
      printf("_UP");
      break;
    case keyDn:
      printf("_DN");
      break;
    case keyPgUp:
      printf("_PGUP");
      break;
    case keyPgDn:
      printf("_PGDN");
      break;
    case keyTab:
      printf("_TAB");
      break;
    default:
      printf("unknown key %i",onKey);
  }
}

void parseButtonEvent (FILE *f, unsigned int condition, unsigned long int length)
{ 
  char delim = ' ';

  putchar('\n');
  ++indent;
  print("on");

  if (condition&IdleToOverUp)
  { putchar(delim); printf("idleToOverUp"); delim = ','; }  

  if (condition&OverUpToIdle)
  { putchar(delim); printf("overUpToIdle"); delim = ','; }  

  if (condition&OverUpToOverDown)
  { putchar(delim); printf("overUpToOverDown"); delim = ','; }  

  if (condition&OverDownToOverUp)
  { putchar(delim); printf("overDownToOverUp"); delim = ','; }  

  if (condition&OverDownToOutDown)
  { putchar(delim); printf("overDownToOutDown"); delim = ','; }  

  if (condition&OutDownToOverDown)
  { putchar(delim); printf("outDownToOverDown"); delim = ','; }  

  if (condition&OutDownToIdle)
  { putchar(delim); printf("outDownToIdle"); delim = ','; }  
 
  if (condition&IdleToOverDown)
  { putchar(delim); printf("idleToOverDown"); delim = ','; }  

  if (condition&OverDownToIdle)
  { putchar(delim); printf("overDownToIdle"); delim = ','; }  
 
  /* keyPress */
  if (condition>0x1FF) {
     putchar(delim);
     parseKeyPressEvent ((unsigned char)(condition>>9));
  }

  printf("\n");
  printActionBlock(f, length);
  println("end");
  --indent;
}

void parseButton2(FILE *f, unsigned long int length)
{
  int condition, buttonID;
  unsigned int actionOffset;
  unsigned long int lastABLength;

  buttonID = getWord(f);
  fgetc(f); // trackAsMenu

  actionOffset = getWord(f);
  lastABLength = length-8-actionOffset;

  if (actionOffset>0) {
    printf("\n");
    println ("defineButton %i ",buttonID);

    /* skip button data */
    while (actionOffset>2) {
      fgetc(f);
      --actionOffset;
    }

    while ((actionOffset=getWord(f))>0)
    {
      lastABLength -= actionOffset;
      condition = getWord(f);
      parseButtonEvent(f,condition,actionOffset-4);
    }

    /* last event*/
    condition = getWord(f);
    parseButtonEvent(f,condition,lastABLength);
  
    println ("end // of defineButton %i ",buttonID);
  }
  else
  { /* no button events */
    while (lastABLength+2>0) {
      fgetc(f);
      --lastABLength;
    }
  }

  fgetc(f); // button end
}

void parseEvent(FILE *f, unsigned long int event, unsigned long int length)
{
  ++indent;
  putchar('\n');

  if (event&eventLoad)
    print("onClipEvent load");

  else if (event&eventEnterFrame)
    print("onClipEvent enterFrame");

  else if (event&eventUnload)
    print("onClipEvent unload");

  else if (event&eventMouseMove)
    print("onClipEvent mouseMove");

  else if (event&eventMouseDown)
    print("onClipEvent mouseDown");

  else if (event&eventMouseUp)
    print("onClipEvent mouseUp");

  else if (event&eventKeyDown)
    print("onClipEvent keyDown");

  else if (event&eventKeyUp)
    print("onClipEvent keyUp");

  else if (event&eventData)
    print("onClipEvent data");

  else if (event&eventInitialize)
    print("onClipEvent initialize");

  else {
    char delim = ' ';

    print("onClipEvent");

    if (event&eventPress)
    { putchar(delim); printf("press"); delim = ','; }

    if (event&eventRelease)
    { putchar(delim); printf("release"); delim = ','; }

    if (event&eventReleaseOutside)
    { putchar(delim); printf("releaseOutside"); delim = ','; }

    if (event&eventRollOver)
    { putchar(delim); printf("rollOver"); delim = ','; }

    if (event&eventRollOut)
    { putchar(delim); printf("rollOut"); delim = ','; }

    if (event&eventDragOver)
    { putchar(delim); printf("dragOver"); delim = ','; }

    if (event&eventDragOut)
    { putchar(delim); printf("dragOut"); delim = ','; }

    if (event&eventKeyPress)
    { putchar(delim); parseKeyPressEvent ((unsigned char)fgetc(f)); length--;}
  }

  printf("\n");
  printActionBlock(f, length);
  println("end");
  --indent;
}

void parsePlaceObject2 (FILE *f, unsigned long int length)
{
  int i,flags,clipID,depth;
  unsigned long int curEvent;

  flags = fgetc(f);

  if (flags & splaceOnClipEvents) {
    printf("\n");

    /* character depth */
    depth = getWord(f);
    // printf("Depth: %i\n",depth);
  
    if (flags & splaceCharacter) clipID = getWord(f);
    if (flags & splaceMatrix) skipMatrix(f);
    if (flags & splaceColorTransform) skipColorTransform(f);
    if (flags & splaceRatio) getWord(f);

    print ("placeMovieClip %i ",clipID);
    if (flags & splaceName) {
      printf("as ");
      putchar('\'');
      while ((i=fgetc(f))!=0) putchar((char)i);
      putchar('\'');
    }
  
    printf("\n");

    getWord(f); // skip ?? - always 0

    if (swfVersion == 5) {
      getWord(f); // all events
      while ((curEvent = (unsigned long int)getWord(f))!=0)
	parseEvent(f, curEvent, getDoubleWord(f));
    }
    else if (swfVersion >= 6) {
      // flash 6 now supports button events for mcs, therefore going long here
      getDoubleWord(f);

      while ((curEvent = getDoubleWord(f))!=0)
        parseEvent(f, curEvent, getDoubleWord(f));
    }

    println("end // of placeMovieClip %i",clipID);
  }
  else {
    /* no events found, skip the rest of placeObject2 */
    skipBuffered(f,length-1);
  }
}

void parseMovieClip (FILE *f)
{
  int clipID;
  int frameNum = 0;
  int framesTotal = 0;
  
  clipID = getWord(f);
  framesTotal = getWord(f);

  printf("\n");
  println("defineMovieClip %i // total frames: %i",clipID,framesTotal);
  indent++;

  while(!feof(f))
  {
    unsigned int type, block;
    unsigned long int length;

    block = getWord(f);
    type = block >> 6;
    length = block & ((1<<6)-1);

    if(length == 63)
	length = getDoubleWord(f);

    if (type == 0)
      break;
    
    switch(type)
    {
      case TAG_DOACTION:
        printFrameNum(frameNum);
        printActionBlock(f, length);
	println("end // of frame %i",frameNum);
        break;

      case TAG_PLACEOBJECT2: /* possibly onClipEvents inside */
        parsePlaceObject2 (f, length);
        break;

      case TAG_SHOWFRAME:
        ++frameNum;
        break;

      default:
	// println("\nMC Tag: %i", type);
        skipBuffered(f,length);
        break;
    }
  }

  --indent;
  println("end // of defineMovieClip %i",clipID);
}

void disassembleSWF (FILE *f, char *fname)
{
  int bits, i, componentID;
  unsigned long int size;
  int frameNum = 0;
  int framesTotal = 0;
  float frameRate, movieWidth, movieHeight;

  swfVersion = fgetc(f); /* version */

  size = getDoubleWord(f);

  /* movie bounds */
  InitBits(f); 
  bits = getBits(f,5);
  getBits(f, bits); // xMin - always 0
  movieWidth = ((float)getBits(f, bits))/20;
  getBits(f, bits); // yMin - always 0
  movieHeight = ((float)getBits(f, bits))/20;

  frameRate = ((float)fgetc(f))/256 + (float)fgetc(f); /* frame rate */
  framesTotal = getWord(f);

  printf("movie '%s'", fname);

  if (wasCompressed) printf(" compressed");
  printf(" // flash %i, total frames: %i, frame rate: ", swfVersion, framesTotal);
  printFloat(frameRate, 1);
  printf(" fps, ");
  printFloat(movieWidth, 1);  putchar('x');  printFloat(movieHeight, 1);
  printf(" px\n");
  
  while(!feof(f))
  {
    unsigned int type, block;
    unsigned long int length;

    block = getWord(f);
    type = block >> 6;
    length = block & ((1<<6)-1);

    if(length == 63)
	length = getDoubleWord(f);

    if (type == 0)
	break;

    switch(type)
    {
      case TAG_DOACTION:
        printFrameNum(frameNum);
        printActionBlock(f, length);
	println("end // of frame %i",frameNum);
        break;

      case TAG_INITMOVIECLIP:
	println("");
	componentID = getWord(f);
        println("initMovieClip %i", componentID);
        printActionBlock(f, length-2);
	println("end // of initMovieClip %i", componentID);
        break;

      case TAG_PLACEOBJECT2: /* possibly onClipEvents inside */
        parsePlaceObject2 (f, length);
        break;

     case TAG_DEFINEBUTTON2: /* possibly button events inside */
        parseButton2 (f, length);
        break;

      case TAG_SHOWFRAME:
        ++frameNum;
        break;

      case TAG_PROTECT:
	println("");
        print("protect");
	if (length>0)
	{
	  /* password found */
	  printf(" '");
	  getWord(f); // always 0?
          for(i=2; i<length-1; ++i) putchar((char)fgetc(f));
	  fgetc(f); // always 0?
	  putchar('\'');
	}
        putchar('\n');
      	break;

      case TAG_ENABLEDEBUGGER:
	println("");
        print("enableDebugger");
	if (length>0)
	{
	  /* password found */
  	  /* debugger always uses password, even for empty one */
	  printf(" '");
	  getWord(f); // always 0?
          for(i=2; i<length-1; ++i) putchar((char)fgetc(f));
	  fgetc(f); // always 0?
	  putchar('\'');
	}
        putchar('\n');
        break;

      case TAG_ENABLEDEBUGGER2:
	/* flash MX debugger */
	println("");
        print("enableDebugger2");
	if (length>0)
	{
	  /* password found */
  	  /* debugger always uses password, even for empty one */
	  printf(" '");
	  getWord(f); // always 0?
          for(i=2; i<length-1; ++i) putchar((char)fgetc(f));
	  fgetc(f); // always 0?
	  putchar('\'');
	}
        putchar('\n');
        break;

      case TAG_DEFINEMOVIECLIP:
        parseMovieClip(f);
        break;

      default:
	//println("\nTag: %i", type);
        skipBuffered(f,length);
    }
  }

  fclose(f);
  printf ("end\n");
}
