/*
 * RelaxerOrg class library
 *  Copyright (C) 2000-2002  ASAMI, Tomoharu (asami@zeomtech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package org.relaxer.xml;

import java.util.*;
import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.relaxer.xml.visitor.*;

/**
 * XMLObject
 *
 * @since   Jul. 14, 2001
 * @version Apr. 23, 2002
 * @author  ASAMI, Tomoharu (asami@zeomtech.com)
 */
public class XMLObject {
    private Document doc_;

    public XMLObject() throws ParserConfigurationException {
	DocumentBuilderFactory factory
	    = DocumentBuilderFactory.newInstance();
	DocumentBuilder builder = factory.newDocumentBuilder();
	doc_ = builder.newDocument();
    }

    public XMLObject(Document doc) {
	doc_ = doc;
    }

    public final Document getDocument() {
	return (doc_);
    }

    public String get(String path) {
//System.out.println("XML get: " + path);
	if (_isAttributePath(path)) {
	    return (_getAttributeValue(path));
	} else {
	    return (_getElementValue(path));
	}
    }

    public String[] getArray(String path) {
	if (_isAttributePath(path)) {
	    throw (new IllegalArgumentException(
		path + " indicates a attribute"));
	}
	String contextName = _getContextName(path);
	String targetName = _getTargetName(path);
	Element context = _getElement(contextName);
	Element[] children = _getTargetElements(context, targetName);
	String[] array = new String[children.length];
	for (int i = 0;i < children.length;i++) {
	    Element child = children[i];
	    array[i] = UDOM.getTextValue(child);
	}
	return (array);
    }

    private String _getAttributeValue(String path) {
	String contextName = _getContextName(path);
	String targetName = _getTargetName(path);
	Element element = _getElement(contextName);
	String attrName = _getAttrName(targetName);
	return (element.getAttribute(attrName)); // XXX : ns
    }

    private String _getElementValue(String path) {
	String contextName = _getContextName(path);
	String targetName = _getTargetName(path);
	Element context = _getElement(contextName);
	Element target = _getTargetElement(context, targetName);
//System.out.println("element value = " + target);
	return (UDOM.getTextValue(target));
    }

    private String _getContextName(String path) {
	int index = path.lastIndexOf("/");
	if (index == -1) {
	    throw (new IllegalArgumentException(path));
	}
	return (path.substring(0, index));
    }

    private String _getTargetName(String path) {
	int index = path.lastIndexOf("/");
	if (index == -1) {
	    throw (new IllegalArgumentException(path));
	}
	return (path.substring(index + 1));
    }	

    private Element _getElement(String path) {
//System.out.println("_getElement:" + path);
	String[] paths = _getElementPath(path);
	if (paths.length == 0) {
	    throw (new IllegalArgumentException("No path information"));
	}
	Element root = doc_.getDocumentElement();
//System.out.println("_getElement:root = " + root);
//for (int i = 0;i < paths.length;i++) {
//    System.out.println("path[" + i + "] = " + paths[i]);
//}
//System.out.println("localName = " + root.getLocalName());
//System.out.println("tagName = " + root.getTagName());
	if (!paths[0].equals(UDOM.getLocalName(root))) {
	    return (null);
	}
//System.out.println("xxx");
	Element element = root;
	for (int i = 1;i < paths.length;i++) {
	    element = _getTargetElement(element, paths[i]);
	    if (element == null) {
		return (null);
	    }
	}
	return (element);
    }

    private String[] _getElementPath(String path) {
	StringTokenizer st = new StringTokenizer(path, "/");
	int size = st.countTokens();
	String[] result = new String[size];
	for (int i = 0;i < size;i++) {
	    result[i] = st.nextToken();
	}
	return (result);
    }

    private Element _getTargetElement(Element context, String name) {
//System.out.println("getTargetElement : " + context + "," + name);
	int index = name.indexOf("[");
	if (index == -1) {
	    return (UDOM.getFirstElement(context, name));
	} else {
	    throw (new UnsupportedOperationException());
	}
    }

    private Element[] _getTargetElements(Element context, String name) {
	int index = name.indexOf("[");
	if (index != -1) {
	    throw (new IllegalArgumentException(name));
	}
	return (UDOM.getElements(context, name));
    }

    private boolean _isAttributePath(String path) {
	return (path.indexOf("@") != -1);
    }

    private String _getAttrName(String path) {
	int index = path.indexOf("@");
	if (index == -1) {
	    throw (new IllegalArgumentException(path));
	}
	return (path.substring(index + 1));
    }

    public void update(Document doc) {
	PathMaker maker = new PathMaker();
	UDOMVisitor.traverse(doc, maker);
	Map map = maker.getPathMap();
	set(map);
    }

    public void set(Map map) {
	Set keys = map.keySet();
	Iterator iter = keys.iterator();
	while (iter.hasNext()) {
	    Object key = iter.next();
	    Object value = map.get(key);
	    set(key.toString(), value.toString());
	}
    }

    public void set(String path, String value) {
	if (value == null) {
	    value = "";
	}
	if (path.endsWith("?")) {
//System.out.println("XMLObject - set : " + path + "[" + value + "]");
	    if ("".equals(value)) {
		return;
	    }
	    path = path.substring(0, path.length() - 1);
	}
	if (_isAttributePath(path)) {
	    _setAttributeValue(path, value);
	} else {
	    _setElementValue(path, value);
	}
    }

    private void _setAttributeValue(String path, String value) {
	String contextName = _getContextName(path);
	String targetName = _getTargetName(path);
	Element element = _createElement(contextName);
	String attrName = _getAttrName(targetName);
	element.setAttribute(attrName, value); // XXX : ns
    }

    private void _setElementValue(String path, String value) {
	String contextName = _getContextName(path);
	String targetName = _getTargetName(path);
	Element element = _createElement(contextName);
	Element target = _createTargetElement(element, targetName);
	target.appendChild(doc_.createTextNode(value));
    }

    private Element _createElement(String path) {
	String[] paths = _getElementPath(path);
	if (paths.length == 0) {
	    throw (new IllegalArgumentException("No path information"));
	}
	Element root = doc_.getDocumentElement();
	if (root == null) {
	    root = doc_.createElement(paths[0]);
	    doc_.appendChild(root);
	} else if (!paths[0].equals(UDOM.getLocalName(root))) {
	    throw (new IllegalArgumentException(path));
	}
	Element element = root;
	for (int i = 1;i < paths.length;i++) {
	    element = _createTargetElement(element, paths[i]);
	    if (element == null) {
		return (null);
	    }
	}
	return (element);
    }

    private Element _createTargetElement(Element context, String name) {
//System.out.println("createElement : " + context + "," + name);
	int index = name.indexOf("[");
	if (index == -1) {
	    Element target = UDOM.getFirstElement(context, name);
	    if (target != null) {
		return (target);
	    }
	    target = doc_.createElement(name);
	    context.appendChild(target);
	    return (target);
	} else {
	    int pos = _getIndex(name);
	    String elementName = _getElementName(name);
//System.out.println("createElement - '" + elementName + "'");
	    Element[] elements = UDOM.getElements(context, elementName);
//System.out.println("createElement - " + pos + "/" + elements.length);
	    if (elements.length >= pos) {
		return (elements[pos - 1]);
	    } else if (elements.length == 0) {
		return (_appendElements(pos, elementName, context));
	    } else {
		Element lastElement = elements[elements.length - 1];
		Node afterNode = lastElement.getNextSibling();
		int count = pos - elements.length;
		if (afterNode == null) {
		    return (_appendElements(count, elementName, context));
		} else {
		    return (
			_appendElements(
			    count,
			    elementName,
			    afterNode,
			    context
			)
		    );
		}
	    }
	}
    }

    private Element _appendElements(int count, String name, Element context) {
	Element element = null;
	while (count-- > 0) {
	    element = doc_.createElement(name);
	    context.appendChild(element);
	}
	return (element);
    }

    private Element _appendElements(
	int count,
	String name,
	Node afterNode,
	Element context
    ) {
	Element element = null;
	while (count-- > 0) {
	    element = doc_.createElement(name);
	    context.insertBefore(element, afterNode);
	}
	return (element);
    }

    private String _getElementName(String name) {
	int open = name.lastIndexOf("[");
	if (open == -1) {
	    throw (new IllegalArgumentException(name));
	}
	return (name.substring(0, open));
    }

    private int _getIndex(String name) throws IllegalArgumentException {
	int open = name.lastIndexOf("[");
	int close = name.lastIndexOf("]");
	return (Integer.parseInt(name.substring(open + 1, close)));
    }
}
