/*
 * The RelaxerOrg class library
 *  Copyright (C) 1997-2002  ASAMI, Tomoharu (asami@relaxer.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package org.relaxer.util;

import java.lang.reflect.*;
import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.relaxer.xml.UDOM;

/**
 * URelaxer
 *
 * @since   Jul.  4, 2001
 * @version Apr. 15, 2002
 * @author  ASAMI, Tomoharu (asami@relaxer.org)
 */
public final class URelaxer {
    public static final String RELAXER_CONTAINER_NS
                                 = "http://www.relaxer.org/xml";
    public static final String RELAXER_CONTAINER_LIST = "list";

    private static Class[] constructorParams__ = { Document.class };
    private static Class[] noParams__ = {};

    public static boolean isRelaxerObject(Object object) {
	return (isRelaxerType(object.getClass()));
    }

    public static boolean isRelaxerType(Class type) {
	try {
	    if (type.getConstructor(constructorParams__) == null) {
		return (false);
	    }
	    if (type.getMethod("makeDocument", noParams__) == null) {
		return (false);
	    }
	} catch (NoSuchMethodException e) {
	    return (false);
	}
	return (true);
    }

    public static Object makeRelaxerObject(Class type, Document doc) {
	Element root = doc.getDocumentElement();
	if (RELAXER_CONTAINER_NS.equals(root.getNamespaceURI())) {
	    if (RELAXER_CONTAINER_LIST.equals(root.getLocalName())) {
		return (_makeRelaxerObjectAsArray(type, doc));
	    } else {
		throw (new UnsupportedOperationException());
	    }
	} else {
	    return (_makeRelaxerObjectAsObject(type, doc));
	}
    }

    private static Object _makeRelaxerObjectAsObject(
	Class type,
	Document doc
    ) {
	try {
	    Constructor constructor = type.getConstructor(
		constructorParams__
	    );
	    Object[] params = { doc };
	    return (constructor.newInstance(params));
	} catch (NoSuchMethodException e) {
	    throw (new IllegalArgumentException(
		"no relaxer document (NoSuchMethod): " + type));
	} catch (IllegalAccessException e) {
	    throw (new IllegalArgumentException(
		"no relaxer document (IllegalAccess): " + type));
	} catch (InstantiationException e) {
	    throw (new IllegalArgumentException(
		"no relaxer document (Instantiation): " + type));
	} catch (InvocationTargetException e) {
	    Throwable target = e.getTargetException();
	    throw (new IllegalArgumentException(
		"no relaxer document (" + target.toString() + "): " + type
	    ));
	}
    }

    private static Object _makeRelaxerObjectAsArray(
	Class type,
	Document doc
    ) {
	Element root = doc.getDocumentElement();
	Element[] elements = UDOM.getElements(root);
	return (_makeRelaxerObjectAsArray(type, elements));
    }

    private static Object _makeRelaxerObjectAsArray(
	Class type,
	Element[] elements
    ) {
	Object array = Array.newInstance(type, elements.length);
	for (int i = 0;i < elements.length;i++) {
	    try {
		Class[] types = { Element.class };
		Constructor constructor = type.getConstructor(types);
		Object[] params = { elements[i] };
		Object value = constructor.newInstance(params);
		Array.set(array, i, value);
	    } catch (NoSuchMethodException e) {
		throw (new IllegalArgumentException(
		    "no relaxer document : " + elements[i]));
	    } catch (IllegalAccessException e) {
		throw (new IllegalArgumentException(
		    "no relaxer document : " + elements[i]));
	    } catch (InstantiationException e) {
		throw (new IllegalArgumentException(
		    "no relaxer document : " + elements[i]));
	    } catch (InvocationTargetException e) {
		throw (new IllegalArgumentException(
		    "no relaxer document : " + elements[i]));
	    }
	}
	return (array);
    }

    public static Object makeRelaxerObject(Class type, Document[] docs) {
	Element[] elements = new Element[docs.length];
	for (int i = 0;i < docs.length;i++) {
	      elements[i] = docs[i].getDocumentElement();
	}
	return (_makeRelaxerObjectAsArray(type, elements));
    }

    public static Object makeRelaxerObject(Object[] objects) {
	return (objects);
    }

    public static Object makeRelaxerObjects(Class type, Document doc) {
	Element root = doc.getDocumentElement();
	if (RELAXER_CONTAINER_NS.equals(root.getNamespaceURI())) {
	    if (RELAXER_CONTAINER_LIST.equals(root.getLocalName())) {
		return (_makeRelaxerObjectAsArray(type, doc));
	    } else {
		throw (new UnsupportedOperationException());
	    }
	}
	Object array = Array.newInstance(type, 1);
	Array.set(array, 0, _makeRelaxerObjectAsObject(type, doc));
	return (array);
    }

    public static Object makeRelaxerObjects(Object value) {
	Class type = value.getClass();
	if (type.isArray()) {
	    return (value);
	}
	Object array = Array.newInstance(type, 1);
	Array.set(array, 0, value);
	return (array);
    }

    public static Document makeRelaxerDocument(Object value) {
	return (makeRelaxerDocument(value.getClass(), value));
    }

    public static Document makeRelaxerDocument(Class type, Object value) {
	if (type.isArray()) {
	    return (_makeRelaxerDocumentByArray(type, value));
	} else {
	    return (_makeRelaxerDocumentByObject(type, value));
	}
    }

    private static Document _makeRelaxerDocumentByObject(
	Class type,
	Object value
    ) {
	try {
	    Method method = type.getMethod("makeDocument", noParams__);
	    Object[] params = {};
	    return ((Document)method.invoke(value, params));
	} catch (NoSuchMethodException e) {
	    throw (
		new IllegalArgumentException("no relaxer object : " + value));
	} catch (IllegalAccessException e) {
	    throw (
		new IllegalArgumentException("no relaxer object : " + value));
	} catch (InvocationTargetException e) {
	    throw (
		new IllegalArgumentException("no relaxer object : " + value));
	}
    }

    private static Document _makeRelaxerDocumentByArray(
	Class arrayType,
	Object array
    ) {
	Document doc = _makeNewDocument();
	Element root = doc.createElementNS(
	    RELAXER_CONTAINER_NS,
	    RELAXER_CONTAINER_LIST
	);
	doc.appendChild(root);
	Class type = arrayType.getComponentType();
	int size = Array.getLength(array);
	Class[] types = { Node.class };
	Object[] params = { root };
	for (int i = 0;i < size;i++) {
	    Object value = Array.get(array, i);
	    try {
		Method method = type.getMethod("makeElement", types);
		method.invoke(value, params);
	    } catch (NoSuchMethodException e) {
		throw (new IllegalArgumentException(
		    "no relaxer object : " + value));
	    } catch (IllegalAccessException e) {
		throw (new IllegalArgumentException(
		    "no relaxer object : " + value));
	    } catch (InvocationTargetException e) {
		throw (new IllegalArgumentException(
		    "no relaxer object : " + value));
	    }
	}
	return (doc);
    }

    public static Document makeRelaxerDocuments(Object value) {
	return (makeRelaxerDocument(value.getClass(), value));
    }	

    public static Document makeRelaxerDocuments(Class type, Object value) {
	if (type.isArray()) {
	    return (_makeRelaxerDocumentByArray(type, value));
	} else {
	    return (
		makeRelaxerDocuments(
		    _makeRelaxerDocumentByObject(type, value)
		)
	    );
	}
    }

    public static Document makeRelaxerDocuments(Document doc) {
	Element element = doc.getDocumentElement();
	Element list = doc.createElementNS(
	    RELAXER_CONTAINER_NS,
	    RELAXER_CONTAINER_LIST
	);
	list.appendChild(element);
	doc.appendChild(list);
	return (doc);
    }

    private static Document _makeNewDocument() {
	try {
	    DocumentBuilderFactory factory
		= DocumentBuilderFactory.newInstance();
	    DocumentBuilder builder = factory.newDocumentBuilder();
	    return (builder.newDocument());
	} catch (ParserConfigurationException e) {
	    throw (new InternalError());
	}
    }
}
