/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2003  ASAMI, Tomoharu (asami@AsamiOffice.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.vfs.ftp;

import java.util.*;
import java.io.*;
import java.net.*;
import com.oroinc.net.ftp.*;
import jp.gr.java_conf.jaba2.text.UString;
import jp.gr.java_conf.jaba2.io.UFile;
import jp.gr.java_conf.jaba2.vfs.*;

/**
 * FTPRealm
 *
 * @since   Jan. 15, 2000
 * @version Sep. 26, 2003
 * @author  ASAMI, Tomoharu (asami@AsamiOffice.com)
 */
public class FTPRealm extends AbstractVFSRealm {
    private URL url_;
    private String server_;
    private String username_;
    private String password_;
    private boolean passive_ = false;
    private String root_;
    private FTPClient ftp_;
    private Set createdDirs_ = new HashSet();
    private FTPFile timestamp_;
    private String timestampName_ = ".VFSTimestamp";

    public FTPRealm(URL url, String username, String password) {
	url_ = url;
	if (!"ftp".equals(url.getProtocol())) {
	    throw (new InternalError());
	}
	server_ = url.getHost();
	root_ = url.getFile();
	username_ = username;
	password_ = password;
    }

    public void setPassive(boolean passive) {
        passive_ = passive;
    }

    public void shallowSetup(IFilter filter) throws IOException {
	_connect();
	_shallowSetup(new FTPRoot(ftp_, root_), filter);
    }

    public void deepSetup(IFilter filter) throws IOException {
	_connect();
	_deepSetup(new FTPRoot(ftp_, root_), filter);
	timestamp_ = _getTimestampFile();
    }

    private FTPFile _getTimestampFile() throws IOException {
	FTPFile[] files = ftp_.listFiles();
	for (int i = 0;i < files.length;i++) {
	    FTPFile file = files[i];
	    if (timestampName_.equals(file.getName())) {
		return (file);
	    }
	}
	return (null);
    }

    public long getLastTimestamp() throws IOException {
	if (timestamp_ != null) {
	    return (timestamp_.getTimestamp().getTime().getTime());
	} else {
	    return (super.getLastTimestamp());
	}
    }

    public void updateTimestamp() throws IOException {
	if (timestamp_ == null) {
	    createObject(timestampName_);
	}
	timestamp_ = _getTimestampFile();
    }

    public void createObject(String pathname)
	throws IOException {

	InputStream in = new ByteArrayInputStream("".getBytes());
	createObject(pathname, in);
	in.close();
    }

    public void createObject(String pathname, InputStream in)
	throws IOException {

	String containerName = UString.getContainerPathname(pathname);
	if (containerName != null) {
	    _ensureDirectory(containerName);
	}
        ftp_.deleteFile(pathname);
        ftp_.setFileType(FTP.BINARY_FILE_TYPE);
        if (passive_) {
            ftp_.enterLocalPassiveMode();
        }
	ftp_.storeFile(pathname, in);
    }

    public void createContainer(String pathname, InputStream in)
	throws IOException {

	_ensureDirectory(pathname);
    }

    private void _connect() throws IOException {
	FTPClient ftp = new FTPClient();
	try {
	    ftp.addProtocolCommandListener(
		new LogCommandListener()
	    );
	    ftp.connect(server_);
	    int reply = ftp.getReplyCode();
	    if (!FTPReply.isPositiveCompletion(reply)) {
		ftp.disconnect();
		ftp = null;
		return;
	    }
	    ftp.setFileType(FTP.BINARY_FILE_TYPE);
	    if (!ftp.login(username_, password_)) {
		ftp.logout();
		ftp = null;
		return;
	    }
	} catch (IOException e) {
	    if (ftp.isConnected()) {
		try {
		    ftp.disconnect();
		} catch (IOException ee) {
		    // do nothing
		} finally {
		    ftp = null;
		}
	    }
	} finally {
	    if (ftp != null) {
		ftp_ = ftp;
	    }
	}
    }

    private void _ensureDirectory(String pathname) throws IOException {
	String[] names = UString.getTokens(pathname, "/");
	for (int i = 0;i < names.length;i++) {
	    _ensureDirectory(names, i);
	}
    }

    private void _ensureDirectory(String[] names, int index)
	throws IOException {

	StringBuffer buffer = new StringBuffer();
	buffer.append(names[0]);
	for (int i = 1;i <= index;i++) {
	    buffer.append("/");
	    buffer.append(names[i]);
	}
	String name = new String(buffer);
	if (createdDirs_.contains(name)) {
	    return;
	}
	ftp_.makeDirectory(name);
	createdDirs_.add(name);
    }
}
