/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2002  ASAMI, Tomoharu (asami@relaxer.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.tmodel;

import jp.gr.java_conf.jaba2.datatype.ISQLDatatype;
import jp.gr.java_conf.jaba2.util.PropertyList;

/**
 * TTable
 *
 * @since   Feb. 14, 2000
 * @version Oct. 14, 2002
 * @author  ASAMI, Tomoharu (asami@relaxer.org)
 */
public class TTable {
    private String name_;
    private TMetadata metadata_;
    private PropertyList properties_ = new PropertyList();
    private String quoteId_ = "\"";

    public TTable(String name) {
	name_ = name;
	metadata_ = new TMetadata();
    }

    public final String getName() {
	return (name_);
    }

    public final void setMetadata(TMetadata metadata) {
	metadata_ = metadata;
    }

    public final TMetadata getMetadata() {
	return (metadata_);
    }

    public final void setQuoteId(String quote) {
	quoteId_ = quote;
    }

    public final String getQuoteId() {
	return (quoteId_);
    }

    public final String getCreateTableCommandReadable() {
	return (getCreateTableCommandReadable(name_));
    }

    public final String getCreateTableCommandReadable(String tableName) {
	StringBuffer buffer = new StringBuffer();
	TColumnInfo[] infos = metadata_.getColumnInfos();
	TColumnInfo[] primaries = metadata_.getPrimaryKeyInfos();
	buffer.append("CREATE TABLE ");
	buffer.append(_escapeNull(quoteId_));
	buffer.append(tableName);
	buffer.append(_escapeNull(quoteId_));
	buffer.append(" (\n");
	if (infos.length > 0) {
	    TColumnInfo info = infos[0];
	    buffer.append("\t");
	    _makeColumnReadable(info, primaries, buffer);
	    for (int i = 1;i < infos.length;i++) {
		info = infos[i];
		buffer.append(",\n");
		buffer.append("\t");
		_makeColumnReadable(info, primaries, buffer);
	    }
	    if (primaries.length > 1) {
		buffer.append(",\n");
		buffer.append("\t");
		buffer.append("PRIMARY KEY (");
		TColumnInfo primary = primaries[0];
		buffer.append(primary.getName());
		for (int i = 1;i < primaries.length;i++) {
		    primary = primaries[i];
		    buffer.append(", ");
		    buffer.append(primary.getName());
		}
		buffer.append(")\n");
	    } else {
		buffer.append("\n");
	    }
	    buffer.append(")\n");
	}
	return (new String(buffer));
    }

    private final void _makeColumnReadable(
	TColumnInfo column,
	TColumnInfo[] primaries,
	StringBuffer buffer
    ) {
	String columnName = column.getName();
	ISQLDatatype type = column.getSQLDatatype();
	buffer.append(_escapeNull(quoteId_));
	buffer.append(columnName);
	buffer.append(_escapeNull(quoteId_));
	buffer.append(" ");
	buffer.append(type.getDDLName());
	if (column.isPrimary()) {
	    if (primaries.length == 1) {
		buffer.append(" PRIMARY KEY");
	    } else {
		buffer.append(" NOT NULL");
	    }
	} else if (column.isUnique()) {
	    buffer.append(" UNIQUE");
	} else if (column.isNotNull()) {
	    buffer.append(" NOT NULL");
	}
    }

    public final String getCreateTableCommand() {
	return (getCreateTableCommand(name_));
    }

    public final String getCreateTableCommand(String tableName) {
	StringBuffer buffer = new StringBuffer();
	TColumnInfo[] infos = metadata_.getColumnInfos();
	TColumnInfo[] primaries = metadata_.getPrimaryKeyInfos();
	buffer.append("CREATE TABLE ");
	buffer.append(_escapeString(quoteId_));
	buffer.append(tableName);
	buffer.append(_escapeString(quoteId_));
	buffer.append(" (");
	if (infos.length > 0) {
	    TColumnInfo info = infos[0];
	    _makeColumn(info, infos, buffer);
	    for (int i = 1;i < infos.length;i++) {
		info = infos[i];
		buffer.append(",");
		_makeColumn(info, infos, buffer);
	    }
	    if (primaries.length > 1) {
		buffer.append(",");
		buffer.append("PRIMARY KEY (");
		TColumnInfo primary = primaries[0];
		buffer.append(primary.getName());
		for (int i = 1;i < primaries.length;i++) {
		    primary = primaries[i];
		    buffer.append(",");
		    buffer.append(primary.getName());
		}
		buffer.append(")");
	    }
	    buffer.append(")");
	}
	return (new String(buffer));
    }

    public final String getDropTableCommand() {
	return (getDropTableCommand(name_));
    }

    public final String getDropTableCommand(String tableName) {
	return (
	    "DROP TABLE " +
	    _escapeString(quoteId_) + tableName + _escapeString(quoteId_)
	);
    }

    private final void _makeColumn(
	TColumnInfo column,
	TColumnInfo[] primaries,
	StringBuffer buffer
    ) {
	String columnName = column.getName();
	ISQLDatatype type = column.getSQLDatatype();
	buffer.append(_escapeString(quoteId_));
	buffer.append(columnName);
	buffer.append(_escapeString(quoteId_));
	buffer.append(" ");
	buffer.append(type.getDDLName());
	if (column.isPrimary()) {
	    if (primaries.length == 1) {
		buffer.append(" PRIMARY KEY");
	    } else {
		buffer.append(" NOT NULL");
	    }
	} else if (column.isUnique()) {
	    buffer.append(" UNIQUE");
	} else if (column.isNotNull()) {
	    buffer.append(" NOT NULL");
	}
    }

    private String _escapeString(String text) {
	if (text == null) {
	    return ("");
	}
	if (text.indexOf("\"") == -1) {
	    return (text);
	}
	StringBuffer buffer = new StringBuffer();
	int size = text.length();
	for (int i = 0;i < size;i++) {
	    char c = text.charAt(i);
	    if (c == '"') {
		buffer.append("\\\"");
	    } else {
		buffer.append(c);
	    }
	}
	return (new String(buffer));
    }

    private String _escapeNull(String text) {
	if (text == null) {
	    return ("");
	} else {
	    return (text);
	}
    }

    public final void setProperty(String key, Object value) {
	properties_.put(key, value);
    }

    public final Object getProperty(String key) {
	return (properties_.get(key));
    }
}
