/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2003  ASAMI, Tomoharu (asami@AsamiOffice.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.io;

import java.io.*;
import java.net.*;
import jp.gr.java_conf.jaba2.text.UString;

/**
 * URL is a utility for URL.
 *
 * @since   Mar. 23, 1998
 * @version Oct. 29, 2003
 * @author  ASAMI, Tomoharu (asami@AsamiOffice.com)
 */
public final class UURL {
    /**
     * Convert file name or URL name to URL. Name's kind is detected
     * automatically.
     *
     * @param name  file name or URL name.
     * @return URL
     */
    public static URL getURLFromFileOrURLName(String name)
	throws MalformedURLException {

	try {
	    if (name.startsWith("jdbc:")) {
		return (new URL(null, name, new JDBCStreamHandler()));
	    }
	    return (new URL(name));
	} catch (MalformedURLException e) {
	}
	return (getURLFromFileName(name));
    }

    /**
     * Convert URL name to URL.
     *
     * @param urlName  URL name to convert
     * @return URL
     */
    public static URL getURLFromURLName(String urlName)
	throws MalformedURLException {

	return (new URL(urlName));
    }

    /**
     * Convert file name to URL.
     *
     * @param filename  file name to convert
     * @return URL
     * @exception MalformedURLException filename is not valid file name form.
     */
    public static URL getURLFromFileName(String filename)
	throws MalformedURLException {

	return (getURLFromFile(new File(filename)));
    }

    /**
     * Convert file name to URL.
     *
     * @param file  file to convert
     * @return URL
     */
    public static URL getURLFromFile(File file) throws MalformedURLException {
	return (file.toURL());
    }

    /**
     * Convert resource name to URL.
     * <br>
     * This method uses a ClassLoader specified by the parameter object
     * to access the resource.
     *
     * @param resourceName  resource name to convert
     * @param object  object related resource.
     * @return URL
     */
    public static URL getURLFromResourceName(
	String resourceName,
	ClassLoader loader
    ) {
        URL url = loader.getResource(resourceName);
        if (url != null) {
            return (url);
        }
	if (resourceName.startsWith("/")) {
	    resourceName = resourceName.substring(1);
	} else {
            resourceName = "/" + resourceName;
        }
        return (loader.getResource(resourceName));
    }

    /**
     * Convert resource name to URL.
     * <br>
     * This method uses a ClassLoader specified by the parameter clazz
     * to access the resource.
     *
     * @param resourceName  resource name to convert
     * @param clazz  class related resource.
     * @return URL
     */
    public static URL getURLFromResourceName(
	String resourceName,
	Class clazz
    ) {
	return (getURLFromResourceName(resourceName, clazz.getClassLoader()));
    }

    /**
     * Convert resource name to URL.
     * <br>
     * This method uses a ClassLoader specified by the parameter object
     * to access the resource.
     *
     * @param resourceName  resource name to convert
     * @param object  object related resource.
     * @return URL
     */
    public static URL getURLFromResourceName(
	String resourceName,
	Object object
    ) {
	return (getURLFromResourceName(resourceName, object.getClass()));
    }

    public static boolean isURL(String src) {
	try {
	    URL url = new URL(src);
	    return (true);
	} catch (MalformedURLException e) {
	    return (false);
	}
    }

    public static String getLastComponent(URL url) {
	return (UString.getLastComponent(url.toExternalForm()));
    }

    public static String getBaseURI(URL url) {
	String uri = url.toExternalForm();
	return (UString.getContainerPathname(uri) + "/");
    }
}
