/*
 * The JabaJaba class library
 *  Copyright (C) 1997-1998  ASAMI, Tomoharu (tasami@ibm.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.awt;

import java.util.List;
import java.util.ArrayList;
import java.awt.LayoutManager2;
import java.awt.Component;
import java.awt.Container;
import java.awt.Point;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.Insets;
import jp.gr.java_conf.jaba2.util.Debug;
import jp.gr.java_conf.jaba2.util.D2Array;

/**
 * TableLayout
 *
 * @since   Jul. 18, 1998
 * @version Jul. 30, 1998
 * @author ASAMI, Tomoharu (tasami@ibm.net)
 */
public class PropertySheetLayout extends AbstractLayoutManager
    implements PropertySheetConstants {

    List entries_ = new ArrayList(); // List<Entry>
    Entry current_ = null;

    public PropertySheetLayout() {
	super(5, 5, 5, 5);
    }

    public void addLayoutComponent(Component comp, Object constraints) {
	String action = (String)constraints;
	if ("first".equalsIgnoreCase(action)) {
	    action = "line";
	}
	if ("line".equalsIgnoreCase(action)) {
	    current_ = new Entry(DEPLOY_LINE);
	    entries_.add(current_);
	    current_.comps[Entry.NAME] = comp;
	} else if ("pack".equalsIgnoreCase(action)) {
	    current_ = new Entry(DEPLOY_PACK);
	    entries_.add(current_);
	    current_.comps[Entry.NAME] = comp;
	} else if ("area".equalsIgnoreCase(action)) {
	    current_ = new Entry(DEPLOY_AREA);
	    entries_.add(current_);
	    current_.comps[Entry.NAME] = comp;
	} else {
	    if (current_.comps[Entry.SEPARATOR] instanceof NullComponent) {
		current_.comps[Entry.SEPARATOR] = comp;
	    } else if (current_.comps[Entry.VALUE] instanceof NullComponent) {
		current_.comps[Entry.VALUE] = comp;
	    } else if (current_.comps[Entry.MORE] instanceof NullComponent) {
		current_.comps[Entry.MORE] = comp;
	    } else {
		throw (new IllegalArgumentException("Already sufficient"));
	    }
	}
    }

    public Dimension preferredLayoutSize(Container target) {
	Insets insets = target.getInsets();
	PanelData data = new PanelData();
	return (
	    _adjustBorder(
		new Dimension(data.maxWidth, data.maxHeight), insets
	    )
	);
    }

    public void layoutContainer(Container target) {
	if (target.getComponentCount() == 0) {
	    return;
	}
	Dimension size = target.getSize();
	if (size.width == 0 || size.height == 0) {
	    return;
	}
	Insets insets = target.getInsets();
	Dimension viewSize = _getViewSize(target);
	PanelData panel = new PanelData();
	List lines = panel.lines;
	int nLines = panel.nLines;
	// calc height for area
	int linesHeight = 0;
	if (nLines > 0) {
	    LineData data = (LineData)panel.lines.get(0);
	    linesHeight = data.indicatorHeight;
	    for (int i = 1;i < nLines;i++) {
		linesHeight += vInnerGap_;
		data = (LineData)panel.lines.get(i);
		linesHeight += data.indicatorHeight;
	    }
	}
	int heightForValueAreas = viewSize.height - linesHeight;
	int heightPerValueArea;
	if (panel.nAreas == 0) {
	    heightPerValueArea = 0;
	} else {
	    heightPerValueArea = heightForValueAreas / panel.nAreas;
	}
	// XXX : divide Packed lines to fit the view size
	//
	int x = 0;
	int y = 0;
	for (int i = 0;i < nLines;i++) {
	    LineData data = (LineData)lines.get(i);
	    Entry entry;
	    Rectangle area;
	    Component comp;
	    switch (data.style) {

	    case DEPLOY_LINE:
		x = 0;
		entry = (Entry)data.entries.get(0);
		entry.areas[Entry.NAME].x = x;
		entry.areas[Entry.NAME].y = y;
		entry.areas[Entry.NAME].width = panel.nameWidth;
		entry.areas[Entry.NAME].height = data.height;
		x += entry.areas[Entry.NAME].width + hInnerGap_;
		entry.areas[Entry.SEPARATOR].x = x;
		entry.areas[Entry.SEPARATOR].y = y;
		entry.areas[Entry.SEPARATOR].width = panel.separatorWidth;
		entry.areas[Entry.SEPARATOR].height = data.height;
		x += entry.areas[Entry.SEPARATOR].width + hInnerGap_;
		entry.areas[Entry.VALUE].x = x;
		entry.areas[Entry.VALUE].y = y;
		entry.areas[Entry.VALUE].width = panel.valueWidth;
		entry.areas[Entry.VALUE].height = data.height;
		x += entry.areas[Entry.VALUE].width + hInnerGap_;
		entry.areas[Entry.MORE].x = x;
		entry.areas[Entry.MORE].y = y;
		entry.areas[Entry.MORE].width = panel.moreWidth;
		entry.areas[Entry.MORE].height = data.height;
		y += data.height + vInnerGap_;
		break;
	    case DEPLOY_PACK:
		throw (new InternalError("under construction"));
	    case DEPLOY_AREA:
		throw (new InternalError("under construction"));
	    default:
		throw (new InternalError("bad style : " + data.style));
	    }
	}
	// do layout
	List comps = new ArrayList();
	List areas = new ArrayList();
	int nEntries = entries_.size();
	for (int i = 0;i < nEntries;i++) {
	    Entry entry = (Entry)entries_.get(i);
	    comps.add(entry.comps[Entry.NAME]);
	    comps.add(entry.comps[Entry.SEPARATOR]);
	    comps.add(entry.comps[Entry.VALUE]);
	    comps.add(entry.comps[Entry.MORE]);
	    areas.add(entry.areas[Entry.NAME]);
	    areas.add(entry.areas[Entry.SEPARATOR]);
	    areas.add(entry.areas[Entry.VALUE]);
	    areas.add(entry.areas[Entry.MORE]);
	}
	_moveComponentsOnView(comps, areas, insets);
    }

    protected int _calcNameWidth() {
	int width = 0;
	int nEntries = entries_.size();
	for (int i = 0;i < nEntries;i++) {
	    Entry entry = (Entry)entries_.get(i);
	    width = Math.max(width, entry.prefs[Entry.NAME].width);
	}
	return (width);
    }

    protected int _calcSeparatorWidth() {
	int width = 0;
	int nEntries = entries_.size();
	for (int i = 0;i < nEntries;i++) {
	    Entry entry = (Entry)entries_.get(i);
	    width = Math.max(width, entry.prefs[Entry.SEPARATOR].width);
	}
	return (width);
    }

    protected int _calcValueWidth() {
	int width = 0;
	int nEntries = entries_.size();
	for (int i = 0;i < nEntries;i++) {
	    Entry entry = (Entry)entries_.get(i);
	    width = Math.max(width, entry.prefs[Entry.VALUE].width);
	}
	return (width);
    }

    protected int _calcMoreWidth() {
	int width = 0;
	int nEntries = entries_.size();
	for (int i = 0;i < nEntries;i++) {
	    Entry entry = (Entry)entries_.get(i);
	    width = Math.max(width, entry.prefs[Entry.MORE].width);
	}
	return (width);
    }

    protected void _syncPrefs() {
	int nEntries = entries_.size();
	for (int i = 0;i < nEntries;i++) {
	    Entry entry = (Entry)entries_.get(i);
	    entry.syncPrefs();
	}
    }

    static class Entry {
	static int NAME = 0;
	static int SEPARATOR = 1;
	static int VALUE = 2;
	static int MORE = 3;

	int style;
	Component[] comps = new Component[4];
	Dimension[] prefs = new Dimension[4];
	Rectangle[] areas = new Rectangle[4];
	int prefLineHeight;
	int prefAreaHeight;

	Entry (int style) {
	    this.style = style;
	    comps[NAME] = new NullComponent();
	    comps[SEPARATOR] = new NullComponent();
	    comps[VALUE] = new NullComponent();
	    comps[MORE] = new NullComponent();
	}

	void syncPrefs() {
	    if (style == DEPLOY_AREA) {
		prefLineHeight = 0;
		Dimension pref;
		pref = comps[NAME].getPreferredSize();
		prefs[NAME] = pref;
		prefLineHeight = Math.max(prefLineHeight, pref.height);
		pref = comps[SEPARATOR].getPreferredSize();
		prefs[SEPARATOR] = pref;
		prefLineHeight = Math.max(prefLineHeight, pref.height);
		pref = comps[MORE].getPreferredSize();
		prefs[MORE] = pref;
		prefLineHeight = Math.max(prefLineHeight, pref.height);
		pref = comps[VALUE].getPreferredSize();
		prefs[VALUE] = pref;
		prefAreaHeight = pref.height;
	    } else {
		prefLineHeight = 0;
		for (int i = NAME;i <= MORE;i++) {
		    Dimension pref = comps[i].getPreferredSize();
		    prefs[i] = pref;
		    prefLineHeight = Math.max(prefLineHeight, pref.height);
		}
		prefAreaHeight = 0;
	    }
	}
    }

    static class LineData {
	int style;
	int width = 0;
	int height = 0;
	int indicatorHeight = 0;
	List entries = new ArrayList(); // List<Entry>

	LineData(int style) {
	    this.style = style;
	}
    }

    class PanelData {
	int nameWidth;
	int separatorWidth;
	int valueWidth;
	int moreWidth;
	List lines = new ArrayList(); // List<LineData>
	int nLines;
	int nAreas = 0;
	int maxWidth = 0;
	int maxHeight = 0;

	PanelData() {
	    LineData data = null;
	    int nEntries = entries_.size();
	    int x = 0;
	    int y = 0;
	    _syncPrefs();
	    nameWidth = _calcNameWidth();
	    separatorWidth = _calcSeparatorWidth();
	    valueWidth = _calcValueWidth();
	    moreWidth = _calcMoreWidth();
	    for (int i = 0;i < nEntries;i++) {
		Entry entry = (Entry)entries_.get(i);
		switch (entry.style) {

		case DEPLOY_LINE:
		    data = new LineData(DEPLOY_LINE);
		    lines.add(data);
		    x = 0;
		    data.height = entry.prefLineHeight;
		    data.indicatorHeight = entry.prefLineHeight;
		    data.entries.add(entry);
		    entry.areas[Entry.NAME] = 
			new Rectangle(x, y, nameWidth, data.height);
		    x += nameWidth + hInnerGap_;
		    entry.areas[Entry.SEPARATOR] =
			new Rectangle(x, y, separatorWidth, data.height);
		    x += separatorWidth + hInnerGap_;
		    entry.areas[Entry.VALUE] = 
			new Rectangle(x, y, valueWidth, data.height);
		    x += valueWidth + hInnerGap_;
		    entry.areas[Entry.MORE] = 
			new Rectangle(x, y, moreWidth, data.height);
		    data.width = x + moreWidth;
		    maxWidth = Math.max(maxWidth, data.width);
		    y += entry.prefLineHeight;
		    maxHeight = y;
		    y += vInnerGap_;
//System.out.println(data.width + '/' + data.height + '/' + maxWidth + '/' + maxHeight);
		    data = null;
		    break;
		case DEPLOY_PACK:
		    if (data == null) {
			data = new LineData(DEPLOY_PACK);
			lines.add(data);
		    }
		    data.height = Math.max(data.height, entry.prefLineHeight);
		    data.indicatorHeight
			= Math.max(data.height, entry.prefLineHeight);
		    data.entries.add(entry);
		    entry.areas[Entry.NAME] =
			new Rectangle(x, y, nameWidth, data.height);
		    x += nameWidth + hInnerGap_;
		    entry.areas[Entry.SEPARATOR] =
			new Rectangle(x, y, separatorWidth, data.height);
		    x += separatorWidth + hInnerGap_;
		    entry.areas[Entry.VALUE] = 
			new Rectangle(x, y, valueWidth, data.height);
		    x += valueWidth + hInnerGap_;
		    entry.areas[Entry.MORE] =
			new Rectangle(x, y, moreWidth, data.height);
		    x += moreWidth;
		    maxWidth = Math.max(maxWidth, x);
		    x += hInnerGap_; // XXX : per field gap?
		    data.width = x;
		    break;
		case DEPLOY_AREA:
		    nAreas++;
		    data = new LineData(DEPLOY_AREA);
		    lines.add(data);
		    data.entries.add(entry);
		    entry.areas[Entry.NAME] = 
			new Rectangle(x, y, nameWidth, entry.prefLineHeight);
		    x += nameWidth + hInnerGap_;
		    entry.areas[Entry.SEPARATOR] =
			new Rectangle(
			    x,
			    y,
			    separatorWidth,
			    entry.prefLineHeight
			);
		    x += valueWidth + hInnerGap_;
		    entry.areas[Entry.MORE] =
			new Rectangle(x, y, moreWidth, entry.prefLineHeight);
		    data.width = x + moreWidth;
		    x = 0;
		    y += entry.prefLineHeight + vInnerGap_;
		    entry.areas[Entry.VALUE] =
			new Rectangle(
			    x, y, valueWidth, entry.prefAreaHeight
			);
		    data.width = Math.max(data.width, valueWidth);
		    data.indicatorHeight = entry.prefLineHeight;
		    data.height =
			data.indicatorHeight +
			vInnerGap_ +
			entry.prefAreaHeight;
		    maxWidth = Math.max(maxWidth, data.width);
		    y += entry.prefAreaHeight;
		    maxHeight = y;
		    y += vInnerGap_;
		    data = null;
		    break;
		default:
		    throw (new InternalError("bad style : " + entry.style));
		}
	    }
	    nLines = lines.size();
	}
    }
}
