/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2002  ASAMI, Tomoharu (asami@AsamiOffice.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.xml.relaxng;

import java.util.*;
import java.io.IOException;
import java.net.URL;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;
import jp.gr.java_conf.jaba2.text.UString;
import jp.gr.java_conf.jaba2.datatype.XMLFacet;
import jp.gr.java_conf.jaba2.xml.*;
import jp.gr.java_conf.jaba2.xml.relaxng.rRelaxng.*;
import jp.gr.java_conf.jaba2.xml.relax.cooked.*;
import jp.gr.java_conf.jaba2.xml.relax.expanded.*;

/**
 * RNGGrammar
 *
 * @since   Jun. 15, 2001
 * @version Dec. 17, 2002
 * @author  ASAMI, Tomoharu (asami@AsamiOffice.com)
 */
public class RNGGrammar {
    private static final int OCCURS_ONE = 1;
    private static final int OCCURS_ZEROONE = 2;
    private static final int OCCURS_ONEMORE = 3;
    private static final int OCCURS_ZEROMORE = 4;
    private String baseURI_;
    private RNGGrammarConfig config_ = new RNGGrammarConfig();
    private Map modules_ = new HashMap(); // Map<String, RNGModule>
    private Map nsByLabel_ = new HashMap(); // Map<String, String>
    private List unresolvedRefs_ = new ArrayList();
    private List parentRefs_ = new ArrayList();

    public RNGGrammar(RNGrammar grammar, String baseURI) {
	baseURI_ = baseURI;
	RNGGrammarBuilder builder = new RNGGrammarBuilder(grammar, baseURI);
	_buildDefine(builder.getDefines());
	_buildStart(builder.getStarts());
	_resolveRefs();
    }

    public RNGGrammar(RNElement element, String baseURI) {
	baseURI_ = baseURI;
	RNGGrammarBuilder builder = new RNGGrammarBuilder(element, baseURI);
	_buildDefine(builder.getDefines());
	_buildStart(builder.getStarts());
/*
	String name = element.getName();
	String uri = _getNamespace(element);
	RNDefine define = new RNDefine();
	define.setName(name);
	define.addElementHedge(element);
	_buildDefine(define);
	RNRef ref = new RNRef();
	ref.setName(name);
	RNGModule module = _getModule(uri);
	module.addExportLabel(name);
*/
	_resolveRefs();
    }

    private void _buildDefine(List defineList) {
//System.out.println("l = " + defineList.size());
	RNDefine[] defines = _getDefines(defineList);
        for (int i = 0;i < defines.length;i++) {
	    RNDefine define = defines[i];
//System.out.println("Z = " + define.getName());
	    _buildDefine(define);
	}
    }

    private RNDefine[] _getDefines(List defineList) {
	RNDefine[] defines = new RNDefine[defineList.size()];
	return ((RNDefine[])defineList.toArray(defines));
    }

    private void _buildStart(List startList) {
	RNStart start = _getStart(startList);
	_buildStart(start);
    }

    private RNStart _getStart(List startList) {
	switch (startList.size()) {

	case 0:
	    throw (new RNGSyntaxErrorException("No start"));
	case 1:
	    return ((RNStart)startList.get(0));
	default:
	    RNStart newStart = new RNStart();
	    RNChoice choice = new RNChoice();
	    int size = startList.size();
	    for (int i = 0;i < size;i++) {
		RNStart oldStart = (RNStart)startList.get(i);
		choice.addElementHedge(oldStart.getElementHedge());
	    }
	    newStart.addElementHedge(choice);
	    return (newStart);
	}
    }

    public RNGModule[] getModules() {
	Collection values = modules_.values();
	RNGModule[] modules = new RNGModule[values.size()];
	return ((RNGModule[])values.toArray(modules));
    }

    public RNStart getStart() {
	throw (new UnsupportedOperationException());
    }

    private void _buildStart(RNStart start) {
	IRNElementHedgeChoice[] nodes = start.getElementHedge();
	_buildStart(nodes);
    }

    private void _buildStart(IRNElementHedgeChoice[] nodes) {
	for (int i = 0;i < nodes.length;i++) {
	    IRNElementHedgeChoice node = nodes[i];
	    if (node instanceof RNElement) {
		_buildStartElement((RNElement)node);
	    } else if (node instanceof RNRef) {
		_buildStartRef((RNRef)node);
	    } else if (node instanceof RNChoice) {
		_buildStartChoice((RNChoice)node);
	    } else if (node instanceof RNExternalRef) {
		_buildStartExternalRef((RNExternalRef)node);
	    }
	}
    }

    private void _buildStartElement(RNElement element) {
	String elementName = _getElementName(element);
	String labelName = _makeLabelNameElement(
	    config_.getStartLabel(),
	    elementName
	);
	_buildElement(element, labelName);
	RNGModule module = _getModule(element);
	module.addExportLabel(labelName);
    }

    private void _buildStartRef(RNRef ref) {
	String name = ref.getName();
	RNGModule module = _getTargetModule(ref);
	module.addExportLabel(name);
    }

    private void _buildStartChoice(RNChoice choice) {
	IRNElementHedgeChoice[] contents = choice.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		_buildStartElement((RNElement)content);
	    } else if (content instanceof RNRef) {
		_buildStartRef((RNRef)content);
	    } else if (content instanceof RNChoice) {
		_buildStartChoice((RNChoice)content);
	    } else if (content instanceof RNExternalRef) {
		_buildStartExternalRef((RNExternalRef)content);
	    }
	}
    }

    private void _buildStartExternalRef(RNExternalRef ref) {
	List list = new ArrayList();
	_buildExternalRef(ref, "", list);
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    CRefWithLabel cRef = (CRefWithLabel)list.get(i);
	    RNGModule module;
	    String namespace = cRef.getNamespace();
	    if (namespace == null) {
		module = _getModule(ref);
	    } else {
		module = _getModule(namespace);
	    }
	    module.addExportLabel(cRef.getLabel());
	}
    }

    private void _resolveRefs() {
	int size = unresolvedRefs_.size();
	for (int i = 0;i < size;i++) {
	    RefPair pair = (RefPair)unresolvedRefs_.get(i);
	    RNGModule contextModule = _getModule(pair.ngRef);
	    RNGModule targetModule = _getTargetModule(pair.ngRef);
	    if (contextModule != targetModule) {
		pair.cRef.setNamespace(targetModule.getNamespaceURI());
		CModule cModule = targetModule.getCModule();
		cModule.addExportLabel(pair.ngRef.getName());
	    }
	}
    }

    private void _resolveParentRefs() {
	// XXX
    }

    private void _buildDefine(RNDefine define) {
//System.out.print("buildDefine:");
//System.out.print("name = " + define.getName());
//System.out.println(" ns = " + define.getNs());
	IRNElementHedgeChoice[] nodes = define.getElementHedge();
	if (nodes.length == 1 &&
	    nodes[0] instanceof RNElement) {

	    _buildElementRule(define, (RNElement)nodes[0]);
	} else {
	    _buildHedgeRule(define, nodes);
	}
    }

    private RNGModule _getModule(RNDefine define) {
	return (_getModule(_getNamespace(define)));
    }

    private RNGModule _getModule(RNElement element) {
	return (_getModule(_getNamespace(element)));
    }

    private RNGModule _getModule(RNAttribute attribute) {
	return (_getModule(_getNamespace(attribute)));
    }

    private RNGModule _getModule(IRNode node) {
	return (_getModule(_getNamespace(node)));
    }

    private String _getNamespace(IRNode node) {
	String ns = _getNodeNamespaceWithPrefix(node);
	if (ns != null) {
	    return (ns);
	}
	for (;;) {
	    IRNode parent = node.getParentRNode();
	    if (parent == null) {
		return ("");
	    }
	    node = parent;
//	    ns = _getNodeNamespaceWithoutPrefix(node);
	    ns = _getNodeNamespaceWithPrefix(node);
	    if (ns != null) {
		return (ns);
	    }
	}
    }

    private String _getContextNamespace(IRNode node) {
	for (;;) {
	    IRNode parent = node.getParentRNode();
	    if (parent == null) {
		return ("");
	    }
	    node = parent;
	    String ns = _getNodeNamespaceWithPrefix(node);
	    if (ns != null) {
		return (ns);
	    }
	}
    }

    private String _getNodeNamespaceWithoutPrefix(IRNode node) {
	if (node instanceof RNElement) {
	    RNElement element = (RNElement)node;
	    String ns = element.getNs();
	    if (ns != null) {
		return (ns);
	    }
	} else if (node instanceof RNAttribute) {
	    RNAttribute attribute = (RNAttribute)node;
	    String ns = attribute.getNs();
	    if (ns == null) {
		return ("");
	    } else {
		return (ns);
	    }
	} else if (node instanceof RNGrammar) {
	    RNGrammar grammar = (RNGrammar)node;
	    String ns = grammar.getNs();
	    if (ns != null) {
		return (ns);
	    }
	} else if (node instanceof RNStart) {
	    RNStart start = (RNStart)node;
	    String ns = start.getNs();
	    if (ns != null) {
		return (ns);
	    }
	} else if (node instanceof RNDefine) {
	    RNDefine define = (RNDefine)node;
	    String ns = define.getNs();
	    if (ns != null) {
		return (ns);
	    }
	}
	return (null);
    }

    private String _getNodeNamespaceWithPrefix(IRNode node) {
	if (node instanceof RNElement) {
	    RNElement element = (RNElement)node;
	    String ns = element.getNs();
	    if (ns != null) {
		return (ns);
	    }
	    String name = element.getName();
	    if (name != null) {
		ns = _getNamespaceByNamePrefix(
		    element.getRNSContext(),
		    name
		);
		if (ns != null) {
		    return (ns);
		}
	    }
	} else if (node instanceof RNAttribute) {
	    RNAttribute attribute = (RNAttribute)node;
	    String ns = attribute.getNs();
	    if (ns != null) {
		return (ns);
	    }
	    String name = attribute.getName();
	    if (name != null) {
		ns = _getNamespaceByNamePrefix(
		    attribute.getRNSContext(),
		    name
		);
		if (ns == null) {
		    return ("");
		} else {
		    return (ns);
		}
	    }
	} else if (node instanceof RNGrammar) {
	    RNGrammar grammar = (RNGrammar)node;
	    String ns = grammar.getNs();
	    if (ns != null) {
		return (ns);
	    }
	} else if (node instanceof RNStart) {
	    RNStart start = (RNStart)node;
	    String ns = start.getNs();
	    if (ns != null) {
		return (ns);
	    }
	} else if (node instanceof RNDefine) {
	    RNDefine define = (RNDefine)node;
	    String ns = define.getNs();
	    if (ns != null) {
		return (ns);
	    }
	    String name = define.getName();
	    if (name == null) {
		throw (new RNGSyntaxErrorException("Unnamed define"));
	    }
	    ns = _getNamespaceByNamePrefix(
		define.getRNSContext(),
		name
	    );
	    if (ns != null) {
		return (ns);
	    }
	}
	return (null);
    }

/*
    private String _getNamespace(RNDefine define) {
	String name = define.getName();
	if (name == null) {
	    throw (new InternalError("syntax error"));
	}
	String ns = _getNamespaceByNamePrefix(name);
	if (ns != null) {
	    return (ns);
	} else {
	    return (_getNamespace(define));
	}
    }

    private String _getNamespace(RNElement element) {
	String name = element.getName();
	if (name == null) {
	    return (_findNamespace(element));
	}
	int index = name.indexOf(":");
	if (index == -1) {
	    return (_findNamespace(element));
	} else {
	    String prefix = name.substring(0, index);
	    RNSContext nsContext = element.getRNSContext();
	    return (nsContext.getUriByPrefix(prefix));
	}
    }

    private String _getNamespace(RNAttribute attribute) {
	String name = attribute.getName();
	if (name == null) {
	    String ns = attribute.getNs();
	    // XXX global?
	    if (ns == null) {
		return ("");
	    } else {
		return (ns);
	    }
	}    
	int index = name.indexOf(":");
	if (index == -1) {
	    String ns = attribute.getNs();
	    // XXX global?
	    if (ns == null) {
		return ("");
	    } else {
		return (ns);
	    }
	} else {
	    String prefix = name.substring(0, index);
	    RNSContext nsContext = attribute.getRNSContext();
	    return (nsContext.getUriByPrefix(prefix));
	}
    }
*/

    private String _getNamespaceByNamePrefix(
	RNSContext nsContext,
	String name
    ) {
	int index = name.indexOf(":");
	if (index == -1) {
	    return (null);
	}
	String prefix = name.substring(0, index);
	return (nsContext.getUriByPrefix(prefix));
    }

/*
    private String _findNamespace(IRNode node) {
	for (;;) {
	    if (node instanceof RNElement) {
		RNElement element = (RNElement)node;
		String ns = element.getNs();
		if (ns != null) {
		    return (ns);
		}
		String name = element.getName();
		if (name != null) {
		    ns = _getNamespaceByNamePrefix(name);
		    if (ns != null) {
			return (ns);
		    }
		}
	    } else if (node instanceof RNAttribute) {
		RNAttribute attribute = (RNAttribute)node;
		String ns = attribute.getNs();
		if (ns != null) {
		    return (ns);
		}
		String name = attribute.getName();
		if (name != null) {
		    ns = _getNamespaceByNamePrefix(name);
		    if (ns != null) {
			return (ns);
		    }
		}
	    } else if (node instanceof RNGrammar) {
		RNGrammar grammar = (RNGrammar)node;
		String ns = grammar.getNs();
		if (ns != null) {
		    return (ns);
		}
	    } else if (node instanceof RNStart) {
		RNStart start = (RNStart)node;
		String ns = start.getNs();
		if (ns != null) {
		    return (ns);
		}
	    } else if (node instanceof RNDefine) {
		RNDefine define = (RNDefine)node;
		String ns = define.getNs();
		if (ns != null) {
		    return (ns);
		}
		ns = _getNamespaceByNamePrefix(define.getName());
		if (ns != null) {
		    return (ns);
		}
	    }
	    IRNode parent = node.getParentRNode();
	    if (parent == null) {
		return ("");
	    }
	    node = parent;
	}
    }
*/

    private RNGModule _getTargetModule(RNRef ref) {
	String name = ref.getName();
	if (name == null || "".equals(name)) {
	    throw (new RNGSyntaxErrorException("Unnamed ref"));
	}
	String ns = (String)nsByLabel_.get(name);
	if (ns == null) {
	    throw (new RNGSyntaxErrorException("Unavailable name = " + name));
	}
	return (_getModule(ns));
    }

    private RNGModule _getModule(String namespaceURI) {
	RNGModule module = (RNGModule)modules_.get(namespaceURI);
	if (module == null) {
	    module = new RNGModule(namespaceURI);
	    module.setBaseURI(baseURI_);
	    modules_.put(namespaceURI, module);
	}
	return (module);
    }

    private void _buildElementRule(RNDefine define, RNElement element) {
	String labelName = define.getName();
	_buildElement(element, labelName);
    }

    private CRefWithLabel _buildElementWithRef(
	RNElement element,
	String context,
	int occurs
    ) {
	String elementName = _getElementName(element);
	String uri = _getNamespace(element);
	String contextUri = _getContextNamespace(element);
	String labelName = _makeLabelNameElement(context, elementName);
	_buildElement(element, labelName);
	CRefWithLabel ref;
	if (uri.equals(contextUri)) {
	    ref = new CRefWithLabel(labelName, _makeCOccurs(occurs));
	} else {
	    ref = new CRefWithLabel(labelName, uri, _makeCOccurs(occurs));
	    RNGModule ngModule = _getModule(element);
	    ngModule.addExportLabel(labelName);	// XXX
	}
	ref.setBase(element.getXMLElement());
	return (ref);
    }

    private void _buildElement(
	RNElement element,
	String labelName
    ) {
	String elementName = _getElementName(element);
	IRNElementHedgeChoice[] contents = element.getElementHedge();
	RNGModule ngModule = _getModule(element);
	CModule cModule = ngModule.getCModule();
	CTagPattern cTag = new CTagPattern(labelName, elementName, cModule);
	cTag.setBase(element.getXMLElement());
	CSequence cSequence = new CSequence();
	String typeName = null;
	XMLFacet[] facets = null;
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
//		    _makeLabelName(labelName, elementName),
		    labelName,
		    OCCURS_ONE
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		_buildCTag(ngAttr, true, cTag, labelName, cModule);
/*
		String value = _getValueText(ngAttr);
		if (value != null) {
		    cTag.addContent(new CValue(value));
		} else {
		    String type = _getTypeName(ngAttr);
		    CAttribute cAttr = new CAttribute(
			ngAttr.getName(),
			type,
			cModule
		    );
		    cAttr.setRequired(true);
		    cTag.addContent(cAttr);
		}
*/
	    } else if (content instanceof RNText) {
		typeName = "string";
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		typeName = data.getType();
		facets = _makeFacets(data);
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String valueType = value.getType();
		cSequence.addParticle(
		    new CValue(value.getContent(), typeName)
		);
	    } else if (content instanceof RNList) {
		RNList list = (RNList)content;
		typeName = list.makeTextDocument();
	    } else if (content instanceof RNEmpty) {
		// do nothing
	    } else if (content instanceof RNGroup) {
		CSequence cGroup = new CSequence();
		_buildGroup((RNGroup)content, labelName, cGroup);
		cSequence.addParticle(cGroup);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, labelName, cChoice);
		cSequence.addParticle(cChoice);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave();
		_buildInterleave(
		    (RNInterleave)content,
		    labelName,
		    cInterleave
		);
		cSequence.addParticle(cInterleave);
	    } else if (content instanceof RNMixed) {
		CMixed cMixed = new CMixed();
		_buildMixed((RNMixed)content, labelName, cMixed);
		cSequence.addParticle(cMixed);
/*
		CMixedElementRule cRule = new CMixedElementRule(
		    labelName,
		    cMixed,
		    cModule
		);
		cModule.addElementRule(cRule);
		cRule.setBase(element.getXMLElement());
		nsByLabel_.put(labelName, cModule.getTargetNamespace());
		cModule.addTagPattern(cTag);
		return;
*/
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    labelName,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cTag.addContent(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    labelName,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    labelName,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    labelName,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNNotAllowed) {
		cSequence.addParticle(new CNone());
	    } else if (content instanceof RNGrammar) {
		List externalContents = new ArrayList();
		_buildGrammar(
		    (RNGrammar)content,
		    labelName,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else {
		throw (new InternalError(content.toString()));
	    }
	}
	if (_hasNotAllowed(cSequence)) {
	    CNoneElementRule cRule = new CNoneElementRule(labelName);
	    cRule.setBase(element.getXMLElement());
	    cModule.addElementRule(cRule);
	    nsByLabel_.put(labelName, cModule.getTargetNamespace());
	} else if (typeName != null) {
	    CRelaxNgElementRule cRule = new CRelaxNgElementRule(
		    labelName,
		    cSequence,
		    typeName,
		    facets,
		    cModule
	    );
	    cRule.setBase(element.getXMLElement());
	    cModule.addElementRule(cRule);
	    nsByLabel_.put(labelName, cModule.getTargetNamespace());
	} else {
	    if (cSequence.getParticleCount() == 0) {
		CEmptyElementRule cRule = new CEmptyElementRule(
		    labelName,
		    cModule
		);
		cModule.addElementRule(cRule);
		cRule.setBase(element.getXMLElement());
		nsByLabel_.put(labelName, cModule.getTargetNamespace());
	    } else {
		CSequenceElementRule cRule = new CSequenceElementRule(
		    labelName,
		    cSequence,
		    cModule
		);
		cModule.addElementRule(cRule);
		cRule.setBase(element.getXMLElement());
		nsByLabel_.put(labelName, cModule.getTargetNamespace());
	    }
	}
	cModule.addTagPattern(cTag);
    }

    private boolean _hasNotAllowed(CSequence sequence) {
	ICParticle[] particles = sequence.getParticles();
	for (int i = 0;i < particles.length;i++) {
	    ICParticle particle = particles[i];
	    if (particle instanceof CNone) {
		return (true);
	    }
	}
	return (false);
    }

    private String _getElementName(RNElement element) {
	return (_getElementNameCandidate(element).replace('"', '\''));
    }

    private String _getElementNameCandidate(RNElement element) {
	String elementName = element.getName();
	if (elementName != null) {
	    return (_getNamePart(elementName));
	}
	IRNNameClassHedgeChoice child = element.getNameClassHedge();
	return (_getAttributeNameCandidateNameClass(child));
    }

    private String _getAttributeName(RNAttribute attribute) {
	return (_getAttributeNameCandidate(attribute).replace('"', '\''));
    }

    private String _getAttributeNameCandidate(RNAttribute attribute) {
	String attributeName = attribute.getName();
	if (attributeName != null) {
	    return (_getNamePart(attributeName));
	}
	IRNNameClassHedgeChoice child = attribute.getNameClassHedge();
	return (_getAttributeNameCandidateNameClass(child));
    }

    private String _getAttributeNameCandidateNameClass(
	IRNNameClassHedgeChoice child
    ) {
	if (child instanceof RNName) {
	    RNName name = (RNName)child;
	    return (_getNamePart(name.getContent()));
	} else if (child instanceof RNAnyName) {
	    return (child.makeTextDocument());
	} else if (child instanceof RNNsName) {
	    return (child.makeTextDocument());
	} else if (child instanceof RNNameChoice) {
	    return (child.makeTextDocument());
	} else if (child instanceof RNExcept) {
	    return (child.makeTextDocument());
	} else {
	    throw (new RNGSyntaxErrorException("Invalid argument name"));
	}
    }

    private String _getAttributeNs(RNAttribute attribute) {
	// XXX : already resolved by NsResolver
//System.out.println("ns = " + attribute.getNs());
	return (attribute.getNs());
    }

    private String _getNamePart(String name) {
	// XXX : already resolved by NsResolver
	int index = name.indexOf(":");
	if (index == -1) {
	    return (name.trim());
	} else {
	    return (name.substring(index + 1).trim());
	}
    }

    private void _buildCTag(
	RNAttribute ngAttr,
	boolean isRequired,
	CTagPattern cTag,
	String context,
	CModule cModule
    ) {
	CAttribute cAttr = _makeCAttribute(
	    ngAttr,
	    isRequired,
	    context,
	    cModule
	);
	if (cAttr == null) {
	    return;
	}
	cTag.addContent(cAttr);
    }

    private CAttribute _makeCAttribute(
	RNAttribute ngAttr,
	boolean isRequired,
	String context,
	CModule cModule
    ) {
	if (isRequired) {
	    return (_makeCAttribute(ngAttr, "", context, cModule));
	} else {
	    return (_makeCAttribute(ngAttr, "?", context, cModule));
	}
    }

    private CAttribute _makeCAttribute(
	RNAttribute ngAttr,
	String occurs,
	String context,
	CModule cModule
    ) {
	IRNTextHedgeChoice content = ngAttr.getTextHedge();
	if (content == null || content instanceof RNText) {
	    return (_makeCAttributeText(ngAttr, occurs, context, cModule));
	} else if (content instanceof RNData) {
	    return (
		_makeCAttributeData(
		    ngAttr,
		    (RNData)content,
		    occurs,
		    context,
		    cModule
		)
	    );
	} else if (content instanceof RNValue) {
	    return (
		_makeCAttributeValue(
		    ngAttr,
		    (RNValue)content,
		    occurs,
		    context,
		    cModule
		)
	    );
	} else if (content instanceof RNList) {
	    return (
		_makeCAttributeList(
		    ngAttr,
		    (RNList)content,
		    occurs,
		    context,
		    cModule
		)
	    );
	} else if (content instanceof RNRef) {
	    return (
		_makeCAttributeRef(
		    ngAttr,
		    (RNRef)content,
		    occurs,
		    context,
		    cModule
		)
	    );
	} else if (content instanceof RNTextChoice) {
	    return (
		_makeCAttributeChoice(
		    ngAttr,
		    (RNTextChoice)content,
		    occurs,
		    context,
		    cModule
		)
	    );
	} else if (content instanceof RNTextOptional) {
	    return (
		_makeCAttributeOptional(
		    ngAttr,
		    (RNTextOptional)content,
		    occurs,
		    context,
		    cModule
		)
	    );
	} else if (content instanceof RNTextOneOrMore) {
	    return (
		_makeCAttributeOneOrMore(
		    ngAttr,
		    (RNTextOneOrMore)content,
		    occurs,
		    context,
		    cModule
		)
	    );
	} else if (content instanceof RNTextZeroOrMore) {
	    return (
		_makeCAttributeZeroOrMore(
		    ngAttr,
		    (RNTextZeroOrMore)content,
		    occurs,
		    context,
		    cModule
		)
	    );
	} else if (content instanceof RNNotAllowed) {
	    return (null);
	} else {
	    throw (new InternalError(content.toString()));
	}
    }

    private CAttribute _makeCAttributeRef(
	RNAttribute ngAttr,
	RNRef ngRef,
	String occurs,
	String context,
	CModule cModule
    ) {
	String attrName = _getAttributeName(ngAttr);
	String ns = _getAttributeNs(ngAttr);
	CAttribute cAttr = new CAttribute(
	    attrName,
	    ngRef.makeTextDocument(),
	    cModule
	);
	if (ns != null) {
	    cAttr.setNamespace(ns);
	}
/*
	CAttribute cAttr = new CAttribute(
	    _getAttributeName(ngAttr),
	    new CRefWithPattern(ngAttr.getName(), cModule),
	    cModule
	);
*/
	cAttr.setOccurs(occurs);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	return (cAttr);
    }

    private CAttribute _makeCAttributeText(
	RNAttribute ngAttr,
	String occurs,
	String context,
	CModule cModule
    ) {
	String attrName = _getAttributeName(ngAttr);
	String ns = _getAttributeNs(ngAttr);
	CAttribute cAttr = new CAttribute(
	    attrName,
	    "string",
	    cModule
	);
	if (ns != null) {
	    cAttr.setNamespace(ns);
	}
	cAttr.setOccurs(occurs);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	return (cAttr);
    }

    private CAttribute _makeCAttributeData(
	RNAttribute ngAttr,
	RNData data,
	String occurs,
	String context,
	CModule cModule
    ) {
	String attrName = _getAttributeName(ngAttr);
	String ns = _getAttributeNs(ngAttr);
	String type = data.getType();
	XMLFacet[] facets = _makeFacets(data);
	if (facets != null) {
	    CAttribute cAttr = new CAttribute(
		attrName,
		type,
		facets,
		cModule
	    );
	    if (ns != null) {
		cAttr.setNamespace(ns);
	    }
	    cAttr.setOccurs(occurs);
	    cAttr.setAnd(true);
	    cAttr.setBase(ngAttr.getXMLElement());
	    cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	    return (cAttr);
	} else {
	    CAttribute cAttr = new CAttribute(
		attrName,
		type,
		cModule
	    );
	    if (ns != null) {
		cAttr.setNamespace(ns);
	    }
	    cAttr.setOccurs(occurs);
	    cAttr.setAnd(true);
	    cAttr.setBase(ngAttr.getXMLElement());
	    cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	    return (cAttr);
	}
    }

    private XMLFacet[] _makeFacets(RNData data) {
	RNParam[] params = data.getParam();
	if (params == null || params.length == 0) {
	    return (null);
	}
	XMLFacet[] facets = new XMLFacet[params.length];
	for (int i = 0;i < params.length;i++) {
	    RNParam param = params[i];
	    facets[i] = new XMLFacet(param.getName(), param.getContent());
	}
	return (facets);
    }

    private CAttribute _makeCAttributeValue(
	RNAttribute ngAttr,
	RNValue ngValue,
	String occurs,
	String context,
	CModule cModule
    ) {
	String attrName = _getAttributeName(ngAttr);
	String ns = _getAttributeNs(ngAttr);
	CAttribute cAttr = new CAttribute(
	    attrName,
	    ngValue.makeTextDocument(),
	    cModule
	);
	if (ns != null) {
	    cAttr.setNamespace(ns);
	}
/*
	String value = ngValue.getContent();
	String type = ngValue.getType();
	CValue cValue;
	if (type != null) {
	    cValue = new CValue(value, type);
	} else {
	    cValue = new CValue(value);
	}
	CAttribute cAttr = new CAttribute(
	    _getAttributeName(ngAttr),
	    cValue,
	    cModule
	);
*/
	cAttr.setOccurs(occurs);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	return (cAttr);
    }

    private CAttribute _makeCAttributeList(
	RNAttribute ngAttr,
	RNList ngList,
	String occurs,
	String context,
	CModule cModule
    ) {
	String attrName = _getAttributeName(ngAttr);
	String ns = _getAttributeNs(ngAttr);
	CAttribute cAttr = new CAttribute(
	    attrName,
	    ngList.makeTextDocument(),
	    cModule
	);
	if (ns != null) {
	    cAttr.setNamespace(ns);
	}
	cAttr.setOccurs(occurs);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	return (cAttr);
    }

    private CAttribute _makeCAttributeChoice(
	RNAttribute ngAttr,
	RNTextChoice ngChoice,
	String occurs,
	String context,
	CModule cModule
    ) {
	String attrName = _getAttributeName(ngAttr);
	String ns = _getAttributeNs(ngAttr);
	CAttribute cAttr = new CAttribute(
	    attrName,
	    ngChoice.makeTextDocument(),
	    cModule
	);
	if (ns != null) {
	    cAttr.setNamespace(ns);
	}
	cAttr.setOccurs(occurs);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	return (cAttr);
    }

    private CAttribute _makeCAttributeOptional(
	RNAttribute ngAttr,
	RNTextOptional ngOptional,
	String occurs,
	String context,
	CModule cModule
    ) {
	String attrName = _getAttributeName(ngAttr);
	String ns = _getAttributeNs(ngAttr);
	CAttribute cAttr = new CAttribute(
	    attrName,
	    ngOptional.makeTextDocument(),
	    cModule
	);
	if (ns != null) {
	    cAttr.setNamespace(ns);
	}
	cAttr.setOccurs(occurs);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	return (cAttr);
    }

    private CAttribute _makeCAttributeZeroOrMore(
	RNAttribute ngAttr,
	RNTextZeroOrMore ngZeroOrMore,
	String occurs,
	String context,
	CModule cModule
    ) {
	String attrName = _getAttributeName(ngAttr);
	String ns = _getAttributeNs(ngAttr);
	CAttribute cAttr = new CAttribute(
	    attrName,
	    ngZeroOrMore.makeTextDocument(),
	    cModule
	);
	if (ns != null) {
	    cAttr.setNamespace(ns);
	}
	cAttr.setOccurs(occurs);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	return (cAttr);
    }

    private CAttribute _makeCAttributeOneOrMore(
	RNAttribute ngAttr,
	RNTextOneOrMore ngOneOrMore,
	String occurs,
	String context,
	CModule cModule
    ) {
	String attrName = _getAttributeName(ngAttr);
	String ns = _getAttributeNs(ngAttr);
	CAttribute cAttr = new CAttribute(
	    attrName,
	    ngOneOrMore.makeTextDocument(),
	    cModule
	);
	if (ns != null) {
	    cAttr.setNamespace(ns);
	}
	cAttr.setOccurs(occurs);
	cAttr.setAnd(true);
	cAttr.setBase(ngAttr.getXMLElement());
	cAttr.setLabel(_makeLabelNameAttr(context, attrName));
	return (cAttr);
    }

/*
    private String _getValueText(RNAttribute attr) {
	IRNAttributeHedgeChoice content = attr.getAttributeHedge();
	if (content instanceof RNValue) {
	    RNValue value = (RNValue)content;
	    return (value.getContent());
	} else {
	    return (null);
	}
    }

    private String _getTypeName(RNAttribute attr) {
	IRNAttributeHedgeChoice content = attr.getAttributeHedge();
	if (content instanceof RNText) {
	    return ("string");
	} else if (content instanceof RNData) {
	    RNData data = (RNData)content;
	    return (data.getType());
	} else if (content instanceof RNAttributeChoice) {
	    throw (new InternalError());
	} else {
	    throw (new InternalError());
	}
    }
*/

    private void _buildGroup(
	RNGroup group,
	String context,
	CSequence cSequence
    ) {
	RNGModule ngModule = _getModule(group);
	IRNElementHedgeChoice[] contents = group.getElementHedge();
	_buildGroup(ngModule, contents, context, cSequence);
    }

    private void _buildGroup(
	List list,
	CSequence cSequence
    ) {
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    ICParticle particle = (ICParticle)list.get(0);
	    cSequence.addParticle(particle);
	}
    }

    private void _buildGroup(
	RNGModule ngModule,
	IRNElementHedgeChoice[] contents,
	String context,
	CSequence cSequence
    ) {
	CModule cModule = ngModule.getCModule();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    context,
		    OCCURS_ONE
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cSequence.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CAttribute cAttr = _makeCAttribute(
		    ngAttr,
		    true,
		    context,
		    cModule
		);
		if (cAttr != null) {
		    cSequence.addParticle(cAttr);
		}
	    } else if (content instanceof RNText) {
		cSequence.addParticle(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		cSequence.addParticle(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		cSequence.addParticle(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cGroup = new CSequence();
		_buildGroup((RNGroup)content, context, cGroup);
		cSequence.addParticle(cGroup);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave();
		_buildInterleave((RNInterleave)content, context, cInterleave);
		cSequence.addParticle(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChildChoice = new CChoice();
		_buildChoice((RNChoice)content, context, cChildChoice);
		cSequence.addParticle(cChildChoice);
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cSequence.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    cSequence.addParticle(cParticle);
		}
	    } else if (content instanceof RNNotAllowed) {
		cSequence.addParticle(new CNone());
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildList(
	RNList list,
	String context,
	CList cList
    ) {
	cList.setExpr(list.makeTextDocument());
/*
	RNGModule ngModule = _getModule(list);
	CModule cModule = ngModule.getCModule();
	IRNTextHedgeChoice[] contents = list.getTextHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNTextHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		throw (new InternalError());
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cList.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		throw (new InternalError());

	    } else if (content instanceof RNText) {
		cList.addParticle(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		cList.addParticle(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		cList.addParticle(
		    new CValue(value.getContent(), typeName)
		);
	    } else if (content instanceof RNGroup) {
		CSequence cGroup = new CSequence();
		_buildGroup((RNGroup)content, context, cGroup);
		cList.addParticle(cGroup);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave();
		_buildInterleave((RNInterleave)content, context, cInterleave);
		cList.addParticle(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChildChoice = new CChoice();
		_buildChoice((RNChoice)content, context, cChildChoice);
		cList.addParticle(cChildChoice);
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cList.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cList.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cList.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cList.addParticle(cParticle);
		}
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    cList.addParticle(cParticle);
		}
	    } else if (content instanceof RNNotAllowed) {
		cList.addParticle(new CNone());
	    } else if (content instanceof RNTextOptional) {
		throw (new UnsupportedOperationException());
	    } else if (content instanceof RNTextZeroOrMore) {
		throw (new UnsupportedOperationException());
	    } else if (content instanceof RNTextOneOrMore) {
		throw (new UnsupportedOperationException());
	    } else {
		throw (new InternalError(content.toString()));
	    }
	}
*/
    }

    private void _buildChoice(
	RNChoice ngChoice,
	String context,
	CChoice cChoice
    ) {
	RNGModule ngModule = _getModule(ngChoice);
	CModule cModule = ngModule.getCModule();
	IRNElementHedgeChoice[] contents = ngChoice.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    context,
		    OCCURS_ONE
		);
		cChoice.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cChoice.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CAttribute cAttr = _makeCAttribute(
		    ngAttr,
		    true,
		    context,
		    cModule
		);
		if (cAttr != null) {
		    cChoice.addParticle(cAttr);
		}
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		cChoice.addParticle(cAttr);
*/
	    } else if (content instanceof RNText) {
		cChoice.addParticle(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		cChoice.addParticle(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		cChoice.addParticle(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cSequence = new CSequence();
		_buildGroup((RNGroup)content, context, cSequence);
		cChoice.addParticle(cSequence);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave();
		_buildInterleave((RNInterleave)content, context, cInterleave);
		cChoice.addParticle(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChildChoice = new CChoice();
		_buildChoice((RNChoice)content, context, cChildChoice);
		cChoice.addParticle(cChildChoice);
	    } else if (content instanceof RNList) {
		CList cList = new CList();
		_buildList((RNList)content, context, cList);
		cChoice.addParticle(cList);
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cChoice.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cChoice.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cChoice.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cChoice.addParticle(cParticle);
		}
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    cChoice.addParticle(cParticle);
		}
	    } else if (content instanceof RNNotAllowed) {
		cChoice.addParticle(new CNone());
	    } else if (content instanceof RNGrammar) {
		List externalContents = new ArrayList();
		_buildGrammar(
		    (RNGrammar)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    cChoice.addParticle(cParticle);
		}
	    } else {
		throw (new InternalError(content.toString()));
	    }
	}
    }

    private void _buildInterleave(
	RNInterleave ngInterleave,
	String context,
	CInterleave cInterleave
    ) {
	RNGModule ngModule = _getModule(ngInterleave);
	CModule cModule = ngModule.getCModule();
	IRNElementHedgeChoice[] contents = ngInterleave.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    context,
		    OCCURS_ONE
		);
		cInterleave.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cInterleave.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CAttribute cAttr = _makeCAttribute(
		    ngAttr,
		    true,
		    context,
		    cModule
		);
		if (cAttr != null) {
		    cInterleave.addParticle(cAttr);
		}
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		cInterleave.addParticle(cAttr);
*/
	    } else if (content instanceof RNText) {
		cInterleave.addParticle(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		cInterleave.addParticle(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		cInterleave.addParticle(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cSequence = new CSequence();
		_buildGroup((RNGroup)content, context, cSequence);
		cInterleave.addParticle(cSequence);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, context, cChoice);
		cInterleave.addParticle(cChoice);
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cInterleave.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cInterleave.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cInterleave.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cInterleave.addParticle(cParticle);
		}
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    cInterleave.addParticle(cParticle);
		}
	    } else if (content instanceof RNNotAllowed) {
		cInterleave.addParticle(new CNone());
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildMixed(
	RNMixed ngMixed,
	String context,
	CMixed cMixed
    ) {
	RNGModule ngModule = _getModule(ngMixed);
	CModule cModule = ngModule.getCModule();
	IRNElementHedgeChoice[] contents = ngMixed.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    context,
		    OCCURS_ONE
		);
		cMixed.addParticle(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		cMixed.addParticle(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CAttribute cAttr = _makeCAttribute(
		    ngAttr,
		    true,
		    context,
		    cModule
		);
		if (cAttr != null) {
		    cMixed.addParticle(cAttr);
		}
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		cMixed.addParticle(cAttr);
*/
	    } else if (content instanceof RNText) {
		cMixed.addParticle(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		cMixed.addParticle(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		cMixed.addParticle(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cSequence = new CSequence();
		_buildGroup((RNGroup)content, context, cSequence);
		cMixed.addParticle(cSequence);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, context, cChoice);
		cMixed.addParticle(cChoice);
	    } else if (content instanceof RNOptional) {
		List optionContents = new ArrayList();
		List optionAttrs = new ArrayList();
		_buildOptional(
		    (RNOptional)content,
		    context,
		    optionContents,
		    optionAttrs
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cMixed.addParticle(cParticle);
		}
		size = optionAttrs.size();
		for (int j = 0;j < size;j++) {
		    CAttribute cAttr = (CAttribute)optionAttrs.get(j);
		    cMixed.addParticle(cAttr);
		}
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cMixed.addParticle(cParticle);
		}
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    context,
		    optionContents
		);
		int size = optionContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)optionContents.get(j);
		    cMixed.addParticle(cParticle);
		}
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    cMixed.addParticle(cParticle);
		}
	    } else if (content instanceof RNNotAllowed) {
		cMixed.addParticle(new CNone());
	    } else {
		throw (new InternalError());
	    }
	}
    }

    private void _buildOptional(
	RNOptional optional,
	String context,
	List result,
	List attrs
    ) {
	IRNElementHedgeChoice[] contents = optional.getElementHedge();
	if (contents.length == 0) {
	    return;
	} else if (contents.length == 1) {
	    IRNElementHedgeChoice content = contents[0];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ZEROONE
		);
		result.add(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    _getModule(optional),
		    "?"
		);
		result.add(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CModule cModule = _getModule(ngAttr).getCModule();
		CAttribute cAttr = _makeCAttribute(
		    ngAttr,
		    false,
		    context,
		    cModule
		);
		if (cAttr != null) {
		    attrs.add(cAttr);
		}
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(false);
		attrs.add(cAttr);
*/
	    } else if (content instanceof RNText) {
		CModule cModule = _getModule(content).getCModule();
		result.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		CModule cModule = _getModule(content).getCModule();
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		result.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		result.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cSequence = new CSequence("?");
		_buildGroup((RNGroup)content, context, cSequence);
		result.add(cSequence);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave("?");
		_buildInterleave((RNInterleave)content, context, cInterleave);
		result.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("?");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		if (size == 0) {
		    // do nothing
		} else {
		    CSequence cSequence = new CSequence("?");
		    _buildGroup(externalContents, cSequence);
		    result.add(cSequence);
		}
	    } else if (content instanceof RNNotAllowed) {
		result.add(new CNone());
	    } else {
		throw (new InternalError());
	    }
	} else {
	    CSequence cSequence = new CSequence("?");
	    _buildGroup(_getModule(optional), contents, context, cSequence);
	    result.add(cSequence);
	}
    }

/*
    private void _buildOptional9(
	RNOptional optional,
	String context,
	List result,
	List attrs
    ) {
	IRNElementHedgeChoice[] contents = optional.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ZEROONE
		);
		result.add(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    _getModule(optional),
		    "?"
		);
		result.add(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CModule cModule = _getModule(ngAttr).getCModule();
		CAttribute cAttr = _makeCAttribute(ngAttr, false, cModule);
		if (cAttr != null) {
		    attrs.add(cAttr);
		}
*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(false);
		attrs.add(cAttr);
*
	    } else if (content instanceof RNText) {
		CModule cModule = _getModule(content).getCModule();
		result.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		CModule cModule = _getModule(content).getCModule();
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		result.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		result.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cSequence = new CSequence("?");
		_buildGroup((RNGroup)content, context, cSequence);
		result.add(cSequence);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave("?");
		_buildInterleave((RNInterleave)content, context, cInterleave);
		result.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("?");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    result.add(cParticle);
		}
	    } else if (content instanceof RNNotAllowed) {
		result.add(new CNone());
	    } else {
		throw (new InternalError());
	    }
	}
    }
*/

    private void _buildOneOrMore(
	RNOneOrMore oneOrMore,
	String context,
	List result
    ) {
	IRNElementHedgeChoice[] contents = oneOrMore.getElementHedge();
	if (contents.length == 0) {
	    return;
	} else if (contents.length == 1) {
	    IRNElementHedgeChoice content = contents[0];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ONEMORE
		);
		result.add(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    _getModule(oneOrMore),
		    "+"
		);
		result.add(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CModule cModule = _getModule(ngAttr).getCModule();
		CAttribute cAttr = _makeCAttribute(
		    ngAttr,
		    "+",
		    context,
		    cModule
		);
		if (cAttr != null) {
		    result.add(cAttr);
		}
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(false);
		result.add(cAttr);
*/
	    } else if (content instanceof RNText) {
		CModule cModule = _getModule(content).getCModule();
		result.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		CModule cModule = _getModule(content).getCModule();
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		result.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		result.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cGroup = new CSequence("+");
		_buildGroup((RNGroup)content, context, cGroup);
		result.add(cGroup);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave("+");
		_buildInterleave((RNInterleave)content, context, cInterleave);
		result.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("+");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		if (size == 0) {
		    // do nothing
		} else {
		    CSequence cSequence = new CSequence("+");
		    _buildGroup(externalContents, cSequence);
		    result.add(cSequence);
		}
	    } else if (content instanceof RNNotAllowed) {
		result.add(new CNone());
	    } else {
		throw (new InternalError());
	    }
	} else {
	    CSequence cSequence = new CSequence("+");
	    _buildGroup(_getModule(oneOrMore), contents, context, cSequence);
	    result.add(cSequence);
	}
    }

/*
    private void _buildOneOrMore9(
	RNOneOrMore oneOrMore,
	String context,
	List result
    ) {
	IRNElementHedgeChoice[] contents = oneOrMore.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ONEMORE
		);
		result.add(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    _getModule(oneOrMore),
		    "+"
		);
		result.add(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CModule cModule = _getModule(ngAttr).getCModule();
		CAttribute cAttr = _makeCAttribute(ngAttr, true, cModule);
		if (cAttr != null) {
		    result.add(cAttr);
		}
*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(true);
		result.add(cAttr);
*

	    } else if (content instanceof RNText) {
		CModule cModule = _getModule(content).getCModule();
		result.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		CModule cModule = _getModule(content).getCModule();
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		result.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		result.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		throw (new InternalError());
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave("+");
		_buildInterleave((RNInterleave)content, context, cInterleave);
		result.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("+");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		if (size == 0) {
		    // do nothing
		} else {
		    CSequence cSequence = new CSequence("+");
		    _buildGroup(externalContents, cSequence);
		    result.add(cSequence);
		}
	    } else if (content instanceof RNNotAllowed) {
		result.add(new CNone());
	    } else {
		throw (new InternalError());
	    }
	}
    }
*/

    private void _buildZeroOrMore(
	RNZeroOrMore zeroOrMore,
	String context,
	List result
    ) {
	IRNElementHedgeChoice[] contents = zeroOrMore.getElementHedge();
	if (contents.length == 0) {
	    return;
	} else if (contents.length == 1) {
	    IRNElementHedgeChoice content = contents[0];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ZEROMORE
		);
		result.add(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    _getModule(zeroOrMore),
		    "*"
		);
		result.add(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CModule cModule = _getModule(ngAttr).getCModule();
		CAttribute cAttr = _makeCAttribute(
		    ngAttr,
		    "*",
		    context,
		    cModule
		);
		if (cAttr != null) {
		    result.add(cAttr);
		}
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(false);
		result.add(cAttr);
*/

	    } else if (content instanceof RNText) {
		CModule cModule = _getModule(content).getCModule();
		result.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		CModule cModule = _getModule(content).getCModule();
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		result.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		result.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cGroup = new CSequence("*");
		_buildGroup((RNGroup)content, context, cGroup);
		result.add(cGroup);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave("*");
		_buildInterleave((RNInterleave)content, context, cInterleave);
		result.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("*");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		if (size == 0) {
		    // do nothing
		} else {
		    CSequence cSequence = new CSequence("*");
		    _buildGroup(externalContents, cSequence);
		    result.add(cSequence);
		}
	    } else if (content instanceof RNNotAllowed) {
		result.add(new CNone());
	    } else {
		throw (new InternalError());
	    }
	} else {
	    CSequence cSequence = new CSequence("*");
	    _buildGroup(_getModule(zeroOrMore), contents, context, cSequence);
	    result.add(cSequence);
	}
    }

/*
    private void _buildZeroOrMore9(
	RNZeroOrMore zeroOrMore,
	String context,
	List result
    ) {
	IRNElementHedgeChoice[] contents = zeroOrMore.getElementHedge();
	for (int i = 0;i < contents.length;i++) {
	    IRNElementHedgeChoice content = contents[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		CRefWithLabel cRef = _buildElementWithRef(
		    (RNElement)content,
		    context,
		    OCCURS_ZEROMORE
		);
		result.add(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    _getModule(zeroOrMore),
		    "*"
		);
		result.add(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		CModule cModule = _getModule(ngAttr).getCModule();
		CAttribute cAttr = _makeCAttribute(ngAttr, true, cModule);
		if (cAttr != null) {
		    result.add(cAttr);
		}
*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    _getModule(ngAttr).getCModule()
		);
		cAttr.setRequired(false);
		result.add(cAttr);
*

	    } else if (content instanceof RNText) {
		CModule cModule = _getModule(content).getCModule();
		result.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		CModule cModule = _getModule(content).getCModule();
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		result.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		result.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNGroup) {
		CSequence cSequence = new CSequence();
		_buildGroup(
		    (RNGroup)content,
		    context,
		    cSequence
		);
		result.add(cSequence);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave("*");
		_buildInterleave((RNInterleave)content, context, cInterleave);
		result.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice("*");
		_buildChoice((RNChoice)content, context, cChoice);
		result.add(cChoice);
	    } else if (content instanceof RNOptional) {
		throw (new InternalError());
	    } else if (content instanceof RNZeroOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNOneOrMore) {
		throw (new InternalError());
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    context,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    result.add(cParticle);
		}
	    } else if (content instanceof RNNotAllowed) {
		result.add(new CNone());
	    } else {
		throw (new InternalError());
	    }
	}
    }
*/

    private void _buildExternalRef(
	RNExternalRef ref,
	String context,
	List result
    ) {
	String uri = ref.getHref();
	if (uri == null) {
	    throw (new RNGSyntaxErrorException("Empty uri"));
	}
	try {
	    RNGGrammar eGrammar = _getGrammar(uri);
	    if (eGrammar == null) {
		throw (new RNGSyntaxErrorException("Unavailable uri = " + uri));
	    }

	} catch (IOException e) {
	    throw (URelaxNg.makeSyntaxErrorExceptionUnavailableUri(uri, e));
	} catch (SAXException e) {
	    throw (URelaxNg.makeSyntaxErrorExceptionUnavailableUri(uri, e));
	} catch (ParserConfigurationException e) {
	    throw (URelaxNg.makeSyntaxErrorExceptionUnavailableUri(uri, e));
	}
    }

    private void _buildGrammar(
	RNGrammar grammar,
	String context,
	List result
    ) {
	RNGGrammar eGrammar = new RNGGrammar(grammar, baseURI_);
	_buildGrammar(eGrammar, context, result);
    }

    private void _buildGrammar(
	RNGGrammar eGrammar,
	String context,
	List result
    ) {
	RNGModule[] eModules = eGrammar.getModules();
	for (int i = 0;i < eModules.length;i++) {
	    RNGModule eModule = eModules[i];
	    String eNs = eModule.getNamespaceURI();
	    RNGModule module = (RNGModule)modules_.get(eNs);
	    String[] labels;
	    if (module != null) {
		result.add(_mergeModule(module, eModule));
	    } else {
		result.add(_addModule(eNs, eModule));
	    }
	}
    }

    private RNGGrammar _getGrammar(String uri)
	throws IOException, SAXException, ParserConfigurationException {

//	uri = URelaxNg.makeUri(uri, baseURI_);
	IRelaxNgFactory factory = RelaxNgFactory.getFactory();
	factory.rSetBaseUri(baseURI_);
	Object node = factory.create(uri);
	if (node instanceof RNGrammar) {
	    return (new RNGGrammar((RNGrammar)node, uri));
	} else if (node instanceof RNElement) {
	    return (new RNGGrammar((RNElement)node, uri));
	}
	return (null);
    }

    private ICParticle _mergeModule(RNGModule body, RNGModule part) {
	CModule cBody = body.getCModule();
	CModule cPart = part.getCModule();
	String[] labels = cPart.getExportLabels();
	String[] roles = cPart.getExportRoles();
	// label
	Map labelMap = new HashMap();
	CElementRule[] eRules = cPart.getElementRules();
	for (int i = 0;i < eRules.length;i++) {
	    CElementRule eRule = eRules[i];
	    String label = eRule.getLabel();
	    if (labelMap.get(label) == null) {
		labelMap.put(label, _getNewLabel(cBody, label));
	    }
	}
	CContentRule[] cRules = cPart.getContentRules();
	for (int i = 0;i < cRules.length;i++) {
	    CContentRule cRule = cRules[i];
	    String label = cRule.getLabel();
	    if (labelMap.get(label) == null) {
		labelMap.put(label, _getNewLabel(cBody, label));
	    }
	}
	for (int i = 0;i < eRules.length;i++) {
	    CElementRule eRule = eRules[i];
	    cBody.addElementRule(_makeNewRule(eRule, labelMap));
	}
	for (int i = 0;i < cRules.length;i++) {
	    CContentRule cRule = cRules[i];
	    cBody.addContentRule(_makeNewRule(cRule, labelMap));
	}
	// role
	Map patternMap = new HashMap();
	CTagPattern[] tags = cPart.getTagPatterns();
	for (int i = 0;i < tags.length;i++) {
	    CTagPattern tag = tags[i];
	    String pattern = tag.getName();
	    if (patternMap.get(pattern) == null) {
		patternMap.put(pattern, _getNewPattern(cBody, pattern));
	    }
	}
	CAttListPattern[] attLists = cPart.getAttListPatterns();
	for (int i = 0;i < attLists.length;i++) {
	    CAttListPattern attList = attLists[i];
	    String pattern = attList.getName();
	    if (patternMap.get(pattern) == null) {
		patternMap.put(pattern, _getNewPattern(cBody, pattern));
	    }
	}
	for (int i = 0;i < tags.length;i++) {
	    CTagPattern tag = tags[i];
	    cBody.addTagPattern(_makeNewRole(tag, patternMap));
	}
	for (int i = 0;i < attLists.length;i++) {
	    CAttListPattern attList = attLists[i];
	    cBody.addAttListPattern(_makeNewRole(attList, patternMap));
	}
	if (labels.length == 0) {
	    throw (new InternalError());
	} else if (labels.length == 1) {
	    String newLabel = (String)labelMap.get(labels[0]);
	    CRefWithLabel ref = new CRefWithLabel(newLabel, "");
	    return (ref);
	} else {
	    CChoice choice = new CChoice();
	    for (int i = 0;i < labels.length;i++) {
		String label = labels[i];
		String newLabel = (String)labelMap.get(label);
		CRefWithLabel ref = new CRefWithLabel(newLabel, "");
		choice.addParticle(ref);
	    }
	    return (choice);
	}
    }

    private ICParticle _addModule(String eNs, RNGModule eModule) {
	modules_.put(eNs, eModule);
	String[] labels = eModule.getCModule().getExportLabels();
	if (labels.length == 0) {
	    throw (new InternalError());
	} else if (labels.length == 1) {
	    CRefWithLabel ref = new CRefWithLabel(labels[0], "", eNs);
	    return (ref);
	} else {
	    CChoice choice = new CChoice(); // occurs
	    for (int i = 0;i < labels.length;i++) {
		String label = labels[i];
		CRefWithLabel ref = new CRefWithLabel(label, "", eNs);
		choice.addParticle(ref);
	    }
	    return (choice);
	}
    }

    private CElementRule _makeNewRule(
	CElementRule eRule,
	Map labelMap
    ) {
	if (eRule instanceof CRelaxNgElementRule) {
	    return (_makeNewRule((CRelaxNgElementRule)eRule, labelMap));
	} else if (eRule instanceof CSequenceElementRule) {
	    return (_makeNewRule((CSequenceElementRule)eRule, labelMap));
	} else {
	    throw (new InternalError(eRule.toString()));
	}
    }

    private CElementRule _makeNewRule(
	CRelaxNgElementRule eRule,
	Map labelMap
    ) {
	return (eRule);		// XXX
    }

    private CElementRule _makeNewRule(
	CSequenceElementRule eRule,
	Map labelMap
    ) {
	return (eRule);		// XXX
    }

    private CContentRule _makeNewRule(
	CContentRule cRule,
	Map labelMap
    ) {
	return (cRule);		// XXX
    }

    private CTagPattern _makeNewRole(
	CTagPattern tag,
	Map roleMap
    ) {
	return (tag);		// XXX
    }

    private CAttListPattern _makeNewRole(
	CAttListPattern attList,
	Map roleMap
    ) {
	return (attList);	// XXX
    }

    private String _getNewLabel(CModule module, String label) {
	if (module.getElementRules(label) == null &&
	    module.getContentRules(label) == null) {

	    return (label);
	}
	int count = 1;
	for (;;) {
	    String newLabel = label + count;
	    if (module.getElementRules(newLabel) != null ||
		module.getContentRules(newLabel) != null) {

		return (newLabel);
	    }
	    count++;
	}
    }

    private String _getNewPattern(CModule module, String pattern) {
	if (module.getTagPatterns(pattern) == null &&
	    module.getAttListPatterns(pattern) == null) {

	    return (pattern);
	}
	int count = 1;
	for (;;) {
	    String newPattern = pattern + count;
	    if (module.getTagPatterns(newPattern) != null ||
		module.getAttListPatterns(newPattern) != null) {

		return (newPattern);
	    }
	    count++;
	}
    }

    private String _makeLabelNameElement(String context, String name) {
	if (name.startsWith("<")) {
	    return (context + "AnyElement");
	} else {
	    return (context + UString.capitalize(name));
	}
    }

    private String _makeLabelNameAttr(String context, String name) {
	if (name.startsWith("<")) {
	    return (context + "AnyAttr");
	} else {
	    return (context + UString.capitalize(name));
	}
    }

    private String _makeCOccurs(int occurs) {
	switch (occurs) {

	case OCCURS_ONE:
	    return ("");
	case OCCURS_ZEROONE:
	    return ("?");
	case OCCURS_ONEMORE:
	    return ("+");
	case OCCURS_ZEROMORE:
	    return ("*");
	default:
	    throw (new InternalError());
	}
    }

    private void _buildHedgeRule(
	RNDefine define,
	IRNElementHedgeChoice[] nodes
    ) {
	String labelName = define.getName();
//System.out.println(labelName);
	RNGModule ngModule = _getModule(define);
	CModule cModule = ngModule.getCModule();
	List list = new ArrayList();
	for (int i = 0;i < nodes.length;i++) {
	    IRNElementHedgeChoice content = nodes[i];
	    if (content instanceof RNElement) {
		RNElement element = (RNElement)content;
		String elementName = _getElementName(element);
		CRefWithLabel cRef = _buildElementWithRef(
		    element,
		    _makeLabelNameElement(labelName, elementName),
		    OCCURS_ONE
		);
		list.add(cRef);
	    } else if (content instanceof RNRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNRef)content,
		    ngModule,
		    ""
		);
		list.add(cRef);
	    } else if (content instanceof RNAttribute) {
		RNAttribute ngAttr = (RNAttribute)content;
		String attrName = _getAttributeName(ngAttr);
		CAttribute cAttr = _makeCAttribute(
		    ngAttr,
		    true,
		    _makeLabelNameAttr(labelName, attrName),
		    cModule
		);
		if (cAttr != null) {
		    list.add(cAttr);
		}
/*
		String type = _getTypeName(ngAttr);
		CAttribute cAttr = new CAttribute(
		    ngAttr.getName(),
		    type,
		    cModule
		);
		cAttr.setRequired(true);
		list.add(cAttr);
*/
	    } else if (content instanceof RNText) {
		list.add(new CData("string", cModule));
	    } else if (content instanceof RNData) {
		RNData data = (RNData)content;
		String typeName = data.getType();
		XMLFacet[] facets = _makeFacets(data);
		list.add(new CData(typeName, facets, cModule));
	    } else if (content instanceof RNValue) {
		RNValue value = (RNValue)content;
		String typeName = value.getType();
		list.add(new CValue(value.getContent(), typeName));
	    } else if (content instanceof RNEmpty) {
		// do nothing
	    } else if (content instanceof RNGroup) {
		CSequence cSequence = new CSequence();
		_buildGroup(
		    (RNGroup)content,
		    labelName,
		    cSequence
		);
		list.add(cSequence);
	    } else if (content instanceof RNInterleave) {
		CInterleave cInterleave = new CInterleave();
		_buildInterleave(
		    (RNInterleave)content,
		    labelName,
		    cInterleave
		);
		list.add(cInterleave);
	    } else if (content instanceof RNChoice) {
		CChoice cChoice = new CChoice();
		_buildChoice((RNChoice)content, labelName, cChoice);
		list.add(cChoice);
	    } else if (content instanceof RNMixed) {
		CMixed cMixed = new CMixed();
		_buildMixed((RNMixed)content, labelName, cMixed);
		list.add(cMixed);
	    } else if (content instanceof RNList) {
		CList cList = new CList();
		_buildList((RNList)content, labelName, cList);
		list.add(cList);
	    } else if (content instanceof RNOptional) {
		_buildOptional(
		    (RNOptional)content,
		    labelName,
		    list,
		    list
		);
	    } else if (content instanceof RNZeroOrMore) {
		List optionContents = new ArrayList();
		_buildZeroOrMore(
		    (RNZeroOrMore)content,
		    labelName,
		    list
		);
	    } else if (content instanceof RNOneOrMore) {
		List optionContents = new ArrayList();
		_buildOneOrMore(
		    (RNOneOrMore)content,
		    labelName,
		    list
		);
	    } else if (content instanceof RNExternalRef) {
		List externalContents = new ArrayList();
		_buildExternalRef(
		    (RNExternalRef)content,
		    labelName,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    list.add(cParticle);
		}
	    } else if (content instanceof RNNotAllowed) {
		list.add(new CNone());
	    } else if (content instanceof RNParentRef) {
		CRefWithLabel cRef = _makeCRefWithLabel(
		    (RNParentRef)content,
		    ngModule,
		    ""
		);
		list.add(cRef);
	    } else if (content instanceof RNGrammar) {
		List externalContents = new ArrayList();
		_buildGrammar(
		    (RNGrammar)content,
		    labelName,
		    externalContents
		);
		int size = externalContents.size();
		for (int j = 0;j < size;j++) {
		    ICParticle cParticle = (ICParticle)externalContents.get(j);
		    list.add(cParticle);
		}
	    } else {
		throw (new InternalError(content.toString()));
	    }
	}
	if (list.size() == 0) {
	    throw (new InternalError("Unidentified spec"));
	}
	CSequence cSequence = new CSequence();
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    cSequence.addParticle((ICParticle)list.get(i));
	}
	CSequenceContentRule cRule = new CSequenceContentRule(
	    labelName,
	    cSequence,
	    cModule
	);
	cModule.addContentRule(cRule);
	cRule.setBase(define.getXMLElement());
	nsByLabel_.put(labelName, cModule.getTargetNamespace());
/*
	if (_canAttPool(list)) {
	    CAttListPattern attPool = new CAttListPattern(
		labelName,
		cModule
	    );
	    int size = list.size();
	    for (int i = 0;i < size;i++) {
		Object node = list.get(i);
		if (!(node instanceof CAttribute)) {
		    throw (new InternalError());
		}
		attPool.addContent((ICPatternContent)node);
	    }
	    cModule.addAttListPattern(attPool);
	} else {
	    CSequence cSequence = new CSequence();
	    int size = list.size();
	    for (int i = 0;i < size;i++) {
		cSequence.addParticle((ICParticle)list.get(i));
	    }
	    CSequenceContentRule cRule = new CSequenceContentRule(
		labelName,
		cSequence,
		cModule
	    );
	    cModule.addContentRule(cRule);
	}
*/
    }
/*
    private boolean _canAttPool(List list) {
	int size = list.size();
	for (int i = 0;i < size;i++) {
	    Object node = list.get(i);
	    if (!(node instanceof CAttribute)) {
		return (false);
	    }
	}
	return (true);
    }
*/

    private CRefWithLabel _makeCRefWithLabel(
	RNRef ngRef,
	RNGModule ngModule,
	String occurs
    ) {
//System.out.println("_makeCRefWithLabel");
	String refLabelName = ngRef.getName();
//System.out.println(refLabelName);
	String uri = _getNamespace(ngRef);
	String contextUri = _getContextNamespace(ngRef);
	CRefWithLabel cRef;
	if (uri.equals(contextUri)) {
	    cRef = new CRefWithLabel(refLabelName, occurs);
	} else {
	    cRef = new CRefWithLabel(refLabelName, uri, occurs);
	    RNGModule parentModule = _getModule(uri);
//	    parentModule.addExportLabel(refLabelName); // XXX
	}
	cRef.setBase(ngRef.getXMLElement());
	unresolvedRefs_.add(new RefPair(ngRef, cRef));
	return (cRef);
/*
	CRefWithLabel cRef = new CRefWithLabel(refLabelName, occurs);
	unresolvedRefs_.add(new RefPair(ngRef, cRef));
	return (cRef);
*/
/*
	RNGModule targetModule = _getModule(ngRef);
	if (ngModule == targetModule) {
	    return (new CRefWithLabel(refLabelName, occurs));
	} else {
	    String namespace = (String)nsByLabel_.get(refLabelName);
	    return (new CRefWithLabel(refLabelName, namespace, occurs));
	}
*/
    }

    private CRefWithLabel _makeCRefWithLabel(
	RNParentRef ngRef,
	RNGModule ngModule,
	String occurs
    ) {
//System.out.println("_makeCRefWithLabel(parent)");
	String refLabelName = ngRef.getName();
//System.out.println(refLabelName);
	String uri = _getNamespace(ngRef);
	String contextUri = _getContextNamespace(ngRef);
	CRefWithLabel cRef;
	if (uri.equals(contextUri)) {
	    cRef = new CRefWithLabel(refLabelName, occurs);
	} else {
	    cRef = new CRefWithLabel(refLabelName, uri, occurs);
	    RNGModule parentModule = _getModule(uri);
//	    parentModule.addExportLabel(refLabelName); // XXX
	}
	cRef.setBase(ngRef.getXMLElement());
	parentRefs_.add(new ParentRefPair(ngRef, cRef));
	return (cRef);
    }

    private static class RefPair {
	RNRef ngRef;
	CRefWithLabel cRef;

	RefPair(RNRef ngRef, CRefWithLabel cRef) {
	    this.ngRef = ngRef;
	    this.cRef = cRef;
	}
    }

    private static class ParentRefPair {
	RNParentRef ngRef;
	CRefWithLabel cRef;

	ParentRefPair(RNParentRef ngRef, CRefWithLabel cRef) {
	    this.ngRef = ngRef;
	    this.cRef = cRef;
	}
    }
}
