/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2000  ASAMI, Tomoharu (tasami@ibm.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.vfs.jar;

import java.io.*;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.jar.*;
import jp.gr.java_conf.jaba2.text.UString;
import jp.gr.java_conf.jaba2.io.UIO;
import jp.gr.java_conf.jaba2.io.UFile;
import jp.gr.java_conf.jaba2.io.UJar;
import jp.gr.java_conf.jaba2.vfs.*;

/**
 * JarRealm
 *
 * @since   Jan. 25, 2000
 * @version Feb.  9, 2000
 * @author  ASAMI, Tomoharu (tasami@ibm.net)
 */
public class JarRealm extends AbstractVFSRealm {
    private File file_;
    private JarFile jarFile_;
    private JarEntry timestamp_;
    private String timestampName_ = ".VFSTimestamp";
    private URL manifest_;

    public JarRealm(File file) throws IOException {
	file_ = file;
	if (file.exists()) {
	    jarFile_ = new JarFile(file);
	    timestamp_ = jarFile_.getJarEntry(timestampName_);
	}
    }

    public JarRealm(File file, File manifest) throws IOException {
	this(file);
	manifest_ = manifest.toURL();
    }

    public void shallowSetup(IFilter filter) throws IOException {
	_shallowSetup(new JarRoot(file_), filter);
    }

    public void deepSetup(IFilter filter) throws IOException {
	_deepSetup(new JarRoot(file_), filter);
    }

    public void flush() throws IOException {
	JarRoot root = (JarRoot)getRoot();
	File temp = File.createTempFile("jaba2", "tmp");
	Manifest manifest;
	if (manifest_ != null) {
	    manifest = new Manifest(manifest_.openStream());
	} else {
	    manifest = new Manifest();
	}
	JarOutputStream out = new JarOutputStream(
	    new BufferedOutputStream(
		new FileOutputStream(temp)
	    ),
	    manifest
	);
	IVFSObject[] objects = root.getObjectChildren();
	for (int i = 0;i < objects.length;i++) {
	    _flushObject((JarObject)objects[i], out);
	}
	IVFSContainer[] containers = root.getContainerChildren();
	for (int i = 0;i < containers.length;i++) {
	    IVFSContainer container = containers[i];
	    if (!"META-INF".equals(container.getName())) {
		_flushContainer((JarContainer)containers[i], out);
	    }
	}
	out.flush();
	out.close();
	UFile.copyFile(temp, file_);
	temp.delete();
    }

    private void _flushContainer(
	JarContainer container,
	JarOutputStream out
    ) throws IOException {
	out.putNextEntry(new JarEntry(container.getPathname()));
	IVFSObject[] objects = container.getObjectChildren();
	for (int i = 0;i < objects.length;i++) {
	    _flushObject((JarObject)objects[i], out);
	}
	IVFSContainer[] containers = container.getContainerChildren();
	for (int i = 0;i < containers.length;i++) {
	    _flushContainer((JarContainer)containers[i], out);
	}
    }

    private void _flushObject(
	JarObject object,
	JarOutputStream out
    ) throws IOException {
	String pathname = object.getPathname();
	JarEntry entry = new JarEntry(pathname);
	InputStream in = object.openInputStream();
	try {
	    byte[] data = UIO.stream2Bytes(in);
	    entry.setSize(data.length);
	    out.putNextEntry(entry);
	    out.write(data, 0, data.length);
	} finally {
	    if (in != null) {
		in.close();
	    }
	}
    }

    public long getLastTimestamp() throws IOException {
	if (timestamp_ != null) {
	    return (timestamp_.getTime());
	} else {
	    return (super.getLastTimestamp());
	}
    }

    public void updateTimestamp() throws IOException {
	if (timestamp_ == null) {
	    timestamp_ = new JarEntry(timestampName_);
	}
	timestamp_.setTime(System.currentTimeMillis());
    }

    public void createObject(String pathname, InputStream in)
	throws IOException {

	JarObject target = _getJarObject(pathname);
	if (target == null) {
	    JarRoot root = (JarRoot)getRoot();
	    target = root.createJarObject(pathname);
	}
	target.setData(in);
    }

    public void createContainer(String pathname, InputStream in)
	throws IOException {

	// XXX : do nothing
    }

    public final JarFile getJarFile() {
	return (jarFile_);
    }

    private JarObject _getJarObject(String path) {
	String[] bodies = UString.getTokens(path, "/");
	if (bodies == null) {
	    throw (new InternalError());
	}
	if (bodies.length == 0) {
	    throw (new InternalError());
	}
	IVFSNode node = getRoot().getNode(bodies[0]);
	if (node == null) {
	    return (null);
	}
	if (bodies.length == 1) {
	    return ((JarObject)node);
	}
	if (!(node instanceof IVFSContainer)) {
	    throw (new InternalError());
	}
	return (_getJarObject((IVFSContainer)node, bodies, 1));
    }

    private JarObject _getJarObject(
	IVFSContainer container,
	String[] bodies,
	int index
    ) {
	IVFSNode node = container.getNode(bodies[index]);
	if (node == null) {
	    return (null);
	}
	if (bodies.length == index + 1) {
	    return ((JarObject)node);
	}
	if (!(node instanceof IVFSContainer)) {
	    throw (new InternalError());
	}
	return (_getJarObject((IVFSContainer)node, bodies, index + 1));
    }
}
