/*
 * RelaxerOrg class library
 *  Copyright (C) 2000-2002  ASAMI, Tomoharu (asami@relaxer.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package org.relaxer.runtime;

import java.util.*;
import java.io.IOException;
import java.net.URL;
import java.beans.Beans;
import java.rmi.RemoteException;
import java.lang.reflect.*;
import javax.xml.parsers.ParserConfigurationException;
import org.xml.sax.SAXException;
import org.relaxer.runtime.rFactory.*;
import org.relaxer.util.UException;
import com.AsamiOffice.util.UReflection;

/**
 * RComponentFactory
 *
 * @since   Mar. 13, 2002
 * @version Apr.  9, 2002
 * @author  ASAMI, Tomoharu (asami@relaxer.org)
 */
public class RComponentFactory {
    private FFactory config_;
    private Map components_ = new HashMap();
    private Map daos_ = new HashMap();
    private Map envEntries_ = new HashMap();

    public RComponentFactory(String uri)
	throws IOException, SAXException, ParserConfigurationException {

	_init(new FFactory(uri));
    }

    public RComponentFactory(URL url)
	throws IOException, SAXException, ParserConfigurationException {

	_init(new FFactory(url));
    }

    public RComponentFactory(FFactory config) {
	_init(config);
    }

    private void _init(FFactory config) {
	config_ = config;
	FComponent[] components = config.getComponent();
	for (int i = 0;i < components.length;i++) {
	    FComponent component = components[i];
	    components_.put(component.getId(), component);
	}
	FDao[] daos = config.getDao();
	for (int i = 0;i < daos.length;i++) {
	    FDao dao = daos[i];
	    daos_.put(dao.getId(), dao);
	}
/* XXX : component?
	FEnvEntry[] envEntries = config.getEnvEntry();
	for (int i = 0;i < envEntries.length;i++) {
	    FEnvEntry envEntry = envEntries[i];
	    envEntries_.put(envEntry.getName(), envEntry);
	}
*/
    }

    public Object getComponent(String id) throws RemoteException {
	try {
	    FComponent component = (FComponent)components_.get(id);
	    if (component == null) {
		return (null);
	    }
	    Object client;
	    IFComponentChoice info = component.getContent();
	    if (info instanceof FComponentBeans) {
		client = _getComponent((FComponentBeans)info);
	    } else if (info instanceof FComponentRmi) {
		client = _getComponent((FComponentRmi)info);
	    } else if (info instanceof FComponentIiop) {
		throw (new UnsupportedOperationException());
	    } else if (info instanceof FComponentEjb) {
		throw (new UnsupportedOperationException());
	    } else if (info instanceof FComponentAsoap) {
		throw (new UnsupportedOperationException());
	    } else if (info instanceof FComponentJaxm) {
		throw (new UnsupportedOperationException());
	    } else {
		throw (new InternalError());
	    }
	    return (client);
	} catch (InvocationTargetException e) {
	    throw (UException.makeRemoteException(e));
	}
    }

    private Object _getComponent(FComponentBeans info)
	throws InvocationTargetException, RemoteException {

	FComponentClient client = info.getComponentClient();
	if (client == null) {
	    return (null);
	}
	String className = client.getClassName();
	FComponentProvider provider = info.getComponentProvider();
	if (provider == null) {
	    return (_makeBean(className, client.getProperty()));
	} else {
	    String providerClassName = provider.getClassName();
	    FComponentProviderGateway gateway
		= provider.getComponentProviderGateway();
	    if (gateway != null) {
		return (
		    _makeBean(
			className, client.getProperty(),
			providerClassName, provider.getProperty(),
			gateway
		    )
		);
	    } else {
		return (
		    _makeBean(
			className, client.getProperty(),
			providerClassName, provider.getProperty()
		    )
		);
	    }
	}
    }

    private Object _getComponent(FComponentRmi info)
	throws InvocationTargetException, RemoteException {

	String regName = info.getName();
	FComponentClient client = info.getComponentClient();
	if (client == null) {
	    return (null);
	}
	String className = client.getClassName();
	return (_makeObject(className, regName));
    }

    public Object getDao(String id) throws RemoteException {
	try {
	    FDao dao = (FDao)daos_.get(id);
	    if (dao == null) {
		return (null);
	    }
	    IFDaoChoice info = dao.getContent();
	    if (info instanceof FDaoJdbc) {
		return (_getDao((FDaoJdbc)info));
	    } else if (info instanceof FDaoXml) {
		return (_getDao((FDaoXml)info));
	    } else {
		throw (new InternalError());
	    }
	} catch (InvocationTargetException e) {
	    throw (UException.makeRemoteException(e));
	}
    }

    private Object _getDao(FDaoJdbc info)
	throws InvocationTargetException {

	String className = info.getClassName();
	String uri = info.getUri();
	String table = info.getTable();
	String username = info.getUsername();
	String password = info.getPassword();
	String driver = info.getDriver();
	try {
	    Class.forName(driver);
	} catch (ClassNotFoundException e) {
	    System.err.println("Invalid jdbc driver = " + driver);
	}
	if (username != null) {
	    return (_makeObject(className, uri, table));
	} else {
	    return (_makeObject(className, uri, table, username, password));
	}
    }

    private Object _getDao(FDaoXml info)
	throws InvocationTargetException {

	String className = info.getClassName();
	return (_makeObject(className));
    }

    public Object getProvider(String id) throws RemoteException {
	try {
	    FComponent component = (FComponent)components_.get(id);
	    if (component == null) {
		return (null);
	    }
	    IFComponentChoice info = component.getContent();
	    if (info instanceof FComponentBeans) {
		return (_getProvider((FComponentBeans)info));
	    } else if (info instanceof FComponentRmi) {
		throw (new UnsupportedOperationException());
	    } else if (info instanceof FComponentIiop) {
		throw (new UnsupportedOperationException());
	    } else if (info instanceof FComponentEjb) {
		throw (new UnsupportedOperationException());
	    } else if (info instanceof FComponentAsoap) {
		throw (new UnsupportedOperationException());
	    } else if (info instanceof FComponentJaxm) {
		throw (new UnsupportedOperationException());
	    } else {
		throw (new InternalError());
	    }
	} catch (InvocationTargetException e) {
	    throw (UException.makeRemoteException(e));
	}
    }

    private Object _getProvider(FComponentBeans info)
	throws InvocationTargetException, RemoteException {

	FComponentProvider provider = info.getComponentProvider();
	String className = provider.getClassName();
	FComponentProviderGateway gateway
	    = provider.getComponentProviderGateway();
	if (gateway == null) {
	    return (_makeBean(className));
	} else {
	    return (_makeBean(className, _getGateway(gateway)));
	}
    }

    private Object _getGateway(FComponentProviderGateway gateway)
	throws RemoteException {

	IFComponentProviderGatewayChoice choice = gateway.getContent();
	Object bean;
	if (choice instanceof FComponentProviderGatewayName) {
	    FComponentProviderGatewayName name =
		(FComponentProviderGatewayName)choice;
	    bean = getComponent(name.getContent());
	} else if (choice instanceof FComponentProviderGatewayClassName) {
	    FComponentProviderGatewayClassName className =
		(FComponentProviderGatewayClassName)choice;
	    bean = _makeBean(className.getContent());
	} else {
	    throw (new InternalError());
	}
	_setProperties(bean, gateway.getProperty());
	return (bean);
    }

    private Object _makeBean(String beanName, FProperty[] properties) {
	Object bean = _makeBean(beanName);
	_setProperties(bean, properties);
	return (bean);
    }

    private Object _makeBean(String beanName) {
	try {
	    return (Beans.instantiate(null, beanName));
	} catch (ClassNotFoundException e) {
	    return (new InternalError(e.getMessage()));
	} catch (IOException e) {
	    return (new InternalError(e.getMessage()));
	}
    }

    private Object _makeBean(
	String beanName,
	FProperty[] beanProperties,
	String providerName,
	FProperty[] providerProperties
    ) throws InvocationTargetException {
	return (
	    _makeBean(
		beanName, beanProperties,
		_makeBean(providerName, providerProperties)
	    )
	);
    }

    private Object _makeBean(String beanName, String providerName)
	throws InvocationTargetException {

	return (_makeBean(beanName, _makeBean(providerName)));
    }

    private Object _makeBean(
	String beanName,
	FProperty[] beanProperties,
	String providerName,
	FProperty[] providerProperties,
	FComponentProviderGateway gateway
    ) throws InvocationTargetException, RemoteException {
	return (
	    _makeBean(
		beanName, beanProperties,
		_makeBean(
		    providerName,
		    providerProperties,
		    _getGateway(gateway)
		)
	    )
	);
    }

    private Object _makeBean(
	String beanName,
	String providerName,
	FComponentProviderGateway gateway
    ) throws InvocationTargetException, RemoteException {
	return (
	    _makeBean(
		beanName,
		_makeBean(
		    providerName,
		    _getGateway(gateway)
		)
	    )
	);
    }

    private Object _makeBean(
	String beanName,
	FProperty[] properties,
	Object provider
    ) throws InvocationTargetException {
	try {
	    Class clazz = Class.forName(beanName);
	    Constructor constructor = _findConstructor(clazz, provider);
	    Object component = constructor.newInstance(
		new Object[] { provider }
	    );
	    _setProperties(component, properties);
	    return (component);
	} catch (ClassNotFoundException e) {
	    return (new InternalError(e.getMessage()));
	} catch (NoSuchMethodException e) {
	    return (new InternalError(e.getMessage()));
	} catch (InstantiationException e) {
	    return (new InternalError(e.getMessage()));
	} catch (IllegalAccessException e) {
	    return (new InternalError(e.getMessage()));
	}
    }

    private Object _makeBean(
	String beanName,
	Object provider
    ) throws InvocationTargetException {
	try {
	    Class clazz = Class.forName(beanName);
	    Constructor constructor = _findConstructor(clazz, provider);
	    Object component = constructor.newInstance(
		new Object[] { provider }
	    );
	    return (component);
	} catch (ClassNotFoundException e) {
	    return (new InternalError(e.getMessage()));
	} catch (NoSuchMethodException e) {
	    return (new InternalError(e.getMessage()));
	} catch (InstantiationException e) {
	    return (new InternalError(e.getMessage()));
	} catch (IllegalAccessException e) {
	    return (new InternalError(e.getMessage()));
	}
    }

    private Constructor _findConstructor(Class clazz, Object arg)
	throws NoSuchMethodException {

	Class type = arg.getClass();
	Constructor[] constructors = clazz.getConstructors();
	for (int i = 0;i < constructors.length;i++) {
	    Constructor constructor = constructors[i];
	    Class[] paramTypes = constructor.getParameterTypes();
	    if (paramTypes.length == 1) {
		if (paramTypes[0].isAssignableFrom(type)) {
		    return (constructor);
		}
	    }
	}
	return (clazz.getConstructor(new Class[] { arg.getClass() }));
    }

    private Object _makeObject(String className)
	throws InvocationTargetException {

	try {
	    Class clazz = Class.forName(className);
	    Constructor constructor = clazz.getConstructor(new Class[0]);
	    Object component = constructor.newInstance(new Object[0]);
	    return (component);
	} catch (ClassNotFoundException e) {
	    return (new InternalError(e.getMessage()));
	} catch (NoSuchMethodException e) {
	    return (new InternalError(e.getMessage()));
	} catch (InstantiationException e) {
	    return (new InternalError(e.getMessage()));
	} catch (IllegalAccessException e) {
	    return (new InternalError(e.getMessage()));
	}
    }

    private Object _makeObject(String className, String param1)
	throws InvocationTargetException {

	try {
	    Class clazz = Class.forName(className);
	    Constructor constructor = clazz.getConstructor(
		new Class[] { String.class }
	    );
	    Object component = constructor.newInstance(
		new Object[] { param1 }
	    );
	    return (component);
	} catch (ClassNotFoundException e) {
	    return (new InternalError(e.getMessage()));
	} catch (NoSuchMethodException e) {
	    return (new InternalError(e.getMessage()));
	} catch (InstantiationException e) {
	    return (new InternalError(e.getMessage()));
	} catch (IllegalAccessException e) {
	    return (new InternalError(e.getMessage()));
	}
    }

    private Object _makeObject(
	String className,
	String param1,
	String param2
    ) throws InvocationTargetException {
	try {
	    Class clazz = Class.forName(className);
	    Constructor constructor = clazz.getConstructor(
		new Class[] { String.class, String.class }
	    );
	    Object component = constructor.newInstance(
		new Object[] { param1, param2 }
	    );
	    return (component);
	} catch (ClassNotFoundException e) {
	    return (new InternalError(e.getMessage()));
	} catch (NoSuchMethodException e) {
	    return (new InternalError(e.getMessage()));
	} catch (InstantiationException e) {
	    return (new InternalError(e.getMessage()));
	} catch (IllegalAccessException e) {
	    return (new InternalError(e.getMessage()));
	}
    }

    private Object _makeObject(
	String className,
	String param1,
	String param2,
	String param3
    ) throws InvocationTargetException {
	try {
	    Class clazz = Class.forName(className);
	    Constructor constructor = clazz.getConstructor(
		new Class[] { String.class, String.class, String.class }
	    );
	    Object component = constructor.newInstance(
		new Object[] { param1, param2, param3 }
	    );
	    return (component);
	} catch (ClassNotFoundException e) {
	    return (new InternalError(e.getMessage()));
	} catch (NoSuchMethodException e) {
	    return (new InternalError(e.getMessage()));
	} catch (InstantiationException e) {
	    return (new InternalError(e.getMessage()));
	} catch (IllegalAccessException e) {
	    return (new InternalError(e.getMessage()));
	}
    }

    private Object _makeObject(
	String className,
	String param1,
	String param2,
	String param3,
	String param4
    ) throws InvocationTargetException {
	try {
	    Class clazz = Class.forName(className);
	    Constructor constructor = clazz.getConstructor(
		new Class[] {
		    String.class,
		    String.class,
		    String.class,
		    String.class
		}
	    );
	    Object component = constructor.newInstance(
		new Object[] {
		    param1,
		    param2,
		    param3,
		    param4
		}
	    );
	    return (component);
	} catch (ClassNotFoundException e) {
	    return (new InternalError(e.getMessage()));
	} catch (NoSuchMethodException e) {
	    return (new InternalError(e.getMessage()));
	} catch (InstantiationException e) {
	    return (new InternalError(e.getMessage()));
	} catch (IllegalAccessException e) {
	    return (new InternalError(e.getMessage()));
	}
    }

    private void _setProperties(Object object, FProperty[] properties) {
	for (int i = 0;i < properties.length;i++) {
	    _setProperty(object, properties[i]);
	}
    }

    private void _setProperty(Object object, FProperty property) {
	try {
	    String name = property.getName();
	    String value = property.getValue();
	    UReflection.setProperty(object, name, value);
	} catch (NoSuchMethodException e) {
	    System.out.println(e.toString()); // XXX
	} catch (InstantiationException e) {
	    System.out.println(e.toString()); // XXX
	} catch (IllegalAccessException e) {
	    System.out.println(e.toString()); // XXX
	} catch (InvocationTargetException e) {
	    System.out.println(e.toString()); // XXX
	}
    }

    //
    private static RComponentFactory factory__;

    public static RComponentFactory getFactory() {
	if (factory__ == null) {
	    try {
		URL url = RComponentFactory.class.getResource(
		    "/META-INF/cdlConfig.xml"
		);
		factory__ = new RComponentFactory(url);
	    } catch (IOException e) {
		throw (new InternalError(e.getMessage()));
	    } catch (SAXException e) {
		throw (new InternalError(e.getMessage()));
	    } catch (ParserConfigurationException e) {
		throw (new InternalError(e.getMessage()));
	    }
	}
	return (factory__);
    }

    public static void setFactory(RComponentFactory factory) {
	factory__ = factory;
    }
}
