/*
 * The JabaJaba class library
 *  Copyright (C) 1997-2002  ASAMI, Tomoharu (asami@AsamiOffice.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package jp.gr.java_conf.jaba2.xml.relaxng;

import java.util.*;
import java.io.IOException;
import java.net.URL;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;
import jp.gr.java_conf.jaba2.util.MultiValueMap;
import jp.gr.java_conf.jaba2.xml.*;
import jp.gr.java_conf.jaba2.xml.relaxng.rRelaxng.*;

/**
 * RNGGrammarBuilder
 *
 * @since   Aug. 17, 2002
 * @version Aug. 18, 2002
 * @author  ASAMI, Tomoharu (asami@AsamiOffice.com)
 */
public class RNGGrammarBuilder {
    private Context root_;
    private Context current_;
    private Set names_ = new HashSet();
    private List starts_ = new ArrayList();

    public RNGGrammarBuilder(RNGrammar grammar) {
	current_ = root_ = new Context();
	_expandGrammar(grammar);
    }

    public List getDefines() {
	ArrayList list = new ArrayList();
	Iterator iter = names_.iterator();
	while (iter.hasNext()) {
	    String name = (String)iter.next();
	    list.add(_getDefine(root_, name));
	}
	return (list);
    }

    private RNDefine _getDefine(Context context, String name) {
	RNDefine[] defines = (RNDefine[])context.defines.get(name);
	if (defines == null || defines.length == 0) {
	    List childrenDefines = new ArrayList();
	    int size = context.children.size();
	    for (int i = 0;i < size;i++) {
		Context childContext = (Context)context.children.get(i);
		RNDefine childDefine = _getDefine(childContext, name);
		if (childDefine != null) {
		    childrenDefines.add(childDefine);
		}
	    }
	    switch (childrenDefines.size()) {
	    case 0:
		return (null);
	    case 1:
		return ((RNDefine)childrenDefines.get(0));
	    default:
		return (_getChoiceDefine(name, childrenDefines));
	    }
	} else if (defines.length == 1) {
	    RNDefine define = defines[0];
	    if ("choice".equals(define.getCombine())) {
		List childrenDefines = new ArrayList();
		childrenDefines.add(define);
		_getChildrenDefines(context, name, childrenDefines);
		return (_getChoiceDefine(name, childrenDefines));
	    } else if ("interleave".equals(define.getCombine())) {
		List childrenDefines = new ArrayList();
		childrenDefines.add(define);
		_getChildrenDefines(context, name, childrenDefines);
		return (_getInterleaveDefine(name, childrenDefines));
	    } else {
		return (define);
	    }
	} else {
	    if (_isChoice(defines)) {
		List childrenDefines = new ArrayList();
		childrenDefines.addAll(Arrays.asList(defines));
		_getChildrenDefines(context, name, childrenDefines);
		return (_getChoiceDefine(name, childrenDefines));
	    } else if (_isInterleave(defines)) {
		List childrenDefines = new ArrayList();
		childrenDefines.addAll(Arrays.asList(defines));
		_getChildrenDefines(context, name, childrenDefines);
		return (_getInterleaveDefine(name, childrenDefines));
	    } else {
		throw (new UnsupportedOperationException());
	    }
	}
    }

    private void _getChildrenDefines(Context context, String name, List list) {
	int size = context.children.size();
	for (int i = 0;i < size;i++) {
	    Context childContext = (Context)context.children.get(i);
	    RNDefine define = _getDefine(childContext, name);
	    if (define != null) {
		list.add(define);
	    }
	}
    }

    private RNDefine _getChoiceDefine(String name, List defines) {
	RNDefine newDefine = new RNDefine();
	newDefine.setName(name);
	RNChoice choice = new RNChoice();
	newDefine.addElementHedge(choice);
	int size = defines.size();
	for (int i = 0;i < size;i++) {
	    RNDefine oldDefine = (RNDefine)defines.get(i);
	    RNGroup group = new RNGroup();
	    group.setElementHedge(oldDefine.getElementHedge());
	    choice.addElementHedge(group);
	}
	return (newDefine);
    }

    private RNDefine _getInterleaveDefine(String name, List defines) {
	RNDefine newDefine = new RNDefine();
	newDefine.setName(name);
	RNInterleave interleave = new RNInterleave();
	newDefine.addElementHedge(interleave);
	int size = defines.size();
	for (int i = 0;i < size;i++) {
	    RNDefine oldDefine = (RNDefine)defines.get(i);
	    RNGroup group = new RNGroup();
	    group.setElementHedge(oldDefine.getElementHedge());
	    interleave.addElementHedge(group);
	}
	return (newDefine);
    }

/*
    public List getDefines() {
	List list = new ArrayList();
	Iterator iter = names_.iterator();
	while (iter.hasNext()) {
	    String name = (String)iter.next();
	    RNDefine[] defines = (RNDefine[])defines_.get(name);
	    if (defines.length == 0) {
		throw (new InternalError());
	    } else if (defines.length == 1) {
		list.add(defines[0]);
	    } else {
		if (_isChoice(defines)) {
		    RNDefine newDefine = new RNDefine();
		    newDefine.setName(name);
		    RNChoice choice = new RNChoice();
		    newDefine.addElementHedge(choice);
		    for (int i = 0;i < defines.length;i++) {
			RNDefine oldDefine = defines[i];
			RNGroup group = new RNGroup();
			group.setElementHedge(oldDefine.getElementHedge());
			choice.addElementHedge(group);
		    }
		    list.add(newDefine);
		} else if (_isInterleave(defines)) {
		    RNDefine newDefine = new RNDefine();
		    newDefine.setName(name);
		    RNInterleave interleave = new RNInterleave();
		    newDefine.addElementHedge(interleave);
		    for (int i = 0;i < defines.length;i++) {
			RNDefine oldDefine = defines[i];
			RNGroup group = new RNGroup();
			group.setElementHedge(oldDefine.getElementHedge());
			interleave.addElementHedge(group);
		    }
		    list.add(newDefine);
		} else {
		    throw (new UnsupportedOperationException());
		}
	    }
	}
	return (list);
    }
*/

    private boolean _isChoice(RNDefine[] defines) {
	for (int i = 0;i < defines.length;i++) {
	    RNDefine define = defines[i];
	    if ("choice".equals(define.getCombine())) {
		return (true);
	    }
	}
	return (false);
    }

    private boolean _isInterleave(RNDefine[] defines) {
	for (int i = 0;i < defines.length;i++) {
	    RNDefine define = defines[i];
	    if ("interleave".equals(define.getCombine())) {
		return (true);
	    }
	}
	return (false);
    }

    public List getStarts() {
	return (starts_);
    }

    private void _expandGrammar(
	RNGrammar grammar
    ) {
	IRNGrammarContentChoice[] contents = grammar.getGrammarContent();
	for (int i = 0;i < contents.length;i++) {
	    IRNGrammarContentChoice content = contents[i];
	    if (content instanceof RNDefine) {
		_addDefine((RNDefine)content);
	    } else if (content instanceof RNInclude) {
		_includeGrammar((RNInclude)content);
	    } else if (content instanceof RNStart) {
		_addStart((RNStart)content);
	    } else if (content instanceof RNDiv) {
		_buildDiv((RNDiv)content);
	    } else {
		throw (new IllegalArgumentException("syntax error"));
	    }
	}
    }

    private void _buildDiv(RNDiv div) {
	IRNGrammarContentChoice[] contents = div.getGrammarContent();
	for (int i = 0;i < contents.length;i++) {
	    IRNGrammarContentChoice content = contents[i];
	    if (content instanceof RNDefine) {
		_addDefine((RNDefine)content);
	    } else if (content instanceof RNInclude) {
		_includeGrammar((RNInclude)content);
	    } else if (content instanceof RNStart) {
		_addStart((RNStart)content);
	    } else if (content instanceof RNDiv) {
		_buildDiv((RNDiv)content);
	    } else {
		throw (new IllegalArgumentException("syntax error"));
	    }
	}
    }

    private void _includeGrammar(
	RNInclude include
    ) {
	Object node = null;
	try {
	    String uri = include.getHref();
	    if (uri == null) {
		return;
	    }
	    IRelaxNgFactory factory = RelaxNgFactory.getFactory();
	    node = factory.create(uri);
	} catch (IOException e) {
	    // warning
	} catch (SAXException e) {
	    // warning
	} catch (ParserConfigurationException e) {
	    // warning
	}
	_pushContext();
	if (node instanceof RNGrammar) {
	    _expandGrammar((RNGrammar)node);
	} else if (node instanceof RNDefine) {
	    _addDefine((RNDefine)node);
	} else if (node instanceof RNElement) {
	    throw (new UnsupportedOperationException());
	} else if (node instanceof RNDiv) {
	    _buildDiv((RNDiv)node);
	} else {
	    throw (new UnsupportedOperationException());
	}
	_popContext();
    }

    private void _addDefine(RNDefine define) {
	String name = define.getName();
	names_.add(name);
	current_.defines.add(define.getName(), define);
    }

    private void _addStart(RNStart start) {
	starts_.add(start);
    }

    private void _pushContext() {
	Context newContext = new Context();
	newContext.parent = current_;
	current_.children.add(newContext);
	current_ = newContext;
    }

    private void _popContext() {
	current_ = current_.parent;
    }

    static class Context {
	List children = new ArrayList();
	MultiValueMap defines = new MultiValueMap(
	    new RNDefine[0]
	);
	Context parent;
    }
}
