/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2000-2009 Sun Microsystems, Inc. All rights reserved. 
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License ("CDDL") (collectively, the "License").  You may
 * not use this file except in compliance with the License.  You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or mq/legal/LICENSE.txt.  See the License for the specific language
 * governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at mq/legal/LICENSE.txt.  Sun designates
 * this particular file as subject to the "Classpath" exception as provided by
 * Sun in the GPL Version 2 section of the License file that accompanied this
 * code.  If applicable, add the following below the License Header, with the
 * fields enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or  to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright holder. 
 */

/*
 * @(#)ConfigRecordDAOImpl.java	1.12 06/29/07
 */ 

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.resources.*;

import java.util.*;
import java.sql.*;
import java.io.IOException;

/**
 * This class implement a generic ConfigRecordDAO.
 */
class ConfigRecordDAOImpl extends BaseDAOImpl implements ConfigRecordDAO {

    protected String tableName;

    // SQLs
    protected String insertSQL;
    protected String selectRecordsSinceSQL;
    protected String selectAllSQL;

    /**
     * Constructor
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    ConfigRecordDAOImpl() throws BrokerException {

        // Initialize all SQLs
        DBManager dbMgr = DBManager.getDBManager();

        tableName = dbMgr.getTableName( TABLE_NAME_PREFIX );

        insertSQL = new StringBuffer(128)
            .append( "INSERT INTO " ).append( tableName )
            .append( " ( " )
            .append( RECORD_COLUMN ).append( ", " )
            .append( CREATED_TS_COLUMN )
            .append( ") VALUES ( ?, ? )" )
            .toString();

        selectRecordsSinceSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( RECORD_COLUMN ).append( ", " )
            .append( CREATED_TS_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( CREATED_TS_COLUMN ).append( " > ?" )
            .toString();

        selectAllSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( RECORD_COLUMN ).append( ", " )
            .append( CREATED_TS_COLUMN )
            .append( " FROM " ).append( tableName )
            .toString();
    }

    /**
     * Get the prefix name of the table.
     * @return table name
     */
    public final String getTableNamePrefix() {
        return TABLE_NAME_PREFIX;
    }

    /**
     * Get the name of the table.
     * @return table name
     */
    public final String getTableName() {
        return tableName;
    }

    /**
     * Insert a new entry.
     * @param conn database connection
     * @param recordData the record data
     * @param timeStamp
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public void insert( Connection conn, byte[] recordData, long timeStamp )
        throws BrokerException {

        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( insertSQL );
            Util.setBytes( pstmt, 1, recordData );
            pstmt.setLong( 2, timeStamp );

            pstmt.executeUpdate();
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + insertSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_PERSIST_CONFIGRECORD_FAILED, 
                    String.valueOf(timeStamp) ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Get all records created since the specified timestamp.
     * @param conn database connection
     * @param timestamp the timestamp
     * @return a List of records.
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public List getRecordsSince( Connection conn, long timestamp )
        throws BrokerException {

        ArrayList list = new ArrayList();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectRecordsSinceSQL );
            pstmt.setLong( 1, timestamp );
            rs = pstmt.executeQuery();
            while ( rs.next() ) {
                try {
                    byte[] buf = Util.readBytes( rs, 1 );
                    list.add( buf );
                } catch (IOException e) {
                    // fail to load one record; just log the record TS
                    IOException ex = DBManager.wrapIOException(
                        "[" + selectRecordsSinceSQL + "]", e );
                    logger.logStack( Logger.ERROR,
                        BrokerResources.X_PARSE_CONFIGRECORD_FAILED,
                        rs.getString( 2 ), ex);
                }
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectRecordsSinceSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_CONFIGRECORDS_FAILED ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return list;
    }

    /**
     * Return all records together with their corresponding timestamps.
     * @param conn database connection
     * @return an array of Object whose first element contains a List of
     *	 timestamps and the second element contains a List of config records of
     *   type byte[].
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public Object[] getAllRecords( Connection conn ) throws BrokerException {

        ArrayList timeList = new ArrayList();
        ArrayList recordList = new ArrayList();
        Object[] data = { timeList, recordList };

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectAllSQL );
            rs = pstmt.executeQuery();
            while ( rs.next() ) {
                long createdTS = -1;
                try {
                    createdTS = rs.getLong( 2 );
                    timeList.add( new Long( createdTS ) );

                    byte[] buf = Util.readBytes( rs, 1 );
                    recordList.add( buf );
                } catch (IOException e) {
                    // fail to load one record; just log the record TS
                    IOException ex = DBManager.wrapIOException(
                        "[" + selectAllSQL + "]", e );
                    logger.logStack( Logger.ERROR,
                        BrokerResources.X_PARSE_CONFIGRECORD_FAILED,
                        String.valueOf( createdTS ), ex);
                }
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectAllSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_CONFIGRECORDS_FAILED ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return data;
    }

    /**
     * Get debug information about the store.
     * @param conn database connection
     * @return a HashMap of name value pair of information
     */
    public HashMap getDebugInfo( Connection conn ) {

        HashMap map = new HashMap();
        int count = -1;

        try {
            // Get row count
            count = getRowCount( null, null );
        } catch ( Exception e ) {
            logger.log( Logger.ERROR, e.getMessage(), e.getCause() );
        }

        map.put( "Config Change Records(" + tableName + ")", String.valueOf( count ) );
        return map;
    }
}
