//-----------------------------------------------------------------------------
// Memory management
//-----------------------------------------------------------------------------

#include "mem.h"

#ifdef DEBUG_MEM
  #include "console.h"
  #include "logfile.h"
  #include <stdlib.h>
  #include <malloc.h>

  #define REGISTER_ALLOCATION_BLOC  128
  memZone *memRegister = NULL;
  unsigned int registerSize = 0, registerAllocatedSize = 0;

  void add_reference(void *m, size_t l, strDescription d)
  {
    if (registerSize == registerAllocatedSize)
    {
      registerAllocatedSize += REGISTER_ALLOCATION_BLOC;
      if (!registerAllocatedSize)
        memRegister = (memZone*) malloc(registerAllocatedSize*sizeof(memZone));
      else
        memRegister = (memZone*) realloc(memRegister, registerAllocatedSize*sizeof(memZone));

      if (!memRegister)
      {
        gConsole->Insertln("^1add_reference: Cannot allocate required size");
        return;
      }

      for (unsigned int i = registerSize; i < registerAllocatedSize; ++i)
      {
        memRegister[i].mem = NULL;
        memRegister[i].len = 0;
      }
    }

    ++registerSize;
    memRegister[registerSize-1].mem = m;
    memRegister[registerSize-1].len = l;
    memset(memRegister[registerSize-1].descr, '\0', sizeof(strDescription));
    if (strlen(d)) strncpy(memRegister[registerSize-1].descr, d, DESCRIPTION_MAXLENGTH);
  }

  void update_reference(void *m, void *new_m, size_t new_l, strDescription new_d)
  {
    if (!registerSize) return;

    // search for reference into memory
    for (unsigned int i = 0; i < registerSize; ++i)
    {
      if (memRegister[i].mem == m)
      {
        memRegister[i].mem = new_m;
        memRegister[i].len = new_l;
        if (strlen(new_d))
        {
          memset(memRegister[registerSize-1].descr, '\0', sizeof(strDescription));
          strncpy(memRegister[registerSize-1].descr, new_d, DESCRIPTION_MAXLENGTH);
        }
        return;
      }
    }
    // reference not found
    gConsole->Insertln("^1Unable to update register, reference not found.");
  }

  void remove_reference(void *m)
  {
    if (!registerSize) return;

    // search for reference into memory
    for (unsigned int i = 0; i < registerSize; ++i)
    {
      if (memRegister[i].mem == m)
      {
        // copy the last entry in the current pos so array is always full
        if (i < registerSize-1)
        {
          memRegister[i].mem = memRegister[registerSize-1].mem;
          memRegister[i].len = memRegister[registerSize-1].len;
          memset(memRegister[i].descr, '\0', sizeof(strDescription));
          strcpy(memRegister[i].descr, memRegister[registerSize-1].descr);
        }

        memRegister[registerSize-1].mem = NULL;
        memRegister[registerSize-1].len = 0;
        memset(memRegister[registerSize-1].descr, '\0', sizeof(strDescription));
        --registerSize;

        return;
      }
    }
    // reference not found
    gConsole->Insertln("^1Unable to remove reference, reference not found.");
  }

  void delete_memregister(void)
  {
    if (memRegister) free(memRegister);
    memRegister = NULL;
    registerSize = registerAllocatedSize = 0;
  }

  int get_references(void)
  {
    return registerSize;
  }

  void log_mem_report(void)
  {
    // search for reference into memory
    gLogFile->Insert("^3Report on actual memory allocation :\n");
    gLogFile->Insert("Number of entries : %d\n", registerSize);
    gLogFile->OpenFile();
    for (unsigned int i = 0; i < registerSize; ++i)
    {
      gLogFile->Insert("\tmem: %d\tlen: %d\tdescription: %s\n", 
        memRegister[i].mem, memRegister[i].len, memRegister[i].descr);
    }
    gLogFile->CloseFile();
  }

  void force_break(void)
  {
    __asm int 3;    // generates a break to allow debugging
  }

  void auto_freemem(void)
  {
    for (unsigned int i = 0; i < registerSize; ++i)
    {
      cake_free(memRegister[i].mem);
    }
  }

  void *cake_malloc(size_t size, strDescription description)
  {
    if (size < 0)
    {
      // TODO: Send Error message
      #ifdef _DEBUG
        __asm int 3;  // generates a break
      #else
        gConsole->Insertln("^1Invalid allocation size");
      #endif
      return NULL;
    }

    void *ptr = malloc(size);

    if (!ptr)
    {
      #ifdef _DEBUG
        __asm int 3;  // generates a break
      #else
        gConsole->Insertln("^1ERROR: Error allocating memory");
      #endif
      return NULL;
    }

    add_reference(ptr, size, description);
    return ptr;
  }

  void *cake_realloc(void *mem, size_t size, strDescription description)
  {
    void *ptr;

    if (size < 0)
    {
      // TODO: Send Error message
      #ifdef _DEBUG
        __asm int 3;  // generates a break
      #else
        gConsole->Insertln("^1Invalid allocation size");
      #endif
      return NULL;
    }

    if (mem && size)
    {
      ptr = realloc(mem, size);
      update_reference(mem, ptr, size, description);
    }
    else if (size)
    {
      ptr = malloc(size);
      add_reference(ptr, size, description);
    }
    else if (!size)
    {
      gLogFile->Insert("^1Reallocation has been aborted\n");
      return mem;     // do nothing
    }
    return ptr;
  }

  void cake_free(void *mem)
  {
    if (mem)
    {
      remove_reference(mem);
      free(mem);
    }
  }

#endif  /* DEBUG_MEM */
