//-----------------------------------------------------------------------------
// Console
//-----------------------------------------------------------------------------

#ifndef __CONSOLE_H__
#define __CONSOLE_H__

#include "shader.h"

#define C_NMESSAGELINES       8     /**< max number of message lines */
#define C_NCOMMHISTORYLINES   1024  /**< max number of lines of the commands history */

/**
 * Console states.
 */
typedef enum
{
  OPEN,             /**< opened console */
  OPENING,          /**< onening console */
  CLOSED,           /**< closed console */
  CLOSING           /**< closing console */
} enum_ConsoleStates;

/**
 * Vertical scrolling directions.
 */
typedef enum
{
  UP,               /**< scroll up */
  DOWN,             /**< scroll down */
  TOP,              /**< jump tp top */
  BOTTOM            /**< jump tp bottom */
} enum_ConsoleScrollDir;

/**
 * Horizontal cursor scrolling directions
 */
typedef enum
{
  LEFT = -1,                                /**< move left */
  RIGHT = 1,                                /**< move right */
  C_EXTREM_LEFT = -(CONSOLE_LINELENGTH+1),  /**< go to extrem left */
  C_EXTREM_RIGHT = CONSOLE_LINELENGTH+1     /**< go to extrem right */
} enum_ConsoleCursorScroll;

/**
* Define a console line.
*/
class ConsLine
{
  private:
    char buffer[CONSOLE_LINELENGTH];

  public:
    char content[CONSOLE_LINELENGTH];       /**< Line content. */
    int length;                             /**< Line length. */
    int nlines;                 /**< Number of lines that content takes when displayed. */
    int *breakPos;              /**< Postions of line breaks. */
    int breakPosSize;           /**< Break position table size. */

    /**
    * Initialize the line.
    */
    void Init(void);

    /**
    * Destroy the line.
    */
    void Shut(void);

    /**
    * Reinitialize the line.
    * Don't realloc breakPos table.
    */
    void ReInit(void);

    /**
    * Update the line.
    * The line update itself in function of given line length.
    * @param linelen The line length.
    * @return The number of lines occupied by the line
    */
    int Update(int linelen);

    /**
    * Set the line content.
    * @param c The line content.
    */
    void SetContent(const char *c);

    /**
    * Add content to line.
    * @param c The content to add at current line.
    * @return The new length of content, and 0 if new string couldn't be added.
    */
    int AddContent(const char *c);

    /**
    * Get the line content.
    * @param p The number of desired content line.
    * @return A pointer to a null ended line. If parameter is negative or bigger
    * than the number of lines, the full line content is returned.
    */
    char *GetContent(int p = -1);
};

/**
 * Console class.
 * The console appears on the top of the screen and can be used to enter
 * commands. It is also used to display messages or variables values.
 * @bug The engine stops sometimes without any error message when the user
 *      press a key in the console.
 * @todo Merge the <a href="http://www.calodox.org/morbac/console">console
 *       library project</a> in cake console
 */
class Console
{
  private:
    float cursorSpeed;              /**< blinking cursor speed (number of blink per second) */
    char* prompt;                   /**< prompt symbol */
    char cursorSymbol;              /**< cursor symbol */

    bool movingConsole;

    char MessageLines[C_NMESSAGELINES][CONSOLE_LINELENGTH];   /**< console lines */
    float MessageLife[C_NMESSAGELINES];                       /**< message lifetime */
    bool startNewMessageLine;         /**< start a new message line */

    ConsLine *ConsoleLines;           /**< console content */
    long NbrUsedLines;                /**< number of console lines */
    /**
     * Number of lines that covers the console lines.
     * During rendering, lines can be splitted in more than only one line in console.
     * This variable stores the number of lines that are necessary to display all the
     * console.
     */
    int NbrTrueLines;
    int MaxTextLineLength;
    bool startNewLine;                /**< defines if a new line must be started for insert function */

    long NbrAllocatedLines;           /**< number of allocated lines */

    int cursorPos;                    /**< cursor position (0 = at the end of line) */
    int scrollVal;                    /**< scrolling cursor value */

    enum_ConsoleStates state;         /**< console state */

    int height;                       /**< console height */
    int width;                        /**< console width */
    int leftPos;                      /**< console left position */
    int topPos;                       /**< console top position */
    int NbrMaxLines;                  /**< max number of displayable lines */
    
    /**
     * Recalculates the number of displayable lines in the console.
     * The function automatically calculates the number of lines that can
     * be displayed in the console, in function of console height.
     */
    void Recalculate_NLines(void);

    int VScrollY;                     /**< vertical scroll position */
    
    bool ConsoleIsMaximized;          /**< flying window state */
    int HBackup, WBackup, LBackup, TBackup;   /**< console window dimensions backup */
  
    char title[CONSOLE_LINELENGTH];

    int font;                   /**< Font shader */
    int back;                   /**< Back shader */
    int titlebar;               /**< TitleBar shader */
    int scrollUp;               /**< ScrollUp button shader */
    int scrollDown;             /**< ScrollDown button shader */
    int scroll;                 /**< Scroll cursor shader */
    int resize;                 /**< Resize logo shader */
    int maximise;               /**< Maximize shader */
    int reduce;                 /**< Reduce shader */
    int close;                  /**< Close shader */

    int ConsoleFontSizeX, ConsoleFontSizeY,
      MiniConsoleFontSizeX, MiniConsoleFontSizeY; /**< console font size */
    int fontRows, fontCols;
    float Coeff;

    void AddLine(void);                   /**< Add a new line to console. */
    void AddMessageLine(char* s);         /**< Add a line to the messages list. */
    void AddToLastMessageLine(char *s);   /**< Add content to the last message line. */
    void UpdateMessageLines(void);        /**< Updates each message lines */

    GLfloat openSpeed;                    /**< opening console speed */
    GLfloat closeSpeed;                   /**< closing console speed */
    bool enableOpeningClosingAnimations;  /**< activate opening and closing animation */

  public:
    ShaderManager shaders;

    bool isActive;                        /**< is console active */
    
    bool showMessages;              /**< display messages (mini console) when console is inactive */
    bool addToMessages;             /**< add console messages to messages list (mini console) */
    GLfloat messageMaxLife;         /**< messages lifetime */

    bool autoCut;                   /**< enable automatic lines autocut */

    bool showVScroll;               /**< display the vertical scroll */
    bool showTitleBar;              /**< display the title bar */
    int titleBarHeight;             /**< title bar height */

    GLfloat ActiveBorderColor[4];   /**< border color for active console */
    GLfloat InactiveBorderColor[4]; /**< border color for inactive console */

    Console(void);
    ~Console(void);

    void Init(void);
    void Shut(void);

    void Update(void);
    void Render(void);

    void Clear(void);
    void Insert(const char* s, ...);      /**< Add text to current line */
    void Insertln(const char* s, ...);    /**< Add a text line to console */
    
    // Command line management
    void ReInitCurrentCommand(void);
    char* GetCurrentCommand(void);
    void AddChar(char c);
    void DelChar(void);
    void SetCurrentCommand(char* s, ...);
    int GetNbrUsedLines(void);

    // Title, prompt
    void SetTitle(char *t);
    void SetPrompt(char *p);
    char* GetPrompt(void);

    // Opening and closing
    void Open(void);
    void Close(void);
    enum_ConsoleStates GetState(void);
    enum_ConsoleStates ToggleState(void);
    void SetState(enum_ConsoleStates s);

    // Scrolling
    void ScrollConsole(enum_ConsoleScrollDir dir);
    void SetVScrollYPos(int y, bool center = false);

    // Type management
    void ToggleType(void);
    void SetType(bool console_type);

    // Resizing and positioning
    void Resize(int w, int h);
    void Maximize(void);
    void Unmaximize(void);
    bool IsMaximized(void);
    void ToggleMaximisation(void);
    int GetWidth(void);
    int GetHeight(void);

    // Position management
    void SetTopPos(int top, int test = 1);
    void SetLeftPos(int left, int test = 1);
    int GetLeft(void);
    int GetTop(void);

    // Cursor management
    void MoveCursor(enum_ConsoleCursorScroll d);
    void SetCursorSpeed(float f);
    void SetCursorSymbol(char c);

    // Font sizing
    void SetFontSize(int xsize = -1, int ysize = -1, int n = 1);
    int GetFontSizeX(int n = 1);
    int GetFontSizeY(int n = 1);
    void SetCoeff(float c);

    // Console shaders
    void SetFont(int f, int rows = -1, int cols = -1);
    void SetBack(int b);
    void SetClose(int c);
    void SetMaximise(int m);
    void SetReduce(int r);
    void SetTitleBar(int t);
    void SetScrollUp(int s);
    void SetScrollDown(int s);
    void SetScroll(int s);
    void SetResize(int r);

    int GetFont(void);
};

#endif  /* __CONSOLE_H__ */
