//-----------------------------------------------------------------------------
// Client
//-----------------------------------------------------------------------------

#ifndef __CLIENT_H__
#define __CLIENT_H__

#include "types.h"
#include "camera.h"
#include "definitions.h"

/**
 * Indexes for different player sounds.
 */
typedef struct
{
  // specific sounds
  int death1;
  int death2;
  int death3;
  int drown;
  int fall1;
  int falling1;
  int gasp;
  int jump1;
  int pain25_1;
  int pain50_1;
  int pain75_1;
  int pain100_1;
  int taunt;

  // common sounds
  int fry;
  int gibimp1;
  int gibimp2;
  int gibimp3;
  int gibsplt1;
  int gurp1;
  int gurp2;
  int land1;
  int talk;
  int watr_in;
  int watr_out;
  int watr_un;
} PlayerSounds;

/**
 * User command.
 * This structure contains all possible commands for user.
 */
typedef struct
{
  float forwardmove;
  float sidemove;
  float upmove;
  float move_mouse_x;
  float move_mouse_y;
} usercmd_t;

/**
 * Client class.
 * The client class represents an entity that can move in the world. It is
 * generally used to represent players or bots. A camera is attached to the
 * client, and for each clients, the world is represented following this
 * own camera.
 * @todo Make friction and acceleration dependent on ground slope, so it is
 *       easier to accelerate on a descending slope, and harder to stop (less
 *       friction).
 */
class Client
{
  public:
    Client(void);
    ~Client(void);

    /**
     * Client initialization.
     */
    void Init(int left = 0, int top = 0, int width = -1, int height = -1);

    /**
     * Loads a player.
     * This function loads the player skin and sounds. For example, the
     * playername parameter only contains "sarge" or "anarki" to load
     * corresponding player settings.
     * @param playername The name of player to be loaded.
     */
    void LoadPlayer(const char *playername);

    void Shut(void);  /**< Client shuting out */

    /**
     * Update client.
     * Calculates client future position and updates camera orientation
     * using current rotation matrix and tranlsation values. The function
     * also simulates the acceleration and friction effects.
     * @param bsp The binary space tree the client is eveolving in
     * @param wld The client's world
     */
    void Update(Q3BSP *bsp, World *wld);

    /**
     * Update client camera.
     * Update the camera associated with current client.
     * @todo Use viewport instead of camera.
     */
    void UpdateCam(void);

    /**
     * Reset client movement.
     */
    void ResetMovement(void);

    /**
     * Set new client angle.
     * @param p the pitch value
     * @param y the yaw value
     * @param r the roll value
     */
    void SetAngle(float p, float y, float r);
    void SetAngle(vec3_t rot);

    /**
     * Set new client position.
     * @param x the x client position
     * @param y the y client position
     * @param z the z client position
     */
    void SetPos(float x, float y, float z);
    void SetPos(vec3_t pos);

    /**
     * Increment the client angle.
     * @param p the pitch increment
     * @param y the yaw increment
     * @param r the roll increment
     */
    void AddAngle(float p, float y, float r);

    /**
     * Increment the client position.
     * @param x the x client offset
     * @param y the y client offset
     * @param z the z client offset
     */
    void AddPos(float x, float y, float z);

    /**
     * Move the client in forward direction.
     */
    void MoveForward(float val = 1.f);

    /**
     * Move the client backward.
     * The function will interpret the command and make the client moving
     * using current speed, acceleration, friction, position and angle.
     */
    void MoveBackward(float val = 1.f);

    /**
     * Move the client on the left.
     * The function will interpret the command and make the client moving
     * using current speed, acceleration, friction, position and angle.
     */
    void MoveLeft(float val = 1.f);

    /**
     * Move the client on the right.
     * The function will interpret the command and make the client moving
     * using current speed, acceleration, friction, position and angle.
     */
    void MoveRight(float val = 1.f);

    /**
     * Move the client up.
     * The function will interpret the command and make the client moving
     * using current speed, acceleration, friction, position and angle.
     */
    void MoveUp(float val = 1.f);

    /**
     * Move the client down.
     * The function will interpret the command and make the client moving
     * using current speed, acceleration, friction, position and angle.
     */
    void MoveDown(float val = 1.f);

    /**
     * Rotate the client using mouse.
     * The function will interpret the rotation using mouse.
     * @param val the yaw rotation value
     */
    void MoveMouseX(float val);

    /**
     * Rotate the client using mouse.
     * The function will interpret the rotation using mouse.
     * @param val the pitch rotation value.
     */
    void MoveMouseY(float val);

    /**
     * Rotates the client using mouse.
     * The function is similar to moveMouseX and moveMouseY and permits to
     * specify the mouse movement for x and y using a single function call.
     * @param xval the yaw rotation value
     * @param yval the pitch rotation value
     */
    void MoveMouseXY(float xval, float yval);

    /**
     * Zoom or unzoom the camera.
     * The function change the camera fov for zooming. If argument is
     * positive fov is increased, if argument is negative, fov is
     * decreased. The function check if current camera zoom is in
     * boundaries. In this case, fov is upadted and function returns 1,
     * if camera zoom is out of boundaries, fov is not updated and
     * funcion returns 0. The result value can be used to avoid calling
     * the function in the main program.
     * @todo Make zoom depend of timer (because zoom is slower on low fps rate)
     * @param val the zoom value to add (in degrees)
     */
    void Zoom(float val);

    int TouchBounds(bboxf_t bbox);

    // Movements

    /**
     * Make the client jump.
     * This has no effect if client camera is in flying mode.
     */
    void Jump(void);
    void Land(void);
    void Fall(void);
    void Falling(void);
    void Die(int depthnum = 3);
    void Gasp(void);
    void Drown(void);
    void Pain(int amount);
    void Taunt(void);
    void Fry(void);
    void Gibimp(int type);
    void Gurp(int type);
    void Talk(void);
    void Water(int type); /**< type -> 0 = in, 1 = out, 2 = un */

    // Triggered actions
    bool JumpTo(vec3_t dest);
    bool TeleportTo(vec3_t dest, float angle, float startwalk = 0.f);

    /**
     * Check the collisions for a client.
     * @param targetpos the position that client want to get (but may not because of collision)
     */
    void CheckCollision(vec3_t targetpos);
    trace_t StepSlideMove(vec3_t vel);
    void StepSlideMove_(vec3_t pos, vec3_t vel, vec3_t mins, vec3_t maxs);
    void ClipVelocity(vec3_t in, vec3_t normal, vec3_t out, float overbounce);

    /**
     * Client camera.
     * The camera is the client eye.
     * @todo Support multiple cameras (for multiple (stereographic) eyes, rearview mirror, etc.)
     */
    Camera cam;         

    /**
     * Define if the client is an active client, if it can move by itself (active) or not (passive).
     * By default, all client are active, but they can be changed to passive mode
     * if required. In passive mode, no collision detection is done.
     */
    bool active_client;

    /**
     * Define if the client is the current client. In the case there are more than one client
     * only one can be the current selected client. This clients receives input commands and
     * have sounds enabled.
     * This value shouldn't be modified manually. The App class is responsible for this
     * variable.
     */
    bool isCurrent;

    bool pitchLimit;      /**< pitch limitation */

    float fov;          /**< camera fov */
    float zoomlimit;      /**< fov value when zooming */
    float jump_height;      /**< height that client can jump */
    float max_speed;      /**< maximum client translation speed */
    float acceleration;     /**< client acceleration */
    float friction;       /**< client friction */
    vec3_t forward, right, up;

    vec3_t velocity;      /**< velocity vector */
    bool flying;        /**< client is flying (can't jump) */
    bool swimming;
    bool falling;

    float viewheight;     /**< height of camera in the client bounding box */
    vec3_t mins, maxs;      /**< client min and max sizes */
    float step_size;      /**< step height that client can climb */
    vec3_t extSpeed;      /**< external speed used for jumping */

    // Interpolation stuff
    float interpolation;    /**< Height we are interpolating on */
    float startheight;      /**< Start height for interpolation */
    float progress;       /**< Current progress (between 0 and 1) */

    colour_t color;       /**< Used for debug */

    bool noclip;        /**< enable collision */

  private:

    int armor;          /**< amount of resting armor */
    int life;         /**< amount of resting life */

    usercmd_t cmd;        /**< user command */

    int nzoom, pzoom;     /**< variables used to avoid unnecessary
                   *   function calls */

    char name[MAX_PLAYERNAME_LENGTH];
    PlayerSounds sounds;
    int gender;         /**< client gender (1 = male, 2 = female, 3 = other) */

    double diveStart;     /**< time where client dived */
    double lastgurptime;    /**< time of last grub */

    Q3BSP *curr_bsp;
    World *curr_world;
};

#endif  /* __CLIENT_H__ */
