//-----------------------------------------------------------------------------
// Camera
//-----------------------------------------------------------------------------

#include "camera.h"
#include "render.h"
#include "timer.h"
#include "cake.h"
#include "math.h"

/*
  pitch -> rotation on horizontal axis (Ox)
  yaw -> rotation on vertical axis (Oy)
  roll -> rotation on depth axis (Oz)
*/

vec3_t v_forward = { 0, 0, -1 };    /**< forward vector (inversed because of OpenGL) */
vec3_t v_right = { 1, 0, 0 };     /**< right vector */
vec3_t v_up = { 0, 1, 0 };        /**< up vector */

Camera::Camera(void)
{
  VectorSet(pos, 0, 0, 10);
  VectorSet(rot, 90, 0, 0);

  fov = 90;

  VectorClear(forward);
  VectorClear(right);
  VectorClear(up);

  viewport_size = 1;

  viewsize = 100.f;

  inWater = false;
  waterwidthdeform = 20.f;
  waterheightdeform = 25.f;
  waterdeformtimefactor = 2.5f;

  lockfrustum = false;
}

Camera::~Camera(void)
{
}

void Camera::Init(float fov, float w, float h)
{
  this->fov = DEG2RAD(fov); // converts fov to radians
  cos_fov = (float)FastCos(fov/2.0f);

  // Projection ratio
  proj_ratio = w/h;
}

void Camera::UpdateRotation(void)
{
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  #if 0 // Disabled because it causes errors in demo interpolations
    if (rot[0] > 360.f) rot[0] -= 360.f;
    else if (rot[0] < 0.f) rot[0] += 360.f;
    if (rot[1] > 360.f) rot[1] -= 360.f;
    else if (rot[1] < 0.f) rot[1] += 360.f;
    if (rot[2] > 360.f) rot[2] -= 360.f;
    else if (rot[2] < 0.f) rot[2] += 360.f;
  #endif

  // in q3 +z is up, -z is down
  glRotatef(-rot[0], 1, 0, 0);
  glRotatef(-rot[1], 0, 0, 1);
  glRotatef(-rot[2], 0, 1, 0);

  // FIXME: do that only if angles have change.
  // Update camera rotation matrix
  float temp[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, temp);

  gRender->CheckGLError(6);

  // copy a 4x4 matrix into a 3x3
  rotation[0] = temp[0];
  rotation[1] = temp[1];
  rotation[2] = temp[2];
  rotation[3] = temp[4];
  rotation[4] = temp[5];
  rotation[5] = temp[6];
  rotation[6] = temp[8];
  rotation[7] = temp[9];
  rotation[8] = temp[10];

  MultVect3x3(rotation, v_forward, forward);
  MultVect3x3(rotation, v_right, right);
  MultVect3x3(rotation, v_up, up);
  eyedist = DotProduct(pos, forward);
}

void Camera::UpdatePosition(void)
{
  glMatrixMode(GL_MODELVIEW);
  glTranslatef(-pos[0], -pos[1], -pos[2]);
  gRender->CheckGLError(2);
}

void Camera::UpdateViewport(void)
{
  float nwidth = viewport_width*(viewsize/100);
  float nheight = viewport_height*(viewsize/100);
  float nleft = viewport_left+(viewport_width-nwidth)/2;
  float ntop = gRender->GetHeight()-viewport_top-viewport_height+(viewport_height-nheight)/2;

  // Water effect on viewport
  if (inWater)
  {
    float nnwidth = nwidth + waterwidthdeform*(FastSin(waterdeformtimefactor*(float)Timer::fTime)+1);
    float nnheight = nheight + waterheightdeform*(FastCos(waterdeformtimefactor*(float)Timer::fTime)+1);
    float nnleft = nleft - (nnwidth - nwidth)/2.f;
    float nntop = ntop - (nnheight - nheight)/2.f;
    glViewport((GLint) nnleft, (GLint) nntop, (GLsizei) nnwidth, (GLsizei) nnheight);
    glScissor((GLint) nleft, (GLint) ntop, (GLsizei) nwidth, (GLsizei) nheight);
  }
  else
  {
    glViewport((GLint) nleft, (GLint) ntop, (GLsizei) nwidth, (GLsizei) nheight);
    glScissor((GLint) nleft, (GLint) ntop, (GLsizei) nwidth, (GLsizei) nheight);
  }
  gRender->CheckGLError(2);
}

void Camera::UpdateProjection(void)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gRender->CheckGLError(2);
  gRender->Perspective(fov, proj_ratio);
}

void Camera::UpdateFrustum(void)
{
  if (!lockfrustum) CalculateFrustum();
}

void Camera::SetFov(float fov)
{
  this->fov = DEG2RAD(fov); // converts fov to radians
}

float Camera::GetFov(void)
{
  return RAD2DEG(this->fov);
}

void Camera::ImportParams(Camera *cam)
{
  if (cam == NULL) return;

  VectorCopy(cam->pos, this->pos);
  VectorCopy(cam->rot, this->rot);
  this->fov = cam->fov;
  this->cos_fov = cam->cos_fov;
  this->proj_ratio = cam->proj_ratio;
  memcpy(this->rotation, cam->rotation, 9*sizeof(float));
  VectorCopy(this->forward, cam->forward);
  VectorCopy(this->right, cam->right);
  VectorCopy(this->up, cam->up);
  this->eyedist = cam->eyedist;
  this->viewsize = cam->viewsize;
  this->waterdeformtimefactor = cam->waterdeformtimefactor;
  this->waterheightdeform = cam->waterheightdeform;
  this->waterwidthdeform = cam->waterwidthdeform;
}
