#!/usr/bin/python -O

from unittest import main, TestCase

from unittestplus.run import combine
from unittestplus.testcaseplus import (
    _compare_indexables, _compare_lengths, _compare_types, _is_int_indexable,
    _tostr, TestCasePlus,
)


class ClassUnderTest(TestCasePlus):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    def testAlwaysPass(self):
        pass

class ModuleFunctionsTest(TestCasePlus):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    def test_is_int_indexable(self):
        self.assertTrue(_is_int_indexable([]), "list")
        self.assertTrue(_is_int_indexable(()), "tuple")
        self.assertTrue(_is_int_indexable('a'), "str")

        self.assertFalse(_is_int_indexable({}), "dict")
        self.assertFalse(_is_int_indexable(set()), "set")
        self.assertFalse(_is_int_indexable(1), "int")
        self.assertFalse(_is_int_indexable(1.1), "float")


    def test_tostr(self):
        items = [
            1, 1.2, 'abc',
            [1, 2, 3], (1, 2, 3), set([1, 2, 3]),
            {'a':1, 'b':2, 'c':3},
        ]
        for i in items:
            self.assertEquals(_tostr(i), str(i),
                'tostr() for %s' % str(type(i)))


    def test_compare_lengths(self):
        _compare_lengths((1, 2, 3), [1, 2, 3], None)
        self.assertRaises(
            lambda: _compare_lengths((1, 2, 3), [1, 2, 3, 4], None),
            AssertionError)
        self.assertRaises(
            lambda: _compare_lengths((1, 2, 3), [1, 2], None),
            AssertionError)


    def test_compare_indexables(self):
        _compare_indexables((1, 2, 3), [1, 2, 3], None)
        self.assertRaises(
            lambda: _compare_indexables((1, 2, 3), [1, 2, 4], None),
            AssertionError)
        self.assertRaises(
            lambda: _compare_indexables((1, 2, 3), [1, 2, 3, 4], None),
            AssertionError)
        self.assertRaises(
            lambda: _compare_indexables((1, 2, 3, 4), [1, 2, 3], None),
            AssertionError)


    def test_compare_types(self):
        _compare_types(0, 1, None)
        self.assertRaises(lambda: _compare_types(0, 0.1, None), AssertionError)


class AssertNoneTest(TestCase):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    def setUp(self):
        self.mytestcase = ClassUnderTest("testAlwaysPass")

    def testAssertNonePass(self):
        self.mytestcase.assertNone(None)
        self.mytestcase.assertNone(None, 'msg')

    def testAssertNoneFail(self):
        assertion = lambda: self.mytestcase.assertNone(0)
        expected_msg = "not None: 0\n  "
        self.mytestcase.assertRaises(assertion, AssertionError, expected_msg)

        assertion = lambda: self.mytestcase.assertNone(0, 'msg')
        expected_msg = "not None: 0\n  msg"
        self.mytestcase.assertRaises(assertion, AssertionError, expected_msg)

    def testAssertNotNonePass(self):
        self.mytestcase.assertNotNone(0)
        self.mytestcase.assertNotNone(0, 'msg')

    def testAssertNonNoneFail(self):
        assertion = lambda: self.mytestcase.assertNotNone(None)
        expected_msg = "is None\n  "
        self.mytestcase.assertRaises(assertion, AssertionError, expected_msg)

        assertion = lambda: self.mytestcase.assertNotNone(None, 'msg')
        expected_msg = "is None\n  msg"
        self.mytestcase.assertRaises(assertion, AssertionError, expected_msg)


class AssertBooleanTest(TestCase):

    def setUp(self):
        self.mytestcase = ClassUnderTest("testAlwaysPass")

    def testAssertTruePasses(self):
        self.mytestcase.assertTrue(True)

    def testAssertTrueFails(self):
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertTrue(1, 'message'),
            AssertionError,
            '1 is not boolean. message')
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertTrue(False, 'message'),
            AssertionError,
            'is False. message')


    def testAssertFalsePasses(self):
        self.mytestcase.assertFalse(False)


    def testAssertFalseFails(self):
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertFalse(0, 'message'),
            AssertionError,
            '0 is not boolean. message')
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertFalse(True, 'message'),
            AssertionError,
            'is True. message')


class AssertEqualsTest(TestCase):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    def setUp(self):
        self.mytestcase = ClassUnderTest("testAlwaysPass")


    def test_shows_values_in_exception_message(self):
        assertion = lambda: self.mytestcase.assertEquals(2, 3, "desc")
        expected_msg = "2 != 3\n  desc"
        self.mytestcase.assertRaises(assertion, AssertionError, expected_msg)


    def test_types_differ(self):
        assertion = lambda: self.mytestcase.assertEquals((1, 2, 3), [1, 2, 3])
        expected_msg = (
            "not equal. types differ:\n"
            "  <type 'tuple'> (1, 2, 3)\n"
            "  <type 'list'> [1, 2, 3]\n")
        self.mytestcase.assertRaises(assertion, AssertionError, expected_msg)

        assertion = lambda: self.mytestcase.assertEquals(
            (1, 2, 3), [1, 2, 3], "message")
        expected_msg += "message"
        self.mytestcase.assertRaises(assertion, AssertionError, expected_msg)


    def test_iterables_of_different_lengths(self):
        assertion = lambda: self.mytestcase.assertEquals(
            [1, 2, 3], [1, 2, 3, 4])
        expected_msg = (
            "not equal, lengths differ: 3 != 4\n"
            "  [1, 2, 3]\n"
            "  [1, 2, 3, 4]\n")
        self.mytestcase.assertRaises(assertion, AssertionError, expected_msg)


    def test_sets(self):
        self.mytestcase.assertEquals(set(), set(), "empty sets should pass")

        one = set([2, 4, 6])
        two = set([6, 4, 2])
        self.mytestcase.assertEquals(one, two, "similar sets should pass")

        three = set([2, 4, 6, 8])
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertEquals(one, three, "desc"),
            AssertionError,
            "set([2, 4, 6]) != set([8, 2, 4, 6])\n  desc")

        four = set([2, 4])
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertEquals(one, four, "desc"),
            AssertionError,
            "set([2, 4, 6]) != set([2, 4])\n  desc")

        five = set([2, 4, 99])
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertEquals(one, five, "desc"),
            AssertionError,
            "set([2, 4, 6]) != set([2, 99, 4])\n  desc")


    def test_epsilon(self):
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertEquals(1.230, 1.239),
            AssertionError)

        self.mytestcase.assertEquals(1.23, 1.24, epsilon=0.011),
        self.mytestcase.assertEquals(1.24, 1.23, epsilon=0.011),

        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertEquals (1.23, 1.24001, epsilon=0.01),
            AssertionError)
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertEquals (1.24001, 1.23, epsilon=0.01),
            AssertionError)



class AssertRaisesTest(TestCase):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    def setUp(self):
        self.mytestcase = ClassUnderTest("testAlwaysPass")


    def test_for_func_that_raises_the_right_thing(self):

        def raise_correctly():
            raise ZeroDivisionError("actual message")

        self.mytestcase.assertRaises(
            raise_correctly, ZeroDivisionError, "actual message")


    def assertFailureModeOfAssertRaises(
        self, func, expected_msg, expected_msg_out):

        try:
            self.mytestcase.assertRaises(
                func, ZeroDivisionError, expected_msg)

        except AssertionError as exc:
            if str(exc) != expected_msg_out:
                message = (
                    "assertRaises should raise with correct message:\n" +
                    "--actual message:----\n" +
                    str(exc) + "\n" +
                    "--expected message:----\n" +
                    expected_msg_out + "\n")
                self.fail(message)

        except Exception as exc:
            self.fail(
                "assertRaises should raise an AssertionError, not %s" % exc)

        else:
            self.fail("assertRaises should raise")


    def test_for_func_that_raises_with_wrong_message(self):

        def raise_bad_message():
            raise ZeroDivisionError("bad message")

        self.mytestcase.assertRaises(raise_bad_message, ZeroDivisionError)

        expected_msg_out = (
            "raised exception with wrong message:\n" +
            "  bad message\n" +
            "  expected message\n")
        self.assertFailureModeOfAssertRaises(
            raise_bad_message, "expected message", expected_msg_out)


    def test_for_func_that_doesnt_raise(self):

        def raise_nothing():
            pass

        expected_msg_out = (
            'didn\'t raise.\n'
            '  Expected ZeroDivisionError("expected message")')
        self.assertFailureModeOfAssertRaises(
            raise_nothing, "expected message", expected_msg_out)


    def test_for_fn_that_raises_wrong_exception_type(self):

        def raise_wrong_type():
            raise TypeError("doesnt matter")

        expected_msg_out = (
            "raised wrong exception type:\n" +
            "  <type 'exceptions.TypeError'>(\"doesnt matter\")\n" + 
            "  <type 'exceptions.ZeroDivisionError'>")
        self.assertFailureModeOfAssertRaises(
            raise_wrong_type, "expected message", expected_msg_out)


    def test_returns_the_exception(self):

        exc = ZeroDivisionError()

        def raise_please():
            raise exc

        actual = self.mytestcase.assertRaises(raise_please, ZeroDivisionError)
        self.assertTrue(actual is exc, "should return the raised exception")


    def assert_warns_on_bad_args(self, func, exc_class, expected_prefix):

        assertion = lambda: self.mytestcase.assertRaises(
            func, exc_class, "WARNING")

        expected_msg = (
            "%s" % expected_prefix +
            "WARNING: TestCasePlus.assertRaises() has a new signature:\n"
            "  assertRaises(self, func, expectedType, expected_msg=None)")
        try:
            assertion()
        except StandardError as exc:
            self.assertEquals(type(exc), TypeError,
                "should raise a TypeError, not: %s" % type(exc))
            self.assertEquals(str(exc), expected_msg,
                "should raise with correct message, not: " + str(exc))
        else:
            self.fail("should raise a warning")


    def test_warns_on_bad_args(self):

        def raise_please():
            raise ZeroDivisionError

        arg1 = "1st arg is not callable\n"
        arg2 = "2nd arg is not exception class\n"
        self.assert_warns_on_bad_args(object(), ZeroDivisionError, arg1)
        self.assert_warns_on_bad_args(Exception, ZeroDivisionError, arg1)
        self.assert_warns_on_bad_args(raise_please, object(), arg2)
        self.assert_warns_on_bad_args(raise_please, lambda: None, arg2)
        self.assert_warns_on_bad_args(Exception, lambda: None, arg1+arg2)


class AssertVertsEqualTest(TestCase):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    def setUp(self):
        self.mytestcase = ClassUnderTest("testAlwaysPass")

    def test_equal(self):
        verts1 = [(1, 2), (2, 3), (4, 5)]
        verts2 = [(1, 2), (2, 3), (4, 5)]
        self.mytestcase.assertVertsEqual(verts1, verts2)
        self.mytestcase.assertVertsEqual(verts1, verts2, 'x')


    def test_notequal(self):
        verts1 = [(1, 2), (2, 3), (4, 5)]
        verts2 = [(1, 2), (2, 3), (4, 66)]
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertVertsEqual(verts1, verts2),
            AssertionError,
            'verts differ at vert 2: (4, 5), (4, 66)\n')


    def test_wronglen(self):
        verts1 = [(1, 2), (2, 3)]
        verts2 = [(1, 2), (2, 3), (4, 5)]
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertVertsEqual(verts1, verts2),
            AssertionError,
            'verts differ in len: 2, 3\n')


    def test_badly_formed(self):
        verts1 = [(1, 2), (3,)]
        verts2 = [(1, 2), (2, 3)]
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertVertsEqual(verts1, verts2),
            AssertionError,
            'actual verts badly formed at vert 1: (3,)\n')

        verts1 = [(1, 2), (3, 4), (5, 6)]
        verts2 = [(1, 2), (3, 4), (5, 6, 7)]
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertVertsEqual(verts1, verts2),
            AssertionError,
            'expected verts badly formed at vert 2: (5, 6, 7)\n')


    def test_degenerate(self):
        self.mytestcase.assertVertsEqual([], [])



class AssertValidColorTest(TestCase):
    # pylint: disable-msg=R0904
    # Too many public methods: we're a subclass of TestCase

    def setUp(self):
        self.mytestcase = ClassUnderTest("testAlwaysPass")


    def test_valid_colors(self):
        self.mytestcase.assertValidColor((0, 0, 0))
        self.mytestcase.assertValidColor((255, 255, 255))
        self.mytestcase.assertValidColor((100, 100, 100), "msg")


    def assertBadColor(self, bad_color):
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertValidColor(bad_color),
            AssertionError,
            "bad color: %s\n" % (bad_color,))
        self.mytestcase.assertRaises(
            lambda: self.mytestcase.assertValidColor(bad_color, "msg"),
            AssertionError,
            "bad color: %s\nmsg" % (bad_color,))


    def test_raises_on_bad_color(self):
        self.assertBadColor(())
        self.assertBadColor((100,))
        self.assertBadColor((100, 100))
        self.assertBadColor((100, 100, 100, 100))
        self.assertBadColor((-1, 100, 100))
        self.assertBadColor((256, 100, 100))
        self.assertBadColor((100, -1, 100))
        self.assertBadColor((100, 256, 100))
        self.assertBadColor((100, 100, -1))
        self.assertBadColor((100, 100, 256))
        self.assertBadColor((100.0, 100, 100))
        self.assertBadColor((100, 100.0, 100))
        self.assertBadColor((100, 100, 100.0))


TestCasePlusTest = combine(
    ModuleFunctionsTest,
    AssertNoneTest,
    AssertEqualsTest,
    AssertRaisesTest,
    AssertValidColorTest,
    AssertVertsEqualTest,
)

if __name__ == "__main__":
    main()

