/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the TrainingPage, DatabasePage and
** OtherPage class
** File name: settingspages.cpp
**
****************************************************************/

#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QSqlQuery>
#include <QFontDialog>
#include <QFileDialog>
#include <QMessageBox>
#include <QSettings>

#include "settingspages.h"
#include "sql/startsql.h"
#include "def/errordefines.h"
#include "errormessage.h"
#include "def/defines.h"

TrainingPage::TrainingPage(QWidget *parent) : QWidget(parent) {
	// Default ticker font if there is no saved ticker font
	tickerFont = QFont(FONT_STANDARD, FONT_SIZE_TICKER);

	// Create group boxes with settings
    createGroupKeyboardLayout();
    createGroupTickerFont();
	// Create group with sound output settings
    createGroupSoundOutput();

	// Set the layout of all widgets created above
	createLayout();

	// Widget connections
	connect(buttonSetFont, SIGNAL(clicked()), this, SLOT(setFont()));
	connect(comboLayout, SIGNAL(activated(int)), this, SLOT(toggleLayout(int)));

	// Read settings
	readSettings();
}

void TrainingPage::createGroupKeyboardLayout() {
	// Group "Keyboard layout"
	groupKeyboardLayout = new QGroupBox(QObject::tr("Tastaturlayout"));

	comboLayout = new QComboBox();
	comboLayout->insertItem(0, QObject::tr("QWERTZ Deutschland"));
	comboLayout->insertItem(1, QObject::tr("QWERTZ Schweiz"));
	comboLayout->insertItem(2, QObject::tr("NEO"));
	comboLayout->insertItem(3, QObject::tr("Dvorak"));
	comboLayout->insertItem(4, QObject::tr("RISTOME"));

	comboLayoutOs = new QComboBox();
	comboLayoutOs->insertItem(0, QObject::tr("MS Windows"));
	comboLayoutOs->insertItem(1, QObject::tr("Apple Macintosh"));

	comboLayoutCh = new QComboBox();
	comboLayoutCh->insertItem(0, QObject::tr("SG"));
	comboLayoutCh->insertItem(1, QObject::tr("SF"));
	/*
	// Radiobutton "DE Qwertz layout"
	radioLayoutDe = new QRadioButton(QObject::tr("QWERTZ (Deutschland)"));
	radioLayoutDe->setToolTip(QObject::tr("Whlen Sie diese Option, wenn Sie "
		"eine deutsche Windows-Tastatur verwenden\n(@-Zeichen auf der Q-Taste)"));

	//Radiobutton "CH Qwertz layout"
	radioLayoutCh = new QRadioButton(QObject::tr("QWERTZ (Schweiz)"));
	radioLayoutCh->setToolTip(QObject::tr("Whlen Sie diese Option, wenn Sie "
		"eine schweizerische Windows-Tastatur verwenden\n(@-Zeichen auf der 2-Taste)"));

	//Radiobutton "NEO layout"
	radioLayoutNeo = new QRadioButton(QObject::tr("NEO"));
	radioLayoutNeo->setToolTip(QObject::tr("Whlen Sie diese Option, wenn Sie "
		"eine Tastatur mit NEO-Layout verwenden"));

	//Radiobutton "Dvorak layout"
	radioLayoutDvorak = new QRadioButton(QObject::tr("Dvorak"));
	radioLayoutDvorak->setToolTip(QObject::tr("Whlen Sie diese Option, wenn Sie "
		"eine Tastatur mit Dvorak-Layout verwenden"));

	//Radiobutton "NEO layout"
	radioLayoutRistome = new QRadioButton(QObject::tr("RISTOME"));
	radioLayoutRistome->setToolTip(QObject::tr("Whlen Sie diese Option, wenn Sie "
		"eine Tastatur mit RISTOME-Layout verwenden"));

	//Radiobutton "Windows layout"
	radioLayoutMac = new QRadioButton(QObject::tr("Windows"));
	radioLayoutMac->setToolTip(QObject::tr("Whlen Sie diese Option, wenn Sie "
		"eine standardmige Windows-Tastatur verwenden\n(Windows-Tasten vorhanden)"));

	//Radiobutton "Macintosh layout"
	radioLayoutMac = new QRadioButton(QObject::tr("Apple Macintosh"));
	radioLayoutMac->setToolTip(QObject::tr("Whlen Sie diese Option, wenn Sie "
		"eine standardmige Apple-Tastatur verwenden\n(Apfel-Tasten vorhanden)"));


	// Layout of group box
	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget(radioLayoutDe);
	layout->addWidget(radioLayoutCh);
	layout->addWidget(radioLayoutNeo);
	layout->addWidget(radioLayoutDvorak);
	layout->addWidget(radioLayoutRistome);
	layout->setMargin(16);
	*/

	// Layout of group box
	QHBoxLayout *layout = new QHBoxLayout;
	layout->addWidget(comboLayout);
	layout->addWidget(comboLayoutOs);
	layout->addWidget(comboLayoutCh);
	layout->setMargin(16);
	groupKeyboardLayout->setLayout(layout);
}

void TrainingPage::createGroupTickerFont() {
	// Group "Ticker"
	groupTickerFont = new QGroupBox(QObject::tr("Laufschrift"));

	// Button "Set ticker font"
	labelTickerFont = new QLabel(QObject::tr("Schriftart:"));
	buttonSetFont = new QPushButton(QObject::tr("&Schriftart ndern"));
	buttonSetFont->setToolTip(QObject::tr("Hier knnen Sie die Schriftart der "
		"Laufschrift verndern\n(eine Schriftgre ber 20 Punkte wird aus "
		"Formatierungs-\ngrnden nicht empfohlen)"));

	// Spinbox "Set ticker speed"
	labelTickerSpeed = new QLabel(QObject::tr("Anfangsgeschwindigkeit (1-4):"));
	spinTickerSpeed = new QSpinBox();
	spinTickerSpeed->setMinimum(1);
	spinTickerSpeed->setMaximum(4);
	spinTickerSpeed->setToolTip(QObject::tr("Hier knnen Sie die "
		"Geschwindigkeit der Laufschrift verndern\n(der Wert 1 entspricht "
		"einem langsamen und der Wert 4\neinem schnellen Laufband)"));

	// Layout of group box vertical
	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget(labelTickerFont);
	layout->addWidget(buttonSetFont);
    layout->addSpacing(10);
	layout->addWidget(labelTickerSpeed);
	layout->addWidget(spinTickerSpeed);
	//layout->addStretch(1);
	layout->setMargin(16);
	groupTickerFont->setLayout(layout);
}

void TrainingPage::createGroupSoundOutput() {
	// Group "Ticker"
	groupSoundOutput = new QGroupBox(QObject::tr("Audioausgabe"));

	// Radiobutton Soundcard output
	radioSoundcard = new QRadioButton(QObject::tr("Soundkarte"));
	radioSoundcard->setToolTip(QObject::tr("Whlen Sie diese Option, wenn Sie "
		"die Soundausgabe ber die Soundkarte erfolgen soll"));

	// Radiobutton PC speaker output
	radioSpeaker = new QRadioButton(QObject::tr("PC-Lautsprecher"));
	radioSpeaker->setToolTip(QObject::tr("Whlen Sie diese Option, wenn Sie "
		"die Soundausgabe ber den PC-Lautsprecher erfolgen soll"));

	// Layout of group box vertical
	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget(radioSoundcard);
    layout->addSpacing(1);
	layout->addWidget(radioSpeaker);
	layout->setMargin(16);
	groupSoundOutput->setLayout(layout);
}

void TrainingPage::createLayout() {
    // Group layout vertical
	QVBoxLayout *boxesLayout = new QVBoxLayout;
    boxesLayout->addWidget(groupTickerFont);
    boxesLayout->addWidget(groupKeyboardLayout);
    boxesLayout->addWidget(groupSoundOutput);
	// Full layout of all widgets vertical
	QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addLayout(boxesLayout);
    //mainLayout->setMargin(5);
    mainLayout->setSpacing(15);
    // Pass layout to parent widget (this)
	this->setLayout(mainLayout);
}

void TrainingPage::setFont() {
	bool ok;
	QFont font = QFontDialog::getFont(&ok, tickerFont, this);
	if (ok) {
		tickerFont = font;
		setFontButtonLabel();
	}
}

void TrainingPage::setFontButtonLabel() {
	buttonSetFont->setText(tickerFont.family() + ", "
		+ QString::number(tickerFont.pointSize()));
}

void TrainingPage::toggleLayout(int index) {
	QSettings settings;
	settings.beginGroup("settings");
	switch (index) {
		case 0:
			comboLayoutOs->setCurrentIndex(settings.value("combo_layout_os",
				KEYBOARD_LAYOUT_MAC).toInt());
			comboLayoutOs->setVisible(true);
			comboLayoutCh->setVisible(false);
			break;
		case 1:
			comboLayoutOs->setCurrentIndex(settings.value("combo_layout_os",
				KEYBOARD_LAYOUT_MAC).toInt());
			comboLayoutOs->setVisible(true);
			comboLayoutCh->setCurrentIndex(settings.value("combo_layout_ch",
				0).toInt());
			comboLayoutCh->setVisible(true);
			break;
		case 2:
		case 3:
		case 4:
			comboLayoutOs->setVisible(false);
			comboLayoutCh->setVisible(false);
			break;
		default:
			comboLayoutOs->setCurrentIndex(settings.value("combo_layout_os",
				KEYBOARD_LAYOUT_MAC).toInt());
			comboLayoutOs->setVisible(true);
			comboLayoutCh->setVisible(false);
			break;
	}
	settings.endGroup();
}

void TrainingPage::readSettings() {
	QString dbNameUser = APP_USER_DB;
	QSettings settings;
	settings.beginGroup("settings");

	comboLayout->setCurrentIndex(settings.value("combo_layout",
		KEYBOARD_LAYOUT_STANDARD).toInt());
	switch (settings.value("combo_layout", KEYBOARD_LAYOUT_STANDARD).toInt()) {
		case 0:
			comboLayoutOs->setCurrentIndex(settings.value("combo_layout_os",
				KEYBOARD_LAYOUT_MAC).toInt());
			comboLayoutOs->setVisible(true);
			comboLayoutCh->setVisible(false);
			break;
		case 1:
			comboLayoutOs->setCurrentIndex(settings.value("combo_layout_os",
				KEYBOARD_LAYOUT_MAC).toInt());
			comboLayoutOs->setVisible(true);
			comboLayoutCh->setCurrentIndex(settings.value("combo_layout_ch",
				0).toInt());
			comboLayoutCh->setVisible(true);
			break;
		case 2:
		case 3:
		case 4:
			comboLayoutOs->setVisible(false);
			comboLayoutCh->setVisible(false);
			break;
		default:
			comboLayoutOs->setCurrentIndex(settings.value("combo_layout_os",
				KEYBOARD_LAYOUT_MAC).toInt());
			comboLayoutOs->setVisible(true);
			comboLayoutCh->setVisible(false);
			break;
	}
	tickerFont.fromString(settings.value("ticker_font",
		tickerFont.toString()).toString());
	spinTickerSpeed->setValue(settings.value("ticker_speed",
		TICKERSPEED_STANDARD).toInt());
	settings.endGroup();
	settings.beginGroup("sound");
	radioSoundcard->setChecked(settings.value("radio_soundcard", true).toBool());
	radioSpeaker->setChecked(settings.value("radio_speaker", false).toBool());
	settings.endGroup();
	setFontButtonLabel();
}

void TrainingPage::writeSettings() {
	QSettings settings;
	settings.beginGroup("settings");
	settings.setValue("combo_layout", comboLayout->currentIndex ());
	settings.setValue("combo_layout_os", comboLayoutOs->currentIndex ());
	settings.setValue("combo_layout_ch", comboLayoutCh->currentIndex ());
	settings.setValue("ticker_font", tickerFont.toString());
	settings.setValue("ticker_speed", spinTickerSpeed->value());
	settings.endGroup();
	settings.beginGroup("sound");
	settings.setValue("radio_soundcard", radioSoundcard->isChecked());
	settings.setValue("radio_speaker", radioSpeaker->isChecked());
	settings.endGroup();
}

//--------------------------------------------------------

DatabasePage::DatabasePage(QWidget *parent) : QWidget(parent) {

	// Create group boxes with settings
    createGroupUserReset();
    createGroupDatabase();

	// Set the layout of all widgets created above
	createLayout();

	// Widget connections
	connect(buttonLessonsReset, SIGNAL(clicked()), this, SLOT(deleteUserLessonList()));
	connect(buttonCharsReset, SIGNAL(clicked()), this, SLOT(deleteUserChars()));
	connect(buttonDatabasePath, SIGNAL(clicked()), this, SLOT(setDatabasePath()));

	// Read settings
	readSettings();
}

void DatabasePage::createGroupUserReset() {
	// Group "Keyboard layout"
	groupUserReset = new QGroupBox(QObject::tr("Benutzerdaten"));

	// Button "Reset user data"
	buttonLessonsReset = new QPushButton(QObject::tr("&Absolvierte Lektionen "
		"zurcksetzen"));
	buttonLessonsReset->setToolTip(QObject::tr("Hier knnen Sie smtliche "
		"gespeicherte Lektionendaten zurcksetzen\n(die Lektionen werden so "
		"in den ursprnglichen Zustand, wie nach\nder Installation, "
		"versetzt)"));

	// Button "Reset user data"
	buttonCharsReset = new QPushButton(QObject::tr("&Aufgezeichnete "
		"Schriftzeichen zurcksetzen"));
	buttonCharsReset->setToolTip(QObject::tr("Hier knnen Sie alle "
		"aufgezeichneten Tastendrcke und Tippfehler zurcksetzen\n(die "
		"Schriftzeichen werden so in den ursprnglichen Zustand, "
		"wie nach der\nInstallation, versetzt)"));

	// Layout of group box
	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget(buttonLessonsReset);
	layout->addWidget(buttonCharsReset);
	//layout->addStretch(1);
	layout->setMargin(16);
	groupUserReset->setLayout(layout);
}

void DatabasePage::createGroupDatabase() {
	// Group "Keyboard layout"
	groupDatabase = new QGroupBox(QObject::tr("Speicherort"));

	labelDatabasePath = new QLabel(QObject::tr("Verzeichnispfad zur Datenbank \"tipp10v2.db\":"));
	lineDatabasePath = new QLineEdit();
	buttonDatabasePath = new QPushButton(QObject::tr("..."));
	buttonDatabasePath->setMaximumWidth(30);
	labelDatabaseExplain = new QLabel(QObject::tr("Wenn im angegebenen Verzeichnis keine Datei \"tipp10v2.db\" existiert, "
		"wird automatisch eine neue, leere Datenbank erzeugt."));
	labelDatabaseExplain->setWordWrap(true);

	// Layout of group box
	QHBoxLayout *hlayout = new QHBoxLayout;
	hlayout->addWidget(lineDatabasePath);
	hlayout->addWidget(buttonDatabasePath);
	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget(labelDatabasePath);
	layout->addLayout(hlayout);
	layout->addWidget(labelDatabaseExplain);
	//layout->addStretch(1);
	layout->setMargin(16);
	groupDatabase->setLayout(layout);
}

void DatabasePage::createLayout() {
	// Full layout of all widgets vertical
	QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addWidget(groupUserReset);
    mainLayout->addWidget(groupDatabase);
    mainLayout->setSpacing(15);
    // Pass layout to parent widget (this)
	this->setLayout(mainLayout);
}

void DatabasePage::setDatabasePath() {
	/*QFileDialog *fd = new QFileDialog(this);
	fd->setFileMode(QFileDialog::AnyFile);
	fd->setFilter("Datenbankdatei (*.db)");
	fd->setViewMode(QFileDialog::Detail);
	//fd->setLabelText("Bitte whlen Sie eine Datenbankdatei aus...");
	fd->setDirectory(currentDatabasePath);
	if (fd->exec()) {
		//QString fileNames = fd->selectedFiles();
	}*/
	QString s = QFileDialog::getExistingDirectory(
		this,
		"Bitte whlen Sie ein Verzeichnis aus...",
		lineDatabasePath->text());
		//QFileDialog::confirmOverwrite());
	if (s.size() != 0) {
		lineDatabasePath->setText(s);
	}
}

void DatabasePage::deleteUserLessonList() {
	QSqlQuery query;
	switch (QMessageBox::question(this, APP_NAME,
		QObject::tr("Es werden alle absolvierten Lektionen des aktuellen "
		"Benutzers gelscht\nund die Lektionenliste in den urprnglichen "
		"Zustand versetzt!\n\nWollen Sie den Vorgang wirklich "
		"fortsetzen?\n\n"), QObject::tr("&Ja"), QObject::tr("&Abbrechen"),
		0, 1)) {
		case 0:
			StartSql *userSql = new StartSql();
			if (!userSql->deleteUserLessonList()) {
				// Error message
				ErrorMessage *errorMessage = new ErrorMessage(this);
				errorMessage->showMessage(ERR_USER_LESSONS_FLUSH,
					TYPE_WARNING, CANCEL_OPERATION);
				return;
			}
			QMessageBox::information(this, APP_NAME,
				QObject::tr("Die absolvierten Lektionen wurden erfolgreich "
				"zurckgesetzt!\n"));
			break;
	}
}

void DatabasePage::deleteUserChars() {
	QSqlQuery query;
	switch (QMessageBox::question(this, APP_NAME,
		QObject::tr("Es werden alle aufgezeichneten Schriftzeichen "
		"(Fehlerquoten) des\n"
		"aktuellen Benutzers gelscht und die Zeichenliste in den\n"
		"urprnglichen "
		"Zustand versetzt!\n\nWollen Sie den Vorgang wirklich "
		"fortsetzen?\n\n"), QObject::tr("&Ja"), QObject::tr("&Abbrechen"),
		0, 1)) {
		case 0:
			StartSql *userSql = new StartSql();
			if (!userSql->deleteUserChars()) {
				// Error message
				ErrorMessage *errorMessage = new ErrorMessage(this);
				errorMessage->showMessage(ERR_USER_ERRORS_FLUSH,
					TYPE_WARNING, CANCEL_OPERATION);
				return;
			}
			QMessageBox::information(this, APP_NAME,
				QObject::tr("Die aufgezeichneten Schriftzeichen wurden erfolgreich "
				"zurckgesetzt!\n"));
			break;
	}
}

void DatabasePage::readSettings() {
	QString dbNameUser = APP_USER_DB;
	QSettings settings;
	settings.beginGroup("database");
	currentDatabasePath = settings.value("path", "Fehler/" + dbNameUser).toString();
	if (currentDatabasePath.size() > dbNameUser.size() + 1) {
		lineDatabasePath->setText(currentDatabasePath.left(currentDatabasePath.size() - dbNameUser.size() - 1));
	}
	settings.endGroup();
}

void DatabasePage::writeSettings() {
	QSettings settings;
	settings.beginGroup("database");
	settings.setValue("path", lineDatabasePath->text() + "/" + QString::QString(APP_USER_DB));
	settings.endGroup();
}

//--------------------------------------------------------

OtherPage::OtherPage(QWidget *parent) : QWidget(parent) {

	// Create group with info window settings
    createGroupDialogCheck();

	// Create group with adaptation settings
    createGroupAdaptation();

	// Set the layout of all widgets created above
	createLayout();

	// Read settings
	readSettings();
}

void OtherPage::createGroupDialogCheck() {
	// Group "Ticker"
	groupDialogCheck = new QGroupBox(QObject::tr("Hinweisfenster"));

	// Check box "Show start window"
	checkIllustration = new QCheckBox(QObject::tr("Willkommenfenster beim "
		"Programmstart anzeigen"));
	checkIllustration->setToolTip(QObject::tr("Hier knnen Sie festlegen, "
		"ob ein Informationsfenster\nmit Ratschlgen beim Start von TIPP10"
		"angezeigt werden soll"));

	// Check box "Show intelligence warning"
	checkIntelligence = new QCheckBox(QObject::tr("Auf aktivierte Intelligenz "
		"beim Start einer freien oder\neigenen Lektion hinweisen"));
	checkIntelligence->setToolTip(QObject::tr("Hier knnen Sie festlegen, "
		"ob ein Hinweisfenster\nangezeigt werden soll, wenn eine freie"
		"oder eigene\nLektion mit aktivierter Intelligenz gestartet wird."));

	// Check box "Show info about publishing your own lesson"
	checkLessonPublish = new QCheckBox(QObject::tr("Vor dem Anlegen einer "
		"eigenen Lektion auf die\nVerffentlichungsmglichkeit als freie "
		"Lektion hinweisen"));
	checkLessonPublish->setToolTip(QObject::tr("xxx")); //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

	// Layout of group box vertical
	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget(checkIllustration);
    layout->addSpacing(1);
	layout->addWidget(checkIntelligence);
    layout->addSpacing(1);
	layout->addWidget(checkLessonPublish);
	layout->setMargin(16);
	groupDialogCheck->setLayout(layout);
}

void OtherPage::createGroupAdaptation() {
	// Group "Ticker"
	groupAdaptation = new QGroupBox(QObject::tr("Sonstiges"));

	// Check box "Intelligence duration toggling"
	checkLimitLesson = new QCheckBox(QObject::tr("Dauer der Lektion "
		"automatisch auf \"Gesamte Lektion\"\numstellen, wenn die "
		"Intelligenz deaktiviert wird"));
	//checkLimitLesson->setToolTip(QObject::tr("xxx")); //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

	// Check box "Database update from tipp10 version 1"
	checkDbUpdate = new QCheckBox(QObject::tr("Im Programmverzeichnis "
		"nach der alten Datenbank\n\"tipp10.db\" (TIPP10 Version 1) "
		"suchen und die Daten\nbernehmen"));
	//checkDbUpdate->setToolTip(QObject::tr("xxx")); //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

	// Layout of group box vertical
	QVBoxLayout *layouta = new QVBoxLayout;
	layouta->addWidget(checkLimitLesson);
    layouta->addSpacing(1);
	layouta->addWidget(checkDbUpdate);
	layouta->setMargin(16);
	groupAdaptation->setLayout(layouta);
}

void OtherPage::createLayout() {
	// Full layout of all widgets vertical
	QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addWidget(groupDialogCheck);
    mainLayout->addWidget(groupAdaptation);
    mainLayout->setSpacing(15);
    // Pass layout to parent widget (this)
	this->setLayout(mainLayout);
}

void OtherPage::readSettings() {
	QString dbNameUser = APP_USER_DB;
	QSettings settings;
	settings.beginGroup("general");
	checkIllustration->setChecked(settings.value("check_illustration", true).toBool());
	checkIntelligence->setChecked(settings.value("check_toggle_intelligence", true).toBool());
	checkLimitLesson->setChecked(settings.value("check_limit_lesson", true).toBool());
	checkDbUpdate->setChecked(settings.value("check_db_update", false).toBool());
	checkLessonPublish->setChecked(settings.value("check_lesson_publish", true).toBool());
	settings.endGroup();
}

void OtherPage::writeSettings() {
	QSettings settings;
	settings.beginGroup("general");
	settings.setValue("check_illustration", checkIllustration->isChecked());
	settings.setValue("check_toggle_intelligence", checkIntelligence->isChecked());
	settings.setValue("check_limit_lesson", checkLimitLesson->isChecked());
	settings.setValue("check_db_update", checkDbUpdate->isChecked());
	settings.setValue("check_lesson_publish", checkLessonPublish->isChecked());
	settings.endGroup();
}
