/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the MainWindow class
** File name: mainwindow.cpp
**
****************************************************************/

#include <QIcon>
#include <QMessageBox>
#include <QMenuBar>
#include <QSize>
#include <QPoint>
#include <QSqlQuery>
#include <QSettings>
#include <QProcess>
#include <QDir>

#include "mainwindow.h"
#include "settingsdialog.h"
#include "updatedialog.h"
#include "def/defines.h"
#include "def/errordefines.h"
#include "errormessage.h"

MainWindow::MainWindow() {
	trainingStarted = false;

	createActions();
	createMenu();

	createWidgets();

	createStart();
	startWidget->fillLessonList(false);
	selectedLesson = -1;
	selectedType = -1;

	setWindowTitle(APP_NAME);
	setWindowIcon(QIcon("img/icon.ico"));
    setMinimumSize(APP_WIDTH_STANDARD, APP_HEIGHT_STANDARD);

    readSettings();

	if (showIllustration) {
		IllustrationDialog illustrationDialog("illustration.png", 560, 400,
			"Einfhrung", "", "check_illustration", 0, this);
		illustrationDialog.exec();
	}
}

MainWindow::~MainWindow() {
	writeSettings();
}

void MainWindow::closeEvent(QCloseEvent *event) {
	if (trainingStarted) {
		// Ask, if training is already started
		switch (QMessageBox::question(this, APP_NAME,
			QObject::tr("Es gehen alle Werte der laufenden Lektion verloren!"
			"\n\nWollen Sie die Anwendung wirklich beenden?\n\n"),
			QObject::tr("&Ja"), QObject::tr("&Abbrechen"), 0, 1)) {
			case 0:
				// Ok button pushed
				break;
			case 1:
				// Cancel button pushed
				event->ignore();
				break;
		}
	}
}

void MainWindow::createMenu() {
	//Mac-Version:
	//-----------
	/*
	evaluationMenu = menuBar()->addMenu(QObject::tr("&Gehe zu"));
	evaluationMenu->addAction(exitAction);
	evaluationMenu->addAction(settingsAction);
	evaluationMenu->addAction(updateAction);
	evaluationMenu->addAction(evalAction);
	helpMenu = menuBar()->addMenu(QObject::tr("&Hilfe"));
	helpMenu->addAction(helpAction);
	helpMenu->addSeparator();
	helpMenu->addAction(aboutAction);
	*/
	// Win-Version:
	// -----------
	// Menu bar items
	fileMenu = menuBar()->addMenu(QString("&%1").arg(APP_NAME));
	fileMenu->addAction(settingsAction);
	//fileMenu->addAction(usersAction);
	fileMenu->addAction(updateAction);
	fileMenu->addSeparator();
	fileMenu->addAction(exitAction);
	evaluationMenu = menuBar()->addMenu(QObject::tr("&Auswertung"));
	evaluationMenu->addAction(evalAction);
	helpMenu = menuBar()->addMenu(QObject::tr("&Hilfe"));
	helpMenu->addAction(helpAction);
	helpMenu->addSeparator();
	helpMenu->addAction(aboutAction);
}

void MainWindow::createActions() {
	//Mac-Version:
	//-----------
	/*
	settingsAction = new QAction(QIcon(":/img/menu_settings.png"),
		QObject::tr("&Preferences"), this);
	updateAction = new QAction(QIcon(":/img/menu_update.png"),
		QObject::tr("Aktualisierung"), this);
	exitAction = new QAction(QObject::tr("&Quit"), this);
	evalAction = new QAction(QIcon(":/img/menu_evaluation.png"),
		QObject::tr("&Ergebnisse"), this);
	helpAction = new QAction(QIcon(":/img/menu_help.png"),
		QObject::tr("&Hilfe"), this);
	aboutAction = new QAction(QObject::tr("&About "), this);
	*/
	// Win-Version:
	// -----------
	// Menu bar actions
	settingsAction = new QAction(QIcon(":/img/menu_settings.png"),
		QObject::tr("&Grundeinstellungen"), this);
	//usersAction = new QAction(QIcon(":/img/menu_users.png"),
	//	QObject::tr("Benutzerverwaltung"), this);
	updateAction = new QAction(QIcon(":/img/menu_update.png"),
		QObject::tr("Aktualisierung"), this);
	exitAction = new QAction(QObject::tr("&Beenden"), this);
	evalAction = new QAction(QIcon(":/img/menu_evaluation.png"),
		QObject::tr("&Ergebnisse"), this);
	helpAction = new QAction(QIcon(":/img/menu_help.png"),
		QObject::tr("&Hilfe"), this);
	aboutAction = new QAction(QObject::tr("&ber ") + APP_NAME, this);

	// Connect bar actions
	connect(aboutAction, SIGNAL(triggered()), this, SLOT(about()));
	connect(exitAction, SIGNAL(triggered()), this, SLOT(close()));
	connect(settingsAction, SIGNAL(triggered()), this, SLOT(showSettings()));
	connect(updateAction, SIGNAL(triggered()), this, SLOT(showUpdate()));
	connect(evalAction, SIGNAL(triggered()), this,
		SLOT(toggleStartToEvaluation()));
	connect(helpAction, SIGNAL(triggered()), this, SLOT(showHelp()));
}

void MainWindow::showSettings() {
	SettingsDialog settingsDialog(this);
	settingsDialog.exec();
	// Fill lesson list after changing program settings
	startWidget->fillLessonList(false);
	startWidget->readSettings();
}

void MainWindow::showUpdate() {
	UpdateDialog updateDialog(this);
	updateDialog.exec();
	// Fill lesson list after online update
	startWidget->fillLessonList(false);
}

void MainWindow::showHelp() {
	// Open help html file in the windows standard browser
	/*QProcess p;
	QString path = QDir::currentPath() + "/help/help.html";
    QStringList s;
    s << "url.dll,FileProtocolHandler" << path;
    p.startDetached(QString("rundll32.exe") , s );*/
    helpBrowser = new HelpBrowser("", 0);
	helpBrowser->show();
}

void MainWindow::about() {
	QSqlQuery query;
	QString dbVersion = "?";
	QString softwareVersion = "?";

	// Get database version info
	if (!query.exec("SELECT * FROM db_version ORDER BY version DESC;")) {
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_DB_VERSION_READABLE,
			TYPE_WARNING, CANCEL_NO);
		// return is not necessary here - "about message" anyway
	} else {
		if (query.first()) {
			// One number represents the DB version
			dbVersion = query.value(0).toString();
			softwareVersion = query.value(1).toString();
		}
	}
	//Show about window with image
	IllustrationDialog illustrationDialog("about.png", 540, 400,
		"ber TIPP10", "Programmversion " + softwareVersion +
		"\nDatenbankversion " + dbVersion, "", 1, this);
	illustrationDialog.exec();

	//Old about box
	/*QMessageBox::about(this, APP_NAME,
			QString("<b>%1<br>" + QObject::tr("Version ") + softwareVersion
			+ "</b> (" + QObject::tr("DB Version ") + dbVersion + ")<br><br>"
			+ QObject::tr("Im Internet: ") + "<u>" + APP_URL + "</u><br><br>"
			+ APP_COPYRIGHT + "<br>" + QObject::tr(
			"This program is free software; you can redistribute it and/or "
			"modify it under the terms of the GNU General Public License "
			"as published by the Free Software Foundation; either version 2 "
			"of the License, or (at your option) any later version.<br>"
			"This program is distributed in the hope that it will be useful, "
			"but WITHOUT ANY WARRANTY; without even the implied warranty of "
			"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
			"GNU General Public License for more details.<br>"
			"You should have received a copy of the GNU General Public License "
			"along with this program; if not, write to the Free Software "
			"Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA "
			"02110-1301, USA.<br>"
			)).arg(APP_NAME));*/
}

void MainWindow::createWidgets() {
	stackedWidget = new QStackedWidget(this);
	setCentralWidget(stackedWidget);
	startWidget = new StartWidget(this);

}

void MainWindow::createStart() {
	menuBar()->show();
    //setCentralWidget(startWidget);
    stackedWidget->addWidget(startWidget);
    stackedWidget->setCurrentWidget(startWidget);
	connect(startWidget, SIGNAL(trainingClicked(int, int, QString)),
		this, SLOT(toggleStartToTraining(int, int, QString)));
	if (!isMaximized() && height() < APP_HEIGHT_STANDARD) {
		resize(APP_WIDTH_STANDARD, APP_HEIGHT_STANDARD);
	}
	setMinimumSize(APP_WIDTH_STANDARD, APP_HEIGHT_STANDARD);
}

void MainWindow::deleteStart() {
    stackedWidget->removeWidget(startWidget);
	disconnect(startWidget, SIGNAL(trainingClicked(int, int, QString)),
		this, SLOT(toggleStartToTraining(int, int, QString)));
	startWidget->writeSettings();
	//delete startWidget;
}

void MainWindow::createTraining(int lesson, int type, QString name) {
	trainingStarted = true;
	menuBar()->hide();
	trainingWidget = new TrainingWidget(lesson, type, name, this);
	//setCentralWidget(trainingWidget);
    stackedWidget->addWidget(trainingWidget);
    stackedWidget->setCurrentWidget(trainingWidget);
    trainingWidget->tickerBoard->setFocus();
    connect(trainingWidget, SIGNAL(lessonReady(int, int)),
		this, SLOT(toggleTrainingToEvaluation(int, int)));
	connect(trainingWidget, SIGNAL(lessonCanceled()),
		this, SLOT(toggleTrainingToStart()));
}

void MainWindow::deleteTraining() {
    stackedWidget->removeWidget(trainingWidget);
	trainingStarted = false;
	disconnect(trainingWidget, SIGNAL(lessonReady(int, int)),
		this, SLOT(toggleTrainingToEvaluation(int, int)));
	disconnect(trainingWidget, SIGNAL(lessonCanceled()),
		this, SLOT(toggleTrainingToStart()));
	delete trainingWidget;
}

void MainWindow::createEvaluation(int row, int type) {
	menuBar()->hide();
	evaluationWidget = new EvaluationWidget(row, type, this);
    //setCentralWidget(evaluationWidget);
    stackedWidget->addWidget(evaluationWidget);
    stackedWidget->setCurrentWidget(evaluationWidget);
	connect(evaluationWidget, SIGNAL(readyClicked()),
		this, SLOT(toggleEvaluationToStart()));
	if (!isMaximized() && height() < APP_HEIGHT_STANDARD) {
		resize(APP_WIDTH_STANDARD, APP_HEIGHT_STANDARD);
	}
	setMinimumSize(APP_WIDTH_STANDARD, APP_HEIGHT_STANDARD);
}

void MainWindow::deleteEvaluation() {
    stackedWidget->removeWidget(evaluationWidget);
	disconnect(evaluationWidget, SIGNAL(readyClicked()),
		this, SLOT(toggleEvaluationToStart()));
	delete evaluationWidget;
}

void MainWindow::toggleStartToTraining(int lesson, int type, QString name) {
	selectedLesson = lesson;
	selectedType = type;
	selectedName = name;
	deleteStart();
	startWidget->fillLessonList(false);
	createTraining(lesson, type, name);
}

void MainWindow::toggleTrainingToStart() {
	deleteTraining();
	createStart();
}

void MainWindow::toggleTrainingToEvaluation(int row, int type) {
	deleteTraining();
	createEvaluation(row, type);
}

void MainWindow::toggleEvaluationToStart() {
	deleteEvaluation();
	createStart();
	startWidget->fillLessonList(false);
}

void MainWindow::toggleStartToEvaluation() {
	deleteStart();
	createEvaluation(0, 0);
}

void MainWindow::readSettings() {
	QSettings settings;
	settings.beginGroup("mainwindow");
	//resize(settings.value("size", QSize(APP_WIDTH_STANDARD, APP_HEIGHT_STANDARD)).toSize());
	//move(settings.value("pos", QPoint(100, 100)).toPoint());
	restoreGeometry(settings.value("geometry").toByteArray());
	settings.endGroup();
	settings.beginGroup("general");
	showIllustration = settings.value("check_illustration", true).toBool();
	settings.endGroup();
}

void MainWindow::writeSettings() {
	QSettings settings;
	settings.beginGroup("mainwindow");
	settings.setValue("size", size());
	settings.setValue("pos", pos());
	settings.setValue("geometry", saveGeometry());
	settings.endGroup();
}
