/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the LessonDialog class
** File name: lessondialog.cpp
**
****************************************************************/

#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QSqlQuery>
#include <QSettings>
#include <QMessageBox>
#include <QFontDialog>
#include <QFileDialog>
#include <QStringList>
#include <QString>

#include "lessondialog.h"
#include "sql/startsql.h"
#include "def/errordefines.h"
#include "errormessage.h"
#include "def/defines.h"

LessonDialog::LessonDialog(QString lessonid, QWidget *parent) : QDialog(parent) {
	currentLessonId = lessonid;

    // Create buttons
    createButtons();

    // Create controls
    createControls();

	// Set the layout of all widgets created above
	createLayout();

	// Widget connections
    connect(buttonSave, SIGNAL(clicked()), this, SLOT(clickSave()));
    connect(buttonCancel, SIGNAL(clicked()), this, SLOT(reject()));
    connect(buttonHelp, SIGNAL(clicked()), this, SLOT(showHelp()));

    setWindowIcon(QIcon("img/icon.ico"));

    setMinimumSize(420, 470);

    buttonSave->setFocus();

    updateContent();
}

void LessonDialog::updateContent() {
	StartSql *startSql;
	// Dialog is in edit modus (lesson id exist)
	if (currentLessonId != "-1") {
		startSql = new StartSql();
		if (!startSql->getOwnLesson(currentLessonId, lineLessonName,
			lineLessonDescription, lineLessonContent,
			radioUnitSentence, radioUnitWord)) {
			// No selected lesson found in combo box
			// -> error message
			ErrorMessage *errorMessage = new ErrorMessage(this);
			errorMessage->showMessage(ERR_USER_LESSON_GET, TYPE_INFO,
				CANCEL_OPERATION);
			return;
		}
	} else {
		lineLessonName->selectAll();
		lineLessonName->setFocus();
	}
}

void LessonDialog::createButtons() {
	// Buttons
	buttonCancel = new QPushButton(QObject::tr("&Abbrechen"));
	buttonSave = new QPushButton(QObject::tr("&Speichern"));
	buttonHelp = new QPushButton(QObject::tr("&Hilfe"));
	buttonSave->setDefault(true);
}

void LessonDialog::createControls() {
	// Labels
	labelLessonName = new QLabel(QObject::tr("Name der Lektion (maximal 20 "
		"Zeichen):"));
	labelLessonDescription = new QLabel(QObject::tr("Kurzbeschreibung (maximal "
		"120 Zeichen):"));
	labelLessonContent = new QLabel(QObject::tr("Diktat (mindestens 20 "
		"Zeilen):"));
	labelLessonNotices = new QLabel(QObject::tr("<u>Kurze Erluterung</u>"
		"<br>&nbsp;<br>Jede Zeile (Zeilenumbruch "
		"am Ende) entspricht einer Einheit fr das Diktat. "
		"Das Diktat kann auf zwei Arten gefhrt werden:<br>&nbsp;<br>"
		"<b>Satzdiktat</b> - die "
		"einzelnen Zeilen (Stze) werden wie hier eingegeben mit einem Zeilenumbruch am "
		"Ende diktiert.<br>&nbsp;<br>"
		"<b>Wortdiktat</b> - die einzelnen Zeilen (Worte) werden mit Leerzeichen getrennt "
		"diktiert, ein Zeilenumbruch erfolgt automatisch nach mindestens ") +
		QString::number(NUM_TOKEN_UNTIL_NEW_LINE) + QObject::tr(" diktierten "
		"Schriftzeichen."));
	labelLessonNotices->setWordWrap(true);
	labelLessonNotices->setMaximumWidth(180);
	labelLessonUnit = new QLabel(QObject::tr("<b>Was passiert bei aktivierter "
		"Intelligenz?</b><br>Bei "
		"aktivierter Intelligenz werden die Zeilen nicht der Reihenfolge nach "
		"diktiert, sondern abhngig von den Tippfehler-Quoten die aktuell "
		"fr den Lernerfolg sinnvollste Zeile in das Diktat einbezogen. Die "
		"Intelligenz zu aktivieren macht nur bei Lektionen Sinn, die aus sehr "
		"vielen Zeilen bestehen.<br>"));
	labelLessonUnit->setWordWrap(true);
	labelLessonUnit->setMaximumWidth(180);
	labelLessonUnitRadio = new QLabel(QObject::tr("Das Diktat soll gefhrt werden als:"));
	labelLessonUnitRadio->setWordWrap(true);
	// Lines (text input)
	lineLessonName = new QLineEdit();
	lineLessonName->setMaxLength(20);
	lineLessonDescription = new QLineEdit();
	lineLessonDescription->setMaxLength(120);
	lineLessonContent = new QTextEdit();
	lineLessonContent->setLineWrapMode(QTextEdit::NoWrap);
	lineLessonContent->setAcceptRichText(false);
	// Radiobutton sentence unit
	radioUnitSentence = new QRadioButton(QObject::tr("Satzdiktat"));
	radioUnitSentence->setToolTip(QObject::tr("Die einzelnen "
		"Einheiten werden mit\neinem Zeilenumbruch am Ende diktiert"));
	radioUnitSentence->setChecked(true);
	// Radiobutton word unit
	radioUnitWord = new QRadioButton(QObject::tr("Wortdiktat"));
	radioUnitWord->setToolTip(QObject::tr("Die einzelnen "
		"Einheiten werden mit\nLeerzeichen getrennt "
		"diktiert,\nein Zeilenumbruch erfolgt automatisch\nnach mindestens ") +
		QString::number(NUM_TOKEN_UNTIL_NEW_LINE) + QObject::tr(" "
		"Schriftzeichen"));
	radioUnitWord->setChecked(false);

	// Dialog is in edit modus (lesson id exist)
	if (currentLessonId != "-1") {
		setWindowTitle(QObject::tr("Eigene Lektion editieren"));
		// Change text of lesson name label
		labelLessonName->setText(QObject::tr("Name der Lektion:"));
		// Disable text field "Name of lesson"
		lineLessonName->setEnabled(false);
	} else {
		setWindowTitle(QObject::tr("Eigene Lektion erstellen"));
		// Select text field "Name of lesson"
		//lineLessonName->selectAll();
		//lineLessonName->setCursorPosition(0);
	}
}

void LessonDialog::createLayout() {
	// Button layout horizontal
	QHBoxLayout *buttonLayoutHorizontal = new QHBoxLayout;
    buttonLayoutHorizontal->addStretch(1);
    buttonLayoutHorizontal->addWidget(buttonCancel);
    buttonLayoutHorizontal->addSpacing(10);
    buttonLayoutHorizontal->addWidget(buttonHelp);
    buttonLayoutHorizontal->addWidget(buttonSave);
    // Group layout vertical
	QHBoxLayout *boxesLayout = new QHBoxLayout;
    boxesLayout->addStretch(1);
    boxesLayout->addWidget(radioUnitSentence);
    boxesLayout->addSpacing(30);
    boxesLayout->addWidget(radioUnitWord);
    boxesLayout->addStretch(1);
	QVBoxLayout *helpLayout = new QVBoxLayout;
    helpLayout->addWidget(labelLessonNotices);
    helpLayout->addSpacing(14);
    helpLayout->addWidget(labelLessonUnit);
    helpLayout->addStretch(1);
	QVBoxLayout *controlLayout = new QVBoxLayout;
    controlLayout->addWidget(labelLessonName);
    controlLayout->addWidget(lineLessonName);
    controlLayout->addSpacing(10);
    controlLayout->addWidget(labelLessonDescription);
    controlLayout->addWidget(lineLessonDescription);
    controlLayout->addSpacing(10);
    controlLayout->addWidget(labelLessonContent);
    controlLayout->addWidget(lineLessonContent);
    controlLayout->addSpacing(10);
    controlLayout->addWidget(labelLessonUnitRadio);
    controlLayout->addLayout(boxesLayout);
	// Full layout of all widgets vertical
	QHBoxLayout *preLayout = new QHBoxLayout;
    preLayout->addLayout(controlLayout);
    preLayout->addSpacing(20);
    preLayout->addLayout(helpLayout);
	// Full layout of all widgets vertical
	QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addLayout(preLayout);
    mainLayout->addSpacing(1);
    mainLayout->addLayout(buttonLayoutHorizontal);
    mainLayout->setMargin(15);
    mainLayout->setSpacing(5);
    // Pass layout to parent widget (this)
	this->setLayout(mainLayout);
}

void LessonDialog::clickSave() {
	StartSql *startSql;
	QStringList contentList;
	int lessonUnit = 0;
	int i;

	// Check whether lesson name is filled out
	if (lineLessonName->text() == "") {
		QMessageBox::information(this, APP_NAME,
			QObject::tr("Bitte geben Sie der Lektion einen Namen\n"));
		return;
	}
	// Check whether lesson content is filled out
	if (lineLessonContent->toPlainText() == "") {
		QMessageBox::information(this, APP_NAME,
			QObject::tr("Bitte geben Sie ein vollstndiges Diktat ein\n"));
		return;
	}

	// Split lesson content to lines
	contentList = lineLessonContent->toPlainText().split("\n", QString::SkipEmptyParts);
	// Delete empty lines
	for (i = 0; i < contentList.size(); i++) {
		if (QString::QString(contentList.at(i).toLocal8Bit().constData()) == "") {
			contentList.removeAt(i);
		}
	}
	// Check whether there is enough lesson content
	if (contentList.size() < 20) {
		QMessageBox::information(this, APP_NAME,
			QObject::tr("Bitte geben Sie mindestens 20 Zeilen Diktat ein\n"));
		return;
	}
	// Check whether there is too much lesson content
	if (contentList.size() > 400) {
		QMessageBox::information(this, APP_NAME,
			QObject::tr("Bitte geben Sie maximal 400 Zeilen Diktat ein\n"));
		return;
	}
	// Save the lesson
	startSql = new StartSql();
	if (currentLessonId == "-1" && startSql->ownLessonExist(lineLessonName->text())) {
		QMessageBox::information(this, APP_NAME,
			QObject::tr("Der Name der Lektion existiert bereits. Bitte geben "
				"Sie der Lektion einen anderen Namen.\n"));
		return;
	}
	if (radioUnitWord->isChecked()) {
		lessonUnit = 1;
	}
	if (!startSql->updateOwnLesson(currentLessonId, lineLessonName->text(),
		lineLessonDescription->text(), contentList, lessonUnit)) {
		// No selected lesson found in combo box
		// -> error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_USER_LESSON_ADD, TYPE_INFO,
			CANCEL_OPERATION);
		return;
	}
	if (!startSql->analyzeOwnLessons()) {
		// No selected lesson found in combo box
		// -> error message
		ErrorMessage *errorMessage = new ErrorMessage(this);
		errorMessage->showMessage(ERR_USER_LESSON_ANALYZE, TYPE_INFO,
			CANCEL_OPERATION);
		return;
	}
	this->accept();
}

void LessonDialog::showHelp() {
	helpBrowser = new HelpBrowser("lessons.html#ownlesson", this);
	helpBrowser->show();
}
