/*
Copyright (c) 2006-2007, Tom Thielicke IT Solutions

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.
*/

/****************************************************************
**
** Implementation of the TrainingSql class
** File name: trainingsql.cpp
**
****************************************************************/

#include <QSqlQuery>
#include <QSqlDriver>
#include <QVariant>
#include <QSqlDatabase>
#include <QDateTime>
#include <QRegExp>

#include "trainingsql.h"
#include "def/defines.h"

// Konstruktor
TrainingSql::TrainingSql() {
	setSynchron(SYNCHRON_DB_WHILE_TRAINING);
}

TrainingSql::~TrainingSql() {
	setSynchron(true);
}

int TrainingSql::getLessonUnit(int lesson, int type) {
	QSqlQuery query;
	int lessonUnit;
	QString tableName;
	QString sqlString;

	switch (type) {
		case 0:
			tableName = "lesson";
			break;
		case 1:
			tableName = "open";
			break;
		case 2:
			tableName = "own";
			break;
	}
	sqlString = "SELECT " + tableName + "_list." + tableName + "_id, " +
		tableName + "_list." + tableName + "_unit FROM " + tableName + "_list WHERE " +
		tableName + "_list." + tableName + "_id = " + QString::number(lesson) + ";";
	// Standard query to get the first lesson
	if (!query.exec(sqlString)) {
        return 0;
	}
	if (!query.first()) {
		return 0;
	}
	lessonUnit = query.value(1).toInt();
	return lessonUnit;
}

void TrainingSql::setSynchron(bool synchron) {
	QSqlQuery query;
	QString queryString;
	if (synchron) {
		queryString = "PRAGMA synchronous=FULL;";
	} else {
		queryString = "PRAGMA synchronous=OFF;";
	}
	query.exec(queryString);
}

bool TrainingSql::updateUsertable(QChar unicodechar, QString columnname) {
	QSqlQuery query;
	//int sqlNum = 0;
	// First a simple select statement to check whether the unicode exists
	// already in table
	/*if (!query.exec("SELECT COUNT(uerrorunicode) FROM usererrors WHERE "
		"uerrorunicode = " + QString::number(unicodechar.unicode()) + ";")) {
        return false;
	}*/

	// Now check the number of rows returned
	// Below commented is the full code to test wheter the current database
	// supports the feature "size()".
	// At the Moment a SQLite Driver Version 3.3.4 is in use - it doesn't
	// support the feature "size()". So we have to go to the end of the table
	// to count the number of rows returned as listet below after the commented
	// code.
	// This is slow - so change it if using an other database driver which
	// supports the feature!

	/*QSqlDatabase defaultDB = QSqlDatabase::database();

	//defaultDB.driver()->hasFeature(QSqlDriver::Transactions);
	if (defaultDB.driver()->hasFeature(QSqlDriver::QuerySize)) {
		sqlNum = query.size();
	} else {
		query.last();
		sqlNum = query.at() + 1;
	}*/

	// Slow function (look at the comment above)

	//sqlNum = query.at() + 1;
	/*query.first();
    if (query.value(0).toInt() == 0) {
		// There is no data record with the current unicode
		// -> insert new data record
		if (!query.exec("INSERT INTO usererrors VALUES("
			+ QString::number(unicodechar.unicode()) + ",0,0,0);")) {
			return false;
		}
	}*/
	// Now we keep save there is already a data record with the current unicode
	// -> update (increment) current data record
	QString charToString = QString::number(unicodechar.unicode());
	//query.exec("INSERT INTO usererrors VALUES(" + temp + ",0,0,0);");
	query.exec("BEGIN;");
	query.exec("INSERT OR IGNORE INTO user_chars VALUES(" + charToString +
		",0,0,0);");
	query.exec("UPDATE user_chars SET " + columnname + "="
		+ columnname + "+1 WHERE user_char_unicode = " + charToString + ";");
	if (!query.exec("COMMIT;")) {
		return false;
	}
	return true;
}

QString TrainingSql::createLesson(int lesson, int type, int unit,
	bool intelligence, bool useEszett) {
	QSqlQuery query;
	QString sqlString = "";
	QString tableName = "";
	QString textOutput = "";
	QString textOutputSub = "";
	int counterToNewLine;

	switch (type) {
		case 0:
			tableName = "lesson_content";
			break;
		case 1:
			tableName = "open_content";
			break;
		case 2:
			tableName = "own_content";
			break;
	}
	sqlString = "SELECT " + tableName + ".content_id, " +
		tableName + ".content_text FROM " + tableName + " WHERE " +
		tableName + ".content_lesson = " + QString::number(lesson) +
		" ORDER BY " + tableName + ".content_id;";
	// Standard query to get the first lesson
	if (!query.exec(sqlString)) {
        return "";
	}
	if (type == 0 || intelligence) {
		if (!query.first()) {
			return "";
		}
		if (query.isNull(0)) {
			return "";
		}
		lessonsDoneList.prepend(query.value(0).toInt());
		textOutput = query.value(1).toString();
		//if (textOutput.contains(QChar(0x9), Qt::CaseSensitive)) {
		textOutput.replace(QChar(0x9), QChar(TOKEN_TAB), Qt::CaseSensitive);
		//}
		if (!useEszett) {
			textOutput.replace(QChar(0x00df), "ss", Qt::CaseSensitive);
		}
	} else {
		counterToNewLine = 0;
		// Without intelligence
		while (query.next()) {
			textOutputSub = query.value(1).toString();
			counterToNewLine += textOutputSub.length();
			if (unit == 0 || (unit == 1 &&
				counterToNewLine > NUM_TOKEN_UNTIL_NEW_LINE)) {
				textOutputSub.append(QChar(TOKEN_NEW_LINE));
				counterToNewLine = 0;
			} else {
				textOutputSub.append(" ");
				counterToNewLine++;
			}
			//if (textOutputSub.contains(QChar(0x9), Qt::CaseSensitive)) {
			textOutputSub.replace(QChar(0x9), QChar(TOKEN_TAB), Qt::CaseSensitive);
			//}
			if (!useEszett) {
				textOutputSub.replace(QChar(0x00df), "ss", Qt::CaseSensitive);
			}
			textOutput.append(textOutputSub);
		}
	}
	// return new lesson
	return textOutput;
}

QString TrainingSql::updateLesson(int lesson, int type, bool intelligence,
	bool useEszett) {
	QSqlQuery query;
	QString errorUnicodeMaxFirst = "";
	QString errorUnicodeMaxSecond = "";
	QString errorUnicodeMaxThird = "";
	QString errorUnicodeMaxFourth = "";
	QString sqlString = "";
	bool errorsExist = false;
	QString tableName = "";
	QString analysisName = "";
	QString textOutput = "";
	QString textOutputSub = "";

	switch (type) {
		case 0:
			tableName = "lesson_content";
			analysisName = "lesson_analysis";
			break;
		case 1:
			tableName = "open_content";
			analysisName = "open_analysis";
			break;
		case 2:
			tableName = "own_content";
			analysisName = "own_analysis";
			break;
	}
	if (intelligence) {
		// Check if user errors exist and take the unicode with the
		// highest error ratio
		if (query.exec("SELECT user_char_unicode, (user_char_target_errornum * "
			"100) / user_char_occur_num AS user_char_weighted FROM user_chars "
			"ORDER BY user_char_weighted DESC;")) {
			if (query.first()) {
				errorsExist = true;
				errorUnicodeMaxFirst = query.value(0).toString();
				if (query.next()) {
					errorUnicodeMaxSecond = query.value(0).toString();
					if (query.next()) {
						errorUnicodeMaxThird = query.value(0).toString();
						if (query.next()) {
							errorUnicodeMaxFourth = query.value(0).toString();
						}
					}
				}
			}
		} else {
			// SQL exec error
			return "";
		}
		sqlString = "SELECT " + tableName + ".content_id, " +
				tableName + ".content_text FROM " + tableName;

		if (errorsExist) {
			// User error exists
			// -> create lesson SQL string sorted by errorUnicodeMax
			sqlString.append(" LEFT JOIN " +
				analysisName + " ON " + tableName + ".content_id = " +
				analysisName + ".analysis_content AND " +
				tableName + ".content_lesson");
			if (lesson != LAST_LESSON || type != 0) {
				sqlString.append(" = " + QString::number(lesson));
			} else {
				// Last training lesson over all previous lessons
				sqlString.append(" >= " + QString::number(BORDER_LESSON_IS_SENTENCE));
			}
			sqlString.append(" AND (" + tableName + ".content_id % 1000) <> 0 "
				"GROUP BY " + tableName + ".content_id");
			if (errorUnicodeMaxFirst != "") {
				sqlString.append(" ORDER BY analysis_char_"
					+ errorUnicodeMaxFirst + " DESC");
				if (errorUnicodeMaxSecond != "") {
					sqlString.append(", analysis_char_" + errorUnicodeMaxSecond
						+ " DESC");
				}
				if (errorUnicodeMaxThird != "") {
					sqlString.append(", analysis_char_" + errorUnicodeMaxThird
						+ " DESC");
				}
				if (errorUnicodeMaxFourth != "") {
					sqlString.append(", analysis_char_" + errorUnicodeMaxFourth
						+ " DESC");
				}
			}
			sqlString.append(";");
		} else {
			// No user error exists
			// -> create standard lesson SQL string sorted by id
			sqlString.append(" WHERE " +
			tableName + ".content_lesson = " + QString::number(lesson)
			+ " ORDER BY " + tableName + ".content_id;");
		}
		// Execute SQL string created above
		if (query.exec(sqlString)) {
			// Find a lesson not occurs last time
			while (query.next()) {
				if (!lessonsDoneList.contains(query.value(0).toInt())) {
					// New unused lesson found
					// -> append to lessonsDoneList
					lessonsDoneList.prepend(query.value(0).toInt());
					// Check if lessonsDoneList exceeds its maximum
					if (lessonsDoneList.size() > NUM_TEXT_UNTIL_REPEAT) {
						// lessonsDoneList exceeded -> remove last item
						lessonsDoneList.removeLast();
					}
					textOutput = query.value(1).toString();
					//if (textOutput.contains(QChar(0x9), Qt::CaseSensitive)) {
					textOutput.replace(QChar(0x9), QChar(TOKEN_TAB), Qt::CaseSensitive);
					//}
					if (!useEszett) {
						textOutput.replace(QChar(0x00df), "ss", Qt::CaseSensitive);
					}
					// return new lesson
					return textOutput;
				}
			}
		} else {
			// SQL exec error
			return "";
		}
		// No unused lesson found
		// -> delete all items expect the first of lessonsDoneList
		while (lessonsDoneList.size() > 1) {
			lessonsDoneList.removeLast();
		}
		return updateLesson(lesson, type, intelligence, useEszett);
	} else {
		// Without intelligence
		sqlString = "SELECT " + tableName + ".content_id, " +
			tableName + ".content_text FROM " + tableName + " WHERE " +
			tableName + ".content_lesson = " + QString::number(lesson) +
			" ORDER BY " + tableName + ".content_id;";
		// Standard query to get the first lesson
		if (!query.exec(sqlString)) {
			return "";
		}
		while (query.next()) {
			textOutputSub = query.value(1).toString();
			textOutputSub.append(QChar(TOKEN_NEW_LINE));
			//if (textOutputSub.contains(QChar(0x9), Qt::CaseSensitive)) {
			textOutputSub.replace(QChar(0x9), QChar(TOKEN_TAB), Qt::CaseSensitive);
			//}
			if (!useEszett) {
				textOutputSub.replace(QChar(0x00df), "ss", Qt::CaseSensitive);
			}
			textOutput.append(textOutputSub);
		}
		return textOutput;
	}
}

QVariant TrainingSql::saveLesson(int lesson, int timelen, int tokenlen,
	int charnum, int errornum, QDateTime timestamp, int type, QString name) {
	QSqlQuery query;
	QVariant lastRowId;
	QString lessonName;
	if (type == 0 && name.length() > 7) {
		lessonName = "bungslektion" + name.mid(7, 3);
	} else {
		lessonName = name;
	}
	if (!query.exec("INSERT INTO user_lesson_list VALUES(NULL," +
		QString::number(lesson) + "," + QString::number(timelen) + "," +
		QString::number(tokenlen) + "," + QString::number(charnum) + "," +
		QString::number(errornum) + ", '" +
		timestamp.toString("yyyyMMddhhmmss") + "', " +
		QString::number(type) + ", '" + lessonName + "');")) {
		return QVariant::Invalid;
	}
	lastRowId = query.lastInsertId();
	return lastRowId;
}
