/*	loadsave.cpp - Created by Giampiero Caprino

This file is part of Train Director 3

Train Director is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; using exclusively version 2.
It is expressly forbidden the use of higher versions of the GNU
General Public License.

Train Director is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Train Director; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>

#if !defined(__unix__) && !defined(__WXMAC__)
#include <malloc.h>
#endif

#include <memory.h>
#include <string.h>
#include "wx/ffile.h"
#include "Traindir3.h"
#include "html.h"
#include "TDFile.h"
#include "Itinerary.h"

extern	void	*w_train_pmap_default[4], *w_train_pmap[4];
extern	void	*e_train_pmap_default[4], *e_train_pmap[4];
extern	void	*w_car_pmap_default[4], *w_car_pmap[4];
extern	void	*e_car_pmap_default[4], *e_car_pmap[4];

Path	*paths;
Vector	*findPath(Track *t, int dir);
void	colorPartialPath(Vector *path, int state, int start);
void	colorPath(Vector *path, int state);
void	compute_train_numbers();
void	new_train_status(Train *t, int status);

bool	performance_hide_canceled = false;

pxmap	*pixmaps;
int	npixmaps, maxpixmaps;

pxmap	*carpixmaps;
int	ncarpixmaps, maxcarpixmaps;

extern	wxString	current_project;

	wxChar	dirPath[1024];

static	int	curtype = 0;
	int	save_prefs = 1;

static wxChar	*linebuff;
static	int	maxline;

static	const wxChar	*getline(TDFile *fp)
{
	int	i;
	wxChar	*buffpos;
	bool	notEOF;

	if(!linebuff) {
	    maxline = 256;
	    linebuff = (wxChar *)malloc(maxline * sizeof(linebuff[0]));
	    if (!linebuff)
		return NULL;
	}
	buffpos = linebuff;
	i = maxline;
	linebuff[maxline - 2] = '\n';
	while ((notEOF = fp->ReadLine(buffpos, i))) {
	    if(linebuff[maxline - 2] != '\n') {
	        i = maxline - 1;
		maxline += 256;
		linebuff = (wxChar *)realloc(linebuff, maxline * sizeof(linebuff[0]));
		if (!linebuff)
		    return NULL;
		buffpos = &linebuff[i];
		i = maxline - i;
		linebuff[maxline - 2] = '\n';
	    } else
		break;
	}
	if(!notEOF)
	    return 0;
	return linebuff;
}

bool	file_create(const wxChar *name, const wxChar *ext, wxFFile& fp)
{
	wxChar	buff[256];
	wxChar	*p;
	extern	int errno;

	wxStrncpy(buff, name, sizeof(buff)/sizeof(wxChar) - 1);
	buff[sizeof(buff)/sizeof(wxChar) - 1] = 0;
	for(p = buff + wxStrlen(buff); p > buff && *p != '.' && *p != '/' && *p != '\\'; --p);
	if(*p == '.')
	    wxStrcpy(p, ext);
	else
	    wxStrcat(p, ext);
	if(fp.Open(buff, wxT("w")))
	    return true;
	wxSnprintf(buff, sizeof(buff)/sizeof(wxChar), wxT("%s '%s' - %s."), L("Can't create file"), name, L("Error"));
	traindir->Error(buff);
	return false;
}

const wxChar	*locase(wxChar *s)
{
	wxChar	*p;

	for(p = s; *p; ++p)
	    *p = wxTolower(*p);
	return s;
}

wxChar	*skipblk(wxChar *p)
{
	while(*p == ' ' || *p == '\t') ++p;
	return p;
}

void	clean_field(Track *layout)
{
	Track	*t;

	while(layout) {
	    t = layout->next;
	    if(layout->station)
		free(layout->station);
	    free(layout);
	    layout = t;
	}
}

Track	*load_field_tracks(const wxChar *name, Itinerary **itinList)
{
	Track	*layout, *t, *lastTrack;
	TextList *tl, *tlast;
	Itinerary *it;
	wxChar	buff[1024];
	int	l;
	int	ttype;
	int	x, y, sw;
	wxChar	*p, *p1;
	TDFile	trkFile(name);

	trkFile.SetExt(wxT(".trk"));
	if(!trkFile.Load()) {
	    wxSnprintf(buff, sizeof(buff)/sizeof(wxChar), wxT("File '%s' not found."), trkFile.name.GetFullPath().c_str());
	    traindir->Error(buff);
	    return 0;
	}
	lastTrack = 0;
	tlast = 0;
	layout = 0;
	while(trkFile.ReadLine(buff, sizeof(buff)/sizeof(wxChar))) {
	    //t = (Track *)malloc(sizeof(Track));
	    //memset(t, 0, sizeof(Track));
	    t = new Track();
	    t->fgcolor = fieldcolors[COL_TRACK];
	    ttype = buff[0];

	    if(!wxStrncmp(buff, wxT("(script "), 8)) {
		p = buff + 8;
		x = wxStrtol(p, &p, 10);
		if(*p == wxT(',')) ++p;
		y = wxStrtol(p, &p, 10);

		wxString	script;
		while(trkFile.ReadLine(buff, sizeof(buff)/sizeof(wxChar)) && buff[0] != ')') {
		    wxStrcat(buff, wxT("\n"));
		    script += buff;
		}

		for(t = layout; t; t = t->next) {
		    if(t->x == x && t->y == y)
			break;
		}
		if(!t)
		    continue;
		if(t->stateProgram)
		    free(t->stateProgram);
		t->stateProgram = wxStrdup(script.c_str());
		continue;
	    }
	    if(!wxStrncmp(buff, wxT("(attributes "), 12)) {
		x = wxStrtol(buff + 12, &p, 10);
		if(*p == wxT(',')) ++p;
		y = wxStrtol(p, &p, 10);
		t = find_track(layout, x, y);
		while(trkFile.ReadLine(buff, sizeof(buff)/sizeof(wxChar)) && buff[0] != ')') {
		    if(!t)
			continue;
		    if(!wxStrcmp(buff, wxT("hidden"))) {
			t->invisible = 1;
			continue;
		    }
		}
		continue;
	    }
#if 0
	    // script lines

	    if(ttype == wxT('\t') || ttype == wxT(' ')) {
		wxStrcat(buff, wxT("\n"));
		append_to_script(buff);
	    }
#endif
	    p = buff + 1;
	    if(*p == wxT(',')) ++p;
	    t->x = wxStrtol(p, &p, 10);
	    if(*p == wxT(',')) ++p;
	    t->y = wxStrtol(p, &p, 10);
	    if(t->x >= ((XMAX - HCOORDBAR) / HGRID) ||
		    t->y >= ((YMAX - VCOORDBAR) / VGRID))
		continue;
	    if(*p == wxT(',')) ++p;
	    t->direction = (trkdir)wxStrtol(p, &p, 10);
	    if(*p == wxT(',')) ++p;
	    if(!layout)
		layout = t;
	    else
		lastTrack->next = t;
	    lastTrack = t;
	    switch(ttype) {
	    case wxT('0'):
		t->type = TRACK;
		t->isstation = (char)wxStrtol(p, &p, 10);
		if(*p == wxT(',')) ++p;
		t->length = wxStrtol(p, &p, 10);
		if(!t->length)
		    t->length = 1;
		if(*p == ',') ++p;
		t->wlinkx = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->wlinky = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->elinkx = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->elinky = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		if(*p == '@') {
		    int	    i;

		    t->speed[0] = wxStrtol(p + 1, &p, 10); 
		    for(i = 1; i < NTTYPES && *p == '/'; ++i) {
			t->speed[i] = wxStrtol(p + 1, &p, 10); 
		    }
		    if(*p == ',') ++p;
		}
		if(!*p || !wxStrcmp(p, wxT("noname"))) {
		    t->isstation = 0;
		    break;
		}
		if(*p == '>') {
		    p = parse_km(t, p + 1);
		    if(*p == ',')
			++p;
		}
		t->station = wxStrdup(p);
		break;

	    case '1':
		t->type = SWITCH;
		t->length = 1;
		t->wlinkx = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->wlinky = wxStrtol(p, &p, 10);
		break;

	/* 2, x, y, type, linkx, linky [itinerary] */

	    case '2':
		t->type = TSIGNAL;
		t->status = ST_RED;
		if((l = t->direction) & 2) {
		    t->fleeted = 1;
		    l &= ~2;
		}
		if(l & 0x100)
		    t->fixedred = 1;
		if(l & 0x200)
		    t->nopenalty = 1;
		if(l & 0x400)
		    t->signalx = 1;
		l &= ~0x700;
		t->direction = (trkdir)((int)t->direction & ~0x700);

		switch(l) {
		case 0:
		    t->direction = E_W;
		    break;

		case 1:
		    t->direction = W_E;
		    break;

		case N_S:
		case S_N:
		case signal_SOUTH_FLEETED:
		case signal_NORTH_FLEETED:
		    /* already there */
		    t->direction = (trkdir)l;
		    break;
		}
		t->wlinkx = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->wlinky = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		if(*p == '@') {
		    p1 = p + 1;
		    p = wxStrchr(p1, ',');
		    if(p)
			*p++ = 0;
		    t->stateProgram = wxStrdup(p1);
		}
		if(p && *p)			/* for itinerary definition */
		    t->station = wxStrdup(p);
		break;

	    case '3':
		t->type = PLATFORM;
		if(t->direction == 0)
		    t->direction = W_E;
		else
		    t->direction = N_S;
		break;

	    case '4':
		t->type = TEXT;
		t->station = wxStrdup(p);
		for(l = 0; t->station[l] && t->station[l] != ','; ++l);
		t->station[l] = 0;
		while(*p && *p != ',') ++p;
		if(*p == ',') ++p;
		t->wlinkx = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->wlinky = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->elinkx = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->elinky = wxStrtol(p, &p, 10);
		if(*p == '>')
		    p = parse_km(t, ++p);
		break;

	    case '5':
		t->type = IMAGE;
		t->station = wxStrdup(p);
		break;

	    case '6':			/* territory information */
		tl = (TextList *)malloc(sizeof(TextList));
		wxStrcat(p, wxT("\n"));	/* put it back, since we removed it */
		tl->txt = wxStrdup(p);
		if(!track_info)
		    track_info = tl;
		else
		    tlast->next = tl;
		tl->next = 0;
		tlast = tl;
		break;

	    case '7':			/* itinerary */
		for(p1 = p; *p && *p != ','; ++p);
		if(!*p)
		    break;
		*p++ = 0;
		it = (Itinerary *)calloc(sizeof(Itinerary), 1);
		it->name = wxStrdup(p1);
		for(p1 = p, l = 0; *p && (*p != ',' || l); ++p) {
		    if(*p == '(') ++l;
		    else if(*p == ')') --l;
		}
		if(!*p)
		    break;
		*p++ = 0;
		it->signame = wxStrdup(p1);
		for(p1 = p, l = 0; *p && (*p != ',' || l); ++p) {
		    if(*p == '(') ++l;
		    else if(*p == ')') --l;
		}
		if(!*p)
		    break;
		*p++ = 0;
		it->endsig = wxStrdup(p1);
		if(*p == '@') {
		    for(p1 = ++p, l = 0; *p && (*p != ',' || l); ++p) {
			if(*p == '(') ++l;
			else if(*p == ')') --l;
		    }
		    if(!*p)
			break;
		    *p++ = 0;
		    it->nextitin = wxStrdup(p1);
		}
		l = 0;
		while(*p) {
		    x = wxStrtol(p, &p, 0);
		    if(*p != ',')
			break;
		    y = wxStrtol(p + 1, &p, 0);
		    if(*p != ',')
			break;
		    sw = wxStrtol(p + 1, &p, 0);
		    add_itinerary(it, x, y, sw);
		    if(*p == ',') ++p;
		}
		it->next = *itinList;	/* all ok, add to the list */
		*itinList = it;
		break;

	    case '8':			/* itinerary placement */
		t->type = ITIN;
		t->station = wxStrdup(p);
		break;

	    case '9':
		t->type = TRIGGER;
		t->wlinkx = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->wlinky = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->elinkx = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		t->elinky = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
		for(l = 0; l < NTTYPES && *p != ','; ++l) {
		    t->speed[l] = wxStrtol(p, &p, 10);
		    if(*p == '/') ++p;
		}
		if(*p == ',') ++p;
		if(!*p || !wxStrcmp(p, wxT("noname")))
		    break;
		t->station = wxStrdup(p);
		break;
	    }
	}
	load_scripts(layout);
	return layout;
}

Track	*load_field(const wxChar *name)
{
	int	l;
	TextList *tl;
	Itinerary *it;
	Track	*t;

	for(l = 0; l < 4; ++l) {
	    e_train_pmap[l] = e_train_pmap_default[l];
	    w_train_pmap[l] = w_train_pmap_default[l];
	    e_car_pmap[l] = e_car_pmap_default[l];
	    w_car_pmap[l] = w_car_pmap_default[l];
	}
	while((tl = track_info)) {
	    track_info = tl->next;
	    free(tl->txt);
	    free(tl);
	}
	while((it = itineraries)) {
	    itineraries = it->next;
	    free_itinerary(it);
	}
#if 0
	if(script_text) {
	    free(script_text);
	    script_text = 0;
	}
#endif
	free_scripts();
	t = load_field_tracks(name, &itineraries);
	sort_itineraries();
	if(t)
	    current_project = name;
	//add_to_script(t);
	//parse_script();
	return t;
}

Track	*find_track(Track *layout, int x, int y)
{
	while(layout) {
	    if(layout->x == x && layout->y == y)
		return(layout);
	   layout = layout->next;
	}
	return 0;
}

void	link_signals(Track *layout)
{
	TrackBase	*t;

	for(t = layout; t; t = t->next)	    /* in case signal was relinked during edit */
	    t->esignal = t->wsignal = 0;

	for(t = layout; t; t = t->next) {

	    /*	link signals with the track they control	*/

	    if(t->type == TSIGNAL) {
		if(!(t->controls = findTrack(t->wlinkx, t->wlinky)))
		    continue;
		if(t->direction == W_E || t->direction == S_N)
		    t->controls->esignal = (Signal *)t;
		else
		    t->controls->wsignal = (Signal *)t;
	    }
	}
}

void	clean_pixmap_cache(void)
{
	int	i;

	for(i = 0; i < npixmaps; ++i)
	    if(pixmaps[i].name) {
		free(pixmaps[i].name);
		delete_pixmap(pixmaps[i].pixels);
	    }
	npixmaps = 0;
	for(i = 0; i < ncarpixmaps; ++i)
	    if(carpixmaps[i].name) {
		free(carpixmaps[i].name);
		delete_pixmap(carpixmaps[i].pixels);
	    }
	ncarpixmaps = 0;
}
 
int	get_pixmap_index(const wxChar *mapname)
{
	int	i;

	for(i = 0; i < npixmaps; ++i)
	    if(!wxStrcmp(mapname, pixmaps[i].name))
		return i;
	if(npixmaps >= maxpixmaps) {
	    maxpixmaps += 10;
	    if(!pixmaps)
		pixmaps = (pxmap *)malloc(sizeof(pxmap) * maxpixmaps);
	    else
		pixmaps = (pxmap *)realloc(pixmaps, sizeof(pxmap) * maxpixmaps);
	}               
	if(!(pixmaps[npixmaps].pixels = (char *)get_pixmap_file(mapname)))
	    return -1;          /* failed! file does not exist */
	pixmaps[npixmaps].name = wxStrdup(mapname);
	return npixmaps++;
}                       
 
int	get_carpixmap_index(const wxChar *mapname)
{
	int	i;

	for(i = 0; i < ncarpixmaps; ++i)
	    if(!wxStrcmp(mapname, carpixmaps[i].name))
		return i;
	if(ncarpixmaps >= maxcarpixmaps) {
	    maxcarpixmaps += 10;
	    if(!carpixmaps)
		carpixmaps = (pxmap *)malloc(sizeof(pxmap) * maxcarpixmaps);
	    else
		carpixmaps = (pxmap *)realloc(carpixmaps, sizeof(pxmap) * maxcarpixmaps);
	}               
	if(!(carpixmaps[ncarpixmaps].pixels = (char *)get_pixmap_file(mapname)))
	    return -1;          /* failed! file does not exist */
	carpixmaps[ncarpixmaps].name = wxStrdup(mapname);
	return ncarpixmaps++;
}                       
 
void	clean_trains(Train *sched)
{
	Train	*t;

	clean_pixmap_cache();
	while(sched) {
	    t = sched->next;
	    delete sched;
	    sched = t;
	}
}

int	trcmp(const void *a, const void *b)
{
	const Train *ap = *(const Train **)a;
	const Train *bp = *(const Train **)b;
	if(ap->timein < bp->timein)
	    return -1;
	if(ap->timein > bp->timein)
	    return 1;
	return 0;
}

Train	*sort_schedule(Train *sched)
{
	Train	**qb, *t;
	int	ntrains;
	int	l;

	for(t = sched, ntrains = 0; t; t = t->next)
	    ++ntrains;
	if(!ntrains)
	    return sched;
	qb = (Train **)malloc(sizeof(Train *) * ntrains);
	for(t = sched, l = 0; l < ntrains; ++l, t = t->next)
	    qb[l] = t;
	qsort(qb, ntrains, sizeof(Train *), trcmp);
	for(l = 0; l < ntrains - 1; ++l)
	    qb[l]->next = qb[l + 1];
	qb[ntrains - 1]->next = 0;
	t = qb[0];
	free(qb);
	return t;
}

wxChar	*convert_station(wxChar *p)
{
	return(p);
}

Train	*cancelTrain(wxChar *p, Train *sched)
{
	Train	*t, *t1;

	t1 = 0;
	for(t = sched; t && wxStrcmp(t->name, p); t = t->next)
	    t1 = t;
	if(!t)
	    return sched;
	if(t == sched)
	    sched = t->next;
	else
	    t1->next = t->next;
	free(t->name);
	free(t);
	return sched;
}

static	Train	*sched;

int	scanline(wxChar *dst, int size, wxChar **ptr)
{
	wxChar	*p = *ptr;
	int	c;

	while((c = *p++) && c != '\n') {
	    if(c == '\r')
		continue;
	    *dst++ = c;
	    if(--size < 2) {
		*dst = 0;
		break;
	    }
	}
	*ptr = p;
	return c != 0;
}

Train	*parse_newformat(TDFile& schFile)
{
	Train	*t;
	TrainStop *stp;
	wxChar	buff[1024];
	int	l;
	wxChar	*p;
	wxString	fileinc;
	wxChar	*nw, *ne;
	void	*pmw, *pme;

	t = 0;
	while(schFile.ReadLine(buff, sizeof(buff)/sizeof(wxChar))) {
	    if(!buff[0] || buff[0] == '#')
		continue;
	    if(buff[0] == '.') {
		t = 0;
		continue;
	    }
	    for(l = 0; buff[l]; ++l)
		if(buff[l] == '\t')
		    buff[l] = ' ';
	    while(l && (buff[l - 1] == ' ' || buff[l - 1] == '\t')) --l;
	    buff[l] = 0;
	    if(!wxStrncmp(buff, wxT("Include: "), 9)) {
		for(p = buff + 9; *p == ' '; ++p);
		if(!*p)
		    continue;

		TDFile	incFile(p);

		if(incFile.Load())
		    parse_newformat(incFile);
		else {
		    fileinc.Printf(wxT("%s/%s"), dirPath, locase(p));
		    TDFile  incFile1(fileinc);

		    if(!incFile1.Load())
			continue;
		    parse_newformat(incFile1);
		}
#if 0
		if(!sched) {
		    sched = t;
		    continue;
		}
		for(t1 = t; t->next; t = t->next);
		t->next = sched;
		sched = t1;
#endif
		t = 0;
		continue;
	    }
	    if(!wxStrncmp(buff, wxT("Cancel: "), 8)) {
		for(p = buff + 8; *p == ' '; ++p);
		if(!*p)
		    continue;
		sched = cancelTrain(p, sched);
		t = 0;
		continue;
	    }
	    if(!wxStrncmp(buff, wxT("Today: "), 7)) {
		for(p = buff + 7; *p == ' '; ++p);
		for(l = 0; *p >= '0' && *p <= '9'; ++p)
		    l |= 1 << (*p - '1');
		run_day = l;
		continue;
	    }
	    if(!wxStrncmp(buff, wxT("Start: "), 7)) {
		p = buff + 7;
		current_time = start_time = parse_time(&p);
		continue;
	    }
	    if(!wxStrncmp(buff, wxT("Train: "), 7)) {
		for(t = sched; t; t = t->next)
		    if(!wxStrcmp(t->name, buff + 7))
			break;
		if(t)
		    continue;
		t = new Train();
		t->name = wxStrdup(buff + 7);
		t->next = sched;
		t->type = curtype;
		t->epix = t->wpix = -1;
		t->ecarpix = t->wcarpix = -1;
		sched = t;
		continue;
	    }
	    if(!t) {
		if(!wxStrncmp(buff, wxT("Type: "), 6)) {
		    if((l = wxStrtol(buff + 6, &p, 0) - 1) >= NTTYPES || l < 0)
			continue;
		    curtype = l;
		    if(!p)
			continue;
		    while(*p == ' ' || *p == '\t') ++p;
		    if(!*p)
			continue;
		    nw = p;
		    while(*p && *p != ' ' && *p != '\t') ++p;
		    if(!*p)
			continue;
		    *p++ = 0;
		    while(*p == ' ' || *p == '\t') ++p;
		    ne = p;
		    while(*p && *p != ' ' && *p != '\t') ++p;
		    l = *p;
		    *p++ = 0;
		    if(!(pmw = get_pixmap_file(locase(nw))))
			continue;
		    if(!(pme = get_pixmap_file(locase(ne))))
			continue;
		    w_train_pmap[curtype] = pmw;
		    e_train_pmap[curtype] = pme;
		    if(!l)
			continue;
		    while(*p == ' ' || *p == '\t') ++p;
		    ne = p;
		    while(*p && *p != ' ' && *p != '\t') ++p;
		    l = *p;
		    *p++ = 0;
		    if(!(pmw = get_pixmap_file(locase(ne))))
			continue;
		    w_car_pmap[curtype] = pmw;
		    e_car_pmap[curtype] = pmw;
		    if(!l)
			continue;
		    while(*p == ' ' || *p == '\t') ++p;
		    if(!*p)
			continue;
		    if(!(pme = get_pixmap_file(locase(p))))
			continue;
		    e_car_pmap[curtype] = pme;
		}
		continue;
	    }
	    p = buff;
	    while(*p == ' ' || *p == '\t') ++p;
	    if(!wxStrncmp(p, wxT("Wait: "), 6)) {
		p += 6;
		while(*p == ' ' || *p == '\t') ++p;
		for(nw = p; *nw && *nw != ' '; ++nw);
		if(*nw)
		   *nw++ = 0;
		else
		    nw = 0;
		t->waitfor = wxStrdup(p);
		t->waittime = nw ? wxAtoi(nw) : 60;
		continue;
	    }
	    if(!wxStrncmp(p, wxT("When: "), 6)) {
		for(p += 6; *p == ' '; ++p);
		for(l = 0; *p >= '0' && *p <= '9'; ++p)
		    l |= 1 << (*p - '1');
		t->days = l;
		continue;
	    }
	    if(!wxStrncmp(p, wxT("Speed: "), 7)) {
		t->maxspeed = wxAtoi(p + 7);
		continue;
	    }
	    if(!wxStrncmp(p, wxT("Type: "), 6)) {
		if((l = wxStrtol(p + 6, &p, 0)) - 1 < NTTYPES)
		    t->type = l - 1;
		if(!p || !*p)
		    continue;
		while(*p == ' ' || *p == '\t') ++p;
		if(!*p)
		    continue;
		nw = p;      
		while(*p && *p != ' ' && *p != '\t') ++p;
		if(!*p)
		    continue;
		*p++ = 0;    
		while(*p == ' ' || *p == '\t') ++p;
		if((t->wpix = get_pixmap_index(locase(nw))) < 0)
		    continue;
		t->epix = get_pixmap_index(locase(p));
		continue;
	    }
	    if(!wxStrncmp(p, wxT("Stock: "), 7)) {
		t->stock = wxStrdup(p + 7);
		continue;
	    }
	    if(!wxStrncmp(p, wxT("Length: "), 8)) {
		t->length = wxStrtol(p + 8, &p, 0);
		t->tail = (Train *)calloc(sizeof(Train), 1);
		t->ecarpix = t->wcarpix = -1;
		while(*p == ' ' || *p == '\t') ++p;
		if(!*p)
		    continue;
		ne = p;
		while(*p && *p != ' ' && *p != '\t') ++p;
		l = *p;
		*p++ = 0;
		t->ecarpix = t->wcarpix = get_carpixmap_index(locase(ne));
		if(!l)
		    continue;
		while(*p == ' ' || *p == '\t') ++p;
		if(!*p)
		    continue;
		t->wcarpix = get_carpixmap_index(locase(p));
		continue;
	    }
	    if(!wxStrncmp(p, wxT("Enter: "), 7)) {
		p += 7;
		t->timein = parse_time(&p);
		if(*p == ',') ++p;
		while(*p == ' ' || *p == '\t') ++p;
		t->entrance = wxStrdup(convert_station(p));
		continue;
	    }
	    if(!wxStrncmp(p, wxT("Notes: "), 7)) {
		p += 7;
		if(t->nnotes < MAXNOTES)
		    t->notes[t->nnotes++] = wxStrdup(p);
		continue;
	    }
	    if(!wxStrcmp(p, wxT("Script:"))) {
		wxString scr;
		while(schFile.ReadLine(buff, sizeof(buff)/sizeof(wxChar))) {
		    if(!buff[0] || buff[0] == '#')
			continue;
		    if(!wxStrcmp(buff, wxT("EndScript"))) {
			break;
		    }
		    wxStrcat(buff, wxT("\n"));
		    scr += buff;
		}
		t->stateProgram = wxStrdup(scr.c_str());
		continue;
	    }
	    stp = (TrainStop *)malloc(sizeof(TrainStop));
	    memset(stp, 0, sizeof(TrainStop));
	    stp->minstop = 30;
	    if(*p == '-') {		/* doesn't stop */
		while(*++p == ' ' || *p == '\t');
		stp->minstop = 0;
	    } else
	        l = parse_time(&p);	/* arrival */
	    if(*p == ',') ++p;
	    while(*p == ' ' || *p == '\t') ++p;
	    if(*p == '-') {
		free(stp);
		if(t->exit)		/* already processed exit point! */
		    continue;
		t->timeout = l;
		while(*++p == ' ' || *p == '\t');
		if(*p == ',') ++p;
		while(*p == ' ' || *p == '\t') ++p;
		t->exit = wxStrdup(convert_station(p));
		continue;
	    }
	    stp->departure = parse_time(&p);
	    if(!stp->minstop)
		stp->arrival = stp->departure;
	    else {
		stp->arrival = l;
		if(stp->departure == stp->arrival)  // 
		    stp->departure = stp->arrival + stp->minstop;
		else if(stp->minstop > stp->departure - stp->arrival)	// +Rask Ingemann Lambertsen
		    stp->minstop = stp->departure - stp->arrival;	// +Rask Ingemann Lambertsen
	    }
	    if(*p == ',') ++p;
	    while(*p == ' ' || *p == '\t') ++p;
	    stp->station = wxStrdup(convert_station(p));
	    if(!t->stops)
		t->stops = stp;
	    else
		t->laststop->next = stp;
	    t->laststop = stp;
	}
	return sched;
}

void	check_delayed_entries(Train *sched)
{
	Train	*t, *t1;
	Track	*trk, *tk1;
	int	firsttime = 1;
	int	i;
	wxChar	buff[256];

	/*  Check entrance conflicts */

	for(t = sched; t; t = t->next) {
	    for(t1 = t->next; t1; t1 = t1->next) {
		if(t->timein != t1->timein)
		    continue;
		if(t->days && t1->days && run_day)
		    if(!(t->days & t1->days))
			continue;
		if(wxStrcmp(t->entrance, t1->entrance))
		    continue;
		for(trk = layout; trk; trk = trk->next)
		    if(trk->type == TRACK && trk->isstation &&
			    !wxStrcmp(t->entrance, trk->station))
			break;
		if(trk)
		    continue;
		if(firsttime) {
		    traindir->layout_error(L("These trains will be delayed on entry:"));
		    traindir->layout_error(wxT("\n"));
		}
		firsttime = 0;
		wxSnprintf(buff, sizeof(buff)/sizeof(wxChar), L("%s and %s both enter at %s on %s"),
			t->name, t1->name, t->entrance, format_time(t->timein));
		wxStrcat(buff, wxT("\n"));
		traindir->layout_error(buff);
	    }
	}
	firsttime = 1;
	for(t = sched; t; t = t->next) {
	    trk = findStationNamed(t->entrance);
	    if(!trk) {
		wxStrcpy(buff, t->entrance);
		for(i = 0; buff[i] && buff[i] != ' '; ++i);
		buff[i] = 0;
		trk = findStationNamed(buff);
	    }
	    tk1 = findStationNamed(t->exit);
	    if(!tk1) {
		wxStrcpy(buff, t->exit);
		for(i = 0; buff[i] && buff[i] != ' '; ++i);
		buff[i] = 0;
		tk1 = findStationNamed(buff);
	    }
	    if(trk && tk1)
		continue;
	    if(firsttime) {
		traindir->layout_error(L("These trains have unknown entry or exit points:"));
		traindir->layout_error(wxT("\n"));
	    }
	    firsttime = 0;
	    wxSnprintf(buff, sizeof(buff)/sizeof(wxChar), L("%s enters from '%s', exits at '%s'"), t->name,
			    t->entrance, t->exit);
	    wxStrcat(buff, wxT("\n"));
	    traindir->layout_error(buff);
	}
	traindir->end_layout_error();
}

static	Path	*find_path(const wxChar *from, const wxChar *to)
{
	Path	*pt;

	for(pt = paths; pt; pt = pt->next) {
	    if(!pt->from || !pt->to || !pt->enter)
		continue;
	    if(sameStation(from, pt->from) && sameStation(to, pt->to))
		return pt;
	}
	return 0;
}

static	void	resolve_path(Train *t)
{
	Path	*pt, *pth;
	TrainStop *ts, *tt;
	long	t0;
	int	f1;

	if(findStationNamed(t->entrance)) {
	    f1 = 1;
	    goto xit;
	}
	f1 = 0;
	for(ts = t->stops; ts; ts = ts->next) {
	    if((pt = find_path(t->entrance, ts->station))) {
		t->entrance = wxStrdup(pt->enter);
		t->timein = ts->arrival - pt->times[t->type];
		f1 = 1;
		goto xit;
	    }
	}
	for(tt = t->stops; tt; tt = tt->next) {
	    for(ts = tt->next; ts; ts = ts->next) {
		if((pt = find_path(tt->station, ts->station))) {
		    t->entrance = wxStrdup(pt->enter);
		    t->timein = ts->arrival - pt->times[t->type];
		    f1 = 1;
		    goto xit;
		}
	    }
	}

xit:
	if(findStation(t->exit)) {
	    if(f1)			/* both entrance and exit in layout */
		return;
	    pth = 0;
	    for(tt = t->stops; tt; tt = tt->next)
		if((pt = find_path(tt->station, t->exit)))
		    pth = pt;
	    if(!pth)
		pth = find_path(t->entrance, t->exit);
	    if(!pth)
		return;
	    t->entrance = wxStrdup(pth->enter);
	    t->timein = t->timeout - pth->times[t->type];
	    return;
	}
	pth = 0;
	for(tt = t->stops; tt; tt = tt->next) {
	    for(ts = tt->next; ts; ts = ts->next) {
		if((pt = find_path(tt->station, ts->station))) {
		    t0 = tt->departure;
		    pth = pt;
		}
	    }
	}
	for(ts = t->stops; ts; ts = ts->next)
	    if((pt = find_path(ts->station, t->exit))) {
		t0 = ts->departure;
		pth = pt;
	    }
	if(pth) {
	    t->exit = wxStrdup(pth->enter);
	    t->timeout = t0 + pth->times[t->type];
	    return;
	}
	if(!f1)
	    return;
	for(ts = t->stops; ts; ts = ts->next)
	    if((pt = find_path(t->entrance, ts->station))) {
		t->exit = wxStrdup(pt->enter);
		t->timeout = t->timein + pt->times[t->type];
		return;
	    }
	if((pt = find_path(t->entrance, t->exit))) {
	    t->exit = wxStrdup(pt->enter);
	    t->timeout = t->timein + pt->times[t->type];
	}
}

void	resolve_paths(Train *schedule)
{
	Train	*t;

	if(!paths)
	    return;
	for(t = schedule; t; t = t->next)
	    resolve_path(t);
}

void	load_paths(const wxChar *name)
{
	Path	*pt;
	wxChar	buff[1024];
	int	l;
	wxChar	*p, *p1;
	int	errors;

	while(paths) {
	    pt = paths->next;
	    if(paths->from) free(paths->from);
	    if(paths->to) free(paths->to);
	    if(paths->enter) free(paths->enter);
	    free(paths);
	    paths = pt;
	}

	TDFile	pthFile(name);

	pthFile.SetExt(wxT(".pth"));
	if(!pthFile.Load())
	    return;
	pt = 0;
	errors = 0;
	while(pthFile.ReadLine(buff, sizeof(buff)/sizeof(wxChar))) {
	    p = skipblk(buff);
	    if(!*p || *p == '#')
		continue;
	    if(!wxStrcmp(p, wxT("Path:"))) {
		if(pt) {			/* end previous entry */
		    if(!pt->from || !pt->to || !pt->enter) {
			++errors;
			paths = pt->next;	/* ignore last entry */
			free(pt);
		    }
		}
		p += 5;
		pt = (Path *)calloc(sizeof(Path), 1);
		pt->next = paths;
		paths = pt;
		continue;
	    }
	    if(!wxStrncmp(p, wxT("From: "), 6))
		pt->from = wxStrdup(skipblk(p + 6));
	    if(!wxStrncmp(p, wxT("To: "), 4))
		pt->to = wxStrdup(skipblk(p + 4));
	    if(!wxStrncmp(p, wxT("Times: "), 7)) {
		p += 7;
		for(p1 = p; *p1 && *p1 != ' '; ++p1);
		if(!*p1)			/* no entry point! */
		    continue;
		*p1++ = 0;
		for(l = 0; l < NTTYPES; ++l) {
		    pt->times[l] = wxStrtol(p, &p, 10) * 60;
		    if(*p == '/' || *p == ',') ++p;
		}
		p = skipblk(p1);
		pt->enter = wxStrdup(p);
	    }
	}
}

Train	*load_trains(const wxChar *name)
{
	Train	*t;
	TrainStop *stp;
	wxChar	buff[1024];
	int	l;
	wxChar	*p, *p1;
	int	newformat;

	TDFile	schFile(name);

	schFile.SetExt(wxT(".sch"));
	if(!schFile.Load())
	    return 0;

	schFile.GetDirName(dirPath, sizeof(dirPath)/sizeof(wxChar));
	sched = 0;
	newformat = 0;
	start_time = 0;
	curtype = 0;
	while(schFile.ReadLine(buff, sizeof(buff)/sizeof(wxChar))) {
	    if(!buff[0])
		continue;
	    if(newformat || !wxStrcmp(buff, wxT("#!trdir"))) {
		newformat = 1;
		t = parse_newformat(schFile);
		if(!t)
		    continue;
		sched = t;
		continue;
	    }
	    if(buff[0] == '#')
		continue;
	    t = new Train();
	    t->next = sched;
	    sched = t;
	    for(p = buff; *p && *p != ','; ++p);
	    if(!*p)
		continue;
	    *p++ = 0;
	    t->name = wxStrdup(buff);
	    t->status = train_READY;
	    t->direction = t->sdirection = (trkdir)wxStrtol(p, &p, 10);
	    if(*p == ',') ++p;
	    t->timein = parse_time(&p);
	    if(*p == ',') ++p;
	    p1 = p;
	    while(*p && *p != ',') ++p;
	    if(!*p)
		continue;
	    *p++ = 0;
	    t->entrance = wxStrdup(p1);
	    t->timeout = parse_time(&p);
	    if(*p == ',') ++p;
	    p1 = p;
	    while(*p && *p != ',') ++p;
	    if(!*p)
		continue;
	    *p++ = 0;
	    t->exit = wxStrdup(p1);
	    t->maxspeed = wxStrtol(p, &p, 10);
	    if(*p == ',') ++p;
	    while(*p) {
		for(p1 = p; *p && *p != ','; ++p);
		if(!*p)
		    continue;
		*p++ = 0;
		stp = (TrainStop *)malloc(sizeof(TrainStop));
		memset(stp, 0, sizeof(TrainStop));
		if(!t->stops)
		    t->stops = stp;
		else
		    t->laststop->next = stp;
		t->laststop = stp;
		stp->station = wxStrdup(p1);
		stp->arrival = parse_time(&p);
		if(*p == ',') ++p;
		stp->departure = parse_time(&p);
		if(*p == ',') ++p;
		stp->minstop = wxStrtol(p, &p, 10);
		if(*p == ',') ++p;
	    }
	}

	/* check correctness of schedule */

	l = 0;
	for(t = sched; t; t = t->next) {
	    if(!t->exit) {
		t->exit = wxStrdup(wxT("?"));
		++l;
	    }
	    if(!t->entrance) {
		t->entrance = wxStrdup(wxT("?"));
		++l;
	    }
	}
	if(l)
	    traindir->Error(L("Some train has unknown entry/exit point!"));
	load_paths(name);
	resolve_paths(sched);

	sched = sort_schedule(sched);

	for(t = sched; t; t = t->next)
	    if(t->stateProgram)
		t->ParseProgram();

	return sched;
}

/* ================================= */

int	save_layout(const wxChar *name, Track *layout)
{
	wxFFile file;
	Track	*t;
	TextList *tl;
	Itinerary *it;
	int	i;
	int	ch;

	if(!file_create(name, wxT(".trk"), file))
	    return 0;
	for(t = layout; t; t = t->next) {
	    switch(t->type) {
	    case TRACK:
		file.Write(wxString::Format(wxT("0,%d,%d,%d,"), t->x, t->y, t->direction));
		file.Write(wxString::Format(wxT("%d,%d,"), t->isstation, t->length));
		file.Write(wxString::Format(wxT("%d,%d,%d,%d,"), t->wlinkx, t->wlinky,
					t->elinkx, t->elinky));
		if(t->speed[0]) {
		    ch = '@';

		    for(i = 0; i < NTTYPES; ++i) {
			file.Write(wxString::Format(wxT("%c%d"), ch, t->speed[i]));
			ch = '/';
		    }
		    file.Write(wxT(','));
		}
		if(t->km)
		    file.Write(wxString::Format(wxT(">%d.%d,"), t->km / 1000, t->km % 1000));
		if(t->isstation && t->station)
		    file.Write(wxString::Format(wxT("%s\n"), t->station));
		else
		    file.Write(wxString::Format(wxT("noname\n")));
		break;

	    case SWITCH:
		file.Write(wxString::Format(wxT("1,%d,%d,%d,"), t->x, t->y, t->direction));
		file.Write(wxString::Format(wxT("%d,%d\n"), t->wlinkx, t->wlinky));
		break;

	    case TSIGNAL:
		file.Write(wxString::Format(wxT("2,%d,%d,%d,"), t->x, t->y,
			t->direction + t->fleeted * 2 +
			(t->fixedred << 8) +
			(t->nopenalty << 9) + (t->signalx << 10)));
		file.Write(wxString::Format(wxT("%d,%d"), t->wlinkx, t->wlinky));
		if(t->stateProgram) {
		    for(i = wxStrlen(t->stateProgram); i >= 0; --i)
			if(t->stateProgram[i] == '/' || t->stateProgram[i] == '\\')
			    break;
		    file.Write(wxString::Format(wxT(",@%s"), t->stateProgram + i + 1));
		}
		if(t->station && *t->station)	/* for itineraries */
		    file.Write(wxString::Format(wxT(",%s"), t->station));
		file.Write(wxString::Format(wxT("\n")));
		break;

	    case PLATFORM:
		file.Write(wxString::Format(wxT("3,%d,%d,%d\n"), t->x, t->y, t->direction == W_E ? 0 : 1));
		break;

	    case TEXT:
		file.Write(wxString::Format(wxT("4,%d,%d,%d,%s,"), t->x, t->y, t->direction, t->station));
		file.Write(wxString::Format(wxT("%d,%d,%d,%d"), t->wlinkx, t->wlinky,
						t->elinkx, t->elinky));
		if(t->km)
		    file.Write(wxString::Format(wxT(">%d.%d"), t->km / 1000, t->km % 1000));
		file.Write(wxString::Format(wxT("\n")));
		break;

	    case IMAGE:
		if(!t->station)
		    t->station = wxStrdup(wxT(""));
		for(i = wxStrlen(t->station); i >= 0; --i)
		    if(t->station[i] == '/' || t->station[i] == '\\')
			break;
		file.Write(wxString::Format(wxT("5,%d,%d,0,%s\n"), t->x, t->y, t->station + i + 1));
		break;

	    case ITIN:
		file.Write(wxString::Format(wxT("8,%d,%d,%d,%s\n"), t->x, t->y, t->direction, t->station));
		break;

	    case TRIGGER:
		file.Write(wxString::Format(wxT("9,%d,%d,%d,"), t->x, t->y, t->direction));
		file.Write(wxString::Format(wxT("%d,%d,%d,%d"), t->wlinkx, t->wlinky,
					t->elinkx, t->elinky));
		ch = ',';
		for(i = 0; i < NTTYPES; ++i) {
		    file.Write(wxString::Format(wxT("%c%d"), ch, t->speed[i]));
		    ch = '/';
		}
		file.Write(wxString::Format(wxT(",%s\n"), t->station));
		break;
	    }
	}
	for(tl = track_info; tl; tl = tl->next)
	    file.Write(wxString::Format(wxT("6,0,0,0,%s\n"), tl->txt));

	for(it = itineraries; it; it = it->next) {
	    file.Write(wxString::Format(wxT("7,0,0,0,%s,%s,%s,"), it->name,
					it->signame, it->endsig));
	    if(it->nextitin)
		file.Write(wxString::Format(wxT("@%s,"), it->nextitin));
	    for(i = 0; i < it->nsects; ++i)
		file.Write(wxString::Format(wxT("%d,%d,%d,"), it->sw[i].x, it->sw[i].y,
					it->sw[i].switched));
	    file.Write(wxString::Format(wxT("\n")));
	}

	for(t = layout; t; t = t->next) {
	    switch(t->type) {
	    case TRACK:
	    case SWITCH:
	    case ITIN:
	    case TRIGGER:

		if(t->stateProgram) {
		    file.Write(wxString::Format(wxT("(script %d,%d\n%s)\n"), t->x, t->y, t->stateProgram));
		}

	    case TSIGNAL:
		if(t->invisible) {
		    file.Write(wxString::Format(wxT("(attributes %d,%d\nhidden\n)\n"), t->x, t->y));
		}
		continue;
	    }
	}

	file.Close();
	return 1;
}

#define	MAXSHORTNAME 10

static void	short_station_name(wxChar *d, const wxChar *s)
{
	int	i;

	for(i = 0; *s && *s != ' ' && i < MAXSHORTNAME - 1; ++i)
	    *d++ = *s++;
	*d = 0;
}

#define OUT_wxHTML  0
#define OUT_TEXT    1
#define	OUT_HTML    2

void	schedule_status_print(HtmlPage& page, int outFormat)
{
	wxString	buff;
	wxString	buffs[9];
	const wxChar	*cols[9];
	Train		*t;
	TrainStop	*ts;
	const wxChar	*eol;

	if(outFormat == OUT_TEXT) {
	    *page.content = wxT("");
	    eol = wxT("\n");
	} else {
	    eol = wxT("<br>\n");
	    page.StartPage(L("Simulation results"));
	    page.AddCenter();
	    page.AddLine(wxT("<table><tr><td valign=top>"));
	}
	buff.Printf(wxT("%s : %s%s"),  L("Time"), format_time(current_time), eol);
	page.Add(buff);
	if(run_day) {
	    buff.Printf(wxT("%s : %s%s"),  L("Day"), format_day(run_day), eol);
	    page.Add(buff);
	}
	buff.Printf(wxT("%s : %ld%s"), L("Total points"), run_points, eol);
	page.Add(buff);
	buff.Printf(wxT("%s : %d%s"),  L("Total min. of delayed entry"), total_delay / 60, eol);
	page.Add(buff);
	buff.Printf(wxT("%s : %ld%s"), L("Total min. trains arrived late"), total_late, eol);
	page.Add(buff);
	if(outFormat == OUT_wxHTML) {
	    buff.Printf(wxT("<br><a href=\"save_perf_text\">%s</a>"), L("Save as text"));
	    page.AddLine(buff);
	    if(performance_hide_canceled) {
		buff.Printf(wxT("<br><a href=\"performance_toggle_canceled\">%s</a>"), L("(show canceled trains)"));
	    } else {
		buff.Printf(wxT("<br><a href=\"performance_toggle_canceled\">%s</a>"), L("(hide canceled trains)"));
	    }
	    page.AddLine(buff);
//	    buff.Printf(wxT("<br><a href=\"save_perf_HTML\">%s</a>"), L("Save as HTML"));
//	    page.AddLine(buff);
	}
///	fprintf(fp, "%s : %ld\n",     LCS("Total performance penalties"), performance());
	//fprintf(fp, "</blockquote>\n");
	if(outFormat == OUT_TEXT) {
	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n"), L("Wrong destinations"),
		perf_tot.wrong_dest, perf_vals.wrong_dest,
		perf_tot.wrong_dest * perf_vals.wrong_dest);
	    page.Add(buff);
	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n"), L("Late trains"),
		perf_tot.late_trains, perf_vals.late_trains,
		perf_tot.late_trains * perf_vals.late_trains);
	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n"), L("Wrong platforms"),
		perf_tot.wrong_platform, perf_vals.wrong_platform,
		perf_tot.wrong_platform * perf_vals.wrong_platform);
	    page.Add(buff);
	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n"), L("Commands denied"),
		perf_tot.denied, perf_vals.denied,
		perf_tot.denied * perf_vals.denied);
	    page.Add(buff);
	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n"), L("Trains waiting at signals"),
		perf_tot.waiting_train, perf_vals.waiting_train,
		perf_tot.waiting_train * perf_vals.waiting_train);

	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n"), L("Thrown switches"),
		perf_tot.thrown_switch, perf_vals.thrown_switch,
		perf_tot.thrown_switch * perf_vals.thrown_switch);
	    page.Add(buff);
	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n"), L("Cleared signals"),
		perf_tot.cleared_signal, perf_vals.cleared_signal,
		perf_tot.cleared_signal * perf_vals.cleared_signal);
	    page.Add(buff);
	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n"), L("Reversed trains"),
		perf_tot.turned_train, perf_vals.turned_train,
		perf_tot.turned_train * perf_vals.turned_train);
	    page.Add(buff);
	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n"), L("Missed station stops"),
		perf_tot.nmissed_stops, perf_vals.nmissed_stops,
		perf_tot.nmissed_stops * perf_vals.nmissed_stops);
	    page.Add(buff);
	    buff.Printf(wxT("%-40s : %5d x %3d = %-5d\n\n"), L("Wrong stock assignments"),
		perf_tot.wrong_assign, perf_vals.wrong_assign,
		perf_tot.wrong_assign * perf_vals.wrong_assign);
	    page.Add(buff);
	} else {
	    page.Add(wxT("</td><td valign=top>\n"));
	    buff.Printf(wxT("<table><tr><td valign=top>%s</td>\n"), L("Wrong destinations"));
	    page.Add(buff);
	    buff.Printf(wxT("<td align=right valign=top>%d&nbsp;x</td><td align=right valign=top>%d&nbsp;=</td>"),
			    perf_tot.wrong_dest, perf_vals.wrong_dest);
	    page.Add(buff);
	    buff.Printf(wxT("<td>%d</td></tr>\n"), perf_tot.wrong_dest * perf_vals.wrong_dest);
	    page.Add(buff);
	    buff.Printf(wxT("<tr><td valign=top>%s</td>\n"), L("Late trains"));
	    page.Add(buff);
	    buff.Printf(wxT("<td align=right valign=top>%d&nbsp;x</td><td align=right valign=top>%d&nbsp;=</td>"),
			    perf_tot.late_trains, perf_vals.late_trains);
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top>%d</td></tr>\n"), perf_tot.late_trains * perf_vals.late_trains);
	    page.Add(buff);
	    buff.Printf(wxT("<tr><td valign=top>%s</td>\n"), L("Wrong platforms"));
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top align=right>%d&nbsp;x</td><td valign=top align=right>%d&nbsp;=</td>"),
			    perf_tot.wrong_platform, perf_vals.wrong_platform);
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top>%d</td></tr>\n"), perf_tot.wrong_platform * perf_vals.wrong_platform);
	    page.Add(buff);
	    buff.Printf(wxT("<tr><td valign=top>%s</td>\n"), L("Commands denied"));
	    page.Add(buff);
	    buff.Printf(wxT("<td align=right valign=top>%d&nbsp;x</td><td align=right valign=top>%d&nbsp;=</td>"),
			    perf_tot.denied, perf_vals.denied);
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top>%d</td></tr>\n"), perf_tot.denied * perf_vals.denied);
	    page.Add(buff);
	    buff.Printf(wxT("<tr><td valign=top>%s</td>\n"), L("Trains waiting at signals"));
	    page.Add(buff);
	    buff.Printf(wxT("<td align=right valign=top>%d&nbsp;x</td><td align=right valign=top>%d&nbsp;=</td>"),
			    perf_tot.waiting_train, perf_vals.waiting_train);
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top>%d</td></tr>\n"), perf_tot.waiting_train * perf_vals.waiting_train);
	    page.Add(buff);

	    buff.Printf(wxT("</table></td><td valign=top><table>"));
	    page.Add(buff);
	    buff.Printf(wxT("<tr><td valign=top>%s</td>\n"), L("Thrown switches"));
	    page.Add(buff);
	    buff.Printf(wxT("<td align=right valign=top>%d&nbsp;x</td><td align=right valign=top>%d&nbsp;=</td>"),
			    perf_tot.thrown_switch, perf_vals.thrown_switch);
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top>%d</td></tr>\n"), perf_tot.thrown_switch * perf_vals.thrown_switch);
	    page.Add(buff);
	    buff.Printf(wxT("<tr><td valign=top>%s</td>\n"), L("Cleared signals"));
	    page.Add(buff);
	    buff.Printf(wxT("<td align=right valign=top>%d&nbsp;x</td><td align=right valign=top>%d&nbsp;=</td>"),
			    perf_tot.cleared_signal, perf_vals.cleared_signal);
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top>%d</td></tr>\n"), perf_tot.cleared_signal * perf_vals.cleared_signal);
	    page.Add(buff);
	    buff.Printf(wxT("<tr><td valign=top>%s</td>\n"), L("Reversed trains"));
	    page.Add(buff);
	    buff.Printf(wxT("<td align=right valign=top>%d&nbsp;x</td><td align=right valign=top>%d&nbsp;=</td>"),
			    perf_tot.turned_train, perf_vals.turned_train);
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top>%d</td></tr>\n"), perf_tot.turned_train * perf_vals.turned_train);
	    page.Add(buff);
	    buff.Printf(wxT("<tr><td valign=top>%s</td>\n"), L("Missed station stops"));
	    page.Add(buff);
	    buff.Printf(wxT("<td align=right valign=top>%d&nbsp;x</td><td align=right valign=top>%d&nbsp;=</td>"),
			    perf_tot.nmissed_stops, perf_vals.nmissed_stops);
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top>%d</td></tr>\n"), perf_tot.nmissed_stops * perf_vals.nmissed_stops);
	    page.Add(buff);
	    buff.Printf(wxT("<tr><td valign=top>%s</td>\n"), L("Wrong stock assignments"));
	    page.Add(buff);
	    buff.Printf(wxT("<td align=right valign=top>%d&nbsp;x</td><td align=right valign=top>%d&nbsp;=</td>"),
			    perf_tot.wrong_assign, perf_vals.wrong_assign);
	    page.Add(buff);
	    buff.Printf(wxT("<td valign=top>%d</td></tr>\n"), perf_tot.wrong_assign * perf_vals.wrong_assign);
	    page.Add(buff);
	    page.Add(wxT("</table></td></tr></table>"));
	    page.AddLine(wxT("</center>"));
	}

	cols[0] = L("Train");
	cols[1] = L("Enters");
	cols[2] = L("At");
	cols[3] = L("Exits");
	cols[4] = L("Before");
	cols[5] = L("Delay");
	cols[6] = L("Late");
	cols[7] = L("Status");
	cols[8] = 0;
	if(outFormat == OUT_TEXT) {
	} else
	    page.StartTable(&cols[0]);

	cols[0] = buffs[0];
	cols[1] = buffs[1];
	cols[2] = buffs[2];
	cols[3] = buffs[3];
	cols[4] = buffs[4];
	cols[5] = buffs[5];
	cols[6] = buffs[6];
	cols[7] = buffs[7];
	cols[8] = 0;
	for(t = schedule; t; t = t->next) {
	    if(performance_hide_canceled && is_canceled(t))
		continue;
	    print_train_info(t);
	    buffs[0].Printf(wxT("<a href=\"traininfopage %s\">%s</a>"), t->name, t->name);
	    cols[0] = buffs[0];
	    buffs[1] = t->entrance;
	    cols[1] = buffs[1];
	    cols[2] = entering_time;
	    buffs[3] = t->exit;
	    cols[3] = buffs[3];
	    cols[4] = leaving_time;
	    cols[5] = current_delay;
	    cols[6] = current_late;
	    cols[7] = current_status;
	    if(outFormat == OUT_TEXT) {
		buff.Printf(wxT("%-10.10s  %-20.20s  %12s  %-20.20s  %12s %3s %3s %s\n"),
		    t->name, cols[1], cols[2], cols[3], cols[4], cols[5], cols[6], cols[7]);
		page.Add(buff);
	    } else
		page.AddTableRow(cols);
	    cols[0] = wxT("&nbsp;");
	    cols[1] = wxT("&nbsp;");
	    cols[2] = wxT("&nbsp;");
	    cols[3] = wxT("&nbsp;");
	    cols[4] = wxT("&nbsp;");
	    cols[5] = wxT("&nbsp;");
	    buff = wxT("");
	    for(ts = t->stops; ts; ts = ts->next) {
		if(outFormat == OUT_TEXT && t->status != train_READY) {
		    wxString s;

		    s.Printf(wxT("   %d: %s"), ts->delay, ts->station);
		    if(buff.length() + s.length() > 93) {
			page.Add(buff);
			page.Add(wxT("\n"));
			buff = wxT("");
		    }
		    buff += s;
		    continue;
		}
		if(!ts->delay)
		    continue;
		buffs[6].Printf(wxT("%c%d"), ts->delay > 0 ? '+' : ' ', ts->delay);
		cols[6] = buffs[6];
		cols[7] = ts->station;
		page.AddTableRow(cols);
	    }
	    if(buff.length() > 0) {
		page.Add(buff);
		page.Add(wxT("\n\n"));
	    }
	}
	if(outFormat != OUT_TEXT) {
	    page.EndTable();
	    page.EndPage();
	}
}

void	show_schedule_status(HtmlPage& dest)
{
	schedule_status_print(dest, OUT_wxHTML);
}

void	save_schedule_status(HtmlPage& dest)
{
	schedule_status_print(dest, OUT_TEXT);
}

void	performance_toggle_canceled()
{
	performance_hide_canceled = !performance_hide_canceled;
}

void	train_print(Train *t, HtmlPage& page)
{
	TrainStop	*ts;
	wxString	buff;
	int		i;
	const wxChar	*beg, *end;
	int		status;
	wxString	buffs[7];
	const wxChar	*cols[7];

	buff.Printf(wxT("%s %s"), L("Train"), t->name);
	page.StartPage(buff);
	cols[0] = L("Station");
	cols[1] = L("Arrival");
	cols[2] = L("Departure");
	cols[3] = L("Min.stop");
	cols[4] = 0 /*"Stopped";
	cols[5] = "Delay";
	cols[6] = 0*/;
	page.StartTable(cols);
	cols[0] = buffs[0];
	cols[1] = buffs[1];
	cols[2] = buffs[2];
	cols[3] = buffs[3];
	cols[4] = 0 /*buffs[4];
	cols[5] = buffs[5];
	cols[6] = 0 */;

	status = 0;
	beg = wxT(""), end = wxT("");
	for(ts = t->stops; ts; ts = ts->next) {
	    if(ts->arrival >= t->timein/* && findStation(ts->station)*/) {
		if(status == 0) {
		    buffs[0].Printf(wxT("<b><a href=\"stationinfopage %s\">%s</a></b>"), t->entrance, t->entrance), cols[0] = buffs[0];
		    buffs[1].Printf(wxT("&nbsp;")), cols[1] = buffs[1];
		    buffs[2].Printf(wxT("<b>%s</b>"), format_time(t->timein)), cols[2] = buffs[2]; 
		    buffs[3].Printf(wxT("&nbsp;")), cols[3] = buffs[3];
		    cols[4] = 0;
		    page.AddTableRow(cols);
		    status = 1;
		}
	    }
	    if(ts->arrival > t->timeout && status == 1) {
		buffs[0].Printf(wxT("<b><a href=\"stationinfopage %s\">%s</a></b>"), t->exit, t->exit), cols[0] = buffs[0];
		buffs[1].Printf(wxT("<b>%s</b>"), format_time(t->timeout)), cols[1] = buffs[1]; 
		buffs[2].Printf(wxT("&nbsp;")), cols[2] = buffs[2];
		buffs[3].Printf(wxT("&nbsp;")), cols[3] = buffs[3];
		cols[4] = 0;
		page.AddTableRow(cols);
		status = 2;
	    }
	    buffs[0] = wxString(ts->station).BeforeFirst(wxT('@'));
	    if(findStationNamed(buffs[0]))
		beg = wxT("<b>"), end = wxT("</b>");
	    else
		beg = wxT(""), end = wxT("");
	    buffs[0].Printf(wxT("%s<a href=\"stationinfopage %s\">%s</a>%s"), beg, ts->station, ts->station, end), cols[0] = buffs[0];
	    if(!ts->arrival || !ts->minstop)
		cols[1] = wxT("&nbsp;");
	    else
		buffs[1].Printf(wxT("%s%s%s"), beg, format_time(ts->arrival), end), cols[1] = buffs[1];
	    buffs[2].Printf(wxT("%s%s%s"), beg, format_time(ts->departure), end), cols[2] = buffs[2];
	    if(status != 1)
		cols[3] = wxT("&nbsp;");
	    else
	        buffs[3].Printf(wxT("%ld"), ts->minstop), cols[3] = buffs[3];
/*	    sprintf(cols[4], ts->stopped ? "<b>Yes</b>" : "<b>No</b>");
	    sprintf(cols[5], "%s%ld%s", beg, (long)ts->delay, end);
*/	    cols[4] = 0;
	    page.AddTableRow(cols);
	}
	if(status < 1) {
	    buffs[0].Printf(wxT("<b><a href=\"stationinfopage %s\">%s</a></b>"), t->entrance, t->entrance), cols[0] = buffs[0];
	    buffs[1].Printf(wxT("&nbsp;")), cols[1] = buffs[1];
	    buffs[2].Printf(wxT("<b>%s</b>"), format_time(t->timein)), cols[2] = buffs[2]; 
	    buffs[3].Printf(wxT("&nbsp;")), cols[3] = buffs[3];
	    cols[4] = 0;
	    page.AddTableRow(cols);
	    ++status;
	}
	if(status < 2) {
	    buffs[0].Printf(wxT("<b><a href=\"stationinfopage %s\">%s</a></b>"), t->exit, t->exit), cols[0] = buffs[0];
	    buffs[1].Printf(wxT("<b>%s</b>"), format_time(t->timeout)), cols[1] = buffs[1]; 
	    buffs[2].Printf(wxT("&nbsp;")), cols[2] = buffs[2];
	    buffs[3].Printf(wxT("&nbsp;")), cols[3] = buffs[3];
	    cols[4] = 0;
	    page.AddTableRow(cols);
	}
	page.EndTable();
	page.Add(wxT("<blockquote><blockquote>\n"));
	if(t->days) {
	    buff.Printf(wxT("%s : "), L("Runs on"));
	    for(i = 0; i < 7; ++i)
		if(t->days & (1 << i))
		    buff += wxString::Format(wxT("%d"), i+1);
	    page.AddLine(buff);
	}
	if(t->nnotes) {
	    buff.Printf(wxT("%s: "), L("Notes"));
	    page.Add(buff);
	    for(status = 0; status < t->nnotes; ++status) {
		buff.Printf(wxT("%s.<br>\n"), t->notes[status]);
		page.Add(buff);
	    }
	}
	page.AddLine(wxT("</blockquote></blockquote>"));
	page.EndPage();
}

int	save_game(const wxChar *name)
{
	wxFFile file;
	Track	*t;
	Train	*tr;
	TrainStop *ts;
	int	i;

	if(!file_create(name, wxT(".sav"), file))
	    return 0;
	file.Write(wxString::Format(wxT("%s\n"), current_project.c_str()));

	file.Write(wxString::Format(wxT("%d,%ld,%d,%d,%d,%d,%d,%d,%d,%ld\n"),
		cur_time_mult, start_time, show_speeds,
		show_blocks, beep_on_alert, run_points, total_delay,
		total_late, time_mult, current_time));

	/* Save the state of every switch */

	for(t = layout; t; t = t->next) {
	    if(t->type != SWITCH || !t->switched)
		continue;
	    file.Write(wxString::Format(wxT("%d,%d,%d\n"), t->x, t->y, t->switched));
	}
	file.Write(wxString::Format(wxT("\n")));

	/* Save the state of every signal */

	for(t = layout; t; t = t->next) {
	    if(t->type != TSIGNAL)
		continue;
	    Signal *sig = (Signal *)t;
	    if(!sig->IsClear() && !t->nowfleeted)
		continue;
	    file.Write(wxString::Format(wxT("%d,%d,%d,%d"), t->x, t->y,
			sig->IsClear(), t->nowfleeted != 0));
	    if(sig->_interpreterData)
		file.Write(wxString::Format(wxT(",%s"), sig->_currentState));
	    file.Write(wxString::Format(wxT("\n")));
	}
	file.Write(wxString::Format(wxT("\n")));

	/* Save the position of every train */

	for(tr = schedule; tr; tr = tr->next) {
	    if(tr->status == train_READY)
		continue;
	    file.Write(wxString::Format(wxT("%s\n"), tr->name));
	    file.Write(wxString::Format(wxT("  %d,%d,%s\n"), tr->status, tr->direction,
				tr->exited ? tr->exited : wxT("")));
	    file.Write(wxString::Format(wxT("  %d,%d,%d,%d,%d,%d,%d,%d,%d\n"), tr->timeexited,
				tr->wrongdest, tr->curspeed, tr->maxspeed,
				tr->curmaxspeed, tr->trackpos, tr->timelate,
				tr->timedelay, tr->timered));
	    file.Write(wxString::Format(wxT("  %ld,%d,%ld,%ld,%d\n"), tr->timedep, tr->pathpos,
				tr->pathtravelled, tr->disttostop, tr->shunting));
	    if(!tr->stoppoint)
		file.Write(wxString::Format(wxT("  0,0,0,")));
	    else
		file.Write(wxString::Format(wxT("  %d,%d,%ld,"), tr->stoppoint->x, tr->stoppoint->y,
				tr->disttoslow));
	    if(!tr->slowpoint)
		file.Write(wxString::Format(wxT("0,0")));
	    else
		file.Write(wxString::Format(wxT("%d,%d"), tr->slowpoint->x, tr->slowpoint->y));
	    file.Write(wxString::Format(wxT(",%d\n"), tr->needfindstop));
	    if(tr->fleet && tr->fleet->size) {
		/* file.Write(wxString::Format("  %d,%d\n", tr->fleet->x, tr->fleet->y)); */
		file.Write(wxT("  "));
		for(i = 0; i < tr->fleet->size; ++i) {
		    t = (Track *)Vector_elementAt(tr->fleet, i);
		    if(i)
			file.Write(wxT(','));
		    file.Write(wxString::Format(wxT("%d,%d"), t->x, t->y));
		}
		file.Write(wxT('\n'));
	    } else
		file.Write(wxString::Format(wxT("  0,0\n")));	    /* length has fleet info at end */
	    if(tr->position)
		file.Write(wxString::Format(wxT("  %d,%d"), tr->position->x, tr->position->y));
	    else
		file.Write(wxString::Format(wxT("  0,0")));
	    file.Write(wxString::Format(wxT(",%d\n"), tr->waittime));
	    file.Write(wxString::Format(wxT("  %d,%s\n"), tr->oldstatus,
				tr->outof ? tr->outof->station : wxT("")));

	    /* Save status of each stop */

	    for(ts = tr->stops; ts; ts = ts->next)
		if(ts->stopped || ts->delay)
		    file.Write(wxString::Format(wxT("    %s,%d,%d\n"), ts->station, ts->stopped, ts->delay));
	    if(tr->tail && tr->tail->path) {
		Train *tail = tr->tail;

		file.Write(wxString::Format(wxT(".\n")));	/* marks beginning of tail path */
		file.Write(wxString::Format(wxT("  %s\n"), tr->stopping ? tr->stopping->station : wxT("")));
		if(tail->fleet && tail->fleet->size) {
		    for(i = 0; i < tail->fleet->size; ++i) {
			t = (Track *)Vector_elementAt(tail->fleet, i);
			file.Write(wxString::Format(wxT("%c%d,%d"), i ? ',' : '!', t->x, t->y));
		    }
		    file.Write(wxT('\n'));
		}
		file.Write(wxString::Format(wxT("  %d,%d,%d,%d"), !tail->position ? -1 : tail->pathpos,
				tail->trackpos, tail->tailentry, tail->tailexit));
		for(i = 0; i < tail->path->size; ++i) {
		    t = (Track *)Vector_elementAt(tail->path, i);
		    file.Write(wxString::Format(wxT(",%d,%d,%d"), t->x, t->y, Vector_flagAt(tail->path, i)));
		}
	    }
	    file.Write(wxT('\n'));
	}
	file.Write(wxT(".\n"));

	/* save white tracks (to allow merging trains) */
	for(t = layout; t; t = t->next)
	    if((t->type == TRACK || t->type == SWITCH) && t->fgcolor == color_white)
		break;

	if(t) {	    /* we found a white track */
	    file.Write(wxT("(white tracks\n"));
	    for(t = layout; t; t = t->next) {
		if((t->type == TRACK || t->type == SWITCH) && t->fgcolor == color_white)
		    file.Write(wxString::Format(wxT("%d,%d\n"), t->x, t->y));
	    }
	    file.Write (wxT(")\n"));
	}
	
	/* Save the position of every stranded train */

	for(tr = stranded; tr; tr = tr->next) {
	    file.Write (wxT("(stranded\n"));
	    file.Write(wxString::Format(wxT("%d,%d,%d,%d,%d,%d,%d,%d,%d"),
		tr->type, tr->position->x, tr->position->y,
		tr->direction, tr->ecarpix, tr->wcarpix,
		tr->maxspeed, tr->curmaxspeed,
		tr->length));
	    if(tr->length) {
		if(tr->tail && tr->tail->path) {
		    int	f;
		    const wxChar *sep = wxT("");

		    file.Write(wxString::Format(wxT(",%d\n"), tr->tail->path->size));
		    for(i = 0; i < tr->tail->path->size; ++i) {
			t = Vector_elementAt(tr->tail->path, i);
			f = Vector_flagAt(tr->tail->path, i);
			file.Write(wxString::Format(wxT("%s%d,%d,%d"), sep, t->x, t->y, f));
			sep = wxT(",");
		    }
		    file.Write(wxT('\n'));
		} else
		    file.Write(wxT(",0\n"));
	    } else
		file.Write(wxT('\n'));
	    file.Write(wxT(")\n"));
	}

	int	m;
	file.Write(wxT("(late minutes\n"));
	for(i = m = 0; i < 24 * 60; ++i) {
	    file.Write(wxString::Format(wxT(" %d"), late_data[i]));
	    if(++m == 15) {	// 15 values per line
		file.Write(wxT("\n"));
		m = 0;
	    }
	}
	file.Write (wxT(")\n"));

	file.Write(wxString::Format(wxT("%d,%d,%d,%d,%d\n"), run_day, terse_status, status_on_top,
		show_seconds, signal_traditional));
	file.Write(wxString::Format(wxT("%d,%d\n"), auto_link, show_grid));
	file.Write(wxString::Format(wxT("%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d\n"),
	    perf_tot.wrong_dest, perf_tot.late_trains, perf_tot.thrown_switch,
	    perf_tot.cleared_signal, perf_tot.denied, perf_tot.turned_train,
	    perf_tot.waiting_train, perf_tot.wrong_platform,
	    perf_tot.ntrains_late, perf_tot.ntrains_wrong,
	    perf_tot.nmissed_stops, perf_tot.wrong_assign));
	file.Write(wxString::Format(wxT("%d\n"), hard_counters));
	file.Write(wxString::Format(wxT("%d\n"), show_canceled));
	file.Write(wxString::Format(wxT("%d\n"), show_links));
	file.Write(wxString::Format(wxT("%d\n"), beep_on_enter));
	file.Write(wxString::Format(wxT("%d\n"), bShowCoord));
	file.Write(wxString::Format(wxT("%d\n"), show_icons));
	file.Write(wxString::Format(wxT("%d\n"), show_tooltip));

	file.Close();
	return 1;
}

void	restore_game(const wxChar *name)
{
	FILE		*fp1;
	const wxChar	*buffptr;
	wxChar		buff[1024];
	wxChar		*p;
	int	x, y;
	Track	*t;
	Train	*tr, *tail;
	TrainStop *ts;

	wxSnprintf(buff, sizeof(buff)/sizeof(wxChar), wxT("%s.sav"), name);
	TDFile fp(buff);
	if(!(fp.Load())) {
#if wxUSE_UNICODE
	    perror(wxSafeConvertWX2MB(buff));	/* No wxPerror()? */
#else
	    perror(buff);
#endif
	    return;
	}
	wxStrcpy(buff, wxT("load "));
	buffptr = getline(&fp);
	wxStrncat(buff, buffptr, sizeof(buff)/sizeof(wxChar) - 6);
	if(wxStrstr(buff, wxT(".zip")) || wxStrstr(buff, wxT(".ZIP")))
	    ;
	else if(!wxStrstr(buff, wxT(".trk")) && !wxStrstr(buff, wxT(".TRK")))
	    wxStrcat(buff, wxT(".trk"));
	if(!(fp1 = wxFopen(buff + 5, wxT("r")))) {		// path not there
	    p = (wxChar *) (buffptr + wxStrlen(buffptr));	// try isolating only the file name
	    while(--p > buffptr && *p != '\\' && *p != '/' && *p != ':');
	    if(p > buffptr)
		wxStrcpy(buff + 5, p + 1);
	} else
	    fclose(fp1);
	// trainsim_cmd(buff);
	traindir->OpenFile(buff + 5, true);

	buffptr = getline(&fp);
	wxSscanf(buffptr, wxT("%d,%ld,%d,%d,%d,%d,%d,%d,%d,%ld"),
		&cur_time_mult, &start_time, &show_speeds,
		&show_blocks, &beep_on_alert, &run_points, &total_delay,
		&total_late, &time_mult, &current_time);

	/* reload state of all switches */

	while((buffptr = getline(&fp))) {
	    if(!buffptr[0])
		break;
	    x = wxStrtol(buffptr, &p, 0);
	    if(*p == ',') ++p;
	    y = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    if(!(t = findSwitch(x, y)))
		continue;
	    t->switched = wxAtoi(p);
	    if(t->switched)
		change_coord(t->x, t->y);
	}

	/* reload state of all signals */

	while((buffptr = getline(&fp))) {
	    Signal  *sig;

	    if(!buffptr[0])
		break;
	    x = wxStrtol(buffptr, &p, 0);
	    if(*p == ',') ++p;
	    y = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    if(!(sig = findSignal(x, y)))
		continue;
	    sig->status = wxStrtol(p, &p, 0) == 1 ? ST_GREEN : ST_RED;
	    if(*p == ',') ++p;
	    sig->nowfleeted = wxStrtol(p, &p, 0);

	    if(*p == ',') ++p;
	    if(*p)
		sig->_currentState = wxStrdup(p); // this is a small memory leak!
	    if(!sig->IsApproach() && sig->IsClear())
		signal_unlock(sig);
	    change_coord(sig->x, sig->y);
	}

	/* reload state of all trains */

	while((buffptr = getline(&fp))) {
	    if(!buffptr[0] || buffptr[0] == '.')
		break;			/* end of file */
	    tr = findTrainNamed(buffptr);
	    if(!tr) {
		/* the train could not be found in the schedule.
		 * Warn the user, and ignore all lines up to the
		 * next empty line.
		 */
		do {
		    buffptr = getline(&fp);
		} while(buffptr && buffptr[0] && buffptr[0] != '.');
		continue;
	    }
	    /* second line */
	    buffptr = getline(&fp);
	    for(p = (wxChar *) buffptr; *p == ' '; ++p);
	    tr->status = (trainstat)wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->direction = (trkdir)wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    if(*p)
		tr->exited = wxStrdup(p);

	    /* third line */
	    buffptr = getline(&fp);
	    for(p = (wxChar *) buffptr; *p == ' '; ++p);
	    tr->timeexited = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->wrongdest = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->curspeed = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->maxspeed = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->curmaxspeed = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->trackpos = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->timelate = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->timedelay = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->timered = wxStrtol(p, &p, 0);

	    /* fourth line */
	    buffptr = getline(&fp);
	    for(p = (wxChar *) buffptr; *p == ' '; ++p);
	    tr->timedep = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->pathpos = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->pathtravelled = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    tr->disttostop = wxStrtol(p, &p, 0);
	    if(*p == ',') {
		++p;
		tr->shunting = wxStrtol(p, &p, 0);
	    }

	    /* fifth line */
	    buffptr = getline(&fp);
	    for(p = (wxChar *) buffptr; *p == ' '; ++p);
	    x = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    y = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    if(!(tr->stoppoint = findTrack(x, y)))
		tr->stoppoint = findSwitch(x, y);
	    tr->disttoslow = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    x = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    y = wxStrtol(p, &p, 0);
	    if(!(tr->slowpoint = findTrack(x, y)))
		tr->slowpoint = findSwitch(x, y);
	    if(*p == ',') {
		++p;
		tr->needfindstop = wxStrtol(p, &p, 0);
	    }

	    /* sixth line */
	    buffptr = getline(&fp);
	    for(p = (wxChar *) buffptr; *p == ' '; ++p);
	    while(*p) {		/* list of fleeting signals */
	      x = wxStrtol(p, &p, 0);
	      if(*p == ',') ++p;
	      y = wxStrtol(p, &p, 0);
/*	    tr->fleet = findSignal(x, y);	*/
	      if(x && y) {
		if(!tr->fleet)
		  tr->fleet = new_Vector();
		Vector_addElement(tr->fleet, (Track *)findSignal(x, y), 0); // TODO
	      }
	    }

	    /* seventh line */
	    buffptr = getline(&fp);
	    for(p = (wxChar *) buffptr; *p == ' '; ++p);
	    x = wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    y = wxStrtol(p, &p, 0);
	    if(!(tr->position = findTrack(x, y)))
		if(!(tr->position = findSwitch(x, y))) {
		    switch(tr->status) {
		    case train_READY:
		    case train_ARRIVED:
		    case train_DERAILED:
		    case train_DELAY:
			break;

		    case train_WAITING:
		    case train_STOPPED:
		    case train_RUNNING:
			do_alert(wxT("Train is not on track!"));
			new_train_status(tr, train_DERAILED);
		    }
		}
	    if(*p == ',') ++p;
	    tr->waittime = wxStrtol(p, &p, 0);

	    /* reset paths!!! */
	    if(tr->position) {
	        tr->path = findPath(tr->position, tr->direction);
	        tr->pathpos = 1;
		if(tr->path)
		    colorPath(tr->path, ST_GREEN);
	        tr->position->fgcolor = conf.fgcolor;
	    }

	    /* eighth line */
	    buffptr = getline(&fp);
	    for(p = (wxChar *) buffptr; *p == ' '; ++p);
	    tr->oldstatus = (trainstat)wxStrtol(p, &p, 0);
	    if(*p == ',') ++p;
	    if(*p)
		tr->outof = findStation(p);

	    /* nineth line */
	    while((buffptr = getline(&fp))) {
		if(!buffptr[0] || buffptr[0] == '.')
		    break;
		if(!(p = wxStrchr(buffptr, ',')))
		    continue;
		*p++ = 0;
		for(ts = tr->stops; ts; ts = ts->next)
		    if(!wxStrcmp(ts->station, buffptr + 4))
			break;
		if(!ts)
		    continue;
		ts->stopped = wxStrtol(p, &p, 0);
		if(*p == ',') ++p;
		ts->delay = wxAtoi(p);
	    }
	    if(!buffptr)
		break;
	    if(buffptr[0] == '.') {	/* tail path info present */
		buffptr = getline(&fp);
		if(!buffptr[0] || buffptr[0] == '.')
		    break;
		if(!(tail = tr->tail))	/* maybe length was removed in .sch */
		    continue;
		for(p = (wxChar *) buffptr; *p == ' '; ++p);
		if(*p)			/* stopping at station name present */
		    tr->stopping = findStation(p);
		buffptr = getline(&fp);
		for(p = (wxChar *) buffptr; *p == ' '; ++p);
		if(*p == '!') {
		    ++p;
		    while(*p) {		/* list of fleeting signals */
		      x = wxStrtol(p, &p, 0);
		      if(*p == ',') ++p;
		      y = wxStrtol(p, &p, 0);
		      if(x && y) {
			if(!tail->fleet)
			  tail->fleet = new_Vector();
			Vector_addElement(tail->fleet, (Track *)findSignal(x, y), 0);	// TODO
		      }
		      if(*p == ',') ++p;
		    }
		    buffptr = getline(&fp);
		    for(p = (wxChar *) buffptr; *p == ' '; ++p);
		}
		tail->pathpos = wxStrtol(p, &p, 0);
		if(*p == ',') ++p;
		tail->trackpos = wxStrtol(p, &p, 0);
		if(*p == ',') ++p;
		tail->tailentry = wxStrtol(p, &p, 0);
		if(*p == ',') ++p;
		tail->tailexit = wxStrtol(p, &p, 0);
		while(*p == ',') {
		    x = wxStrtol(p + 1, &p, 0);
		    if(*p == ',') ++p;
		    y = wxStrtol(p, &p, 0);
		    if(!tail->path)
			tail->path = new_Vector();
		    if(!(t = findTrack(x, y)))
			if(!(t = findSwitch(x, y)))
			    t = findText(x, y);
		    if(!t) {		/* maybe layout changed? */
			if(tail->path)	/* disable length for this train */
			    Vector_delete(tail->path);
			tail->path = 0;
			tr->tail = 0;
			tr->length = 0;
			break;
		    }
		    if(*p == ',') ++p;
		    Vector_addElement(tail->path, t, wxStrtol(p, &p, 0));
		}
		if(tail->path) {
		    tail->position = 0;
		    if(tr->status == train_ARRIVED) {
			Vector_delete(tail->path);
			tail->path = 0;
		    } else {
			colorPartialPath(tail->path, ST_RED, tail->pathpos + 1);
			if(tr->path) {
			    colorPath(tr->path, ST_GREEN);
			    tr->position->fgcolor = conf.fgcolor;
			}
			if(tail->pathpos >= 0 && tail->pathpos < tail->path->size)
			    tail->position = (Track *)Vector_elementAt(tail->path, tail->pathpos);
			else
			    tail->pathpos = 0;
		    }
		}
	    }
	    update_schedule(tr);
	}
	
	while((buffptr = getline(&fp)) && buffptr[0] == '(') {
	    if(!wxStrncmp(buffptr, wxT("(white tracks"), 13)) {
		while((buffptr = getline(&fp)) && buffptr[0] != ')') {
		    wxSscanf(buffptr, wxT("%d,%d"), &x, &y);
		    if(!(t = findTrack(x, y)))
			t = findSwitch(x, y);
		    if(t) {
			t->fgcolor = color_white;
			change_coord(x, y);
		    }
		}
		continue;
	    }
	    if(!wxStrncmp(buffptr, wxT("(stranded"), 9)) {
		while((buffptr = getline(&fp)) && buffptr[0] != ')') {
		    Train *stk = new Train();
		    stk->next = stranded;
		    stranded = stk;
		    stk->name = wxStrdup(wxT(""));
		    p = (wxChar *) buffptr;
		    stk->type = wxStrtoul(p, &p, 0);
		    if(*p == ',') ++p;
		    x = wxStrtoul(p, &p, 0);
		    if(*p == ',') ++p;
		    y = wxStrtoul(p, &p, 0);
		    stk->position = findTrack(x, y);
		    if(!stk->position)
			stk->position = findSwitch(x, y);
		    stk->flags = TFLG_STRANDED;
		    stk->status = train_ARRIVED;
		    if(*p == ',') ++p;
		    stk->direction = (trkdir)wxStrtoul(p, &p, 0);
		    if(*p == ',') ++p;
		    stk->ecarpix = wxStrtoul(p, &p, 0);
		    if(*p == ',') ++p;
		    stk->wcarpix = wxStrtoul(p, &p, 0);
		    if(*p == ',') ++p;
		    stk->maxspeed = wxStrtoul(p, &p, 0);
		    if(*p == ',') ++p;
		    stk->curmaxspeed = wxStrtoul(p, &p, 0);
		    if(*p == ',') ++p;
		    stk->length = wxStrtoul(p, &p, 0);
		    if(stk->length) {
			int	tailLength, l, f;
			int	pathLength = 0;

			if(*p == ',') ++p;
			tailLength = wxStrtoul(p, &p, 0);
			if(tailLength) {
			    if(!(buffptr = getline(&fp)))
				break;
			    p = (wxChar *) buffptr;
			    stk->tail = new Train();
			    stk->tail->path = new_Vector();
			    for(l = 0; l < tailLength; ++l) {
				if(*p == ',') ++p;
				x = wxStrtoul(p, &p, 0);
				if(*p == ',') ++p;
				y = wxStrtoul(p, &p, 0);
				if(*p == ',') ++p;
				f = wxStrtoul(p, &p, 0);
				if(!(t = findTrack(x, y)))
				    t = findSwitch(x, y);
				if(!t)
				    break;
				change_coord(x, y);

				Vector_addElement(stk->tail->path, t, f);
				pathLength += t->length;
			    }
			    stk->tail->length = pathLength;
			    stk->tail->position = Vector_elementAt(stk->tail->path, 0);
			}
		    }
		    stk->position->fgcolor = color_black;
		}
		continue;
	    }
	    if(!wxStrncmp(buffptr, wxT("(late minutes"), 13)) {
		x = 0;
		while((buffptr = getline(&fp)) && buffptr[0] != ')') {
		    p = (wxChar *)buffptr;
		    while(*p) {
			late_data[x % (24 * 60)] = wxStrtoul(p, &p, 0);
			++x;
		    }
		}
		continue;
	    }
	}
	if(buffptr) {
	    wxSscanf(buffptr, wxT("%d,%d,%d,%d,%d"), &run_day, &terse_status, &status_on_top,
		&show_seconds, &signal_traditional);
	}

	if((buffptr = getline(&fp)))
	    wxSscanf(buffptr, wxT("%d,%d"), &auto_link, &show_grid);
	memset(&perf_tot, 0, sizeof(perf_tot));
	if((buffptr = getline(&fp)))
	    wxSscanf(buffptr, wxT("%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d"),
	    &perf_tot.wrong_dest, &perf_tot.late_trains, &perf_tot.thrown_switch,
	    &perf_tot.cleared_signal, &perf_tot.denied, &perf_tot.turned_train,
	    &perf_tot.waiting_train, &perf_tot.wrong_platform,
	    &perf_tot.ntrains_late, &perf_tot.ntrains_wrong,
	    &perf_tot.nmissed_stops, &perf_tot.wrong_assign);
	if((buffptr = getline(&fp)) && *buffptr)
	    hard_counters = wxAtoi(buffptr);
	if((buffptr = getline(&fp)) && *buffptr)
	    show_canceled = wxAtoi(buffptr);
	if((buffptr = getline(&fp)) && *buffptr)
	    show_links = wxAtoi(buffptr);
	if((buffptr = getline(&fp)) && *buffptr)
	    beep_on_enter = wxAtoi(buffptr);
	if((buffptr = getline(&fp)) && *buffptr)
	    bShowCoord = wxAtoi(buffptr) != 0;
	if((buffptr = getline(&fp)) && *buffptr)
	    show_icons = wxAtoi(buffptr) != 0;
	if((buffptr = getline(&fp)) && *buffptr)
	    show_tooltip = wxAtoi(buffptr) != 0;
	compute_train_numbers();
}

void	print_track_info(HtmlPage& page)
{
	wxString	buff;
	TextList *tl;

	buff.Printf(wxT("%s : %s"), L("Territory"), current_project.c_str());
	page.StartPage(buff);
	page.AddRuler();
	page.Add(wxT("<blockquote>\n"));
	for(tl = track_info; tl; tl = tl->next) {
	    buff.Printf(wxT("%s\n"), tl->txt);
	    page.Add(buff);
	}
	page.EndPage();
}

static	const wxChar	*en_headers[] = { wxT("Station Name"), wxT("Coordinates"), wxT("&nbsp;"), wxT("Entry/Exit"), wxT("Coordinates"), 0 };
static	const wxChar	*headers[sizeof(en_headers)/sizeof(en_headers[0])];

void	print_entry_exit_stations(HtmlPage& page)
{
	Track	**stations = get_station_list();
	Track	**entry_exit = get_entry_list();
	Track	*trk;
	int	i, j;
	wxString	buff;

	if(!headers[0])
	    localizeArray(headers, en_headers);

	page.StartPage(L("Stations and Entry/Exit Points"));
	page.Add(wxT("<blockquote>\n"));
	page.StartTable(headers);
	for(i = j = 0; (stations && stations[i]) || (entry_exit && entry_exit[j]); ) {
	    page.Add(wxT("<tr><td>"));
	    if((trk = stations[i])) {
		buff.Printf(wxT("<a href=\"stationinfopage %s\">%s</a>"), trk->station, trk->station);
		page.Add(buff);
		page.Add(wxT("</td><td>"));
		buff.Printf(wxT("%d, %d"), trk->x, trk->y);
		page.Add(buff);
		++i;
	    } else {
		page.Add(wxT("&nbsp;</td><td>&nbsp;"));
	    }
	    page.Add(wxT("</td><td>&nbsp;&nbsp;&nbsp;</td>\n<td>"));
	    if((trk = entry_exit[j])) {
		buff.Printf(wxT("<a href=\"stationinfopage %s\">%s</a>"), trk->station, trk->station);
		page.Add(buff);
		page.Add(wxT("</td><td>"));
		buff.Printf(wxT("%d, %d"), trk->x, trk->y);
		page.Add(buff);
		++j;
	    } else {
		page.Add(wxT("&nbsp;</td><td>&nbsp;"));
	    }
	    page.Add(wxT("</td></tr>\n"));
	}
	page.EndTable();
	page.EndPage();
}

/*	Set default preferences		*/

void default_prefs(void)
{
	terse_status = 1;
	status_on_top = 1;
	beep_on_alert = 1;
	beep_on_enter = 0;
	show_speeds = 1;
	auto_link = 1;
	show_grid = 0;
	show_blocks = 1;
	show_seconds = 0;
	show_icons = 1;
	signal_traditional = 0;
	hard_counters = 0;
	save_prefs = 1;
	bShowCoord = true;
}

