/*	TSignal.cpp - Created by Giampiero Caprino

This file is part of Train Director 3

Train Director is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; using exclusively version 2.
It is expressly forbidden the use of higher versions of the GNU
General Public License.

Train Director is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Train Director; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>

#if !defined(__unix__) && !defined(__WXMAC__)
#include <malloc.h>
#endif

#ifdef WIN32
#define strcasecmp stricmp
#endif

#include "Traindir3.h"
#include "itracks.h"
#include "itools.h"
#include "isignals.h"
#include "iswitch.h"
#include "iactions.h"
#include "imovers.h"
#include "TDFile.h"

int	pathIsBusy(Train *tr, Vector *path, int dir);
Vector	*findPath(Track *t, int dir);
Track	*findNextTrack(trkdir direction, int x, int y);
const Char *GetColorName(int color);

int	trace_script = 0;

bool	gMustBeClearPath;

extern	int	signals_changed;

static  void    *n_sig_pmap[2];         /* R, G */
static	void	*n_sigx_pmap[2];
static const char *n_sig_xpm[] = {
"7 9 3 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"X      c #0000FFFFFFFF",*/
"       ",
"  ...  ",
" .GGG. ",
" .GGG. ",
" .GGG. ",
"  ...  ",
"   .   ",
"   .   ",
" ..... "};
static const char *n_sigx_xpm[] = {
"7 9 3 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"X      c #0000FFFFFFFF",*/
"       ",
" ..... ",
" .GGG. ",
" .GGG. ",
" .GGG. ",
" ..... ",
"   .   ",
"   .   ",
" ..... "};

static  void    *s_sig_pmap[2];         /* R, G */
static  void    *s_sigx_pmap[2];
static const char *s_sig_xpm[] = {
"7 9 3 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"X      c #0000FFFFFFFF",*/
"       ",
" ..... ",
"   .   ",
"   .   ",
"  ...  ",
" .GGG. ",
" .GGG. ",
" .GGG. ",
"  ...  "};
static const char *s_sigx_xpm[] = {
"7 9 3 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"X      c #0000FFFFFFFF",*/
"       ",
" ..... ",
"   .   ",
"   .   ",
" ..... ",
" .GGG. ",
" .GGG. ",
" .GGG. ",
" ..... "};

static  void    *n_sig2_pmap[4];         /* R, G */
static  void    *n_sig2x_pmap[4];
static const char *n_sig2_xpm[] = {
"7 13 4 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"G      c #0000FFFFFFFF",*/
NULL, /*"X      c #0000FFFFFFFF",*/
"  ...  ",
" .GGG. ",
" .GGG. ",
" .GGG. ",
"  ...  ",
"  ...  ",
" .XXX. ",
" .XXX. ",
" .XXX. ",
"  ...  ",
"   .   ",
"   .   ",
" ..... "};
static const char *n_sig2x_xpm[] = {
"7 13 4 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"G      c #0000FFFFFFFF",*/
NULL, /*"X      c #0000FFFFFFFF",*/
" ..... ",
" .GGG. ",
" .GGG. ",
" .GGG. ",
" ..... ",
" ..... ",
" .XXX. ",
" .XXX. ",
" .XXX. ",
" ..... ",
"   .   ",
"   .   ",
" ..... "};

static  void    *s_sig2_pmap[4];         /* R, G */
static  void    *s_sig2x_pmap[4];         /* R, G */
static const char *s_sig2_xpm[] = {
"7 13 4 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"G      c #0000FFFFFFFF",*/
NULL, /*"X      c #0000FFFFFFFF",*/
" ..... ",
"   .   ",
"   .   ",
"  ...  ",
" .XXX. ",
" .XXX. ",
" .XXX. ",
"  ...  ",
"  ...  ",
" .GGG. ",
" .GGG. ",
" .GGG. ",
"  ...  "};
static const char *s_sig2x_xpm[] = {
"7 13 4 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"G      c #0000FFFFFFFF",*/
NULL, /*"X      c #0000FFFFFFFF",*/
" ..... ",
"   .   ",
"   .   ",
" ..... ",
" .XXX. ",
" .XXX. ",
" .XXX. ",
" ..... ",
" ..... ",
" .GGG. ",
" .GGG. ",
" .GGG. ",
" ..... "};

static	void	*e_sig_pmap[2];		/* R, G */
static	void	*e_sigx_pmap[2];
static const char *e_sig_xpm[] = {
"9 7 3 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"X      c #0000FFFFFFFF",*/
"         ",
"         ",
".    ... ",
".   .GGG.",
".....GGG.",
".   .GGG.",
".    ... "};
static const char *e_sigx_xpm[] = {
"9 7 3 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"X      c #0000FFFFFFFF",*/
"         ",
"         ",
".   .....",
".   .GGG.",
".....GGG.",
".   .GGG.",
".   ....."};

static	void	*w_sig_pmap[2];		/* R, G */
static	void	*w_sigx_pmap[2];
static const char *w_sig_xpm[] = {
"9 7 3 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"X      c #0000FFFFFFFF",*/
"         ",
"         ",
" ...    .",
".GGG.   .",
".GGG.....",
".GGG.   .",
" ...    ."
};
static const char *w_sigx_xpm[] = {
"9 7 3 1",
"       c #FFFFFFFFFFFF",
".      c #000000000000",
NULL, /*"X      c #0000FFFFFFFF",*/
"         ",
"         ",
".....   .",
".GGG.   .",
".GGG.....",
".GGG.   .",
".....   ."
};

static	void	*e_sig2_pmap[4];		/* RR, GR, GG, GO */
static	void	*e_sig2x_pmap[4];
static const char *e_sig2_xpm[] = {
	"13 7 4 1",
	"       c #FFFFFFFFFFFF",
	".      c #000000000000",
	NULL, /*"G      c #0000FFFFFFFF",*/
	NULL, /*"X      c #0000FFFFFFFF",*/
	"             ",
	"             ",
	".   ...  ... ",
	".  .XXX..GGG.",
	"....XXX..GGG.",
	".  .XXX..GGG.",
	".   ...  ... "};
static const char *e_sig2x_xpm[] = {
	"13 7 4 1",
	"       c #FFFFFFFFFFFF",
	".      c #000000000000",
	NULL, /*"G      c #0000FFFFFFFF",*/
	NULL, /*"X      c #0000FFFFFFFF",*/
	"             ",
	"             ",
	".  ..........",
	".  .XXX..GGG.",
	"....XXX..GGG.",
	".  .XXX..GGG.",
	".  .........."};

static	void	*e_sigP_pmap[4];		/* RR, GR, GG, GO */
static const char *e_sigP_xpm[] = {
	"13 7 4 1",
	"       c #FFFFFFFFFFFF",
	".      c #000000000000",
	NULL, /*"G      c #0000FFFFFFFF",*/
	NULL, /*"X      c #0000FFFFFFFF",*/
	"             ",
	"             ",
        ". ...... ... ",
        ". XXXXX..GGG.",
        "....X.X..GGG.",
        ". ..XXX..GGG.",
        ". ...... ... "};


static	void	*w_sig2_pmap[4];		/* RR, GR, GG, GO */
static	void	*w_sig2x_pmap[4];
static const char *w_sig2_xpm[] = {
	"13 7 4 1",
	"       c #FFFFFFFFFFFF",
	".      c #000000000000",
	NULL, /*"G      c #0000FFFFFFFF",*/
	NULL, /*"X      c #0000FFFFFFFF",*/
	"             ",
	"             ",
	" ...  ...   .",
	".GGG..XXX.  .",
	".GGG..XXX....",
	".GGG..XXX.  .",
	" ...  ...   ."};
static const char *w_sig2x_xpm[] = {
	"13 7 4 1",
	"       c #FFFFFFFFFFFF",
	".      c #000000000000",
	NULL, /*"G      c #0000FFFFFFFF",*/
	NULL, /*"X      c #0000FFFFFFFF",*/
	"             ",
	"             ",
	"..........  .",
	".GGG..XXX.  .",
	".GGG..XXX....",
	".GGG..XXX.  .",
	"..........  ."};

static	void	*w_sigP_pmap[4];		/* RR, GR, GG, GO */
static const char *w_sigP_xpm[] = {
	"13 7 4 1",
	"       c #FFFFFFFFFFFF",
	".      c #000000000000",
	NULL, /*"G      c #0000FFFFFFFF",*/
	NULL, /*"X      c #0000FFFFFFFF",*/
	"             ",
	"             ",
	" ... ...... .",
	".GGG..XXX.. .",
	".GGG..X.X....",
	".GGG..XXXXX .",
	" ... ...... ."};

// static
void	Signal::InitPixmaps()
{
	char	bufffg[64];
	char	buff[64];
	int	fgr, fgg, fgb;
	int	r, g, b;
	const char *green_name = "G      c #0000d8000000";

	getcolor_rgb(fieldcolors[COL_TRACK], &fgr, &fgg, &fgb);
	getcolor_rgb(fieldcolors[COL_BACKGROUND], &r, &g, &b);

	sprintf(bufffg, ".      c #%02x00%02x00%02x00", fgr, fgg, fgb);
	sprintf(buff, "       c #%02x00%02x00%02x00", r, g, b);
	sprintf(buff, "       c lightgray", r, g, b);
        n_sig_xpm[1] = s_sig_xpm[1] = e_sig_xpm[1] = w_sig_xpm[1] = 
	n_sigx_xpm[1] = s_sigx_xpm[1] = e_sigx_xpm[1] = w_sigx_xpm[1] = buff;

        n_sig_xpm[2] = s_sig_xpm[2] = e_sig_xpm[2] = w_sig_xpm[2] =
	n_sigx_xpm[2] = s_sigx_xpm[2] = e_sigx_xpm[2] = w_sigx_xpm[2] = bufffg;
        n_sig_xpm[3] = s_sig_xpm[3] = e_sig_xpm[3] = w_sig_xpm[3] =
	n_sigx_xpm[3] = s_sigx_xpm[3] = e_sigx_xpm[3] = w_sigx_xpm[3] = "G      c red";
        n_sig_pmap[0] = get_pixmap(n_sig_xpm);
        s_sig_pmap[0] = get_pixmap(s_sig_xpm);
        e_sig_pmap[0] = get_pixmap(e_sig_xpm);
        w_sig_pmap[0] = get_pixmap(w_sig_xpm);
        n_sigx_pmap[0] = get_pixmap(n_sigx_xpm);
        s_sigx_pmap[0] = get_pixmap(s_sigx_xpm);
        e_sigx_pmap[0] = get_pixmap(e_sigx_xpm);
        w_sigx_pmap[0] = get_pixmap(w_sigx_xpm);

        n_sig_xpm[3] = s_sig_xpm[3] = e_sig_xpm[3] = w_sig_xpm[3] =
	n_sigx_xpm[3] = s_sigx_xpm[3] = e_sigx_xpm[3] = w_sigx_xpm[3] = green_name;
        n_sig_pmap[1] = get_pixmap(n_sig_xpm);
        s_sig_pmap[1] = get_pixmap(s_sig_xpm);
        e_sig_pmap[1] = get_pixmap(e_sig_xpm);
        w_sig_pmap[1] = get_pixmap(w_sig_xpm);
        n_sigx_pmap[1] = get_pixmap(n_sigx_xpm);
        s_sigx_pmap[1] = get_pixmap(s_sigx_xpm);
        e_sigx_pmap[1] = get_pixmap(e_sigx_xpm);
        w_sigx_pmap[1] = get_pixmap(w_sigx_xpm);

	e_sig2_xpm[1] = w_sig2_xpm[1] =
		n_sig2_xpm[1] = s_sig2_xpm[1] =
		e_sig2x_xpm[1] = w_sig2x_xpm[1] =
		n_sig2x_xpm[1] = s_sig2x_xpm[1] =
		e_sigP_xpm[1] = w_sigP_xpm[1] = buff;
	e_sig2_xpm[2] = w_sig2_xpm[2] =
		n_sig2_xpm[2] = s_sig2_xpm[2] =
		e_sig2x_xpm[2] = w_sig2x_xpm[2] =
		n_sig2x_xpm[2] = s_sig2x_xpm[2] =
		e_sigP_xpm[2] = w_sigP_xpm[2] = bufffg;
	e_sig2_xpm[3] = w_sig2_xpm[3] =
		n_sig2_xpm[3] = s_sig2_xpm[3] =
		e_sig2x_xpm[3] = w_sig2x_xpm[3] =
		n_sig2x_xpm[3] = s_sig2x_xpm[3] =
		e_sigP_xpm[3] = w_sigP_xpm[3] = "G      c red";
	e_sig2_xpm[4] = w_sig2_xpm[4] =
		n_sig2_xpm[4] = s_sig2_xpm[4] =
		e_sig2x_xpm[4] = w_sig2x_xpm[4] =
		n_sig2x_xpm[4] = s_sig2x_xpm[4] = "X      c red";
	e_sigP_xpm[4] = w_sigP_xpm[4] = "X      c gray";
	e_sig2_pmap[0] = get_pixmap(e_sig2_xpm);
	w_sig2_pmap[0] = get_pixmap(w_sig2_xpm);
	e_sigP_pmap[0] = get_pixmap(e_sigP_xpm);
	w_sigP_pmap[0] = get_pixmap(w_sigP_xpm);
	n_sig2_pmap[0] = get_pixmap(n_sig2_xpm);
	s_sig2_pmap[0] = get_pixmap(s_sig2_xpm);
	e_sig2x_pmap[0] = get_pixmap(e_sig2x_xpm);
	w_sig2x_pmap[0] = get_pixmap(w_sig2x_xpm);
	n_sig2x_pmap[0] = get_pixmap(n_sig2x_xpm);
	s_sig2x_pmap[0] = get_pixmap(s_sig2x_xpm);

	e_sig2_xpm[3] = w_sig2_xpm[3] =
	n_sig2_xpm[3] = s_sig2_xpm[3] =
	e_sig2x_xpm[3] = w_sig2x_xpm[3] =
	n_sig2x_xpm[3] = s_sig2x_xpm[3] = green_name;
	e_sigP_xpm[3] = w_sigP_xpm[3] = green_name;
	e_sig2_xpm[4] = w_sig2_xpm[4] =
	n_sig2_xpm[4] = s_sig2_xpm[4] =
	e_sig2x_xpm[4] = w_sig2x_xpm[4] =
	n_sig2x_xpm[4] = s_sig2x_xpm[4] = "X      c red";
	e_sigP_xpm[4] = w_sigP_xpm[4] = "X      c gray";
	e_sig2_pmap[1] = get_pixmap(e_sig2_xpm);
	w_sig2_pmap[1] = get_pixmap(w_sig2_xpm);
	e_sigP_pmap[1] = get_pixmap(e_sigP_xpm);
	w_sigP_pmap[1] = get_pixmap(w_sigP_xpm);
	n_sig2_pmap[1] = get_pixmap(n_sig2_xpm);
	s_sig2_pmap[1] = get_pixmap(s_sig2_xpm);
	e_sig2x_pmap[1] = get_pixmap(e_sig2x_xpm);
	w_sig2x_pmap[1] = get_pixmap(w_sig2x_xpm);
	n_sig2x_pmap[1] = get_pixmap(n_sig2x_xpm);
	s_sig2x_pmap[1] = get_pixmap(s_sig2x_xpm);

	e_sig2_xpm[3] = w_sig2_xpm[3] =
	n_sig2_xpm[3] = s_sig2_xpm[3] =
	e_sig2x_xpm[3] = w_sig2x_xpm[3] =
	n_sig2x_xpm[3] = s_sig2x_xpm[3] = green_name;
	e_sig2_xpm[4] = w_sig2_xpm[4] =
	n_sig2_xpm[4] = s_sig2_xpm[4] =
	e_sig2x_xpm[4] = w_sig2x_xpm[4] =
	n_sig2x_xpm[4] = s_sig2x_xpm[4] = "X      c #0000d8000000";
	e_sigP_xpm[4] = w_sigP_xpm[4] = "X      c white";
	e_sig2_pmap[2] = get_pixmap(e_sig2_xpm);
	w_sig2_pmap[2] = get_pixmap(w_sig2_xpm);
	e_sigP_pmap[2] = get_pixmap(e_sigP_xpm);
	w_sigP_pmap[2] = get_pixmap(w_sigP_xpm);
	n_sig2_pmap[2] = get_pixmap(n_sig2_xpm);
	s_sig2_pmap[2] = get_pixmap(s_sig2_xpm);
	e_sig2x_pmap[2] = get_pixmap(e_sig2x_xpm);
	w_sig2x_pmap[2] = get_pixmap(w_sig2x_xpm);
	n_sig2x_pmap[2] = get_pixmap(n_sig2x_xpm);
	s_sig2x_pmap[2] = get_pixmap(s_sig2x_xpm);

	e_sig2_xpm[3] = w_sig2_xpm[3] =
	n_sig2_xpm[3] = s_sig2_xpm[3] =
	e_sig2x_xpm[3] = w_sig2x_xpm[3] =
	n_sig2x_xpm[3] = s_sig2x_xpm[3] = "G      c red";
	e_sigP_xpm[3] = w_sigP_xpm[3] = "G      c red";
	e_sig2_xpm[4] = w_sig2_xpm[4] =
	n_sig2_xpm[4] = s_sig2_xpm[4] =
	e_sig2x_xpm[4] = w_sig2x_xpm[4] =
	n_sig2x_xpm[4] = s_sig2x_xpm[4] = "X      c orange";
	e_sigP_xpm[4] = w_sigP_xpm[4] = "X      c white";
	e_sig2_pmap[3] = get_pixmap(e_sig2_xpm);
	w_sig2_pmap[3] = get_pixmap(w_sig2_xpm);
	e_sigP_pmap[3] = get_pixmap(e_sigP_xpm);
	w_sigP_pmap[3] = get_pixmap(w_sigP_xpm);
	/*n_sig2_pmap[3] = get_pixmap(n_sig2_xpm);
	s_sig2_pmap[3] = get_pixmap(s_sig2_xpm);*/
	e_sig2x_pmap[3] = get_pixmap(e_sig2x_xpm);
	w_sig2x_pmap[3] = get_pixmap(w_sig2x_xpm);
}

// static
void	Signal::FreePixmaps()
{
	int	i;

	for(i = 0; i < 4; ++i) {
	    delete_pixmap(e_sig2_pmap[i]);
	    delete_pixmap(w_sig2_pmap[i]);
	    delete_pixmap(e_sigP_pmap[i]);
	    delete_pixmap(w_sigP_pmap[i]);
	    delete_pixmap(n_sig2_pmap[i]);
	    delete_pixmap(s_sig2_pmap[i]);
	    delete_pixmap(e_sig2x_pmap[i]);
	    delete_pixmap(w_sig2x_pmap[i]);
	    delete_pixmap(n_sig2x_pmap[i]);
	    delete_pixmap(s_sig2x_pmap[i]);
	}
	for(i = 0; i < 2; ++i) {
	    delete_pixmap(n_sig_pmap[i]);
	    delete_pixmap(s_sig_pmap[i]);
	    delete_pixmap(e_sig_pmap[i]);
	    delete_pixmap(w_sig_pmap[i]);
	    delete_pixmap(n_sigx_pmap[i]);
	    delete_pixmap(s_sigx_pmap[i]);
	    delete_pixmap(e_sigx_pmap[i]);
	    delete_pixmap(w_sigx_pmap[i]);
	}
}

void	Signal::Draw()
{
	grcolor color = color_red;
	int	i;
	void	*p = 0;
	Signal	*t = this;

	i = 0;					/* RR */
	if(!t->_interpreterData || !(p = FindIcon())) {
	    if(t->fleeted) {
		if(t->status == ST_GREEN) {
		    if(t->nowfleeted)
			i = 2;			/* GG */
		    else
			i = 1;			/* GR */
		} else if(t->nowfleeted)
		    i = 3;				/* RO */
		switch(t->direction) {
		case W_E:
		    p = signal_traditional ?
			(t->signalx ? e_sig2x_pmap[i] : e_sig2_pmap[i]) : e_sigP_pmap[i];
		    break;
		case E_W:
		    p = signal_traditional ?
			(t->signalx ? w_sig2x_pmap[i] : w_sig2_pmap[i]) : w_sigP_pmap[i];
		    break;
		case N_S:
		    p = t->signalx ? s_sig2x_pmap[i] : s_sig2_pmap[i];
		    break;
		case S_N:
		    p = t->signalx ? n_sig2x_pmap[i] : n_sig2_pmap[i];
		    break;
		}
		if(p)
		    draw_pixmap(t->x, t->y, p);
		if(editing && show_links && t->controls)
		    draw_link(t->x, t->y, t->controls->x, t->controls->y, conf.linkcolor);
		return;
	    }
	    if(t->status == ST_GREEN)
		i = 1;
	    switch(t->direction) {
	    case W_E:
		    p = t->signalx ? e_sigx_pmap[i] : e_sig_pmap[i];
		    break;
	    case E_W:
		    p = t->signalx ? w_sigx_pmap[i] : w_sig_pmap[i];
		    break;
	    case N_S:
		    p = t->signalx ? s_sigx_pmap[i] : s_sig_pmap[i];
		    break;
	    case S_N:
		    p = t->signalx ? n_sigx_pmap[i] : n_sig_pmap[i];
		    break;
	    }
	}
	if(p)
	    draw_pixmap(t->x, t->y, p);
	if(editing && show_links && t->controls)
	    draw_link(t->x, t->y, t->controls->x, t->controls->y, conf.linkcolor);
}



SignalAspect::SignalAspect()
{
	_next = 0;
	_name = 0;
	_action = wxStrdup(wxT("none"));
	memset(_iconN, 0, sizeof(_iconN));
	memset(_iconE, 0, sizeof(_iconE));
	memset(_iconS, 0, sizeof(_iconS));
	memset(_iconW, 0, sizeof(_iconW));
}


SignalAspect::~SignalAspect()
{
	int	    i;

	for(i = 0; i < MAX_FLASHING_ICONS; ++i) {
	    if(_iconN[i])
		free(_iconN[i]);
	    if(_iconE[i])
		free(_iconE[i]);
	    if(_iconS[i])
		free(_iconS[i]);
	    if(_iconW[i])
		free(_iconW[i]);
	}
	if(_name)
	    free(_name);
}


void	Signal::ParseAspect(const wxChar **pp)
{
	wxChar	line[256];
	const wxChar	*p = *pp;
	wxChar	**dst;
	SignalAspect *asp = new SignalAspect();
	SignalInterpreterData *interp = (SignalInterpreterData *)_interpreterData;

	p = scan_line(p, line);
	if(line[0])
	    asp->_name = wxStrdup(line);
	do {
	    dst = 0;
	    if(match(&p, wxT("IconN:")))
		dst = &asp->_iconN[0];
	    else if(match(&p, wxT("IconE:")))
		dst = &asp->_iconE[0];
	    else if(match(&p, wxT("IconS:")))
		dst = &asp->_iconS[0];
	    else if(match(&p, wxT("IconW:")))
		dst = &asp->_iconW[0];
	    if(dst) {
		p = scan_line(p, line);
		if(line[0]) {
		    if(wxStrchr(line, ' ')) {
			this->_isFlashing = true;
			int	nxt = 0;
			wxChar	*p1, *pp;

			pp = line;
			do {
			    for(p1 = pp; *pp && *pp != ' '; ++pp);
			    if(p1 != pp) {
				int oc = *pp;
				*pp = 0;
				*dst++ = wxStrdup(p1);
				*pp = oc;
				while(*pp == ' ') ++pp;
				if(++nxt >= MAX_FLASHING_ICONS)
				    break;
			    }
			} while(*pp);
		    } else
			*dst = wxStrdup(line);
		}
		continue;
	    }
	    if(match(&p, wxT("Action:"))) {
		p = scan_line(p, line);
		if(!line[0])
		    continue;
		if(asp->_action)
		    free(asp->_action);
		asp->_action = wxStrdup(line);
		continue;
	    }
	    break;
	    // unknown. Should we give an error?
	} while(1);
	asp->_next = interp->_aspects;
	interp->_aspects = asp;
	*pp = p;
}


void	Signal::ParseProgram()
{
	const wxChar	*p;

	if(!this->stateProgram || !*this->stateProgram)
	    return;
	Script	*s = find_script(this->stateProgram);
	SignalInterpreterData *interp;
	if(!s) {
	    s = new_script(this->stateProgram);
	    // return;
	}
	if(!s->ReadFile())
	    return;

	if(!_interpreterData)
	    _interpreterData = new SignalInterpreterData;

	interp = (SignalInterpreterData *)_interpreterData;
	p = s->_text;
	while(*p) {
	    const wxChar	*p1 = p;
	    while(*p1 == ' ' || *p1 == '\t' || *p1 == '\r' || *p1 == '\n')
		++p1;
	    p = p1;
	    if(match(&p, wxT("Aspect:"))) {
		p1 = p;
		ParseAspect(&p);
	    } else if(match(&p, wxT("OnClick:"))) {
		p1 = p;
		interp->_onClick = ParseStatements(&p);
	    } else if(match(&p, wxT("OnCleared:"))) {
		p = next_token(p);
		p1 = p;
		interp->_onCleared = ParseStatements(&p);
	    } else if(match(&p, wxT("OnInit:"))) {
		p = next_token(p);
		p1 = p;
		interp->_onInit = ParseStatements(&p);
	    } else if(match(&p, wxT("OnUpdate:"))) {
		p = next_token(p);
		p1 = p;
		interp->_onUpdate = ParseStatements(&p);
	    } else if(match(&p, wxT("OnAuto:"))) {
		p = next_token(p);
		p1 = p;
		interp->_onAuto = ParseStatements(&p);
	    } else if(match(&p, wxT("OnCross:"))) {
		p = next_token(p);
		p1 = p;
		interp->_onCross = ParseStatements(&p);
	    }
	    if(p1 == p)	    // error! couldn't parse token
		break;
	}
}

Signal	*Signal::GetNextSignal()
{
	Signal	*sig = this;

	Track	*t;
	trkdir	dir;
	Vector	*path;

	if(!sig->controls)
	    return 0;

	path = findPath(sig->controls, sig->direction);
	t = (Track *)Vector_elementAt(path, path->size - 1);
	dir = (trkdir)Vector_flagAt(path, path->size - 1);
	Vector_delete(path);
	Track *t1;
	if(!(t1 = findNextTrack(dir, t->x, t->y))) {
	    wxSnprintf(expr_buff + wxStrlen(expr_buff), sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT("no next track after (%d,%d)"), t->x, t->y);
	    return 0;
	}
	t = t1;
	sig = (Signal *)((dir == W_E || dir == S_N) ?
			    t->esignal : t->wsignal);
	if(!sig) {
	    wxSnprintf(expr_buff + wxStrlen(expr_buff), sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT("no signal after (%d,%d)"), t->x, t->y);
	    return 0;
	}
	wxSnprintf(expr_buff + wxStrlen(expr_buff), sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT("(%d,%d)"), sig->x, sig->y);
	return sig;
}

bool	Signal::GetNextPath(Vector **ppath)
{
	Signal	*s = this;

	*ppath = 0;
	if(s->fixedred) {
	    wxStrncat(expr_buff, wxT("signal is always red"), sizeof(expr_buff)-1);
	    return true;
	}
	*ppath = findPath(s->controls, s->direction);
	if(!*ppath) {
	    wxStrncat(expr_buff, wxT("no valid path"), sizeof(expr_buff)-1);
	    return false;
	}
	if(gMustBeClearPath) {
//	    if(!s->IsClear()) { // t->status == ST_GREEN) {
//		Vector_delete(path);
//		return true;
//	    }
	    if(pathIsBusy(NULL, *ppath, s->direction)) {
		Vector_delete(*ppath);
		*ppath = 0;
		wxStrncat(expr_buff, wxT("path is busy"), sizeof(expr_buff)-1);
		return true;
	    }
	}
	return true;
}

//
//	Execution of the Abstract Syntax Tree
//

wxChar	expr_buff[EXPR_BUFF_SIZE];

void	Trace(const wxChar *msg)
{
	if(!trace_script)
	    return;
	traindir->AddAlert(msg);
}


Signal	*SignalInterpreterData::GetNextSignal(Signal *sig)
{
	return sig->GetNextSignal();
}

bool	SignalInterpreterData::GetNextPath(Signal *sig, Vector **ppath)
{
	gMustBeClearPath = _mustBeClearPath;
	bool res = sig->GetNextPath(ppath);
	gMustBeClearPath = false;
	return res;
}

bool	SignalInterpreterData::Evaluate(ExprNode *n, ExprValue& result)
{
	ExprValue left(None);
	ExprValue right(None);
	const wxChar	*prop;
	Signal	*sig;

	if(!n)
	    return false;
        switch(n->_op) {

	case NextSignalRef:

	    sig = GetNextSignal(_signal);
	    if(!sig)
		return false;
	    result._op = SignalRef;
	    result._signal = sig;
	    return true;

	case Dot:
	    
	    result._op = Addr;
	    if(!(n->_left)) {
		result._signal = this->_signal;		// .<property> ->   this->signal
		result._op = SignalRef;
		if(!result._signal) {
		    wxStrcat(expr_buff, wxT("no current signal for '.'"));
		    return false;
		}
		TraceCoord(result._signal->x, result._signal->y);
	    } else if(n->_left && n->_left->_op == Dot) {
		bool oldForAddr = _forAddr;
		_forAddr = true;
		if(!Evaluate(n->_left, result)) {	// <signal>.<property>
		    _forAddr = oldForAddr;
		    return false;
		}
		_forAddr = oldForAddr;

		if(result._op != SignalRef)
		    return false;
		result._signal = GetNextSignal(result._signal);
		if(!result._signal) {
		    wxStrcat(expr_buff, wxT("no current signal for '.'"));
		    return false;
		}
		TraceCoord(result._signal->x, result._signal->y);
	    } else {
		if(!Evaluate(n->_left, result))
		    return false;
	    }
	    result._txt = n->_txt;
	    if(_forAddr)
		return true;

	    prop = n->_txt;
	    if(!prop)
		return false;

	    switch(result._op) {
	    
	    case SwitchRef:

		if(!wxStrcmp(prop, wxT("thrown"))) {
		    result._op = Number;
		    if(!result._track || result._track->type != SWITCH)
			result._val = 0;
		    else
			result._val = result._track->switched;
		    return true;
		}

	    case Addr:
	    case TrackRef:

		if(!result._track)
		    return false;
		return result._track->GetPropertyValue(prop, result);

	    case SignalRef:

		if(!result._signal)
		    return false;
		return result._signal->GetPropertyValue(prop, result);

	    case TrainRef:

		if(!result._train)
		    return false;
		return result._train->GetPropertyValue(prop, result);

	    }
	    return false;

	default:

	    return InterpreterData::Evaluate(n, result);
	}
	return false;
}


void	*Signal::FindIcon()
{
	SignalInterpreterData *interp = (SignalInterpreterData *)_interpreterData;
	SignalAspect	*asp = interp->_aspects;
	wxChar		**p = 0;
	int		ix;
	const wxChar	*curState;

	if(this->_currentState)
	    curState = this->_currentState;
	else if(this->status == ST_GREEN)
	    curState = wxT("green");
	else
	    curState = wxT("red");

	while(asp) {
	    if(!wxStricmp(asp->_name, curState))
		break;
	    asp = asp->_next;
	}
	if(!asp)
	    return 0;
	switch(this->direction) {
	case W_E:
		p = asp->_iconE;
		break;

	case E_W:
		p = asp->_iconW;
		break;

	case N_S:
		p = asp->_iconS;
		break;

	case S_N:
		p = asp->_iconN;
		break;
	}
	if(!p || !*p)
	    return 0;
	if(_isFlashing) {
	    if(!p[_nextFlashingIcon])
		_nextFlashingIcon = 0;
	    p = &p[_nextFlashingIcon];
	}
	if((ix = get_pixmap_index(*p)) < 0)
	    return 0;
	return pixmaps[ix].pixels;
}




bool	Signal::IsClear()
{
	if(this->_currentState) {
	    if(_isShuntingSignal) {
		return wxStrcmp(this->_currentState, wxT("yellow")) != 0;
	    }
	    return wxStrcmp(GetAction(), wxT("stop")) != 0;	// !Rask
	}
	return this->status == ST_GREEN;
}


void	Signal::OnClear()
{
	signals_changed = 1;
	if(_interpreterData) {
	    SignalInterpreterData interp((SignalInterpreterData *)_interpreterData);
	    if(interp._onCleared) {
		interp._signal = this;
		interp._mustBeClearPath = true;
		wxSnprintf(expr_buff, sizeof(expr_buff)/sizeof(wxChar), wxT("%s::OnClear(%d,%d)"), this->stateProgram, this->x, this->y);
		Trace(expr_buff);
		interp.Execute(interp._onCleared);
		return;
	    }
	}

	this->status = ST_GREEN;
	_currentState = wxT("green");
	_nextFlashingIcon = 0;	    // in case new aspect is not flashing
}


void	Signal::OnUnclear()
{
	signals_changed = 1;
	if(_interpreterData) {
	    SignalInterpreterData interp((SignalInterpreterData *)_interpreterData);
	    if(interp._onClick) {
		interp._signal = this;
		wxSnprintf(expr_buff, sizeof(expr_buff)/sizeof(wxChar), wxT("%s::OnUnclear(%d,%d)"), this->stateProgram, this->x, this->y);
		Trace(expr_buff);
		interp.Execute(interp._onClick);
		return;
	    }
	}

	this->status = ST_RED;
	_currentState = wxT("red");
	_nextFlashingIcon = 0;	    // in case new aspect is not flashing
}



void	Signal::OnCross()
{
//	if(this->aspect_changed)
//	    return;

	if(_interpreterData) {
	    SignalInterpreterData interp((SignalInterpreterData *)_interpreterData);
	    if(interp._onCross) {
		interp._signal = this;
		wxSnprintf(expr_buff, sizeof(expr_buff)/sizeof(wxChar), wxT("%s::OnCross(%d,%d)"), this->stateProgram, this->x, this->y);
		Trace(expr_buff);
		interp.Execute(interp._onCross);
		return;
	    }
	}
	signals_changed = 1;
	this->status = ST_RED;
	SetAspect(wxT("red"));
	_nextFlashingIcon = 0;	    // in case new aspect is not flashing
}


void	Signal::OnUnlock()
{
	signals_changed = 1;
	this->status = ST_GREEN;
}


void	Signal::OnUnfleet()
{
	signals_changed = 1;
	this->status = ST_GREEN;
}

void	Signal::OnUpdate()
{
	if(this->aspect_changed)
	    return;

	if(!_interpreterData)
	    return;

	SignalInterpreterData interp((SignalInterpreterData *)_interpreterData);
	if(!interp._onUpdate)
	    return;

	interp._signal = this;
	wxSnprintf(expr_buff, sizeof(expr_buff)/sizeof(wxChar), wxT("%s::OnUpdate(%d,%d)"), this->stateProgram, this->x, this->y);
	Trace(expr_buff);
	interp.Execute(interp._onUpdate);
}

void	Signal::OnInit()
{
	if(!_interpreterData)
	    return;
	SignalInterpreterData interp((SignalInterpreterData *)_interpreterData);
	if(!interp._onInit)
	    return;
	interp._signal = this;
	interp._mustBeClearPath = true;
	wxSnprintf(expr_buff, sizeof(expr_buff)/sizeof(wxChar), wxT("%s::OnInit(%d,%d)"), this->stateProgram, this->x, this->y);
	Trace(expr_buff);
	interp.Execute(interp._onInit);
	return;
}

void	Signal::OnAuto()
{
	if(!_interpreterData)
	    return;

	SignalInterpreterData interp((SignalInterpreterData *)_interpreterData);
	if(!interp._onAuto)
	    return;

	interp._signal = this;
	interp._mustBeClearPath = true;
	wxSnprintf(expr_buff, sizeof(expr_buff)/sizeof(wxChar), wxT("%s::OnAuto(%d,%d)"), this->stateProgram, this->x, this->y);
	Trace(expr_buff);
	interp.Execute(interp._onAuto);
}


void	Signal::OnClicked()
{
	if(!_interpreterData)
	    return;

	SignalInterpreterData interp((SignalInterpreterData *)_interpreterData);
	if(!interp._onClick)
	    return;

	interp._signal = this;
	interp._mustBeClearPath = true;
	wxSnprintf(expr_buff, sizeof(expr_buff)/sizeof(wxChar), wxT("%s::OnClicked(%d,%d)"), this->stateProgram, this->x, this->y);
	Trace(expr_buff);
	interp.Execute(interp._onClick);
}


void	Signal::OnFlash()
{
	SignalAspect *asp;

	if(!_interpreterData)
	    return;
	if(!_currentState)
	    return;

	SignalInterpreterData interp((SignalInterpreterData *)_interpreterData);
	wxChar	    **p;

	for(asp = interp._aspects; asp; asp = asp->_next)
	    if(!wxStrcmp(_currentState, asp->_name)) {
		int	nxt = _nextFlashingIcon + 1;

		if(nxt >= MAX_FLASHING_ICONS)
		    nxt = 0;
		p = 0;
		switch(this->direction) {
		case W_E:
			p = asp->_iconE;
			break;

		case E_W:
			p = asp->_iconW;
			break;

		case N_S:
			p = asp->_iconS;
			break;

		case S_N:
			p = asp->_iconN;
			break;
		}
		if(!p || ! p[nxt])
		    nxt = 0;
		_nextFlashingIcon = nxt;
    		change_coord(this->x, this->y);
		break;
	    }
}


void	Signal::SetAspect(const wxChar *aspect)
{
	if(!_currentState || wxStrcmp(_currentState, aspect)) {
	    signals_changed = 1;
    	    change_coord(this->x, this->y);
	    this->aspect_changed = 1;
	}

	_currentState = aspect;
	_nextFlashingIcon = 0;	    // in case new aspect is not flashing
}


const wxChar *Signal::GetAspect()
{
	if(_currentState)
	    return _currentState;
	if(this->status == ST_RED)
	    return wxT("red");
	return wxT("green");
}

const wxChar	*Signal::GetAction()
{
	const wxChar	*name = GetAspect();
	SignalInterpreterData *interp = (SignalInterpreterData *)_interpreterData;
	SignalAspect *asp;

	if(!interp) {
	    if(!wxStrcmp(name, wxT("red")))
		return wxT("stop");
	    return wxT("proceed");
	}
	for(asp = interp->_aspects; asp; asp = asp->_next) {
	    if(!wxStrcmp(name, asp->_name) && asp->_action)
		return asp->_action;
	}
	return wxT("proceed");	    // broken signal? maybe we should stop.
}


bool	Signal::IsApproach()
{
	return wxStrcmp(GetAction(), wxT("none")) == 0 || _isShuntingSignal;
}


bool	Signal::IsShuntingSignal()
{
	return _isShuntingSignal;
}

bool	Signal::GetSpeedLimit(int *limit)
{
	wxChar	buff[256];
	const wxChar	*action = GetAction();
	action = scan_word(action, buff);

	if(!wxStrcmp(buff, wxT("speedLimit"))) {
	    *limit = wxAtoi(action);
	    return true;
	}
	return false;
}

bool	Signal::GetApproach(ExprValue& result)
{
	bool	res;
	Vector	*path;
	int	i;

	res = GetNextPath(&path);
	if(!path)
	    return res;

	for(i = 0; i < path->size; ++i) {
	    Track	*trk = (Track *)Vector_elementAt(path, i);
	    trkdir	dir = (trkdir)Vector_flagAt(path, i);
	    if(dir == E_W || dir == N_S) {
		if(trk->wsignal) {
		    Signal *sig = trk->wsignal;
		    if(!sig->IsShuntingSignal() && sig->IsApproach()) {
			result._op = Addr;
			result._track = sig;
			break;
		    }
		}
	    } else if(trk->esignal) {
		Signal *sig = trk->esignal;
		if(!sig->IsShuntingSignal() && sig->IsApproach()) {
		    result._op = Addr;
		    result._track = sig;
		    break;
		}
	    }
	}
	res = (i >= path->size) ? 0 : 1;
	Vector_delete(path);
	wxSnprintf(expr_buff + wxStrlen(expr_buff),
	    sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT(" found"));
	return res;
}


bool	Signal::GetPropertyValue(const wxChar *prop, ExprValue& result)
{
	bool	res;
	Vector	*path;
	int	i;

	Signal	*s = this;
	wxStrncat(expr_buff, prop, sizeof(expr_buff)-1);

	if(!wxStrcmp(prop, wxT("aspect"))) {
	    result._op = String;
	    result._txt = s->GetAspect();
	    wxSnprintf(expr_buff + wxStrlen(expr_buff),
		sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT("{%s}"), result._txt);
	    return true;
	}
	if(!wxStrcmp(prop, wxT("auto"))) {
	    result._op = Number;
	    result._val = s->fleeted;
	    wxSnprintf(expr_buff + wxStrlen(expr_buff),
		sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT("{%d}"), result._val);
	    return true;
	}
	if(!wxStrcmp(prop, wxT("enabled"))) {
	    result._op = Number;
	    result._val = s->fleeted && s->nowfleeted;
	    wxSnprintf(expr_buff + wxStrlen(expr_buff),
		sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT("{%d}"), result._val);
	    return true;
	}

	result._op = Number;
	result._val = 0;
	if(!wxStrcmp(prop, wxT("switchThrown"))) {
	    res = s->GetNextPath(&path);
	    if(!path)
		return res;

	    for(i = 0; i < path->size; ++i) {
		Track	*trk = (Track *)Vector_elementAt(path, i);

		if(trk->type != SWITCH)
		    continue;
		if(trk->switched) {
		    result._val = 1;
		    break;
		}
	    }
	    wxSnprintf(expr_buff + wxStrlen(expr_buff),
		sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff),
		wxT("{%s}"), result._val ? wxT("switchThrown") : wxT("switchNotThrown"));
	    Vector_delete(path);
	    return true;
	}
	if(!wxStrcmp(prop, wxT("nextLimit"))) {
	    res = s->GetNextPath(&path);
	    if(!path)
		return res;

	    int	    j;
	    int	    lowSpeed = 1000;
	    
	    for(i = 0; i < path->size; ++i) {
		Track	*trk = (Track *)Vector_elementAt(path, i);

		for(j = 0; j < NTTYPES; ++j)
		    if(trk->speed[j] && trk->speed[j] < lowSpeed)
			lowSpeed = trk->speed[j];
	    }
	    result._val = lowSpeed;
	    Vector_delete(path);
	    wxSnprintf(expr_buff + wxStrlen(expr_buff),
		sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT("{%d}"), lowSpeed);
	    return true;
	}
	if(!wxStrcmp(prop, wxT("nextLength"))) {
	    res = s->GetNextPath(&path);
	    if(!path)
		return res;

	    int	    length = 0;
	    
	    for(i = 0; i < path->size; ++i) {
		Track	*trk = (Track *)Vector_elementAt(path, i);
		length += trk->length;
	    }
	    result._val = length;
	    Vector_delete(path);
	    wxSnprintf(expr_buff + wxStrlen(expr_buff),
		sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT("{%d}"), length);
	    return true;
	}
	if(!wxStrcmp(prop, wxT("nextApproach"))) {
	    return GetApproach(result);
	}
	if(!wxStrcmp(prop, wxT("nextIsApproach"))) {
	    res = GetApproach(result);
	    result._op = Number;
	    result._val = res == true;
	    return true;
	}
	if(!wxStrcmp(prop, wxT("nextStation"))) {
	    result._op = String;
	    result._txt = wxT("");

	    res = s->GetNextPath(&path);
	    if(!path)
		return res;

	    for(i = 0; i < path->size; ++i) {
		Track	*trk = (Track *)Vector_elementAt(path, i);

		if(!trk->isstation)
		    continue;
		result._txt = trk->station;
		break;
	    }
	    Vector_delete(path);
	    wxSnprintf(expr_buff + wxStrlen(expr_buff), sizeof(expr_buff)/sizeof(wxChar) - wxStrlen(expr_buff), wxT("{%s}"), result._txt);
	    return true;
	}
	if(!wxStrcmp(prop, wxT("color"))) {
	    result._op = String;
	    result._txt = wxT("");
	    if(s->controls)
		result._txt = GetColorName(s->controls->fgcolor);
	    return true;
	}

	return false;
}


bool	Signal::SetPropertyValue(const wxChar *prop, ExprValue& val)
{
	Signal	*s = this;
    
	if(!wxStrcmp(prop, wxT("thrown"))) {
	    // call t->Throw(val._val);
	} else if(!wxStrcmp(prop, wxT("aspect"))) {
	    s->SetAspect(val._txt);
	} else if(!wxStrcmp(prop, wxT("auto"))) {
	    s->fleeted = val._val;
	} else if(!wxStrcmp(prop, wxT("enabled"))) {
	    s->nowfleeted = val._val;
	} else if(!wxStrcmp(prop, wxT("fleeted"))) {
	    s->fleeted = val._val;
	    s->nowfleeted = val._val;
	} else if(!wxStrcmp(prop, wxT("shunting"))) {
	    s->_isShuntingSignal = val._val != 0;
	} else if(!wxStrcmp(prop, wxT("click"))) {
	    track_selected(s->x, s->y);
	} else if(!wxStrcmp(prop, wxT("rclick"))) {
	    track_selected1(s->x, s->y);
	} else if(!wxStrcmp(prop, wxT("ctrlclick"))) {
	    Coord c(s->x, s->y);
	    track_control_selected(c);
	} else
	    return false;
	return true;
}



