# SuperDateTime plugin.pm by Greg Brown Feb 2005    www.gregbrown.net/squeeze
#	Copyright (c) 2005 
#	All rights reserved.
#
# DESCRIPTION
# SlimServer plugin screensaver datetime replacement. Graphically displays current weather conditions and forecasts.  
# Will also optionally display stock quotes and upcoming/current game information for MLB, NBA, NHL, NFL, and college 
# football and basketball teams at user-configurable intervals.
#
# REMOTE CONTROL
# Pressing up or down cycles the information shown on line one during time display.  It will also immediately
# display the time if it is not currently being shown.
# Pressing right or left cycles through the available games and long weather forecasts that normally cycle automatically.
# Pressing FWD or REW cycles through the long weather forecasts.
# Pressing '+' will force an immediate data refresh.
#
# This screen saver is based on the standard SlimServer DateTime screensaver written written by kdf.
# The graphical weather icons and the code to support them are based on the WeatherTime screensaver written by Martin Rehfeld.
#
# VERSION HISTORY
# 5.6.2  08/30/08  Alarm display support.  
#                  Touched font files to resolve overlapping text issue caused by
#                  SqueezeCenter fontcache bug.
#                  SDT will refresh data when there are no players with displays (Duet only configurations)
# 5.6.0  08/28/08  BOOM Support.  Player-specific configuration.
# 5.5.1  04/11/08  Fixed MLB parsing
# 5.5.0            Now offers API for other plugins to add display info.
#                  Added all remaing MLB icons.  Thanks B. Yudichak!
#                  Added Minnesota and Tampa Bay MLB icons.  Thanks M.Meyer!            
#						 Major overhaul of display code for future expandability.
#						 Fuzzytime support  http://www.tux.org/~peterw/slim/FuzzyTime.html
#                  Supports negative forecasts
#                  Save last viewing top item on restart
#                  Highly configurable display with smart spacing
#                  Up/down moves both screens for period.
#                  Holding down jumps to current conditions
#                  10 day forecasts
#                  Data updates sequential and less prone to blocking
#                  *** SlimServer 7.0 or later is required. ***
# 5.0.5  04/02/07  Fixed ESPN MLB parsing to handle minor '07 season website changes.
#                  Divided up MLB processing to decrease HTML parsing hangs.
#                  Added a few additional MLB logos.
# 5.0.4  01/19/07  Fixed bug where Top-25 conference selection is overwritten on sever restart.
# *** Complete version history is available on the website listed above. ***
#
# INSTALLATION
# 1. Unzip downloaded zip file into SlimServer\server\Plugins\ directory.  This should result in a SuperDateTime folder
#    being created within the Plugins folder that contains Plugin.pm.  The folder name is case sensitive.
# 2. Restart slimserver.
# 3. Ensure plugin is enabled from Server Settings / Plugins web page.
# 4. Configure custom plugin settings (city, teams, etc.) from Server Setttings / Plugins web page.
# 5. Enable screensaver for when a player is turned off from Player Settings web page.   
#
# [OPTIONAL] SlimServer can be configured to temporarily display SuperDateTime when powered on by holding the
#            size button for a couple seconds.  The display is returned to it's previous state by holding the
#            size button again for a couple seconds.  This is useful to view sports/weather/etc information
#            while listening to music.
#
# 6. Open up \SlimServer\Server\IR\Default.map in your favorite raw text editor.
# 7. Scroll down to # textsize modes
# 8. Edit the text in that section to mirror:
#      size = dead
#      size.single = textsize_toggle
#      size.hold   = modefunction_Plugins::SuperDateTime::Plugin->showme
#      #size           = textsize_toggle
#      textsize_small  = textsize_small
#      textsize_medium = textsize_medium
#      textsize_large  = textsize_large
# 9. Restart SlimServer
# 
# FEEDBACK
# Please direct all feedback to GoCubs on the Slim Devices public forums at forums.slimdevices.com
#
# KNOWN BUGS/ISSUES
# 1.) Text may get truncated if it does not fit on the screen.  Selecting a smaller font fixes the problem.
# 2.) When using the "slide" scroll option, time display may appear to skip a second when the display
# transitions from showing game information or long description forecast to the time.
# 3.) In celius mode, a long description forecast that contains a temperature group (ie mid to upper 30s) instead of
# a specific high/low value (Low 30F) may not make sense after the conversion (ie mid to upper -2s).
# 4.) When SuperDateTime is used as an offsaver in SoftSqueeze, data refreshes will continue to occur once SoftSqueeze is
# closed.  This is because SoftSqueeze is put in "off" state when it is closed, which activates the offsaver. 
#
#	This program is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with this program; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
#	02111-1307 USA
#
# This code is derived from code with the following copyright message:
#
# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.

package Plugins::SuperDateTime::Plugin;

use strict;

use base qw(Slim::Plugin::Base);
use Slim::Utils::Misc;
use Slim::Utils::Strings qw (string);
use Slim::Networking::SimpleAsyncHTTP;
use Slim::Utils::Prefs;
use Slim::Utils::Log;
use Plugins::SuperDateTime::Settings;
use Plugins::SuperDateTime::PlayerSettings;

my $prefs = preferences('plugin.superdatetime');

my $log = Slim::Utils::Log->addLogCategory({
	'category'     => 'plugin.superdatetime',
	'defaultLevel' => 'WARN',
#	'defaultLevel' => 'DEBUG',
	'description'  => getDisplayName(),
});

use vars qw($VERSION);
$VERSION = substr(q$Revision: 5.6.1 $,10);

$Plugins::SuperDateTime::Plugin::apiVersion = 1.0;

sub getDisplayName {
	return 'PLUGIN_SCREENSAVER_SUPERDATETIME';
}

sub enabled {
	return ($::VERSION ge '7.0');
}

my $killTicker = 0; #Flag used to exit out of ticker mode
my $status = '';  #Used for the status indicator
my %overlay; #Used to show alarm indicator
my $errorCount = 0; #Used to keep track of network errors
my $activeClients = 0; #Number of players currently displaying SuperDateTime

#These preferences are used often so they're locally cached. Preset to default values.
my $showtime = 7; #How long to show the time when no games are active
my $showgame = 3; #How long to show an upcoming/completed game
my $showactivetime = 5; #How long to show the time when games are active
my $showactivegame = 3; #How long to show an active game
my $activeGames = 0; #Flag set if any games are active
my $newActiveGames; #Flag set during a data refresh if any games are active
my %scrollType = (); #Type of display scrolling between games per player

my %displayInfo; #Contains all the info shown when hitting up/down
my %displayInfoBuild; #Used while building display info in pieces

my %nowshowing; #What's now showing, 0 for time, Negative for long forecasts, etc
my %weathershowing; #Used for when the user chooses to just view the long weather forecasts via remote
my %topNowShowing; #Which item is being shown on top line during time display
my %displayLine1; #What is currently being shown on the display line 1 while cycling
my %displayLine2; #What is currently being shown on the display line 2 while cycling
my %tickline1; #Used to store previous value of display line 1 during ticker display

my %wetData = ();	#Stores weather-related data
my %sportsData = (); #Stores all sport info for use from jive interface

#Track the average high/low for today/tomorrow
my %averages = ('last' => ''); #Used to store raw averages data and when they were last refreshed

my @divideMLB = (); #Array used to divide MLB processing

#Used to store pointers to the data providers
my %providers =  ('1'  => \&get10day,
                  '2'  => \&getAverages,
                  '3'  => \&getWunderground,
                  '4'  => \&getMLB,
                  '5'  => \&getNFL,
                  '6'  => \&getNBA,
                  '7'  => \&getNHL,
                  '8'  => \&getCBB,                  
                  '9'  => \&getCFB,
                  '10' => \&getStocks);

my @WETdisplayItems2 = ();

my @displayItems2;
$displayItems2[0] =\@WETdisplayItems2;

my @refreshTracker = (); #Used to keep track of which fields to continue to refresh
my $lastRefresh = ''; #Date that pertains to @refreshTracker data

#Arrays to contain the user selected teams to monitor for each sport and icon mappings
my @MLBteams = ();
my %MLBmap = ('Cubs'  => 10, 'White Sox'  => 11, 'Atlanta' => 12, 'San Francisco' => 13, 'LA Angels' => 14, 'Detroit' => 15, 'Houston' => 16, 'Cincinnati' => 17, 'Arizona' => 18, 'Minnesota' => 19, 'Tampa Bay' => 20, 'Boston' => 21, 'Cleveland' => 22, 'Texas' => 24, 'NY Yankees' => 25, 'Baltimore' => 26, 'Toronto' => 27, 'Kansas City' => 28, 'Oakland' => 29, 'Seattle' => 30, 'Pittsburgh' => 31, 'NY Mets' => 32, 'Philadelphia' => 33, 'Florida' => 34, 'Washington' => 35, "Milwaukee" => 36, 'St. Louis' => 37, 'LA Dodgers' => 38, 'San Diego' => 39, 'Colorado' => 40 ); #Graphical icon mapping
my @NBAteams = ();
my @NHLteams = ();
my @NFLteams = ();
my %NFLmap = ('Chicago'  => 10, 'Detroit'  => 11, 'Green Bay' => 12); #Graphical icon mapping
my @CBBteams = ();
my @CFBteams = ();

#Graphical display stuff
my %xmax = ();
my %ymax = ();

my %hashDisp;

#
# Map ASCII characters to custom @Charset elements
#
my %Codepage = ( ' ' =>  0, '1' =>  1, '2' =>  2, '3' =>  3, '4' =>  4,
                 '5' =>  5, '6' =>  6, '7' =>  7, '8' =>  8, '9' =>  9,
                 '0' => 10, '-' => 11, '' => 12, '.' => 13, '%' => 14,
                 'A' => 15, 'B' => 16, 'C' => 17, 'D' => 18, 'E' => 19,
                 'F' => 20, 'G' => 21, 'H' => 22, 'I' => 23, 'J' => 24,
                 'K' => 25, 'L' => 26, 'M' => 27, 'N' => 28, 'O' => 29,
                 'P' => 30, 'Q' => 31, 'R' => 32, 'S' => 33, 'T' => 34,
                 'U' => 35, 'V' => 36, 'W' => 37, 'X' => 38, 'Y' => 39,
                 'Z' => 40, '/' => 41, ':' => 42 );

#
# Custom 7x5 mono charset for 3-line forecast display on Squeezebox2
#
my @Charset = ('







','
  *
 **
* *
  *
  *
  *
*****
','
 ***
*   *
    *
   *
  *
 *
*****
','
 ***
*   *
    *
  **
    *
*   *
 ***
','
   *
  **
 * *
*****
   *
   *
   *
','
*****
*
*
****
    *
*   *
 ***
','
 ***
*
*
****
*   *
*   *
 ***
','
*****
    *
    *
   *
  *
 *
*
','
 ***
*   *
*   *
 ***
*   *
*   *
 ***
','
 ***
*   *
*   *
 ****
    *
    *
 ***
','
 ***
*   *
*   *
*   *
*   *
*   *
 ***
','



 ****



','
 **
*  *
*  *
 **



','





  **
  **
','
**   
**  *
   *
  *
 *
*  **
   **
','
 ***
*   *
*   *
*****
*   *
*   *
*   *
','
****
*   *
*   *
****
*   *
*   *
****
','
 ***
*   *
*
*
*
*   *
 ***
','
****
*   *
*   *
*   *
*   *
*   *
****
','
*****
*
*
****
*
*
*****
','
*****
*
*
****
*
*
*
','
 ***
*   *
*
* ***
*   *
*   *
 ***
','
*   *
*   *
*   *
*****
*   *
*   *
*   *
','
 ***
  *
  *
  *
  *
  *
 ***
','
    *
    *
    *
    *
    *
*   *
 ***
','
*   *
*  *
* *
**
* *
*  *
*   *
','
*
*
*
*
*
*
*****
','
*   *
** **
* * *
*   *
*   *
*   *
*   *
','
*   *
*   *
**  *
* * *
*  **
*   *
*   *
','
 ***
*   *
*   *
*   *
*   *
*   *
 ***
','
****
*   *
*   *
****
*
*
*
','
 ***
*   *
*   *
*   *
*   *
* * *
*  **
 ** *
','
****
*   *
*   *
****
* *
*  *
*   *
','
 ****
*
*
 ***
    *
    *
****
','
*****
  *
  *
  *
  *
  *
  *
','
*   *
*   *
*   *
*   *
*   *
*   *
 *** 
','
*   *
*   *
*   *
*   *
*   *
 * *
  *
','
*   *
*   *
*   *
*   *
* * *
** **
*   *
','
*   *
*   * 
 * * 
  *  
 * * 
*   * 
*   *
','
*   *
*   *
 * *
  *
  *
  *
  *
','
*****
    *
   *
  *
 *
*
*****
','

    *
   *
  *
 *
*

','

 **
 **

 **
 **

');

#
# map standard weather.com icons to custom icons
#
my %Iconmap = ( '1'  => 2, '2'  => 2, '3'  => 2, '4'  => 2, '5'  => 23, 
                '6'  => 27,'7'  => 24,'8'  => 3, '9'  => 1, '10' => 3, 
                '11' => 25,'12' => 1, '13' => 18,'14' => 3, '15' => 19, 
                '16' => 20,'17' => 2, '18' => 26,'19' => 32,'20' => 33, 
                '21' => 10,'22' => 9, '23' => 11,'24' => 11,'25' => 30, 
                '26' => 0, '27' => 14,'28' => 6, '29' => 13,'30' => 5, 
                '31' => 12,'32' => 4, '33' => 13,'34' => 5, '35' => 2, 
                '36' => 31,'37' => 8, '38' => 8, '39' => 7, '40' => 28, 
                '41' => 21,'42' => 22,'43' => 22,'44' => 9, '45' => 15, 
                '46' => 16,'47' => 17,'48' => 29, '0' => 2, 'na' => 9,
                'NA' => 9, 'N/A' => 9 );

#
# Custom weather condition icons (40x32 pixel)
#
#234567890123456789012345678901234567890 - icon 0
my @Icons = ('





              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
',
#234567890123456789012345678901234567890 - icon 1
'





              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************

     *         *        *
     *         *   *    *
    *    *    *    *   *
    *    *    *   *    *    *
   *    *    *    *    *    *
   *         *        *    *
  *         *         *    *
',
#234567890123456789012345678901234567890 - icon 2
'





              ****
           ****  ****
        ****        **
       **            **
      **              **
     **     **         ** ***
     *      **          *** **
  ****     **            *   **
 **        **                 *
**        **                  **
*         **                   ***
*        ** **                   **
*        ******                   *
**       *** **                  **
 ***        **                  **
   ******************************
           **
          **
     *    **   *        *
     * * **    *   *    *
    *  ****   *    *   *
    *  *****  *   *    *    *
   *    **   *    *    *    *
   *    *    *        *    *
  *         *         *    *
',
#234567890123456789012345678901234567890 - icon 3
'





              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************

                           * *
      * *                *  *  *
    *  *  *               * * *
     * * *      * *      * *** *
    * *** *   *  *  *     * * *
     * * *     * * *     *  *  *
    *  *  *   * *** *      * *
      * *      * * *
              *  *  *
                * *
',
#234567890123456789012345678901234567890 - icon 4
'
                     *
        *            *
         *          *
          *         *
          *         *
           *       *          *
            *      *         *
                           **
                          *
 *            *****      *
  **        *********
    **     ***********
      *   *************
          *************
         ***************
         ***************
         ***************  *******
         ***************
         ***************
    ***   *************
****      *************
           ***********
            *********
              *****      *
                          *
           *               **
          *       *          *
         *        *           *
         *        *
        *          *
       *           *
                   *
',
#234567890123456789012345678901234567890 - icon 5
'
                     *
        *            *
         *          *
          *         *
          *         *
           *       *          *
            *      *         *
                           **
                          *
 *            *****      *
  **        *********
    **     *********** ****
      *   **********  **   *
          *********         **
         *********           * **
         *******              *  *
         ******                  **
         *****                    **
         *****                     *
    ***   *****                    *
****      *************************
           ***********
            *********
              *****      *
                          *
           *               **
          *       *          *
         *        *           *
         *        *
        *          *
       *           *
                   *
',
#234567890123456789012345678901234567890 - icon 6
'
                  *       *        *
                   *      *       *
                   *      *       *
                    *     *      *
                                *
              ****     *******
           ****  **** *********
        ****        ************     **
       **            ************  **
      **              ***********
     **                **********
     *                  *** *****
  ****                   *   ****
 **                           *** *
**                            ***  **
*                              ***   **
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
',
#234567890123456789012345678901234567890 - icon 7
'
                  *       *        *
                   *      *       *
                   *      *       *
                    *     *      *
                                *
              ****     *******
           ****  **** *********
        ****        ************     **
       **            ************  **
      **              ***********
     **                **********
     *                  *** *****
  ****                   *   ****
 **                           *** *
**                            ***  **
*                              ***   **
*                                **
*                                 *
**                               **
 ***                            **
   ******************************

     *         *        *
     *         *   *    *
    *    *    *    *   *
    *    *    *   *    *    *
   *    *    *    *    *    *
   *         *        *    *
  *         *         *    *
',
#234567890123456789012345678901234567890 - icon 8
'
                  *       *        *
                   *      *       *
                   *      *       *
                    *     *      *
                                *
              ****     *******
           ****  **** *********
        ****        ************     **
       **            ************  **
      **              ***********
     **     **         **********
     *      **          *** *****
  ****     **            *   ****
 **        **                 *** *
**        **                  ***  **
*         **                   ***   **
*        ** **                   **
*        ******                   *
**       *** **                  **
 ***        **                  **
   ******************************
           **
          **
     *    **   *        *
     * * **    *   *    *
    *  ****   *    *   *
    *  *****  *   *    *    *
   *    **   *    *    *    *
   *    *    *        *    *
  *         *         *    *
',
#234567890123456789012345678901234567890 - icon 9
'






                *****
              *********
             ***    ***
             *       ***
                     ***
                     ***
                    ****
                   ****
                  ****
                 ****
                 ***
                ***
                ***
                ***
                ***


                ***
                ***
                ***
',
#234567890123456789012345678901234567890 - icon 10
'




    
 
                               
**   **    *     ****** **   **
**   **   ***        ** **   **
**   **   * *       **   ** **
*******  **  *     **     ***
**   **  *****    **       *
**   **  *   *   **        *
**   ** *     * **         *
**   ** *     * ********   *
  
  *     *     *    *     *                
    *      *     *    *      *  
 *     *  *       *       * 
     *       *         *    *      
      *            *        
    *       *           *

  *       *       *        *              
                
      *       *        *          
                 
           *       *         
',
#234567890123456789012345678901234567890 - icon 11 - Windy - Thanks Yannzola
'
                       *
        *****          **
       **   **         **
      **     *         ******
      *      ****      ******
      *         **     ******
   ****          *     ******
  **           ******* **********
 **                    **********
 *                     **********
 *          ********** **********
 **                    **************
  ***                  **************
    ****************** **************
                       **************
                       ****************
         *******       *****************
        **     **     **
        *       **   **
       **    *   *  **
       *    **     **
       *    *     **
   *****    **   **
  **         ** **   ************
 **           ***
 *
 *
 *             ************************
 *
 **
  ***
    *****************************
',
#234567890123456789012345678901234567890 - icon 12
'
                     
        
         
          
          
           
          
                     
                          
              *****      
            **     **
           *     *   *
          *     * **  *
          *           *
         *          *  *
         *    *        *
         *   **   *    *  
         **   *  ***   *
         ***     *    *
          **    **    *
          ** ***      *
           ****   *  *
            **  ** **
              *****      
                          
          
         
      
        
        
       
                   
',
#234567890123456789012345678901234567890 - icon 13
'
                     
        
         
          
          
           
          
                     
                          
              *****      
            **     **
           *        ** ****
          *        *  **   *
          *       *         **
         *      **           * **
         *     *              *  *
         *    *                  **
         *   *                    **
         *   *                     *
          *   *                    *
          *   *********************
           *         *
            **     **
              *****      
                          
          
         
      
        
        
       
                   
',
#234567890123456789012345678901234567890 - icon 14
'
                  
                   
                   
                               
                         *****       
              ****     **     **
           ****  **** *         *
        ****        **           *     
       **            **          *  
      **              **         *
     **                ** ***    *
     *                  *** **   *
  ****                   *   *  *
 **                           ** 
**                            **   
*                              ***   
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
',
#234567890123456789012345678901234567890 - icon 15
'
                  
                   
                   
                               
                         *****       
              ****     **     **
           ****  **** *         *
        ****        **           *     
       **            **          *  
      **              **         *
     **                ** ***    *
     *                  *** **   *
  ****                   *   *  *
 **                           ** 
**                            **   
*                              ***   
*                                **
*                                 *
**                               **
 ***                            **
   ******************************

     *         *        *
     *         *   *    *
    *    *    *    *   *
    *    *    *   *    *    *
   *    *    *    *    *    *
   *         *        *    *
  *         *         *    *
',
#234567890123456789012345678901234567890 - icon 16
'
                  
                   
                   
                               
                         *****       
              ****     **     **
           ****  **** *         *
        ****        **           *     
       **            **          *  
      **              **         *
     **                ** ***    *
     *                  *** **   *
  ****                   *   *  *
 **                           ** 
**                            **   
*                              ***   
*                                **
*                                 *
**                               **
 ***                            **
   ******************************

                           * *
      * *                *  *  *
    *  *  *               * * *
     * * *      * *      * *** *
    * *** *   *  *  *     * * *
     * * *     * * *     *  *  *
    *  *  *   * *** *      * *
      * *      * * *
              *  *  *
                * *
',
#234567890123456789012345678901234567890 - icon 17
'
                  
                   
                   
                               
                         *****       
              ****     **     **
           ****  **** *         *
        ****        **           *     
       **            **          *  
      **              **         *
     **     **         ** ***    *
     *      **          *** **   *
  ****     **            *   *  *
 **        **                 ** 
**        **                  **   
*         **                   ***   
*        ** **                   **
*        ******                   *
**       *** **                  **
 ***        **                  **
   ******************************
           **
          **
     *    **   *        *
     * * **    *   *    *
    *  ****   *    *   *
    *  *****  *   *    *    *
   *    **   *    *    *    *
   *    *    *        *    *
  *         *         *    *
',
#234567890123456789012345678901234567890 - icon 18
'





              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************

                         * *
                       *  *  *
                        * * *
                       * *** *
                        * * *
                       *  *  *
                         * *
                    
                     
                   
',
#234567890123456789012345678901234567890 - icon 19
'












                    * *
                  *  *  *
                   * * *
                  * *** *
                   * * *
                  *  *  * 
                    * *    * *
      * *                *  *  *
    *  *  *               * * *
     * * *      * *      * *** *
    * *** *   *  *  *     * * *
     * * *     * * *     *  *  *
    *  *  *   * *** *      * *
      * *      * * *
              *  *  *
                * *




',
#234567890123456789012345678901234567890 - icon 20
'




              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
        *  *  *     * * *
         * * *     *  *  *
        * *** *      * *    * *
   * *   * * *            *  *  *
 *  *  **  *  *            * * *
  * * *   * *   * *       * *** *
 * *** *      *  *  *      * * *
  * * *        * * *      *  *  *
 *  *  *      * *** *       * *
   * *         * * *
              *  *  *
                * *
',
#234567890123456789012345678901234567890 - icon 21
'
                  *       *        *
                   *      *       *
                   *      *       *
                    *     *      *
                                *
              ****     *******
           ****  **** *********
        ****        ************     **
       **            ************  **
      **              ***********
     **                **********
     *                  *** *****
  ****                   *   ****
 **                           *** *
**                            ***  **
*                              ***   **
*                                **
*                                 *
**                               **
 ***                            **
   ******************************

                           * *
      * *                *  *  *
    *  *  *               * * *
     * * *      * *      * *** *
    * *** *   *  *  *     * * *
     * * *     * * *     *  *  *
    *  *  *   * *** *      * *
      * *      * * *
              *  *  *
                * *   
',
#234567890123456789012345678901234567890 - icon 22
'

              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
  * *** *         *  *  *   * * *
   * * *           * * *   *  *  *
  *  *  *         * *** *    * *
    * *  *  *  *   * * *
          * * *   *  *  *
         * *** *    * *     * *
   * *    * * *           *  *  *
 *  *  * *  *  *           * * *
  * * *    * *  * *       * *** *
 * *** *      *  *  *      * * *
  * * *        * * *      *  *  *
 *  *  *      * *** *       * *
   * *         * * *
              *  *  *
                * *
',
#234567890123456789012345678901234567890 - icon 23
'

              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
   * *** *    *    *  *  *       
    * * *     *     * * *     *        
   *  *  *   *     * *** *   *         
     * *     *      * * *    *      
  *         *      *  *  *  *    
 *          *        * *    *
    * *                   
  *  *  *    *             
   * * *    *    * *       
  * *** *   *  *  *  *   *
   * * *   *    * * *    *
  *  *  *  *   * *** *  *    
 *  * *   *     * * *   *     
*         *    *  *  * *      
*                * *   *     
',
#234567890123456789012345678901234567890 - icon 24
'

              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
      *                 *  *         
      *  *  *  *  *          *          
  *  *   *   * * *     *  * *  *
 *   *  *   * *** *         *  *    
 *      *    * * *   *  *  *  *
*      *    *  *  *        *    
*  *   *      * *   *  *  *  
   *  *                      *
  *       * *      *  *     *
  *   * *  *  *         *   * 
 *    *  * * *   *  *  *   *  
 *   *  * *** *        *   *  
*    *   * * *  *  *  *   *
    *   *  *  *       *     
    *     * *  *  *  *       
',
#234567890123456789012345678901234567890 - icon 25
'





              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
       *    *    *    *    *         
                             
       *    *    *    *    * 
                           
     *    *    *    *    *  
                           
     *    *    *    *    *  
                           
    *         *        *   
                                             
                      
',
#234567890123456789012345678901234567890 - icon 26
'




              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
     *     *     *     *     *         
        *     *     *     *      
     *     *     *     *     * 
        *     *     *     *    
     *     *     *     *     *  
        *     *     *     *   
     *     *     *     *     *  
        *     *     *     *   
     *     *     *     *     *   
        *     *     *     *    
                       
                      
',
#234567890123456789012345678901234567890 - icon 27
'





              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
         *   *  *   *     *   *
     *     *   * *    * *
     * *     * *   *    *  * *
    *    * *  * *  *   *
    * *  *    *   * *  *  * *
   *    *  * *  * *    *    *
   *         *        *    *
  *  *   *  *   *  *  *  * *                    
',
#234567890123456789012345678901234567890 - icon 28
'





              ****
           ****  ****
        ****        **
       **            **
      **              **
     **                ** ***
     *                  *** **
  ****                   *   **
 **                           *
**                            **
*                              ***
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
       *    *    *    *     *
    *  *    *    *    * *   *
    * *    *  * * *  *  *  *
   *  * *  * *  * *  * *   *
   * *  * *  * * *  *  *  *  *
  *  * *  * *  * *  *  *  *  *
  * *  *    *   *  *  *  *  *
 *    *    *    *     *     *
',
#234567890123456789012345678901234567890 - icon 29
'
                  *       *        *
                   *      *       *
                   *      *       *
                    *     *      *
                                *
              ****     *******
           ****  **** *********
        ****        ************     **
       **            ************  **
      **              ***********
     **                **********
     *                  *** *****
  ****                   *   ****
 **                           *** *
**                            ***  **
*                              ***   **
*                                **
*                                 *
**                               **
 ***                            **
   ******************************
       *    *    *    *     *
    *  *    *    *    * *   *
    * *    *  * * *  *  *  *
   *  * *  * *  * *  * *   *
   * *  * *  * * *  *  *  *  *
  *  * *  * *  * *  *  *  *  *
  * *  *    *   *  *  *  *  *
 *    *    *    *     *     *
',
#234567890123456789012345678901234567890 - icon 30 Frigid - Thanks Yannzola
'


    *****                 *
   **   **                *
   *     *            *   *   *
   * *** *            *** * ***
   * *   *         *    *****    *
   * **  *         *     ***     *
   * *   *         *      *      *
   * *** *    ***  *      *      *  ***
   * *   *      ****      *      ****
   * **  *        ***     *      **
   * *   *      *** ***   *   *** ***
   * *** *    ***     *** * ***     ***
   * *   *              *****
   * **  *               ***
   * *   *              *****
   * *** *    ***     *** * ***     ***
   * *   *      *** ***   *   *** ***
   * **  *        ***     *     ***
   * *   *      ****      *      ****
   * *** *    ***  *      *      *  ***
  ** *** **        *      *      *
  * ***** *        *     ***     *
  * ***** *        *    *****    *
  * ***** *           *** * ***
  ** *** **           *   *   *
   **   **                *
    *****                 *
',
#234567890123456789012345678901234567890 - icon 31 HOT - Thanks Yannzola
'

                         *
    *****                *
   **   **               *
   *     *    *          *          *
   * *** *     *         *         *
   * *   *      *        *        *
   * *** *       *               *
   * *** *        *    *****    *
   * *** *           *********
   * *** *          ***********
   * *** *         *************
   * *** *         *************
   * *** *        ***************
   * *** *        ***************
   * *** * ****** *************** ******
   * *** *        ***************
   * *** *        ***************
   * *** *         *************
   * *** *         *************
   * *** *          ***********
   * *** *           *********
  ** *** **       *    *****    *
  * ***** *      *               *
  * ***** *     *        *        *
  * ***** *    *         *         *
  ** *** **   *          *          *
   **   **               *
    *****                *
                         *
',
#234567890123456789012345678901234567890 - icon 32 Dust - Thanks Yannzola
'

                 *******
                **     **
               **  *    *
               *        **
          ****** *  *    *
         **            * *
        **         *     *****
        *    *   *   *       **
        *  *             *    **
    *****                   *  *
   **        *   ******  *     *
  **   *  *     **    **       *
  *            **      **      *
  *       *  * *   *    *  *****
  *  *  *      *        ****   **
  *         ****    *           **
  **   *   **     *    *  *   *  *
   ***    **  *             *    *
     ******           *   *      ****
          *     * *           *     **
          **            *            **
           ***      *     *       *   *
             ******          *        *
                  *    *        *    **
                  *  *     ****    ***
                  **   *  **  ******
                   **    **
                    ******
',
#234567890123456789012345678901234567890 - icon 33 Fog - Thanks Yannzola
'
   ***********************************



   ***********************************



   ***********************************
                *********
               ***********

   ***********************************
             ***************
             ***************

   ***********************************
             ***************
   ***********************************

   ***********************************
                *********
   ***********************************

   ***********************************

   ***********************************

   ***********************************

   ***********************************
');

my $TWClogo = '
       ************************
       ************************
       ************************
       ************************
       ************************
       ************************
       *    *******************
       ** *  *** **************
       ** * * *   *************
       ** * * * ***************
       ** * * **  *************
       ************************
       * * * ****** ** ********
       * * * * *  *  *  ** *  *
       * * *    **  ** *     **
       ** * * **    ** *  ** **
       ** * **       * * *   **
       ************************
       **   **************** **
       * **  *  *   *   * ** **
       * ** * **  * * *    * **
       * ** *     * * *  *** **
       **   *     * * * *  ** *
       ************************



            * *
* * * * ** *****   *  **   **  * ** *
* * ****  * * * * *** *   *   * ** * *
** ***  *** * * * *   *   *   * ** * *
 * *  ***** *** *  ** * ** **  * * * *
';

our %mapping = ('arrow_up' => 'up',
					'arrow_right' => 'right',
					'arrow_left' => 'left',
					'knob_right' => 'up',
					'knob_left' => 'down',
					'knob_right.repeat' => 'up',
					'knob_left.repeat' => 'down',
					'add.single' => 'refresh',
					'arrow_down' => 'down',
					'arrow_down.hold' => 'down.hold',
					'fwd' => 'wnext',
					'rew' => 'wprev',
					'knob' => 'done',
					'size.hold'  => 'size.hold');


##################################################
### Super Functions ###
##################################################
sub initPlugin {
	my $class = shift;
	$class->SUPER::initPlugin();
	
	$log->info("Initializing");

	Plugins::SuperDateTime::Settings->new;
	Plugins::SuperDateTime::PlayerSettings->new;

	Slim::Buttons::Common::addSaver('SCREENSAVER.superdatetime', 
	                                 getScreensaverSuperDatetime(), 
	                                 \&setScreensaverSuperDateTimeMode,
	                                 \&leaveScreensaverSuperDateTimeMode,
	                                 getDisplayName()
	                               );

	Slim::Hardware::IR::addModeDefaultMapping('SCREENSAVER.superdatetime',\%mapping);
	Slim::Hardware::IR::addModeDefaultMapping('OFF.superdatetime',\%mapping);

	# Get previous settings or set default
	if ($prefs->get('city') eq '') {
		$prefs->set('city','60614'); #Default to Chicago... cuz Chicago is where it's at!
	}

	if ($prefs->get('stock1format') eq '') {
		$prefs->set('stock1format','%n');
	}

	if ($prefs->get('stock2format') eq '') {
		$prefs->set('stock2format','%l %c %z %v');
	}

	if ($prefs->get('refresh') eq '') {
		$prefs->set('refresh','5');
	}

	if ($prefs->get('time') eq '') {
		$prefs->set('time', $showtime);
	}

	if ($prefs->get('score') eq '') {
		$prefs->set('score', $showgame);
	}

	if ($prefs->get('atime') eq '') {
		$prefs->set('atime', $showactivetime);
	}

	if ($prefs->get('ascore') eq '') {
		$prefs->set('ascore', $showactivegame);
	}

	if ($prefs->get('offset') eq '') {
		$prefs->set('offset', '00');
	}

	if ($prefs->get('temperature') eq '') {
		$prefs->set('temperature', 0);
	}

	if ($prefs->get('windunit') eq '') {
		$prefs->set('windunit', 0);
	}

	if ($prefs->get('teamlogos') eq '') {
		$prefs->set('teamlogos', 2);
	}

	if ($prefs->get('lweather') eq '') {
		$prefs->set('lweather', 1);
	}

	if ($prefs->get('cbballconf') eq '') {
		$prefs->set('cbballconf', 0);
	}

	if ($prefs->get('cfballconf') eq '') {	
		$prefs->set('cfballconf', 0);
	}

	$showtime = $prefs->get('time');
	$showactivetime = $prefs->get('atime');

	$showgame = $prefs->get('score');
	$showactivegame = $prefs->get('ascore');
	
	@MLBteams = @{ $prefs->get('mlb') || [] };
	if (scalar(@MLBteams) == 0) {
		$prefs->set('mlb', \@MLBteams);
    }

	@NBAteams = @{ $prefs->get('nba') || [] };
	if (scalar(@NBAteams) == 0) {
		$prefs->set('nba', \@NBAteams);
    }

	@NHLteams = @{ $prefs->get('nhl') || [] };
	if (scalar(@NHLteams) == 0) {
		$prefs->set('nhl', \@NHLteams);
    }

	@NFLteams = @{ $prefs->get('nfl') || [] };
	if (scalar(@NFLteams) == 0) {
		$prefs->set('nfl', \@NFLteams);
	}

	@CBBteams = @{ $prefs->get('cbb') || [] };
	if (scalar(@CBBteams) == 0) {
		$prefs->set('cbb', \@CBBteams);
	}

	@CFBteams = @{ $prefs->get('cfb') || []};
	if (scalar(@CFBteams) == 0) {
		$prefs->set('cfb', \@CFBteams);
	}
	
	Slim::Control::Request::addDispatch(['sdtTop'],[1, 1, 0, \&sdtTop]);
	Slim::Control::Request::addDispatch(['sdtCurrent'],[1, 1, 0, \&sdtCurrent]);
	Slim::Control::Request::addDispatch(['sdtForecast'],[1, 1, 0, \&sdtForecast]);
	Slim::Control::Request::addDispatch(['sdtForecastDetail','_forecastNum'],[1, 1, 0, \&sdtForecastDetail]);
	Slim::Control::Request::addDispatch(['sdt10day'],[1, 1, 0, \&sdt10day]);
	Slim::Control::Request::addDispatch(['sdtScores'],[1, 1, 0, \&sdtScores]);
	Slim::Control::Request::addDispatch(['sdtGameList','_sport'],[1, 1, 0, \&sdtGameList]);
	
	my @menu = ({
		text   => 'SuperDateTime',
		id     => 'pluginSuperDateTime',
		weight => 15,
		actions => {
				go => {
					player => 0,
					cmd	 => [ 'sdtTop' ],
				}
		},
	});

	Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + 75, \&duetCheck);

	Slim::Control::Jive::registerPluginMenu(\@menu, 'extras');
}

sub sdtTop {
	my $request = shift;
	my $client = $request->client();
	
	my @menu = ();

	push @menu, {
		text => 'Current Conditions',
		window => { menuStyle => 'album' },
        actions  => {
          go  => {
              player => 0,
              cmd    => [ 'sdtCurrent' ],
              params => {
              	menu => 'sdtCurrent',
              },
          },
        },
	};
	
	push @menu, {
		text => 'Forecast',
		window => { menuStyle => 'album' },
        actions  => {
          go  => {
              player => 0,
              cmd    => [ 'sdtForecast' ],
              params => {
              	menu => 'stdForecast',
              },
          },
        },
	};
	
	push @menu, {
		text => '10-Day Forecast',
		window => { menuStyle => 'album' },
        actions  => {
          go  => {
              player => 0,
              cmd    => [ 'sdt10day' ],
              params => {
              	menu => 'sdt10day',
              },
          },
        },
	};	
	
	push @menu, {
		text => 'Scores',
		window => { menuStyle => 'album' },
        actions  => {
          go  => {
              player => 0,
              cmd    => [ 'sdtScores' ],
              params => {
              	menu => 'sdtScores',
              },
          },
        },
	};		
	
	my $numitems = scalar(@menu);
	
	$request->addResult("count", $numitems);
	$request->addResult("offset", 0);
	my $cnt = 0;
	for my $eachPreset (@menu[0..$#menu]) {
		$request->setResultLoopHash('item_loop', $cnt, $eachPreset);
		$cnt++;
	}
	
	$request->setStatusDone();
}

sub duetCheck {
	#Look for any players that don't have a screen.  This implies they probably have a Duet that they want to utilize the plugin with
	#Duets are always active, so update plugin accordingly
	my @players = Slim::Player::Client::clients();
	$log->info("No active SDT clients so checking for duet only configuration.");
	if ($activeClients == 0) {
		for my $player (@players) {
			if ($player->display->isa('Slim::Display::NoDisplay')) {
					$activeClients++;
					Slim::Utils::Timers::killTimers(undef, \&refreshData); #Paranoia check
					Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + 5, \&refreshData, $player, -1);
					$log->info("Non-display player found, configuring SDT to always refresh data.");
			}
		}
	}
}

sub sdt10day {
	my $request = shift;
	my $client = $request->client();
	
	my @menu = ();
	my $i=1;
	
	while ($i <10) {
		push @menu, {
			'icon-id' => 'plugins/SuperDateTime/html/images/'.$wetData{'d'.$i}{'forecastIcon'}.'_52x52_p.gif',
			text => $wetData{'d'.$i}{'day'}. '. '. $wetData{'d'.$i}{'date'}."\n".
						$wetData{'d'.$i}{'highF'} . '/' . $wetData{'d'.$i}{'lowF'} . ' Precip. '. $wetData{'d'.$i}{'precip'} . "\n".
						$wetData{'d'.$i}{'condition'},
		};
				
		$i++;
	}
	
	my $numitems = scalar(@menu);
	
	$request->addResult("base", {window => { titleStyle => 'album' }});	
	$request->addResult("count", $numitems);
	$request->addResult("offset", 0);
	my $cnt = 0;
	for my $eachPreset (@menu[0..$#menu]) {
		$request->setResultLoopHash('item_loop', $cnt, $eachPreset);
		$cnt++;
	}
	
	$request->setStatusDone();
}

sub sdtForecast {
	my $request = shift;
	my $client = $request->client();
	
	my @menu = ();
	my $i = 0;

	while ($i <3) {
		my $hilo;
		if ($wetData{$i}{'forecastType'} eq 'HIGH') {
			$hilo = 'High ';
		}
		else {
			$hilo = 'Low ';
		}
	
		push @menu, {
			'icon-id' => 'plugins/SuperDateTime/html/images/'.$wetData{$i}{'forecastIcon'}.'_52x52_p.gif',
			text => $wetData{$i}{'forecastTOD'}."\n".
						$hilo . $wetData{$i}{'forecastTempF'} . ' Precip. '. $wetData{$i}{'forecastPrec'} . "\n".
						$wetData{$i}{'skyCondition'},
         actions  => {
           go  => {
               player => 0,
               cmd    => [ 'sdtForecastDetail', $i ],
               params => {
               	menu => 'nowhere',
               },
           },
         },
		};
				
		$i++;
	}
		
	my $numitems = scalar(@menu);
	
	$request->addResult("base", {window => { titleStyle => 'album' }});	
	$request->addResult("count", $numitems);
	$request->addResult("offset", 0);
	my $cnt = 0;
	for my $eachPreset (@menu[0..$#menu]) {
		$request->setResultLoopHash('item_loop', $cnt, $eachPreset);
		$cnt++;
	}
	
	$request->setStatusDone();
}

sub sdtForecastDetail {
	my $request = shift;
	my $client = $request->client();
	my $forecastNum = $request->getParam('_forecastNum');

	my @menu = ();
	
	push @menu, {
		text => $WETdisplayItems2[$forecastNum],
	};

	my $numitems = scalar(@menu);

	$request->addResult("count", $numitems);
	$request->addResult("offset", 0);
	my $cnt = 0;
	for my $eachItem (@menu[0..$#menu]) {
		$request->setResultLoopHash('item_loop', $cnt, $eachItem);
		$cnt++;
	}
	$request->setStatusDone();
}

sub sdtCurrent {
	my $request = shift;
	my $client = $request->client();
	
	my @menu = ();
	my $i = 0;

	my $hilo;
	if ($wetData{$i}{'forecastType'} eq 'HIGH') {
		$hilo = 'High';
	}
	else {
		$hilo = 'Low';
	}
	
	push @menu, {
		'icon-id' => 'plugins/SuperDateTime/html/images/'.$wetData{-1}{'forecastIcon'}.'_52x52_p.gif',
			  text => "$wetData{'temperatureF'}($wetData{'feelslikeF'}) $wetData{'windspeed_mh'}"."\n".
						"$hilo $wetData{-1}{'forecastTempF'} Precip. $wetData{-1}{'forecastPrec'}" . "\n".
						$wetData{-1}{'skyCondition'}."\n".
						"Humidity: $wetData{'humidity'}"."\n".
						"Pressure: $wetData{'pressureIN'}$wetData{'pressureT'}"."\n".
						"Dewpoint: $wetData{'dewpointF'}",
         actions  => {
           go  => {
               player => 0,
               cmd    => [ 'sdtForecastDetail', -1 ],
               params => {
               	menu => 'nowhere',
               },
           },
         },
	};
		
	my $numitems = scalar(@menu);
	
	$request->addResult("base", {window => { titleStyle => 'album' }});	
	$request->addResult("count", $numitems);
	$request->addResult("offset", 0);
	my $cnt = 0;
	for my $eachItem (@menu[0..$#menu]) {
		$request->setResultLoopHash('item_loop', $cnt, $eachItem);
		$cnt++;
	}
	
	$request->setStatusDone();
}

sub sdtScores {
	my $request = shift;
	my $client = $request->client();
	
	my @menu = ();
	for my $key ( sort keys %sportsData ) {
		my $icon;
		if ($key eq 'College Basketball' || $key eq 'College Football') {
			$icon = 'plugins/SuperDateTime/html/images/NCAA.png'
		}
		else {
			$icon = 'plugins/SuperDateTime/html/images/'.$key.'.png'
		}
		
		push @menu, {
				'icon-id' => $icon,
				text => $key,
				window => { menuStyle => 'album' },
				actions  => {
				           go  => {
				               player => 0,
				               cmd    => [ 'sdtGameList', $key ],
				               params => {
				               	menu => 'nowhere',
				               },
				           },
         },
		}
		
	}	
	
	my $numitems = scalar(@menu);
	
	$request->addResult("base", {window => { titleStyle => 'album' }});	
	$request->addResult("count", $numitems);
	$request->addResult("offset", 0);

	my $cnt = 0;
	for my $eachItem (@menu[0..$#menu]) {
		$request->setResultLoopHash('item_loop', $cnt, $eachItem);
		$cnt++;
	}
	
	$request->setStatusDone();
}

sub sdtGameList {
	my $request = shift;
	my $client = $request->client();
	my $sport = $request->getParam('_sport');

	my @menu = ();
	my $hashRef = \%sportsData;
	
	for my $key (sort keys %{$hashRef->{$sport}} ) {
		#print "$key => $value\n";

		push @menu, {
			text => "$sportsData{$sport}{$key}{'awayTeam'} $sportsData{$sport}{$key}{'awayScore'} \n $sportsData{$sport}{$key}{'homeTeam'} $sportsData{$sport}{$key}{'homeScore'} \n $sportsData{$sport}{$key}{'gameTime'}",
		};
	}	

	my $numitems = scalar(@menu);

	$request->addResult("count", $numitems);
	$request->addResult("offset", 0);
	my $cnt = 0;
	for my $eachItem (@menu[0..$#menu]) {
		$request->setResultLoopHash('item_loop', $cnt, $eachItem);
		$cnt++;
	}
	$request->setStatusDone();
}

sub FtoC {
	my $temp = shift;

	$temp = ($temp-32)*5/9;
	$temp = int($temp + .5 * ($temp <=> 0)); #Funky round	
	
	return $temp;
}

sub getAverages {  #Set up Async HTTP request for averages
	my ($days_advance, $client, $refreshItem) = @_;

	if (!defined $days_advance) {
		$days_advance = 0;
	}

	if ($wetData{'0'}{'forecastTOD'} ne $averages{'last'}) { #See if the averages & 10day need to be refreshed due to a period change    
		my $dayNum = "";
		if ($days_advance > 0) {
			$dayNum = "?dayNum=$days_advance";
  		}

		my $url = 'http://www.weather.com/weather/wxdetail/' . $prefs->get('city') . $dayNum;
  		# tomorrow ### http://www.weather.com/weather/wxdetail/95616?dayNum=1

  		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotAverages,
																		\&gotErrorViaHTTP,
																		{caller => 'getAverages',
																		 dayNum => $days_advance,
							   										 client => $client,
							   										 refreshItem => $refreshItem});

		$log->info("async request: $url");
		$http->get($url);
	}
	else {
		$log->info("Skipping averages...");
		refreshData(undef, $client, $refreshItem);
	}
}

sub gotAverages {  #Average data was received
	my $http = shift;
	
	my $params = $http->params();
	my $client = $params->{'client'};
  	my $refreshItem = $params->{'refreshItem'};
  
	$log->info("got " . $http->url());
	my $dayNum = 0;

	if ($http->url() =~ m/\?dayNum=(\d+)/) {
		$dayNum = $1;
	}

	my $content = $http->content();
	  
	if ($content =~ m/Avg. High:<\/TD>.*?<B>(-?\d+)(F|C)<\/B><\/TD>/s) {
		$averages{$dayNum}{'average_high_F'} = $1;
	}

	if ($content =~ m/Record High:<\/TD>.*?<B>(-?\d+)(F|C) \((\d+)\)<\/B><\/TD>/s) {
		$averages{$dayNum}{'record_high_F'} = $1;
		$averages{$dayNum}{'record_high_year'} = $3;
	}
  
	if ($content =~ m/Avg. Low:<\/TD>.*?<B>(-?\d+)(F|C)<\/B><\/TD>/s) {
		$averages{$dayNum}{'average_low_F'} = $1;
	}
  
	if ($content =~ m/Record Low:<\/TD>.*?<B>(-?\d+)(F|C) \((\d+)\)<\/B>/s) {
		$averages{$dayNum}{'record_low_F'} = $1;
		$averages{$dayNum}{'record_low_year'} = $3;
	}
	
	if ($content =~ m/Sunrise:<\/TD>.*?<B>(\d+:\d+) (AM|PM).*<\/B>/s) {
		$averages{$dayNum}{'sunrise'} = "$1 $2";
	}
	
	if ($content =~ m/Sunset:<\/TD>.*?<B>(\d+:\d+) (AM|PM).*<\/B>/s) {
		$averages{$dayNum}{'sunset'} = "$1 $2"
	}	

	if ($dayNum == 0) { #Today's was received okay, let's get tomorrows
		getAverages(1, $client, $refreshItem);
	}
	else { #Just received tomorrow's averages
		for (my $i = 0; $i <= 2; $i++) { #Update average info for each wetData period
			my $name = $wetData{$i}{'forecastTOD'};
		
			$wetData{-1}{'average_F'} = $averages{0}{'average_high_F'};
			$wetData{-1}{'average_C'} = FtoC($averages{0}{'average_high_F'});
			
			$wetData{-1}{'record_F'} = $averages{0}{'record_high_F'};
			$wetData{-1}{'record_C'} = FtoC($averages{0}{'record_high_F'});
			
			$wetData{-1}{'record_year'} = $averages{0}{'record_high_year'};
			
			$wetData{-1}{'sunrise'} = $averages{0}{'sunrise'};
			$wetData{-1}{'sunset'}  = $averages{0}{'sunset'};

			if ($name eq 'Today') {
				$wetData{$i}{'average_F'} = $averages{0}{'average_high_F'};
				$wetData{$i}{'average_C'} = FtoC($averages{0}{'average_high_F'});

				$wetData{$i}{'record_F'} = $averages{0}{'record_high_F'};
				$wetData{$i}{'record_C'} = FtoC($averages{0}{'record_high_F'});
			
				$wetData{$i}{'record_year'} = $averages{0}{'record_high_year'};
				
				$wetData{$i}{'sunrise'} = $averages{0}{'sunrise'};
				$wetData{$i}{'sunset'}  = $averages{0}{'sunset'};
			}
			elsif ($name eq 'Tonight') {
				$wetData{$i}{'average_F'} = $averages{0}{'average_low_F'};
				$wetData{$i}{'average_C'} = FtoC($averages{0}{'average_low_F'});
			
				$wetData{$i}{'record_F'} = $averages{0}{'record_low_F'};
				$wetData{$i}{'record_C'} = FtoC($averages{0}{'record_low_F'});
			
				$wetData{$i}{'record_year'} = $averages{0}{'record_low_year'};
				
				$wetData{$i}{'sunrise'} = $averages{0}{'sunrise'};
				$wetData{$i}{'sunset'}  = $averages{0}{'sunset'};				
			}
			elsif ($name eq 'Tomorrow') {
				$wetData{$i}{'average_F'} = $averages{1}{'average_high_F'};
				$wetData{$i}{'average_C'} = FtoC($averages{1}{'average_high_F'});
				
				$wetData{$i}{'record_F'} = $averages{1}{'record_high_F'};
				$wetData{$i}{'record_C'} = FtoC($averages{1}{'record_high_F'});
			
				$wetData{$i}{'record_year'} = $averages{1}{'record_high_year'};
				
				$wetData{$i}{'sunrise'} = $averages{1}{'sunrise'};
				$wetData{$i}{'sunset'}  = $averages{1}{'sunset'};				
			}
			elsif ($name eq 'Tomorrow Night') {
				$wetData{$i}{'average_F'} = $averages{1}{'average_low_F'};
				$wetData{$i}{'average_C'} = FtoC($averages{1}{'average_low_F'});
				
				$wetData{$i}{'record_F'} = $averages{1}{'record_low_F'};
				$wetData{$i}{'record_C'} = FtoC($averages{1}{'record_low_F'});
			
				$wetData{$i}{'record_year'} = $averages{1}{'record_low_year'};

				$wetData{$i}{'sunrise'} = $averages{1}{'sunrise'};
				$wetData{$i}{'sunset'}  = $averages{1}{'sunset'};
			} 
			else {
				$log->warn("unknown name: $name");
			}
		}
	
		$averages{'last'} = $wetData{'0'}{'forecastTOD'}; #update last average update indicator
		refreshData(undef, $client, $refreshItem);
	}
}

sub drawEach {
	my ($timerObj, $client, $period, $formatLoc) = @_;
	    #^Should be undef
	
	my $destLoc = 0;
	
	if (defined $displayInfoBuild{$client}{'TOPdisplayItems1'}) {
		$destLoc = scalar @{$displayInfoBuild{$client}{'TOPdisplayItems1'}};
	}
	
	$log->debug("Drawing scrn $destLoc");
	
	my	@show1icon  = @{ $prefs->client($client)->get('show1icon') || [] };
	my	@show13line = @{ $prefs->client($client)->get('show13line') || [] };
	my @d3line1t   = @{ $prefs->client($client)->get('v3line1t') || [] };
	my @d3line1m   = @{ $prefs->client($client)->get('v3line1m') || [] };
	my @d3line1b   = @{ $prefs->client($client)->get('v3line1b') || [] };
	my @d1period   = @{ $prefs->client($client)->get('v1period') || [] };
	my @weatherformat1t = @{ $prefs->client($client)->get('weatherformat1t') || [] };
	my @weatherformat1b = @{ $prefs->client($client)->get('weatherformat1b') || [] };
	my @weatherformat2t = @{ $prefs->client($client)->get('weatherformat2t') || [] };
	my @weatherformat2b = @{ $prefs->client($client)->get('weatherformat2b') || [] };
	
	clearCanvas($client);
	#$displayInfoBuild{$client}{'btmLinePreSpace'}[$destLoc] = '';
	
	#Draw icons
	if ($show1icon[$formatLoc] == 1) {
		drawIcon($client,0,$ymax{$client}-1, $Icons[$Iconmap{$wetData{$period}{'forecastIcon'}}]);		
		$displayInfoBuild{$client}{'hasIcon'}[$destLoc] = 1;
	}
	
	#Draw 3line stuff
	if ($show13line[$formatLoc] == 1) {
		my $line1 = uc(replaceMacrosPer($d3line1t[$formatLoc], $period));
		my $line2 = uc(replaceMacrosPer($d3line1m[$formatLoc], $period));
		my $line3 = uc(replaceMacrosPer($d3line1b[$formatLoc], $period));		
	
		if ($show1icon[$formatLoc] == 1) {
			drawText($client,42,$ymax{$client}-25, $line3);
			drawText($client,42,$ymax{$client}-13, $line2);
			drawText($client,42,$ymax{$client}-1,  $line1);
		}
		else {
			drawText($client,0,$ymax{$client}-25, $line3);
			drawText($client,0,$ymax{$client}-13, $line2);
			drawText($client,0,$ymax{$client}-1,  $line1);
		}

		#Save the character lengths of each line
		$displayInfoBuild{$client}{'CharLen1'}[$destLoc] = length($line1);
		$displayInfoBuild{$client}{'CharLen2'}[$destLoc] = length($line2);
		$displayInfoBuild{$client}{'CharLen3'}[$destLoc] = length($line3);		
	}
	
	#If necessary can make the width calculation based on max length of 3line text
	$displayInfoBuild{$client}{'forecastG'}[$destLoc] = getFramebuf($client,$xmax{$client});
	
	push(@{$displayInfoBuild{$client}{'TOPdisplayItems1'}},  replaceMacrosPer($weatherformat1t[$formatLoc], $period));	
	push(@{$displayInfoBuild{$client}{'BOTdisplayItems1'}},  replaceMacrosPer($weatherformat1b[$formatLoc], $period));
	push(@{$displayInfoBuild{$client}{'TOPdisplayItems2'}}, replaceMacrosPer($weatherformat2t[$formatLoc], $period));	
	push(@{$displayInfoBuild{$client}{'BOTdisplayItems2'}}, replaceMacrosPer($weatherformat2b[$formatLoc], $period));
	
	#See if all the icons have been drawn. 13 total without customs.  0 based index.  Remove 2 from scalar for defaults
	if ($destLoc == 12 + (scalar @{ $prefs->get('v1period') || [] } -2)) {
		doneDrawing($client);
	}	
}

sub replaceMacros {
	my $string     = shift;

	for ($string) {
		my $date = Slim::Utils::DateTime::longDateF(undef, preferences('plugin.datetime')->get('dateformat'));
		s/%2/$date/;
		
		s/%t/$wetData{'temperatureF'}/;
		s/%T/$wetData{'temperatureC'}/;
		s/%h/$wetData{'humidity'}/;
		s/%p/$wetData{'pressureIN'}$wetData{'pressureT'}/;
		s/%P/$wetData{'pressureMB'}$wetData{'pressureT'}/;
		s/%d/$wetData{'dewpointF'}/;
		s/%D/$wetData{'dewpointC'}/;
		s/%f/$wetData{'feelslikeF'}/;
		s/%F/$wetData{'feelslikeC'}/;
		s/%w/$wetData{'windspeed_mh'}/;
		s/%W/$wetData{'windspeed_kh'}/;
		s/%q/$wetData{'windspeed_kth'}/;
		s/%Q/$wetData{'windspeed_ms'}/;		

		#Wunderground
		s/%e/$wetData{'wu_temperatureF'}/;
		s/%E/$wetData{'wu_temperatureC'}/;
		s/%H/$wetData{'wu_humidity'}/;
		s/%l/$wetData{'wu_pressureIN'}/;
		s/%L/$wetData{'wu_pressureMB'}/;
		s/%m/$wetData{'wu_dewpointF'}/;
		s/%M/$wetData{'wu_dewpointC'}/;
		s/%j/$wetData{'wu_windspeed_mh'}/;
		s/%J/$wetData{'wu_windspeed_kh'}/;
		s/%k/$wetData{'wu_windspeed_kth'}/;
		s/%K/$wetData{'wu_windspeed_ms'}/;		
	}

	return $string;
}

sub replaceMacrosPer {
	my $string  = shift;
	my $location = shift;

	$string = replaceMacros($string);

	for ($string) {
		s/%a/$wetData{$location}{'average_F'}/;
		s/%A/$wetData{$location}{'average_C'}/;
		s/%c/$wetData{$location}{'record_F'}/;
		s/%C/$wetData{$location}{'record_C'}/;
		s/%g/$wetData{$location}{'record_year'}/;
		s/%s/$wetData{$location}{'sunrise'}/;
		s/%S/$wetData{$location}{'sunset'}/;

		s/%z/$wetData{$location}{'forecastType'} $wetData{$location}{'forecastTempF'}/;
		s/%Z/$wetData{$location}{'forecastType'} $wetData{$location}{'forecastTempC'}/;
		s/LOW/ LOW/;
		s/%x/$wetData{$location}{'forecastPrec'}/;
		
		if ($location != -1) { #Make sure it's not a special "currently" case
			s/%y/$wetData{$location}{'forecastTOD'}/;
			s/%v/$wetData{$location}{'skyCondition'}/;
		}
		else { #Current conditions
			s/%y/Currently/;
			s/%v/$wetData{'-1'}{'skyCondition'}/;		
		}
		
		#10day stuff
		s/%_3/$wetData{$location}{'day'}/;
		s/%_4/$wetData{$location}{'date'}/;
		s/%_5/$wetData{$location}{'highF'}/;
		s/%_6/$wetData{$location}{'highC'}/;
		s/%_7/$wetData{$location}{'lowF'}/;
		s/%_8/$wetData{$location}{'lowC'}/;
		s/%_9/$wetData{$location}{'precip'}/;
		s/%_0/$wetData{$location}{'condition'}/;
	}		

	return $string;
}

sub getWeather {  #Set up Async HTTP request for Weather
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;

	my $url = 'http://www.weather.com/weather/local/' . $prefs->get('city');
	my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotWeather,
							  \&gotErrorViaHTTP,
							  {caller => 'getWeather',
							   callerProc => \&getWeather,
							   client => $client,
							   refreshItem => $refreshItem});
	$log->info("async request: $url");
	
	$http->get($url);
}

sub gotWeather {  #Weather data was received
	my $http = shift;
	
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};

	$log->info("got " . $http->url());
	#$::d_plugins && msg("SuperDateTime: content type is " . $http->headers()->{'Content-Type'} . "\n");

	my $content = $http->content();
	my @ary=split /\n/,$content; #break large string into array
      
   #Counter variables
	my $forecastNum = -1; #which of the 3 forecasts currently being processed
	my $forecastTODNum = -1; #which of the 3 forecasts currently being processed for time of day
	my $forecastIconNum = -1;
	
	for (@ary) {
		if (/obsTempTextA"*>(-*\d+)\s*&deg\;F.+Feels Like<BR>\s*(-*\d+)\s*\&d/) {
				$wetData{'temperatureF'} = $1;
				$wetData{'feelslikeF'} = $2;
				$wetData{'temperatureC'} = FtoC($1);       
				$wetData{'feelslikeC'} = FtoC($2);
		}
		elsif (/obsTempTextA"*>(-*\d+)\s*&deg\;F/) { #Some html has actual/feels like separated
			$wetData{'temperatureF'} = $1;
			$wetData{'temperatureC'} = FtoC($1);       
		}
		elsif (/Feels Like<br\/*>\s*(-*\d+)\s*\&d/) { #Some html has actual/feels like separated
			$wetData{'feelslikeF'} = $1;
			$wetData{'feelslikeC'} = FtoC($1);
		}
		elsif (/<TD VALIGN="top"  CLASS="obsTextA">(\d+%)<\/td>/) {
			$wetData{'humidity'} = $1;
		}
		elsif (/class="crowval">(\d+%)<\/div>/) {
			$wetData{'humidity'} = $1;
		}		
		elsif (/From (.+) at (\d+) mph/) {
			$wetData{'windspeed_mh'} = $1 . $2;

			$wetData{'windspeed_kh'} = $2*1.609344;
			$wetData{'windspeed_kh'} = $1 . int($wetData{'windspeed_kh'} + .5 * ($wetData{'windspeed_kh'} <=> 0)); #Funky round

			$wetData{'windspeed_ms'} = $2*16.09344/36;
			$wetData{'windspeed_ms'} = $1 . int($wetData{'windspeed_ms'} + .5 * ($wetData{'windspeed_ms'} <=> 0)); #Funky round
			$wetData{'windspeed_kth'} = $2/1.1515;
			$wetData{'windspeed_kth'} = $1 . int($wetData{'windspeed_kth'} + .5 * ($wetData{'windspeed_kth'} <=> 0)); #Funky round
		}
		elsif (/CLASS="obsTextA">CALM/) {
				$wetData{'windspeed_mh'} = "Calm";
				$wetData{'windspeed_kh'} = "Calm";
				$wetData{'windspeed_ms'} = "Calm";
				$wetData{'windspeed_kth'} = "Calm";
		}
		elsif (/Wind:<\/div><div class="crowval">CALM/) {
				$wetData{'windspeed_mh'} = "Calm";
				$wetData{'windspeed_kh'} = "Calm";
				$wetData{'windspeed_ms'} = "Calm";
				$wetData{'windspeed_kth'} = "Calm";
		}		
		elsif (/wxicons\/52\/(\d+).png.*class="obsTextA">(.+)<\/strong/) {
			#Current
			$wetData{'-1'}{'skyCondition'} = $2; #Text
			$wetData{'-1'}{'forecastIcon'} = $1;
      }
		elsif (/wxicons\/52\/(\d+).gif.*<BR><B CLASS=obsTextA>(.+)<\/B><\/TD>/) {
			#Current
			$wetData{'-1'}{'skyCondition'} = $2; #Text
			$wetData{'-1'}{'forecastIcon'} = $1;
      }
		elsif (/>(\d+.\d+)&nbsp\;in..*\/(.*)_pressure/) { #Bar. pressure value
			$wetData{'pressureIN'} = $1;
			$wetData{'pressureMB'} = $1 * 33.8639;
			$wetData{'pressureMB'} = int($wetData{'pressureMB'} + .5 * ($wetData{'pressureMB'} <=> 0)); #Funky round	
      
      	if ($2 eq 'steady') {
      		$wetData{'pressureT'} = '~';
      	}
      	elsif ($2 eq 'down') {
      		$wetData{'pressureT'} = '-';
      	}
      	else {
      		$wetData{'pressureT'} = '+';
      	}
      }
		elsif (/>(\d+.\d+)&nbsp\;in./) { #Bar. pressure value
			$wetData{'pressureIN'} = $1;
			$wetData{'pressureMB'} = $1 * 33.8639;
			$wetData{'pressureMB'} = int($wetData{'pressureMB'} + .5 * ($wetData{'pressureMB'} <=> 0)); #Funky round	
      }
		elsif (/up_pressure/) { #Bar. pressure trend
			$wetData{'pressureT'} = '+';
      }
		elsif (/steady_pressure/) { #Bar. pressure trend
			$wetData{'pressureT'} = '~';
      }
      elsif (/down_pressure/) { #Bar. pressure trend
			$wetData{'pressureT'} = '-';
		}
		elsif (/<TD VALIGN="top"  CLASS="obsTextA">(-?\d+)&deg\;F<\/td>/) { #Dew point
			$wetData{'dewpointF'} = $1;
			$wetData{'dewpointC'} = FtoC($1);
		}
		elsif (/Dew Point:<\/div><div class="crowval">(-?\d+)&deg\;F/) { #Dew point
			$wetData{'dewpointF'} = $1;
			$wetData{'dewpointC'} = FtoC($1);
		}		
		elsif (/<font CLASS="blueFont10">(.+)<\/font><\/td>/) {
			$forecastNum++;
			$wetData{$forecastNum}{'skyCondition'} = $1;
		}
  		elsif (/<font class="obsTemp"><nobr>(-*\d+)&deg; F<\/nobr><\/font><\/td>/) {			
			$wetData{$forecastNum}{'forecastTempF'} = $1;
			$wetData{$forecastNum}{'forecastTempC'} = FtoC($1);
		}      
		elsif (/wxicons\/31\/(\d+).gif/) {
			$forecastIconNum++;
			$wetData{$forecastIconNum}{'forecastIcon'} = $1;
      }		
		elsif (/<font CLASS="blueVerdanaText11">(.+)<br>/) {
			$wetData{$forecastNum}{'forecastType'} = uc($1);
		}
		elsif (/class="whiteArialLink11"><B>(.+)<\/font><\/B><\/A>/) {
			$forecastTODNum++;
			$wetData{$forecastTODNum}{'forecastTOD'} = $1;
		}
		elsif (/<FONT CLASS="blueFont10"><DIV STYLE="padding:5px 5px 5px 0px;">(.+ )(-?\d+)(F|s)(.*\.+)/) {
			my $tempSymbol;
			my $firstHalf;
			
			#Temperature will either be a specific number (50F) or a group (50s)
			if ($3 eq 'F') {
				$tempSymbol = '';
			}
			else {
				$tempSymbol = 's';
			}

			if ($prefs->get('temperature') == 0) {  #Fahrenheit
				$firstHalf = $1 . $2 . $tempSymbol;
			}
			else { #Celsius
				my $temp = ($2-32)*5/9;       
				$temp = int($temp + .5 * ($temp <=> 0)); #Funky round
				
				$firstHalf = $1 . $temp . $tempSymbol;
			}			

			if ($4 =~ /(.*at )(\d+) to (\d+) mph(.*\.+)/) { #Wind portion
				my $wind;
				my $windUnit = $prefs->get('windunit');
				
				if ($windUnit == 0) { #miles/hr
					$wind = $2 . ' to ' . $3 . ' mph';
				}
				elsif ($windUnit == 1) { #km/hr
					my $wind1;
					my $wind2;
					$wind1 = $2*1.609344;
					$wind1 = int($wind1 + .5 * ($wind1 <=> 0)); #Funky round
					$wind2 = $3*1.609344;
					$wind2 = int($wind2 + .5 * ($wind2 <=> 0)); #Funky round
					$wind = $wind1 . ' to ' . $wind2 . ' km/hr';
				}
				elsif ($windUnit == 3) { #m/s
					my $wind1;
					my $wind2;
					$wind1 = $2*16.09344/36;
					$wind1 = int($wind1 + .5 * ($wind1 <=> 0)); #Funky round
					$wind2 = $3*16.09344/36;
					$wind2 = int($wind2 + .5 * ($wind2 <=> 0)); #Funky round
					$wind = $wind1 . ' to ' . $wind2 . ' m/s';
				}				
				else { #kt/hr
					my $wind1;
					my $wind2;
					$wind1 = $2/1.1515;
					$wind1 = int($wind1 + .5 * ($wind1 <=> 0)); #Funky round
					$wind2 = $3/1.1515;
					$wind2 = int($wind2 + .5 * ($wind2 <=> 0)); #Funky round
	 				$wind = $wind1 . ' to ' . $wind2 . ' kt';
				}
				push(@WETdisplayItems2, $firstHalf . $1 . $wind . $4);
			}
			else { #No wind numbers to convert
				push(@WETdisplayItems2, $firstHalf . $4);
			}
		}
		elsif (/blueVerdanaText.*">(\d+%)<\/td>/) { #Precip
				$wetData{$forecastNum}{'forecastPrec'} = $1;
		}
		elsif (/<FONT CLASS="blueFont10"><DIV STYLE="padding:5px 5px 5px 0px;">(.+\.)/) { #Description didn't parse
			$log->warn("Error parsing weather forecast desc. \nTEXT:" . $1);
			push(@WETdisplayItems2, $1 . '*');			
      }
	}

	#Set current to that of the first period forecast
	$wetData{'-1'}{'forecastPrec'} = $wetData{'0'}{'forecastPrec'};
	$wetData{'-1'}{'forecastType'} = $wetData{'0'}{'forecastType'};
	$wetData{'-1'}{'forecastTempF'} = $wetData{'0'}{'forecastTempF'};
	$wetData{'-1'}{'forecastTempC'} = $wetData{'0'}{'forecastTempC'};

	refreshData(undef, $client, $refreshItem);
}

sub get10day {  #Set up Async HTTP request for 10day
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;
	
	if ($wetData{'0'}{'forecastTOD'} ne $averages{'last'}) { #See if the averages & 10day need to be refreshed due to a period change    	
		my $url = 'http://www.weather.com/weather/print/' . $prefs->get('city');
		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&got10day,
							  \&gotErrorViaHTTP,
							  {caller => 'get10day',
							   callerProc => \&get10day,							  
							   client => $client,
							   refreshItem => $refreshItem});
		$log->info("async request: $url");
	
		$http->get($url);
	}
	else {
		$log->info("Skipping 10day");
		refreshData(undef, $client, $refreshItem);
	}
}

sub got10day {  #10day weather data was received
	my $http = shift;
	
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};

	$log->info("got " . $http->url());
	#$::d_plugins && msg("SuperDateTime: content type is " . $http->headers()->{'Content-Type'} . "\n");

	my $content = $http->content();
	
	#my @ary=split /FONT CLASS=blkVerdanaText10">.*<BR>.*<\/FONT><\/TD>/,$content; #break large string into day array
   my @ary=split /\n/,$content; #break large string into array
   
   my $dayNum = 1;
	for (@ary) {
		if (/15%">(.*)<BR> (.*)<\/TD>/) {
			$wetData{'d'.$dayNum}{'day'} = $1;
			$wetData{'d'.$dayNum}{'date'} = $2;		
		}
		elsif (/10%"><IMG SRC.*wxicons\/31\/(\d+).gif/) {
			$wetData{'d'.$dayNum}{'forecastIcon'} = $1;
		}
		elsif (/35%">(.*)<\/TD>/) {
			$wetData{'d'.$dayNum}{'condition'} = $1;	
		}
		elsif (/25%" ALIGN="CENTER"><B>(.*)&deg;\/(.*)&deg;/) {
			$wetData{'d'.$dayNum}{'highF'} = $1;
			$wetData{'d'.$dayNum}{'highC'} = FtoC($1);
			$wetData{'d'.$dayNum}{'lowF'} = $2;
			$wetData{'d'.$dayNum}{'lowC'} = FtoC($2);		
		}
		elsif (/15%" ALIGN="CENTER">(\d+) %</) {
					$wetData{'d'.$dayNum}{'precip'} = $1 . '%';
					$dayNum++;
		}
		elsif (/FONT CLASS="blkVerdanaText10">(.*) <BR> (.*)<\/FONT></) {
			$wetData{'d'.$dayNum}{'day'} = $1;
			$wetData{'d'.$dayNum}{'date'} = $2;
		}
		elsif (/TD (C|>).*IMG SRC.*wxicons\/31\/(\d+).gif/) {
			$wetData{'d'.$dayNum}{'forecastIcon'} = $2;
		}
		elsif (/CLASS="blkVerdanaText10"><B>(.*)&deg;\/(.*)&deg;/) {
			$wetData{'d'.$dayNum}{'highF'} = $1;
			$wetData{'d'.$dayNum}{'highC'} = FtoC($1);
			$wetData{'d'.$dayNum}{'lowF'} = $2;
			$wetData{'d'.$dayNum}{'lowC'} = FtoC($2);
		}
		elsif (/CLASS="blkVerdanaText10">(\d+) %</) {
			$wetData{'d'.$dayNum}{'precip'} = $1 . '%';
			$dayNum++;
		}		
		elsif (/CLASS="blkVerdanaText10">(.*)<\/FONT/) { #**Need to move to bottom condition
			$wetData{'d'.$dayNum}{'condition'} = $1;
		}
	}
	
	refreshData(undef, $client, $refreshItem);
}

sub getWunderground {  #Set up Async HTTP request for Weather
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;
	
	if ($prefs->get('wunder') ne '') { #Make sure wunderground data is needed  
		my $url = 'http://www.wunderground.com/weatherstation/WXDailyHistory.asp?ID=' . $prefs->get('wunder') . '&format=1';
		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotWunderground,
						  \&gotErrorViaHTTP,
						  {caller => 'getWunderground',
							callerProc => \&getWunderground,
						   client => $client,
						   refreshItem => $refreshItem});
	
		$log->info("async request: $url");
		$http->get($url);
	}
	else {
		$log->info("No wunderground.com value set.  Skipping...");
		refreshData(undef, $client, $refreshItem);
	}

}

sub gotWunderground {  #Weather data was received
	my $http = shift;
	
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};

   $log->info("got " . $http->url());
	my $content = $http->content();
	my @ary=split /\n/,$content; #break large string into array

	#$::d_plugins && msg("WU DATA: ".$ary[scalar(@ary)-3]."\n");

	if($ary[scalar(@ary)-3] =~ /.*,(.*),(.*),(.*),(.*),.*,(.*),.*,(.*),.*,(.*),(.*),.*,.*,/) {
		$wetData{'wu_temperatureF'} = $1;
		$wetData{'wu_temperatureC'} = FtoC($1); 
		
		$wetData{'wu_dewpointF'} = $2;
		$wetData{'wu_dewpointC'} = FtoC($2);
		
		$wetData{'wu_pressureIN'} = $3;
		$wetData{'wu_pressureMB'} = $3 * 33.8639;
		$wetData{'wu_pressureMB'} = int($wetData{'wu_pressureMB'} + .5 * ($wetData{'wu_pressureMB'} <=> 0)); #Funky round	
		
		if ($5==0) {
			$wetData{'wu_windspeed_mh'}  = 'Calm';
			$wetData{'wu_windspeed_kh'}  = 'Calm';
			$wetData{'wu_windspeed_kth'} = 'Calm';
			$wetData{'wu_windspeed_ms'}  = 'Calm';
		}
		else {
			my $winddir = $4;
			if ($winddir eq "South") {
				$winddir = "S";
			}
			elsif ($winddir eq "North") {
				$winddir = "N";
			}
			elsif ($winddir eq "East") {
				$winddir = "E";
			}
			elsif ($winddir eq "West") {
				$winddir = "W";
			}
			
			$wetData{'wu_windspeed_mh'} = $winddir . $5;
			
			$wetData{'wu_windspeed_kh'} = $5*1.609344;
			$wetData{'wu_windspeed_kh'} = $wetData{'wu_windspeed_kh'} . int($wetData{'wu_windspeed_kh'} + .5 * ($wetData{'wu_windspeed_kh'} <=> 0)); #Funky round
			
			$wetData{'wu_windspeed_ms'} = $5*16.09344/36;
			$wetData{'wu_windspeed_ms'} = $wetData{'wu_windspeed_ms'} . int($wetData{'wu_windspeed_ms'} + .5 * ($wetData{'wu_windspeed_ms'} <=> 0)); #Funky round

			$wetData{'wu_windspeed_kth'} = $5/1.1515;
			$wetData{'wu_windspeed_kth'} = $wetData{'wu_windspeed_kth'} . int($wetData{'wu_windspeed_kth'} + .5 * ($wetData{'wu_windspeed_kth'} <=> 0)); #Funky round	
		}
		
		$wetData{'wu_humidity'} = $6 . '%';
	}

	refreshData(undef, $client, $refreshItem);
}

sub getStocks {  #Set up Async HTTP request for Stocks
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;

	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
		
	my $offset = $prefs->get('offset');	
	while ($offset != 0) {
		if ($offset >0) {
			$offset = $offset -1;
			$hour = $hour -1;
			if ($hour == -1) {
				$hour = 23;
				$wday--;
			}
		}
		else {
			$offset = $offset +1;
			$hour = $hour +1;
			if ($hour == 24) {
				$hour = 0;
				$wday++;
			}
		}
	} 	

	my $stocks = $prefs->get('stocks');

               #Market hours
	if ($stocks ne "" && $wday >0 && $wday <6 && $hour <18 && $hour >8) {
		my $url = 'http://finance.yahoo.com/d/quotes.csv?s=' . $stocks . '&f=sl1d1t1c1ohgvpn&e=.csv';
		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotStocks,
							  \&gotErrorViaHTTP,
							  {caller => 'getStocks',
							   callerProc => \&getStocks,							  
							   client => $client,
							   refreshItem => $refreshItem});
		
		$log->info("async request: $url");
		$http->get($url);
	}
	else {
		$log->info("Skipping stocks...");
		
		if ($hour <9 || $wday == 0 || $wday == 6) {
			$log->info("Out of stock info display window.");
		}
		elsif ($stocks ne '') {
			saveCycles('getStocks');
		}

		refreshData(undef, $client, $refreshItem);
	}
}

sub gotStocks {  #Stock data was received
	my $http = shift;
	
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};
	
	$log->info("got " . $http->url());

	my $content = $http->content();
	my @ary=split /\n/,$content; #break large string into array

	#$::d_plugins && msg("STOCK DATA: ".$ary[0]."\n");
	
	for (@ary) {  
		if(/"(.*)",(.*),"(.*)","(.*)",(.*),(.*),(.*),(.*),(.*),(.*),"(.*)"/) {
			my $ticker    = $1;
			my $name;
			my $lasttrade = Slim::Utils::Misc::delimitThousands($2);
			my $lastdate  = $3;
			my $lasttime  = $4;
			my $change    = $5;
			my $open      = Slim::Utils::Misc::delimitThousands($6);
			my $high      = Slim::Utils::Misc::delimitThousands($7);
			my $low       = Slim::Utils::Misc::delimitThousands($8);
			my $volume    = $9;
			my $prev;
		   my $pchange   = Slim::Utils::Misc::delimitThousands($10);

			if ($volume >1000000) {
			  $volume = $volume / 1000000;
			  
			  $volume = sprintf("%.2f", $volume);
			  
			  $volume = Slim::Utils::Misc::delimitThousands($volume);
			  $volume = $volume . 'M';
			}
			else {
				$volume = Slim::Utils::Misc::delimitThousands($volume);
			}
						
			if ($ticker eq "^IXIC") {  #Volume is not reported for Nasdaq composite
				$volume = '';
			}
						
			if ($10 > 0) {
				$pchange = ($5/$10)*100;
				$pchange = sprintf("%.2f%", $pchange);
			}
			
			$name = $11;			

			my $line1 = $prefs->get('stock1format');
			my $line2 = $prefs->get('stock2format');
			
			for ($line1) {
				s/%s/$ticker/;
				s/%l/$lasttrade/;
				s/%d/$lastdate/;
				s/%t/$lasttime/;
				s/%c/$change/;
				s/%o/$open/;
				s/%h/$high/;
				s/%w/$low/;
				s/%v/$volume/;
				s/%n/$name/;
				s/%p/$prev/;
				s/%z/$pchange/;
			}
			
			for ($line2) {
				s/%s/$ticker/;
				s/%l/$lasttrade/;
				s/%d/$lastdate/;
				s/%t/$lasttime/;
				s/%c/$change/;
				s/%o/$open/;
				s/%h/$high/;
				s/%w/$low/;
				s/%v/$volume/;
				s/%n/$name/;
				s/%p/$prev/;
				s/%z/$pchange/;				
			}
			
			addDisplayItem('getStocks', $line1, $line2, $showgame);
		}
	}

	refreshData(undef, $client, $refreshItem);
}

sub refreshSoon {
	my $client = shift;

	#Save top showing preference incase server is shut down
	$prefs->client($client)->set('topNowShowing', $topNowShowing{$client});

	#Clear previous display info
	@WETdisplayItems2 =();
	
	$averages{'last'} = ''; #Make sure averages/10day will get refreshed too
	
	#Need to reload updated prefs
	@MLBteams = @{ $prefs->get('mlb') || [] };
	$refreshTracker[0] = 1; #Need to refresh MLB again
	@NBAteams = @{ $prefs->get('nba') || [] };
	$refreshTracker[1] = 1; #Need to refresh NBA again
	@NHLteams = @{ $prefs->get('nhl') || [] };
	$refreshTracker[2] = 1; #Need to refresh NHL again
	@NFLteams = @{ $prefs->get('nfl') || [] };
	@CBBteams = @{ $prefs->get('cbb') || [] };
	@CFBteams = @{ $prefs->get('cfb') || [] };
	$showtime = $prefs->get('time');
	$showgame = $prefs->get('score');
	$showactivetime = $prefs->get('atime');
	$showactivegame = $prefs->get('ascore');
	
	#No need to refresh data unless there are active clients
	if ($activeClients >0) {
		Slim::Utils::Timers::killTimers(undef, \&refreshData);
		#Refresh in 8 seconds.  In case user keeps changing settings so there are less refreshes maybe.....
		Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + 8, \&refreshData, $client, -1);
	}
}

sub refreshPlayerSettings {
	my $timerObj = shift; #Should be undef
	my $client = shift;

	my $prefs = preferences('plugin.superdatetime');

	my @show1icon = @{ $prefs->client($client)->get('show1icon') || [] };
	if (scalar(@show1icon) == 0) {
		$show1icon[0] = 1;
		$show1icon[1] = 1;
		$prefs->client($client)->set('show1icon', \@show1icon);
	}

	my @show13line = @{ $prefs->client($client)->get('show13line') || [] };
	if (scalar(@show13line) == 0) {
		$show13line[0] = 1;
		$show13line[1] = 1;
		$prefs->client($client)->set('show13line', \@show13line);
	}

	my @d3line1t = @{ $prefs->client($client)->get('v3line1t') || [] };
	if (scalar(@d3line1t) == 0) {
		$d3line1t[0] = '';
		$d3line1t[1] = '';
		$prefs->client($client)->set('v3line1t', \@d3line1t);
	}

	my @d3line1m = @{ $prefs->client($client)->get('v3line1m') || [] };
	if (scalar(@d3line1m) == 0) {
		$d3line1m[0] = '%z';
		$d3line1m[1] = ' %_4';
		$prefs->client($client)->set('v3line1m', \@d3line1m);
	}

	my @d3line1b = @{ $prefs->client($client)->get('v3line1b') || [] };
	if (scalar(@d3line1b) == 0) {
		$d3line1b[0] = 'PREC %x';
		$d3line1b[1] = 'PREC %_9';
		$prefs->client($client)->set('v3line1b', \@d3line1b);
	}

	my @weatherformat1b = @{ $prefs->client($client)->get('weatherformat1b') || [] };
	if (scalar(@weatherformat1b) == 0) {
		$weatherformat1b[0] = '%t/%h %1';
		$weatherformat1b[1] = '%_5/%_7';
		$prefs->client($client)->set('weatherformat1b', \@weatherformat1b);
	}

	my @weatherformat1t = @{ $prefs->client($client)->get('weatherformat1t') || [] };
	if (scalar(@weatherformat1t) == 0) {
		$weatherformat1t[0] = '%y: %v';
		$weatherformat1t[1] = '%_3: %_0';
		$prefs->client($client)->set('weatherformat1t', \@weatherformat1t);
	}

	my @weatherformat2b = @{ $prefs->client($client)->get('weatherformat2b') || [] };
	if (scalar(@weatherformat2b) == 0) {
		$weatherformat2b[0] = '%f  %a(%c %g)';
		$weatherformat2b[1] = '%_0';
		$prefs->client($client)->set('weatherformat2b', \@weatherformat2b);
	}

	my @weatherformat2t = @{ $prefs->client($client)->get('weatherformat2t') || [] };
	if (scalar(@weatherformat2t) == 0) {
		$weatherformat2t[0] = '%s - %S';
		$weatherformat2t[1] = '%_3: %_5/%_7';
		$prefs->client($client)->set('weatherformat2t', \@weatherformat2t);
	}
	
	my @d1period = @{ $prefs->client($client)->get('v1period') || [] };
	if (scalar(@d1period) == 0) {
		$d1period[0] = -1; 
		$d1period[1] = -1;
		$prefs->client($client)->set('v1period', \@d1period);
	}

	if ($prefs->client($client)->get('pref_scroll') eq '') {
		$scrollType{$client} = 'Slide';
	}
	else {
		$scrollType{$client} = $prefs->client($client)->get('pref_scroll');
	}
	
}

sub refreshData {
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;
	$refreshItem = $refreshItem + 1;
	
	if ($refreshItem == 0) { #New data refresh, weather.com
		$log->info("Data refresh in process...");
		$errorCount = 0; #Reset network error counter for new refresh
		%displayInfoBuild = (); #Paranoia, should already be empty

		#Do we need to take into account $isdst???  How do we figure out if East Coast is observing if located in timezone that doesnt observe?
		my ($mday,$mon,$year,$hour, $isdst) = (gmtime(time-(60*60*4)))[3,4,5,2,8];	#Figure out EST based on GMT time
		if ($mday < 10) {
			$mday = '0' . $mday;
		}

		$mon++; #Starts at 0
		if ($mon < 10) {
			$mon = '0' . $mon;
		}
		$year = $year + 1900;
	
		$log->debug("Eastern Hour/Date:$hour/$mday");

		if ((($lastRefresh ne ($year . $mon . $mday)) && $hour >4) || $lastRefresh eq '') { #5AM EST start refreshing a new day
			@refreshTracker = ();
			#Should this be moved into their own subs.... would need separate $lastRefreshes then
			$refreshTracker[0] = 1; #Need to refresh MLB again
			$refreshTracker[1] = 1; #Need to refresh NBA again
			$refreshTracker[2] = 1; #Need to refresh NHL again
			$lastRefresh = $year.$mon.$mday;
		}

		#Update status indicator to show that a data refresh is in progress
		$status = '*';

		#Is this okay since old data is still being displayed?
		$newActiveGames = 0; #Reset active game flag for upcoming sports data refresh

		getWeather(undef, $client, $refreshItem);
	}
	elsif (defined $providers{$refreshItem}) { #Dynamic provider
		$providers{$refreshItem}->(undef, $client, $refreshItem);
	}
	else {  #DONE
		if ($newActiveGames == 1) {
			$activeGames = 1;
		}
	
		$log->info("Drawing screens...");
		#Refresh complete so it's safe to draw/cache icons for each up/down mode
		
		my @players = Slim::Player::Client::clients();
		for my $player (@players) {
			#Draw default modes
			#Current 0
			Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + .1, \&drawEach, $player, -1, 0);
			#Default 1
			Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + .2, \&drawEach, $player, 0, 0);
			#Default 2
			Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + .3, \&drawEach, $player, 1, 0);
			#Default 3
			Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + .4, \&drawEach, $player, 2, 0);
		
			#Draw custom modes
			my @d1period = @{ $prefs->client($client)->get('v1period') || [] };

			my $cumTime = .6;
			for(my $i=2; $i < scalar @d1period; $i++) {
				Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + $cumTime, \&drawEach, $player, $d1period[$i], $i);
				$cumTime = $cumTime + .11;
			}		
		
			#Draw 10 Day Forecasts for each player
			for (my $i=1; $i < 10; $i++) {
				Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + $cumTime, \&drawEach, $player, 'd'.$i, 1);
				$cumTime = $cumTime + .11;
			}		
		}
	}
}

#Called when no more icons need to be drawn
sub doneDrawing {
	my $client = shift;
	
	$log->info("Done drawing screens.");
		
	#Icons have been drawn, replace old display data with fresh data
	%displayInfo = %displayInfoBuild;
	%displayInfoBuild = (); #Done building, no longer needed		
		
	if ($status eq '*') {
		$log->info("Data refresh completed.");
		$status = '';
	}
	else {
		$log->warn("Data refresh completed with errors.");
	}
		
	$activeGames = $newActiveGames;
		
	Slim::Utils::Timers::killTimers(undef, \&refreshData); #Paranoia check
	Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + ($prefs->get('refresh') * 60), \&refreshData, $client, -1); #Set up next refresh timer

}

sub getMLB {  #Set up Async HTTP request for MLB
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;

	#Do we need to take into account $isdst???  How do we figure out if East Coast is observing if located in timezone that doesnt observe?
	my ($mday,$mon,$year,$hour, $isdst) = (gmtime(time-(60*60*4)))[3,4,5,2,8];	#Figure out EST based on GMT time
	$mon++; #Starts at 0

	if (scalar(@MLBteams)>0 && $mon <11 && $mon >3 && $refreshTracker[0] !=0) { #Make sure a baseball team is chosen and its baseball season
		my $url = 'http://scores.espn.go.com/mlb/scoreboard?date=' . $lastRefresh;
		#my $url = 'http://scores.espn.go.com/mlb/scoreboard?date=20080411';
		#my $url = 'http://sports-ak.espn.go.com/mlb/scoreboard/printable/scoreboard_1_0_0?date=' . $year . $mon . $mday;
	
		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotMLB,
								  \&gotErrorViaHTTP,
								  {caller => 'getMLB',
							      callerProc => \&getMLB,								  
								   client => $client,
								   refreshItem => $refreshItem});							  
		
		$log->info("aync request: $url");
		$http->get($url);
	}
	else {
		$log->info("Skipping MLB...");
		if (scalar(@MLBteams)>0) {
			saveCycles('getMLB');
		}
		refreshData(undef, $client, $refreshItem);
	}
}

sub registerProvider {
	my $subRef = shift;
	
	my $size = (scalar keys %providers) + 1;
	
	$providers{$size} = $subRef;
}

sub addDisplayItem {
	my ($origin, $item1, $item2, $int, $expires, $time) = @_;
	
	if (!defined $expires) {
		$expires = 0;
	}

	if (!defined $time) {
		$time = Time::HiRes::time();
	}
	
	push(@{$displayInfoBuild{'cycleOrigin'}}, $origin);
	push(@{$displayInfoBuild{'cycleItems1'}}, $item1);
	push(@{$displayInfoBuild{'cycleItems2'}}, $item2);	
	push(@{$displayInfoBuild{'cycleInts'}}, $int);
	push(@{$displayInfoBuild{'cycleTimeAdded'}}, $time);
	push(@{$displayInfoBuild{'cycleTimeExpires'}}, $expires);
}

#Preserves previous display info for a given source.  Optionally a time can be specified which will be compared against the 
#cycleTimeAdded value.either before (b) or after (a) based on $ba argument.
sub saveCycles {
	my ($origin, $time, $ba) = @_;

	$log->debug("Preserving previous $origin info... $ba $time");
	
	if (exists $displayInfo{'cycleOrigin'}) {
		my $total = scalar @{$displayInfo{'cycleOrigin'}};
		for(my $counter=0 ; $counter < $total; $counter++) {
			if ($displayInfo{'cycleOrigin'}[$counter] eq $origin) {
				if (defined $time) {
					if ($ba eq 'b' && $displayInfo{'cycleTimeAdded'}[$counter] < $time) {
						addDisplayItem($origin, $displayInfo{'cycleItems1'}[$counter], $displayInfo{'cycleItems2'}[$counter], $displayInfo{'cycleInts'}[$counter], $displayInfo{'cycleTimeExpires'}[$counter], $displayInfo{'cycleTimeAdded'}[$counter]);
					}
					elsif ($ba eq 'a' && $displayInfo{'cycleTimeAdded'}[$counter] > $time) {
						addDisplayItem($origin, $displayInfo{'cycleItems1'}[$counter], $displayInfo{'cycleItems2'}[$counter], $displayInfo{'cycleInts'}[$counter], $displayInfo{'cycleTimeExpires'}[$counter], $displayInfo{'cycleTimeAdded'}[$counter]);
					}
				}
				else {
					addDisplayItem($origin, $displayInfo{'cycleItems1'}[$counter], $displayInfo{'cycleItems2'}[$counter], $displayInfo{'cycleInts'}[$counter], $displayInfo{'cycleTimeExpires'}[$counter], $displayInfo{'cycleTimeAdded'}[$counter]);
				}
			}
		}
	}			
}

#Preserves non-expired display info for a given source
sub processExpired {
	my $origin = shift;
	
	my $time = Time::HiRes::time();

	$log->debug("Preserving non-expired $origin info...");
	
	my $total = scalar @{$displayInfo{'cycleOrigin'}};
	for(my $counter=0 ; $counter < $total; $counter++) {
		if (($displayInfo{'cycleOrigin'}[$counter] eq $origin) && $displayInfo{'cycleTimeExpires'}[$counter] > $time) {
			addDisplayItem($origin, $displayInfo{'cycleItems1'}[$counter], $displayInfo{'cycleItems2'}[$counter], $displayInfo{'cycleInts'}[$counter], $displayInfo{'cycleTimeExpires'}[$counter], $displayInfo{'cycleTimeAdded'}[$counter]);
		}
	}
}

sub gotMLB {
	my $http = shift;
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};
	
	@divideMLB = ();
	delete $sportsData{'MLB'};
	$refreshTracker[0] = 0;
	
	$log->info($http->url());
	#$::d_plugins && msg("SuperDateTime: content type is " . $http->headers()->{'Content-Type'} . "\n");

	my $content = $http->content();
   
	@divideMLB = split /teamTop/,$content; #break large string into array of games

	Slim::Utils::Timers::killTimers(undef, \&processMLB); #Just in case there's a rogue timer
	Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + .1, \&processMLB, $client, $refreshItem);
}

sub processMLB {
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;

	$log->debug("Processing MLB game...");

	my $line = pop @divideMLB;

	my @todaysGames = split /\n/, $line;  #break down to speed up regexp processing

	for (@todaysGames) {		
		#if (/>(\d+:\d+ .?M ET)<.+class="teamLine".+\d+">(.+)<\/a>.+teamLine".+id=\d+">(.+)<\/a><\/strong>/) { #Upcoming game
		if (/>(\d+:\d+ \w?M ET).+team=\w\w\w">(.+)<\/a>.+team=\w\w\w">(.+)<\/a><\/strong>/) { #Upcoming game
			$log->debug("Upcoming game @ $1 $2 vs $3");
			#my $gametime = $1; my $team1 = $2; my $team2 = $3;
			$sportsData{'MLB'}{$3.$1}{'homeTeam'}  =$3;
			$sportsData{'MLB'}{$3.$1}{'awayTeam'}  =$2;
			$sportsData{'MLB'}{$3.$1}{'gameTime'}  =convertTime($1);
			
			if ((teamCheck($2, \@MLBteams)==1) || (teamCheck($3, \@MLBteams)==1)) {
				displayLines($client, $1, $2, $3, '', '', 'MLB', 'getMLB', \&shortenMLB, \%MLBmap);
			}				
		}		
		elsif (/\d">(.+)<\/td><\/tr><tr><td class="teamLine".+=\w\w\w">(.+)<\/a><\/strong>.+teamLine".+=\w\w\w">(.+)<\/a><\/strong>.+awayRunsTotal.+".*>(\d+)<\/td><td.+awayTeamHits.+homeRunsTotal.+" ?>(\d+)<\/td><td class.+homeTeamHits/) { #Completed/Active game
			#my $gametime = $1; my $team1 = $2; my $team2 = $3; my $team1score = $4; my $team2score = $5;
			$log->debug("Game scores @ $1 $2 $4 vs $3 $5");
			$sportsData{'MLB'}{$3.$1}{'homeTeam'}  =$3;
			$sportsData{'MLB'}{$3.$1}{'awayTeam'}  =$2;
			$sportsData{'MLB'}{$3.$1}{'homeScore'} =$5;
			$sportsData{'MLB'}{$3.$1}{'awayScore'} =$4;
			$sportsData{'MLB'}{$3.$1}{'gameTime'}  =convertTime($1);

			if ((teamCheck($2, \@MLBteams)==1) || (teamCheck($3, \@MLBteams)==1)) {	
				displayLines($client, $1, $2, $3, $4, $5, 'MLB', 'getMLB', \&shortenMLB, \%MLBmap);
			}				
		}	
	}

	$log->debug("Done processing MLB game.");
   
	if (scalar @divideMLB > 0) {
		#Slim::Utils::Timers::killTimers(undef, \&processMLB); #No way there are any right!?!?
		Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + .1, \&processMLB, $client, $refreshItem);	
	}
	else {
		refreshData(undef, $client, $refreshItem);
	}
}

sub teamCheck {
	my $teamToCheck  = shift;
	my $teamArrayRef = shift;

	#Get rid of any ranking, if present (college).
	if ($teamToCheck =~ /(\(\d+\)) (.+)/) {
		$teamToCheck = $2;
	}
	
  	my $endingValue = scalar(@$teamArrayRef);

	for(my $counter=0 ; $counter < $endingValue; $counter++)
	{
		if (($teamToCheck eq @$teamArrayRef[$counter]) || (@$teamArrayRef[$counter] eq '1') || (@$teamArrayRef[$counter] eq 'All') || (@$teamArrayRef[$counter] eq 'all')) {
			return 1;
		}
	}
	
	return 0;
}

sub displayLines {
	#Figure out/Format display lines
	my ($client, $gametime, $team1, $team2, $score1, $score2, $topline, $origin, $shortenRef, $iconHashRef) = @_;

	$gametime = convertTime($gametime);  #Clean up time display and convert to local time zone

	my $logoteam1 = '';
	my $logoteam2 = '';
	
	if ($prefs->get('teamlogos') == 0 || !defined $iconHashRef) { #No logos
		$logoteam1 = $shortenRef->($team1);
		$logoteam2 = $shortenRef->($team2) 
	}
	elsif ($prefs->get('teamlogos') == 1) { #Logo no text
		if (defined $iconHashRef->{$team1}) { #Are there logos for team1?
			$logoteam1 = $client->symbols("$origin-".$iconHashRef->{$team1});
		}
		else { #No logo available
			$logoteam1 = $shortenRef->($team1);
		}
			
		if (defined $MLBmap{$team2}) { #Are there logos for team2
			$logoteam2 = $client->symbols("$origin-".$iconHashRef->{$team2});
		}
		else { #No logo available
			$logoteam2 = $shortenRef->($team2);
		}		
	}
	else { #Logo and text
		if (defined $iconHashRef->{$team1}) { #Are there logos for team1?
			$logoteam1 = $client->symbols("$origin-".$iconHashRef->{$team1});
		}
			
		if (defined $iconHashRef->{$team2}) { #Are there logos for team2
			$logoteam2 = $client->symbols("$origin-".$iconHashRef->{$team2});
		}	
		
		$logoteam1 = $logoteam1 . $shortenRef->($team1);
		$logoteam2 = $logoteam2 . $shortenRef->($team2);
	}
		
	if ($score1 eq '' && $showgame>0) {  #upcoming game
		if ($origin eq 'getMLB') {
			$refreshTracker[0] = 1;
		}
		elsif ($origin eq 'getNBA') {
			$refreshTracker[1] = 1;
		}
		elsif ($origin eq 'getNHL') {
			$refreshTracker[2] = 1;
		}
		addDisplayItem($origin, $topline, $logoteam1 . ' @ ' . $logoteam2 . '-' . $gametime, $showgame);
	}
	elsif ($gametime eq 'F' && $showgame>0) {  #Finished game		
		addDisplayItem($origin, $topline, $logoteam1 . ' ' . $score1 . ' @ ' . $logoteam2 . ' ' . $score2 . ' -' . $gametime, $showgame);		
	}
	elsif ($gametime ne 'F' && $showactivegame >0 && $score1 ne '') { #active game
		if ($origin eq 'getMLB') {
			$refreshTracker[0] = 1;
		}
		elsif ($origin eq 'getNBA') {
			$refreshTracker[1] = 1;
		}
		elsif ($origin eq 'getNHL') {
			$refreshTracker[2] = 1;
		}		
		$newActiveGames = 1;
		addDisplayItem($origin, $topline, $logoteam1 . ' ' . $score1 . ' @ ' . $logoteam2 . ' ' . $score2 . ' -' . $gametime, $showactivegame);
	}
}

sub getNBA {  #Set up Async HTTP request for NBA
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;
	
	if (scalar(@NBAteams)>0 && $refreshTracker[1] !=0) { #Make sure a basketball team is chosen
		#my $url ='http://sports.espn.go.com/nba/scoreboard?date=20050623';
		#my $url = 'http://sports.espn.go.com/nba/scoreboard?date=' . $year . $mon . $mday;
		my $url = 'http://sports.espn.go.com/nba/scoreboard?date=' . $lastRefresh;

		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotNBA,
														  \&gotErrorViaHTTP,
														  {caller => 'getNBA',
							   						   callerProc => \&getNBA,														  
														   client => $client,
														   refreshItem => $refreshItem});

		$log->info("async request: $url");
		$http->get($url);
	}
	else {
		$log->info("Skipping NBA...");
		if (scalar(@NBAteams)>0) {
			saveCycles('getNBA');
		}

		refreshData(undef, $client, $refreshItem);
	}

}

sub gotNBA {
	my $http = shift;
	
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};

	delete $sportsData{'NBA'};

	$refreshTracker[1] = 0;
   
	my $team1 = '';
	my $team2 = '';
	my $gametime = '';
    
	my $team1score ='';
	my $team2score ='';

	$log->info("got " . $http->url());
	#$::d_plugins && msg("SuperDateTime: content type is " . $http->headers()->{'Content-Type'} . "\n");

	my $content = $http->content();
    
	my @ary=split /\n/,$content; #break large string into array

	for (@ary) {
		if (/<div><div id="LSTeams_Clock">.+-clock">(.+)<\/div><\/div><!-- away/) {
			$gametime = $1;
		}
		elsif (/<div><div id="LSTeams_Clock">.+-clock">(.+)<\/div><\/div>/) {
			$gametime = $1;
		}
		elsif (/<div class="game"/) {
			if ($team2 ne "") {  #Save previously found game
				$sportsData{'NBA'}{$team2.$gametime}{'homeTeam'}  =$team2;
				$sportsData{'NBA'}{$team2.$gametime}{'awayTeam'}  =$team1;
				$sportsData{'NBA'}{$team2.$gametime}{'homeScore'} =$team2score;
				$sportsData{'NBA'}{$team2.$gametime}{'awayScore'} =$team1score;
				$sportsData{'NBA'}{$team2.$gametime}{'gameTime'}  =convertTime($gametime);
				
				if ((teamCheck($team1,\@NBAteams)==1) || (teamCheck($team2,\@NBAteams)==1)) {
					displayLines($client, $gametime, $team1, $team2, $team1score, $team2score, 'NBA', 'getNBA', \&shortenNBA, undef);
				}
			}
			$team1 = '';
			$team1score = '';
			$team2 = "";
			$team2score = '';
		}
		elsif (/-- leave empty --><div class="lsData" align="center".*>(\d+)<\/div><div class/) {
			if ($team1score eq '') {
				$team1score = $1;
			}
			elsif ($team2score eq '') {
				$team2score = $1;
			}
		}

		if (/clubhouse\?team=.+">(.*)<\/a><\/div><\/div><!-- home.+clubhouse\?team=.+">(.*)<\/a><\/div><\/div/){
			$team1 = $1;
			$team2 = $2;
		}   
		elsif (/clubhouse\?team=.+">(.*)<\/a>/){
			if ($team1 eq "") {
				$team1 = $1;
			}
			else {
				$team2 = $1;
			}
		}   
	} #end for

	if ($team2 ne "") {  #Save previously found game
		$sportsData{'NBA'}{$team2.$gametime}{'homeTeam'}  =$team2;
		$sportsData{'NBA'}{$team2.$gametime}{'awayTeam'}  =$team1;
		$sportsData{'NBA'}{$team2.$gametime}{'homeScore'} =$team2score;
		$sportsData{'NBA'}{$team2.$gametime}{'awayScore'} =$team1score;
		$sportsData{'NBA'}{$team2.$gametime}{'gameTime'}  =convertTime($gametime);

		if (teamCheck($team1,\@NBAteams) || teamCheck($team2,\@NBAteams)) {
			displayLines($client, $gametime, $team1, $team2, $team1score, $team2score, 'NBA', 'getNBA', \&shortenNBA, undef);
		}
	}

	refreshData(undef, $client, $refreshItem);	
}

sub getCBB {  #Set up Async HTTP request for CBB
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;

	if ((scalar(@CBBteams)>0) && ($prefs->get('cbballconf') != 0)) { #Make sure a college basketball team is entered and not 'none'
		my ($mday,$mon,$year) = (localtime(time))[3,4,5];
 
		if ($mday < 10) {
			$mday = '0' . $mday;
		}
		$mon++;

		if ($mon < 10) {
			$mon = '0' . $mon;
		}
		$year = $year + 1900;
    
 		my $url;
		#$url = 'http://sports.espn.go.com/ncb/scoreboard?confId=50&date=20051210';

 		if ($prefs->get('cbballconf') != 1000) { #Top 25
 			$url = "http://sports.espn.go.com/ncb/scoreboard?confId=".$prefs->get('cbballconf')."&cast=false";
 		}
 		else {
 			$url = 'http://sports.espn.go.com/ncb/scoreboard';
 		}

		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotCBB,
													  \&gotErrorViaHTTP,
													  {caller => 'getCBB',
							   					   callerProc => \&getCBB,													  
													   client => $client,
													   refreshItem => $refreshItem});
	
		$log->info("aync request: $url");
		$http->get($url);
	}
	else {
		$log->info("Skipping CBB...");
		refreshData(undef, $client, $refreshItem);
	}
}

sub gotCBB {
	my $http = shift;
	
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};
   
   delete $sportsData{'College Basketball'};

	$log->info("got " . $http->url());
	#$::d_plugins && msg("SuperDateTime: content type is " . $http->headers()->{'Content-Type'} . "\n");

	my $content = $http->content();
	my @ary=split /"game"/,$content; #break large string into array

	for (@ary) {
		if (/\d-clock">(.*)<\/div><\/div>.*teamId=\d+">(.*)<\/a>.*\n.*teamId=\d+">(.*)<\/a>.*\n.*\d-asT">(\d+)<.*\d-hsT">(\d+)</) {
			$sportsData{'College Basketball'}{$3.$1}{'homeTeam'}  =$3;
			$sportsData{'College Basketball'}{$3.$1}{'awayTeam'}  =$2;
			$sportsData{'College Basketball'}{$3.$1}{'homeScore'} =$5;
			$sportsData{'College Basketball'}{$3.$1}{'awayScore'} =$4;
			$sportsData{'College Basketball'}{$3.$1}{'gameTime'}  = convertTime($1);

			if ((teamCheck($2,\@CBBteams)==1) || (teamCheck($3,\@CBBteams)==1)) {
				displayLines($client, $1, $2, $3, $4, $5, 'College Basketball', 'getCBB', \&shortenCBB, undef);
			}
		}
	} #end for

	refreshData(undef, $client, $refreshItem);	
}

sub getCFB {  #Set up Async HTTP request for college football
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;
	
	if (scalar(@CFBteams)>0 && ($prefs->get('cfballconf') != 0)) { #Make sure a team is entered and not 'none'  
		my $url;	
		#$url = 'http://sports-ak.espn.go.com/ncf/scoreboard?confId=80';

		if ($prefs->get('cfballconf') != 1000) { #Top 25
		 	$url = 'http://sports-ak.espn.go.com/ncf/scoreboard?confId='.$prefs->get('cfballconf');
		}
		else {
			$url = 'http://sports-ak.espn.go.com/ncf/scoreboard';
		}

		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotCFB,
											  \&gotErrorViaHTTP,
											  {caller => 'getCFB',
							   				callerProc => \&getCFB,											  
											   client => $client,
											   refreshItem => $refreshItem});
														  
		$log->info("aync request: $url");
		$http->get($url);
	}
	else {
		$log->info("Skipping CFB...");
		refreshData(undef, $client, $refreshItem);
	}
}

sub gotCFB {
	my $http = shift;
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};

	$log->info("got " . $http->url());
	#$::d_plugins && msg("SuperDateTime: content type is " . $http->headers()->{'Content-Type'} . "\n");

	my $content = $http->content();

	delete $sportsData{'College Football'};

	my @ary=split /"dateChange">-/,$content; #break large string into day array

	for (@ary) {
		if (m/Today/) { #Find today's games 
			#my @todaysGames=split /teamTop/,$1;  #break large string into individual games
			my @todaysGames=split /teamTop/;  #break large string into individual games

			for (@todaysGames) {
				if(/>(.*) ET<\/td>.*teamId=\d.*"> ?(.*)<\/a>.*teamId=\d.*"> ?(.*)<\/a> <s/) { #upcoming game
					#$gametime = $1; $team1 = $2; $team2 = $3;
					$sportsData{'College Football'}{$3.$1}{'homeTeam'}  =$3;
					$sportsData{'College Football'}{$3.$1}{'awayTeam'}  =$2;
					$sportsData{'College Football'}{$3.$1}{'gameTime'}  =convertTime($1);

					if ((teamCheck($2,\@CFBteams)==1) || (teamCheck($3,\@CFBteams)==1)) {
						CFBdisplayLines($client, $1, $2, $3, '', '');
					}
				}
				elsif(/>(.*)<\/td>.*teamId=\d.*"> ?(.*)<\/a>.*teamId=\d.*"> ?(.*)<\/a> .*tScoreLine.*>(\d.*)<\/td>.*tScoreLine.*>(\d.*)<\/td>/s) { #active game
					#$gametime = $1; $team1 = $2; $team2 = $3; $team1score = $4; $team2score = $5;
					$sportsData{'College Football'}{$3.$1}{'homeTeam'}  =$3;
					$sportsData{'College Football'}{$3.$1}{'awayTeam'}  =$2;
					$sportsData{'College Football'}{$3.$1}{'homeScore'} =$5;
					$sportsData{'College Football'}{$3.$1}{'awayScore'} =$4;
					$sportsData{'College Football'}{$3.$1}{'gameTime'}  =convertTime($1);

					if ((teamCheck($2,\@CFBteams)==1) || (teamCheck($3,\@CFBteams)==1)) {
						CFBdisplayLines($client, $1, $2, $3, $4, $5);
					}
				}
			}	 
		}
	}

	refreshData(undef, $client, $refreshItem);
}

sub CFBdisplayLines {
	#Figure out/Format display lines
	my ($client, $CFBgametime, $CFBteam1, $CFBteam2, $CFBscore1, $CFBscore2) = @_;

	my $rank1 = '';
	my $rank2 = '';
	
	if ($CFBteam1 =~ /(\(\d+\)) (.+)/) {
		$rank1 = $1;
		$CFBteam1 = $2;
	}

	if ($CFBteam2 =~ /(\(\d+\)) (.+)/) {
		$rank2 = $1;
		$CFBteam2 = $2;
	}

	$CFBgametime = convertTime($CFBgametime);  #Clean up time display and convert to local time zone

	#Add team icons
	my $icon1='';
	my $icon2='';
	#if (defined $CFBmap{$CFBteam1}) { #Are there icons for each team?
	#	$icon1 = $client->symbols('cfb-'.$CFBmap{$CFBteam1});
	#}
					
	#if (defined $CFBmap{$CFBteam2}) {
	#	$icon2 = $client->symbols('cfb-'.$CFBmap{$CFBteam2});
	#}

	if ($CFBscore1 eq '' && $showgame>0) {  #upcoming game
		addDisplayItem('getCFB', 'College Football', $icon1.$rank1.shortenCFB($CFBteam1) . ' @ ' . $icon2.$rank2.shortenCFB($CFBteam2) . '-' . $CFBgametime, $showgame);
	}
	elsif ($CFBgametime ne 'F' && $showactivegame >0 && $CFBscore1 ne '') { #active game
		$newActiveGames = 1;
		addDisplayItem('getCFB', 'College Football', $icon1.$rank1.shortenCFB($CFBteam1) . ' ' . $CFBscore1 . ' @ ' . $icon2.$rank2.shortenCFB($CFBteam2) . ' ' . $CFBscore2 . ' -' . $CFBgametime, $showactivegame);
	}
	else {  #Must be finished game
		addDisplayItem('getCFB', 'College Football', $icon1.$rank1.shortenCFB($CFBteam1) . ' ' . $CFBscore1 . ' @ ' . $icon2.$rank2.shortenCFB($CFBteam2) . ' ' . $CFBscore2 . ' -' . $CFBgametime, $showgame);
	}
}

sub getNHL {  #Set up Async HTTP request for NHL
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;
	
	if (scalar(@NHLteams)>0 && $refreshTracker[2] !=0) { #Make sure a hockey team is chosen  
		#my $url ='http://sports.espn.go.com/nhl/scoreboard?date=20051009';
		#my $url = 'http://sports.espn.go.com/nhl/scoreboard?date=' . $year . $mon . $mday;
		
		my $url = 'http://sports.espn.go.com/nhl/scoreboard?date=' . $lastRefresh;

		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotNHL,
														  \&gotErrorViaHTTP,
														  {caller => 'getNHL',
							   						   callerProc => \&getNHL,														  
														   client => $client,
														   refreshItem => $refreshItem});

		$log->info("aync request: $url");
		$http->get($url);
	}
	else {
		$log->info("Skipping NHL...");
		if (scalar(@NHLteams)>0) {
			saveCycles('getNHL');
		}

		refreshData(undef, $client, $refreshItem);
	}

}

sub gotNHL {
	my $http = shift;
	
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};

	$refreshTracker[2] = 0;
	delete $sportsData{'NHL'};
	$log->info("got " . $http->url());
	#$::d_plugins && msg("SuperDateTime: content type is " . $http->headers()->{'Content-Type'} . "\n");

	my $content = $http->content();
        
	my @ary=split /\n/,$content; #break large string into array

	for (@ary) {
		if (/-st">(.+)<\/td><.+team=\w+">(.+)<\/a>.+team=\w+">(.+)<\/a> <span id=".+-atot">(\d+)<\/td>.+-htot">(\d+)</) { #Active or complete game
			#$gametime = $1; $team1 = $2; $team2 = $3; $team1score = $4; $team2score = $5;

			$sportsData{'NHL'}{$3.$1}{'homeTeam'}  =$3;
			$sportsData{'NHL'}{$3.$1}{'awayTeam'}  =$2;
			$sportsData{'NHL'}{$3.$1}{'homeScore'} =$5;
			$sportsData{'NHL'}{$3.$1}{'awayScore'} =$4;
			$sportsData{'NHL'}{$3.$1}{'gameTime'}  =convertTime($1);

			if ((teamCheck($2,\@NHLteams)==1) || (teamCheck($3,\@NHLteams)==1)) {
				displayLines($client, $1, $2, $3, $4, $5, 'NHL', 'getNHL', \&shortenNHL, undef);
			}
		}
		elsif (/-st">(.+)<\/td><.+team=\w+">(.+)<\/a>.+team=\w+">(.+)<\/a> <span id="/) { #Upcoming game
			#$gametime = $1; $team1 = $2; $team2 = $3;
			$sportsData{'NHL'}{$3.$1}{'homeTeam'}  =$3;
			$sportsData{'NHL'}{$3.$1}{'awayTeam'}  =$2;
			$sportsData{'NHL'}{$3.$1}{'gameTime'}  =convertTime($1);

			if ((teamCheck($2,\@NHLteams)==1) || (teamCheck($3,\@NHLteams)==1)) {
				displayLines($client, $1, $2, $3, '', '', 'NHL', 'getNHL', \&shortenNHL, undef);
			}
		}
	}

	refreshData(undef, $client, $refreshItem);
}

sub getNFL {  #Set up Async HTTP request for NFL
	my $timerObj = shift; #Should be undef
	my $client = shift;
	my $refreshItem = shift;
	
	if (scalar(@NFLteams)>0) { #Make sure a NFL team is chosen  
		my $url = 'http://sports.espn.go.com/nfl/scoreboard';
	
		my $http = Slim::Networking::SimpleAsyncHTTP->new(\&gotNFL,
												  \&gotErrorViaHTTP,
												  {caller => 'getNFL',
							   				   callerProc => \&getNFL,												  
												   client => $client,
												   refreshItem => $refreshItem});

		$log->info("aync request: $url");
		$http->get($url);
	}
	else {
		$log->info("Skipping NFL...");
		refreshData(undef, $client, $refreshItem);
	}
}

sub gotNFL {
	my $http = shift;
	my $params = $http->params();
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};

	$log->info("got " . $http->url());
	#$::d_plugins && msg("SuperDateTime: content type is " . $http->headers()->{'Content-Type'} . "\n");

	delete $sportsData{'NFL'};

	my $content = $http->content();

	my @ary=split /"dateChange">-/,$content; #break large string into day array

	for (@ary) {
		if (m/Today's/) { #Find today's games 
			#my @todaysGames=split /teamTop/,$1;  #break large string into individual games

			my @todaysGames=split /teamTop/;  #break large string into individual games

			for (@todaysGames) {
				if(/>(.*) ET<\/td>.*teamId=\d.*">(.*)<\/a>.*teamId=\d.*">(.*)<\/a> /) { #upcoming game
					#$gametime = $1;
					#$team1 = $2;
					#$team2 = $3;
					$sportsData{'NFL'}{$3.$1}{'homeTeam'}  =$3;
					$sportsData{'NFL'}{$3.$1}{'awayTeam'}  =$2;
					$sportsData{'NFL'}{$3.$1}{'gameTime'}  =convertTime($1);

					if ((teamCheck($2,\@NFLteams)==1) || (teamCheck($3,\@NFLteams)==1)) {
						displayLines($client, $1, $2, $3, '', '', 'NFL', 'getNFL', \&shortenNFL, \%NFLmap);
					}
				}
				elsif(/>(.*)<\/td>.*teamId=\d.*">(.*)<\/a>.*teamId=\d.*">(.*)<\/a> .*tScoreLine.*>(\d.*)<\/td>.*tScoreLine.*>(\d.*)<\/td>/s) { #active game
					#$gametime = $1;
					#$team1 = $2;
					#$team2 = $3;
					#$team1score = $4;
					#$team2score = $5;
					$sportsData{'NFL'}{$3.$1}{'homeTeam'}  =$3;
					$sportsData{'NFL'}{$3.$1}{'awayTeam'}  =$2;
					$sportsData{'NFL'}{$3.$1}{'homeScore'} =$5;
					$sportsData{'NFL'}{$3.$1}{'awayScore'} =$4;
					$sportsData{'NFL'}{$3.$1}{'gameTime'}  =convertTime($1);
		
					if ((teamCheck($2,\@NFLteams)==1) || (teamCheck($3,\@NFLteams)==1)) {
						displayLines($client, $1, $2, $3, $4, $5, 'NFL', 'getNFL', \&shortenNFL, \%NFLmap);
					}
				}
			}	 
		}
	}
   
	refreshData(undef, $client, $refreshItem);
}

sub convertTime {
  my $gametime = shift;

  my $hour = 0;
  my $minute = 0;
  my $ampm = '';
  my $offset = $prefs->get('offset');

  if ($gametime =~ /(\d+:\d+), (\d+).*Period/) { #Active NHL game
		$gametime = $1 . '/' . $2;
  }
  elsif ($gametime =~ /(\d+) Period, End/) { #NHL game, end of period
		$gametime = 'E'.$1;
  }
  elsif ($gametime =~ /(\D)\D+(\d+)/) { #Active MLB game, shorten inning info
		$gametime = $1 . $2;
  }
  elsif ($gametime =~ /(\d*\d+:\d\d) (\d)\w\w/) { #Active NBA/NFL game
		$gametime = $1 . '/' . $2;
  }
  elsif ($gametime =~ /(\d+):(\d+) (\w\w)/) { #Future game, convert timezone
    $hour = $1;
    $minute = $2;
    $ampm = lc($3);

    while ($offset != 0) {
      if ($offset >0) {
        $offset = $offset -1;
        $hour = $hour +1;
        if ($hour == 13) {
          $hour = 1;
        }
        elsif ($hour == 12) {
          if ($ampm eq 'am') {
            $ampm = 'pm';
          }
          else {
            $ampm = 'am';
          }
        }
      }
      else {
        $offset = $offset +1;
        $hour = $hour -1;
        if ($hour == 0) {
          $hour = 12;
        }
        elsif ($hour == 11) {
          if ($ampm eq 'am') {
            $ampm = 'pm';
          }
          else {
            $ampm = 'am';
          }
        }
      }
    } 
     
    $gametime = $hour . ':' . $minute . $ampm;
  }
  elsif ($gametime =~ /Final - OT/) { #NBA OT Final
    $gametime = 'FOT';
  }
  elsif ($gametime =~ /(\D)\D.*/) { #Final or Delayed MLB game
    $gametime = $1;
  }

  return $gametime;
}


sub shortenMLB {
  my $long = shift;
  
  if ($long =~ m/^Chicago Cubs/) { $long = 'Cubs';}
  elsif ($long =~ m/^NY Yankees/) { $long = 'Yankees';}
  elsif ($long =~ m/^San Francisco/) { $long = 'Giants';}
  elsif ($long =~ m/^White Sox/) { $long = 'WSox';}
  elsif ($long =~ m/^LA Dodgers/) { $long = 'Dodgers';}
  elsif ($long =~ m/^LA Angels/) { $long = 'Angels';}
  elsif ($long =~ m/^Kansas City/) { $long = 'Royals';}
  elsif ($long =~ m/^Tampa Bay/) { $long = 'TB';}
  elsif ($long =~ m/^Cincinnati/) { $long = 'Reds';}
  elsif ($long =~ m/^Philadelphia/) { $long = 'Phillies';}
  elsif ($long =~ m/^Washington/) { $long = 'Nationals';}
  elsif ($long =~ m/^Cleveland/) { $long = 'Indians';}
  elsif ($long =~ m/^Baltimore/) { $long = 'Orioles';}
  elsif ($long =~ m/^Florida/) { $long = 'Marlins';}
  elsif ($long =~ m/^Atlanta/) { $long = 'Braves';}
  elsif ($long =~ m/^Colorado/) { $long = 'Rockies';}
  elsif ($long =~ m/^Detroit/) { $long = 'Tigers';}
  elsif ($long =~ m/^Pittsburgh/) { $long = 'Pirates';}
  elsif ($long =~ m/^NY Mets/) { $long = 'Mets';}
  elsif ($long =~ m/^Houston/) { $long = 'Astros';}
  elsif ($long =~ m/^Milwaukee/) { $long = 'Brewers';}
  elsif ($long =~ m/^St. Louis/) { $long = 'Cardinals';}
    
  return $long;
}

sub shortenNBA {
  my $long = shift;
  
  if ($long =~ m/^San Antonio/) { $long = 'Spurs';}
  elsif ($long=~ m/^Washington/) { $long = 'Wizards';}
  elsif ($long =~ m/^Philadelphia/) { $long = '76ers';}
  elsif ($long =~ m/^Sacramento/) { $long = 'Kings';}
  elsif ($long =~ m/^LA Clippers/) { $long = 'Clippers';}
  elsif ($long =~ m/^LA Lakers/) { $long = 'Lakers';}
  elsif ($long =~ m/^Golden State/) { $long = 'Warriors';}
  elsif ($long =~ m/^New Orleans/) { $long = 'Hornets';}
  elsif ($long =~ m/^NO\/Oklahoma City/) { $long = 'Hornets';}
  elsif ($long =~ m/^Minnesota/) { $long = 'TWolves';}
  elsif ($long =~ m/^Charlotte/) { $long = 'Bobcats';}
  elsif ($long =~ m/^Cleveland/) { $long = 'Cavaliers';}
  elsif ($long =~ m/^Milwaukee/) { $long = 'Bucks';}
  elsif ($long =~ m/^New Jersey/) { $long = 'Nets';}
  elsif ($long =~ m/^New York/) { $long = 'Knicks';}
  elsif ($long =~ m/^Chicago/) { $long = 'Bulls';}
  elsif ($long =~ m/^Toronto/) { $long = 'Raptors';}
  elsif ($long =~ m/^Houston/) { $long = 'Rockets';}
  elsif ($long =~ m/^Utah/) { $long = 'Jazz';}
  elsif ($long =~ m/^Phoenix/) { $long = 'Suns';}
  elsif ($long =~ m/^Orlando/) { $long = 'Magic';}
  elsif ($long =~ m/^Atlanta/) { $long = 'Hawks';}
  elsif ($long =~ m/^Miami/) { $long = 'Heat';}

  return $long;
}

sub shortenCBB {
	my $long = shift;
  
#	YOU CAN MODIFY THIS LIKE THE EXAMPLE BELOW TO SHORTEN YOUR TEAM NAMES...
	if ($long =~ m/^Illinois$/) { $long = 'Illini';}
	elsif ($long=~ m/^Notre Dame/) { $long = 'ND';}
	elsif ($long=~ m/^Southern Methodist/) { $long = 'SMU';}
	elsif ($long=~ m/^Ohio State/) { $long = 'OSU';}
	elsif ($long=~ m/^Washington State/) { $long = 'WSU';}
	elsif ($long=~ m/^Boston College/) { $long = 'BC';}
	elsif ($long=~ m/^Wisconsin$/) { $long = 'Wisc.';}
	elsif ($long=~ m/^Bowling Green$/) { $long = 'BGSU';}
	elsif ($long=~ m/^Brigham Young/) { $long = 'BYU';}
	elsif ($long=~ m/^Northwestern/) { $long = 'NW';}
	elsif ($long=~ m/^San Diego State/) { $long = 'SDSU';}
	elsif ($long=~ m/^San Jose State/) { $long = 'SJSU';}

	return $long;
}

sub shortenCFB {
	my $long = shift;
  
#	YOU CAN MODIFY THIS LIKE THE EXAMPLES BELOW TO SHORTEN YOUR TEAM NAMES...
	if ($long =~ m/^Illinois$/) { $long = 'Illini';}
	elsif ($long=~ m/^Notre Dame/) { $long = 'ND';}
	elsif ($long=~ m/^Southern Methodist/) { $long = 'SMU';}
	elsif ($long=~ m/^Ohio State/) { $long = 'OSU';}
	elsif ($long=~ m/^Washington State/) { $long = 'WSU';}
	elsif ($long=~ m/^Boston College/) { $long = 'BC';}
	elsif ($long=~ m/^Wisconsin$/) { $long = 'Wisc.';}
	elsif ($long=~ m/^Bowling Green$/) { $long = 'BGSU';}
	elsif ($long=~ m/^Brigham Young/) { $long = 'BYU';}
	elsif ($long=~ m/^Northwestern/) { $long = 'NW';}
	elsif ($long=~ m/^San Diego State/) { $long = 'SDSU';}
	elsif ($long=~ m/^San Jose State/) { $long = 'SJSU';}

	return $long;
}

sub shortenNHL {
	my $long = shift;
  
	if ($long =~ m/^Vancouver/) { $long = 'Canucks';}
	elsif ($long=~ m/^Edmonton/) { $long = 'Oilers';}
	elsif ($long =~ m/^New Jersey/) { $long = 'Devils';}
	elsif ($long =~ m/^NY Rangers/) { $long = 'Rangers';}
	elsif ($long =~ m/^Philadelphia/) { $long = 'Flyers';}
	elsif ($long =~ m/^NY Islanders/) { $long = 'Islanders';}
	elsif ($long =~ m/^Pittsburgh/) { $long = 'Penguins';}
	#elsif ($long =~ m/^Montreal/) { $long = 'Canadiens';}
	#elsif ($long =~ m/^Ottawa/) { $long = 'Senators';}
	elsif ($long =~ m/^Buffalo/) { $long = 'Sabres';}
	elsif ($long =~ m/^Boston/) { $long = 'Bruins';}
	#elsif ($long =~ m/^Toronto/) { $long = 'Maple Leafs';}
	#elsif ($long =~ m/^Florida/) { $long = 'Panthers';}
	#elsif ($long =~ m/^Atlanta/) { $long = 'Thrashers';}
	#elsif ($long =~ m/^Tampa Bay/) { $long = 'Lightning';}
	elsif ($long =~ m/^Washington/) { $long = 'Capitals';}
	#elsif ($long =~ m/^Carolina/) { $long = 'Hurricanes';}
	#elsif ($long =~ m/^Detroit/) { $long = 'Red Wings';}
	#elsif ($long =~ m/^Nashville/) { $long = 'Predators';}
	#elsif ($long =~ m/^Chicago/) { $long = 'Black Hawks';}
	#elsif ($long =~ m/^Columbus/) { $long = 'Blue Jackets';}
  	elsif ($long =~ m/^St. Louis/) { $long = 'Blues';}
	elsif ($long =~ m/^Minnesota/) { $long = 'Wild';}
	#elsif ($long =~ m/^Colorado/) { $long = 'Avalanche';}
	elsif ($long =~ m/^Calgary/) { $long = 'Flames';}
	elsif ($long =~ m/^Anaheim/) { $long = 'Ducks';}
	elsif ($long =~ m/^Dallas/) { $long = 'Stars';}
	elsif ($long =~ m/^Los Angeles/) { $long = 'Kings';}
	elsif ($long =~ m/^San Jose/) { $long = 'Sharks';}
	#elsif ($long =~ m/^Phoenix/) { $long = 'Coyotes';}
	
	return $long;
}

sub shortenNFL {
  my $long = shift;
  
  if ($long =~ m/^Chicago/) { $long = 'Bears';}
  elsif ($long=~ m/^Green Bay/) { $long = 'Packers';}
  elsif ($long=~ m/^Indianapolis/) { $long = 'Colts';}
  elsif ($long=~ m/^Philadelphia/) { $long = 'Eagles';}
  elsif ($long=~ m/^Pittsburgh/) { $long = 'Steelers';}
  elsif ($long=~ m/^New Orleans/) { $long = 'Saints';}
  elsif ($long=~ m/^New England/) { $long = 'Patriots';}
  elsif ($long=~ m/^San Francisco/) { $long = '49ers';}
  elsif ($long=~ m/^Jacksonville/) { $long = 'Jaguars';}
  elsif ($long=~ m/^Tennessee/) { $long = 'Titans';}
  #elsif ($long=~ m/^Tampa Bay/) { $long = 'Buccaneers';}
  elsif ($long=~ m/^Washington/) { $long = 'Redskins';}
  elsif ($long=~ m/^New York Jets/) { $long = 'Jets';}
  elsif ($long=~ m/^New York Giants/) { $long = 'Giants';}
  elsif ($long=~ m/^Cleveland/) { $long = 'Browns';}
  elsif ($long=~ m/^St. Louis/) { $long = 'Rams';}
  elsif ($long=~ m/^Kansas City/) { $long = 'Chiefs';}
  elsif ($long=~ m/^Minnesota/) { $long = 'Vikings';}
  elsif ($long=~ m/^Baltimore/) { $long = 'Ravens';}
  elsif ($long=~ m/^Buffalo/) { $long = 'Bills';}
  elsif ($long=~ m/^Houston/) { $long = 'Texans';}


  return $long;
}

sub setMode() {
	my $class  = shift;
	my $client = shift;

	$client->lines(\&lines);

	# setting this param will call client->update() frequently
	$client->modeParam('modeUpdateInterval', 1); # seconds

}

sub getFunctions {
	return {
		'up' => sub  {
			my $client = shift;
			my $button = shift;
			$client->bumpUp() if ($button !~ /repeat/);
		},
		'down' => sub  {
			my $client = shift;
			my $button = shift;
			$client->bumpDown() if ($button !~ /repeat/);;
		},
		'left' => sub  {
			my $client = shift;
			Slim::Buttons::Common::popModeRight($client);
		},
		'right' => sub  {
			my $client = shift;
			
			my $saver = Slim::Player::Source::playmode($client) eq 'play' ? 'screensaver' : 'idlesaver';
		
			if ($prefs->client($client)->get($saver) ne 'SCREENSAVER.superdatetime') {
				$prefs->client($client)->set($saver,'SCREENSAVER.superdatetime');
			} else {
				$prefs->client($client)->set($saver, $Slim::Player::Player::defaultPrefs->{$saver});
			}
		},
		'stop' => sub {
			my $client = shift;
			Slim::Buttons::Common::pushMode($client, 'SCREENSAVER.superdatetime');
		},
		'showme' => sub { #Function to force screensaver into screensaver mode
			my $client = shift;
			Slim::Buttons::Common::pushMode($client, 'SCREENSAVER.superdatetime');
		}	
	};
}

sub lines {
	my $client = shift;
	
	my $saver = Slim::Player::Source::playmode($client) eq 'play' ? 'screensaver' : 'idlesaver';
	my $line2 = $client->string('SETUP_SCREENSAVER_USE');
	my $overlay2 = Slim::Buttons::Common::checkBoxOverlay($client, $prefs->client($client)->get($saver) eq 'SCREENSAVER.superdatetime');
	
	return {
		'line'    => [ $client->string('PLUGIN_SCREENSAVER_SUPERDATETIME'), $line2 ],
		'overlay' => [ undef, $overlay2 ]
	};
}

our %screensaverSuperDateTimeFunctions = (
	'done' => sub  {
					my ($client, $funct, $functarg) = @_;
					Slim::Buttons::Common::popMode($client);
					$client->update();
					#pass along ir code to new mode if requested
					if (defined $functarg && $functarg eq 'passback') {
						Slim::Hardware::IR::resendButton($client);
					}
	},
	'up' => sub  {
		my $client = shift;
		
		$weathershowing{$client} = 1; #Reset weather showing in case up was hit while displaying weather
		
		if ($scrollType{$client} eq 'Ticker') { #Set flag to reset out of ticker mode
			$killTicker = 1;
		}
		
		if ($nowshowing{$client} == 0) { #Is the time currently showing?
			if($topNowShowing{$client} == (scalar @{$displayInfo{$client}{'TOPdisplayItems1'}}-1)) {
				$topNowShowing{$client} = 0;
			}
			else {
				$topNowShowing{$client}++;
			}
		}
		else {
			$nowshowing{$client} = 0; #Show the time
		}
		
		killClientTimers($client);
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + 10, \&nextDisplayItem);
				
		$client->update(); #Refresh the display
	},
	'down' => sub  {
		my $client = shift;
		
		$weathershowing{$client} = 1; #Reset weather showoing incase down was hit while displaying weather
		
		if ($scrollType{$client} eq 'Ticker') { #Set flag to reset out of ticker mode
			$killTicker = 1;
		}
		
		if ($nowshowing{$client} == 0) { #Is the time currently showing?
			if($topNowShowing{$client} == 0) {
				$topNowShowing{$client} = scalar @{$displayInfo{$client}{'TOPdisplayItems1'}}-1;
			}
			else {
				$topNowShowing{$client}--;
			}
		}
		else {
			$nowshowing{$client} = 0; #Show the time
		}
		
		killClientTimers($client);			
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + 10, \&nextDisplayItem);	
		
		$client->update(); #Refresh the display
	},
	'down.hold' => sub  {
		my $client = shift;
		
		$weathershowing{$client} = 1; #Reset weather showing in case up was hit while displaying weather
		
		if ($scrollType{$client} eq 'Ticker') { #Set flag to reset out of ticker mode
			$killTicker = 1;
		}
		
		if ($nowshowing{$client} == 0) { #Is the time currently showing?
			$topNowShowing{$client} = 0;
		}
		else {
			$nowshowing{$client} = 0; #Show the time
		}
		
		killClientTimers($client);
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + 10, \&nextDisplayItem);
				
		$client->update(); #Refresh the display
	},	
	'refresh' => sub  {
		my $client = shift;

		$client->showBriefly( {
			line => ['Refreshing data...']
		},
		{
			scroll => 1,
			block  => 1,
		} );		

		Slim::Utils::Timers::killTimers(undef, \&refreshData);
		$averages{'last'} = ''; #Make sure averages/10day will get refreshed too
		refreshData(undef, $client, -1);
	},	
	'wnext' => sub  {
		my $client = shift;

		if (scalar @WETdisplayItems2 >0) { #Make sure there are some long forecasts to show
			killClientTimers($client);
			nextWeatherItem($client,1);
		}
		else {
			$client->bumpLeft();
		}
	},
	'wprev' => sub  {
		my $client = shift;
		
		if (scalar @WETdisplayItems2 >0) { #Make sure there are some long forecasts to show
			killClientTimers($client);
			nextWeatherItem($client,-1);
		}
		else {
			$client->bumpRight();
		}
	},	
	'right' => sub  {
		my $client = shift;
		
		if ((totalGames() + currentWeatherInterval()) >0) { #Are there games?
			killClientTimers($client);
			nextDisplayItem($client);
		}
		else {
			$client->bumpLeft();
		}
		
	},	
	'left' => sub  {
		my $client = shift;

		if ((totalGames() + currentWeatherInterval()) >0) { #Are there games?
			killClientTimers($client);
			prevDisplayItem($client);
		}
		else {
			$client->bumpRight();
		}
	},
	'size.hold' => sub  {
		my $client = shift;
		if ($client->power()) { #Client power is on
			Slim::Buttons::Common::popModeRight($client); #Bring client back to where they were b4 screensaver
		}
	},	
	
);

sub currentWeatherInterval {
	if (scalar @WETdisplayItems2 > 0) {
		my $lweather = $prefs->get('lweather');
		if ($lweather == 4) {
			return 1;
		}
		if ($activeGames == 1 && $lweather == 2) {
			return 1;
		}
		elsif ($activeGames == 0 && $lweather == 3) {
			return 1;
		}
		else {
			return 0;
		}
	}
	else {
		return 0;
	}
}

sub totalGames {
	#return scalar @MLBdisplayItems2 + scalar @NBAdisplayItems2 + scalar @NFLdisplayItems2 + scalar @NHLdisplayItems2 + scalar @CBBdisplayItems2 + scalar @CFBdisplayItems2 + scalar @STKdisplayItems1;	
	my $total = 0;
	
	if (defined $displayInfo{'cycleItems1'}) {
		$total = scalar @{$displayInfo{'cycleItems1'}};
	}

	return $total;	
}

sub getScreensaverSuperDatetime {

	return \%screensaverSuperDateTimeFunctions;
}

sub setScreensaverSuperDateTimeMode() {
	my $client = shift;
	
	#***TESTING***   Need to make sure this only occurs once? instead of each setmode?
	my $key;
	my $value;
	while (($key, $value) = each(%NFLmap)){
		Slim::Display::Graphics::setCustomChar( 'getNFL-'.$value, chr(hex $value), 'nfl.2' );
	}
	
	while (($key, $value) = each(%MLBmap)){
		Slim::Display::Graphics::setCustomChar( 'getMLB-'.$value, chr(hex $value), 'mlb.2' );
	}
	
	#Add 'special' character for spacing
	Slim::Display::Graphics::setCustomChar( 'getMLB-41', chr(hex 41), 'mlb.2' );
	
	$activeClients++;

	#$::d_plugins && msg("SuperDateTime: ENTERING Active Clients: " . $activeClients . "\n");

	#Set default for top line per client
	if ($prefs->client($client)->get('topNowShowing') ne '') {
		$topNowShowing{$client} = $prefs->client($client)->get('topNowShowing');
	}
	else {
		$topNowShowing{$client} = 1;
	}

	refreshPlayerSettings(undef, $client);

	if ($activeClients == 1) {
		Slim::Utils::Timers::killTimers(undef, \&refreshData); #Just in case there's a rogue timer
		Slim::Utils::Timers::setTimer(undef, Time::HiRes::time() + 6, \&refreshData, $client, -1);
	}

	$nowshowing{$client} = 0; #Start on time display
	Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + 10, \&nextDisplayItem);
			
	$client->lines(\&screensaverSuperDateTimelines);
	
	#Slim::Utils::Timers::listTimers();
	# setting this param will call client->update() frequently
	$client->modeParam('modeUpdateInterval', 1); # seconds #7.0

	#Graphical forecasts code...
	# get display size for player if at least Squeezebox2
	if( $client && $client->isa( "Slim::Player::Squeezebox2")) {
		$xmax{$client} = $client->display()->displayWidth();
		$ymax{$client} = $client->display()->bytesPerColumn() * 8;
		
		$log->debug("Found graphic display $xmax{$client} x $ymax{$client} ($client)");
	}
	# only use text on SqueezeboxG and SLIMP3
	else {
		$xmax{$client} = 0;
		$ymax{$client} = 0;
	}
		
	if (scalar $displayInfo{$client}{'forecastG'}==0 && ($client->display->vfdmodel() eq Slim::Display::Squeezebox2->vfdmodel())) {  #Only set the weather channel logo the first time
		clearCanvas($client); #Show weatherchannel logo at startup
		drawIcon($client,29,$ymax{$client}-1,$TWClogo);;
   	$displayInfo{$client}{'forecastG'}[0] = getFramebuf($client,75); #75 gfx width
   	$displayInfo{$client}{'forecastG'}[1] = $displayInfo{$client}{'forecastG'}[0];
   	$displayInfo{$client}{'forecastG'}[2] = $displayInfo{$client}{'forecastG'}[0];
   	$displayInfo{$client}{'forecastG'}[3] = $displayInfo{$client}{'forecastG'}[0];
	}
	
	
}

sub leaveScreensaverSuperDateTimeMode {
	my $client = shift;
	
	#Save top showing preference incase server is shut down
	$prefs->client($client)->set('topNowShowing', $topNowShowing{$client});
	
	$activeClients--;
	#$::d_plugins && msg("SuperDateTime: LEAVING Active Clients: " . $activeClients . "\n");
	
	if ($activeClients == 0) {
		Slim::Utils::Timers::killTimers(undef, \&refreshData);
	}
	
	killClientTimers($client);
	Slim::Utils::Timers::killTimers($client, \&_flashAlarm);
}

sub gotErrorViaHTTP {
	my $http = shift;
	my $params = $http->params();
	my $caller = $params->{'caller'};
	my $callerProc = $params->{'callerProc'};
	my $client = $params->{'client'};
	my $refreshItem = $params->{'refreshItem'};

	$log->warn("error getting " . $http->url());
	$log->warn($http->error());
	$errorCount++;

	if ($errorCount >3) {
		$log->warn("Network error count reached during $caller.");
		$status = '?';
		saveCycles($caller);
		refreshData(undef, $client, $refreshItem);
	}
	elsif ($caller eq "getAverages") {#Special case
		$log->info("Trying getAverages again.");
		getAverages($params->{'dayNum'}, $client, $refreshItem);
	}
	elsif (defined $callerProc) {
		$log->info("Trying $caller again.");
		$callerProc->(undef, $client, $refreshItem);
	}
}

sub nextWeatherItem {
	my $client = shift;
	my $caller = shift;
	
	$killTicker = 3;

	if (defined $weathershowing{$client}) {
		if ($caller == -1) {
			if ($weathershowing{$client} == 1) {
				$weathershowing{$client} = 3;
			}
			else {
				$weathershowing{$client}--;
			}
		}
		else {
			if ($weathershowing{$client} == 3) {
				$weathershowing{$client} = 1;
			}
			else {
				$weathershowing{$client}++;
			}
		}
	}
	else {
		if ($caller == -1) {
			$weathershowing{$client} = 3;
		}
		else {
			$weathershowing{$client} = 1;
		}
	}

	#*********This needs to be checked!!!!!!!!!
	$displayLine1{$client} = $displayInfo{$client}{'TOPdisplayItems1'}[$weathershowing{$client}];
	$displayLine2{$client} = $WETdisplayItems2[$weathershowing{$client}-1];
	$nowshowing{$client} = -1; #Don't display the time...

	if ($weathershowing{$client} == 3) {		
		#Need to give time for display lines to be called (+2)
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + 2, \&setWeatherTimer,2);
	}
	else {
		#Need to give time for display lines to be called (+2)
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + 2, \&setWeatherTimer,1);
	}

	#Update display
	if ($scrollType{$client} eq 'Basic') {
		$client->update(); #Refresh the display
	}
	else {
		if ($caller == -1) {
			if ($client->display->hasScreen2()) {
				my $prevHash = $client->curDisplay();
				delete $prevHash->{'screen1'};
					
				my $hash = $client->curLines();
				delete $hash->{'screen1'};
				$client->pushRight($prevHash, $hash);
			}
			else {
				$client->pushRight();
			}
		}
		else {
			if ($client->display->hasScreen2()) {
				my $prevHash = $client->curDisplay();
				delete $prevHash->{'screen1'};
					
				my $hash = $client->curLines();
				delete $hash->{'screen1'};
				$client->pushLeft($prevHash, $hash);
			}
			else {
				$client->pushLeft();
			}
		}
	}	
}

sub nextDisplayItem {
	my $client = shift;
	
	my $timerInterval = 0;

	if ($nowshowing{$client} < 0) {#Showing weather
		$nowshowing{$client} = 0;
		$timerInterval = $showtime;
	}
	elsif (defined($displayInfo{'cycleItems1'}[$nowshowing{$client}])) {
		$displayLine1{$client} = $displayInfo{'cycleItems1'}[$nowshowing{$client}];
		$displayLine2{$client} = $displayInfo{'cycleItems2'}[$nowshowing{$client}];
		$timerInterval = $displayInfo{'cycleInts'}[$nowshowing{$client}];
		$nowshowing{$client}++;
	}
	else { #TIME
		$nowshowing{$client} = 0;
		
		if (($activeGames == 1) && ($showactivetime == 0)) { #Don't show time when active games
			return nextDisplayItem($client);
		}
		#This probably needs to be re-evaluated...
		elsif ((($activeGames == 0) && ($showtime == 0)) && #Don't show time when no active games
				((totalGames() + scalar @WETdisplayItems2) >0)) #make sure there are some games to show
		{ 
			return nextDisplayItem($client);
		}
		else { #Show the time
			if ($activeGames == 0) {
				$timerInterval = $showtime;
			}
			else {
				$timerInterval = $showactivetime;
			}
		}		
	}
		
	#Update alarm clock status indicator
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
	#my $alarmOn = Slim::Utils::Prefs::clientGet($client, "alarm", 0) || Slim::Utils::Prefs::clientGet($client, "alarm", $wday);
	#my $alarm = preferences('server')->client($client)->get('alarm');
	#my $alarmOn = $alarm->[ 0 ] || $alarm->[ $wday ];

	#if ($alarmOn) {
	#	$overlay = $client->symbols('bell');
	#}
	#else {
	#	$overlay = '';
	#}

	if ($scrollType{$client} eq 'Ticker' && $killTicker == 0) {
		my $tickerScreen;
		if ($client->display->hasScreen2()) {
			$tickerScreen = 2;
		}
		else {
			$tickerScreen = 1;
		}
		
		my ($complete, $queue) = $client->scrollTickerTimeLeft($tickerScreen);
		if ($nowshowing{$client} == 0) {  #NEED TO ONLY OCCUR IF TICKER MODE
			Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + $timerInterval + $complete, \&nextDisplayItem);
		}
	}
   else {
		#Make sure there are items besides the time showing
		if ((totalGames() + currentWeatherInterval()) >0) {
			#Update display
			if ($scrollType{$client} eq 'Basic') {
				$client->update(); #Refresh the display
			}
			else {				
				if ($client->display->hasScreen2()) {
					my $prevHash = $client->curDisplay();
					delete $prevHash->{'screen1'};
					
					my $hash = $client->curLines();
					delete $hash->{'screen1'};
					$client->pushLeft($prevHash, $hash);
				}
				else {
					$client->pushLeft();
				}				
			}
		}

		if ($nowshowing{$client} < 0) { #Weather forecast
			#Need to give time for display lines to be called (+2)
			Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + 2.5, \&setWeatherTimer);
		}
		else {
			Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + $timerInterval, \&nextDisplayItem);
		}
	}

	$killTicker = 0;
}

sub setWeatherTimer {
	#This creates a timer to set the nextDisplayItem timer for a weather forecast.
	#This is necessary because display length is based on how long it takes to scroll...
	my $client = shift;
	my $caller = shift;
	
	my $tickerScreen;
	if ($client->display->hasScreen2()) {
		$tickerScreen = 2;
	}
	else {
		$tickerScreen = 1;
	}	
	my ($complete, $queue) = $client->scrollTickerTimeLeft($tickerScreen);
	
	if ($caller == 1) { #User viewing weather via fast forward/reverse buttons
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + $complete + preferences('server')->client($client)->get('scrollPause'), \&nextWeatherItem);	
	}
	elsif ($caller == 2) { #User viewing weather via fast forward/reverse buttons and at last item
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + $complete + preferences('server')->client($client)->get('scrollPause'), \&nextDisplayItem);
	}
	else { #Viewing weather forecast via left/right buttons or regular display intervals
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + $complete + preferences('server')->client($client)->get('scrollPause'), \&nextDisplayItem);
	}
}

#Used during ticker display mode to set the top display line to the text at halfway point
sub topLineSet {

	my $client = shift;
	my $topLine = shift;
	
	$tickline1{$client} = $topLine;	
}

sub prevDisplayItem {
	my $client = shift;
	my $timerInterval;

	if ($nowshowing{$client} == 0) {
		if (defined($displayInfo{'cycleItems1'})) {
			$nowshowing{$client} = scalar @{$displayInfo{'cycleOrigin'}};
			$displayLine1{$client} = $displayInfo{'cycleItems1'}[$nowshowing{$client}-1];
			$displayLine2{$client} = $displayInfo{'cycleItems2'}[$nowshowing{$client}-1];
			$timerInterval = $displayInfo{'cycleInts'}[$nowshowing{$client}-1];
		}
		else {
			$timerInterval = 10;
		}
	}
	elsif ($nowshowing{$client} <= 1) {
		$nowshowing{$client} = 0;
		
		if ($activeGames == 0) {
			$timerInterval = $showtime;
		}
		else {
			$timerInterval = $showactivetime;
		}								
	}
	else {
		$nowshowing{$client}--;
		$displayLine1{$client} = $displayInfo{'cycleItems1'}[$nowshowing{$client}-1];
		$displayLine2{$client} = $displayInfo{'cycleItems2'}[$nowshowing{$client}-1];
		$timerInterval = $displayInfo{'cycleInts'}[$nowshowing{$client}-1];	
	}

	#Update display
	if ($scrollType{$client} eq 'Basic') {
		$client->update(); #Refresh the display
	}
	else {
		if ($client->display->hasScreen2()) {
			my $hash = screensaverSuperDateTimelines($client);
			delete $hash->{'screen1'};
			#my $curHash = $client->curDisplay();
			#delete $curHash->{'screen2'};
			$client->pushRight(undef, $hash);
		}
		else {
			$client->pushRight();
		}
	}

	if ($nowshowing{$client} < 0) { #Weather forecast
		#Need to give time for display lines to be called (+2)
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + 2, \&setWeatherTimer);
	}
	else {
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + $timerInterval, \&nextDisplayItem);
	}
}

sub killClientTimers {
	my $client = shift;

	Slim::Utils::Timers::killTimers($client, \&nextWeatherItem);
	Slim::Utils::Timers::killTimers($client, \&setWeatherTimer);
	Slim::Utils::Timers::killTimers($client, \&nextDisplayItem);
	Slim::Utils::Timers::killTimers($client, \&topLineSet);
}

sub clearCanvas {
	my $client = shift;

	for( my $xi = 0; $xi < $xmax{$client}; $xi++) {
		for( my $yi = 0; $yi < $ymax{$client}; $yi++) {
			$hashDisp{$client}[$xi][$yi] = 0;
		}
	}
}

sub drawIcon {
	my $client = shift;
	my $xpos = shift;
	my $ypos = shift;
	my $icon = shift;

	#$::d_plugins && Slim::Utils::Misc::msg("SuperDateTime-Icon ($xpos,$ypos): $icon\n");
	if ($xmax{$client} && $ymax{$client}) {
		my $firstline = 1;
		my $xs = $xpos < 0 ? 0 : $xpos;
		my $yi = $ypos > $ymax{$client} ? $ymax{$client} : $ypos;
		for my $line (split('\n',$icon)) {
			# first line must be skipped (empty)
			if ($firstline) {
				$firstline = 0;
				next;
			}
			chomp $line;
			for( my $xi = $xs; $xi < length($line)+$xs && $xi < $xmax{$client} && $yi >= 0; $xi++) {
				if (substr($line,$xi-$xs,1) eq "*") {
					$hashDisp{$client}[$xi][$yi] = 1;
				}
			}
			$yi--;
		}
	}
}

sub drawText {
	my $client = shift;
	my $xpos = shift;
	my $ypos = shift;
	my $text = shift;

	#$::d_plugins && Slim::Utils::Misc::msg("SuperDateTime-Text ($xpos,$ypos): $text\n");
	if ($xmax{$client} && $ymax{$client}) {
		for (my $ci = 0; $ci < length($text); $ci++) {
			my $c = substr($text,$ci,1);
			my $firstline = 1;
			my $xs = $xpos < 0 ? 0 : $xpos + $ci*6;
			my $yi = $ypos > $ymax{$client} ? $ymax{$client} : $ypos;
			for my $line (split('\n',$Charset[$Codepage{$c}])) {
				# first line must be skipped (empty)
				if ($firstline) {
					$firstline = 0;
					next;
				}
				chomp $line;
				for( my $xi = $xs; $xi < length($line)+$xs && $xi < $xmax{$client} && $yi >= 0; $xi++) {
					if (substr($line,$xi-$xs,1) eq "*") {
						$hashDisp{$client}[$xi][$yi] = 1;
					}
				}
				$yi--;
			}
		}
	}
}

# convert %hashDisp into line framebuffer format
sub getFramebuf {
	my $client = shift;
	my $width = shift;
	my $line1 = "";
	
	for( my $x = 0; $x < $width && $x < $xmax{$client}; $x++) {
		my $byte;
		for( my $y = $ymax{$client}; $y > 0; $y -= 8) {
			$byte = ($hashDisp{$client}[$x][$y-1] << 7)
			      + ($hashDisp{$client}[$x][$y-2] << 6)
			      + ($hashDisp{$client}[$x][$y-3] << 5)
			      + ($hashDisp{$client}[$x][$y-4] << 4)
			      + ($hashDisp{$client}[$x][$y-5] << 3)
			      + ($hashDisp{$client}[$x][$y-6] << 2)
			      + ($hashDisp{$client}[$x][$y-7] << 1)
			      +  $hashDisp{$client}[$x][$y-8];
			$line1 .= pack("C", $byte);
		}
	}
	return $line1;
}

sub screensaverSuperDateTimelines {
	my $client = shift;
	my $args   = shift;
	
	my $bottomPad = '';
	
	my $flash  = $args->{'flash'}; # set when called from animation callback
	
	my $narrow = $client->display->isa('Slim::Display::Boom');
	my $currentAlarm = Slim::Utils::Alarm->getCurrentAlarm($client);
	my $nextAlarm = Slim::Utils::Alarm->getNextAlarm($client);
	# show alarm symbol if active or set for next 24 hours
	my $alarmOn = defined $currentAlarm || ( defined $nextAlarm && ($nextAlarm->nextDue - time < 86400) );
	$overlay{$client} = undef;
	if ($alarmOn && !$flash) {
		if (defined $currentAlarm && $currentAlarm->snoozeActive) {
			$overlay{$client} = $client->symbols('sleep');
		} else {
			$overlay{$client} = $client->symbols('bell');
			# Include the next alarm time in the overlay if there's room
			if (!$narrow && !defined $currentAlarm) {
				# Remove seconds from alarm time
				my $timeStr = Slim::Utils::DateTime::timeF($nextAlarm->time % 86400, $prefs->client($client)->timeformat, 1);
				$timeStr =~ s/(\d?\d\D\d\d)\D\d\d/$1/;
				$overlay{$client} .=  " $timeStr";
			}
		}
	}

	
	my $s2line1;
	my $s2line2;

	my $lastTicker = 0; #Flag set to indicate showing last ticker item
	my $hash;

	if ($nowshowing{$client} == 0 || ($client->display->hasScreen2())) { #Show time and temperature	
		my $time;
		if (defined($Plugins::FuzzyTime::Plugin::apiVersion)) {
			$time = Plugins::FuzzyTime::Public::timeF($client,undef,preferences('plugin.datetime')->get('timeformat'));
		}
		else {
			$time = Slim::Utils::DateTime::timeF(undef, preferences('plugin.datetime')->get('timeformat'));
		}
					
		if (defined($displayInfo{$client}{'TOPdisplayItems1'}[$topNowShowing{$client}])) { #Show next forecast
			#Figure out how much extra spacing is necessary from the 3line text to the lower line		
			my $max;
			if ($displayInfo{$client}{'CharLen2'}[$topNowShowing{$client}] > $displayInfo{$client}{'CharLen3'}[$topNowShowing{$client}]) {
				$max = $displayInfo{$client}{'CharLen2'}[$topNowShowing{$client}];
			}
			else {
				$max = $displayInfo{$client}{'CharLen3'}[$topNowShowing{$client}];
			}
		
			#If they're using a large font, include top line in maximum for spacing
			if ($client->textSize() == 2) { #Large text
				if ($displayInfo{$client}{'CharLen1'}[$topNowShowing{$client}] > $max) {
					$max = $displayInfo{$client}{'CharLen1'}[$topNowShowing{$client}];
				}
			}
			
			if ($max >2) {
				$max = $max - 1;
			}
					
			#Create a text string with the proper number of blank spaces		
			for (my $count=1; $count< $max; $count++)
			{
				$bottomPad = $bottomPad . $client->symbols("getMLB-41");
			}

			if ($displayInfo{$client}{'hasIcon'}[$topNowShowing{$client}] == 1) {
				$bottomPad = ' ' . $bottomPad . $client->symbols("getMLB-41"). $client->symbols("getMLB-41") . $client->symbols("getMLB-41") . $client->symbols("getMLB-41") . $client->symbols("getMLB-41");
			}
	
			$s2line1 = $displayInfo{$client}{'TOPdisplayItems1'}[$topNowShowing{$client}];
			$s2line1 =~ s/%1/$time/;
			$s2line2 = $bottomPad . $displayInfo{$client}{'BOTdisplayItems1'}[$topNowShowing{$client}];
			$s2line2 =~ s/%1/$time/;
			
			if ($nowshowing{$client} == 0) {
				$displayLine1{$client} = $s2line1;
				$displayLine2{$client} = $s2line2;
			}
		}
		else { #Show the time/date
			$displayLine1{$client} = Slim::Utils::DateTime::longDateF(undef, preferences('plugin.datetime')->get('dateformat'));
			$displayLine2{$client} = $time;
			$s2line1 = $displayLine1{$client};
			$s2line2 = $displayLine2{$client};
		}
	}

	if ($scrollType{$client} eq 'Ticker' && $killTicker !=3) { #TICKERS
		my $tickerScreen;
		if ($client->display->hasScreen2()) {
			$tickerScreen = 2;
		}
		else {
			$tickerScreen = 1;
		}		
		my ($complete, $queue) = $client->scrollTickerTimeLeft($tickerScreen);
		#$log->debug("Q:$queue C:$complete $nowshowing{$client} $displayLine1{$client} Kill:$killTicker");

		#Slim::Utils::Timers::listTimers();
		if ($nowshowing{$client} == 0) { #Time
			if ($complete == 0 || $killTicker == 1) { #Show the time
				$killTicker = 0; #Reset kill ticker in case it's toggeled
				#$::d_plugins && msg("SuperDateTime: *A* ". $nowshowing{$client} ." Show Time\n");
				$hash = {
					'overlay' => [$overlay{$client} . $status, undef],
					'center'  => [$displayLine1{$client}, $displayLine2{$client}],
				};
			}
			else { #Last ticker item still displaying
				$lastTicker = 1;
				#$log->debug("*C* $nowshowing{$client} Last Ticker Still Displaying");
				$hash = {
					'overlay'  => [ $overlay{$client} . $status, undef],
					'center'   => [ $tickline1{$client}, undef ],
					'ticker'   => [ undef, undef ],
				};
			}
		}	
		elsif ($queue <1 && $complete >0) { #Queue empty, item is still showing
			nextDisplayItem($client);
			if ($nowshowing{$client} != 0) { #Add item to ticker
				#$log->debug("*B* $nowshowing{$client} Add item to ticker");

				Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + ($complete*.5), \&topLineSet, $displayLine1{$client});	
						
				$hash = { 
					'overlay' => [ $overlay{$client} . $status, undef ],
					'center'  => [ $tickline1{$client}, undef ],
					'ticker'  => [ undef, $displayLine2{$client} ],
				};
			}
			else { #Last ticker item about to show time
				#$log->debug("*D* ". $nowshowing{$client} ." Last ticker item about to show time.");
				$lastTicker = 1;
				$hash = {
					'overlay' => [ $overlay{$client} . $status, undef ],
					'center'  => [ $tickline1{$client}, undef ],
					'ticker'  => [ undef, undef ],
				};		
			}
		}
		elsif ($complete > 0) { #Showing a ticker item
			#$log->debug("*E* ". $nowshowing{$client} ." Showing a ticker item.");
			$hash = {
				'overlay' => [ $overlay{$client} . $status, undef ],
				'center'  => [ $tickline1{$client}, undef ],
				'ticker'  => [ undef, undef ],
			};
		}
		else {  #Add first ticker item
			#$::d_plugins && msg("SuperDateTime: *F* ". $nowshowing{$client} ." Add first item to ticker.\n");
			$tickline1{$client} = $displayLine1{$client};
				$hash = {
					'overlay' => [ $overlay{$client} . $status, undef ],
					'center'  => [ $tickline1{$client}, undef ],
					'ticker'  => [ undef, $displayLine2{$client} ],
				};
		}
	}
	else {  #BASIC OR SLIDE
		if ($client->textSize() != 2) { #Not large text
			if ($nowshowing{$client} == 0 &&  $bottomPad ne '') { #Time with special spacing				
				$hash = {
					'overlay' => [ $overlay{$client} . $status, undef ],
					'center'  => [ $displayLine1{$client}, undef ],
					'line'    => [ undef, $displayLine2{$client} ],						    
				};
			}
			elsif ($nowshowing{$client} == 0) { #Time without special spacing
				$hash = {
					'overlay' => [ $overlay{$client} . $status, undef ],
					'center'  => [ $displayLine1{$client}, $displayLine2{$client} ],						    
				};
			}			
			elsif ($nowshowing{$client} < 0) { #Weather with scrolling
				$hash = {
					'overlay' => [ $overlay{$client} . $status ],
					'center'  => [ $displayLine1{$client} ],
					'line'    => [ undef, $displayLine2{$client} ],
				};
			}
			else {
				$hash = {  #Game
					'overlay' => [ $overlay{$client} . $status, undef ],
					'center'  => [ $displayLine1{$client}, $displayLine2{$client} ],
				};
			} 
	   } #End of not large text
	   else { #Large text
	   	if ($nowshowing{$client} == 0) { #Time
					$hash = {
						'overlay' => [ $overlay{$client} . $status, undef ],
						'center'  => [ undef,$displayLine2{$client}],
					};
			}
			else { #Sport scores/stocks in large text
				$hash = {
					'overlay' => [ $overlay{$client} . $status, undef ],
					'line'    => [ undef,$displayLine2{$client}],
				};
			}
	   }
	}
	
	my $dispHash;
	if ($client->display->hasScreen2()) { #Dual screens
		my $time;
		if (defined($Plugins::FuzzyTime::Plugin::apiVersion)) {
			$time = Plugins::FuzzyTime::Public::timeF($client,undef,preferences('plugin.datetime')->get('timeformat'));
		}
		else {
			$time = Slim::Utils::DateTime::timeF(undef, preferences('plugin.datetime')->get('timeformat'));
		}
				
		my $timeHash;
		if ($bottomPad ne '') { #Time with special spacing				
			$timeHash = {
				'center' => [ $s2line1, undef ],
				'line'   => [ undef, $s2line2 ],
			};
		}
		else {
			$timeHash = {
				'center'  => [ $s2line1, $s2line2 ],
			};		
		}

		if ($nowshowing{$client} != 0 || $lastTicker == 1) { #Not time
			$dispHash->{'screen2'} = $hash;
		}
		else {
			my $line1 = $displayInfo{$client}{'TOPdisplayItems2'}[$topNowShowing{$client}];
			$line1 =~ s/%1/$time/; #Add the time if its included
			
			my $line2 = $displayInfo{$client}{'BOTdisplayItems2'}[$topNowShowing{$client}];
			$line2 =~ s/%1/$time/; #Add the time if its included
		
			my $screen2 = {
					'overlay' => [ $overlay{$client} . $status, undef ],
					'center'  => [ $line1, $line2 ],
			};
			$dispHash->{'screen2'} = $screen2;
		}
		
		if ($xmax{$client} && $ymax{$client}) {
			$timeHash->{'bits'} = $displayInfo{$client}{'forecastG'}[$topNowShowing{$client}];
		}
		$dispHash->{'screen1'} = $timeHash;
	}
	else { #Single screen
		$hash->{'fonts'} = { 'graphic-280x16'  => { 'overlay' => [ 'small.1' ]},
								   'graphic-320x32'  => { 'overlay' => [ 'standard.1' ]},
								   'text' =>            { 'displayoverlays' => 1 },
								 };

		if ($xmax{$client} && $ymax{$client} && $nowshowing{$client} == 0 && $lastTicker == 0) {
			$hash->{'bits'} = $displayInfo{$client}{'forecastG'}[$topNowShowing{$client}];
		}
		$dispHash->{'screen1'} = $hash;
	}
	
	if ($currentAlarm && !$flash) {
		# schedule another update to remove the alarm symbol during alarm
		Slim::Utils::Timers::setTimer($client, Time::HiRes::time + 0.5, \&_flashAlarm);
	}
	
	return $dispHash;		
}

sub _flashAlarm {
	my $client = shift;
	
	$client->update( screensaverSuperDateTimelines($client, { flash => 1 }) );
}

1;

__END__

# Local Variables:
# tab-width:4
# indent-tabs-mode:t
# End:
