//Please read license.txt for licensing and copyright information

#ifndef SOLIDCOIN_WALLET_H
#define SOLIDCOIN_WALLET_H

#include "bignum.h"
#include "key.h"
#include "script.h"

class CWalletTx;
class CReserveKey;
class CWalletDB;

class CWallet : public CCryptoKeyStore
{
private:
    bool SelectCoinsMinConf(int64 nTargetValue, int nConfMine, int nConfTheirs, std::set<std::pair<const CWalletTx*,unsigned int> >& setCoinsRet, int64& nValueRet) const;
    bool SelectCoins(int64 nTargetValue, std::set<std::pair<const CWalletTx*,unsigned int> >& setCoinsRet, int64& nValueRet) const;

    CWalletDB *pwalletdbEncryption;
public:
    int64 m_nWalletLockTime;
    bool fFileBacked;
    std::string strWalletFile;
    std::string strWalletName;
    std::set<int64> setKeyPool;
    typedef std::map<unsigned int, CMasterKey> MasterKeyMap;
    MasterKeyMap mapMasterKeys;
    unsigned int nMasterKeyMaxID;
    std::map<uint256, CWalletTx> mapWallet;
    std::vector<uint256> vWalletUpdated;
    std::map<uint256, int> mapRequestCount;
    std::map<CSolidCoinAddress, std::string> mapAddressBook;
    std::vector<unsigned char> vchDefaultKey;

    /*
    CWallet()
    {
        fFileBacked = false;
        nMasterKeyMaxID = 0;
        pwalletdbEncryption = NULL;
    }
    */
    CWallet(const std::string &strWalletNameIn)
    {
        fFileBacked = true;
        nMasterKeyMaxID = 0;
        pwalletdbEncryption = NULL;
        setWalletName(strWalletNameIn);
    }

    const std::string GetWalletFileName()
    {
        return strWalletFile;
    }
    const std::string GetWalletName()
    {
        return strWalletName;
    }
    void setWalletName(const std::string &name)
    {
        strWalletName = name;
        strWalletFile = "wallet_"+strWalletName+".dat";
    }

    // keystore implementation
    bool AddKey(const CKey& key);
    bool LoadKey(const CKey& key) { return CCryptoKeyStore::AddKey(key); }
    bool AddCryptedKey(const std::vector<unsigned char> &vchPubKey, const std::vector<unsigned char> &vchCryptedSecret);
    bool LoadCryptedKey(const std::vector<unsigned char> &vchPubKey, const std::vector<unsigned char> &vchCryptedSecret) { return CCryptoKeyStore::AddCryptedKey(vchPubKey, vchCryptedSecret); }

    bool Unlock(const std::string& strWalletPassphrase);
    bool ChangeWalletPassphrase(const std::string& strOldWalletPassphrase, const std::string& strNewWalletPassphrase);
    bool EncryptWallet(const std::string& strWalletPassphrase);

    bool AddToWallet(const CWalletTx& wtxIn);
    bool AddToWalletIfInvolvingMe(const CTransaction& tx, const CBlock* pblock, bool fUpdate = false);
    bool EraseFromWallet(uint256 &hashTx);
    void WalletUpdateSpent(const CTransaction& prevout);
    int ScanForWalletTransactions(CBlockIndex* pindexStart, bool fUpdate = false);
    void ReacceptWalletTransactions();
    void ResendWalletTransactions();
    bool GetTrustedAddress(const std::string &pubkey, CTransaction *pTransaction, unsigned int *txout, int64 *pValue);
    int64 GetBalance() const;
    bool GetFees(int64 nValue, int64& nFeeRet);
    bool CreateTransaction(const std::vector<std::pair<CScript, int64> >& vecSend, CWalletTx& wtxNew, CReserveKey& reservekey, int64& nFeeRet);
    bool CreateTransaction(CScript scriptPubKey, int64 nValue, CWalletTx& wtxNew, CReserveKey& reservekey, int64& nFeeRet);
    bool CommitTransaction(CWalletTx& wtxNew, CReserveKey& reservekey);
    bool BroadcastTransaction(CWalletTx& wtxNew);
    std::string SendMoney(CScript scriptPubKey, int64 nValue, CWalletTx& wtxNew, int64 nMaxFee);
    std::string SendMoneyToSolidCoinAddress(const CSolidCoinAddress& address, int64 nValue, CWalletTx& wtxNew, int64 nMaxFee);

    bool TopUpKeyPool();
    void ReserveKeyFromKeyPool(int64& nIndex, CKeyPool& keypool);
    void KeepKey(int64 nIndex);
    void ReturnKey(int64 nIndex);
    std::vector<unsigned char> GetOrReuseKeyFromPool();
    int64 GetOldestKeyPoolTime();

    bool IsMine(const CTxIn& txin) const;
    int64 GetDebit(const CTxIn& txin) const;
    bool IsMine(const CTxOut& txout) const
    {
        return ::IsMine(*this, txout.scriptPubKey);
    }
    int64 GetCredit(const CTxOut& txout) const
    {
        if (txout.nValue<0)    throw std::runtime_error("CWallet::GetCredit() : value out of range");
        return (IsMine(txout) ? txout.nValue : 0);
    }
    bool IsChange(const CTxOut& txout) const
    {
        CSolidCoinAddress address;
        if (ExtractAddress(txout.scriptPubKey, this, address))
        {
            if (!mapAddressBook.count(address)) return true;
        }
        return false;
    }
    int64 GetChange(const CTxOut& txout) const
    {
        if (txout.nValue<0)  throw std::runtime_error("CWallet::GetChange() : value out of range");
        return (IsChange(txout) ? txout.nValue : 0);
    }
    bool IsMine(const CTransaction& tx) const
    {
        BOOST_FOREACH(const CTxOut& txout, tx.vout)
            if (IsMine(txout))
                return true;
        return false;
    }

    bool IsMine(uint256 &hashTx)
    {
        std::map<uint256, CWalletTx>::iterator mi = mapWallet.find(hashTx);
        if (mi != mapWallet.end())  return true;
        return false;
    }
    bool IsFromMe(const CTransaction& tx) const
    {
        return (GetDebit(tx) > 0);
    }
    int64 GetDebit(const CTransaction& tx) const
    {
        int64 nDebit = 0;
        BOOST_FOREACH(const CTxIn& txin, tx.vin)
        {
            nDebit += GetDebit(txin);
            if(nDebit<0)    throw std::runtime_error("CWallet::GetDebit() : value out of range");
        }
        return nDebit;
    }
    int64 GetCredit(const CTransaction& tx) const
    {
        int64 nCredit = 0;
        BOOST_FOREACH(const CTxOut& txout, tx.vout)
        {
            nCredit += GetCredit(txout);
            if (nCredit<0)   throw std::runtime_error("CWallet::GetCredit() : value out of range");
        }
        return nCredit;
    }
    int64 GetChange(const CTransaction& tx) const
    {
        int64 nChange = 0;
        BOOST_FOREACH(const CTxOut& txout, tx.vout)
        {
            nChange += GetChange(txout);
            if (nChange<0)   throw std::runtime_error("CWallet::GetChange() : value out of range");
        }
        return nChange;
    }
    void SetBestChain(const CBlockLocator& loc)
    {
        CWalletDB walletdb(strWalletFile);
        walletdb.WriteBestBlock(loc);
    }



    int LoadWallet(bool bClean);
//    bool BackupWallet(const std::string& strDest);

    // requires cs_mapAddressBook lock
    bool SetAddressBookName(const CSolidCoinAddress& address, const std::string& strName);

    // requires cs_mapAddressBook lock
    bool DelAddressBookName(const CSolidCoinAddress& address);

    void UpdatedTransaction(const uint256 &hashTx)
    {
        vWalletUpdated.push_back(hashTx);
    }

    void PrintWallet(const CBlock& block);

    void Inventory(const uint256 &hash)
    {
        std::map<uint256, int>::iterator mi = mapRequestCount.find(hash);
        if (mi != mapRequestCount.end())
            (*mi).second++;
    }

    int GetKeyPoolSize()
    {
        return setKeyPool.size();
    }

    bool GetTransaction(const uint256 &hashTx, CWalletTx& wtx);

    bool SetDefaultKey(const std::vector<unsigned char> &vchPubKey);
};


class CReserveKey
{
protected:
    CWallet* pwallet;
    int64 nIndex;
    std::vector<unsigned char> vchPubKey;
public:
    CReserveKey()
    {
        pwallet=0;
    }

    ~CReserveKey()
    {

        if (!fShutdown) ReturnKey();
    }

    void SetWallet(CWallet* pwalletIn)
    {
        nIndex = -1;
        pwallet = pwalletIn;
    }
    void ReturnKey();
    std::vector<unsigned char> GetReservedKey();
    void KeepKey();
};


// A transaction with a bunch of additional info that only the owner cares about.
// It includes any unrecorded transactions needed to link it back to the block chain.
class CWalletTx : public CMerkleTx
{
public:
    const CWallet* pwallet;

    std::vector<CMerkleTx> vtxPrev;
    std::map<std::string, std::string> mapValue;
    std::vector<std::pair<std::string, std::string> > vOrderForm;
    unsigned int fTimeReceivedIsTxTime;
    int64 nTimeReceived;  // time received by this node
    char fFromMe;
    std::string strFromAccount;
    std::vector<char> vfSpent;

    // memory only
    mutable char fDebitCached;
    mutable char fCreditCached;
    mutable char fAvailableCreditCached;
    mutable char fChangeCached;
    mutable int64 nDebitCached;
    mutable int64 nCreditCached;
    mutable int64 nAvailableCreditCached;
    mutable int64 nChangeCached;


    // memory only UI hints
    mutable unsigned int nTimeDisplayed;
    mutable int nLinesDisplayed;
    mutable char fConfirmedDisplayed;
    mutable unsigned char bConfirmedDisplayed;

    CWalletTx();
    CWalletTx(const CWallet* pwalletIn);
    CWalletTx(const CWallet* pwalletIn, const CMerkleTx& txIn);
    CWalletTx(const CWallet* pwalletIn, const CTransaction& txIn);
    void Init(const CWallet* pwalletIn);

    IMPLEMENT_SERIALIZE
    (
        CWalletTx* pthis = const_cast<CWalletTx*>(this);
        if (fRead)
            pthis->Init(NULL);
        char fSpent = false;

        if (!fRead)
        {
            pthis->mapValue["fromaccount"] = pthis->strFromAccount;

            std::string str;
            BOOST_FOREACH(char f, vfSpent)
            {
                str += (f ? '1' : '0');
                if (f)
                    fSpent = true;
            }
            pthis->mapValue["spent"] = str;
        }

        nSerSize += SerReadWrite(s, *(CMerkleTx*)this, nType, nVersion,ser_action);
        READWRITE(vtxPrev);
        READWRITE(mapValue);
        READWRITE(vOrderForm);
        READWRITE(fTimeReceivedIsTxTime);
        READWRITE(nTimeReceived);
        READWRITE(fFromMe);
        READWRITE(fSpent);

        if (fRead)
        {
            pthis->strFromAccount = pthis->mapValue["fromaccount"];

            if (mapValue.count("spent"))
                BOOST_FOREACH(char c, pthis->mapValue["spent"])
                    pthis->vfSpent.push_back(c != '0');
            else
                pthis->vfSpent.assign(vout.size(), fSpent);
        }

        pthis->mapValue.erase("fromaccount");
        pthis->mapValue.erase("version");
        pthis->mapValue.erase("spent");
    )

    // marks certain txout's as spent
    // returns true if any update took place
    bool UpdateSpent(const std::vector<char>& vfNewSpent);
    void MarkDirty();
    void MarkSpent(unsigned int nOut, bool bSpent);
    bool IsSpent(unsigned int nOut) const;
    int64 GetDebit() const;
    int64 GetCredit(bool fUseCache=true) const;
    int64 GetAvailableCredit(bool fUseCache=true) const;
    int64 GetChange() const;
    int64 GetConfirmed() const;
    void GetAmounts(int64& nGeneratedImmature, int64& nGeneratedMature, std::list<std::pair<CSolidCoinAddress, int64> >& listReceived, std::list<std::pair<CSolidCoinAddress, int64> >& listSent, int64& nFee, std::string& strSentAccount) const;
    void GetAccountAmounts(const std::string& strAccount, int64& nGenerated, int64& nReceived,int64& nSent, int64& nFee) const;
    bool IsFromMe() const;
    bool IsConfirmed() const;
    bool WriteToDisk();
    int64 GetTxTime() const;
    int GetRequestCount() const;
    void AddSupportingTransactions(CTxDB& txdb);
    bool AcceptWalletTransaction(CTxDB& txdb, bool fCheckInputs=true);
    bool AcceptWalletTransaction();
    void RelayWalletTransaction(CTxDB& txdb);
    void RelayWalletTransaction();
};


//
// Private key that includes an expiration date in case it never gets used.
//
class CWalletKey
{
public:
    CPrivKey vchPrivKey;
    int64 nTimeCreated;
    int64 nTimeExpires;
    std::string strComment;
    //// todo: add something to note what created it (user, getnewaddress, change)
    ////   maybe should have a map<string, string> property map

    CWalletKey(int64 nExpires=0)
    {
        nTimeCreated = (nExpires ? SolidTime_Get() : 0);
        nTimeExpires = nExpires;
    }

    IMPLEMENT_SERIALIZE
    (
        if (!(nType & SER_GETHASH))
            READWRITE(nVersion);
        READWRITE(vchPrivKey);
        READWRITE(nTimeCreated);
        READWRITE(nTimeExpires);
        READWRITE(strComment);
    )
};


// Account information. Stored in wallet with key "acc"+string account name
class CAccount
{
public:
    std::vector<unsigned char> vchPubKey;

    CAccount()
    {
        SetNull();
    }

    void SetNull()
    {
        vchPubKey.clear();
    }

    IMPLEMENT_SERIALIZE
    (
        if (!(nType & SER_GETHASH))
            READWRITE(nVersion);
        READWRITE(vchPubKey);
    )
};



//
// Internal transfers.
// Database key is acentry<account><counter>
//
class CAccountingEntry
{
public:
    std::string strAccount;
    int64 nCreditDebit;
    int64 nTime;
    std::string strOtherAccount;
    std::string strComment;

    CAccountingEntry()
    {
        SetNull();
    }

    void SetNull()
    {
        nCreditDebit = 0;
        nTime = 0;
        strAccount.clear();
        strOtherAccount.clear();
        strComment.clear();
    }

    IMPLEMENT_SERIALIZE
    (
        if (!(nType & SER_GETHASH))
            READWRITE(nVersion);
        // Note: strAccount is serialized as part of the key, not here.
        READWRITE(nCreditDebit);
        READWRITE(nTime);
        READWRITE(strOtherAccount);
        READWRITE(strComment);
    )
};

#endif
