//Please read license.txt for licensing and copyright information

#include "block.h"

using namespace std;

const uint64 TRUST_FUND_AMOUNT = 1000000;
const unsigned int MAX_BLOCK_SIZE = 204800;
const int MAX_BLOCK_SIGOPS = MAX_BLOCK_SIZE/50;
CScript g_TrustedNodeScript;
CScript g_CPFScript;
std::string g_TrustedNodePublicKey;

char *g_TrustPubKeys[TRUST_FUND_NUM] = {
    "0489a46fe1a47dec74fb9e08cc0579b0a17e2667eca7edd1a5a4f28c6a53efb5af936b03dd7542bae2de1e5f2afa515209cfd3092c2818690234a62d291428a289",
    "0484e16a91b5d30f474856e5b6093fb43f0223a4013695c0ad2b04a2897b6137cd9fff4324821630a20db1a4c201a40e0fc98738cd73c0bc4cc969650abb6298ac",
    "04637ea5acb71dff440cd99c888bf294dd857051d91cf4f23f88451ed6b0ea9eac1908caee1dda765ff6a2013fa8d36498a9edeabf6a565762859ca42176e1f77f",
    "04bfd5db443c347f359b0771997e1cc6e7f73e362e8bf4c4a8e11983401c3b4c0f0b402a980a74973f010162bdbbf81576831bc8dc492077e5c1564ca56a3590c8",
    "040066e4ae95e759fa6f38c42920dc2f18d36b5452ec13006fdd207321ca963b3d50560912e8d3eb5c936c642c304dfcb2d04c05356d57da9ffc5fd7125ef7fdf8",

    "043d52842e0b15506b2192e25ac17e06b84a3a4fcb5dde47ed31e7961bc90a55224c2d719f7ec6a2fee0494bace94ee4d045e43ca6d4bfe9ab3887d29026b0bf4d",
    "041cc11ad7e6799531d06387b3af25170d57361ae347eceb6940f61529ca3ef88762bbe336e955f32100e30b42e4ab5047807dcb98d4e0256dc89afa0f954d6aaf",
    "048898f7b2e183eadc5f248e3050f9aeeed3d7e7f382058b70cacd06f36f6dfedc6924253dcd662af9e33af81ccb5a6cf487f8c9d7cb740ed3e12f50378090c334",
    "04712e3516ce7e4c3dc584c3e13a7d8212499aabb372cc6879fbc21b99098e6e748eeceb0d1f9f5237e92003e865f74d024028d3c3e8a3ad6fceade454860402aa",
    "041ac115c2d62e7e62914f3e6586d143952c9ead2876e903b76db85ed20775720b1cb3550a8deed288bfc1c72d0d9d70d121f3816dff97d8402b1b860927770ec8"
};

CSolidCoinAddress g_TrustPubKeysHash[TRUST_FUND_NUM];

uint256 g_BlockGenesisHash("0x00000fbec280ced7f08336d89a68911f0bced51e936e07268bfead880ccb3d40");
CBigNum g_bnBlockProofOfWorkLimit(~uint256(0) >> 17);
CBigNum g_bnBlockBestChainWork = 0;
CBigNum g_bnBlockBestInvalidWork = 0;
const int64 g_qBlockTotalEstimate = 52000; // Conservative estimate of total nr of blocks on main chain
const int64 g_qBlockInitialThreshold = 120; // Regard blocks up until N-threshold as "initial download"
std::map<uint256, CBlockIndex*> g_BlockIndexMap;
CBlockIndex* g_pBlockGenesisIndex = 0;
CBlockIndex* g_pBlockBestIndex = 0;
int64 g_qBlockBestHeight = -1;

uint256 g_BlockBestChainHash = 0;
int64 g_qTimeBestReceived = 0;
std::map<uint256, CBlock*> g_BlockOrphanMap;
std::multimap<uint256, CBlock*> g_BlockOrphanMapByPrev;

CScript g_GenScript;
CScript g_TrustedScript[TRUST_FUND_NUM];

int64 Block_GetTotalNumEstimate(void)
{
    if(fTestNet)    return 0;
    else            return g_qBlockTotalEstimate;
}

CBigNum GetBlockWork(uint32 dwBits)
{
    CBigNum bnTarget;
    bnTarget.SetCompact(dwBits);
    if (bnTarget <= 0)  return 0;
    return (CBigNum(1)<<256) / (bnTarget+1);
}


bool Block_IsInitialDownload(void)
{
    if (g_pBlockBestIndex == NULL || g_qBlockBestHeight < (Block_GetTotalNumEstimate()-g_qBlockInitialThreshold))  return true;
    static int64 nLastUpdate;
    static CBlockIndex* pindexLastBest;
    if (g_pBlockBestIndex != pindexLastBest)
    {
        pindexLastBest = g_pBlockBestIndex;
        nLastUpdate = SolidTime_Get();
    }
    return (SolidTime_Get() - nLastUpdate < 10 &&   g_pBlockBestIndex->blk.nTime < SolidTime_Get() - 24 * 60 * 60);
}

void Block_InvalidChainFound(CBlockIndex* pindexNew)
{
    if (pindexNew->bnChainWork > g_bnBlockBestInvalidWork)
    {
        g_bnBlockBestInvalidWork = pindexNew->bnChainWork;
        CTxDB().WriteBestInvalidWork(g_bnBlockBestInvalidWork);
        //MainFrameRepaint();
    }
    debugprintf(WARN, "InvalidChainFound: invalid block=%s  height=%"PRI64d"  work=%s\n", pindexNew->GetBlockHash().ToString().substr(0,20).c_str(), pindexNew->blk.nBlockNum, pindexNew->bnChainWork.ToString().c_str());
    debugprintf(WARN, "InvalidChainFound:  current best=%s  height=%"PRI64d"  work=%s\n", g_BlockBestChainHash.ToString().substr(0,20).c_str(), g_qBlockBestHeight, g_bnBlockBestChainWork .ToString().c_str());
    if (g_pBlockBestIndex && g_bnBlockBestInvalidWork > g_bnBlockBestChainWork  + GetBlockWork(g_pBlockBestIndex->blk.dwBits) * 6)
        debugprintf(WARN, "InvalidChainFound: WARNING: Displayed transactions may not be correct!  You may need to upgrade, or other nodes may need to upgrade.\n");
}

bool Block_LoadIndex(bool fAllowNew)
{

    if(g_TrustedNodePublicKey.length()) g_TrustedNodeScript = CScript() << ParseHex(g_TrustedNodePublicKey) << OP_CHECKSIG;

    g_GenScript = CScript() << ParseHex(GEN_KEY) << OP_CHECKSIG;
    g_CPFScript = CScript() << ParseHex(CPF_KEY) << OP_CHECKSIG;



    for(int x=0;x<TRUST_FUND_NUM;x++)
    {
        const std::vector<unsigned char> vchPubKey(g_TrustPubKeys[x],g_TrustPubKeys[x]+strlen(g_TrustPubKeys[x]));
        g_TrustPubKeysHash[x] = CSolidCoinAddress(vchPubKey);
        g_TrustedScript[x] = CScript() << ParseHex(g_TrustPubKeys[x]) << OP_CHECKSIG;;
    }

    if (fTestNet)
    {
        //hashGenesisBlock = uint256("0x00000007199508e34a9ff81e6ec0c477a4cccff2a4767a8eee39c11db367b008");
        g_BlockGenesisHash = uint256("0x00000212745b8ec01ea165dedb8613b8e96a39a316e0031bf5938667f3e76c2e");
        //bnProofOfWorkLimit = CBigNum(~uint256(0) >> 19);
        g_bnBlockProofOfWorkLimit = CBigNum(~uint256(0) >> 17);
        pchMessageStart[0] = 0xf0;
        pchMessageStart[1] = 0xbf;
        pchMessageStart[2] = 0xb0;
        pchMessageStart[3] = 0xd8;
    }

    // Load block index
    CTxDB txdb("cr");
    if (!txdb.LoadBlockIndex()) return false;
    txdb.Close();

    // Init with genesis block
    if (g_BlockIndexMap.empty())
    {
        if (!fAllowNew) return false;
        CTransaction txNew;
        CBlock block;
        const char* pszTimestamp;
        unsigned char trustname[8]={0};

        if (fTestNet)
        {
            //pszTimestamp = "The Times 03/Jan/2009 Chancellor on brink of second bailout for banks";
            pszTimestamp = "New York Times | 6/October/2011 | Anti-Wall Street protesters marched to the capital";
            txNew.vin.resize(1);
            txNew.vout.resize(1);
            txNew.vin[0].prevout.SetNull();
            txNew.vin[0].scriptSig = CScript() << CBigNum(0) << vector<unsigned char>((const unsigned char*)pszTimestamp, (const unsigned char*)pszTimestamp + strlen(pszTimestamp));
            txNew.vout[0].nValue = Block_GetCoinBaseValue(0,0);
            txNew.vout[0].scriptPubKey = g_GenScript;
            block.vtx.push_back(txNew);

            txNew.vout[0].nValue = (TRUST_FUND_AMOUNT+200000)*COIN;
            for(int x=0;x<TRUST_FUND_NUM;x++)
            {
                sprintf((char*)trustname,"TRUST%02d",x+1);
                txNew.vin[0].scriptSig = CScript() << CBigNum(x+1) << vector<unsigned char>(trustname, trustname+7);
                txNew.vout[0].scriptPubKey = g_TrustedScript[x];
                block.vtx.push_back(txNew);
            }

            block.blk.nBlockNum=0;
            block.blk.hashPrevBlock = 0;
            block.blk.hashMerkleRoot = block.BuildMerkleTree();
            block.blk.nVersion = 1;
            block.blk.nTime    = 1318051965;
            block.blk.dwBits    = 503840767 ;

            block.blk.nNonce1   = 152300;
            block.blk.nNonce2   = 0;
            block.blk.nNonce3   = 0;
            block.blk.nNonce4   = 0;
            strcpy(block.blk.miner_id,"RealSolid");
            assert(block.blk.hashMerkleRoot == uint256("0x8d699b0a5524b3d8773ad46f0b974add6e922f422e7d99a8816ef17d712056ee"));
            //if(block.blk.hashMerkleRoot != uint256("0x20b13f4cf37a0d49fb9e377b7ca7562de7518537cbd54f4c358f36b42e2a3ac3"))
            {
            }
        }
        else
        {
            pszTimestamp = "New York Times | October 9 2011 | Recession Officially Over, U.S. Incomes Kept Falling.";
            txNew.vin.resize(1);
            txNew.vout.resize(1);
            txNew.vin[0].prevout.SetNull();
            txNew.vin[0].scriptSig = CScript() << CBigNum(0) << vector<unsigned char>((const unsigned char*)pszTimestamp, (const unsigned char*)pszTimestamp + strlen(pszTimestamp));
            txNew.vout[0].nValue = Block_GetCoinBaseValue(0,0);
            txNew.vout[0].scriptPubKey = g_GenScript;
            block.vtx.push_back(txNew);

            txNew.vout[0].nValue = (TRUST_FUND_AMOUNT+200000)*COIN;
            for(int x=0;x<TRUST_FUND_NUM;x++)
            {
                sprintf((char*)trustname,"TRUST%02d",x+1);
                txNew.vin[0].scriptSig = CScript() << CBigNum(x+1) << vector<unsigned char>(trustname, trustname+7);
                txNew.vout[0].scriptPubKey = g_TrustedScript[x];
                block.vtx.push_back(txNew);
            }

            block.blk.nBlockNum=0;
            block.blk.hashPrevBlock = 0;
            block.blk.hashMerkleRoot = block.BuildMerkleTree();
            block.blk.nVersion = 1;
            block.blk.nTime    = 1318263519;
            block.blk.dwBits    = 504365055 ;

            block.blk.nNonce1   = 778981;
            block.blk.nNonce2   = 0;
            block.blk.nNonce3   = 0;
            block.blk.nNonce4   = 0;
            strcpy(block.blk.miner_id,"RealSolid");
            assert(block.blk.hashMerkleRoot == uint256("0x41d9f7968dbcc8fa5139ffa98ca40257a42abb2e6b0068f85a2981ca94f1a3d2"));
        }

        int nGenerateNewGenBlock=0;
        if(nGenerateNewGenBlock)
        {
            CBigNum bn=g_bnBlockProofOfWorkLimit;
            bn=bn/8;

            block.blk.nNonce1=0;
            block.blk.nNonce2=0;
            block.blk.nNonce3=0;
            block.blk.nNonce4=0;
            block.blk.dwBits=bn.GetCompact();
            debugprintf(INFO, "proof of work: %s\n",bn.ToString().c_str());
            uint256 target=bn.getuint256();
            while (block.GetHash() > target)    ++block.blk.nNonce1;


            FILE * pFile;
            pFile = fopenSOLID("genblock.txt","w");
            fprintf(pFile,"timestamp text: %s \n",pszTimestamp);
            fprintf(pFile,"block.nTime = %"PRI64d" \n", block.blk.nTime);
            fprintf(pFile,"block.dwBits = %u \n", block.blk.dwBits);
            fprintf(pFile,"block.nBlockNum = %"PRI64d" \n", block.blk.nBlockNum);
            fprintf(pFile,"block.nNonce1 = %"PRI64d" \n", block.blk.nNonce1);
            fprintf(pFile,"block.nNonce2 = %"PRI64d" \n", block.blk.nNonce2);
            fprintf(pFile,"block.nNonce3 = %"PRI64d" \n", block.blk.nNonce3);
            fprintf(pFile,"block.nNonce4 = %d\n", block.blk.nNonce4);
            fprintf(pFile,"block.GetHash = %s\n", block.GetHash().ToString().c_str());
            fprintf(pFile,"block.hashMerkleRoot = %s\n", block.blk.hashMerkleRoot.ToString().c_str());
            fclose (pFile);
        }


        debugprintf(INFO, "%s\n", block.GetHash().ToString().c_str());
        debugprintf(INFO, "%s\n", g_BlockGenesisHash.ToString().c_str());
        debugprintf(INFO, "%s\n", block.blk.hashMerkleRoot.ToString().c_str());
        block.print();
        assert(block.GetHash() == g_BlockGenesisHash);

        // Start new block file
        unsigned int nFile;
        unsigned int nBlockPos;
        if (!block.WriteToDisk(nFile, nBlockPos))       return error("LoadBlockIndex() : writing genesis block to disk failed");
        if (!block.AddToBlockIndex(nFile, nBlockPos))   return error("LoadBlockIndex() : genesis block not accepted");
    }

    return true;
}



double GetDifficulty(uint32 dwBits);
int64 Block_GetCoinBaseValue(uint32 dwWorkBits, int64 nHeight)
{
    if(nHeight<50000)
    {
        if(nHeight==0)  return 1200000*COIN;

        int64 qBaseValue = 32;
        qBaseValue -= (nHeight/1000000);
        if(qBaseValue<4) qBaseValue=4;

        qBaseValue*=COIN;

        double dCoinInflationModifier = (GetDifficulty(dwWorkBits)/100000.0);
        qBaseValue += dCoinInflationModifier*qBaseValue;
        if( (nHeight%2) == 0)
        {
            return ((int64)(qBaseValue*CPF_PERCENT))+1;
        }
        return qBaseValue;
    }
    else if(nHeight<180000)
    {
        double dBaseValue = 5;
        dBaseValue -=  ((int64)(nHeight/1000000)) *0.1;
        if(dBaseValue<1) dBaseValue=1;
        dBaseValue=dBaseValue*COIN;
        double dCoinInflationModifier = (GetDifficulty(dwWorkBits)/100000.0);
        dBaseValue += dCoinInflationModifier*dBaseValue;
        if( (nHeight%2) == 0)
        {
            return ((int64)(dBaseValue*CPF_PERCENT))+1;
        }
        return (int64)dBaseValue;
    }


    double dDifficulty = GetDifficulty(dwWorkBits);

    //We need to work out how many years it has been running to calculate estimated
    //electricity price increases and hardware improvements
    double dYearsPassed = ( (double)(nHeight-179999) /(60*24*365));

    //moores law type adjustment. Assume 50% better efficiency per year
    double dKH_Per_Watt = 3.0 * pow(1.50,dYearsPassed);

    //We start at 8c per kilowatt hour, compound increase it 10% per year
    //this is the estimated increased cost of electricity per year
    double dCostPerKiloWattHour = 0.08*pow(1.10,dYearsPassed);
    double dTotalKWh = ((dDifficulty*131)/dKH_Per_Watt) / (3600*1000);

    double dBaseValue = dTotalKWh*dCostPerKiloWattHour;

    if( (nHeight%2) == 0)
    {
        //trust blocks are only worth 5% of normal blocks
        dBaseValue = (dBaseValue*CPF_PERCENT);
    }

    if(dBaseValue<0.0001)   dBaseValue=0.0001;      //0.0001 SC is the minimum amount and each block needs at least that to be worth something
    dBaseValue+=0.05;   //also add a minimum amount to cover the existing fee

    dBaseValue=dBaseValue*COIN;
    return (int64)dBaseValue;
}


bool Block_IsPowerBlock(const uint256 &trustedhash,const uint256 &userhash)
{
    unsigned char trustedcount=trustedhash.SumOfBytes();
    unsigned char usercount=userhash.SumOfBytes();
    if(trustedcount>=128 && usercount>=239) return true;        //1 in 16 chance for user, 1 in 2 for trusted, = 1 in 32 total
    return false;
}

void Block_PrintTree()
{
    // precompute tree structure
    map<CBlockIndex*, vector<CBlockIndex*> > mapNext;
    for (map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.begin(); mi != g_BlockIndexMap.end(); ++mi)
    {
        CBlockIndex* pindex = (*mi).second;
        mapNext[pindex->pprev].push_back(pindex);
        // test
        //while (rand() % 3 == 0)
        //    mapNext[pindex->pprev].push_back(pindex);
    }

    vector<pair<int, CBlockIndex*> > vStack;
    vStack.push_back(make_pair(0, g_pBlockGenesisIndex));

    int nPrevCol = 0;
    while (!vStack.empty())
    {
        int nCol = vStack.back().first;
        CBlockIndex* pindex = vStack.back().second;
        vStack.pop_back();

        // print split or gap
        if (nCol > nPrevCol)
        {
            for (int i = 0; i < nCol-1; i++)
                debugprintf(INFO, "| ");
            debugprintf(INFO, "|\\\n");
        }
        else if (nCol < nPrevCol)
        {
            for (int i = 0; i < nCol; i++)
                debugprintf(INFO, "| ");
            debugprintf(INFO, "|\n");
       }
        nPrevCol = nCol;

        // print columns
        for (int i = 0; i < nCol; i++)
            debugprintf(INFO, "| ");

        // print item
        CBlock block;
        block.ReadFromDisk(pindex);
        debugprintf(INFO, "%"PRI64d" (%u,%u) %s  %s  tx %d",pindex->blk.nBlockNum,pindex->nFile,pindex->nBlockPos,block.GetHash().ToString().substr(0,20).c_str(),DateTimeStrFormat("%x %H:%M:%S", block.blk.nTime).c_str(),block.vtx.size());
        {
            LOCK_WALLET_ACCESS();
            Wallet_PrintWallets(block);
        }

        // put the main timechain first
        vector<CBlockIndex*>& vNext = mapNext[pindex];
        for (int i = 0; i < vNext.size(); i++)
        {
            if (vNext[i]->pnext)
            {
                swap(vNext[0], vNext[i]);
                break;
            }
        }

        // iterate children
        for (int i = 0; i < vNext.size(); i++)
            vStack.push_back(make_pair(nCol+i, vNext[i]));
    }
}




bool CBlock::SetBestChain(CTxDB& txdb, CBlockIndex* pindexNew)
{
    uint256 hash = GetHash();

    txdb.TxnBegin();
    if (g_pBlockGenesisIndex == NULL && hash == g_BlockGenesisHash)
    {
        ConnectBlock(txdb, pindexNew,false);
        txdb.WriteHashBestChain(hash);
        if (!txdb.TxnCommit())  return error("SetBestChain() : TxnCommit failed");
        g_pBlockGenesisIndex = pindexNew;
    }
    else if (blk.hashPrevBlock == g_BlockBestChainHash)
    {
        // Adding to current best branch
        if (!ConnectBlock(txdb, pindexNew,true) || !txdb.WriteHashBestChain(hash))
        {
            txdb.TxnAbort();
            Block_InvalidChainFound(pindexNew);
            return error("SetBestChain() : ConnectBlock failed");
        }
        if (!txdb.TxnCommit())
            return error("SetBestChain() : TxnCommit failed");

        // Add to current best branch
        pindexNew->pprev->pnext = pindexNew;

        // Delete redundant memory transactions
        BOOST_FOREACH(CTransaction& tx, vtx)
        {
            tx.RemoveFromMemoryPool();
        }
    }
    else
    {
        // New best branch
        if (!Block_Reorganize(txdb, pindexNew))
        {
            txdb.TxnAbort();
            Block_InvalidChainFound(pindexNew);
            return error("SetBestChain() : Reorganize failed");
        }
    }

    // Update best block in wallet (so we can detect restored wallets)
    if (!Block_IsInitialDownload())
    {
        LOCK_WALLET_ACCESS();
        const CBlockLocator locator(pindexNew);
        Wallet_SetBestChain(locator);
    }

    // New best block
    g_BlockBestChainHash = hash;
    g_pBlockBestIndex = pindexNew;
    g_qBlockBestHeight = g_pBlockBestIndex->blk.nBlockNum;
    g_bnBlockBestChainWork  = pindexNew->bnChainWork;
    g_qTimeBestReceived = SolidTime_Get();
    nTransactionsUpdated++;
    debugprintf(INFO, "SetBestChain: new best=%s  height=%"PRI64d"  work=%s\n", g_BlockBestChainHash.ToString().substr(0,20).c_str(), g_qBlockBestHeight, g_bnBlockBestChainWork .ToString().c_str());
    return true;
}


void CBlock::CalculateChainWork(CBlockIndex* pindex)
{
    if(pindex->pprev)
    {
        if((blk.nBlockNum%2) == 0)
        {
            CBigNum bnTarget;
            if(blk.nBlockNum<43000)
            {
                bnTarget.SetCompact(pindex->pprev->blk.dwBits);
                bnTarget *= vtx[1].vout[0].nValue;
                bnTarget /= 1000000*COIN;
                bnTarget = (CBigNum(1)<<256) / (bnTarget+1);
            }
            else
            {
                bnTarget.SetCompact(pindex->pprev->blk.dwBits);
                bnTarget = (CBigNum(1)<<256) / (bnTarget+1);
                bnTarget *= vtx[1].vout[0].nValue;
                bnTarget /= 1000000*COIN;
            }

            pindex->bnChainWork = pindex->pprev->bnChainWork + bnTarget;
            //debugprintf(VERBOSE, "last work: %s\nnew work: %s\ncombined: %s\n\n",pindex->pprev->bnChainWork.ToString().c_str(),bnTarget.ToString().c_str(),pindex->bnChainWork.ToString().c_str());
        }
        else
            pindex->bnChainWork=pindex->pprev->bnChainWork + GetBlockWork(pindex->blk.dwBits);
    }
    else    //genesis block
    {
        pindex->bnChainWork = GetBlockWork(pindex->blk.dwBits);
    }
}

//bool CBlock::CreateBlockIndex(CBlockIndex*)


bool CBlock::AddToBlockIndex(unsigned int nFile, unsigned int nBlockPos)
{
    uint256 hash = GetHash();

    // Construct new block index object
    // add it to block index even if it fails this test, just in case we later need it?
    CBlockIndex* pindexNew = new CBlockIndex(nFile, nBlockPos, *this);
    if (!pindexNew) return error("AddToBlockIndex() : new CBlockIndex failed");
    map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.insert(make_pair(hash, pindexNew)).first;
    pindexNew->phashBlock = &((*mi).first);
    map<uint256, CBlockIndex*>::iterator miPrev = g_BlockIndexMap.find(blk.hashPrevBlock);
    if (miPrev != g_BlockIndexMap.end())
    {
        pindexNew->pprev = (*miPrev).second;
    }
    CalculateChainWork(pindexNew);
    if(pindexNew->bnChainWork == g_bnBlockBestChainWork)
    {
        if(g_TrustedNodePublicKey.length())                                 return error("AddToBlockIndex() : trusted node got valid regular block already");
        if(pindexNew->blk.nBlockNum && (pindexNew->blk.nBlockNum%2)==0)     return error("AddToBlockIndex() : received different block hash with same amount of work in trust block");
    }

    CTxDB txdb;
    txdb.TxnBegin();
    txdb.WriteBlockIndex(CDiskBlockIndex(pindexNew));
    if (!txdb.TxnCommit())
        return false;

    // New best
    if (pindexNew->bnChainWork > g_bnBlockBestChainWork || pindexNew->bnChainWork == g_bnBlockBestChainWork)
    {
        if (!SetBestChain(txdb, pindexNew)) return false;
    }

    txdb.Close();

    if (pindexNew == g_pBlockBestIndex)
    {
        // Notify UI to display prev block's coinbase if it was ours
        LOCK_WALLET_ACCESS();
        static uint256 hashPrevBestCoinBase;
        Wallet_UpdatedTransaction(hashPrevBestCoinBase);
        hashPrevBestCoinBase = vtx[0].GetHash();
    }

    MainFrameRepaint();
    return true;
}


CBlock* Block_Create(CWallet *pWallet, const std::string &mining_id,CReserveKey& reservekey)
{
    CTransaction txNew;
    CBlockIndex* pindexPrev = g_pBlockBestIndex;


    // Create new block
    auto_ptr<CBlock> pblock(new CBlock());

    txNew.vin.resize(1);
    txNew.vout.resize(1);
    txNew.vin[0].prevout.SetNull();     // Create coinbase tx

    pblock->blk.nBlockNum = pindexPrev->blk.nBlockNum+1;
    pblock->blk.dwBits= Block_GetNextWorkRequired(pindexPrev);
    int64 qBlockValue=Block_GetCoinBaseValue(pblock->blk.dwBits,pblock->blk.nBlockNum);
    int64 nFees = 0;

    if( (pblock->blk.nBlockNum%2) == 0 && g_TrustedNodePublicKey.length()>0)     //trusted node block
    {
        CWallet *pTrustedWallet = Wallet_Get("trusted");
        if(pTrustedWallet==0)   return 0;

        if(pblock->blk.nBlockNum>=42000)
        {
            //from block 42000 onward we calculate 5% of each block correctly
            qBlockValue=Block_GetCoinBaseValue(pindexPrev->blk.dwBits,pblock->blk.nBlockNum); //get last index value
        }
        //debugprintf(VERBOSE, "[%d] %"PRI64d" value\n",(int)pblock->blk.nTime,qBlockValue);

        //coin base is a payout to CPF
        txNew.vout[0].scriptPubKey = CScript() << ParseHex(CPF_KEY) << OP_CHECKSIG;
        txNew.vin[0].scriptSig = CScript() << CBigNum(pblock->blk.nBlockNum);       //we merkle with this set, but its actually changed after the fact to store the signature
        pblock->vtx.push_back(txNew);

        //second tx is the trusted node losing CPF payment supporting the network
        CTransaction txTrusted,txPrev;
        unsigned int trustedtx=0;
        int64 qTrustedValue;

        pblock->m_trust_pubkey=ParseHex(g_TrustedNodePublicKey);

        if(!pTrustedWallet->GetTrustedAddress(g_TrustedNodePublicKey,&txPrev,&trustedtx,&qTrustedValue)) return 0;
        txTrusted.vin.push_back(CTxIn(txPrev.GetHash(),trustedtx));
        txTrusted.vout.resize(1);
        txTrusted.vout[0].scriptPubKey = CScript() << pblock->m_trust_pubkey << OP_CHECKSIG;
        txTrusted.vout[0].nValue=qTrustedValue-qBlockValue;
        nFees+=qBlockValue;
        if (!SignSignature(*pTrustedWallet, txPrev, txTrusted, 0))  return 0;
        txTrusted.CheckTransaction(pblock->blk.nBlockNum);
        pblock->vtx.push_back(txTrusted);

        //we want to get the private key in the wallet so we can sign block data later on in the function
        if (!pTrustedWallet->GetKey(Hash160(pblock->m_trust_pubkey), pblock->m_trust_privkey))
        {
            debugprintf(ERR, "block_create: Couldnt find private key");
            return 0;
        }

        if (pblock->m_trust_privkey.GetPubKey() != pblock->m_trust_pubkey)      //make sure the key we found has same public key
        {
            debugprintf(ERR, "block_create: Private key found didn't match pubkey");
            return 0;
        }
    }
    else    //normal block
    {
        if( (pblock->blk.nBlockNum%2)==0)
        {
            pblock->blk=pindexPrev->blk;
            qBlockValue=Block_GetCoinBaseValue(pblock->blk.dwBits,pblock->blk.nBlockNum);
            pindexPrev = pindexPrev->pprev; //go back to the block our newest it think we are the block before

        }
        if(pblock->blk.nBlockNum>=50000 && pindexPrev->pprev && Block_IsPowerBlock(pindexPrev->GetBlockHash(),pindexPrev->pprev->GetBlockHash()))
        {
            pblock->m_bPower=true;
            qBlockValue*=2;
        }
        txNew.vin[0].scriptSig = CScript() << CBigNum(GetRand(0xFFFFFFFF)) <<  CBigNum(pblock->blk.nBlockNum);
        txNew.vout[0].scriptPubKey << reservekey.GetReservedKey() << OP_CHECKSIG;
        pblock->vtx.push_back(txNew);
    }


    {
        CTxDB txdb("r");

        // Priority order to process transactions
        list<COrphan> vOrphan; // list memory doesn't move
        map<uint256, vector<COrphan*> > mapDependers;
        multimap<double, CTransaction*> mapPriority;
        for (map<uint256, CTransaction>::iterator mi = mapTransactions.begin(); mi != mapTransactions.end(); ++mi)
        {
            CTransaction& tx = (*mi).second;
            if (tx.IsCoinBase() || !tx.IsFinal())   continue;

            COrphan* porphan = NULL;
            double dPriority = 0;
            BOOST_FOREACH(const CTxIn& txin, tx.vin)
            {
                // Read prev transaction
                CTransaction txPrev;
                CTxIndex txindex;
                if (!txPrev.ReadFromDisk(txdb, txin.prevout, txindex))
                {
                    // Has to wait for dependencies
                    if (!porphan)
                    {
                        // Use list for automatic deletion
                        vOrphan.push_back(COrphan(&tx));
                        porphan = &vOrphan.back();
                    }
                    mapDependers[txin.prevout.hash].push_back(porphan);
                    porphan->setDependsOn.insert(txin.prevout.hash);
                    continue;
                }
                int64 nValueIn = txPrev.vout[txin.prevout.n].nValue;
                int64 nConf = txindex.GetDepthInMainChain();    // Read block header
                dPriority += (double)nValueIn * nConf;

                if (fDebug && Setting_GetBOOL("printpriority")) debugprintf(ALWAYS, "priority     nValueIn=%-12I64d nConf=%-5"PRI64d" dPriority=%-20.1f\n", nValueIn, nConf, dPriority);
            }

            // Priority is sum(valuein * age) / txsize
            dPriority /= ::GetSerializeSize(tx, SER_NETWORK);

            if (porphan)    porphan->dPriority = dPriority;
            else            mapPriority.insert(make_pair(-dPriority, &(*mi).second));

            if (fDebug && Setting_GetBOOL("printpriority"))
            {
                debugprintf(ALWAYS, "priority %-20.1f %s\n%s", dPriority, tx.GetHash().ToString().substr(0,10).c_str(), tx.ToString().c_str());
                if (porphan)
                    porphan->print();
                debugprintf(ALWAYS, "\n");
            }
        }

        // Collect transactions into block
        map<uint256, CTxIndex> mapTestPool;
        uint64 nBlockSize = 1000;
        int nBlockSigOps = 100;
        while (!mapPriority.empty())
        {
            // Take highest priority transaction off priority queue
            double dPriority = -(*mapPriority.begin()).first;
            CTransaction& tx = *(*mapPriority.begin()).second;
            mapPriority.erase(mapPriority.begin());

            // Size limits
            unsigned int nTXSize = ::GetSerializeSize(tx, SER_NETWORK);
            if (nBlockSize + nTXSize >= MAX_BLOCK_SIZE)     continue;
            int nTxSigOps = tx.GetSigOpCount();
            if (nBlockSigOps + nTxSigOps >= MAX_BLOCK_SIGOPS)   continue;

            // Connecting shouldn't fail due to dependency on other memory pool transactions because we're already processing them in order of dependency
            map<uint256, CTxIndex> mapTestPoolTmp(mapTestPool);
            if (!tx.ConnectInputs(txdb, mapTestPoolTmp, CDiskTxPos(1,1,1), pindexPrev, nFees, false, true,false)) continue;
            swap(mapTestPool, mapTestPoolTmp);

            // Added
            pblock->vtx.push_back(tx);
            nBlockSize += nTXSize;
            nBlockSigOps += nTxSigOps;

            // Add transactions that depend on this one to the priority queue
            uint256 hash = tx.GetHash();
            if (mapDependers.count(hash))
            {
                BOOST_FOREACH(COrphan* porphan, mapDependers[hash])
                {
                    if (!porphan->setDependsOn.empty())
                    {
                        porphan->setDependsOn.erase(hash);
                        if (porphan->setDependsOn.empty())
                            mapPriority.insert(make_pair(-porphan->dPriority, porphan->ptx));
                    }
                }
            }
        }
    }

    pblock->vtx[0].vout[0].nValue = qBlockValue+nFees;
    pblock->blk.hashPrevBlock  = pindexPrev->GetBlockHash();
    pblock->blk.nTime          = SolidTime_Get();
    if(pblock->blk.nTime<=pindexPrev->blk.nTime) pblock->blk.nTime=pindexPrev->blk.nTime+1;

    memset(pblock->blk.miner_id,0,12);
    int nLen=mining_id.length();
    if(nLen>12) nLen=12;
    memcpy(pblock->blk.miner_id,mining_id.c_str(),nLen);

    pblock->blk.hashMerkleRoot = pblock->BuildMerkleTree();


    debugprintf(INFO, "[%d] Block_Create: %"PRI64d" height %s prevhash \n",(int)pblock->blk.nTime,pblock->blk.nBlockNum,pblock->blk.hashPrevBlock.ToString().c_str());
    return pblock.release();
}


bool CBlock::VerifyBlock() const
{
    // These are checks that are independent of context that can be verified before saving an orphan block. Size limits
    if (vtx.empty())                                                return error("VerifyBlock() : no transactions");
    if(::GetSerializeSize(*this, SER_NETWORK) > MAX_BLOCK_SIZE)     return error("VerifyBlock() : block size exceeds maximum");
    if (!Block_CheckProofOfWork(GetHash(), blk.dwBits))             return error("VerifyBlock() : proof of work failed");
    if (blk.nTime > SolidTime_Get() + 15)                           return error("VerifyBlock() : block timestamp too far in the future");

    if(blk.nBlockNum < (g_qBlockBestHeight-6))                     return error("VerifyBlock() : block too old");

    if(blk.nBlockNum==0)
    {
        if(vtx.size()!=TRUST_FUND_NUM+1) return error("VerifyBlock() : genesis block is malformed");
        for (int i = 0; i < vtx.size(); i++)
        {
            if (!vtx[i].IsCoinBase())    return error("VerifyBlock() : genesis block is malformed (2)");
        }
    }
    else if((blk.nBlockNum%2) == 1)
    {
        if (!vtx[0].IsCoinBase())    return error("VerifyBlock() : first tx is not coinbase (%"PRI64d")",blk.nBlockNum);
        //if (vtx[0].GetValueOut() > GetBlockValue(blk.nBlockNum, nFees)) return error("VerifyBlock() : Incorrect value in coinbase (%s)",FormatMoney(vtx[0].GetValueOut()));
        for (int i = 1; i < vtx.size(); i++)
        {
            if (vtx[i].IsCoinBase())    return error("VerifyBlock() : more than one coinbase");
        }
    }
    else    //trusted node check
    {
        if (vtx.size()<2)                                       return error("VerifyBlock() : trusted block less than 2 tx");
        if(!vtx[0].IsCoinBase())                                return error("VerifyBlock() : trusted block first tx is not coinbase");
        if(vtx[0].vout[0].scriptPubKey !=g_CPFScript)           return error("VerifyBlock() : Trusted block coinbase not going to CPF");

        if(blk.nBlockNum>=91500)
        {
            if(vtx[0].vout.size()!=1)                           return error("VerifyBlock() : trusted block more than one output on generate");

            uint256 trustedhash = GetHash();
            CKey key;
            std::vector<unsigned char> sigtest,sig,pubkey,pubkeytest;
            pubkeytest = vtx[1].vout[0].scriptPubKey;

            if(pubkeytest.size()<66 || pubkeytest[0]!= 65)      return error("VerifyBlock() : trusted pubkey checks failed");
            std::copy( pubkeytest.begin()+1, pubkeytest.begin()+66, std::back_inserter(pubkey));
            if(!key.SetPubKey(pubkey))                          return error("VerifyBlock() : couldn't set trust pubkey");

            sigtest = vtx[0].vin[0].scriptSig;
            if(sigtest.size()<0x40 || sigtest.size()>0x50 || sigtest[0]!= sigtest.size()-1)
                                                                return error("VerifyBlock() : trusted sig checks failed");
            std::copy( sigtest.begin()+1, sigtest.end(), std::back_inserter(sig));
            if(!key.Verify(trustedhash,sig))                    return error("VerifyBlock() : trusted block wasn't signed correctly");
        }
        //if(!vtx[0].vout.size()!=1)                          return error("shit myself");
        if(vtx[1].vin.size()!=1 || vtx[1].vout.size()!=1)       return error("VerifyBlock() : trusted block second tx more than 1 input or output");
        if(vtx[1].vout[0].nValue<(TRUST_FUND_AMOUNT*COIN))      return error("VerifyBlock() : trusted block doesnt have enough funds");
        for (int i = 1; i < vtx.size(); i++)
        {
            if (vtx[i].IsCoinBase())    return error("VerifyBlock() : more than 1 coinbase");
        }
    }

    // Check transactions
    BOOST_FOREACH(const CTransaction& tx, vtx)
    {
        if (!tx.CheckTransaction(blk.nBlockNum))        return error("VerifyBlock() : CheckTransaction failed");
    }
    if (GetSigOpCount() > MAX_BLOCK_SIGOPS)         return error("VerifyBlock() : too many nonstandard transactions");
    if (blk.hashMerkleRoot != BuildMerkleTree())    return error("VerifyBlock() : hashMerkleRoot mismatch");
    return true;
}

bool CBlock::AcceptBlock()
{
    // Check for duplicate
    uint256 hash = GetHash();
    if (g_BlockIndexMap.count(hash))  return error("AcceptBlock() : block already in g_BlockIndexMap");

    map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(blk.hashPrevBlock);
    if (mi == g_BlockIndexMap.end())  return error("AcceptBlock() : prev block not found");

    CBlockIndex* pindexPrev = (*mi).second;

    // Check timestamp against prev
    if (blk.nTime <= pindexPrev->blk.nTime)   return error("AcceptBlock() : block's timestamp is too early");

    if(pindexPrev->pnext && (blk.nBlockNum%2)==1 )
    {
        //if we have a pnext it means we are replacing the current top level block with a better one
        //so compare the time we have now to that one as well, make sure its at least as new as ours
        //if (blk.nTime < pindexPrev->pnext->blk.nTime)   return error("AcceptBlock() %"PRI64d" : block's timestamp is too early 2",blk.nBlockNum);
    }

    if(blk.nBlockNum != pindexPrev->blk.nBlockNum+1)
    {
        return error("AcceptBlock() : wrong block number");
    }

    // Check proof of work
    uint32 dwNewWork=Block_GetNextWorkRequired(pindexPrev);
    if (blk.dwBits != dwNewWork)
    {
        return error("AcceptBlock() : incorrect proof of work Have (%08X) - Needed (%08X)",blk.dwBits,dwNewWork);
    }

    // Check that all transactions are finalized
    BOOST_FOREACH(const CTransaction& tx, vtx)
    {
        if (!tx.IsFinal(blk.nBlockNum, blk.nTime))      return error("AcceptBlock() : contains a non-final transaction");
    }

    // Check that the block chain matches the known block chain up to a checkpoint
    if (!fTestNet)
    {
        if  ((blk.nBlockNum ==  10000 && hash != uint256("0x0000415c813462cf0a5c35851b6b45eedc29534e271326332f51c58fb22a7716"))  ||     //first 30,080 coins reserved for bounties
             (blk.nBlockNum ==  20000 && hash != uint256("0x00002af3376e3f4801389ec4630a5c7b9f422f18f0f65bce6e01e6cd7676e71d")) ||
             (blk.nBlockNum ==  30000 && hash != uint256("0x000048ca1c38978f28b972a6bd1a0c69e0e7f52675836d713a2852687b3f19b4")) ||
             (blk.nBlockNum ==  40000 && hash != uint256("0x00005cc03415f2bbf76b380a6409563f554a92e5ee300f7adeb282e1d1f765ee")) ||
             (blk.nBlockNum ==  52000 && hash != uint256("0x00001debaad58d3dc18ad050c34af74e131fa2cf74b6cf8a1790b6c2557c0c1a")) ||
             (blk.nBlockNum ==  88000 && hash != uint256("0x00002d696e33ffd795d3d9e4fddad271fd58ddaec7804d5f7f6900d9d2417f13"))
             )
             {
                 return error("AcceptBlock() : rejected by checkpoint lockin at %"PRI64d"", blk.nBlockNum);
             }
    }

    // Write block to history file
    if (!CheckDiskSpace(::GetSerializeSize(*this, SER_DISK)))   return error("AcceptBlock() : out of disk space");
    unsigned int nFile = -1,nBlockPos = 0;
    if (!WriteToDisk(nFile, nBlockPos))         return error("AcceptBlock() : WriteToDisk failed");
    if (!AddToBlockIndex(nFile, nBlockPos))     return error("AcceptBlock() : AddToBlockIndex failed");

    // Relay inventory, but don't relay old inventory during initial block download
    if (g_BlockBestChainHash == hash)
    {
        MUTEX_LOCK(cs_vNodes);
        BOOST_FOREACH(CNode* pnode, vNodes)
        {
            if (g_qBlockBestHeight > (pnode->qStartingHeight != -1 ? pnode->qStartingHeight - 2000 : 88000))   pnode->PushInventory(CInv(MSG_BLOCK, hash));
        }
    }

    return true;
}



bool CBlock::DisconnectBlock(CTxDB& txdb, CBlockIndex* pindex)
{
    // Disconnect in reverse order
    for (int i = vtx.size()-1; i >= 0; i--)
        if (!vtx[i].DisconnectInputs(txdb))
            return false;

    // Update block index on disk without changing it in memory.
    // The memory index structure will be changed after the db commits.
    if (pindex->pprev)
    {
        CDiskBlockIndex blockindexPrev(pindex->pprev);
        blockindexPrev.m_hashNext = 0;
        if (!txdb.WriteBlockIndex(blockindexPrev))  return error("DisconnectBlock() : WriteBlockIndex failed");
    }

    return true;
}

bool CBlock::ConnectBlock(CTxDB& txdb, CBlockIndex* pindex, bool bAllowOldBlockChecking)
{
    // Check it again in case a previous version let a bad block in
    if (!VerifyBlock())  return false;


    if (bAllowOldBlockChecking && blk.nTime < (SolidTime_Get() - 25) )
    {
        if(Setting_GetBOOL("verifyoldblocks"))
        {
            return error("ConnectBlock() : block timestamp too far in the past");
        }
    }
    //ConnectBlock
    // issue here: it doesn't know the version
    unsigned int nTxPos = pindex->nBlockPos + ::GetSerializeSize(CBlock(), SER_DISK) - 1 + GetSizeOfCompactSize(vtx.size());

    map<uint256, CTxIndex> mapQueuedChanges;
    int64 nFees = 0;
    for (int i = 0; i < vtx.size(); i++)
    {
        bool bTrustedTX=false;
        CDiskTxPos posThisTx(pindex->nFile, pindex->nBlockPos, nTxPos);
        nTxPos += ::GetSerializeSize(vtx[i], SER_DISK);
        if(i==1 && blk.nBlockNum && (blk.nBlockNum%2)==0) bTrustedTX=true;
        if (!vtx[i].ConnectInputs(txdb, mapQueuedChanges, posThisTx, pindex, nFees, true, false,bTrustedTX))   return false;
    }

    int64 qBlockValue= Block_GetCoinBaseValue(blk.dwBits,blk.nBlockNum);
    if((pindex->blk.nBlockNum%2)==0)
    {
        if(pindex->blk.nBlockNum>=42000)        //CHECKPOINT for proper work bits for cpf payments
        {
            qBlockValue= Block_GetCoinBaseValue(pindex->pprev->blk.dwBits,blk.nBlockNum);
        }
    }
    else    //user block
    {
        //is it a power block? this only comes in block 50000
        if(pindex->blk.nBlockNum>=50000 && pindex->pprev &&  pindex->pprev->pprev && Block_IsPowerBlock(pindex->pprev->GetBlockHash(),pindex->pprev->pprev->GetBlockHash()))
        {
            qBlockValue*=2;
        }

    }

    if (vtx[0].GetValueOut() != qBlockValue+nFees)  return error("ConnectBlock() : Coinbase value incorrect");;
    //if (vtx[0].vout[0].nValue != qBlockValue+nFees)  return error("ConnectBlock() : Coinbase value incorrect");;


    // Write queued txindex changes
    for (map<uint256, CTxIndex>::iterator mi = mapQueuedChanges.begin(); mi != mapQueuedChanges.end(); ++mi)
    {
        if (!txdb.UpdateTxIndex((*mi).first, (*mi).second))
            return error("ConnectBlock() : UpdateTxIndex failed");
    }


    // Update block index on disk without changing it in memory. The memory index structure will be changed after the db commits.
    if (pindex->pprev)
    {
        CDiskBlockIndex blockindexPrev(pindex->pprev);
        blockindexPrev.m_hashNext = pindex->GetBlockHash();
        if (!txdb.WriteBlockIndex(blockindexPrev))  return error("ConnectBlock() : WriteBlockIndex failed");
    }

    // Watch for transactions paying to me
    BOOST_FOREACH(CTransaction& tx, vtx)
    {
        LOCK_WALLET_ACCESS();
        Wallet_Sync(tx, this, true);
    }

    return true;
}

bool Block_Reorganize(CTxDB& txdb, CBlockIndex* pindexNew)
{
    debugprintf(WARN, "REORGANIZE\n");

    // Find the fork
    CBlockIndex* pfork = g_pBlockBestIndex;
    CBlockIndex* plonger = pindexNew;
    int nForkSizeCount=0;
    while (pfork != plonger)
    {
        while (plonger->blk.nBlockNum > pfork->blk.nBlockNum)
        {
            if (!(plonger = plonger->pprev))    return error("Reorganize() : plonger->pprev is null");
            nForkSizeCount++;
        }
        if (pfork == plonger)   break;
        if (!(pfork = pfork->pprev))    return error("Reorganize() : pfork->pprev is null");
    }

    if(g_pBlockBestIndex->blk.nBlockNum-pfork->blk.nBlockNum > 5)   return error("Reorganize() : fork size is too great");
    //if(nForkSizeCount>5)    return error("Reorganize() : fork size is too great");


    // List of what to disconnect
    vector<CBlockIndex*> vDisconnect;
    for (CBlockIndex* pindex = g_pBlockBestIndex; pindex != pfork; pindex = pindex->pprev)
        vDisconnect.push_back(pindex);

    // List of what to connect
    vector<CBlockIndex*> vConnect;
    for (CBlockIndex* pindex = pindexNew; pindex != pfork; pindex = pindex->pprev)
        vConnect.push_back(pindex);
    reverse(vConnect.begin(), vConnect.end());

    // Disconnect shorter branch
    vector<CTransaction> vResurrect;
    vector<uint256> vWalletRemove;
    BOOST_FOREACH(CBlockIndex* pindex, vDisconnect)
    {
        CBlock block;
        if (!block.ReadFromDisk(pindex))            return error("Reorganize() : ReadFromDisk for disconnect failed");
        if (!block.DisconnectBlock(txdb, pindex))   return error("Reorganize() : DisconnectBlock failed");
        bool bTrustedBlock=(block.blk.nBlockNum%2)==0? true:false;
        for(int i=0;i<block.vtx.size();i++) // Queue memory transactions to resurrect
        {
            CTransaction *pTX = &block.vtx[i];
            vWalletRemove.push_back(pTX->GetHash());
            if(!pTX->IsCoinBase())
            {
                if(!bTrustedBlock || (bTrustedBlock && i>1))
                {
                    vResurrect.push_back(*pTX);
                }
            }
        }
    }

    //clean all shitty transactions from our wallet that were in the old reorg chain
    {
        LOCK_WALLET_ACCESS();
        int nWallet=0;
        CWallet *pWallet=0;
        while((pWallet=Wallet_Get(nWallet++)) )
        {
            BOOST_FOREACH(uint256& hash, vWalletRemove)
            {
                if(pWallet->IsMine(hash))
                {
                    pWallet->EraseFromWallet(hash);
                }
            }
        }
    }

    // Connect longer branch
    vector<CTransaction> vDelete;
    for (int i = 0; i < vConnect.size(); i++)
    {
        CBlockIndex* pindex = vConnect[i];
        CBlock block;
        if (!block.ReadFromDisk(pindex))        return error("Reorganize() : ReadFromDisk for connect failed");
        if (!block.ConnectBlock(txdb, pindex,false))
        {
            txdb.TxnAbort();    // Invalid block
            return error("Reorganize() : ConnectBlock failed");
        }
        BOOST_FOREACH(const CTransaction& tx, block.vtx)    // Queue memory transactions to delete
        {
            vDelete.push_back(tx);
        }
    }
    if (!txdb.WriteHashBestChain(pindexNew->GetBlockHash()))
        return error("Reorganize() : WriteHashBestChain failed");

    // Make sure it's successfully written to disk before changing memory structure
    if (!txdb.TxnCommit())
        return error("Reorganize() : TxnCommit failed");

    // Disconnect shorter branch
    BOOST_FOREACH(CBlockIndex* pindex, vDisconnect)
        if (pindex->pprev)
            pindex->pprev->pnext = NULL;

    // Connect longer branch
    BOOST_FOREACH(CBlockIndex* pindex, vConnect)
        if (pindex->pprev)
            pindex->pprev->pnext = pindex;



    // Resurrect memory transactions that were in the disconnected branch
    BOOST_FOREACH(CTransaction& tx, vResurrect)
    {
        tx.AcceptToMemoryPool(0, txdb, false);
    }

    // Delete redundant memory transactions that are in the connected branch
    BOOST_FOREACH(CTransaction& tx, vDelete)
        tx.RemoveFromMemoryPool();


    return true;
}


//sees if we should add this to our chain or not
bool Block_Process(CNode* pfrom, CBlock* pblock)
{
    // Check for duplicate
    uint256 hash = pblock->GetHash();
    if (g_BlockIndexMap.count(hash))      return error("ProcessBlock() : already have block %"PRI64d" %s", g_BlockIndexMap[hash]->blk.nBlockNum, hash.ToString().substr(0,20).c_str());
    if (g_BlockOrphanMap.count(hash))    return error("ProcessBlock() : already have block (orphan) %s", hash.ToString().substr(0,20).c_str());
    if (!pblock->VerifyBlock())          return error("ProcessBlock() : VerifyBlock FAILED");

    // If don't already have its previous block, shunt it off to holding area until we get it
    if (!g_BlockIndexMap.count(pblock->blk.hashPrevBlock))
    {
        CBlock* pblock2 = new CBlock(*pblock);
        if(g_BlockOrphanMap.size()>100)   //store only upto 100 orphan blocks to stop flooding attacks, push a random one out when full
        {
            map<uint256, CBlock*>::iterator mi = g_BlockOrphanMap.begin();
            CBlock* pblockOrphan = (*mi).second;
            g_BlockOrphanMapByPrev.erase(pblockOrphan->blk.hashPrevBlock);
            g_BlockOrphanMap.erase(mi);
        }
        g_BlockOrphanMap.insert(make_pair(hash, pblock2));
        g_BlockOrphanMapByPrev.insert(make_pair(pblock2->blk.hashPrevBlock, pblock2));

        // Ask this node to give us some blocks
        if (pfrom)  pfrom->PushGetBlocks(g_pBlockBestIndex, Block_GetOrphanRoot(pblock2));
        return true;
    }
    else
    {
        // Store to disk
        if (!pblock->AcceptBlock()) return error("ProcessBlock() : AcceptBlock FAILED");
    }

    // Recursively process any orphan blocks that depended on this one
    vector<uint256> vWorkQueue;
    vWorkQueue.push_back(hash);
    for (int i = 0; i < vWorkQueue.size(); i++)
    {
        uint256 hashPrev = vWorkQueue[i];
        for (multimap<uint256, CBlock*>::iterator mi = g_BlockOrphanMapByPrev.lower_bound(hashPrev);
             mi != g_BlockOrphanMapByPrev.upper_bound(hashPrev);
             ++mi)
        {
            CBlock* pblockOrphan = (*mi).second;
            if (pblockOrphan->AcceptBlock())    vWorkQueue.push_back(pblockOrphan->GetHash());
            g_BlockOrphanMap.erase(pblockOrphan->GetHash());
            delete pblockOrphan;
        }
        g_BlockOrphanMapByPrev.erase(hashPrev);
    }

    debugprintf(INFO, "ProcessBlock: ACCEPTED\n");
    return true;
}

bool CBlock::ReadFromDisk(const CBlockIndex* pindex, bool fReadTransactions)
{
    if (!fReadTransactions)
    {
        *this = pindex->GetBlockHeader();
        return true;
    }
    if (!ReadFromDisk(pindex->nFile, pindex->nBlockPos, fReadTransactions))
        return false;
    if (GetHash() != pindex->GetBlockHash())
        return error("CBlock::ReadFromDisk() : GetHash() doesn't match index");
    return true;
}

uint256 Block_GetOrphanRoot(const CBlock* pblock)
{
    // Work back to the first block in the orphan chain
    while (g_BlockOrphanMap.count(pblock->blk.hashPrevBlock))
        pblock = g_BlockOrphanMap[pblock->blk.hashPrevBlock];
    return pblock->GetHash();
}

unsigned int Block_GetNextWorkRequired(const CBlockIndex* pindexLast)
{
    int64 nTargetTimespan = 12 * 60 * 60; // 12 hours, but it's really 6 due to trusted blocks not being taken into account
    int64 nTargetSpacing = 2 * 60;    //2 minute blocks
    int64 nInterval = nTargetTimespan / nTargetSpacing;
    int64 nActualTimespan=0;

    if(pindexLast == NULL)  return g_bnBlockProofOfWorkLimit.GetCompact();
    if(pindexLast->blk.nBlockNum==0) return pindexLast->blk.dwBits;

    int64 qBlockNum = pindexLast->blk.nBlockNum+1;
    if((qBlockNum%2) == 0) return g_bnBlockProofOfWorkLimit.GetCompact();

    // Only change once per interval
    if((pindexLast->blk.nBlockNum%nInterval) != 0)
    {
        if(pindexLast->pprev==0 || pindexLast->pprev->pprev==0) return g_bnBlockProofOfWorkLimit.GetCompact();
        return pindexLast->pprev->blk.dwBits;
    }

    uint32 dwBits=pindexLast->pprev->blk.dwBits;

    // Go back by what we want to be 6 hours worth of blocks
    const CBlockIndex* pindexFirst = pindexLast->pprev;     //start on an odd number, take away the even (trusted)
    for (int i = 0; pindexFirst && i < nInterval-1; i+=2)
    {
        if(pindexFirst->pprev==0) break;
        nActualTimespan +=  pindexFirst->blk.nTime-pindexFirst->pprev->blk.nTime;
        if(pindexFirst->pprev->pprev==0) break;
        pindexFirst=pindexFirst->pprev->pprev;
    }

    // Limit adjustment step
    nTargetTimespan/=2;
    int64 nOnePercent = (nTargetTimespan/100);        //divide by 2 since each OTHER block shouldnt be counted

    if (nActualTimespan < nTargetTimespan)  //is time taken for a block less than 2 minutes?
    {
        if(nActualTimespan<(nOnePercent*33))        nActualTimespan=(nOnePercent*88); //pretend it was 88% of our desired to limit increase
        else if(nActualTimespan<(nOnePercent*66))   nActualTimespan=(nOnePercent*92); //pretend it was 92% of our desired to limit increase
        else                                        nActualTimespan=(nOnePercent*97); //pretend it was 97% of our desired to limit increase
    }
    else if (nActualTimespan > nTargetTimespan*4)   nActualTimespan = nTargetTimespan*4;

    // Retarget
    CBigNum bnNew;
    bnNew.SetCompact(dwBits);
    bnNew *= nActualTimespan;
    bnNew /= nTargetTimespan;
    if (bnNew > g_bnBlockProofOfWorkLimit) bnNew = g_bnBlockProofOfWorkLimit;
    debugprintf(INFO, "GetNextWorkRequired RETARGET (%"PRI64d" blockheight)\n",pindexLast->blk.nBlockNum);
    debugprintf(INFO, "nTargetTimespan = %"PRI64d"    nActualTimespan = %"PRI64d"\n", nTargetTimespan, nActualTimespan);
    debugprintf(INFO, "Before: %08x  %s\n", dwBits, CBigNum().SetCompact(dwBits).getuint256().ToString().c_str());
    debugprintf(INFO, "After:  %08x  %s\n", bnNew.GetCompact(), bnNew.getuint256().ToString().c_str());
    return bnNew.GetCompact();
}

bool Block_CheckProofOfWork(const uint256 &hash, unsigned int nBits)
{
    CBigNum bnTarget;
    bnTarget.SetCompact(nBits);
    if (bnTarget <= 0 || bnTarget > g_bnBlockProofOfWorkLimit)  return error("Block_CheckProofOfWork() : nBits below minimum work");
    if (hash > bnTarget.getuint256())                           return error("Block_CheckProofOfWork() : hash (%s) doesn't match nBits(%08x)",hash.ToString().c_str(), nBits);
    return true;
}

//assumed wallet lock before entry
bool Block_CheckWork(CBlock* pblock, CWallet& wallet, CReserveKey& reservekey, uint256 *outhash)
{
    uint256 hash = pblock->GetHash();
    if(outhash) *outhash=hash;
    uint256 hashTarget = CBigNum().SetCompact(pblock->blk.dwBits).getuint256();

    if (hash > hashTarget)  return false;

    debugprintf(ALWAYS, "proof-of-work found (%"PRI64d")  \n  hash: %s  \ntarget: %s\n", g_qBlockBestHeight,hash.GetHex().c_str(), hashTarget.GetHex().c_str());
    pblock->print();
    debugprintf(ALWAYS, "%s ", DateTimeStrFormat("%x %H:%M", SolidTime_Get()).c_str());
    debugprintf(ALWAYS, "generated %s\n", FormatMoney(pblock->vtx[0].vout[0].nValue).c_str());

    // Found a solution
    if (pblock->blk.hashPrevBlock != g_BlockBestChainHash)
    {
        if(pblock->blk.nBlockNum!=g_qBlockBestHeight || (g_qBlockBestHeight%2)!=1 )
        {
            return error("SolidCoinMiner : generated block is stale");
        }
    }

    reservekey.KeepKey();   // Remove key from key pool
    wallet.mapRequestCount[hash] = 0;  // Track how many getdata requests this block gets

    if(pblock->blk.nBlockNum>=91500 && (pblock->blk.nBlockNum%2) == 0)     //only trusted nodes on trusted blocks will get here
    {
        //we want to sign important block data here so no one else can modify it, including the transactions, blocknumber, time and prior hash
        uint256 trustblockhash = hash;

        std::vector<unsigned char> sig;
        if(!pblock->m_trust_privkey.Sign(trustblockhash,sig))
        {
            debugprintf(ERR, "block_create: Couldnt sign trust hash");
            return 0;
        }

        //verify our signing
        if (!pblock->m_trust_privkey.Verify(trustblockhash, sig))
        {
            debugprintf(ERR, "block_create: Couldnt verify signature");
            return 0;
        }

        pblock->vtx[0].vin[0].scriptSig = CScript() << sig;     //store the trusted nodes signing of the block data in the generate
    }

    if (!Block_Process(NULL, pblock))    // Process this block the same as if we had received it from another node
        return error("SolidCoinMiner : ProcessBlock, block not accepted");
    //Sleep(2000);
    return true;
}



CBlockIndex* Block_InsertBlockIndex(const uint256 &hash)
{
    if (hash == 0)  return NULL;

    // Return existing
    map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(hash);
    if (mi != g_BlockIndexMap.end())    return (*mi).second;

    // Create new
    CBlockIndex* pindexNew = new CBlockIndex();
    if (!pindexNew)     throw runtime_error("LoadBlockIndex() : new CBlockIndex failed");
    mi = g_BlockIndexMap.insert(make_pair(hash, pindexNew)).first;
    pindexNew->phashBlock = &((*mi).first);

    return pindexNew;
}



CBlockIndex::CBlockIndex()
{
    phashBlock = NULL;
    pprev = NULL;
    pnext = NULL;
    nFile = 0;
    nBlockPos = 0;
    bnChainWork = 0;

    blk.nVersion       = 0;
    blk.hashMerkleRoot = 0;
    blk.nBlockNum = 0;
    blk.nTime          = 0;
    blk.dwBits          = 0;
    blk.nNonce1         = 0;
    blk.nNonce2         = 0;
    blk.nNonce3         = 0;
    blk.nNonce4         = 0;
    memset(blk.miner_id,0,12);
}

CBlockIndex::CBlockIndex(unsigned int nFileIn, unsigned int nBlockPosIn, CBlock& block)
{
    phashBlock = NULL;
    pprev = NULL;
    pnext = NULL;
    nFile = nFileIn;
    nBlockPos = nBlockPosIn;
    bnChainWork = 0;

    blk = block.blk;
    blk.hashPrevBlock=0;
}

CBlock CBlockIndex::GetBlockHeader() const
{
    CBlock block;
    block.blk=blk;
    block.blk.hashPrevBlock=0;
    if (pprev)  block.blk.hashPrevBlock = pprev->GetBlockHash();
    return block;
}




bool CBlockIndex::IsInMainChain() const
{
    return (pnext || this == g_pBlockBestIndex);
}

bool CBlockIndex::CheckIndex() const
{
    return Block_CheckProofOfWork(GetBlockHash(), blk.dwBits);
}

bool CBlockIndex::EraseBlockFromDisk()
{
    // Open history file
    CAutoFile fileout = OpenBlockFile(nFile, nBlockPos, "rb+");
    if (!fileout)   return false;

    // Overwrite with empty null block
    CBlock block;
    block.SetNull();
    fileout << block;
    return true;
}

std::string CBlockIndex::ToString() const
{
    return strprintf("CBlockIndex(nprev=%08x, pnext=%08x, nFile=%d, nBlockPos=%-6d nHeight=%"PRI64d", merkle=%s, hashBlock=%s)",
        pprev, pnext, nFile, nBlockPos, blk.nBlockNum,
        blk.hashMerkleRoot.ToString().substr(0,10).c_str(),
        GetBlockHash().ToString().substr(0,20).c_str());
}

void CBlockIndex::print() const
{
    debugprintf(INFO, "%s\n", ToString().c_str());
}


CBlockLocator::CBlockLocator()
{

}

CBlockLocator::CBlockLocator(const CBlockIndex* pindex)
{
    Set(pindex);
}

CBlockLocator::CBlockLocator(uint256 hashBlock)
{
    std::map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(hashBlock);
    if (mi != g_BlockIndexMap.end())
        Set((*mi).second);
}

void CBlockLocator::SetNull()
{
    vHave.clear();
}

bool CBlockLocator::IsNull()
{
    return vHave.empty();
}

void CBlockLocator::Set(const CBlockIndex* pindex)
{
    vHave.clear();
    int nStep = 1;
    while (pindex)
    {
        vHave.push_back(pindex->GetBlockHash());

        // Exponentially larger steps back
        for (int i = 0; pindex && i < nStep; i++)
            pindex = pindex->pprev;
        if (vHave.size() > 10)
            nStep *= 2;
    }
    vHave.push_back(g_BlockGenesisHash);
}

int CBlockLocator::GetDistanceBack()
{
    // Retrace how far back it was in the sender's branch
    int nDistance = 0;
    int nStep = 1;
    BOOST_FOREACH(const uint256& hash, vHave)
    {
        std::map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(hash);
        if (mi != g_BlockIndexMap.end())
        {
            CBlockIndex* pindex = (*mi).second;
            if (pindex->IsInMainChain())
                return nDistance;
        }
        nDistance += nStep;
        if (nDistance > 10)
            nStep *= 2;
    }
    return nDistance;
}

CBlockIndex* CBlockLocator::GetBlockIndex()
{
    // Find the first block the caller has in the main chain
    BOOST_FOREACH(const uint256& hash, vHave)
    {
        std::map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(hash);
        if (mi != g_BlockIndexMap.end())
        {
            CBlockIndex* pindex = (*mi).second;
            if (pindex->IsInMainChain())
                return pindex;
        }
    }
    return g_pBlockGenesisIndex;
}

uint256 CBlockLocator::GetBlockHash()
{
    // Find the first block the caller has in the main chain
    BOOST_FOREACH(const uint256& hash, vHave)
    {
        std::map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.find(hash);
        if (mi != g_BlockIndexMap.end())
        {
            CBlockIndex* pindex = (*mi).second;
            if (pindex->IsInMainChain())
                return hash;
        }
    }
    return g_BlockGenesisHash;
}

