//Please read license.txt for licensing and copyright information

#include "headers.h"
#include "strlcpy.h"
#include <boost/program_options/detail/config_file.hpp>
#include <boost/program_options/parsers.hpp>
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/interprocess/sync/interprocess_mutex.hpp>
#include <boost/interprocess/sync/interprocess_recursive_mutex.hpp>
#include <boost/foreach.hpp>
#include <boost/asio.hpp>

using namespace std;
using namespace boost;

std::map<std::string, std::vector<std::string> > g_CMDSettingsMap;
std::map<std::string, std::vector<std::string> > g_SettingsMap;

bool fDebug = false;
bool fPrintToConsole = false;
bool fPrintToDebugger = false;
char pszSetDataDir[MAX_PATH] = "";
bool fRequestShutdown = false;
bool fShutdown = false;
bool fDaemon = false;
bool fServer = false;
bool fCommandLine = false;
string strMiscWarning;
bool fTestNet = false;
bool fNoListen = false;
bool fLogTimestamps = false;
int fLogDebugLevel = 1;



// Workaround for "multiple definition of `_tls_used'"
// http://svn.boost.org/trac/boost/ticket/4258
extern "C" void tss_cleanup_implemented() { }





// Init openssl library multithreading support
static boost::interprocess::interprocess_mutex** ppmutexOpenSSL;
void locking_callback(int mode, int i, const char* file, int line)
{
    if (mode & CRYPTO_LOCK)
        ppmutexOpenSSL[i]->lock();
    else
        ppmutexOpenSSL[i]->unlock();
}

// Init
class CInit
{
public:
    CInit()
    {
        // Init openssl library multithreading support
        ppmutexOpenSSL = (boost::interprocess::interprocess_mutex**)OPENSSL_malloc(CRYPTO_num_locks() * sizeof(boost::interprocess::interprocess_mutex*));
        for (int i = 0; i < CRYPTO_num_locks(); i++)
            ppmutexOpenSSL[i] = new boost::interprocess::interprocess_mutex();
        CRYPTO_set_locking_callback(locking_callback);

#ifdef __WXMSW__
        // Seed random number generator with screen scrape and other hardware sources
        RAND_screen();
#endif

        // Seed random number generator with performance counter
        RandAddSeed();
    }
    ~CInit()
    {
        // Shutdown openssl library multithreading support
        CRYPTO_set_locking_callback(NULL);
        for (int i = 0; i < CRYPTO_num_locks(); i++)
            delete ppmutexOpenSSL[i];
        OPENSSL_free(ppmutexOpenSSL);
    }
}
instance_of_cinit;









void RandAddSeed()
{
    // Seed with CPU performance counter
    int64 nCounter = GetPerformanceCounter();
    RAND_add(&nCounter, sizeof(nCounter), 1.5);
    memset(&nCounter, 0, sizeof(nCounter));
}

void RandAddSeedPerfmon()
{
    RandAddSeed();

    // This can take up to 2 seconds, so only do it every 10 minutes
    static int64 nLastPerfmon;
    if (SolidTime_Get() < nLastPerfmon + 10 * 60)
        return;
    nLastPerfmon = SolidTime_Get();

#ifdef __WXMSW__
    // Don't need this on Linux, OpenSSL automatically uses /dev/urandom
    // Seed with the entire set of perfmon data
    unsigned char pdata[250000];
    memset(pdata, 0, sizeof(pdata));
    DWORD dwSize = sizeof(pdata);
    long ret = RegQueryValueExA(HKEY_PERFORMANCE_DATA, "Global", NULL, NULL, pdata, &dwSize);
    RegCloseKey(HKEY_PERFORMANCE_DATA);
    if (ret == ERROR_SUCCESS)
    {
        RAND_add(pdata, dwSize, dwSize/100.0);
        memset(pdata, 0, dwSize);
        debugprintf(INFO, "%s RandAddSeed() %d bytes\n", DateTimeStrFormat("%x %H:%M", SolidTime_Get()).c_str(), dwSize);
    }
#endif
}

uint64 GetRand(uint64 nMax)
{
    if (nMax == 0)
        return 0;

    // The range of the random source must be a multiple of the modulus
    // to give every possible output value an equal possibility
    uint64 nRange = (UINT64_MAX / nMax) * nMax;
    uint64 nRand = 0;
    do
        RAND_bytes((unsigned char*)&nRand, sizeof(nRand));
    while (nRand >= nRange);
    return (nRand % nMax);
}

int GetRandInt(int nMax)
{
    return GetRand(nMax);
}








inline int OutputDebugStringF(DebugLevel level, const char* pszFormat, ...)
{
    int ret = 0;
    if (fPrintToConsole)
    {
        // print to console
        va_list arg_ptr;
        va_start(arg_ptr, pszFormat);
        ret = vprintf(pszFormat, arg_ptr);
        va_end(arg_ptr);
    }
    else
    {
        if(level >= fLogDebugLevel)
		{
			// print to debug.log
			static FILE* fileout = NULL;

			if (!fileout)
			{
				char pszFile[MAX_PATH+100];
				GetDataDir(pszFile);
				strlcat(pszFile, "/debug.log", sizeof(pszFile));
				fileout = fopenSOLID(pszFile, "a");
				if (fileout) setbuf(fileout, NULL); // unbuffered
			}
			if (fileout)
			{
				static bool fStartedNewLine = true;

				// Debug print useful for profiling
				if (fLogTimestamps && fStartedNewLine)              fprintf(fileout, "%s ", DateTimeStrFormat("%x %H:%M:%S", SolidTime_Get()).c_str());
				if (pszFormat[strlen(pszFormat) - 1] == '\n')       fStartedNewLine = true;
				else                                                fStartedNewLine = false;

				va_list arg_ptr;
				va_start(arg_ptr, pszFormat);
				ret = vfprintf(fileout, pszFormat, arg_ptr);
				va_end(arg_ptr);
			}
		}
	}

#ifdef __WXMSW__
    if (fPrintToDebugger)
    {
        static CCriticalSection cs_OutputDebugStringF;

        // accumulate a line at a time
        {
            MUTEX_LOCK(cs_OutputDebugStringF);
            static char pszBuffer[50000];
            static char* pend;
            if (pend == NULL)
                pend = pszBuffer;
            va_list arg_ptr;
            va_start(arg_ptr, pszFormat);
            int limit = END(pszBuffer) - pend - 2;
            int ret = _vsnprintf(pend, limit, pszFormat, arg_ptr);
            va_end(arg_ptr);
            if (ret < 0 || ret >= limit)
            {
                pend = END(pszBuffer) - 2;
                *pend++ = '\n';
            }
            else
                pend += ret;
            *pend = '\0';
            char* p1 = pszBuffer;
            char* p2;
            while ((p2 = strchr(p1, '\n')) )
            {
                p2++;
                char c = *p2;
                *p2 = '\0';
                OutputDebugStringA(p1);
                *p2 = c;
                p1 = p2;
            }
            if (p1 != pszBuffer)
                memmove(pszBuffer, p1, pend - p1 + 1);
            pend -= (p1 - pszBuffer);
        }
    }
#endif
    return ret;
}


// Safer snprintf
//  - prints up to limit-1 characters
//  - output string is always null terminated even if limit reached
//  - return value is the number of characters actually printed
int my_snprintf(char* buffer, size_t limit, const char* format, ...)
{
    if (limit == 0)
        return 0;
    va_list arg_ptr;
    va_start(arg_ptr, format);
    int ret = _vsnprintf(buffer, limit, format, arg_ptr);
    va_end(arg_ptr);
    if (ret < 0 || ret >= limit)
    {
        ret = limit - 1;
        buffer[limit-1] = 0;
    }
    return ret;
}


string strprintf(const char* format, ...)
{
    char buffer[50000];
    char* p = buffer;
    int limit = sizeof(buffer);
    int ret;
    loop
    {
        va_list arg_ptr;
        va_start(arg_ptr, format);
        ret = _vsnprintf(p, limit, format, arg_ptr);
        va_end(arg_ptr);
        if (ret >= 0 && ret < limit)
            break;
        if (p != buffer)
            delete[] p;
        limit *= 2;
        p = new char[limit];
        if (p == NULL)
            throw std::bad_alloc();
    }
    string str(p, p+ret);
    if (p != buffer)
        delete[] p;
    return str;
}


bool error(const char* format, ...)
{
    char buffer[50000];
    int limit = sizeof(buffer);
    va_list arg_ptr;
    va_start(arg_ptr, format);
    int ret = _vsnprintf(buffer, limit, format, arg_ptr);
    va_end(arg_ptr);
    if (ret < 0 || ret >= limit)
    {
        ret = limit - 1;
        buffer[limit-1] = 0;
    }
    debugprintf(ERR, "ERROR: %s\n", buffer);
    return false;
}


void ParseString(const string& str, char c, vector<string>& v)
{
    if (str.empty())
        return;
    string::size_type i1 = 0;
    string::size_type i2;
    loop
    {
        i2 = str.find(c, i1);
        if (i2 == str.npos)
        {
            v.push_back(str.substr(i1));
            return;
        }
        v.push_back(str.substr(i1, i2-i1));
        i1 = i2+1;
    }
}


string FormatMoney(int64 n, bool fPlus)
{
    // Note: not using straight sprintf here because we do NOT want
    // localized number formatting.
    int64 n_abs = (n > 0 ? n : -n);
    int64 quotient = n_abs/COIN;
    int64 remainder = n_abs%COIN;
    string str = strprintf("%"PRI64d".%04"PRI64d, quotient, remainder);

    // Right-trim excess 0's before the decimal point:

    int nTrim = 0;
    //for (int i = str.size()-1; (str[i] == '0' && isdigit(str[i-2])); --i)
      //  ++nTrim;
    //if (nTrim)
      //  str.erase(str.size()-nTrim, nTrim);

    if (n < 0)
        str.insert((unsigned int)0, 1, '-');
    else if (fPlus && n > 0)
        str.insert((unsigned int)0, 1, '+');

    return str;
}


bool ParseMoney(const string& str, int64& nRet)
{
    return ParseMoney(str.c_str(), nRet);
}

bool ParseMoney(const char* pszIn, int64& nRet)
{
    string strWhole;
    int64 nUnits = 0;
    const char* p = pszIn;
    while (isspace(*p))
        p++;
    for (; *p; p++)
    {
        if (*p == '.')
        {
            p++;
            int64 nMult = CENT*10;
            while (isdigit(*p) && (nMult > 0))
            {
                nUnits += nMult * (*p++ - '0');
                nMult /= 10;
            }
            break;
        }
        if (isspace(*p))
            break;
        if (!isdigit(*p))
            return false;
        strWhole.insert(strWhole.end(), *p);
    }
    for (; *p; p++)
        if (!isspace(*p))
            return false;
    if (strWhole.size() > 14)
        return false;
    if (nUnits < 0 || nUnits > COIN)
        return false;
    int64 nWhole = atoi64(strWhole);
    int64 nValue = nWhole*COIN + nUnits;

    nRet = nValue;
    return true;
}


vector<unsigned char> ParseHex(const char* psz)
{
    static char phexdigit[256] =
    { -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      0,1,2,3,4,5,6,7,8,9,-1,-1,-1,-1,-1,-1,
      -1,0xa,0xb,0xc,0xd,0xe,0xf,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,0xa,0xb,0xc,0xd,0xe,0xf,-1,-1,-1,-1,-1,-1,-1,-1,-1
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, };

    // convert hex dump to vector
    vector<unsigned char> vch;
    loop
    {
        while (isspace(*psz))
            psz++;
        char c = phexdigit[(unsigned char)*psz++];
        if (c == (char)-1)
            break;
        unsigned char n = (c << 4);
        c = phexdigit[(unsigned char)*psz++];
        if (c == (char)-1)
            break;
        n |= c;
        vch.push_back(n);
    }
    return vch;
}

vector<unsigned char> ParseHex(const string& str)
{
    return ParseHex(str.c_str());
}


void ParseParameters(int argc, std::string * argv)
{
    g_CMDSettingsMap.clear();
    for (int i = 1; i < argc; i++)
    {
        char psz[10000];
        strlcpy(psz, argv[i].c_str(), sizeof(psz));
        char* pszValue = (char*)"";
        if (strchr(psz, '='))
        {
            pszValue = strchr(psz, '=');
            *pszValue++ = '\0';
        }
        #ifdef __WXMSW__
        _strlwr(psz);
        if (psz[0] == '/')  psz[0] = '-';
        #endif
        if (psz[0] != '-')  break;
        g_CMDSettingsMap[(psz+1) ].push_back(pszValue);
    }
}


const char* wxGetTranslation(const char* pszEnglish)
{
#ifdef GUI
    // Wrapper of wxGetTranslation returning the same const char* type as was passed in
    static CCriticalSection cs;
    MUTEX_LOCK(cs);
    // Look in cache
    static map<string, char*> mapCache;
    map<string, char*>::iterator mi = mapCache.find(pszEnglish);
    if (mi != mapCache.end())
        return (*mi).second;

    // wxWidgets translation
    wxString strTranslated = wxGetTranslation(wxString(pszEnglish, wxConvUTF8));

    // We don't cache unknown strings because caller might be passing in a
    // dynamic string and we would keep allocating memory for each variation.
    if (strcmp(pszEnglish, strTranslated.utf8_str()) == 0)
        return pszEnglish;

    // Add to cache, memory doesn't need to be freed.  We only cache because
    // we must pass back a pointer to permanently allocated memory.
    char* pszCached = new char[strlen(strTranslated.utf8_str())+1];
    strcpy(pszCached, strTranslated.utf8_str());
    mapCache[pszEnglish] = pszCached;
    return pszCached;
#else
    return pszEnglish;
#endif
}


bool WildcardMatch(const char* psz, const char* mask)
{
    loop
    {
        switch (*mask)
        {
        case '\0':
            return (*psz == '\0');
        case '*':
            return WildcardMatch(psz, mask+1) || (*psz && WildcardMatch(psz+1, mask));
        case '?':
            if (*psz == '\0')
                return false;
            break;
        default:
            if (*psz != *mask)
                return false;
            break;
        }
        psz++;
        mask++;
    }
}

bool WildcardMatch(const string& str, const string& mask)
{
    return WildcardMatch(str.c_str(), mask.c_str());
}








void FormatException(char* pszMessage, std::exception* pex, const char* pszThread)
{
#ifdef __WXMSW__
    char pszModule[MAX_PATH];
    pszModule[0] = '\0';
    GetModuleFileNameA(NULL, pszModule, sizeof(pszModule));
#else
    const char* pszModule = "solidcoin";
#endif
    if (pex)
        snprintf(pszMessage, 1000,
            "EXCEPTION: %s       \n%s       \n%s in %s       \n", typeid(*pex).name(), pex->what(), pszModule, pszThread);
    else
        snprintf(pszMessage, 1000,
            "UNKNOWN EXCEPTION       \n%s in %s       \n", pszModule, pszThread);
}

void LogException(std::exception* pex, const char* pszThread)
{
    char pszMessage[10000];
    FormatException(pszMessage, pex, pszThread);
    debugprintf(ERR, "\n%s", pszMessage);
}

void PrintException(std::exception* pex, const char* pszThread)
{
    char pszMessage[10000];
    FormatException(pszMessage, pex, pszThread);
    debugprintf(ERR, "\n\n************************\n%s\n", pszMessage);
    fprintf(stderr, "\n\n************************\n%s\n", pszMessage);
    strMiscWarning = pszMessage;
#ifdef GUI
    if (wxTheApp && !fDaemon)
        MyMessageBox(pszMessage, "SolidCoin", wxOK | wxICON_ERROR);
#endif
    throw;
}

void ThreadOneMessageBox(string strMessage)
{
    // Skip message boxes if one is already open
    static bool fMessageBoxOpen;
    if (fMessageBoxOpen)
        return;
    fMessageBoxOpen = true;
    ThreadSafeMessageBox(strMessage, "SolidCoin", wxOK | wxICON_EXCLAMATION);
    fMessageBoxOpen = false;
}

void PrintExceptionContinue(std::exception* pex, const char* pszThread)
{
    char pszMessage[10000];
    FormatException(pszMessage, pex, pszThread);
    debugprintf(ERR, "\n\n************************\n%s\n", pszMessage);
    fprintf(stderr, "\n\n************************\n%s\n", pszMessage);
    strMiscWarning = pszMessage;
#ifdef GUI
    if (wxTheApp && !fDaemon)
        boost::thread(boost::bind(ThreadOneMessageBox, string(pszMessage)));
#endif
}








#ifdef __WXMSW__
typedef WINSHELLAPI BOOL (WINAPI *PSHGETSPECIALFOLDERPATHW)(HWND hwndOwner, LPWSTR lpszPath, int nFolder, BOOL fCreate);

string MyGetSpecialFolderPath(int nFolder, bool fCreate)
{
    char pszPath[MAX_PATH+100] = {0};
    wchar_t wpszPath[MAX_PATH+100] = {0};

    // SHGetSpecialFolderPath isn't always available on old Windows versions
    HMODULE hShell32 = LoadLibraryA("shell32.dll");
    if (hShell32)
    {
        PSHGETSPECIALFOLDERPATHW pSHGetSpecialFolderPath = (PSHGETSPECIALFOLDERPATHW)GetProcAddress(hShell32, "SHGetSpecialFolderPathW");
        if (pSHGetSpecialFolderPath)    (*pSHGetSpecialFolderPath)(NULL, wpszPath, nFolder, fCreate);
        FreeModule(hShell32);

        ::WideCharToMultiByte(CP_UTF8,0,wpszPath,-1,pszPath,MAX_PATH,0,0);
    }

    // Backup option
    if (pszPath[0] == '\0')
    {
        if (nFolder == CSIDL_STARTUP)
        {
            strcpy(pszPath, getenv("USERPROFILE"));
            strcat(pszPath, "\\Start Menu\\Programs\\Startup");
        }
        else if (nFolder == CSIDL_APPDATA)
        {
            strcpy(pszPath, getenv("APPDATA"));
        }
    }

    return pszPath;
}
#endif

string GetDefaultDataDir()
{
    // Windows: C:\Documents and Settings\username\Application Data\SolidCoin
    // Mac: ~/Library/Application Support/SolidCoin
    // Unix: ~/.solidcoin
#ifdef __WXMSW__
    // Windows
    return MyGetSpecialFolderPath(CSIDL_APPDATA, true) + "\\SolidCoin2";
#else
    char* pszHome = getenv("HOME");
    if (pszHome == NULL || strlen(pszHome) == 0)
        pszHome = (char*)"/";
    string strHome = pszHome;
    if (strHome[strHome.size()-1] != '/')
        strHome += '/';
#ifdef __WXMAC_OSX__
    // Mac
    strHome += "Library/Application Support/";
    filesystem::create_directory(strHome.c_str());
    return strHome + "SolidCoin2";
#else
    // Unix
    return strHome + ".solidcoin2";
#endif
#endif
}

void GetDataDir(char* pszDir)
{
    // pszDir must be at least MAX_PATH length.
    int nVariation;
    if (pszSetDataDir[0] != 0)
    {
        strlcpy(pszDir, pszSetDataDir, MAX_PATH);
        nVariation = 0;
    }
    else
    {
        // This can be called during exceptions by printf, so we cache the
        // value so we don't have to do memory allocations after that.
        static char pszCachedDir[MAX_PATH];
        if (pszCachedDir[0] == 0)
            strlcpy(pszCachedDir, GetDefaultDataDir().c_str(), sizeof(pszCachedDir));
        strlcpy(pszDir, pszCachedDir, MAX_PATH);
        nVariation = 1;
    }
    if (fTestNet)
    {
        char* p = pszDir + strlen(pszDir);
        #ifdef __WXMSW__
        if (p > pszDir && p[-1] != '/' && p[-1] != '\\')    *p++ = '\\';
        #else
        if (p > pszDir && p[-1] != '/' && p[-1] != '\\')    *p++ = '/';
        #endif

        strcpy(p, "testnet");
        nVariation += 2;
    }
    static bool pfMkdir[4];
    if (!pfMkdir[nVariation])
    {
        pfMkdir[nVariation] = true;
        boost::filesystem::create_directory(pszDir);
    }
}

string GetDataDir()
{
    char pszDir[MAX_PATH];
    GetDataDir(pszDir);
    return pszDir;
}

string GetConfigFile()
{
    namespace fs = boost::filesystem;
    fs::path pathConfig(CMDSetting_Get("conf", "solidcoin.conf"));
    if (!pathConfig.is_complete())  pathConfig = fs::path(GetDataDir()) / pathConfig;
    return pathConfig.string();
}


CCriticalSection g_CS_Settings;

bool CMDSetting_Exist(const std::string& strArg)
{
    MUTEX_LOCK(g_CS_Settings);
    if (g_CMDSettingsMap.count(strArg))   return true;
    return false;
}

std::string CMDSetting_Get(const std::string& strArg, const std::string& strDefault)
{
    MUTEX_LOCK(g_CS_Settings);
    if (g_CMDSettingsMap.count(strArg))  return g_CMDSettingsMap[strArg][0];
    return strDefault;
}
bool Setting_Exist(const std::string& strArg)
{
    MUTEX_LOCK(g_CS_Settings);
    if(g_CMDSettingsMap.count(strArg))    return true;
    if(g_SettingsMap.count(strArg))     return true;
    return false;
}

std::string Setting_Get(const std::string& strArg)
{
    MUTEX_LOCK(g_CS_Settings);
    if (g_CMDSettingsMap.count(strArg))  return g_CMDSettingsMap[strArg][0];
    if (g_SettingsMap.count(strArg))  return g_SettingsMap[strArg][0];
    return "";
}

int64 Setting_GetINT64(const std::string& strArg)
{
    MUTEX_LOCK(g_CS_Settings);
    if (g_SettingsMap.count(strArg))  return atoi64(g_SettingsMap[strArg][0]);
    return 0;
}

bool Setting_GetBOOL(const std::string& strArg)
{
    MUTEX_LOCK(g_CS_Settings);

    if (g_CMDSettingsMap.count(strArg))
    {
        if (g_CMDSettingsMap[strArg][0].empty())    return true;  //if it has nothing in it we assume its true
        return (g_CMDSettingsMap[strArg][0] == "1");
    }
    if (g_SettingsMap.count(strArg))
    {
        if (g_SettingsMap[strArg][0].empty())    return true;  //if it has nothing in it we assume its true
        return (g_SettingsMap[strArg][0] == "1");
    }
    return false;   //if the setting doesnt exist, return false
}

std::vector<std::string> Setting_GetVector(const std::string &strArg)
{
    MUTEX_LOCK(g_CS_Settings);

    std::vector<std::string> vec;
    if (g_CMDSettingsMap.count(strArg))
    {
        BOOST_FOREACH( std::string setting, g_CMDSettingsMap[strArg])  vec.push_back(setting);
    }
    if (g_SettingsMap.count(strArg))
    {
        BOOST_FOREACH( std::string setting, g_SettingsMap[strArg])  vec.push_back(setting);
    }
    return vec;
}
void Setting_SetBOOL(const std::string& strArg, bool bVal)
{
    MUTEX_LOCK(g_CS_Settings);

    std::string boolset="0";
    if(bVal) boolset="1";
    if (g_SettingsMap.count(strArg))    g_SettingsMap[strArg][0]=boolset;
    else                                g_SettingsMap[strArg].push_back(boolset);
    SaveConfigFile();
}
void Setting_SetINT64(const std::string& strArg, int64 val)
{
    MUTEX_LOCK(g_CS_Settings);
    if (g_SettingsMap.count(strArg))    g_SettingsMap[strArg][0]=i64tostr(val);
    else                                g_SettingsMap[strArg].push_back(i64tostr(val));
    SaveConfigFile();
}
void Setting_Set(const std::string& strArg, const std::string& strVal)
{
    MUTEX_LOCK(g_CS_Settings);
    if (g_SettingsMap.count(strArg))    g_SettingsMap[strArg][0]=strVal;
    else                                g_SettingsMap[strArg].push_back(strVal);
    SaveConfigFile();
}


void InitConfigFile(void)
{
    #define ADD_SETTING(a,b) g_SettingsMap[a].push_back(b);
    #define ADD_POOL(num, name,website,host,port,user,pass) {  std::string pool=str(boost::format("pool_%d_") % num);ADD_SETTING(pool+"name",name);ADD_SETTING(pool+"website",website);ADD_SETTING(pool+"host",host);ADD_SETTING(pool+"port",port);ADD_SETTING(pool+"user",user);ADD_SETTING(pool+"pass",pass);ADD_SETTING(pool+"threads","1");ADD_SETTING(pool+"usage","5");ADD_SETTING(pool+"autostart","0"); }

    g_SettingsMap.clear();

    ADD_POOL(0,"Solo [SELF]","","127.0.0.1","8555", "user","pass");
    ADD_POOL(1,"Coinotron [POOL]","http://coinotron.com/coinotron/AccountServlet?action=register", "coinotron.com","9322","user","pass");
    ADD_POOL(2,"SquidNet [POOL]","http://www.squidnet.org","miners.squidnet.org","8000","user","pass");
    ADD_POOL(3,"Mine-For.US [POOL]","http://mine-for.us/","go.mine-for.us","5000","user","pass");
    ADD_POOL(4,"Other","","localhost","8555","user","pass");
    ADD_POOL(5,"nixcoin [POOL]","http://nixcoin.com/users/sign_up","nixcoin.com","8400","user","pass");

    ADD_SETTING("debuglevel","2");
    ADD_SETTING("logtimestamps","0");

    ADD_SETTING("noirc","0");
    ADD_SETTING("maxconnections","125");
    ADD_SETTING("rpcallowkeepalive","0");
    ADD_SETTING("rpcuser","user");
    ADD_SETTING("rpcpassword","pass");
    ADD_SETTING("rpctimeout","30");
    ADD_SETTING("rpcport","8555");
    ADD_SETTING("rpcconnect","127.0.0.1");
    ADD_SETTING("rpcssl","0");
    ADD_SETTING("rpcsslciphers","TLSv1+HIGH:!SSLv2:!aNULL:!eNULL:!AH:!3DES:@STRENGTH");
    ADD_SETTING("rpcsslcertificatechainfile","server.cert");
    ADD_SETTING("rpcsslprivatekeyfile","server.pem");
    ADD_SETTING("keypool","100");
    ADD_SETTING("upnp","1");
    ADD_SETTING("server","1");
    ADD_SETTING("testnet","0");

    ADD_SETTING("proxy_use","0");
    ADD_SETTING("proxy_address","127.0.0.1");
    ADD_SETTING("proxy_port","9050");

    ADD_SETTING("timeout","5000");
    ADD_SETTING("daemon","0");      //on linux this means no gui
    ADD_SETTING("dns","1");
    ADD_SETTING("nolisten","0");

    ADD_SETTING("mining_id","Solo Mining");
    ADD_SETTING("mining_sharetarget","1");
    ADD_SETTING("mining_workmaxperiod","60");
    ADD_SETTING("mining_servertimewindow","8");

    ADD_SETTING("maxreceivebuffer","10000");
    ADD_SETTING("maxsendbuffer","10000");

    ADD_SETTING("gui_showtx","50");
    ADD_SETTING("gui_startminimized","0");
    ADD_SETTING("gui_mintotray","1");
    ADD_SETTING("gui_minonclose","1");
    ADD_SETTING("gui_showminingpanel","0");

    ADD_SETTING("wallet_default","main");
    ADD_SETTING("wallet_flushmindelay","30");

    ADD_SETTING("time_disablentp","0");
    ADD_SETTING("time_ntpserver","pool.ntp.org");
    ADD_SETTING("time_ntpupdateminutes","30");
    ADD_SETTING("time_solidcoin","0");
}

void ReadConfigFile(void)
{
    InitConfigFile();

    std::map<std::string, std::vector<std::string> > Settings=g_SettingsMap;
    namespace fs = boost::filesystem;
    namespace pod = boost::program_options::detail;

    fs::ifstream streamConfig(GetConfigFile());
    if (!streamConfig.good())   return;

    set<string> setOptions;
    setOptions.insert("*");

    for (pod::config_file_iterator it(streamConfig, setOptions), end; it != end; ++it)
    {
        // Don't overwrite existing settings so command line settings override solidcoin.conf
        std::string strKey=it->string_key;
        if(g_SettingsMap.count(strKey))
        {
            Settings[strKey][0]=it->value[0];   //if its in our settings map, its a single setting, and we want to overwrite default
        }
        else
        {
            Settings[strKey].push_back(it->value[0]);   //else its a multi value, so add numerous
        }
    }
    g_SettingsMap=Settings;
}

void SaveConfigFile(void)
{
    MUTEX_LOCK(g_CS_Settings);

    namespace fs = boost::filesystem;
    namespace pod = boost::program_options::detail;

    bool bTestNet=fTestNet;
    fTestNet=false;
    fs::ofstream streamConfig(GetConfigFile());
    fTestNet=bTestNet;
    if (!streamConfig.good())   return;


    std::map<std::string, std::vector<std::string> >::iterator iter;

    for(iter=g_SettingsMap.begin();iter!=g_SettingsMap.end();++iter)
    {
        BOOST_FOREACH( std::string setting, iter->second)
        {
            streamConfig << iter->first;
            streamConfig << "=";
            streamConfig << setting;
            streamConfig << "\n";
        }
    }
}


std::vector<MINING_POOL> g_MiningPoolList;

void InitMiningPools(void)
{
    int nItem=0;
    while(1)
    {
        MINING_POOL pool;
        std::string pn=str(boost::format("pool_%d_") % nItem);
        if(!Setting_Exist(pn+"name")) break;

        pool.name=Setting_Get(pn+"name");
        pool.host=Setting_Get(pn+"host");
        pool.user=Setting_Get(pn+"user");
        pool.pass=Setting_Get(pn+"pass");
        pool.website=Setting_Get(pn+"website");
        pool.bAutoStart=Setting_GetBOOL(pn+"autostart");
        pool.nPort=(int)Setting_GetINT64(pn+"port");
        pool.nThreads=(int)Setting_GetINT64(pn+"threads");
        pool.nUsage=(int)Setting_GetINT64(pn+"usage");
        pool.nRunning=0;
        pool.pThreads=0;
        pool.qLastHashPerSec=0;
        pool.qTotalHash=0;
        pool.qTotalShares=0;
        g_MiningPoolList.push_back(pool);
        nItem++;
    }

}

void SaveMiningPools(void)
{
    int num=0;
    BOOST_FOREACH(MINING_POOL pool, g_MiningPoolList)
    {
        std::string pn=str(boost::format("pool_%d_") % num);
        Setting_Set(pn+"name",pool.name);
        Setting_Set(pn+"host",pool.host);
        Setting_Set(pn+"user",pool.user);
        Setting_Set(pn+"pass",pool.pass);
        Setting_SetINT64(pn+"port",pool.nPort);
        Setting_SetINT64(pn+"threads",pool.nThreads);
        Setting_SetINT64(pn+"usage",pool.nUsage);
        num++;
    }
}

#ifndef __WXMSW__
string GetPidFile()
{
    namespace fs = boost::filesystem;
    fs::path pathConfig(CMDSetting_Get("pid", "solidcoind.pid"));
    if (!pathConfig.is_complete())
        pathConfig = fs::path(GetDataDir()) / pathConfig;
    return pathConfig.string();
}

void CreatePidFile(string pidFile, pid_t pid)
{
    FILE* file = fopenSOLID(pidFile.c_str(), "w");
    if (file)
    {
        fprintf(file, "%d\n", pid);
        fclose(file);
    }
}
#endif

int GetFilesize(FILE* file)
{
    int nSavePos = ftell(file);
    int nFilesize = -1;
    if (fseek(file, 0, SEEK_END) == 0)
        nFilesize = ftell(file);
    fseek(file, nSavePos, SEEK_SET);
    return nFilesize;
}

void ShrinkDebugFile()
{
    // Scroll debug.log if it's getting too big
    string strFile = GetDataDir() + "/debug.log";
    FILE* file = fopenSOLID(strFile.c_str(), "r");
    if (file && GetFilesize(file) > 10 * 1000000)
    {
        // Restart the file with some of the end
        char pch[200000];
        fseek(file, -sizeof(pch), SEEK_END);
        int nBytes = fread(pch, 1, sizeof(pch), file);
        fclose(file);

        file = fopenSOLID(strFile.c_str(), "w");
        if (file)
        {
            fwrite(pch, 1, nBytes, file);
            fclose(file);
        }
    }
}








//
// "Never go to sea with two chronometers; take one or three."
// Our three time sources are:
//  - System clock
//  - Median of other nodes's clocks
//  - The user (asking the user to fix the system clock if the first two disagree)

CCriticalSection g_CS_Time;
int64 TIME_MINUTES_UPDATE_NTP=30;
bool g_bTimeDisableNTP=false;
bool g_bUseSolidCoinTime=false;
bool g_bUpdatingNTP=false;
bool g_bTimeGotData=false;
int64 g_nTimeLastUpdateMilli=0;
int64 g_nTimeLastNTPTime=0;
std::string g_TimeNTPServer;

bool SolidTime_Init(void)
{
    TIME_MINUTES_UPDATE_NTP=Setting_GetINT64("time_ntpupdateminutes");
    if(TIME_MINUTES_UPDATE_NTP<0) TIME_MINUTES_UPDATE_NTP=30;

    g_bUseSolidCoinTime=Setting_GetBOOL("time_solidcoin");
    g_bTimeDisableNTP=Setting_GetBOOL("time_disablentp");
    if(g_bTimeDisableNTP)   g_bUseSolidCoinTime=false;
    g_TimeNTPServer=Setting_Get("time_ntpserver");
    if(g_TimeNTPServer.length()<=0) g_TimeNTPServer= "pool.ntp.org";

    if(g_bTimeDisableNTP==false)
    {
        int64 nNTPTime=GetNTPTime();
        int64 nSysTime=(int64)time(NULL);

        g_nTimeLastUpdateMilli = GetTimeMillis();
        g_nTimeLastNTPTime = nNTPTime;
        if(nNTPTime==0)
        {
            if(wxMessageBox("Error connecting to NTP server to verify your system time.\n\nDo you want to use your system time instead?","NTP Time Error",wxICON_WARNING|wxYES_NO)==wxNO)
            {
                return false;
            }
            nNTPTime=g_nTimeLastNTPTime=nSysTime;
        }
        int64 nDiff=nNTPTime-nSysTime;
        if( (nDiff<-5 || nDiff>5) && g_bUseSolidCoinTime==false)
        {
            if(wxMessageBox("Your system clock is too far off global NTP (time) servers. SolidCoin 2.0 requires very accurate timing for every node on the network. Fix your system clock and ensure it verifies its time regularly with NTP servers.\n\nAlternatively you can let SolidCoin use its own time management which should work well in most cases.\n\nDo you want to let SolidCoin manage time internally?","SolidCoing Timing",wxYES_NO|wxICON_WARNING)==wxNO)
            {
                return false;
            }
            //they want to use solidcoin time now
            g_bUseSolidCoinTime=true;
            Setting_SetBOOL("time_solidcoin",true);
        }

    }
    return true;
}



void HandleReceivedTimeData( const boost::system::error_code& error, size_t bytesReceived )
{
    g_bTimeGotData=true;
}


int64 GetNTPTime(void)
{
    int nTries=0;

    int64 nRet=0;
    for(nTries=0;nTries<2;nTries++)
    {
        try
        {
            int nTimeOutRECV=0;
            using boost::asio::ip::udp;
            boost::asio::io_service io_service;

            udp::resolver resolver(io_service);
            udp::resolver::query query(udp::v4(), g_TimeNTPServer.c_str(), "ntp");
            udp::endpoint receiver_endpoint = *resolver.resolve(query);
            udp::endpoint sender_endpoint;

            boost::uint8_t data[128] = { 0x1B,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 };

            udp::socket socket(io_service);
            socket.open(udp::v4());
            socket.send_to(boost::asio::buffer(data,48),receiver_endpoint);

            //socket.receive_from(boost::asio::buffer(data),sender_endpoint);
            g_bTimeGotData=false;
            socket.async_receive_from(boost::asio::buffer(data,48), sender_endpoint,boost::bind( HandleReceivedTimeData,boost::asio::placeholders::error,boost::asio::placeholders::bytes_transferred));

            while(nTimeOutRECV<50 && g_bTimeGotData==false)
            {
                io_service.poll();
                io_service.reset();
                boost::this_thread::sleep( boost::posix_time::milliseconds(100));
                nTimeOutRECV++;
            }

            if(g_bTimeGotData==false) continue;


            typedef boost::uint32_t u32;
            const u32 iPart(static_cast<u32>(data[40]) << 24 | static_cast<u32>(data[41]) << 16 | static_cast<u32>(data[42]) << 8 | static_cast<u32>(data[43]) );
            const u32 fPart(static_cast<u32>(data[44]) << 24 | static_cast<u32>(data[45]) << 16 | static_cast<u32>(data[46]) << 8 | static_cast<u32>(data[47]) );

            using namespace boost::posix_time;
            const ptime pt(boost::gregorian::date(1900,1,1),milliseconds(iPart * 1.0E3+ fPart * 1.0E3 / 0x100000000ULL ));

            ::posix_time::ptime epoch(boost::gregorian::date(1970,1,1));    // Convert ptime to epoch time_t
            time_duration::sec_type x = (pt - epoch).total_seconds();
            nRet=int64(x);
            break;
        }
        catch (std::exception& e)
        {

        }
        Sleep(1000);
    }
    if(nTries==5)   debugprintf(ERR, "Error updating time from NTP server\n");
    return nRet;
}

void UpdateNTPTime(void* parg)
{
    int64 nCurNTP=GetNTPTime();
    if(nCurNTP==0)
    {
        g_bUpdatingNTP=false;
        return;
    }

    MUTEX_LOCK(g_CS_Time);
    g_nTimeLastNTPTime=nCurNTP;
    g_nTimeLastUpdateMilli = GetTimeMillis();
    g_bUpdatingNTP=false;   //allows us to get time again in future
}

//static int64 nTimeOffset = 0;
int64 SolidTime_Get()
{
    int64 nTimeRet=0;
    int64 nCurTime=GetTimeMillis();
    bool bUpdateNTP=false;

    {
        MUTEX_LOCK(g_CS_Time);
        nCurTime-=g_nTimeLastUpdateMilli;
        nCurTime/=1000;
        if(nCurTime > (60*TIME_MINUTES_UPDATE_NTP) && g_bUpdatingNTP==false)
        {
            g_bUpdatingNTP=true;
            bUpdateNTP=true;
        }

        nCurTime += g_nTimeLastNTPTime;
        if(g_bUseSolidCoinTime && g_bTimeDisableNTP==false)
        {
            nTimeRet=nCurTime;
        }
        else
        {
            nTimeRet=(int64)time(NULL);
        }
    }

    if(g_bTimeDisableNTP==false)
    {
        nCurTime = nCurTime-nTimeRet;
        if( (nCurTime<-5 || nCurTime>5) && g_bUseSolidCoinTime==false)
        {
            static int64 lasttime=0;
            if(GetTimeMillis()-lasttime>60000)      //restrict how often we warn about this
            {
                lasttime=GetTimeMillis();
                debugprintf(WARN, "NTP time differs from system clock by more than 5 seconds %"PRI64d"\n",nCurTime);
            }
        }
        if(bUpdateNTP) CreateThread(UpdateNTPTime, NULL);
    }


    return nTimeRet;
}

void AddTimeData(unsigned int ip, int64 nTime)
{
    int64 nOffsetSample = nTime - SolidTime_Get();



    // Ignore duplicates
    static set<unsigned int> setKnown;
    if (!setKnown.insert(ip).second)
        return;

    // Add data
    static vector<int64> vTimeOffsets;
    if (vTimeOffsets.empty())
        vTimeOffsets.push_back(0);
    vTimeOffsets.push_back(nOffsetSample);
    debugprintf(INFO, "Added time data, samples %d, offset %+"PRI64d" (%+"PRI64d" minutes)\n", vTimeOffsets.size(), vTimeOffsets.back(), vTimeOffsets.back()/60);
    if (vTimeOffsets.size() >= 5 && vTimeOffsets.size() % 2 == 1)
    {
        sort(vTimeOffsets.begin(), vTimeOffsets.end());
        int64 nMedian = vTimeOffsets[vTimeOffsets.size()/2];
        // Only let other nodes change our time by so much
        if (abs64(nMedian) < 70 * 60)
        {
            //nTimeOffset = nMedian;
        }
        else
        {
            //nTimeOffset = 0;

            static bool fDone;
            if (!fDone)
            {
                // If nobody has a time different than ours but within 5 minutes of ours, give a warning
                bool fMatch = false;
                BOOST_FOREACH(int64 nOffset, vTimeOffsets)
                    if (nOffset != 0 && abs64(nOffset) < 5 * 60)
                        fMatch = true;

                if (!fMatch)
                {
                    fDone = true;
                    string strMessage = _("Warning: Please check that your computer's date and time are correct.  If your clock is wrong SolidCoin will not work properly.");
                    strMiscWarning = strMessage;
                    debugprintf(WARN, "*** %s\n", strMessage.c_str());
                    boost::thread(boost::bind(ThreadSafeMessageBox, strMessage+" ", string("SolidCoin"), wxOK | wxICON_EXCLAMATION, (wxWindow*)NULL, -1, -1));
                }
            }
        }
        //BOOST_FOREACH(int64 n, vTimeOffsets) debugprintf(INFO, "%+"PRI64d"  ", n);
        //printf("|  nTimeOffset = %+"PRI64d"  (%+"PRI64d" minutes)\n", nTimeOffset, nTimeOffset/60);
    }
}







string EncodeBase64(const unsigned char* pch, size_t len)
{
    static const char *pbase64 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

    string strRet="";
    strRet.reserve((len+2)/3*4);

    int mode=0, left=0;
    const unsigned char *pchEnd = pch+len;

    while (pch<pchEnd)
    {
        int enc = *(pch++);
        switch (mode)
        {
            case 0: // we have no bits
                strRet += pbase64[enc >> 2];
                left = (enc & 3) << 4;
                mode = 1;
                break;

            case 1: // we have two bits
                strRet += pbase64[left | (enc >> 4)];
                left = (enc & 15) << 2;
                mode = 2;
                break;

            case 2: // we have four bits
                strRet += pbase64[left | (enc >> 6)];
                strRet += pbase64[enc & 63];
                mode = 0;
                break;
        }
    }

    if (mode)
    {
        strRet += pbase64[left];
        strRet += '=';
        if (mode == 1)
            strRet += '=';
    }

    return strRet;
}

string EncodeBase64(const string& str)
{
    return EncodeBase64((const unsigned char*)str.c_str(), str.size());
}

vector<unsigned char> DecodeBase64(const char* p, bool* pfInvalid)
{
    static const int decode64_table[256] =
    {
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, 62, -1, -1, -1, 63, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1,
        -1, -1, -1, -1, -1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
        15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1, -1, 26, 27, 28,
        29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48,
        49, 50, 51, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
    };

    if (pfInvalid)
        *pfInvalid = false;

    vector<unsigned char> vchRet;
    vchRet.reserve(strlen(p)*3/4);

    int mode = 0;
    int left = 0;

    while (1)
    {
         int dec = decode64_table[*p];
         if (dec == -1) break;
         p++;
         switch (mode)
         {
             case 0: // we have no bits and get 6
                 left = dec;
                 mode = 1;
                 break;

              case 1: // we have 6 bits and keep 4
                  vchRet.push_back((left<<2) | (dec>>4));
                  left = dec & 15;
                  mode = 2;
                  break;

             case 2: // we have 4 bits and get 6, we keep 2
                 vchRet.push_back((left<<4) | (dec>>2));
                 left = dec & 3;
                 mode = 3;
                 break;

             case 3: // we have 2 bits and get 6
                 vchRet.push_back((left<<6) | dec);
                 mode = 0;
                 break;
         }
    }

    if (pfInvalid)
        switch (mode)
        {
            case 0: // 4n base64 characters processed: ok
                break;

            case 1: // 4n+1 base64 character processed: impossible
                *pfInvalid = true;
                break;

            case 2: // 4n+2 base64 characters processed: require '=='
                if (left || p[0] != '=' || p[1] != '=' || decode64_table[p[2]] != -1)
                    *pfInvalid = true;
                break;

            case 3: // 4n+3 base64 characters processed: require '='
                if (left || p[0] != '=' || decode64_table[p[1]] != -1)
                    *pfInvalid = true;
                break;
        }

    return vchRet;
}

string DecodeBase64(const string& str)
{
    vector<unsigned char> vchRet = DecodeBase64(str.c_str());
    return string((const char*)&vchRet[0], vchRet.size());
}





string FormatVersion(int nVersion)
{
    int nMajor =nVersion/100000;
    int nMinor =(nVersion-(nMajor*100000)) / 1000;
    return strprintf("%d.%02d", nMajor, nMinor);
}

string FormatFullVersion()
{
    string s = FormatVersion(VERSION);
    if (VERSION_IS_BETA) {
        s += "-";
        s += _("beta");
    }
    return s;
}




#ifdef DEBUG_LOCKORDER
//
// Early deadlock detection.
// Problem being solved:
//    Thread 1 locks  A, then B, then C
//    Thread 2 locks  D, then C, then A
//     --> may result in deadlock between the two threads, depending on when they run.
// Solution implemented here:
// Keep track of pairs of locks: (A before B), (A before C), etc.
// Complain if any thread trys to lock in a different order.
//

struct CLockLocation
{
    std::string mutexName;
    std::string sourceFile;
    int sourceLine;

    CLockLocation(const char* pszName, const char* pszFile, int nLine)
    {
        mutexName = pszName;
        sourceFile = pszFile;
        sourceLine = nLine;
    }
};

typedef std::vector< std::pair<CCriticalSection*, CLockLocation> > LockStack;

static boost::interprocess::interprocess_mutex dd_mutex;
static std::map<std::pair<CCriticalSection*, CCriticalSection*>, LockStack> lockorders;
static boost::thread_specific_ptr<LockStack> lockstack;


static void potential_deadlock_detected(const LockStack& s1, const LockStack& s2)
{
    debugprintf(WARN, "POTENTIAL DEADLOCK DETECTED\n");
    debugprintf(WARN, "Previous lock order was:\n");
    BOOST_FOREACH(const PAIRTYPE(CCriticalSection*, CLockLocation)& i, s2)
    {
        debugprintf(WARN, " %s  %s:%d\n", i.second.mutexName.c_str(), i.second.sourceFile.c_str(), i.second.sourceLine);
    }
    debugprintf(INFO, "Current lock order is:\n");
    BOOST_FOREACH(const PAIRTYPE(CCriticalSection*, CLockLocation)& i, s1)
    {
        debugprintf(WARN, " %s  %s:%d\n", i.second.mutexName.c_str(), i.second.sourceFile.c_str(), i.second.sourceLine);
    }
}

static void push_lock(CCriticalSection* c, const CLockLocation& locklocation)
{
    bool fOrderOK = true;
    if (lockstack.get() == NULL)
        lockstack.reset(new LockStack);

    dd_mutex.lock();

    (*lockstack).push_back(std::make_pair(c, locklocation));

    BOOST_FOREACH(const PAIRTYPE(CCriticalSection*, CLockLocation)& i, (*lockstack))
    {
        if (i.first == c) break;

        std::pair<CCriticalSection*, CCriticalSection*> p1 = std::make_pair(i.first, c);
        if (lockorders.count(p1))
            continue;
        lockorders[p1] = (*lockstack);

        std::pair<CCriticalSection*, CCriticalSection*> p2 = std::make_pair(c, i.first);
        if (lockorders.count(p2))
        {
            potential_deadlock_detected(lockorders[p2], lockorders[p1]);
            break;
        }
    }
    dd_mutex.unlock();
}

static void pop_lock()
{
    (*lockstack).pop_back();
}

void CCriticalSection::Enter(const char* pszName, const char* pszFile, int nLine)
{
    push_lock(this, CLockLocation(pszName, pszFile, nLine));
    mutex.lock();
}
void CCriticalSection::Leave()
{
    mutex.unlock();
    pop_lock();
}
bool CCriticalSection::TryEnter(const char* pszName, const char* pszFile, int nLine)
{
    push_lock(this, CLockLocation(pszName, pszFile, nLine));
    bool result = mutex.try_lock();
    if (!result) pop_lock();
    return result;
}

#else

void CCriticalSection::Enter(const char*, const char*, int)
{
    mutex.lock();
}

void CCriticalSection::Leave()
{
    mutex.unlock();
}

bool CCriticalSection::TryEnter(const char*, const char*, int)
{
    bool result = mutex.try_lock();
    return result;
}

#endif /* DEBUG_LOCKORDER */




#define PHI 0x9e3779b9
#define BLOCKHASH_1_PADSIZE (1024*1024*4)
static uint32 BlockHash_1_Q[4096],BlockHash_1_c,BlockHash_1_i;
static unsigned char *BlockHash_1_MemoryPAD8;
static uint32 *BlockHash_1_MemoryPAD32;

uint32 BlockHash_1_rand(void)
{
    uint32 x, r = 0xfffffffe;
    uint64 t, a = 18782LL;
    BlockHash_1_i = (BlockHash_1_i + 1) & 4095;
    t = a * BlockHash_1_Q[BlockHash_1_i] + BlockHash_1_c;
    BlockHash_1_c = (t >> 32);
    x = t + BlockHash_1_c;
    if (x < BlockHash_1_c)
    {
        x++;
        BlockHash_1_c++;
    }
    return (BlockHash_1_Q[BlockHash_1_i] = r - x);
}

void BlockHash_Init()
{
    static unsigned char SomeArrogantText1[]="Back when I was born the world was different. As a kid I could run around the streets, build things in the forest, go to the beach and generally live a care free life. Sure I had video games and played them a fair amount but they didn't get in the way of living an adventurous life. The games back then were different too. They didn't require 40 hours of your life to finish. Oh the good old days, will you ever come back?";
    static unsigned char SomeArrogantText2[]="Why do most humans not understand their shortcomings? The funny thing with the human brain is it makes everyone arrogant at their core. Sure some may fight it more than others but in every brain there is something telling them, HEY YOU ARE THE MOST IMPORTANT PERSON IN THE WORLD. THE CENTER OF THE UNIVERSE. But we can't all be that, can we? Well perhaps we can, introducing GODria, take 2 pills of this daily and you can be like RealSolid, lord of the universe.";
    static unsigned char SomeArrogantText3[]="What's up with kids like artforz that think it's good to attack other's work? He spent a year in the bitcoin scene riding on the fact he took some other guys SHA256 opencl code and made a miner out of it. Bravo artforz, meanwhile all the false praise goes to his head and he thinks he actually is a programmer. Real programmers innovate and create new work, they win through being better coders with better ideas. You're not real artforz, and I hear you like furries? What's up with that? You shouldn't go on IRC when you're drunk, people remember the weird stuff.";
    BlockHash_1_MemoryPAD8 = new unsigned char[BLOCKHASH_1_PADSIZE+8];  //need the +8 for memory overwrites
    BlockHash_1_MemoryPAD32 = (uint32*)BlockHash_1_MemoryPAD8;

    BlockHash_1_Q[0] = 0x6970F271;
    BlockHash_1_Q[1] = 0x6970F271 + PHI;
    BlockHash_1_Q[2] = 0x6970F271 + PHI + PHI;
    for (int i = 3; i < 4096; i++)  BlockHash_1_Q[i] = BlockHash_1_Q[i - 3] ^ BlockHash_1_Q[i - 2] ^ PHI ^ i;
    BlockHash_1_c=362436;
    BlockHash_1_i=4095;

    int count1=0,count2=0,count3=0;
    for(int x=0;x<(BLOCKHASH_1_PADSIZE/4)+2;x++)  BlockHash_1_MemoryPAD32[x] = BlockHash_1_rand();
    for(int x=0;x<BLOCKHASH_1_PADSIZE+8;x++)
    {
        switch(BlockHash_1_MemoryPAD8[x]&3)
        {
            case 0: BlockHash_1_MemoryPAD8[x] ^= SomeArrogantText1[count1++]; if(count1>=sizeof(SomeArrogantText1)) count1=0; break;
            case 1: BlockHash_1_MemoryPAD8[x] ^= SomeArrogantText2[count2++]; if(count2>=sizeof(SomeArrogantText2)) count2=0; break;
            case 2: BlockHash_1_MemoryPAD8[x] ^= SomeArrogantText3[count3++]; if(count3>=sizeof(SomeArrogantText3)) count3=0; break;
            case 3: BlockHash_1_MemoryPAD8[x] ^= 0xAA; break;
        }
    }
}

void BlockHash_DeInit()
{
    delete[] BlockHash_1_MemoryPAD8;
}

void BlockHash_1(uint256 &final_hash, unsigned char *p512bytes)
{
    //0->127   is the block header      (128)
    //128->191 is blake(blockheader)    (64)
    //192->511 is scratch work area     (320)
    ;
    unsigned char *work1 = p512bytes;
    unsigned char *work2=work1+128;
    unsigned char *work3=work1+192;
    #define READ_PAD8(offset) BlockHash_1_MemoryPAD8[(offset)&(BLOCKHASH_1_PADSIZE-1)]
    #define READ_PAD32(offset) (*((uint32*)&BlockHash_1_MemoryPAD8[(offset)&(BLOCKHASH_1_PADSIZE-1)]))

    blake512_hash(work2,work1,128);

    //setup the 320 scratch with some base values
    work3[0] = work2[15];
    for(int x=1;x<320;x++)
    {
        work3[x-1] ^= work2[x&63];
        if(work3[x-1]<0x80) work3[x]=work2[(x+work3[x-1])&63];
        else                work3[x]=work1[(x+work3[x-1])&127];
    }


    uint64 qCount = *((uint64*)&work3[310]);
    int nExtra=READ_PAD8(qCount+work3[300])>>3;
    for(int x=1;x<512+nExtra;x++)
    {
        qCount+= READ_PAD32( qCount );
        if((qCount>>8)&0x00878787)        work3[qCount%320]++;

        qCount-= READ_PAD8( qCount+work3[qCount%160] );
        if(qCount&0x80000000)   { qCount+= READ_PAD8( qCount&0x8080FFFF ); }
        else                    { qCount+= READ_PAD32( qCount&0x7F60FAFB ); }

        qCount+= READ_PAD32( qCount+work3[qCount%160] );
        if((qCount>>8)&0x00F00000)        work3[qCount%320]++;

        qCount+= READ_PAD32( *((uint32*)&work3[qCount&0xFF]) );
        work3[x%320]=work2[x&63]^qCount;

        qCount+= READ_PAD32( (qCount>>32)+work3[x%200] );
        *((uint32*)&work3[qCount%316]) ^= (qCount>>24)&0xFFFFFFFF;
        if((qCount&0x07)==0x03) x++;
        qCount-= READ_PAD8( (x*x) );
        if((qCount&0x07)==0x01) x++;
    }


    //FILE *fp = fopenSOLID("hashdump.dat","wb");
    //fwrite(work1,1,512,fp);
    //fclose(fp);

    //SHA256_CTX sha256;
    //SHA256_Init(&sha256);
    //SHA256_Update(&sha256, work1, 512);
    //SHA256_Final((unsigned char*)&final_hash, &sha256);
    SHA256(work1, 512, (unsigned char*)&final_hash);
    return;
}

#ifdef __WXMSW__

wchar_t* strto_utf16(const std::string &str)
{
    int nChars =::MultiByteToWideChar(CP_UTF8,0,str.c_str(),str.length(),NULL,0);
    if (nChars == 0)
    {
        wchar_t *pOut = new wchar_t[1];
        pOut[0]=0;
        return pOut;
    }
    wchar_t *pOut = new wchar_t[nChars+1];
    pOut[nChars]=0;
    ::MultiByteToWideChar(CP_UTF8,0,str.c_str(),str.length(),pOut,nChars);
    return pOut;
}

FILE* fopenSOLID(const char *file, const char *mode)
{
    char file_temp[256]={0};
    strlcpy(file_temp,file,255);
    for(int x=0;x<256;x++) if(file_temp[x]=='/') file_temp[x]='\\';  //convert / to \  if you know what i mean
    wchar_t *filestr=strto_utf16(file_temp);
    wchar_t *modestr=strto_utf16(mode);
    FILE *fp=_wfopen(filestr,modestr);
    delete[] filestr;
    delete[] modestr;
    return fp;
}

#else
FILE* fopenSOLID(const char *file, const char *mode)
{
    return fopen(file,mode);
}
#endif
