//Please read license.txt for licensing and copyright information

#include "headers.h"
#include "cryptopp/sha.h"
#include "db.h"
#include "net.h"
#include "init.h"
#include "wallet_func.h"
#undef printf
#include <boost/asio.hpp>
#include <boost/iostreams/concepts.hpp>
#include <boost/iostreams/stream.hpp>
#include <boost/algorithm/string.hpp>
#ifdef USE_SSL
#include <boost/asio/ssl.hpp>
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/lexical_cast.hpp>
typedef boost::asio::ssl::stream<boost::asio::ip::tcp::socket> SSLStream;
#endif
#include "json/json_spirit_reader_template.h"
#include "json/json_spirit_writer_template.h"
#include "json/json_spirit_utils.h"
#define printf OutputDebugStringF

using namespace std;
using namespace boost;
using namespace boost::asio;
using namespace json_spirit;


void Mining_AddPool(MINING_POOL *pPool);
void Mining_RemovePool(int id);
typedef Value(*rpcfn_type)(const Array& params, bool fHelp);
extern map<string, rpcfn_type> mapCallTable;
extern CCriticalSection cs_main;

extern CCriticalSection cs_mapTransactions;



extern int64 g_MiningTestWorkShares;
extern int64 g_MiningTestWorkInvalids;
extern int64 g_qShareTimeWindow;
extern uint256 g_MiningShareTargetHash;
extern string g_MiningIDString;
extern int64 g_MiningShareTargetInt;

vector<CBlock*> g_NewBlockList;
static map<uint256, pair<CBlock*, uint32> > g_mapNewBlock;
CCriticalSection cs_getwork;
CReserveKey g_MiningReserveKey;


Object JSONRPCError(int code, const string& message)
{
    Object error;
    error.push_back(Pair("code", code));
    error.push_back(Pair("message", message));
    return error;
}



int64 AmountFromValueStr(const Value& value)
{
    int64 nAmount = boost::lexical_cast<int64>(value.get_str());
    if(nAmount<=0)   throw JSONRPCError(-3, "Invalid amount");
    return nAmount;
}

Value ValueFromAmountStr(int64 amount)
{
    string result = boost::lexical_cast<string>(amount);
    return result;
    //boost::format("%");
    //return (double)amount / (double)COIN;
}

void WalletTxToJSON(const CWalletTx& wtx, Object& entry)
{
    entry.push_back(Pair("confirmations", wtx.GetDepthInMainChain()));
    entry.push_back(Pair("txid", wtx.GetHash().GetHex()));
    entry.push_back(Pair("time", (boost::int64_t)wtx.GetTxTime()));
    BOOST_FOREACH(const PAIRTYPE(string,string)& item, wtx.mapValue)
        entry.push_back(Pair(item.first, item.second));
}

string AccountFromValue(const Value& value)
{
    string strAccount = value.get_str();
    if (strAccount == "*")
        throw JSONRPCError(-11, "Invalid account name");
    return strAccount;
}

Value help(const Array& params, bool fHelp)
{
    if (fHelp || params.size() > 1)
        throw runtime_error(
            "help [command]\n"
            "List commands, or get help for a command.");

    string strCommand;
    if (params.size() > 0)
        strCommand = params[0].get_str();

    string strRet;
    set<rpcfn_type> setDone;
    for (map<string, rpcfn_type>::iterator mi = mapCallTable.begin(); mi != mapCallTable.end(); ++mi)
    {
        string strMethod = (*mi).first;
        // We already filter duplicates, but these deprecated screw up the sort order
        if (strMethod == "getamountreceived" ||
            strMethod == "getallreceived" ||
            (strMethod.find("label") != string::npos))
            continue;
        if (strCommand != "" && strMethod != strCommand)
            continue;
        try
        {
            Array params;
            rpcfn_type pfn = (*mi).second;
            if (setDone.insert(pfn).second)
                (*pfn)(params, true);
        }
        catch (std::exception& e)
        {
            // Help text is returned in an exception
            string strHelp = string(e.what());
            if (strCommand == "")
                if (strHelp.find('\n') != -1)
                    strHelp = strHelp.substr(0, strHelp.find('\n'));
            strRet += strHelp + "\n";
        }
    }
    if (strRet == "")
        strRet = strprintf("help: unknown command: %s\n", strCommand.c_str());
    strRet = strRet.substr(0,strRet.size()-1);
    return strRet;
}


Value stop(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "stop\n"
            "Stop solidcoin server.");

    // Shutdown will take long enough that the response should get back
    CreateThread(Shutdown, NULL);
    return "solidcoin server stopping";
}

Value getblocknumber(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "getblocknumber\n"
            "Returns the block number of the latest block in the longest block chain.");

    return g_qBlockBestHeight;
}

double GetDifficulty(uint32 dwBits)
{
    // Floating point number that is a multiple of the minimum difficulty minimum difficulty = 1.0.
    uint32 dwDiff1=g_bnBlockProofOfWorkLimit.GetCompact();
    if(dwBits==0)
    {
        if(g_pBlockBestIndex == NULL) return 1.0;
        if((g_pBlockBestIndex->blk.nBlockNum%2) == 0 )
        {
            if(g_pBlockBestIndex->pprev==0) return 1.0;
            dwBits = g_pBlockBestIndex->pprev->blk.dwBits;
        }
        else dwBits = g_pBlockBestIndex->blk.dwBits;
    }
    int nShift = (dwBits >> 24) & 0xff;
    double dDiff =  (double)(dwDiff1&0x00FFFFFF) / (double)(dwBits & 0x00ffffff);
    while (nShift < ((dwDiff1>>24)&0xFF) )
    {
        dDiff *= 256.0;
        nShift++;
    }
    while (nShift > ((dwDiff1>>24)&0xFF))
    {
        dDiff /= 256.0;
        nShift--;
    }
    return dDiff;
}

Value getdifficulty(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "getdifficulty\n"
            "Returns the proof-of-work difficulty as a multiple of the minimum difficulty.");

    return GetDifficulty(0);
}


Value sc_getmining(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "sc_getmining\n"
            "Returns status of all miners.");

    MUTEX_LOCK(cs_main);
    Array ret;

    static int64 qLastMilliTime=0;
    static int64 qLastTestworkShares=0;
    #define DIFF1   131072


    int64 qTime = GetTimeMillis()-qLastMilliTime;
    qLastMilliTime=GetTimeMillis();
    int64 qShares = (g_MiningTestWorkShares-qLastTestworkShares)*DIFF1;
    qLastTestworkShares=g_MiningTestWorkShares;

    Object testwork;

    testwork.push_back(Pair("testwork_totalhash",    (g_MiningTestWorkShares*DIFF1) ));
    testwork.push_back(Pair("testwork_invalids",    g_MiningTestWorkInvalids));
    testwork.push_back(Pair("testwork_hashpersec",   (int64)(qShares/((double)qTime/1000.0)) ));
    ret.push_back(testwork);


    for(int x=0;x<g_MiningPoolList.size();x++)
    {
        Object miner;
        MINING_POOL *pPool=&g_MiningPoolList[x];
        miner.push_back(Pair("name",       pPool->name));
        miner.push_back(Pair("running",    pPool->nRunning?true:false));
        miner.push_back(Pair("hashpersec",    pPool->qLastHashPerSec));
        miner.push_back(Pair("totalhash",    pPool->qTotalHash));
        miner.push_back(Pair("totalshares",    pPool->qTotalShares));
        ret.push_back(miner);
    }
    return ret;
}


Value sc_setmining(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 2)
        throw runtime_error(
            "sc_setmining <poolid> <on>\n"
            "<poolid> A number which specifies the pool to use as defined in solidcoin.conf\n"
            "<generate> is true or false to turn generation on or off.\n");


    int nPoolID = params[0].get_int();
    bool bON = params[1].get_bool();
    {
        MUTEX_LOCK(cs_main);
        if(nPoolID<0 || nPoolID>=g_MiningPoolList.size())  throw runtime_error("sc_setmining - incorrect pool ID.");
        MINING_POOL *pPool=&g_MiningPoolList[nPoolID];
        if(!bON && pPool->nRunning)         //close her down
        {
            Mining_RemovePool(pPool->id);
            pPool->nRunning=0;
            pPool->qLastHashPerSec=0;
            pPool->qTotalHash=0;
            pPool->qTotalShares=0;
        }
        else if(bON && !pPool->nRunning)    //start her up
        {
            pPool->id=nPoolID;
            Mining_AddPool(pPool);
            pPool->nRunning=1;
            pPool->qLastHashPerSec=0;
            pPool->qTotalHash=0;
            pPool->qTotalShares=0;
        }
    }
    UIThreadCall(boost::bind(CalledUpdateMiningUI));
    return Value::null;
}

Value sc_getinfo(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 0)
        throw runtime_error(
            "sc_getinfo\n"
            "Returns an object containing various state info about solidcoind.");

    Object obj;
    obj.push_back(Pair("version",       (int)VERSION));
    obj.push_back(Pair("blocks",        g_qBlockBestHeight));
    obj.push_back(Pair("connections",   (int)vNodes.size()));
    obj.push_back(Pair("proxy",         (g_bUseProxy ? addrProxy.ToStringIPPort() : string())));
    obj.push_back(Pair("difficulty",    (double)GetDifficulty(0)));
    obj.push_back(Pair("testnet",       fTestNet));
    obj.push_back(Pair("mintxfee",      ValueFromAmountStr(MIN_TX_FEE)));
    obj.push_back(Pair("errors",        GetWarnings("statusbar")));
    return obj;
}

Value sc_getwalletinfo(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;

    if (fHelp || params.size() != 1 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_getwalletinfo <wallet>\n"
            "Returns an object containing various state info for the given wallet.");
    Object obj;
    if (pWallet->IsCrypted())           obj.push_back(Pair("unlocked_until", (boost::int64_t)pWallet->m_nWalletLockTime));
    obj.push_back(Pair("balance", ValueFromAmountStr(pWallet->GetBalance())));
    obj.push_back(Pair("keypoololdest", (boost::int64_t)pWallet->GetOldestKeyPoolTime()));
    obj.push_back(Pair("keypoolsize",   pWallet->GetKeyPoolSize()));
    return obj;
}

Value sc_getnewaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if (fHelp || params.size()<1 || params.size() > 2 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "getnewaddress <wallet> [account]\n"
            "Returns a new solidcoin address for receiving payments from [wallet].  "
            "If [account] is specified (recommended), it is added to the address book "
            "so payments received with the address will be credited to [account].");

    if(!pWallet->IsLocked())           pWallet->TopUpKeyPool();
    if(pWallet->GetKeyPoolSize() < 1)  throw JSONRPCError(-12, "Error: Keypool ran out, please call keypoolrefill first");

    string strAccount;  // Parse the account first so we don't generate a key if there's an error
    if (params.size() > 1)  strAccount = AccountFromValue(params[1]);
    CSolidCoinAddress address(pWallet->GetOrReuseKeyFromPool());    // Generate a new key that is added to wallet
    pWallet->SetAddressBookName(address, strAccount);   // This could be done in the same main CS as GetKeyFromKeyPool.
    return address.ToString();
}


// requires cs_main, cs_mapWallet, cs_mapAddressBook locks
CSolidCoinAddress GetAccountAddress(CWallet *pWallet, string strAccount, bool bForceNew=false)
{
    CWalletDB walletdb(pWallet->strWalletFile);
    CAccount account;
    {
        walletdb.ReadAccount(strAccount, account);
        bool bKeyUsed = false;
        if (!account.vchPubKey.empty())     // Check if the current key has been used
        {
            CScript scriptPubKey;
            scriptPubKey.SetSolidCoinAddress(account.vchPubKey);
            for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin();
                 it != pWallet->mapWallet.end() && !account.vchPubKey.empty();
                 ++it)
            {
                const CWalletTx& wtx = (*it).second;
                BOOST_FOREACH(const CTxOut& txout, wtx.vout)
                    if (txout.scriptPubKey == scriptPubKey)
                        bKeyUsed = true;
            }
        }

        // Generate a new key
        if (account.vchPubKey.empty() || bForceNew || bKeyUsed)
        {
            if (pWallet->GetKeyPoolSize() < 1)
            {
                if (bKeyUsed || bForceNew)
                    throw JSONRPCError(-12, "Error: Keypool ran out, please call topupkeypool first");
            }
            else
            {
                account.vchPubKey = pWallet->GetOrReuseKeyFromPool();
                pWallet->SetAddressBookName(CSolidCoinAddress(account.vchPubKey), strAccount);
                walletdb.WriteAccount(strAccount, account);
            }
        }
    }
    return CSolidCoinAddress(account.vchPubKey);
}

Value sc_getaccountaddress(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if (fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_getaccountaddress <wallet> <account>\n"
            "Returns the current solidcoin address for receiving payments to this account.");

    string strAccount = AccountFromValue(params[1]);    // Parse the account first so we don't generate a key if there's an error
    Value ret = GetAccountAddress(pWallet,strAccount).ToString();
    return ret;
}

Value sc_setaccount(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || (params.size()!=2 && params.size()!=3) || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_setaccount <wallet> <solidcoinaddress> <account>\n"
            "Sets the account associated with the given address.");

    CSolidCoinAddress address(params[1].get_str());
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");

    string strAccount;
    if (params.size() > 2)  strAccount = AccountFromValue(params[2]);
    // Detect when changing the account of an address that is the 'unused current key' of another account:
    if (pWallet->mapAddressBook.count(address))
    {
        string strOldAccount = pWallet->mapAddressBook[address];
        if (address == GetAccountAddress(pWallet,strOldAccount))    GetAccountAddress(pWallet,strOldAccount, true);
    }
    pWallet->SetAddressBookName(address, strAccount);
    return Value::null;
}


Value sc_getaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()!=2  || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_getaccount <wallet> <solidcoinaddress>\n"
            "Returns the account associated with the given address.");

    CSolidCoinAddress address(params[1].get_str());
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");
    string strAccount;
    map<CSolidCoinAddress, string>::iterator mi = pWallet->mapAddressBook.find(address);
    if (mi != pWallet->mapAddressBook.end() && !(*mi).second.empty())   strAccount = (*mi).second;
    return strAccount;
}


Value sc_getaddressesbyaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()!=2  || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "getaddressesbyaccount <wallet> <account>\n"
            "Returns the list of addresses for the given account in given wallet.");

    string strAccount = AccountFromValue(params[1]);
    Array ret;  // Find all addresses that have the given account
    BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, string)& item, pWallet->mapAddressBook)
    {
        const CSolidCoinAddress& address = item.first;
        const string& strName = item.second;
        if (strName == strAccount)  ret.push_back(address.ToString());
    }
    return ret;
}

Value sc_gettxfee(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()!=2  || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_gettxfee <wallet> <amount>\n"
            "Returns the transaction fee for an amount to be sent");

    int64 nAmount = params[1].get_int64();
    int64 nFees=0;
    if(!pWallet->GetFees(nAmount,nFees)) nFees=0;
    Object obj;
    obj.push_back(Pair("fee",ValueFromAmountStr(nFees)));
    return obj;
}

Value sc_sendtoaddress(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if ( fHelp || params.size() < 3 || params.size() > 4 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_sendtoaddress <wallet> <solidcoinaddress> <amount> [maxfee]\n"
            "<amount> is an integer where 10000 = 1SC\n"
            "[maxfee] is an integer specifying the max fee you will pay. If fee is over that the send won't happen\n");

    CSolidCoinAddress address(params[1].get_str());
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");

    // Amount
    int64 nMinFee=0;
    int64 nAmount = params[2].get_int64();
    if(params.size() > 3)  nMinFee=params[3].get_int64();


    // Wallet comments
    CWalletTx wtx;
    int nCommentID=2;
    nCommentID=3;
    //if (params.size() > nCommentID && params[nCommentID].type() != null_type && !params[nCommentID].get_str().empty()) wtx.mapValue["comment"] = params[nCommentID].get_str();
    //if (params.size() > (nCommentID+1) && params[(nCommentID+1)].type() != null_type && !params[(nCommentID+1)].get_str().empty()) wtx.mapValue["to"]      = params[(nCommentID+1)].get_str();

    if(pWallet->IsLocked()) throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");

    string strError = pWallet->SendMoneyToSolidCoinAddress(address, nAmount, wtx,nMinFee);
    if (strError != "") throw JSONRPCError(-4, strError);
    return wtx.GetHash().GetHex();
}

Value sc_getreceivedbyaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 2 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_getreceivedbyaddress <wallet> <solidcoinaddress> [minconf=1]\n"
            "Returns the total amount received by <solidcoinaddress> in transactions with at least [minconf] confirmations.");

    // SolidCoin address
    CSolidCoinAddress address = CSolidCoinAddress(params[1].get_str());
    CScript scriptPubKey;
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");
    scriptPubKey.SetSolidCoinAddress(address);
    if (!IsMine(*pWallet,scriptPubKey))
    {
        return ValueFromAmountStr(0);
    }

    // Minimum confirmations
    int64 nMinDepth = 1;
    if (params.size() > 2)  nMinDepth = params[2].get_int();

    // Tally
    int64 nAmount = 0;
    {
        for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
        {
            const CWalletTx& wtx = (*it).second;
            if (wtx.IsCoinBase() || !wtx.IsFinal()) continue;

            BOOST_FOREACH(const CTxOut& txout, wtx.vout)
            {
                if (txout.scriptPubKey == scriptPubKey && wtx.GetDepthInMainChain() >= nMinDepth)   nAmount += txout.nValue;
            }
        }
    }

    return ValueFromAmountStr(nAmount);
}


void GetAccountAddresses(CWallet *pWallet, string strAccount, set<CSolidCoinAddress>& setAddress)
{
    BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, string)& item, pWallet->mapAddressBook)
    {
        const CSolidCoinAddress& address = item.first;
        const string& strName = item.second;
        if (strName == strAccount)
            setAddress.insert(address);
    }
}

Value sc_getreceivedbyaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 2 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_getreceivedbyaccount <wallet> <account> [minconf=1]\n"
            "Returns the total amount received by addresses with <account> in transactions with at least [minconf] confirmations.");

    int64 nMinDepth = 1;    // Minimum confirmations
    string strAccount;  // Get the set of pub keys that have the label
    strAccount = AccountFromValue(params[1]);
    if (params.size() > 2)  nMinDepth = params[2].get_int();
    set<CSolidCoinAddress> setAddress;
    GetAccountAddresses(pWallet,strAccount, setAddress);

    // Tally
    int64 nAmount = 0;
    for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
    {
        const CWalletTx& wtx = (*it).second;
        if (wtx.IsCoinBase() || !wtx.IsFinal()) continue;
        BOOST_FOREACH(const CTxOut& txout, wtx.vout)
        {
            CSolidCoinAddress address;
            if (ExtractAddress(txout.scriptPubKey, pWallet, address) && setAddress.count(address))
            {
                if (wtx.GetDepthInMainChain() >= nMinDepth) nAmount += txout.nValue;
            }
        }
    }
    return ValueFromAmountStr(nAmount);
}

int64 GetAccountBalance(CWallet *pWallet, CWalletDB& walletdb, const string& strAccount, int64 nMinDepth)
{
    int64 nBalance = 0;
    // Tally wallet transactions
    for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
    {
        const CWalletTx& wtx = (*it).second;
        if (!wtx.IsFinal()) continue;

        int64 nGenerated, nReceived, nSent, nFee;
        wtx.GetAccountAmounts(strAccount, nGenerated, nReceived, nSent, nFee);

        if (nReceived != 0 && wtx.GetDepthInMainChain() >= nMinDepth)
            nBalance += nReceived;
        nBalance += nGenerated - nSent - nFee;
    }

    nBalance += walletdb.GetAccountCreditDebit(strAccount);     // Tally internal accounting entries
    return nBalance;
}

int64 GetAccountBalance(CWallet *pWallet, const string& strAccount, int64 nMinDepth)
{
    CWalletDB walletdb(pWallet->strWalletFile);
    return GetAccountBalance(pWallet,walletdb, strAccount, nMinDepth);
}

Value sc_getbalance(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()<1 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())) )
    {
        throw runtime_error("sc_getbalance <wallet> [account] [minconf=1]\n"
            "If [account] is not specified, returns the server's total available balance.\n"
            "If [account] is specified, returns the balance in the account.");
    }

    string strAccount;
    int64 nMinDepth = 1;

    if(params.size()==1)   return ValueFromAmountStr(pWallet->GetBalance());
    if(params.size() > 2)  nMinDepth = params[2].get_int();
    strAccount=params[1].get_str();

    if (strAccount == "*") {
        // Calculate total balance a different way from GetBalance()  GetBalance() sums up all unspent TxOuts)
        // getbalance and getbalance '*' should always return the same number.
        int64 nBalance = 0;
        for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
        {
            const CWalletTx& wtx = (*it).second;
            if (!wtx.IsFinal()) continue;

            int64 allGeneratedImmature, allGeneratedMature, allFee;
            allGeneratedImmature = allGeneratedMature = allFee = 0;
            string strSentAccount;
            list<pair<CSolidCoinAddress, int64> > listReceived;
            list<pair<CSolidCoinAddress, int64> > listSent;
            wtx.GetAmounts(allGeneratedImmature, allGeneratedMature, listReceived, listSent, allFee, strSentAccount);
            if (wtx.GetDepthInMainChain() >= nMinDepth)
            {
                BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress,int64)& r, listReceived)
                {
                    nBalance += r.second;
                }
            }
            BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress,int64)& r, listSent)
            {
                nBalance -= r.second;
            }
            nBalance -= allFee;
            nBalance += allGeneratedMature;
        }
        return  ValueFromAmountStr(nBalance);
    }
    strAccount = AccountFromValue(params[1]);
    int64 nBalance = GetAccountBalance(pWallet,strAccount, nMinDepth);
    return ValueFromAmountStr(nBalance);
}



Value sc_movecmd(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()<4 || params.size() > 5 || !(pWallet=Wallet_Get(params[0].get_str())) )
        throw runtime_error(
            "sc_move <wallet> <fromaccount> <toaccount> <amount> [comment]\n"
            "Move from one account in your wallet to another.");

    string strComment,strFrom,strTo;
    int64 nAmount =0;

    strFrom = AccountFromValue(params[1]);
    strTo = AccountFromValue(params[2]);
    nAmount = params[3].get_int64();
    if (params.size() > 3)  strComment = params[4].get_str();

    CWalletDB walletdb(pWallet->strWalletFile);
    walletdb.TxnBegin();
    int64 nNow = SolidTime_Get();

    CAccountingEntry debit;
    debit.strAccount = strFrom;
    debit.nCreditDebit = -nAmount;
    debit.nTime = nNow;
    debit.strOtherAccount = strTo;
    debit.strComment = strComment;
    walletdb.WriteAccountingEntry(debit);

    CAccountingEntry credit;
    credit.strAccount = strTo;
    credit.nCreditDebit = nAmount;
    credit.nTime = nNow;
    credit.strOtherAccount = strFrom;
    credit.strComment = strComment;
    walletdb.WriteAccountingEntry(credit);
    walletdb.TxnCommit();
    return true;
}

Value sc_sendfrom(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 4 || params.size() > 7 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_sendfrom <wallet> <fromaccount> <tosolidcoinaddress> <amount> [minconf=1] [comment] [comment-to]\n"
            "<amount> is an integer. 10000 units = 1SC\n"
            "requires wallet passphrase to be set with walletpassphrase first if wallet is encrypted");

    string strAccount;
    strAccount = AccountFromValue(params[1]);
    CSolidCoinAddress address = CSolidCoinAddress(params[2].get_str());
    if (!address.IsValid()) throw JSONRPCError(-5, "Invalid solidcoin address");
    int64 nAmount = 0;
    int nMinDepth = 1;
    CWalletTx wtx;

    wtx.strFromAccount = strAccount;
    nAmount=params[3].get_int64();
    if (params.size() > 4)  nMinDepth = params[4].get_int();
    if (params.size() > 5 && params[5].type() != null_type && !params[5].get_str().empty()) wtx.mapValue["comment"] = params[5].get_str();
    if (params.size() > 6 && params[6].type() != null_type && !params[6].get_str().empty()) wtx.mapValue["to"]      = params[6].get_str();

    if(pWallet->IsLocked()) throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");

    int64 nBalance = GetAccountBalance(pWallet,strAccount, nMinDepth);  // Check funds
    if (nAmount > nBalance) throw JSONRPCError(-6, "Account has insufficient funds");

    string strError = pWallet->SendMoneyToSolidCoinAddress(address, nAmount, wtx,0);
    if (strError != "") throw JSONRPCError(-4, strError);
    return wtx.GetHash().GetHex();
}


Value sc_sendmany(const Array& params, bool fHelp)
{
    MUTEX_LOCK(cs_main);
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 3 || params.size() > 5 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_sendmany <wallet> <fromaccount> {address:amount,...} [minconf=1] [comment]\n"
            "amounts are an integer\n"
            "requires wallet passphrase to be set with walletpassphrase if wallet is encrypted");

    CWalletTx wtx;
    string strAccount;
    Object sendTo;
    int nMinDepth = 1;

    strAccount = AccountFromValue(params[1]);
    wtx.strFromAccount = strAccount;
    sendTo = params[2].get_obj();
    if (params.size() > 3)  nMinDepth = params[3].get_int();
    if (params.size() > 4 && params[4].type() != null_type && !params[4].get_str().empty()) wtx.mapValue["comment"] = params[4].get_str();

    set<CSolidCoinAddress> setAddress;
    vector<pair<CScript, int64> > vecSend;

    int64 totalAmount = 0;
    BOOST_FOREACH(const Pair& s, sendTo)
    {
        CSolidCoinAddress address(s.name_);
        if(!address.IsValid())             throw JSONRPCError(-5, string("Invalid solidcoin address:")+s.name_);
        if(setAddress.count(address))      throw JSONRPCError(-8, string("Invalid parameter, duplicated address: ")+s.name_);
        setAddress.insert(address);

        CScript scriptPubKey;
        scriptPubKey.SetSolidCoinAddress(address);
        int64 nAmount = 0;
        nAmount=AmountFromValueStr(s.value_);
        totalAmount += nAmount;
        vecSend.push_back(make_pair(scriptPubKey, nAmount));
    }

    if(pWallet->IsLocked()) throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");

    // Check funds
    int64 nBalance = GetAccountBalance(pWallet,strAccount, nMinDepth);
    if (totalAmount > nBalance) throw JSONRPCError(-6, "Account has insufficient funds");

    // Send
    CReserveKey keyChange;
    keyChange.SetWallet(pWallet);
    int64 nFeeRequired = 0;
    bool fCreated = pWallet->CreateTransaction(vecSend, wtx, keyChange, nFeeRequired);
    if (!fCreated)
    {
        if (totalAmount + nFeeRequired > pWallet->GetBalance()) throw JSONRPCError(-6, "Insufficient funds");
        throw JSONRPCError(-4, "Transaction creation failed");
    }
    if (!pWallet->CommitTransaction(wtx, keyChange))    throw JSONRPCError(-4, "Transaction commit failed");
    return wtx.GetHash().GetHex();
}


struct tallyitem
{
    int64 nAmount;
    int64 nConf;
    tallyitem()
    {
        nAmount = 0;
        nConf = INT64_MAX;
    }
};

Value ListReceived(CWallet *pWallet, const Array& params, bool fByAccounts)
{
    int nMinDepth = 1;              // Minimum confirmations
    bool fIncludeEmpty = false;     // Whether to include empty accounts

    if (params.size() > 1)  nMinDepth = params[1].get_int();
    if (params.size() > 2)  fIncludeEmpty = params[2].get_bool();

    // Tally
    map<CSolidCoinAddress, tallyitem> mapTally;
    {
        for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
        {
            const CWalletTx& wtx = (*it).second;
            if (wtx.IsCoinBase() || !wtx.IsFinal()) continue;

            int64 nDepth = wtx.GetDepthInMainChain();
            if (nDepth < nMinDepth)     continue;

            BOOST_FOREACH(const CTxOut& txout, wtx.vout)
            {
                CSolidCoinAddress address;
                if (!ExtractAddress(txout.scriptPubKey, pWallet, address) || !address.IsValid())    continue;
                tallyitem& item = mapTally[address];
                item.nAmount += txout.nValue;
                item.nConf = min(item.nConf, nDepth);
            }
        }
    }

    // Reply
    Array ret;
    map<string, tallyitem> mapAccountTally;
    {
        BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, string)& item, pWallet->mapAddressBook)
        {
            const CSolidCoinAddress& address = item.first;
            const string& strAccount = item.second;
            map<CSolidCoinAddress, tallyitem>::iterator it = mapTally.find(address);
            if (it == mapTally.end() && !fIncludeEmpty)
                continue;

            int64 nAmount = 0;
            int64 nConf = INT64_MAX;
            if (it != mapTally.end())
            {
                nAmount = (*it).second.nAmount;
                nConf = (*it).second.nConf;
            }

            if (fByAccounts)
            {
                tallyitem& item = mapAccountTally[strAccount];
                item.nAmount += nAmount;
                item.nConf = min(item.nConf, nConf);
            }
            else
            {
                Object obj;
                obj.push_back(Pair("address",       address.ToString()));
                obj.push_back(Pair("account",       strAccount));
                obj.push_back(Pair("label",         strAccount)); // deprecated
                obj.push_back(Pair("amount",    ValueFromAmountStr(nAmount)));  //only show int amount with SolidCoin API
                obj.push_back(Pair("confirmations", (nConf == INT64_MAX ? 0 : nConf)));
                ret.push_back(obj);
            }
        }
    }

    if (fByAccounts)
    {
        for (map<string, tallyitem>::iterator it = mapAccountTally.begin(); it != mapAccountTally.end(); ++it)
        {
            int64 nAmount = (*it).second.nAmount;
            int64 nConf = (*it).second.nConf;
            Object obj;
            obj.push_back(Pair("account",       (*it).first));
            obj.push_back(Pair("label",         (*it).first)); // deprecated
            obj.push_back(Pair("amount",    ValueFromAmountStr(nAmount)));      //only show int amount with SolidCoin API
            obj.push_back(Pair("confirmations", (nConf == INT64_MAX ? (int64)0 : nConf)));
            ret.push_back(obj);
        }
    }

    return ret;
}


Value sc_listreceivedbyaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 1 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_listreceivedbyaddress <wallet> [minconf=1] [includeempty=false]\n"
            "[minconf] is the minimum number of confirmations before payments are included.\n"
            "[includeempty] whether to include addresses that haven't received any payments.\n"
            "Returns an array of objects containing:\n"
            "  \"address\" : receiving address\n"
            "  \"account\" : the account of the receiving address\n"
            "  \"amount\" : total amount received by the address\n"
            "  \"confirmations\" : number of confirmations of the most recent transaction included");

    return ListReceived(pWallet,params, false);
}

Value sc_listreceivedbyaccount(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 1 || params.size() > 3 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_listreceivedbyaccount <wallet> [minconf=1] [includeempty=false]\n"
            "[minconf] is the minimum number of confirmations before payments are included.\n"
            "[includeempty] whether to include accounts that haven't received any payments.\n"
            "Returns an array of objects containing:\n"
            "  \"account\" : the account of the receiving addresses\n"
            "  \"amount\" : total amount received by addresses with this account\n"
            "  \"confirmations\" : number of confirmations of the most recent transaction included");

    return ListReceived(pWallet, params, true);
}

void ListTransactions(CWallet *pWallet, const CWalletTx& wtx, const string& strAccount, int nMinDepth, bool fLong, Array& ret)
{
    int64 nGeneratedImmature, nGeneratedMature, nFee;
    string strSentAccount;
    list<pair<CSolidCoinAddress, int64> > listReceived;
    list<pair<CSolidCoinAddress, int64> > listSent;
    wtx.GetAmounts(nGeneratedImmature, nGeneratedMature, listReceived, listSent, nFee, strSentAccount);

    bool fAllAccounts = (strAccount == string("*"));

    // Generated blocks assigned to account ""
    if ((nGeneratedMature+nGeneratedImmature) != 0 && (fAllAccounts || strAccount == ""))
    {
        Object entry;
        entry.push_back(Pair("account", string("")));
        if (nGeneratedImmature)
        {
            entry.push_back(Pair("category", wtx.GetDepthInMainChain() ? "immature" : "orphan"));
            entry.push_back(Pair("amount", ValueFromAmountStr(nGeneratedImmature)));    //only show int amount with SolidCoin API
        }
        else
        {
            entry.push_back(Pair("category", "generate"));
            entry.push_back(Pair("amount", ValueFromAmountStr(nGeneratedMature)));  //only show int amount with SolidCoin API
        }
        if (fLong)  WalletTxToJSON(wtx, entry);
        ret.push_back(entry);
    }

    // Sent
    if ((!listSent.empty() || nFee != 0) && (fAllAccounts || strAccount == strSentAccount))
    {
        BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, int64)& s, listSent)
        {
            Object entry;
            entry.push_back(Pair("account", strSentAccount));
            entry.push_back(Pair("address", s.first.ToString()));
            entry.push_back(Pair("category", "send"));
            entry.push_back(Pair("amount", ValueFromAmountStr(-s.second)));
            entry.push_back(Pair("fee", ValueFromAmountStr(-nFee)));

            if (fLong)  WalletTxToJSON(wtx, entry);
            ret.push_back(entry);
        }
    }

    // Received
    if (listReceived.size() > 0 && wtx.GetDepthInMainChain() >= nMinDepth)
    {
        BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, int64)& r, listReceived)
        {
            string account;
            if (pWallet->mapAddressBook.count(r.first)) account = pWallet->mapAddressBook[r.first];
            if (fAllAccounts || (account == strAccount))
            {
                Object entry;
                entry.push_back(Pair("account", account));
                entry.push_back(Pair("address", r.first.ToString()));
                entry.push_back(Pair("category", "receive"));
                entry.push_back(Pair("amount", ValueFromAmountStr(r.second)));      //only show int amount with SolidCoin API
                if (fLong)  WalletTxToJSON(wtx, entry);
                ret.push_back(entry);
            }
        }
    }

}

void AcentryToJSON(const CAccountingEntry& acentry, const string& strAccount, Array& ret)
{
    bool fAllAccounts = (strAccount == string("*"));

    if (fAllAccounts || acentry.strAccount == strAccount)
    {
        Object entry;
        entry.push_back(Pair("account", acentry.strAccount));
        entry.push_back(Pair("category", "move"));
        entry.push_back(Pair("time", (boost::int64_t)acentry.nTime));
        entry.push_back(Pair("amount", ValueFromAmountStr(acentry.nCreditDebit)));      //only show int amount with SolidCoin API
        entry.push_back(Pair("otheraccount", acentry.strOtherAccount));
        entry.push_back(Pair("comment", acentry.strComment));
        ret.push_back(entry);
    }
}

Value sc_listtransactions(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size() < 1 || params.size() > 4 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_listtransactions <wallet> [account] [count=10] [from=0]\n"
            "Returns up to [count] most recent transactions skipping the first [from] transactions for account [account].");

    string strAccount = "*";
    int nCount = 10;
    int nFrom = 0;

    if (params.size() > 1)  strAccount = params[1].get_str();
    if (params.size() > 2)  nCount = params[2].get_int();
    if (params.size() > 3)  nFrom = params[3].get_int();

    Array ret;
    CWalletDB walletdb(pWallet->strWalletFile);

    // Firs: get all CWalletTx and CAccountingEntry into a sorted-by-time multimap:
    typedef pair<CWalletTx*, CAccountingEntry*> TxPair;
    typedef multimap<int64, TxPair > TxItems;
    TxItems txByTime;

    for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
    {
        CWalletTx* wtx = &((*it).second);
        txByTime.insert(make_pair(wtx->GetTxTime(), TxPair(wtx, (CAccountingEntry*)0)));
    }
    list<CAccountingEntry> acentries;
    walletdb.ListAccountCreditDebit(strAccount, acentries);
    BOOST_FOREACH(CAccountingEntry& entry, acentries)
    {
        txByTime.insert(make_pair(entry.nTime, TxPair((CWalletTx*)0, &entry)));
    }

    // Now: iterate backwards until we have nCount items to return:
    TxItems::reverse_iterator it = txByTime.rbegin();
    if (txByTime.size() > nFrom) std::advance(it, nFrom);
    for (; it != txByTime.rend(); ++it)
    {
        CWalletTx *const pwtx = (*it).second.first;
        if (pwtx != 0)  ListTransactions(pWallet,*pwtx, strAccount, 0, true, ret);
        CAccountingEntry *const pacentry = (*it).second.second;
        if (pacentry != 0)  AcentryToJSON(*pacentry, strAccount, ret);
        if (ret.size() >= nCount) break;
    }
    // ret is now newest to oldest

    // Make sure we return only last nCount items (sends-to-self might give us an extra):
    if (ret.size() > nCount)
    {
        Array::iterator last = ret.begin();
        std::advance(last, nCount);
        ret.erase(last, ret.end());
    }
    std::reverse(ret.begin(), ret.end()); // oldest to newest

    return ret;
}

Value sc_listaccounts(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet *pWallet=0;
    if (fHelp || params.size()<1 || params.size() > 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_listaccounts <wallet> [minconf=1]\n"
            "Returns Object that has account names as keys, account balances as values.");


    int64 nMinDepth = 1;
    if(params.size() > 1)  nMinDepth = params[1].get_int();

    map<string, int64> mapAccountBalances;
    {
        BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, string)& entry, pWallet->mapAddressBook) {
            if (pWallet->HaveKey(entry.first)) // This address belongs to me
                mapAccountBalances[entry.second] = 0;
        }

        for (map<uint256, CWalletTx>::iterator it = pWallet->mapWallet.begin(); it != pWallet->mapWallet.end(); ++it)
        {
            const CWalletTx& wtx = (*it).second;
            int64 nGeneratedImmature, nGeneratedMature, nFee;
            string strSentAccount;
            list<pair<CSolidCoinAddress, int64> > listReceived;
            list<pair<CSolidCoinAddress, int64> > listSent;
            wtx.GetAmounts(nGeneratedImmature, nGeneratedMature, listReceived, listSent, nFee, strSentAccount);
            mapAccountBalances[strSentAccount] -= nFee;
            BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, int64)& s, listSent)
                mapAccountBalances[strSentAccount] -= s.second;
            if (wtx.GetDepthInMainChain() >= nMinDepth)
            {
                mapAccountBalances[""] += nGeneratedMature;
                BOOST_FOREACH(const PAIRTYPE(CSolidCoinAddress, int64)& r, listReceived)
                    if (pWallet->mapAddressBook.count(r.first))
                        mapAccountBalances[pWallet->mapAddressBook[r.first]] += r.second;
                    else
                        mapAccountBalances[""] += r.second;
            }
        }
    }

    list<CAccountingEntry> acentries;
    CWalletDB(pWallet->strWalletFile).ListAccountCreditDebit("*", acentries);
    BOOST_FOREACH(const CAccountingEntry& entry, acentries)
        mapAccountBalances[entry.strAccount] += entry.nCreditDebit;

    Object ret;
    BOOST_FOREACH(const PAIRTYPE(string, int64)& accountBalance, mapAccountBalances) {
        ret.push_back(Pair(accountBalance.first, ValueFromAmountStr(accountBalance.second)));
    }
    return ret;
}


Value sc_gettransaction(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if (fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_gettransaction <wallet> <txid>\n"
            "Get detailed information about <txid>");

    uint256 hash;
    hash.SetHex(params[1].get_str());

    Object entry;
    {
        if (!pWallet->mapWallet.count(hash))    throw JSONRPCError(-5, "Invalid or non-wallet transaction id");
        const CWalletTx& wtx = pWallet->mapWallet[hash];

        int64 nCredit = wtx.GetCredit();
        int64 nDebit = wtx.GetDebit();
        int64 nNet = nCredit - nDebit;
        int64 nFee = (wtx.IsFromMe() ? wtx.GetValueOut() - nDebit : 0);

        entry.push_back(Pair("amount", ValueFromAmountStr(nNet - nFee)));
        if (wtx.IsFromMe()) entry.push_back(Pair("fee", ValueFromAmountStr(nFee)));

        WalletTxToJSON(pWallet->mapWallet[hash], entry);

        Array details;
        ListTransactions(pWallet,pWallet->mapWallet[hash], "*", 0, false, details);
        entry.push_back(Pair("details", details));
    }

    return entry;
}


Value sc_backupwallet(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if (fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_backupwallet <wallet> <destination>\n"
            "Safely copies <wallet> to destination, which can be a directory or a path with filename.");

    string strDest = params[1].get_str();
    BackupWallet(*pWallet, strDest);
    return Value::null;
}

Value sc_keypoolrefill(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 1 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "keypoolrefill <wallet>\n"
            "Fills the keypool, requires wallet passphrase to be set if wallet is encrypted.");

    if (pWallet->IsLocked())    throw JSONRPCError(-13, "Error: Please enter the wallet passphrase with walletpassphrase first.");
    pWallet->TopUpKeyPool();
    if (pWallet->GetKeyPoolSize() < Setting_GetINT64("keypool"))    throw JSONRPCError(-4, "Error refreshing keypool.");
    return Value::null;
}

struct WALLET_SLEEP
{
    int nTime;
    std::string walletname;
};
void ThreadCleanWalletPassphrase(void* parg)
{
    WALLET_SLEEP *pSleep = (WALLET_SLEEP*)parg;
    while(1)
    {
        {
            LOCK_WALLET_ACCESS();
            CWallet *pWallet=Wallet_Get(pSleep->walletname);
            if(pWallet)
            {
                if(SolidTime_Get()>pWallet->m_nWalletLockTime)
                {
                    if(pWallet->m_nWalletLockTime!=0) pWallet->Lock();  //if its still positive, we should lock
                    break;
                }
            }
        }
        Sleep(1000);
    }
    delete pSleep;
}

Value sc_walletpassphrase(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 3 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "walletpassphrase <wallet> <passphrase> <timeout>\n"
            "Stores the wallet decryption key in memory for <timeout> seconds.");

    if (!pWallet->IsCrypted())  throw JSONRPCError(-15, "Error: running with an unencrypted wallet, but walletpassphrase was called.");
    if (!pWallet->IsLocked())   throw JSONRPCError(-17, "Error: Wallet is already unlocked.");

    // Note that the walletpassphrase is stored in params[1] which is not mlock()ed
    string strWalletPass;
    strWalletPass.reserve(100);
    mlock(&strWalletPass[0], strWalletPass.capacity());
    strWalletPass = params[1].get_str();

    if (strWalletPass.length() > 0)
    {
        if (!pWallet->Unlock(strWalletPass))
        {
            fill(strWalletPass.begin(), strWalletPass.end(), '\0');
            munlock(&strWalletPass[0], strWalletPass.capacity());
            throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");
        }
        fill(strWalletPass.begin(), strWalletPass.end(), '\0');
        munlock(&strWalletPass[0], strWalletPass.capacity());
    }
    else
        throw runtime_error(
            "walletpassphrase <passphrase> <timeout>\n"
            "Stores the wallet decryption key in memory for <timeout> seconds.");

    int nSleepTime=params[2].get_int();
    pWallet->TopUpKeyPool();
    WALLET_SLEEP *pSleep= new WALLET_SLEEP;
    pWallet->m_nWalletLockTime=nSleepTime+SolidTime_Get();
    pSleep->nTime=nSleepTime;
    pSleep->walletname=pWallet->GetWalletName();
    CreateThread(ThreadCleanWalletPassphrase, pSleep);

    return Value::null;
}

Value sc_walletpassphrasechange(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 3 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "walletpassphrasechange <wallet> <oldpassphrase> <newpassphrase>\n"
            "Changes the wallet passphrase from <oldpassphrase> to <newpassphrase>.");

    if (!pWallet->IsCrypted())  throw JSONRPCError(-15, "Error: running with an unencrypted wallet, but walletpassphrasechange was called.");

    string strOldWalletPass;
    strOldWalletPass.reserve(100);
    mlock(&strOldWalletPass[0], strOldWalletPass.capacity());
    strOldWalletPass = params[1].get_str();

    string strNewWalletPass;
    strNewWalletPass.reserve(100);
    mlock(&strNewWalletPass[0], strNewWalletPass.capacity());
    strNewWalletPass = params[2].get_str();

    if (strOldWalletPass.length() < 1 || strNewWalletPass.length() < 1)
        throw runtime_error(
            "walletpassphrasechange <oldpassphrase> <newpassphrase>\n"
            "Changes the wallet passphrase from <oldpassphrase> to <newpassphrase>.");

    if (!pWallet->ChangeWalletPassphrase(strOldWalletPass, strNewWalletPass))
    {
        fill(strOldWalletPass.begin(), strOldWalletPass.end(), '\0');
        fill(strNewWalletPass.begin(), strNewWalletPass.end(), '\0');
        munlock(&strOldWalletPass[0], strOldWalletPass.capacity());
        munlock(&strNewWalletPass[0], strNewWalletPass.capacity());
        throw JSONRPCError(-14, "Error: The wallet passphrase entered was incorrect.");
    }
    fill(strNewWalletPass.begin(), strNewWalletPass.end(), '\0');
    fill(strOldWalletPass.begin(), strOldWalletPass.end(), '\0');
    munlock(&strOldWalletPass[0], strOldWalletPass.capacity());
    munlock(&strNewWalletPass[0], strNewWalletPass.capacity());

    return Value::null;
}

Value sc_walletlock(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 1 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "walletlock <wallet>\n"
            "Removes the wallet encryption key from memory, locking the wallet.\n"
            "After calling this method, you will need to call walletpassphrase again\n"
            "before being able to call any methods which require the wallet to be unlocked.");
    if (!pWallet->IsCrypted())  throw JSONRPCError(-15, "Error: running with an unencrypted wallet, but walletlock was called.");
    pWallet->m_nWalletLockTime = 0;
    pWallet->Lock();
    return Value::null;
}

Value sc_encryptwallet(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "encryptwallet <wallet> <passphrase>\n"
            "Encrypts the wallet with <passphrase>.");

    if (pWallet->IsCrypted())   throw JSONRPCError(-15, "Error: running with an encrypted wallet, but encryptwallet was called.");

    string strWalletPass;
    strWalletPass.reserve(100);
    mlock(&strWalletPass[0], strWalletPass.capacity());
    strWalletPass = params[1].get_str();

    if (strWalletPass.length() < 1)
        throw runtime_error(
            "encryptwallet <passphrase>\n"
            "Encrypts the wallet with <passphrase>.");

    if (!pWallet->EncryptWallet(strWalletPass))
    {
        fill(strWalletPass.begin(), strWalletPass.end(), '\0');
        munlock(&strWalletPass[0], strWalletPass.capacity());
        throw JSONRPCError(-16, "Error: Failed to encrypt the wallet.");
    }
    fill(strWalletPass.begin(), strWalletPass.end(), '\0');
    munlock(&strWalletPass[0], strWalletPass.capacity());

    return Value::null;
}

static const string strMessageMagic = "SolidCoin Signed Message:\n";

Value sc_signmessage(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 3 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_signmessage <wallet> <solidcoinaddress> <message>\n"
            "Sign a message with the private key of an address");

    if (pWallet->IsCrypted())           throw JSONRPCError(-15, "Error: Please enter the wallet passphrase with walletpassphrase first.");

    string strAddress = params[1].get_str();
    string strMessage = params[2].get_str();

    CSolidCoinAddress addr(strAddress);
    if (!addr.IsValid())                throw JSONRPCError(-3, "Invalid address");

    CKey key;
    if (!pWallet->GetKey(addr, key))    throw JSONRPCError(-4, "Private key not available");

    CDataStream ss(SER_GETHASH);
    ss << strMessageMagic;
    ss << strMessage;

    vector<unsigned char> vchSig;
    if (!key.SignCompact(Hash(ss.begin(), ss.end()), vchSig))
        throw JSONRPCError(-5, "Sign failed");

    return EncodeBase64(&vchSig[0], vchSig.size());
}

Value sc_verifymessage(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    if(fHelp || params.size() != 3)
        throw runtime_error(
            "sc_verifymessage <solidcoinaddress> <signature> <message>\n"
            "Verify a signed message");

    string strAddress = params[0].get_str();
    string strSign = params[1].get_str();
    string strMessage = params[2].get_str();

    CSolidCoinAddress addr(strAddress);
    if (!addr.IsValid())
        throw JSONRPCError(-3, "Invalid address");

    bool fInvalid = false;
    vector<unsigned char> vchSig = DecodeBase64(strSign.c_str(), &fInvalid);

    if (fInvalid)   throw JSONRPCError(-5, "Malformed base64 encoding");

    CDataStream ss(SER_GETHASH);
    ss << strMessageMagic;
    ss << strMessage;

    CKey key;
    if (!key.SetCompactSignature(Hash(ss.begin(), ss.end()), vchSig))
        return false;

    return (key.GetAddress() == addr);
}

Value sc_validateaddress(const Array& params, bool fHelp)
{
    LOCK_WALLET_ACCESS();
    CWallet* pWallet=0;
    if(fHelp || params.size() != 2 || !(pWallet=Wallet_Get(params[0].get_str())))
        throw runtime_error(
            "sc_validateaddress <wallet> <solidcoinaddress>\n"
            "Return information about <solidcoinaddress>.");

    CSolidCoinAddress address(params[1].get_str());
    bool isValid = address.IsValid();

    Object ret;
    ret.push_back(Pair("isvalid", isValid));
    if (isValid)
    {
        // Call Hash160ToAddress() so we always return current ADDRESSVERSION version of the address:
        string currentAddress = address.ToString();
        ret.push_back(Pair("address", currentAddress));
        ret.push_back(Pair("ismine", (pWallet->HaveKey(address) > 0)));
        {
            if (pWallet->mapAddressBook.count(address))
                ret.push_back(Pair("account", pWallet->mapAddressBook[address]));
        }
    }
    return ret;
}

Value sc_gethash(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "sc_gethash [data]\n"
            "  \"data\" : block data, must be 128 bytes\n"
            "Returns Object named hash.");

    vector<unsigned char> vchData = ParseHex(params[0].get_str());
    if (vchData.size() != 128)  throw JSONRPCError(-8, "Invalid parameter");

    CBlock data;
    data.blk = *((BLOCK_DATA*)&vchData[0]);
    uint256 rethash=data.GetHash();

    Object result;
    result.push_back(Pair("hash",   HexNumStr(BEGIN(rethash), END(rethash))));
    return result;


}

Value sc_testwork(const Array& params, bool fHelp)
{
    if (fHelp || params.size()!=1)
        throw runtime_error(
            "sc_testwork [data]\n"
            "Tries to solve the block and returns true if it was successful.\n"
            "  \"share_valid\" : boolean indicating if the work meets the target for a share\n"
            "  \"block_valid\" : boolean indicating if the work meets the target for a block\n"
            "  \"block_hash\" : hash of the work in hex format\n"
            "  \"block_txid\" : transaction id hex string of the resulting generate for this work\n"
            "  \"block_power\" : boolean indicating if the block was a PowerBlock\n");


    vector<unsigned char> vchData = ParseHex(params[0].get_str());
    if (vchData.size()%128 != 0 )  throw JSONRPCError(-8, "Invalid parameter");

    Object result;
    Array works;

    int nCount=vchData.size()/128;
    for(int x=0;x<nCount;x++)
    {
        Object work;
        CBlock data;
        data.blk = *((BLOCK_DATA*)&vchData[x*128]);
        bool bBlockValid=false,bShareValid=false;
        bool bPowerBlock=false,bFoundBlock=false;
        uint256 txid;
        uint256 blockhash;

        {
            MUTEX_LOCK(cs_getwork);
            BOOST_FOREACH(CBlock* pBlock, g_NewBlockList)
            {
                int64 nTimeDiff =SolidTime_Get()-data.blk.nTime;
                if(nTimeDiff<0) nTimeDiff*=-1;
                if(data.blk.hashMerkleRoot==pBlock->blk.hashMerkleRoot)
                {
                    if(nTimeDiff<g_qShareTimeWindow)
                    {
                        pBlock->blk.nTime = data.blk.nTime;
                        pBlock->blk.nNonce1 = data.blk.nNonce1;
                        pBlock->blk.nNonce2 = data.blk.nNonce2;
                        pBlock->blk.nNonce3 = data.blk.nNonce3;
                        pBlock->blk.nNonce4 = data.blk.nNonce4;
                        bPowerBlock=pBlock->m_bPower;
                        {
                            MUTEX_LOCK(cs_main);
                            LOCK_WALLET_ACCESS();
                            bShareValid=bBlockValid=Block_CheckWork(pBlock, *Wallet_GetDefault(), g_MiningReserveKey,&blockhash);
                        }
                        txid=pBlock->vtx[0].GetHash();
                        //not a valid block, see if its a valid share
                        if(bBlockValid==false && g_MiningShareTargetInt>0 && pBlock->GetHash()<=g_MiningShareTargetHash)    bShareValid=true;
                        if(bShareValid) g_MiningTestWorkShares+=g_MiningShareTargetInt;
                        else            g_MiningTestWorkInvalids++;
                    }
                    bFoundBlock=true;
                    break;
                }
            }
        }
        if(bFoundBlock==false)
        {
            blockhash=data.GetHash();
            g_MiningTestWorkInvalids++;
        }

        work.push_back(Pair("share_valid",     bShareValid));
        work.push_back(Pair("block_valid",     bBlockValid));
        work.push_back(Pair("block_hash",       blockhash.ToString().c_str()));
        if(bBlockValid)
        {
            work.push_back(Pair("block_txid",     txid.ToString().c_str()));
            work.push_back(Pair("block_power",     bPowerBlock));
        }
        works.push_back(work);
    }
    result.push_back(Pair("work", works));
    //obj.push_back(Pair("generate",      (bool)false));
    //result.push_back(Pair("block_value",   HexNumStr(BEGIN(hashTarget_real), END(hashTarget_real))));
    //result.push_back(Pair("target_share",   HexNumStr(BEGIN(hashTarget_share), END(hashTarget_share))));
    //0011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233001122330011223300112233
    return result;

}
Value sc_getwork(const Array& params, bool fHelp)
{
    if (fHelp || params.size()>0)
        throw runtime_error(
            "sc_getwork\n"
            "Returns formatted hash data to work on:\n"
            "  \"data\" : block data\n"
            "  \"target_share\" : little endian hash target of a share\n"
            "  \"target_real\" : little endian hash target of the real diff\n");

    if(!fTestNet)
    {
        if (vNodes.empty())             throw JSONRPCError(-9, "SolidCoin is not connected!");
        if (Block_IsInitialDownload())   throw JSONRPCError(-10, "SolidCoin is downloading blocks...");
    }

    static unsigned int nTransactionsUpdatedLast=0;
    static CBlockIndex* pindexPrev=0;
    static int64 nStart=0;
    BLOCK_DATA blk;

    {
        MUTEX_LOCK(cs_getwork);
        CBlock *pBlock=0;
        if (pindexPrev != g_pBlockBestIndex || (nTransactionsUpdated != nTransactionsUpdatedLast && SolidTime_Get() - nStart > 20))
        {
            if (pindexPrev != g_pBlockBestIndex)
            {
                std::vector<CBlock*>::iterator iter;
                for(iter=g_NewBlockList.begin();iter!=g_NewBlockList.end();)
                {
                    if( (*iter)->blk.nBlockNum < g_qBlockBestHeight-3)
                    {
                        CBlock *pBlock=*iter;
                        iter=g_NewBlockList.erase(iter);
                        delete pBlock;

                    }
                    else
                    {
                        ++iter;
                    }
                }
            }
            nTransactionsUpdatedLast = nTransactionsUpdated;
            pindexPrev = g_pBlockBestIndex;
            extern string g_TrustedNodePublicKey;
            if((pindexPrev->blk.nBlockNum%2)==1 && g_TrustedNodePublicKey.length())
            {
                //Sleep( (40+(rand()%30)) * 1000);  //supernode testing
            }
            nStart = SolidTime_Get();
            {
                MUTEX_LOCK(cs_main);
                MUTEX_LOCK2(cs_mapTransactions);
                LOCK_WALLET_ACCESS();
                g_MiningReserveKey.SetWallet(Wallet_GetDefault());
                pBlock = Block_Create(Wallet_GetDefault(), g_MiningIDString,g_MiningReserveKey);
            }
            g_NewBlockList.push_back(pBlock);
        }
        else
        {
            pBlock = g_NewBlockList[g_NewBlockList.size()-1];  //get last item as its most recent
        }
        blk=pBlock->blk;
    }

    static uint64 nonce2=0;
    blk.nTime = SolidTime_Get();
    blk.nNonce1 = 0;
    blk.nNonce2 = nonce2++;
    blk.nNonce3 = 0;
    blk.nNonce4 = 0;

    //char pmidstate[32];char pdata[128];char phash1[64];
    Object result;
    result.push_back(Pair("data",     HexStr((unsigned char*)&blk,((unsigned char*)&blk)+128 )));
    uint256 hashTarget_real = CBigNum().SetCompact(blk.dwBits).getuint256();
    if(g_MiningShareTargetInt<=0)   result.push_back(Pair("target_share",   HexNumStr(BEGIN(hashTarget_real), END(hashTarget_real))));
    else                            result.push_back(Pair("target_share",   HexNumStr(BEGIN(g_MiningShareTargetHash), END(g_MiningShareTargetHash))));
    result.push_back(Pair("target_real",   HexNumStr(BEGIN(hashTarget_real), END(hashTarget_real))));
    //string str1=HexNumStr(BEGIN(hashTarget_real), END(hashTarget_real));
    //string str2=HexNumStr(BEGIN(g_MiningShareTargetHash), END(g_MiningShareTargetHash));
    //debugprintf(VERBOSE, "hash1:%s\nhash2:%s",str1.c_str(),str2.c_str());
    return result;
}

Value sc_getblockbynumber(const Array& params, bool fHelp)
{
    if (fHelp || params.size() != 1)
        throw runtime_error(
            "sc_getblockbynumber height\n"
            "Dumps the block existing at specified height");

    int64 height = params[0].get_int64();
    if (height > g_qBlockBestHeight)
        throw runtime_error(
            "sc_getblockbynumber height\n"
            "Dumps the block existing at specified height");

    string blkname = strprintf("blk%d", height);

    CBlockIndex* pindex=0;
    bool found = false;

    for (map<uint256, CBlockIndex*>::iterator mi = g_BlockIndexMap.begin();mi != g_BlockIndexMap.end(); ++mi)
    {
    	pindex = (*mi).second;
        if ((pindex->blk.nBlockNum == height) && (pindex->IsInMainChain()))
        {
            found = true;
            break;
        }
    }

    if (!found)
        throw runtime_error(
            "sc_getblockbynumber height\n"
            "Dumps the block existing at specified height");

    CBlock block;
    block.ReadFromDisk(pindex);
    block.BuildMerkleTree();
    char miner_id[13]={0};
    memcpy(miner_id,block.blk.miner_id,12);
    for(int x=0;x<12;x++) if(miner_id[x]<0) miner_id[x]=' ';

    Object obj;
    obj.push_back(Pair("hash", block.GetHash().ToString().c_str()));
    obj.push_back(Pair("version", block.blk.nVersion));
    obj.push_back(Pair("prev_block", block.blk.hashPrevBlock.ToString().c_str()));
    obj.push_back(Pair("mrkl_root", block.blk.hashMerkleRoot.ToString().c_str()));
    obj.push_back(Pair("time", (uint64_t)block.blk.nTime));
    obj.push_back(Pair("bits", (uint64_t)block.blk.dwBits));
    obj.push_back(Pair("blocknum", (uint64_t)block.blk.nBlockNum));
    obj.push_back(Pair("nonce1", (uint64_t)block.blk.nNonce1));
    obj.push_back(Pair("nonce2", (uint64_t)block.blk.nNonce2));
    obj.push_back(Pair("nonce3", (uint64_t)block.blk.nNonce3));
    obj.push_back(Pair("nonce4", (uint64_t)block.blk.nNonce4));
    obj.push_back(Pair("miner_id", miner_id));
    obj.push_back(Pair("miner_idhex", HexNumStr(block.blk.miner_id, block.blk.miner_id+12)));

    obj.push_back(Pair("n_tx", (int)block.vtx.size()));
    obj.push_back(Pair("size", (int)::GetSerializeSize(block, SER_NETWORK)));

    Array tx;
    for (int i = 0; i < block.vtx.size(); i++)
    {
    	Object txobj;
        txobj.push_back(Pair("hash", block.vtx[i].GetHash().ToString().c_str()));
        txobj.push_back(Pair("version", block.vtx[i].nVersion));
        txobj.push_back(Pair("lock_time", (uint64_t)block.vtx[i].nLockTime));
        txobj.push_back(Pair("size",(int)::GetSerializeSize(block.vtx[i], SER_NETWORK)));

        Array tx_vin;
        for (int j = 0; j < block.vtx[i].vin.size(); j++)
        {
            Object vino;

            Object vino_outpt;

            vino_outpt.push_back(Pair("hash",block.vtx[i].vin[j].prevout.hash.ToString().c_str()));
            vino_outpt.push_back(Pair("n", (uint64_t)block.vtx[i].vin[j].prevout.n));

            vino.push_back(Pair("prev_out", vino_outpt));

            if (block.vtx[i].vin[j].prevout.IsNull())
                vino.push_back(Pair("coinbase", HexStr(block.vtx[i].vin[j].scriptSig.begin(),block.vtx[i].vin[j].scriptSig.end(), false).c_str()));
            else
                vino.push_back(Pair("scriptSig",block.vtx[i].vin[j].scriptSig.ToString().c_str()));

            if (block.vtx[i].vin[j].nSequence != UINT_MAX)
                vino.push_back(Pair("sequence", (uint64_t)block.vtx[i].vin[j].nSequence));

            tx_vin.push_back(vino);
        }

        Array tx_vout;
        for (int j = 0; j < block.vtx[i].vout.size(); j++) {
            Object vouto;

            vouto.push_back(Pair("value",
                (double)block.vtx[i].vout[j].nValue / (double)COIN));
            vouto.push_back(Pair("scriptPubKey",
            block.vtx[i].vout[j].scriptPubKey.ToString().c_str()));

            tx_vout.push_back(vouto);
        }

        txobj.push_back(Pair("in", tx_vin));
        txobj.push_back(Pair("out", tx_vout));

        tx.push_back(txobj);
    }

    obj.push_back(Pair("tx", tx));

    Array mrkl;
    for (int i = 0; i < block.vMerkleTree.size(); i++)
    	mrkl.push_back(block.vMerkleTree[i].ToString().c_str());

    obj.push_back(Pair("mrkl_tree", mrkl));

    return obj;
}

// Call Table

pair<string, rpcfn_type> pCallTable[] =
{
    make_pair("help",                   &help),
    make_pair("stop",                   &stop),
    make_pair("getblocknumber",         &getblocknumber),
    make_pair("getdifficulty",          &getdifficulty),

    make_pair("sc_backupwallet",            &sc_backupwallet),
    make_pair("sc_encryptwallet",           &sc_encryptwallet),
    make_pair("sc_getaccountaddress",       &sc_getaccountaddress),
    make_pair("sc_getaccount",              &sc_getaccount),
    make_pair("sc_getaddressesbyaccount",   &sc_getaddressesbyaccount),
    make_pair("sc_getblockbynumber",        &sc_getblockbynumber),
    make_pair("sc_getnewaddress",           &sc_getnewaddress),
    make_pair("sc_getreceivedbyaccount",    &sc_getreceivedbyaccount),
    make_pair("sc_getwork",                 &sc_getwork),
    make_pair("sc_gethash",                 &sc_gethash),
    make_pair("sc_getinfo",                 &sc_getinfo),
    make_pair("sc_getwalletinfo",           &sc_getwalletinfo),
    make_pair("sc_gettxfee",                &sc_gettxfee),
    make_pair("sc_gettransaction",          &sc_gettransaction),
    make_pair("sc_getreceivedbyaddress",    &sc_getreceivedbyaddress),
    make_pair("sc_getbalance",              &sc_getbalance),
    make_pair("sc_getmining",               &sc_getmining),
    make_pair("sc_keypoolrefill",           &sc_keypoolrefill),
    make_pair("sc_listaccounts",            &sc_listaccounts),
    make_pair("sc_listreceivedbyaccount",   &sc_listreceivedbyaccount),
    make_pair("sc_listreceivedbyaddress",   &sc_listreceivedbyaddress),
    make_pair("sc_listtransactions",        &sc_listtransactions),
    make_pair("sc_move",                    &sc_movecmd),
    make_pair("sc_sendtoaddress",           &sc_sendtoaddress),
    make_pair("sc_sendfrom",                &sc_sendfrom),
    make_pair("sc_sendmany",                &sc_sendmany),
    make_pair("sc_setaccount",              &sc_setaccount),
    make_pair("sc_setmining",               &sc_setmining),
    make_pair("sc_testwork",                &sc_testwork),
    make_pair("sc_validateaddress",         &sc_validateaddress),
    make_pair("sc_walletpassphrase",        &sc_walletpassphrase),
    make_pair("sc_walletpassphrasechange",  &sc_walletpassphrasechange),
    make_pair("sc_walletlock",              &sc_walletlock),
    make_pair("sc_signmessage",              &sc_signmessage),
    make_pair("sc_verifymessage",              &sc_verifymessage),

};
map<string, rpcfn_type> mapCallTable(pCallTable, pCallTable + sizeof(pCallTable)/sizeof(pCallTable[0]));

string pAllowInSafeMode[] =
{
    "help",
    "stop",
    "getblockcount",
    "getblocknumber",
    "getconnectioncount",
    "getdifficulty",
    "sc_getmining",
    "sc_setmining",
    "getinfo",
    "getnewaddress",
    "getaccountaddress",
    "setlabel", // deprecated
    "getaccount",
    "getlabel", // deprecated
    "getaddressesbyaccount",
    "getaddressesbylabel", // deprecated
    "backupwallet",
    "keypoolrefill",
    "walletpassphrase",
    "walletlock",
    "validateaddress",
    "sc_getnewaddress",
    "sc_getwork",
    "sc_getaccount",
    "sc_testwork",
    "sc_gethash",
    "sc_getinfo",
};
set<string> setAllowInSafeMode(pAllowInSafeMode, pAllowInSafeMode + sizeof(pAllowInSafeMode)/sizeof(pAllowInSafeMode[0]));

