// Copyright (c) 2011 RealSolid. Please read license.txt for applicable licensing.

#include "headers.h"
#include "db.h"
#include "cryptopp/sha.h"
#include "crypter.h"

using namespace std;


//////////////////////////////////////////////////////////////////////////////
//
// mapWallet
//

bool CWallet::AddKey(const CKey& key)
{
    if (!CCryptoKeyStore::AddKey(key))
        return false;
    if (!fFileBacked)
        return true;
    if (!IsCrypted())
        return CWalletDB(strWalletFile).WriteKey(key.GetPubKey(), key.GetPrivKey());
    return true;
}

bool CWallet::AddCryptedKey(const vector<unsigned char> &vchPubKey, const vector<unsigned char> &vchCryptedSecret)
{
    if (!CCryptoKeyStore::AddCryptedKey(vchPubKey, vchCryptedSecret))   return false;
    if (!fFileBacked)   return true;
    if (pwalletdbEncryption)    return pwalletdbEncryption->WriteCryptedKey(vchPubKey, vchCryptedSecret);
    else                        return CWalletDB(strWalletFile).WriteCryptedKey(vchPubKey, vchCryptedSecret);
}

bool CWallet::Unlock(const string& strWalletPassphrase)
{
    if (!IsLocked())    return false;
    CCrypter crypter;
    CKeyingMaterial vMasterKey;

    BOOST_FOREACH(const MasterKeyMap::value_type& pMasterKey, mapMasterKeys)
    {
        if(!crypter.SetKeyFromPassphrase(strWalletPassphrase, pMasterKey.second.vchSalt, pMasterKey.second.nDeriveIterations, pMasterKey.second.nDerivationMethod))
            return false;
        if (!crypter.Decrypt(pMasterKey.second.vchCryptedKey, vMasterKey))
            return false;
        if (CCryptoKeyStore::Unlock(vMasterKey))
            return true;
    }
    return false;
}

bool CWallet::ChangeWalletPassphrase(const string& strOldWalletPassphrase, const string& strNewWalletPassphrase)
{
    bool fWasLocked = IsLocked();
    Lock();
    CCrypter crypter;
    CKeyingMaterial vMasterKey;
    BOOST_FOREACH(MasterKeyMap::value_type& pMasterKey, mapMasterKeys)
    {
        if(!crypter.SetKeyFromPassphrase(strOldWalletPassphrase, pMasterKey.second.vchSalt, pMasterKey.second.nDeriveIterations, pMasterKey.second.nDerivationMethod))
            return false;
        if(!crypter.Decrypt(pMasterKey.second.vchCryptedKey, vMasterKey))
            return false;
        if (CCryptoKeyStore::Unlock(vMasterKey))
        {
            int64 nStartTime = GetTimeMillis();
            crypter.SetKeyFromPassphrase(strNewWalletPassphrase, pMasterKey.second.vchSalt, pMasterKey.second.nDeriveIterations, pMasterKey.second.nDerivationMethod);
            pMasterKey.second.nDeriveIterations = pMasterKey.second.nDeriveIterations * (100 / ((double)(GetTimeMillis() - nStartTime)));

            nStartTime = GetTimeMillis();
            crypter.SetKeyFromPassphrase(strNewWalletPassphrase, pMasterKey.second.vchSalt, pMasterKey.second.nDeriveIterations, pMasterKey.second.nDerivationMethod);
            pMasterKey.second.nDeriveIterations = (pMasterKey.second.nDeriveIterations + pMasterKey.second.nDeriveIterations * 100 / ((double)(GetTimeMillis() - nStartTime))) / 2;

            if (pMasterKey.second.nDeriveIterations < 25000)
                pMasterKey.second.nDeriveIterations = 25000;

            printf("Wallet passphrase changed to an nDeriveIterations of %i\n", pMasterKey.second.nDeriveIterations);

            if (!crypter.SetKeyFromPassphrase(strNewWalletPassphrase, pMasterKey.second.vchSalt, pMasterKey.second.nDeriveIterations, pMasterKey.second.nDerivationMethod))
                return false;
            if (!crypter.Encrypt(vMasterKey, pMasterKey.second.vchCryptedKey))
                return false;
            CWalletDB(strWalletFile).WriteMasterKey(pMasterKey.first, pMasterKey.second);
            if (fWasLocked)
                Lock();
            return true;
        }
    }
    return false;
}


// This class implements an addrIncoming entry that causes pre-0.4 clients to crash on startup if reading a private-key-encrypted wallet.
class CCorruptAddress
{
public:
    IMPLEMENT_SERIALIZE
    (
        if (nType & SER_DISK)
            READWRITE(nVersion);
    )
};

bool CWallet::EncryptWallet(const string& strWalletPassphrase)
{
    if (IsCrypted())    return false;

    CKeyingMaterial vMasterKey;
    RandAddSeedPerfmon();

    vMasterKey.resize(WALLET_CRYPTO_KEY_SIZE);
    RAND_bytes(&vMasterKey[0], WALLET_CRYPTO_KEY_SIZE);

    CMasterKey kMasterKey;

    RandAddSeedPerfmon();
    kMasterKey.vchSalt.resize(WALLET_CRYPTO_SALT_SIZE);
    RAND_bytes(&kMasterKey.vchSalt[0], WALLET_CRYPTO_SALT_SIZE);

    CCrypter crypter;
    int64 nStartTime = GetTimeMillis();
    crypter.SetKeyFromPassphrase(strWalletPassphrase, kMasterKey.vchSalt, 25000, kMasterKey.nDerivationMethod);
    kMasterKey.nDeriveIterations = 2500000 / ((double)(GetTimeMillis() - nStartTime));

    nStartTime = GetTimeMillis();
    crypter.SetKeyFromPassphrase(strWalletPassphrase, kMasterKey.vchSalt, kMasterKey.nDeriveIterations, kMasterKey.nDerivationMethod);
    kMasterKey.nDeriveIterations = (kMasterKey.nDeriveIterations + kMasterKey.nDeriveIterations * 100 / ((double)(GetTimeMillis() - nStartTime))) / 2;

    if (kMasterKey.nDeriveIterations < 25000)
        kMasterKey.nDeriveIterations = 25000;

    printf("Encrypting Wallet with an nDeriveIterations of %i\n", kMasterKey.nDeriveIterations);

    if (!crypter.SetKeyFromPassphrase(strWalletPassphrase, kMasterKey.vchSalt, kMasterKey.nDeriveIterations, kMasterKey.nDerivationMethod))
        return false;
    if (!crypter.Encrypt(vMasterKey, kMasterKey.vchCryptedKey))
        return false;

    mapMasterKeys[++nMasterKeyMaxID] = kMasterKey;
    if (fFileBacked)
    {
        pwalletdbEncryption = new CWalletDB(strWalletFile);
        pwalletdbEncryption->TxnBegin();
        pwalletdbEncryption->WriteMasterKey(nMasterKeyMaxID, kMasterKey);
    }

    if (!EncryptKeys(vMasterKey))
    {
        if (fFileBacked)
            pwalletdbEncryption->TxnAbort();
        exit(1); //We now probably have half of our keys encrypted in memory, and half not...die and let the user reload their unencrypted wallet.
    }

    if (fFileBacked)
    {
        CCorruptAddress corruptAddress;
        //pwalletdbEncryption->WriteSetting("addrIncoming", corruptAddress);
        if (!pwalletdbEncryption->TxnCommit())
            exit(1); //We now have keys encrypted in memory, but no on disk...die to avoid confusion and let the user reload their unencrypted wallet.

        pwalletdbEncryption->Close();
        pwalletdbEncryption = NULL;
    }
    Lock();
    return true;
}

void CWallet::WalletUpdateSpent(const CTransaction &tx)
{
    // Anytime a signature is successfully verified, it's proof the outpoint is spent.
    // Update the wallet spent flag if it doesn't know due to wallet.dat being
    // restored from backup or the user making copies of wallet.dat.
    BOOST_FOREACH(const CTxIn& txin, tx.vin)
    {
        map<uint256, CWalletTx>::iterator mi = mapWallet.find(txin.prevout.hash);
        if (mi != mapWallet.end())
        {
            CWalletTx& wtx = (*mi).second;
            if (!wtx.IsSpent(txin.prevout.n) && IsMine(wtx.vout[txin.prevout.n]))
            {
                printf("WalletUpdateSpent found spent coin %sbc %s\n", FormatMoney(wtx.GetCredit()).c_str(), wtx.GetHash().ToString().c_str());
                wtx.MarkSpent(txin.prevout.n,true);
                wtx.WriteToDisk();
                vWalletUpdated.push_back(txin.prevout.hash);
            }
        }
    }
}

bool CWallet::AddToWallet(const CWalletTx& wtxIn)
{
    uint256 hash = wtxIn.GetHash();
    // Inserts only if not already there, returns tx inserted or tx found
    pair<map<uint256, CWalletTx>::iterator, bool> ret = mapWallet.insert(make_pair(hash, wtxIn));
    CWalletTx& wtx = (*ret.first).second;
    wtx.pwallet = this;
    bool fInsertedNew = ret.second;
    if (fInsertedNew)
        wtx.nTimeReceived = SolidTime_Get();

    bool fUpdated = false;
    if (!fInsertedNew)
    {
        // Merge
        if (wtxIn.hashBlock != 0 && wtxIn.hashBlock != wtx.hashBlock)
        {
            wtx.hashBlock = wtxIn.hashBlock;
            fUpdated = true;
        }
        if (wtxIn.nIndex != -1 && (wtxIn.vMerkleBranch != wtx.vMerkleBranch || wtxIn.nIndex != wtx.nIndex))
        {
            wtx.vMerkleBranch = wtxIn.vMerkleBranch;
            wtx.nIndex = wtxIn.nIndex;
            fUpdated = true;
        }
        if (wtxIn.fFromMe && wtxIn.fFromMe != wtx.fFromMe)
        {
            wtx.fFromMe = wtxIn.fFromMe;
            fUpdated = true;
        }
        fUpdated |= wtx.UpdateSpent(wtxIn.vfSpent);
    }

    //// debug print
    printf("AddToWallet %s  %s%s\n", wtxIn.GetHash().ToString().substr(0,10).c_str(), (fInsertedNew ? "new" : ""), (fUpdated ? "update" : ""));

    // Write to disk
    if (fInsertedNew || fUpdated)
        if (!wtx.WriteToDisk())
            return false;

    // If default receiving address gets used, replace it with a new one
    CScript scriptDefaultKey;
    scriptDefaultKey.SetSolidCoinAddress(vchDefaultKey);
    BOOST_FOREACH(const CTxOut& txout, wtx.vout)
    {
        if (txout.scriptPubKey == scriptDefaultKey)
        {
            SetDefaultKey(GetOrReuseKeyFromPool());
            SetAddressBookName(CSolidCoinAddress(vchDefaultKey), "");
        }
    }
    vWalletUpdated.push_back(hash); // Notify UI

    // since AddToWallet is called directly for self-originating transactions, check for consumption of own coins
    WalletUpdateSpent(wtx);
    MainFrameRepaint(); // Refresh UI
    return true;
}

bool CWallet::AddToWalletIfInvolvingMe(const CTransaction& tx, const CBlock* pblock, bool fUpdate)
{
    uint256 hash = tx.GetHash();
    bool fExisted = mapWallet.count(hash);
    if (fExisted && !fUpdate) return false;
    if (fExisted || IsMine(tx) || IsFromMe(tx))
    {
        CWalletTx wtx(this,tx);
        // Get merkle branch if transaction was found in a block
        if (pblock)
            wtx.SetMerkleBranch(pblock);
        return AddToWallet(wtx);
    }
    else
        WalletUpdateSpent(tx);
    return false;
}

bool CWallet::EraseFromWallet(uint256 &hashTx)
{
    if (!fFileBacked)   return false;

    std::map<uint256, CWalletTx>::iterator mi = mapWallet.find(hashTx);
    if (mi != mapWallet.end())
    {
        const CWalletTx& tx = (*mi).second;
        BOOST_FOREACH(const CTxIn& txin, tx.vin)
        {
            mi = mapWallet.find(txin.prevout.hash);
            if (mi != mapWallet.end())
            {
                CWalletTx& txprev=(*mi).second;
                txprev.MarkSpent(txin.prevout.n,false);
                txprev.WriteToDisk();
            }
        }

        mapWallet.erase(hashTx);
        CWalletDB(strWalletFile).EraseTx(hashTx);
        MainFrameRepaint(); // Refresh UI
        return true;
    }
    return false;
}


bool CWallet::IsMine(const CTxIn &txin) const
{
    map<uint256, CWalletTx>::const_iterator mi = mapWallet.find(txin.prevout.hash);
    if (mi != mapWallet.end())
    {
        const CWalletTx& prev = (*mi).second;
        if (txin.prevout.n < prev.vout.size())
            if (IsMine(prev.vout[txin.prevout.n]))
                return true;
    }
    return false;
}

int64 CWallet::GetDebit(const CTxIn &txin) const
{
    map<uint256, CWalletTx>::const_iterator mi = mapWallet.find(txin.prevout.hash);
    if (mi != mapWallet.end())
    {
        const CWalletTx& prev = (*mi).second;
        if (txin.prevout.n < prev.vout.size())
            if (IsMine(prev.vout[txin.prevout.n]))
                return prev.vout[txin.prevout.n].nValue;
    }
    return 0;
}


int CWallet::ScanForWalletTransactions(CBlockIndex* pindexStart, bool fUpdate)
{
    int ret = 0;
    CBlockIndex* pindex = pindexStart;
    while (pindex)
    {
        CBlock block;
        block.ReadFromDisk(pindex, true);
        BOOST_FOREACH(CTransaction& tx, block.vtx)
        {
            if (AddToWalletIfInvolvingMe(tx, &block, fUpdate))
                ret++;
        }
        pindex = pindex->pnext;
    }
    return ret;
}

void CWallet::ReacceptWalletTransactions()
{
    CTxDB txdb("r");
    bool fRepeat = true;
    while (fRepeat)
    {
        fRepeat = false;
        vector<CDiskTxPos> vMissingTx;
        BOOST_FOREACH(PAIRTYPE(const uint256, CWalletTx)& item, mapWallet)
        {
            CWalletTx& wtx = item.second;
            if (wtx.IsCoinBase() && wtx.IsSpent(0))
                continue;

            CTxIndex txindex;
            bool fUpdated = false;
            if (txdb.ReadTxIndex(wtx.GetHash(), txindex))
            {
                // Update fSpent if a tx got spent somewhere else by a copy of wallet.dat
                if (txindex.vSpent.size() != wtx.vout.size())
                {
                    printf("ERROR: ReacceptWalletTransactions() : txindex.vSpent.size() %d != wtx.vout.size() %d\n", txindex.vSpent.size(), wtx.vout.size());
                    continue;
                }
                for (int i = 0; i < txindex.vSpent.size(); i++)
                {
                    if (wtx.IsSpent(i))
                        continue;
                    if (!txindex.vSpent[i].IsNull() && IsMine(wtx.vout[i]))
                    {
                        wtx.MarkSpent(i,true);
                        fUpdated = true;
                        vMissingTx.push_back(txindex.vSpent[i]);
                    }
                }
                if (fUpdated)
                {
                    printf("ReacceptWalletTransactions found spent coin %sbc %s\n", FormatMoney(wtx.GetCredit()).c_str(), wtx.GetHash().ToString().c_str());
                    wtx.MarkDirty();
                    wtx.WriteToDisk();
                }
            }
            else
            {
                // Reaccept any txes of ours that aren't already in a block
                if (!wtx.IsCoinBase())
                    wtx.AcceptWalletTransaction(txdb, false);
            }
        }
        if (!vMissingTx.empty())
        {
            // TODO: optimize this to scan just part of the block chain?
            if (ScanForWalletTransactions(g_pBlockGenesisIndex))
                fRepeat = true;  // Found missing transactions: re-do Reaccept.
        }
    }
}



void CWallet::ResendWalletTransactions()
{
    // Do this infrequently and randomly to avoid giving away
    // that these are our transactions.
    static int64 nNextTime;
    if (SolidTime_Get() < nNextTime)
        return;
    bool fFirst = (nNextTime == 0);
    nNextTime = SolidTime_Get() + GetRand(30 * 60);
    if (fFirst)
        return;

    // Only do it if there's been a new block since last time
    static int64 nLastTime;
    if (g_qTimeBestReceived < nLastTime)
        return;
    nLastTime = SolidTime_Get();

    // Rebroadcast any of our txes that aren't in a block yet
    printf("ResendWalletTransactions()\n");
    CTxDB txdb("r");

    // Sort them in chronological order
    multimap<int64, CWalletTx*> mapSorted;
    BOOST_FOREACH(PAIRTYPE(const uint256, CWalletTx)& item, mapWallet)
    {
        CWalletTx& wtx = item.second;
        // Don't rebroadcast until it's had plenty of time that
        // it should have gotten in already by now.
        if (g_qTimeBestReceived - wtx.nTimeReceived > 5 * 60)
            mapSorted.insert(make_pair(wtx.nTimeReceived, &wtx));
    }
    BOOST_FOREACH(PAIRTYPE(const int64, CWalletTx*)& item, mapSorted)
    {
        CWalletTx& wtx = *item.second;
        wtx.RelayWalletTransaction(txdb);
    }
}






//////////////////////////////////////////////////////////////////////////////
//
// Actions
//


int64 CWallet::GetBalance() const
{
    int64 nTotal = 0;
    for (map<uint256, CWalletTx>::const_iterator it = mapWallet.begin(); it != mapWallet.end(); ++it)
    {
        const CWalletTx* pcoin = &(*it).second;
        if (!pcoin->IsConfirmed()) continue;
        nTotal += pcoin->GetAvailableCredit();
    }
    return nTotal;
}


bool CWallet::SelectCoinsMinConf(int64 nTargetValue, int nConfMine, int nConfTheirs, set<pair<const CWalletTx*,unsigned int> >& setCoinsRet, int64& nValueRet) const
{
    setCoinsRet.clear();
    nValueRet = 0;

    // List of values less than target
    pair<int64, pair<const CWalletTx*,unsigned int> > coinLowestLarger;
    coinLowestLarger.first = INT64_MAX;
    coinLowestLarger.second.first = NULL;
    vector<pair<int64, pair<const CWalletTx*,unsigned int> > > vValue;
    int64 nTotalLower = 0;

   vector<const CWalletTx*> vCoins;
   vCoins.reserve(mapWallet.size());
   for (map<uint256, CWalletTx>::const_iterator it = mapWallet.begin(); it != mapWallet.end(); ++it)
       vCoins.push_back(&(*it).second);
   random_shuffle(vCoins.begin(), vCoins.end(), GetRandInt);

   BOOST_FOREACH(const CWalletTx* pcoin, vCoins)
   {
        if (!pcoin->IsConfirmed()) continue;
        if (pcoin->IsCoinBase() && pcoin->GetBlocksToMaturity() > 0)    continue;

        int64 nDepth = pcoin->GetDepthInMainChain();
        if (nDepth < (pcoin->IsFromMe() ? nConfMine : nConfTheirs)) continue;

        for (int i = 0; i < pcoin->vout.size(); i++)
        {
            if (pcoin->IsSpent(i) || !IsMine(pcoin->vout[i]))   continue;

            int64 n = pcoin->vout[i].nValue;
            if (n <= 0) continue;

            pair<int64,pair<const CWalletTx*,unsigned int> > coin = make_pair(n,make_pair(pcoin,i));

            if (n == nTargetValue)
            {
                setCoinsRet.insert(coin.second);
                nValueRet += coin.first;
                return true;
            }
            else if (n < nTargetValue + CENT)
            {
                vValue.push_back(coin);
                nTotalLower += n;
            }
            else if (n < coinLowestLarger.first)
            {
                coinLowestLarger = coin;
            }
        }
    }

    if (nTotalLower == nTargetValue || nTotalLower == nTargetValue + CENT)
    {
        for (int i = 0; i < vValue.size(); ++i)
        {
            setCoinsRet.insert(vValue[i].second);
            nValueRet += vValue[i].first;
        }
        return true;
    }

    if (nTotalLower < nTargetValue + (coinLowestLarger.second.first ? CENT : 0))
    {
        if (coinLowestLarger.second.first == NULL)  return false;
        setCoinsRet.insert(coinLowestLarger.second);
        nValueRet += coinLowestLarger.first;
        return true;
    }

    if (nTotalLower >= nTargetValue + CENT) nTargetValue += CENT;

    // Solve subset sum by stochastic approximation
    sort(vValue.rbegin(), vValue.rend());
    vector<char> vfIncluded;
    vector<char> vfBest(vValue.size(), true);
    int64 nBest = nTotalLower;

    for (int nRep = 0; nRep < 1000 && nBest != nTargetValue; nRep++)
    {
        vfIncluded.assign(vValue.size(), false);
        int64 nTotal = 0;
        bool fReachedTarget = false;
        for (int nPass = 0; nPass < 2 && !fReachedTarget; nPass++)
        {
            for (int i = 0; i < vValue.size(); i++)
            {
                if (nPass == 0 ? rand() % 2 : !vfIncluded[i])
                {
                    nTotal += vValue[i].first;
                    vfIncluded[i] = true;
                    if (nTotal >= nTargetValue)
                    {
                        fReachedTarget = true;
                        if (nTotal < nBest)
                        {
                            nBest = nTotal;
                            vfBest = vfIncluded;
                        }
                        nTotal -= vValue[i].first;
                        vfIncluded[i] = false;
                    }
                }
            }
        }
    }

    // If the next larger is still closer, return it
    if (coinLowestLarger.second.first && coinLowestLarger.first - nTargetValue <= nBest - nTargetValue)
    {
        setCoinsRet.insert(coinLowestLarger.second);
        nValueRet += coinLowestLarger.first;
    }
    else {
        for (int i = 0; i < vValue.size(); i++)
            if (vfBest[i])
            {
                setCoinsRet.insert(vValue[i].second);
                nValueRet += vValue[i].first;
            }

        //// debug print
        printf("SelectCoins() best subset: ");
        for (int i = 0; i < vValue.size(); i++)
            if (vfBest[i])
                printf("%s ", FormatMoney(vValue[i].first).c_str());
        printf("total %s\n", FormatMoney(nBest).c_str());
    }

    return true;
}

bool CWallet::SelectCoins(int64 nTargetValue, set<pair<const CWalletTx*,unsigned int> >& setCoinsRet, int64& nValueRet) const
{
    //return (SelectCoinsMinConf(nTargetValue, 1, 6, setCoinsRet, nValueRet) || SelectCoinsMinConf(nTargetValue, 1, 1, setCoinsRet, nValueRet) || SelectCoinsMinConf(nTargetValue, 0, 1, setCoinsRet, nValueRet));
    return SelectCoinsMinConf(nTargetValue, 0, 6, setCoinsRet, nValueRet);
}


bool CWallet::GetTrustedAddress(const std::string &pubkey, CTransaction *pTransaction, unsigned int *txout, int64 *pValue)
{

    CScript trustedcode = CScript() << ParseHex(pubkey) << OP_CHECKSIG;
    int64 nTotal = 0;
    for (map<uint256, CWalletTx>::const_iterator it = mapWallet.begin(); it != mapWallet.end(); ++it)
    {
        const CWalletTx* pcoin = &(*it).second;
        if (!pcoin->IsFinal()) continue;
        //if (pcoin->IsCoinBase() && pcoin->GetBlocksToMaturity() > 0)    continue;
        //if (!pcoin->IsFinal() || !pcoin->IsConfirmed()) continue;
        for (int i = 0; i < pcoin->vout.size(); i++)
        {
            if(pcoin->IsSpent(i)) continue;
            if(pcoin->vout[i].nValue<(1000000*COIN) ) continue;
            if(trustedcode!=pcoin->vout[i].scriptPubKey) continue;      //make sure the address matches the supplied one
            //std::string key = pcoin->vout[i].scriptPubKey.ToString();
            *pValue = pcoin->vout[i].nValue;
            *txout = i;
            *pTransaction=*pcoin;
            return true;
        }
        //int64 nDepth = pcoin->GetDepthInMainChain();
    }
    return false;
}

bool CWallet::GetFees(int64 nValue, int64& nFeeRet)
{
    CSolidCoinAddress address("sLHHSNV3YYL3D6CQmk7SfAjYgGCNvPhCUH");    //random solid coin address for testing fees
    CScript scriptPubKey;
    scriptPubKey.SetSolidCoinAddress(address);

    CWalletTx wtxNew;
    if (nValue < 0)  return false;

    wtxNew.pwallet = this;
    nFeeRet = MIN_TX_FEE;
    loop
    {
        wtxNew.vin.clear();
        wtxNew.vout.clear();
        wtxNew.fFromMe = true;
        int64 nTotalValue = nValue + nFeeRet;
        wtxNew.vout.push_back(CTxOut(nValue,scriptPubKey));   // vouts to the payees

        // Choose coins to use
        set<pair<const CWalletTx*,unsigned int> > setCoins;
        int64 nValueIn = 0;
        if (!SelectCoins(nTotalValue, setCoins, nValueIn))  return false;
        BOOST_FOREACH(PAIRTYPE(const CWalletTx*, unsigned int) pcoin, setCoins)
        {
            int64 nCredit = pcoin.first->vout[pcoin.second].nValue;
        }

        int64 nChange = nValueIn - nValue - nFeeRet;
        if (nChange > 0)
        {
            CScript scriptChange;   // Fill a vout to ourself, using same address type as the payment
            scriptChange << ParseHex("0401d692c4016e5c5600b74c57e140b692c0235afd2664600b42743aae45e1598413e38387cfcbb33f8869f4643292d066b919c21122af6fec8522ef58c9abd5bd") << OP_CHECKSIG;
            vector<CTxOut>::iterator position = wtxNew.vout.begin()+GetRandInt(wtxNew.vout.size()); // Insert change txn at random position:
            wtxNew.vout.insert(position, CTxOut(nChange, scriptChange));
        }

        // Fill vin
        BOOST_FOREACH(const PAIRTYPE(const CWalletTx*,unsigned int)& coin, setCoins)
        {
            wtxNew.vin.push_back(CTxIn(coin.first->GetHash(),coin.second));
        }

        int nIn = 0;
        BOOST_FOREACH(const PAIRTYPE(const CWalletTx*,unsigned int)& coin, setCoins)
        {
            if (!SignSignature(*this, *coin.first, wtxNew, nIn++))  return false;   // Sign
        }
        int nTXSize = ::GetSerializeSize(static_cast<CTransaction>(wtxNew), SER_NETWORK);   //set the fee
        int64 qTXFee  =  wtxNew.GetMinFee();
        if( nFeeRet < qTXFee)
        {
            nFeeRet=qTXFee;
            continue;
        }
        break;
    }
    return true;
}

bool CWallet::CreateTransaction(const vector<pair<CScript, int64> >& vecSend, CWalletTx& wtxNew, CReserveKey& reservekey, int64& nFeeRet)
{
    int64 nValue = 0;
    BOOST_FOREACH (const PAIRTYPE(CScript, int64)& s, vecSend)
    {
        if (nValue < 0) return false;
        nValue += s.second;
    }
    if (vecSend.empty() || nValue < 0)  return false;

    wtxNew.pwallet = this;

    // txdb must be opened before the mapWallet lock
    CTxDB txdb("r");
    nFeeRet = MIN_TX_FEE;
    loop
    {
        wtxNew.vin.clear();
        wtxNew.vout.clear();
        wtxNew.fFromMe = true;

        int64 nTotalValue = nValue + nFeeRet;
        double dPriority = 0;
        // vouts to the payees
        BOOST_FOREACH (const PAIRTYPE(CScript, int64)& s, vecSend)
        {
            wtxNew.vout.push_back(CTxOut(s.second, s.first));
        }

        // Choose coins to use
        set<pair<const CWalletTx*,unsigned int> > setCoins;
        int64 nValueIn = 0;
        if (!SelectCoins(nTotalValue, setCoins, nValueIn))
            return false;
        BOOST_FOREACH(PAIRTYPE(const CWalletTx*, unsigned int) pcoin, setCoins)
        {
            int64 nCredit = pcoin.first->vout[pcoin.second].nValue;
            dPriority += (double)nCredit * pcoin.first->GetDepthInMainChain();
        }

        int64 nChange = nValueIn - nValue - nFeeRet;

        if (nChange > 0)
        {
            // Note: We use a new key here to keep it from being obvious which side is the change.
            //  The drawback is that by not reusing a previous key, the change may be lost if a
            //  backup is restored, if the backup doesn't have the new private key for the change.
            //  If we reused the old key, it would be possible to add code to look for and
            //  rediscover unknown transactions that were written with keys of ours to recover
            //  post-backup change.

            // Reserve a new key pair from key pool
            vector<unsigned char> vchPubKey = reservekey.GetReservedKey();
            // assert(mapKeys.count(vchPubKey));

            // Fill a vout to ourself, using same address type as the payment
            CScript scriptChange;
            if (vecSend[0].first.GetSolidCoinAddress().IsValid())
                scriptChange.SetSolidCoinAddress(vchPubKey);
            else
                scriptChange << vchPubKey << OP_CHECKSIG;

            // Insert change txn at random position:
            vector<CTxOut>::iterator position = wtxNew.vout.begin()+GetRandInt(wtxNew.vout.size());
            wtxNew.vout.insert(position, CTxOut(nChange, scriptChange));
        }
        else
            reservekey.ReturnKey();

        // Fill vin
        BOOST_FOREACH(const PAIRTYPE(const CWalletTx*,unsigned int)& coin, setCoins)
            wtxNew.vin.push_back(CTxIn(coin.first->GetHash(),coin.second));

        // Sign
        int nIn = 0;
        BOOST_FOREACH(const PAIRTYPE(const CWalletTx*,unsigned int)& coin, setCoins)
        {
            if (!SignSignature(*this, *coin.first, wtxNew, nIn++))  return false;
        }

        //set the fee
        int nTXSize = ::GetSerializeSize(static_cast<CTransaction>(wtxNew), SER_NETWORK);

        int64 qTXFee  =  wtxNew.GetMinFee();
        if( nFeeRet < qTXFee)
        {
            nFeeRet=qTXFee;
            continue;
        }

        dPriority /= nTXSize;

        wtxNew.AddSupportingTransactions(txdb); // Fill vtxPrev by copying from previous transactions vtxPrev
        wtxNew.fTimeReceivedIsTxTime = true;
        break;

    }
    return true;
}

bool CWallet::CreateTransaction(CScript scriptPubKey, int64 nValue, CWalletTx& wtxNew, CReserveKey& reservekey, int64& nFeeRet)
{
    vector< pair<CScript, int64> > vecSend;
    vecSend.push_back(make_pair(scriptPubKey, nValue));
    return CreateTransaction(vecSend, wtxNew, reservekey, nFeeRet);
}

// Call after CreateTransaction unless you want to abort
bool CWallet::CommitTransaction(CWalletTx& wtxNew, CReserveKey& reservekey)
{
    printf("CommitTransaction:\n%s", wtxNew.ToString().c_str());
    // This is only to keep the database open to defeat the auto-flush for the
    // duration of this scope.  This is the only place where this optimization
    // maybe makes sense; please don't do it anywhere else.
    CWalletDB* pwalletdb = fFileBacked ? new CWalletDB(strWalletFile,"r") : NULL;

    // Take key pair from key pool so it won't be used again
    reservekey.KeepKey();

    // Add tx to wallet, because if it has change it's also ours,
    // otherwise just for transaction history.
    AddToWallet(wtxNew);

    // Mark old coins as spent
    set<CWalletTx*> setCoins;
    BOOST_FOREACH(const CTxIn& txin, wtxNew.vin)
    {
        CWalletTx &coin = mapWallet[txin.prevout.hash];
        coin.pwallet = this;
        coin.MarkSpent(txin.prevout.n,true);
        coin.WriteToDisk();
        vWalletUpdated.push_back(coin.GetHash());
    }

    if (fFileBacked)    delete pwalletdb;

    // Track how many getdata requests our transaction gets
    mapRequestCount[wtxNew.GetHash()] = 0;
    if (!wtxNew.AcceptToMemoryPool())
    {
        // This must not fail. The transaction has already been signed and recorded.
        uint256 txhash=wtxNew.GetHash();
        EraseFromWallet(txhash);
        printf("CommitTransaction() : Error: Transaction not valid");
        return false;
    }
    wtxNew.RelayWalletTransaction();


    MainFrameRepaint();
    return true;
}




// requires cs_main lock
string CWallet::SendMoney(CScript scriptPubKey, int64 nValue, CWalletTx& wtxNew, int64 nMaxFee)
{
    CReserveKey reservekey;
    reservekey.SetWallet(this);
    int64 nFeeRequired;
    if (IsLocked())
    {
        string strError = _("Error: Wallet locked, unable to create transaction  ");
        printf("SendMoney() : %s", strError.c_str());
        return strError;
    }
    if (!CreateTransaction(scriptPubKey, nValue, wtxNew, reservekey, nFeeRequired))
    {
        string strError;
        if (nValue + nFeeRequired > GetBalance())   strError = strprintf(_("Error: This transaction requires a transaction fee of at least %s because of its amount, complexity, or use of recently received funds  "), FormatMoney(nFeeRequired).c_str());
        else                                        strError = _("Error: Transaction creation failed  ");
        printf("SendMoney() : %s", strError.c_str());
        return strError;
    }

    if(nMaxFee && nFeeRequired>nMaxFee)
    {
        string strError = "Fee was greater than max fee";
        printf("SendMoney() : %s", strError.c_str());
        return strError;
    }
    //if (fAskFee && !ThreadSafeAskFee(nFeeRequired, _("Sending..."), NULL))  return "ABORTED";

    if (!CommitTransaction(wtxNew, reservekey))
        return _("Error: The transaction was rejected.  This might happen if some of the coins in your wallet were already spent, such as if you used a copy of wallet.dat and coins were spent in the copy but not marked as spent here.");

    MainFrameRepaint();
    return "";
}



// requires cs_main lock
string CWallet::SendMoneyToSolidCoinAddress(const CSolidCoinAddress& address, int64 nValue, CWalletTx& wtxNew, int64 nMaxFee)
{
    // Check amount
    if (nValue <= 0)                            return _("Invalid amount");
    if (nValue + MIN_TX_FEE > GetBalance())     return _("Insufficient funds");

    // Parse solidcoin address
    CScript scriptPubKey;
    scriptPubKey.SetSolidCoinAddress(address);

    return SendMoney(scriptPubKey, nValue, wtxNew, nMaxFee);
}




int CWallet::LoadWallet(bool bClean)
{
    if (!fFileBacked)   return false;
    int nLoadWalletRet = CWalletDB(strWalletFile,"cr+").LoadWallet(this,bClean);
    if (nLoadWalletRet != DB_LOAD_OK)   return nLoadWalletRet;
    if (!HaveKey(Hash160(vchDefaultKey)))
    {
        // Create new keyUser and set as default key
        RandAddSeedPerfmon();

        SetDefaultKey(GetOrReuseKeyFromPool());
        if (!SetAddressBookName(CSolidCoinAddress(vchDefaultKey), ""))
            return DB_LOAD_FAIL;
    }

    CreateThread(ThreadFlushWalletDB, &strWalletFile);
    return DB_LOAD_OK;
}


bool CWallet::SetAddressBookName(const CSolidCoinAddress& address, const string& strName)
{
    mapAddressBook[address] = strName;
    if (!fFileBacked)
        return false;
    return CWalletDB(strWalletFile).WriteName(address.ToString(), strName);
}

bool CWallet::DelAddressBookName(const CSolidCoinAddress& address)
{
    mapAddressBook.erase(address);
    if (!fFileBacked)
        return false;
    return CWalletDB(strWalletFile).EraseName(address.ToString());
}


void CWallet::PrintWallet(const CBlock& block)
{
    if (mapWallet.count(block.vtx[0].GetHash()))
    {
        CWalletTx& wtx = mapWallet[block.vtx[0].GetHash()];
        printf("    mine:  %"PRI64d"  %"PRI64d" %"PRI64d"", wtx.GetDepthInMainChain(), wtx.GetBlocksToMaturity(), wtx.GetCredit());
    }
    printf("\n");
}

bool CWallet::GetTransaction(const uint256 &hashTx, CWalletTx& wtx)
{
    map<uint256, CWalletTx>::iterator mi = mapWallet.find(hashTx);
    if (mi != mapWallet.end())
    {
        wtx = (*mi).second;
        return true;
    }
    return false;
}

bool CWallet::SetDefaultKey(const std::vector<unsigned char> &vchPubKey)
{
    if (fFileBacked)
    {
        if (!CWalletDB(strWalletFile).WriteDefaultKey(vchPubKey))
            return false;
    }
    vchDefaultKey = vchPubKey;
    return true;
}


bool CWallet::TopUpKeyPool()
{
    if (IsLocked())
        return false;

    CWalletDB walletdb(strWalletFile);

    // Top up key pool
    int64 nTargetSize = max(Setting_GetINT64("keypool"), (int64)0);
    while (setKeyPool.size() < nTargetSize+1)
    {
        int64 nEnd = 1;
        if (!setKeyPool.empty())
            nEnd = *(--setKeyPool.end()) + 1;
        if (!walletdb.WritePool(nEnd, CKeyPool(GenerateNewKey())))
            throw runtime_error("TopUpKeyPool() : writing generated key failed");
        setKeyPool.insert(nEnd);
        printf("keypool added key %"PRI64d", size=%d\n", nEnd, setKeyPool.size());
    }
    return true;
}

void CWallet::ReserveKeyFromKeyPool(int64& nIndex, CKeyPool& keypool)
{
    nIndex = -1;
    keypool.vchPubKey.clear();

    if (!IsLocked())    TopUpKeyPool();
    if(setKeyPool.empty())  return; // Get the oldest key

    CWalletDB walletdb(strWalletFile);
    nIndex = *(setKeyPool.begin());
    setKeyPool.erase(setKeyPool.begin());
    if (!walletdb.ReadPool(nIndex, keypool))    throw runtime_error("ReserveKeyFromKeyPool() : read failed");
    if (!HaveKey(Hash160(keypool.vchPubKey)))   throw runtime_error("ReserveKeyFromKeyPool() : unknown key in key pool");
    assert(!keypool.vchPubKey.empty());
    printf("keypool reserve %"PRI64d"\n", nIndex);
}

void CWallet::KeepKey(int64 nIndex)
{
    // Remove from key pool
    if (fFileBacked)
    {
        CWalletDB walletdb(strWalletFile);
        walletdb.ErasePool(nIndex);
    }
    printf("keypool keep %"PRI64d"\n", nIndex);
}

void CWallet::ReturnKey(int64 nIndex)
{
    // Return to key pool
    setKeyPool.insert(nIndex);
    printf("keypool return %"PRI64d"\n", nIndex);
}

vector<unsigned char> CWallet::GetOrReuseKeyFromPool()
{
    int64 nIndex = 0;
    CKeyPool keypool;
    ReserveKeyFromKeyPool(nIndex, keypool);
    if(nIndex == -1)
        return vchDefaultKey;
    KeepKey(nIndex);
    return keypool.vchPubKey;
}

int64 CWallet::GetOldestKeyPoolTime()
{
    int64 nIndex = 0;
    CKeyPool keypool;
    ReserveKeyFromKeyPool(nIndex, keypool);
    if (nIndex == -1)
        return SolidTime_Get();
    ReturnKey(nIndex);
    return keypool.nTime;
}

vector<unsigned char> CReserveKey::GetReservedKey()
{
    if (nIndex == -1)
    {
        CKeyPool keypool;
        pwallet->ReserveKeyFromKeyPool(nIndex, keypool);
        if (nIndex != -1)
            vchPubKey = keypool.vchPubKey;
        else
        {
            printf("CReserveKey::GetReservedKey(): Warning: using default key instead of a new key, top up your keypool.");
            vchPubKey = pwallet->vchDefaultKey;
        }
    }
    assert(!vchPubKey.empty());
    return vchPubKey;
}

void CReserveKey::KeepKey()
{
    if(pwallet==0) return;
    if (nIndex != -1)   pwallet->KeepKey(nIndex);
    nIndex = -1;
    vchPubKey.clear();
}

void CReserveKey::ReturnKey()
{
    if(pwallet==0) return;
    if (nIndex != -1)   pwallet->ReturnKey(nIndex);
    nIndex = -1;
    vchPubKey.clear();
}



