// NAME
//      $RCSfile: StandardSocket.java,v $
// DESCRIPTION
//      [given below in javadoc format]
// DELTA
//      $Revision: 1.12 $
// CREATED
//      $Date: 2006/01/30 15:50:38 $
// COPYRIGHT
//      Westhawk Ltd
// TO DO
//

/*
 * Copyright (C) 2000 - 2006 by Westhawk Ltd
 * <a href="www.westhawk.co.uk">www.westhawk.co.uk</a>
 *
 * Permission to use, copy, modify, and distribute this software
 * for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that
 * both the copyright notice and this permission notice appear in
 * supporting documentation.
 * This software is provided "as is" without express or implied
 * warranty.
 * author <a href="mailto:snmp@westhawk.co.uk">Tim Panton</a>
 */

package uk.co.westhawk.snmp.net;

import java.io.*;
import java.net.*;
import uk.co.westhawk.snmp.stack.*;

/**
 * This is a wrapper class around the standard DatagramSocket. This
 * sends packets over UDP, which is standard in SNMP.
 *
 * @see java.net.DatagramSocket
 * @author <a href="mailto:snmp@westhawk.co.uk">Tim Panton</a>
 * @version $Revision: 1.12 $ $Date: 2006/01/30 15:50:38 $
 */
public class StandardSocket implements ContextSocketFace 
{
    static final String     version_id =
        "@(#)$Id: StandardSocket.java,v 1.12 2006/01/30 15:50:38 birgit Exp $ Copyright Westhawk Ltd";

    private DatagramSocket  soc=null;
    private InetAddress     sendToHostAddr;
    private int             sendToHostPort;
    private InetAddress     receiveFromHostAddr;
    private int             receiveFromHostPort;
    private InetAddress     locBindAddr = null;

public StandardSocket()
{
}

public void create(int port, String bindAddr) throws IOException
{
    sendToHostPort = port;
    receiveFromHostPort = sendToHostPort; // initialise (once!)
    try
    {
        locBindAddr = null;
        if (bindAddr != null)
        {
            locBindAddr = InetAddress.getByName(bindAddr);
        }
        soc = new DatagramSocket(sendToHostPort, locBindAddr);
    }
    catch (SocketException exc)
    {
        String str = "Socket problem " + exc.getMessage();
        throw (new IOException(str));
    }
}


public void create(String host, int port, String bindAddr) throws IOException
{
    sendToHostPort = port;
    receiveFromHostPort = sendToHostPort; // initialise (once!)
    try
    {
        sendToHostAddr = InetAddress.getByName(host);
        receiveFromHostAddr = sendToHostAddr; // initialise (once!)
        locBindAddr = null;
        if (bindAddr != null)
        {
            locBindAddr = InetAddress.getByName(bindAddr);
        }
        InetSocketAddress isa = new InetSocketAddress(locBindAddr, 0);
        soc = new DatagramSocket(isa);
    }
    catch (SocketException exc)
    {
        String str = "Socket problem " + exc.getMessage();
        throw (new IOException(str));
    }
    catch (UnknownHostException exc)
    {
        String str = "Cannot find host " + exc.getMessage();
        throw (new IOException(str));
    }
}

public String getReceivedFromHostAddress()
{
    String res = null;
    if (receiveFromHostAddr != null)
    {
        res = receiveFromHostAddr.getHostAddress();
    }
    return res;
}

public String getSendToHostAddress()
{
    String res = null;
    if (sendToHostAddr != null)
    {
        res = sendToHostAddr.getHostAddress();
    }
    return res;
}

public String getLocalSocketAddress()
{
    String res = null;
    if (soc != null)
    {
        SocketAddress sa = soc.getLocalSocketAddress();
        if (sa != null)
        {
            res = sa.toString();
        }
    }
    return res;
}

public String getRemoteSocketAddress()
{
    String res = null;
    if (soc != null)
    {
        SocketAddress sa = soc.getRemoteSocketAddress();
        if (sa != null)
        {
            res = sa.toString();
        }
    }
    return res;
}

public StreamPortItem receive(int maxRecvSize) throws IOException
{
    StreamPortItem item = null;
    if (soc != null)
    {
        byte [] data = new byte[maxRecvSize];
        DatagramPacket p = new DatagramPacket(data, maxRecvSize);

        // timeout will throw an exception every 1000 secs whilst idle
        // it is caught and ignored, but as a side effect it loops,
        // checking 'me'
        soc.setSoTimeout(1000);

        soc.receive(p);
        receiveFromHostAddr = p.getAddress();
        receiveFromHostPort = p.getPort();

        ByteArrayInputStream in = null;
        in = new ByteArrayInputStream(p.getData(), 0, p.getLength());
        item = new StreamPortItem(receiveFromHostAddr.getHostAddress(), 
                                  receiveFromHostPort, in);
        p = null;
    }
    return item;
}

public void send(byte[] packet) throws IOException
{
    if (soc != null)
    {
        DatagramPacket pack = new DatagramPacket(packet, packet.length, 
              sendToHostAddr, sendToHostPort);
        soc.send(pack);
    }
}

public void close()
{
    if (soc != null)
    {
        soc.close();
        soc = null;
    }
}

}
