/***************************************************************************

    Midway MCR-3 system

    driver by Christopher Kirmse, Aaron Giles

    Games supported:
        * Demolition Derby (Monoboard version) (Turbo Chip Squeak)
        * Sarge (Turbo Chip Squeak)
        * Max RPM (Turbo Chip Squeak)
        * Rampage (Sounds Good)
        * Power Drive (Sounds Good)
        * Star Guards (Sounds Good)
        * Spy Hunter (Chip Squeak Deluxe)
        * Crater Raider
        * Turbo Tag (prototype) (Chip Squeak Deluxe)

    Known bugs:
        * Spy Hunter crashes at the end of the boat level
        * sprite placement on the scrolling games seems a bit off


****************************************************************************

    Game Hardware

    Sarge:
        * Mono Board (A080-91787-C000)
        * Turbo Cheap Squeak (A080-91779-B000)
        * MCR/2 Dual Power Amp (A082-90910-F000)

    Spy Hunter:
        * MCR/III CPU (A080-91442-J000)
        * Super Sound I/O (A082-90913-F000)
        * MCR/III Video Gen (A082-91433-J000)
        * Cheap Squeak Deluxe (A080-91671-E000)
        * Absolute Position Board (A080-91649-C000)
        * Lamp Driver Board (A080-91673-B000)
        * MCR/2 Dual Power Amp (A082-90910-F000)

****************************************************************************

    Memory map

****************************************************************************

    ========================================================================
    CPU #1
    ========================================================================
    0000-DFFF   R     xxxxxxxx    Program ROM
    E000-E7FF   R/W   xxxxxxxx    NVRAM
    E800-E9FF   R/W   xxxxxxxx    Sprite RAM
    F000-F7FF   R/W   xxxxxxxx    Background video RAM
    F800-F8FF     W   xxxxxxxx    Palette RAM
    ========================================================================
    0000        R     x-xxxxxx    Input ports
                R     x-------    Service switch (active low)
                R     --x-----    Tilt
                R     ---xxx--    External inputs
                R     ------x-    Right coin
                R     -------x    Left coin
    0000        W     xxxxxxxx    Data latch OP0 (coin meters, 2 led's and cocktail 'flip')
    0001        R     xxxxxxxx    External inputs
    0002        R     xxxxxxxx    External inputs
    0003        R     xxxxxxxx    DIP switches
    0004        R     xxxxxxxx    External inputs
    0004        W     xxxxxxxx    Data latch OP4 (comm. with external hardware)
    0007        R     xxxxxxxx    Audio status
    001C-001F   W     xxxxxxxx    Audio latches 1-4
    00E0        W     --------    Watchdog reset
    00E8        W     xxxxxxxx    Unknown (written at initialization time)
    00F0-00F3   W     xxxxxxxx    CTC communications
    ========================================================================
    Interrupts:
        NMI ???
        INT generated by CTC
    ========================================================================


    ========================================================================
    CPU #2 (Super Sound I/O)
    ========================================================================
    0000-3FFF   R     xxxxxxxx    Program ROM
    8000-83FF   R/W   xxxxxxxx    Program RAM
    9000-9003   R     xxxxxxxx    Audio latches 1-4
    A000          W   xxxxxxxx    AY-8910 #1 control
    A001        R     xxxxxxxx    AY-8910 #1 status
    A002          W   xxxxxxxx    AY-8910 #1 data
    B000          W   xxxxxxxx    AY-8910 #2 control
    B001        R     xxxxxxxx    AY-8910 #2 status
    B002          W   xxxxxxxx    AY-8910 #2 data
    C000          W   xxxxxxxx    Audio status
    E000          W   xxxxxxxx    Unknown
    F000        R     xxxxxxxx    Audio board switches
    ========================================================================
    Interrupts:
        NMI ???
        INT generated by external circuitry 780 times/second
    ========================================================================

***************************************************************************/


#include "driver.h"
#include "cpu/z80/z80.h"
#include "deprecat.h"
#include "machine/z80ctc.h"
#include "audio/mcr.h"
#include "mcr.h"

#include "turbotag.lh"


#define MASTER_CLOCK		XTAL_20MHz



/*************************************
 *
 *  Local variables and tables
 *
 *************************************/

static UINT8 input_mux;
static UINT8 latched_input;

static UINT8 last_op4;

static UINT8 maxrpm_adc_control;
static UINT8 maxrpm_adc_select;
static UINT8 maxrpm_last_shift;
static INT8 maxrpm_p1_shift;
static INT8 maxrpm_p2_shift;



/*************************************
 *
 *  Demolition Derby (mono) I/O ports
 *
 *************************************/

static READ8_HANDLER( demoderm_ip1_r )
{
	return input_port_read(space->machine, "MONO.IP1") |
		(input_port_read(space->machine, input_mux ? "MONO.IP1.ALT2" : "MONO.IP1.ALT1") << 2);
}


static READ8_HANDLER( demoderm_ip2_r )
{
	return input_port_read(space->machine, "MONO.IP2") |
		(input_port_read(space->machine, input_mux ? "MONO.IP2.ALT2" : "MONO.IP2.ALT1") << 2);
}


static WRITE8_HANDLER( demoderm_op6_w )
{
	/* top 2 bits select the input */
	if (data & 0x80) input_mux = 0;
	if (data & 0x40) input_mux = 1;

	/* low 5 bits control the turbo CS */
	turbocs_data_w(space, offset, data);
}



/*************************************
 *
 *  Max RPM I/O ports
 *
 *************************************/

static READ8_HANDLER( maxrpm_ip1_r )
{
	return latched_input;
}


static READ8_HANDLER( maxrpm_ip2_r )
{
	static const UINT8 shift_bits[5] = { 0x00, 0x05, 0x06, 0x01, 0x02 };
	UINT8 start = input_port_read(space->machine, "MONO.IP0");
	UINT8 shift = input_port_read(space->machine, "SHIFT");

	/* reset on a start */
	if (!(start & 0x08))
		maxrpm_p1_shift = 0;
	if (!(start & 0x04))
		maxrpm_p2_shift = 0;

	/* increment, decrement on falling edge */
	if (!(shift & 0x01) && (maxrpm_last_shift & 0x01))
	{
		maxrpm_p1_shift++;
		if (maxrpm_p1_shift > 4)
			maxrpm_p1_shift = 4;
	}
	if (!(shift & 0x02) && (maxrpm_last_shift & 0x02))
	{
		maxrpm_p1_shift--;
		if (maxrpm_p1_shift < 0)
			maxrpm_p1_shift = 0;
	}
	if (!(shift & 0x04) && (maxrpm_last_shift & 0x04))
	{
		maxrpm_p2_shift++;
		if (maxrpm_p2_shift > 4)
			maxrpm_p2_shift = 4;
	}
	if (!(shift & 0x08) && (maxrpm_last_shift & 0x08))
	{
		maxrpm_p2_shift--;
		if (maxrpm_p2_shift < 0)
			maxrpm_p2_shift = 0;
	}

	maxrpm_last_shift = shift;

	return ~((shift_bits[maxrpm_p1_shift] << 4) + shift_bits[maxrpm_p2_shift]);
}


static WRITE8_HANDLER( maxrpm_op5_w )
{
	/* latch the low 4 bits as input to the ADC0844 */
	maxrpm_adc_control = data & 0x0f;

	/* remaining bits go to standard connections */
	mcrmono_control_port_w(space, offset, data);
}


static WRITE8_HANDLER( maxrpm_op6_w )
{
	static const char *const inputs[] = { "MONO.IP1", "MONO.IP1.ALT1", "MONO.IP1.ALT2", "MONO.IP1.ALT3" };
	/*
        Reflective Sensor Control:
            4 bits of input from OP5 are routed to a transceiver at U2, and
            ultimately on to the low 4 I/O pins of the ADC0844. The /EN on
            the transceiver is directly connected to J2-2.

            In order to perform a read or a write to the ADC0844, the /RD and
            /WR signals are directly controlled via J2-8 and J2-7 respectively.
            The input from the /WR is controlled by enabling the transceiver
            above to allow the 4 bits of input to flow through. The output
            from an /RD is controlled by disabling the transceiver and allowing
            the 8 bits of output to flow through J2-13 through J2-20. These are
            read via IP1.
    */
	/* bit 7 = /RD (J2-8) on ADC0844 */
	/* bit 6 = /WR (J2-7) on ADC0844 */
	/* bit 5 = /EN (J2-2) on input latch */

	/* when the read is toggled is when the ADC value is latched */
	if (!(data & 0x80))
		latched_input = input_port_read(space->machine, inputs[maxrpm_adc_select]);

	/* when both the write and the enable are low, it's a write to the ADC0844 */
	/* unfortunately the behavior below doesn't match up with the inputs on the */
	/* schematics and wiring diagrams, so they must be wrong */
	if (!(data & 0x40) && !(data & 0x20))
		maxrpm_adc_select = (maxrpm_adc_control >> 1) & 3;

	/* low 5 bits control the turbo CS */
	turbocs_data_w(space, offset, data);
}



/*************************************
 *
 *  Rampage I/O ports
 *
 *************************************/

static READ8_HANDLER( rampage_ip4_r )
{
	return input_port_read(space->machine, "MONO.IP4") | (soundsgood_status_r(space,0) << 7);
}


static WRITE8_HANDLER( rampage_op6_w )
{
	/* bit 5 controls reset of the Sounds Good board */
	soundsgood_reset_w(space->machine, (~data >> 5) & 1);

	/* low 5 bits go directly to the Sounds Good board */
	soundsgood_data_w(space, offset, data);
}



/*************************************
 *
 *  Power Drive I/O ports
 *
 *************************************/

static READ8_HANDLER( powerdrv_ip2_r )
{
	return input_port_read(space->machine, "MONO.IP2") | (soundsgood_status_r(space, 0) << 7);
}


static WRITE8_HANDLER( powerdrv_op5_w )
{
	/*
        Lamp Board:
            Very simple board with direct lamp controls.
            Pin J1-10 controls lamp 1.
            Pin J1-8 controls lamp 2.
            Pin J1-6 controls lamp 3.
    */
	/* bit 3 -> J1-10 = lamp 1 */
	/* bit 2 -> J1-8 = lamp 2 */
	/* bit 1 -> J1-6 = lamp 3 */
	set_led_status(0, (data >> 3) & 1);
	set_led_status(1, (data >> 2) & 1);
	set_led_status(2, (data >> 1) & 1);

	/* remaining bits go to standard connections */
	mcrmono_control_port_w(space, offset, data);
}


static WRITE8_HANDLER( powerdrv_op6_w )
{
	/* bit 5 controls reset of the Sounds Good board */
	soundsgood_reset_w(space->machine, (~data >> 5) & 1);

	/* low 5 bits go directly to the Sounds Good board */
	soundsgood_data_w(space, offset, data);
}



/*************************************
 *
 *  Star Guards I/O ports
 *
 *************************************/

static READ8_HANDLER( stargrds_ip0_r )
{
	UINT8 result = input_port_read(space->machine, "MONO.IP0");
	if (input_mux)
		result = (result & ~0x0a) | (input_port_read(space->machine, "MONO.IP0.ALT") & 0x0a);
	return (result & ~0x10) | ((soundsgood_status_r(space, 0) << 4) & 0x10);
}


static WRITE8_HANDLER( stargrds_op5_w )
{
	/* bit 1 controls input muxing on port 0 */
	input_mux = (data >> 1) & 1;

	/* bit 2 controls light #0 */
	/* bit 3 controls light #1 */
	/* bit 4 controls light #2 */
	set_led_status(0, (data >> 2) & 1);
	set_led_status(1, (data >> 3) & 1);
	set_led_status(2, (data >> 4) & 1);

	/* remaining bits go to standard connections */
	mcrmono_control_port_w(space, offset, data);
}


static WRITE8_HANDLER( stargrds_op6_w )
{
	/* bit 6 controls reset of the Sounds Good board */
	soundsgood_reset_w(space->machine, (~data >> 6) & 1);

	/* unline the other games, the STROBE is in the high bit instead of the low bit */
	soundsgood_data_w(space, offset, (data << 1) | (data >> 7));
}



/*************************************
 *
 *  Spy Hunter I/O ports
 *
 *************************************/

static READ8_HANDLER( spyhunt_ip1_r )
{
	return input_port_read(space->machine, "SSIO.IP1") | (csdeluxe_status_r(space, 0) << 5);
}


static READ8_HANDLER( spyhunt_ip2_r )
{
	/* multiplexed steering wheel/gas pedal */
	return input_port_read(space->machine, input_mux ? "SSIO.IP2.ALT" : "SSIO.IP2");
}


static WRITE8_HANDLER( spyhunt_op4_w )
{
	/* Spy Hunter uses port 4 for talking to the Chip Squeak Deluxe */
	/* (and for toggling the lamps and muxing the analog inputs) */

	/* mux select is in bit 7 */
	input_mux = (data >> 7) & 1;

	/*
        Lamp Driver:
            A 3-to-8 latching demuxer is connected to the input bits.
            Three of the inputs (J1-11,10,12) specify which output to write
            to, and the fourth input (J1-14) is the data value. A fifth input
            (J1-13) controls the strobe to latch the data value for the
            demuxer. The eight outputs directly control 8 lamps.
    */
	/* bit 5 = STR1 (J1-13) */
	if (((last_op4 ^ data) & 0x20) && !(data & 0x20))
	{
		static const char *const lampname[8] =
		{
			"lamp0", "lamp1", "lamp2", "lamp3",
			"lamp4", "lamp5", "lamp6", "lamp7"
		};
		/* bit 3 -> J1-14 (DATA) */
		/* bit 2 -> J1-11 (A2) */
		/* bit 1 -> J1-10 (A1) */
		/* bit 0 -> J1-12 (A0) */
		output_set_value(lampname[data & 7], (data >> 3) & 1);
	}
	last_op4 = data;

	/* low 5 bits go to control the Chip Squeak Deluxe */
	csdeluxe_data_w(space, offset, data);
}



/*************************************
 *
 *  Turbo Tag kludges
 *
 *************************************/

static READ8_HANDLER( turbotag_ip2_r )
{
	/* multiplexed steering wheel/gas pedal */
	if (input_mux)
		return input_port_read(space->machine, "SSIO.IP2.ALT");

	return input_port_read(space->machine, "SSIO.IP2") + 5 * (video_screen_get_frame_number(space->machine->primary_screen) & 1);
}


static READ8_HANDLER( turbotag_kludge_r )
{
	/* The checksum on the ttprog1.bin ROM seems to be bad by 1 bit */
	/* The checksum should come out to $82 but it should be $92     */
	/* Unfortunately, the game refuses to start if any bad ROM is   */
	/* found; to work around this, we catch the checksum byte read  */
	/* and modify it to what we know we will be getting.            */
	if (cpu_get_previouspc(space->cpu) == 0xb29)
		return 0x82;
	else
		return 0x92;
}



/*************************************
 *
 *  MCR Monoboard CPU memory handlers
 *
 *************************************/

/* address map verified from schematics */
static ADDRESS_MAP_START( mcrmono_map, ADDRESS_SPACE_PROGRAM, 8 )
	ADDRESS_MAP_UNMAP_HIGH
	AM_RANGE(0x0000, 0xdfff) AM_ROM
	AM_RANGE(0xe000, 0xe7ff) AM_RAM AM_BASE(&generic_nvram) AM_SIZE(&generic_nvram_size)
	AM_RANGE(0xe800, 0xe9ff) AM_RAM AM_BASE(&spriteram) AM_SIZE(&spriteram_size)
	AM_RANGE(0xea00, 0xebff) AM_RAM
	AM_RANGE(0xec00, 0xec7f) AM_MIRROR(0x0380) AM_WRITE(mcr3_paletteram_w) AM_BASE(&paletteram)
	AM_RANGE(0xf000, 0xf7ff) AM_RAM_WRITE(mcr3_videoram_w) AM_BASE(&videoram) AM_SIZE(&videoram_size)
	AM_RANGE(0xf800, 0xffff) AM_ROM		/* schematics show a 2716 @ 2B here, but nobody used it */
ADDRESS_MAP_END

/* I/O map verified from schematics */
static ADDRESS_MAP_START( mcrmono_portmap, ADDRESS_SPACE_IO, 8 )
	ADDRESS_MAP_UNMAP_HIGH
	ADDRESS_MAP_GLOBAL_MASK(0xff)
	AM_RANGE(0x00, 0x00) AM_MIRROR(0x78) AM_READ_PORT("MONO.IP0")
	AM_RANGE(0x01, 0x01) AM_MIRROR(0x78) AM_READ_PORT("MONO.IP1")
	AM_RANGE(0x02, 0x02) AM_MIRROR(0x78) AM_READ_PORT("MONO.IP2")
	AM_RANGE(0x03, 0x03) AM_MIRROR(0x78) AM_READ_PORT("MONO.IP3")
	AM_RANGE(0x04, 0x04) AM_MIRROR(0x78) AM_READ_PORT("MONO.IP4")
	AM_RANGE(0x05, 0x05) AM_MIRROR(0x78) AM_WRITE(mcrmono_control_port_w)
	AM_RANGE(0x07, 0x07) AM_MIRROR(0x78) AM_WRITE(watchdog_reset_w)
	AM_RANGE(0xf0, 0xf3) AM_MIRROR(0x0c) AM_DEVREADWRITE("ctc", z80ctc_r, z80ctc_w)
ADDRESS_MAP_END



/*************************************
 *
 *  Spy Hunter main CPU memory handlers
 *
 *************************************/

/* address map verified from schematics */
static ADDRESS_MAP_START( spyhunt_map, ADDRESS_SPACE_PROGRAM, 8 )
	ADDRESS_MAP_UNMAP_HIGH
	AM_RANGE(0x0000, 0xdfff) AM_ROM
	AM_RANGE(0xe000, 0xe7ff) AM_RAM_WRITE(spyhunt_videoram_w) AM_BASE(&videoram) AM_SIZE(&videoram_size)
	AM_RANGE(0xe800, 0xebff) AM_MIRROR(0x0400) AM_RAM_WRITE(spyhunt_alpharam_w) AM_BASE(&spyhunt_alpharam)
	AM_RANGE(0xf000, 0xf7ff) AM_RAM AM_BASE(&generic_nvram) AM_SIZE(&generic_nvram_size)
	AM_RANGE(0xf800, 0xf9ff) AM_RAM AM_BASE(&spriteram) AM_SIZE(&spriteram_size)
	AM_RANGE(0xfa00, 0xfa7f) AM_MIRROR(0x0180) AM_WRITE(mcr3_paletteram_w) AM_BASE(&paletteram)
ADDRESS_MAP_END

/* upper I/O map determined by PAL; only SSIO ports and scroll registers are verified from schematics */
static ADDRESS_MAP_START( spyhunt_portmap, ADDRESS_SPACE_IO, 8 )
	ADDRESS_MAP_UNMAP_HIGH
	ADDRESS_MAP_GLOBAL_MASK(0xff)
	SSIO_INPUT_PORTS
	AM_RANGE(0x84, 0x86) AM_WRITE(mcr_scroll_value_w)
	AM_RANGE(0xe0, 0xe0) AM_WRITE(watchdog_reset_w)
	AM_RANGE(0xe8, 0xe8) AM_WRITENOP
	AM_RANGE(0xf0, 0xf3) AM_DEVREADWRITE("ctc", z80ctc_r, z80ctc_w)
ADDRESS_MAP_END



/*************************************
 *
 *  Port definitions
 *
 *************************************/

/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( demoderm )
	PORT_START("MONO.IP0")	/* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE( 0x20, IP_ACTIVE_LOW )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP1")	/* J2 10-13,15-18 */	/* The high 6 bits contain the steering wheel value */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)

	PORT_START("MONO.IP1.ALT1")	/* J2 10-13,15-18 */	/* The high 6 bits contain the steering wheel value */
	PORT_BIT( 0x3f, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(1)

	PORT_START("MONO.IP1.ALT2")	/* IN1 (muxed) -- the high 6 bits contain the steering wheel value */
	PORT_BIT( 0x3f, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(3)

	PORT_START("MONO.IP2")	/* J3 1-8 */	/* The high 6 bits contain the steering wheel value */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)

	PORT_START("MONO.IP2.ALT1")	/* J3 1-8 */	/* The high 6 bits contain the steering wheel value */
	PORT_BIT( 0x3f, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(2)

	PORT_START("MONO.IP2.ALT2")	/* IN2 (muxed) -- the high 6 bits contain the steering wheel value */
	PORT_BIT( 0x3f, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(4)

	PORT_START("MONO.IP3")	/* DIPSW @ B3 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x01, "2P Upright" )
	PORT_DIPSETTING(    0x00, "4P Cocktail" )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Harder ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Reward Screen" )
	PORT_DIPSETTING(    0x08, "Expanded" )
	PORT_DIPSETTING(    0x00, "Limited" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("MONO.IP4")	/* J4 1-7,9 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(4)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(4)

INPUT_PORTS_END


/* not verified, no manual found */
static INPUT_PORTS_START( sarge )
	PORT_START("MONO.IP0")	/* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x20, IP_ACTIVE_LOW )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP1")	/* J2 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_2WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_2WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_2WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_2WAY PORT_PLAYER(1)
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)

	PORT_START("MONO.IP2")	/* J3 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_2WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_2WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_2WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_2WAY PORT_PLAYER(2)
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)

	PORT_START("MONO.IP3")
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
/* 0x00 says 2 Coins/2 Credits in service mode, but gives 1 Coin/1 Credit */
	PORT_BIT( 0xc7, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("MONO.IP4")	/* J4 1-7,9 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( maxrpm )
	PORT_START("MONO.IP0")	/* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("MONO.IP1")	/* J2 10-13,15-18 */
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(0x30,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(2)

	PORT_START("MONO.IP2")	/* J3 1-8 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_SPECIAL )

	PORT_START("MONO.IP3")
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
/* 0x00 says 2 Coins/2 Credits in service mode, but gives 1 Coin/1 Credit */
	PORT_BIT( 0xc7, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("MONO.IP4")	/* J4 1-7,9 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("MONO.IP1.ALT1")
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(0x30,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(1)

	PORT_START("MONO.IP1.ALT2")
	PORT_BIT( 0xff, 0x74, IPT_PADDLE ) PORT_MINMAX(0x34,0xb4) PORT_SENSITIVITY(40) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(2)

	PORT_START("MONO.IP1.ALT3")
	PORT_BIT( 0xff, 0x74, IPT_PADDLE ) PORT_MINMAX(0x34,0xb4) PORT_SENSITIVITY(40) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(1)

	PORT_START("SHIFT")	/* fake for shifting */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1) PORT_NAME("P1 Shift Up")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1) PORT_NAME("P1 Shift Down")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2) PORT_NAME("P2 Shift Up")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2) PORT_NAME("P2 Shift Down")
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( rampage )
	PORT_START("MONO.IP0")	/* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x20, IP_ACTIVE_LOW )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP1")	/* J2 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP2")	/* J3 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP3")
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x03, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x04, 0x04, "Score Option" )
	PORT_DIPSETTING(    0x04, "Keep score when continuing" )
	PORT_DIPSETTING(    0x00, "Lose score when continuing" )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_1C ) )
	PORT_DIPNAME( 0x70, 0x70, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x70, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x60, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x50, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x80, 0x80, "Rack Advance (Cheat)" ) PORT_CODE(KEYCODE_F1)
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("MONO.IP4")	/* J4 1-7,9 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* status from Sounds Good board */
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( powerdrv )
	PORT_START("MONO.IP0")	/* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x20, IP_ACTIVE_LOW )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("MONO.IP1")	/* J2 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_TOGGLE PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_TOGGLE PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)

	PORT_START("MONO.IP2")	/* J3 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(3)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_TOGGLE PORT_PLAYER(3)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* status from Sounds Good board */

	PORT_START("MONO.IP3")
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C ) )
/*  PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )*/
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_2C ) )
	PORT_BIT( 0x0c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x30, "Factory" )
	PORT_DIPSETTING(    0x10, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Hardest ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Rack Advance (Cheat)" ) PORT_CODE(KEYCODE_F1)
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("MONO.IP4")	/* J4 1-7,9 */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( stargrds )
	PORT_START("MONO.IP0")	/* J2 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* status from Sounds Good board */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("MONO.IP1")	/* J2 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_PLAYER(1)

	PORT_START("MONO.IP2")	/* J3 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_PLAYER(2)

	PORT_START("MONO.IP3")
	PORT_BIT( 0x03, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0c, 0x0c, "Energy Units" )
	PORT_DIPSETTING(    0x08, "8" )
	PORT_DIPSETTING(    0x0c, "10" )
	PORT_DIPSETTING(    0x04, "12" )
	PORT_DIPSETTING(    0x00, "14" )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0xc0, 0xc0, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0xc0, DEF_STR( Medium ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Hardest ) )

	PORT_START("MONO.IP4")	/* J4 1-7,9 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP ) PORT_PLAYER(3)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN ) PORT_PLAYER(3)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT ) PORT_PLAYER(3)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT ) PORT_PLAYER(3)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP ) PORT_PLAYER(3)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN ) PORT_PLAYER(3)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT ) PORT_PLAYER(3)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT ) PORT_PLAYER(3)

	PORT_START("MONO.IP0.ALT")	/* IN0 (muxed) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )		/* same as MONO.IN0 */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )		/* same as MONO.IN0 */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )		/* same as MONO.IN0 */
INPUT_PORTS_END


/* verified from wiring diagram, plus DIP switches from manual */
static INPUT_PORTS_START( spyhunt )
	PORT_START("SSIO.IP0")	/* J4 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_NAME("Gear Shift") PORT_CODE(KEYCODE_ENTER) PORT_TOGGLE
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("SSIO.IP1")	/* J4 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Oil Slick")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("Missiles")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Weapon Truck")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Smoke Screen")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Machine Guns")
	PORT_BIT( 0x60, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* status from CS deluxe, never read */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("SSIO.IP2")	/* J5 1-8 */
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(0x30,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)

	PORT_START("SSIO.IP3")	/* DIPSW @ B3 */
	PORT_DIPNAME( 0x01, 0x01, "Game Timer" )
	PORT_DIPSETTING(    0x00, "1:00" )
	PORT_DIPSETTING(    0x01, "1:30" )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("SSIO.IP4")	/* J6 1-8 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("SSIO.DIP")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("SSIO.IP2.ALT")
	PORT_BIT( 0xff, 0x74, IPT_PADDLE ) PORT_MINMAX(0x34,0xb4) PORT_SENSITIVITY(40) PORT_KEYDELTA(10)
INPUT_PORTS_END


/* not verified, no manual found */
static INPUT_PORTS_START( crater )
	PORT_START("SSIO.IP0")	/* J4 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("SSIO.IP1")	/* J4 10-13,15-18 */
	PORT_BIT( 0xff, 0x00, IPT_DIAL ) PORT_SENSITIVITY(25) PORT_KEYDELTA(10) PORT_CODE_DEC(KEYCODE_Z) PORT_CODE_INC(KEYCODE_X) PORT_REVERSE

	PORT_START("SSIO.IP2")	/* J5 1-8 */
	PORT_BIT( 0x03, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_2WAY
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_2WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("SSIO.IP3")	/* DIPSW @ B3 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("SSIO.IP4")	/* J6 1-8 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("SSIO.DIP")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/* not verified, no manual found */
static INPUT_PORTS_START( turbotag )
	PORT_START("SSIO.IP0")	/* J4 1-8 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_NAME("Gear Shift") PORT_CODE(KEYCODE_ENTER) PORT_TOGGLE
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START("SSIO.IP1")	/* J4 10-13,15-18 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Left Button")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("Left Trigger")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Center Button")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Right Button")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Right Trigger")
	PORT_BIT( 0x60, IP_ACTIVE_HIGH, IPT_SPECIAL )	/* status from CS deluxe, never read */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("SSIO.IP2")	/* J5 1-8 */
	PORT_BIT( 0xff, 0x3c, IPT_PEDAL ) PORT_MINMAX(60,180) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)

	PORT_START("SSIO.IP3")	/* DIPSW @ B3 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("SSIO.IP4")	/* J6 1-8 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("SSIO.DIP")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("SSIO.IP2.ALT")
	PORT_BIT( 0xff, 0x60, IPT_PADDLE ) PORT_SENSITIVITY(40) PORT_KEYDELTA(10)
INPUT_PORTS_END



/*************************************
 *
 *  Graphics definitions
 *
 *************************************/

static const UINT32 spyhunt_charlayout_xoffset[64] =
{
	   0,  0,  2,  2,  4,  4,  6,  6,  8,  8, 10, 10, 12, 12, 14, 14,
	  16, 16, 18, 18, 20, 20, 22, 22, 24, 24, 26, 26, 28, 28, 30, 30,
	  32, 32, 34, 34, 36, 36, 38, 38, 40, 40, 42, 42, 44, 44, 46, 46,
	  48, 48, 50, 50, 52, 52, 54, 54, 56, 56, 58, 58, 60, 60, 62, 62
};

static const gfx_layout spyhunt_charlayout =
{
	64,32,
	RGN_FRAC(1,2),
	4,
	{ RGN_FRAC(1,2), RGN_FRAC(1,2)+1, 0, 1 },
	EXTENDED_XOFFS,
	{
		  0*32,  0*32,  2*32,  2*32,  4*32,  4*32,  6*32,  6*32,
		  8*32,  8*32, 10*32, 10*32, 12*32, 12*32, 14*32, 14*32,
		 16*32, 16*32, 18*32, 18*32, 20*32, 20*32, 22*32, 22*32,
		 24*32, 24*32, 26*32, 26*32, 28*32, 28*32, 30*32, 30*32
	},
	128*8,
	spyhunt_charlayout_xoffset,
	NULL
};


static const gfx_layout spyhunt_alphalayout =
{
	16,16,
	RGN_FRAC(1,1),
	2,
	{ 0, 1 },
	{ 0, 0, 2, 2, 4, 4, 6, 6, 8, 8, 10, 10, 12, 12, 14, 14 },
	{ 0, 0, 2*8, 2*8, 4*8, 4*8, 6*8, 6*8, 8*8, 8*8, 10*8, 10*8, 12*8, 12*8, 14*8, 14*8 },
	16*8
};


static GFXDECODE_START( mcr3 )
	GFXDECODE_SCALE( "gfx1", 0, mcr_bg_layout,     0, 4, 2, 2 )
	GFXDECODE_ENTRY( "gfx2", 0, mcr_sprite_layout, 0, 4 )
GFXDECODE_END


static GFXDECODE_START( spyhunt )
	GFXDECODE_ENTRY( "gfx1", 0, spyhunt_charlayout,  1*16, 1 )
	GFXDECODE_ENTRY( "gfx2", 0, mcr_sprite_layout,   0*16, 4 )
	GFXDECODE_ENTRY( "gfx3", 0, spyhunt_alphalayout, 4*16, 1 )
GFXDECODE_END



/*************************************
 *
 *  Machine drivers
 *
 *************************************/

/* Core MCR3 system with no sound */
static MACHINE_DRIVER_START( mcr3_base )

	/* basic machine hardware */
	MDRV_CPU_ADD("maincpu", Z80, MASTER_CLOCK/4)
	MDRV_CPU_CONFIG(mcr_daisy_chain)
	MDRV_CPU_VBLANK_INT_HACK(mcr_interrupt,2)

	MDRV_Z80CTC_ADD("ctc", MASTER_CLOCK/4 /* same as "maincpu" */, mcr_ctc_intf)

	MDRV_WATCHDOG_VBLANK_INIT(16)
	MDRV_MACHINE_START(mcr)
	MDRV_MACHINE_RESET(mcr)
	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_UPDATE_BEFORE_VBLANK)

	MDRV_SCREEN_ADD("screen", RASTER)
	MDRV_SCREEN_REFRESH_RATE(30)
	MDRV_SCREEN_VBLANK_TIME(ATTOSECONDS_IN_USEC(2500) /* not accurate */)
	MDRV_SCREEN_FORMAT(BITMAP_FORMAT_INDEXED16)
	MDRV_SCREEN_SIZE(32*16, 30*16)
	MDRV_SCREEN_VISIBLE_AREA(0*16, 32*16-1, 0*16, 30*16-1)

	MDRV_GFXDECODE(mcr3)
	MDRV_PALETTE_LENGTH(64)

	MDRV_VIDEO_START(mcr)
	MDRV_VIDEO_UPDATE(mcr)
MACHINE_DRIVER_END


/*************************************/


/* Core MCR monoboard system with no sound */
static MACHINE_DRIVER_START( mcrmono )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mcr3_base)

	MDRV_CPU_MODIFY("maincpu")
	MDRV_CPU_PROGRAM_MAP(mcrmono_map)
	MDRV_CPU_IO_MAP(mcrmono_portmap)

	/* video hardware */
	MDRV_VIDEO_START(mcrmono)
	MDRV_VIDEO_UPDATE(mcr3)
MACHINE_DRIVER_END


/* Sarge/Demolition Derby Mono/Max RPM = MCR monoboard with Turbo Chip Squeak */
static MACHINE_DRIVER_START( mono_tcs )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mcrmono)
	MDRV_IMPORT_FROM(turbo_chip_squeak)
MACHINE_DRIVER_END


/* Rampage/Power Drive/Star Guards = MCR monoboard with Sounds Good */
static MACHINE_DRIVER_START( mono_sg )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mcrmono)
	MDRV_IMPORT_FROM(sounds_good)
MACHINE_DRIVER_END


/*************************************/


/* Core scrolling system with SSIO sound */
static MACHINE_DRIVER_START( mcrscroll )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mcr3_base)
	MDRV_IMPORT_FROM(mcr_ssio)

	MDRV_CPU_MODIFY("maincpu")
	MDRV_CPU_PROGRAM_MAP(spyhunt_map)
	MDRV_CPU_IO_MAP(spyhunt_portmap)

	/* video hardware */
	MDRV_SCREEN_MODIFY("screen")
	MDRV_SCREEN_SIZE(30*16, 30*16)
	MDRV_SCREEN_VISIBLE_AREA(0, 30*16-1, 0, 30*16-1)
	MDRV_GFXDECODE(spyhunt)
	MDRV_PALETTE_LENGTH(64+4)

	MDRV_PALETTE_INIT(spyhunt)
	MDRV_VIDEO_START(spyhunt)
	MDRV_VIDEO_UPDATE(spyhunt)
MACHINE_DRIVER_END


/* Spy Hunter = scrolling system with an SSIO and a chip squeak deluxe */
static MACHINE_DRIVER_START( mcrsc_csd )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mcrscroll)
	MDRV_IMPORT_FROM(chip_squeak_deluxe_stereo)
MACHINE_DRIVER_END



/*************************************
 *
 *  ROM definitions
 *
 *************************************/

ROM_START( demoderm )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pro0.3b",      0x00000, 0x8000, CRC(2e24527b) SHA1(df8d1129b52ca0f4326c7c7e4f10d81b4ced65b5) )
	ROM_LOAD( "pro1.5b",      0x08000, 0x8000, CRC(034c00fc) SHA1(0f0e8f123a34c330021bce76ed7f38bcb2e9af4e) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )	/* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x10000, "tcscpu", 0 )	/* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "tcs_u5.bin",   0x0c000, 0x2000, CRC(eca33b2c) SHA1(938b021ea3b0f23aed7a98a930a58af371a02303) )
	ROM_LOAD( "tcs_u4.bin",   0x0e000, 0x2000, CRC(3490289a) SHA1(a9d56ea60bb901267da41ab408f8e1ed3742b0ac) )

	ROM_REGION( 0x04000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg0.15a",      0x00000, 0x2000, CRC(a35d13b8) SHA1(18d2d900a787e082ba30c13034ed6c64305a79b4) )
	ROM_LOAD( "bg1.14b",      0x02000, 0x2000, CRC(22ca93f3) SHA1(8a907ee6d1fd88d472d868eb47e7ae7d44097e67) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "dd_fg-0.a4",   0x00000, 0x4000, CRC(e57a4de6) SHA1(d1b2396a85b984e171d751ef8e1cf970ac4ff9fb) )
	ROM_LOAD( "dd_fg-4.a3",   0x04000, 0x4000, CRC(55aa667f) SHA1(d611dbf9e8ef383d02514b0edb9ea36670193bf0) )
	ROM_LOAD( "dd_fg-1.a6",   0x08000, 0x4000, CRC(70259651) SHA1(55967aaf2a7617c8f5a199d1e07128d79ce16970) )
	ROM_LOAD( "dd_fg-5.a5",   0x0c000, 0x4000, CRC(5fe99007) SHA1(9d640b4715333efdc6300dc353991d6934929399) )
	ROM_LOAD( "dd_fg-2.a8",   0x10000, 0x4000, CRC(6cab7b95) SHA1(8faff7458ab5ff2dd096dd78b1449a4096cc6345) )
	ROM_LOAD( "dd_fg-6.a7",   0x14000, 0x4000, CRC(abfb9a8b) SHA1(14ab416bc76db25ad97353c9072048c64ec95344) )
	ROM_LOAD( "dd_fg-3.a10",  0x18000, 0x4000, CRC(801d9b86) SHA1(5a8c72d1060eea1a3ad67b98aa6eff13f6837af6) )
	ROM_LOAD( "dd_fg-7.a9",   0x1c000, 0x4000, CRC(0ec3f60a) SHA1(4176b246b0ea7bce9498c20e12678f16f7173529) )
ROM_END


ROM_START( sarge )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "cpu_3b.bin",   0x00000, 0x8000, CRC(da31a58f) SHA1(29b97caf61f8f59042519a6b501cd1d15099dd59) )
	ROM_LOAD( "cpu_5b.bin",   0x08000, 0x8000, CRC(6800e746) SHA1(018c2b622b3654530ebc2c299b3f745777163d4b) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )	/* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x10000, "tcscpu", 0 )  /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "tcs_u5.bin",   0xc000, 0x2000, CRC(a894ef8a) SHA1(7f53927fc185fff8ba1b1747f0d565e089d879e6) )
	ROM_LOAD( "tcs_u4.bin",   0xe000, 0x2000, CRC(6ca6faf3) SHA1(4647e633dd11f55a65c3acf81adeb3af93624991) )

	ROM_REGION( 0x04000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "til_15a.bin",  0x00000, 0x2000, CRC(685001b8) SHA1(300abd808efe5b44b022082eebc591c7c255697c) )
	ROM_LOAD( "til_14b.bin",  0x02000, 0x2000, CRC(8449eb45) SHA1(6cc43639998d55fe7ffac7e9b091d35ea169e048) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "spr_8e.bin",   0x00000, 0x8000, CRC(93fac29d) SHA1(3d144208eca3b5377689e69da4505187a3d20d4f) )
	ROM_LOAD( "spr_6e.bin",   0x08000, 0x8000, CRC(7cc6fb28) SHA1(5fe24d5114551b4a4bf303cd612da332555c0f93) )
	ROM_LOAD( "spr_5e.bin",   0x10000, 0x8000, CRC(c832375c) SHA1(dfb7782b13e1e959e0ecd5da771cd38962f6952b) )
	ROM_LOAD( "spr_4e.bin",   0x18000, 0x8000, CRC(c382267d) SHA1(6b459e9ec7948a529b5308357851a0bede085aef) )

    ROM_REGION( 0x0007, "pals", 0) /* PAL's located on the Mono Board (91787) */
    ROM_LOAD( "a59a26axlcxhd.13j.bin",  0x0000, 0x0001, NO_DUMP ) /* PLS153N */
    ROM_LOAD( "a59a26axlbxhd.2j.bin",   0x0000, 0x0001, NO_DUMP ) /* PLS153N */
    ROM_LOAD( "a59a26axlaxhd.3j.bin",   0x0000, 0x0001, NO_DUMP ) /* PLS153N */
    ROM_LOAD( "0066-314bx-xxqx.6h.bin", 0x0000, 0x0001, NO_DUMP ) /* Unknown PAL Type */
    ROM_LOAD( "0066-316bx-xxqx.5h.bin", 0x0000, 0x0001, NO_DUMP ) /* Unknown PAL Type */
    ROM_LOAD( "0066-315bx-xxqx.5g.bin", 0x0000, 0x0001, NO_DUMP ) /* Unknown PAL Type */
    ROM_LOAD( "0066-313bx-xxqx.4g.bin", 0x0000, 0x0001, NO_DUMP ) /* Unknown PAL Type */
ROM_END


ROM_START( maxrpm )
	ROM_REGION( 0x12000, "maincpu", 0 )
	ROM_LOAD( "pro.0",        0x00000, 0x8000, CRC(3f9ec35f) SHA1(ebdcf480aee5569af631106cc6478adc26c4ac24) )
	ROM_LOAD( "pro.1",        0x08000, 0x8000, CRC(f628bb30) SHA1(0514906b9764a7f02a730a610affea4d42a4510d) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )	/* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x10000, "tcscpu", 0 )  /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "turbskwk.u5",   0x8000, 0x4000, CRC(55c3b759) SHA1(89d690a007a996e9c7df7b365942e4da755d15d7) )
	ROM_LOAD( "turbskwk.u4",   0xc000, 0x4000, CRC(31a2da2e) SHA1(582434b5d6bc8e84f39191976d8aa0ef9245f253) )

	ROM_REGION( 0x08000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg-0",         0x00000, 0x4000, CRC(e3fb693a) SHA1(9543c099cae4f56ef09f4e678891c90ef55928ed) )
	ROM_LOAD( "bg-1",         0x04000, 0x4000, CRC(50d1db6c) SHA1(5ef669b868edf2d0b7f16879523318e8d6a3f9b3) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "fg-0",         0x00000, 0x8000, CRC(1d1435c1) SHA1(6a53ef047bb763ea30d2e8098cb1dd6b337ccf4f) )
	ROM_LOAD( "fg-1",         0x08000, 0x8000, CRC(e54b7f2a) SHA1(b95f9d71ec98e60db57ce57006c013e5a521d8ab) )
	ROM_LOAD( "fg-2",         0x10000, 0x8000, CRC(38be8505) SHA1(5e01e5e6317dd722d19f23863908ffc5833f22c3) )
	ROM_LOAD( "fg-3",         0x18000, 0x8000, CRC(9ae3eb52) SHA1(a96835caece971692790cd7385ab618373eb821f) )
ROM_END


ROM_START( rampage )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pro0rev3.3b",  0x00000, 0x8000, CRC(2f7ca03c) SHA1(1e3a1f213fd67938adf14ea0d04dab687ea8f4ef) )
	ROM_LOAD( "pro1rev3.5b",  0x08000, 0x8000, CRC(d89bd9a4) SHA1(3531464ffe49dfaf2755d9e2dc1aea23819b3a5d) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )	/* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x40000, "sgcpu", 0 )  /* 256k for the Sounds Good board */
	ROM_LOAD16_BYTE( "ramp_u7.snd",  0x00000, 0x8000, CRC(cffd7fa5) SHA1(7c5cecce1d428f847fea37d53eb09c6f62055c6f) )	/* these are Revision 2 sound ROMs */
	ROM_LOAD16_BYTE( "ramp_u17.snd", 0x00001, 0x8000, CRC(e92c596b) SHA1(4e2d87398f2e7b637cbad6cb16d832dfa8f8288c) )
	ROM_LOAD16_BYTE( "ramp_u8.snd",  0x10000, 0x8000, CRC(11f787e4) SHA1(1fa195bf9169608099d17be5801738a4e17bec3d) )
	ROM_LOAD16_BYTE( "ramp_u18.snd", 0x10001, 0x8000, CRC(6b8bf5e1) SHA1(aa8c0260dcd19a795bfc23197cd87348a685d20b) )

	ROM_REGION( 0x08000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg-0",         0x00000, 0x04000, CRC(c0d8b7a5) SHA1(692219388a3124fb48db7e35c4127b0fe066a289) )
	ROM_LOAD( "bg-1",         0x04000, 0x04000, CRC(2f6e3aa1) SHA1(ae86ce90bb6bf660e38c0f91e7ce90d44be82d60) )

	ROM_REGION( 0x40000, "gfx2", 0 )
	ROM_LOAD( "fg-0",         0x00000, 0x10000, CRC(0974be5d) SHA1(be347faaa345383dc6e5c2b3789c372d6bd25905) )
	ROM_LOAD( "fg-1",         0x10000, 0x10000, CRC(8728532b) SHA1(327df92db7e3506b827d497859980cd2de51f45d) )
	ROM_LOAD( "fg-2",         0x20000, 0x10000, CRC(9489f714) SHA1(df17a45cdc6a9310856d64f89954be79bbeac12e) )
	ROM_LOAD( "fg-3",         0x30000, 0x10000, CRC(81e1de40) SHA1(7e7818792845ec3687b3202eeade60a298ef513e) )
ROM_END


ROM_START( rampage2 )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pro0rev2.3b",  0x00000, 0x8000, CRC(3f1d0293) SHA1(d68f04b9b3fc377b9e57b823db4e7f9cfedbcf99) )
	ROM_LOAD( "pro1rev2.5b",  0x08000, 0x8000, CRC(58523d75) SHA1(5cd512864568ec7793bda0164f21e7d72a7ea817) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )	/* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x40000, "sgcpu", 0 )  /* 256k for the Sounds Good board */
	ROM_LOAD16_BYTE( "ramp_u7.snd",  0x00000, 0x8000, CRC(cffd7fa5) SHA1(7c5cecce1d428f847fea37d53eb09c6f62055c6f) )    /* these are Revision 2 sound ROMs */
	ROM_LOAD16_BYTE( "ramp_u17.snd", 0x00001, 0x8000, CRC(e92c596b) SHA1(4e2d87398f2e7b637cbad6cb16d832dfa8f8288c) )
	ROM_LOAD16_BYTE( "ramp_u8.snd",  0x10000, 0x8000, CRC(11f787e4) SHA1(1fa195bf9169608099d17be5801738a4e17bec3d) )
	ROM_LOAD16_BYTE( "ramp_u18.snd", 0x10001, 0x8000, CRC(6b8bf5e1) SHA1(aa8c0260dcd19a795bfc23197cd87348a685d20b) )

	ROM_REGION( 0x08000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg-0",         0x00000, 0x04000, CRC(c0d8b7a5) SHA1(692219388a3124fb48db7e35c4127b0fe066a289) )
	ROM_LOAD( "bg-1",         0x04000, 0x04000, CRC(2f6e3aa1) SHA1(ae86ce90bb6bf660e38c0f91e7ce90d44be82d60) )

	ROM_REGION( 0x40000, "gfx2", 0 )
	ROM_LOAD( "fg-0",         0x00000, 0x10000, CRC(0974be5d) SHA1(be347faaa345383dc6e5c2b3789c372d6bd25905) )
	ROM_LOAD( "fg-1",         0x10000, 0x10000, CRC(8728532b) SHA1(327df92db7e3506b827d497859980cd2de51f45d) )
	ROM_LOAD( "fg-2",         0x20000, 0x10000, CRC(9489f714) SHA1(df17a45cdc6a9310856d64f89954be79bbeac12e) )
	ROM_LOAD( "fg-3",         0x30000, 0x10000, CRC(81e1de40) SHA1(7e7818792845ec3687b3202eeade60a298ef513e) )
ROM_END


ROM_START( powerdrv )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pdrv3b.bin",   0x00000, 0x8000, CRC(d870b704) SHA1(300d6ff3c92a45f5c4f28c171280174924aecb6c) )
	ROM_LOAD( "pdrv5b.bin",   0x08000, 0x8000, CRC(fa0544ad) SHA1(55a9cf8c8648761443e4a5a3b214f4d6236cbaff) )
	ROM_FILL(                 0x0e000, 0x2000, 0xff )	/* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x40000, "sgcpu", 0 )  /* 256k for the Sounds Good board */
	ROM_LOAD16_BYTE( "pdsndu7.bin",  0x00000, 0x8000, CRC(78713e78) SHA1(11382c024536f743e051ba208ae02d0f5e07cf5e) )
	ROM_LOAD16_BYTE( "pdsndu17.bin", 0x00001, 0x8000, CRC(c41de6e4) SHA1(0391afd96ee80dd1d4a34e661e5df1e01fbbd57a) )
	ROM_LOAD16_BYTE( "pdsndu8.bin",  0x10000, 0x8000, CRC(15714036) SHA1(77ca5f703eb7f146e13d9c01f4427f6aaa31df39) )
	ROM_LOAD16_BYTE( "pdsndu18.bin", 0x10001, 0x8000, CRC(cae14c70) SHA1(04e92f1f144cc8ff13a09a3d38aa65ac05c41c0b) )

	ROM_REGION( 0x08000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "pdrv15a.bin",  0x00000, 0x04000, CRC(b858b5a8) SHA1(da622bde13c7156a826d658e176feccf18f33a4b) )
	ROM_LOAD( "pdrv14b.bin",  0x04000, 0x04000, CRC(12ee7fc2) SHA1(aca7b20efa8b0e2217691fd8de6adf68a2048331) )

	ROM_REGION( 0x40000, "gfx2", 0 )
	ROM_LOAD( "pdrv8e.bin",   0x00000, 0x10000, CRC(dd3a2adc) SHA1(0792591eb55603c809d08ee622ecb2c9f5731038) )
	ROM_LOAD( "pdrv6e.bin",   0x10000, 0x10000, CRC(1a1f7f81) SHA1(d66ff3ef3855e086e665531aef0e079842c48fcb) )
	ROM_LOAD( "pdrv5e.bin",   0x20000, 0x10000, CRC(4cb4780e) SHA1(e1fad431257d9ec5e35175e4dc09d21e36ba1fa0) )
	ROM_LOAD( "pdrv4e.bin",   0x30000, 0x10000, CRC(de400335) SHA1(49438bc7c2ba236dcbd4ddc3c985460887dcf110) )
ROM_END


ROM_START( stargrds )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pro-0.3b",  0x00000, 0x8000, CRC(3ad94aa2) SHA1(1e17ac70fddee1f3d0dd71172e15a7a256168a70) )
	ROM_LOAD( "pro-1.5b",  0x08000, 0x8000, CRC(dba428b0) SHA1(72efa2f02e95f05a5503ced136fbdf3fcdf57554) )
	ROM_FILL(              0x0e000, 0x2000, 0xff )	/* upper 8k is not mapped on monoboard */

	ROM_REGION( 0x40000, "sgcpu", 0 )  /* 256k for the Sounds Good board */
	ROM_LOAD16_BYTE( "snd0.u7",      0x00000, 0x8000, CRC(7755a493) SHA1(a888fba45a2a31de5b3082bfc5ccef94dafc4d16) )
	ROM_LOAD16_BYTE( "snd1.u17",     0x00001, 0x8000, CRC(d98d14ae) SHA1(51dbb97655ab8a389ca67f0e796ab57894f5bb32) )

	ROM_REGION( 0x10000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg-0.15a",     0x00000, 0x08000, CRC(ecfaef3e) SHA1(145b0543552e678ef441e4a20afe80cd21e43cf6) )
	ROM_LOAD( "bg-1.14b",     0x08000, 0x08000, CRC(2c75569d) SHA1(13808ff0fdb413a4ac6ddef1422b86c118277899) )

	ROM_REGION( 0x40000, "gfx2", 0 )
	ROM_LOAD( "fg-0.8e",         0x00000, 0x10000, CRC(22797aaa) SHA1(806ea29f8a43f104d3154322f492e7fa053da751) )
	ROM_LOAD( "fg-1.6e",         0x10000, 0x10000, CRC(413fa119) SHA1(cb609db2e6694a5bb563afd6b28c19afe65a8487) )
	ROM_LOAD( "fg-2.5e",         0x20000, 0x10000, CRC(7036cfe6) SHA1(7778c5a8bf457ba0c1fa0e17978709e1f7ccb8a5) )
	ROM_LOAD( "fg-3.4e",         0x30000, 0x10000, CRC(cc5cc003) SHA1(afe8a0b0542c6f98de9386a07d6586af7de6bdf6) )
ROM_END


ROM_START( spyhunt )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "cpu_pg0.6d",   0x0000, 0x2000, CRC(1721b88f) SHA1(c7a641f0c05bd343ebc79e1c1be3a26da5fb77f0) )
	ROM_LOAD( "cpu_pg1.7d",   0x2000, 0x2000, CRC(909d044f) SHA1(67237c3efde568d52e9f8b0d36df726d05a9d9e4) )
	ROM_LOAD( "cpu_pg2.8d",   0x4000, 0x2000, CRC(afeeb8bd) SHA1(fde32863d08a745dfe19f1c1382810eab6aebcec) )
	ROM_LOAD( "cpu_pg3.9d",   0x6000, 0x2000, CRC(5e744381) SHA1(5b75e4f44dfd63d6e35294c606b84231c216e57d) )
	ROM_LOAD( "cpu_pg4.10d",  0x8000, 0x2000, CRC(a3033c15) SHA1(e9811450a7c952561912777d679fe45a6b5a794a) )
	ROM_LOAD( "cpu_pg5.11d",  0xa000, 0x4000, CRC(53a4f7cd) SHA1(051b07ae993e14b329507710c0f7cadaa952f9ae) )

	ROM_REGION( 0x10000, "ssiocpu", 0 )
	ROM_LOAD( "snd_0sd.a8",   0x0000, 0x1000, CRC(c95cf31e) SHA1(d1b0e299a27e306ddbc0654fd3a9d981c92afe8c) )
	ROM_LOAD( "snd_1sd.a7",   0x1000, 0x1000, CRC(12aaa48e) SHA1(c6b835fc45e4484a4d52b682ce015caa242c8b4f) )

	ROM_REGION( 0x8000, "csdcpu", 0 )  /* 32k for the Chip Squeak Deluxe */
	ROM_LOAD16_BYTE( "csd_u7a.u7",   0x00000, 0x2000, CRC(6e689fe7) SHA1(38ad2e9f12b9d389fb2568ebcb32c8bd1ac6879e) )
	ROM_LOAD16_BYTE( "csd_u17b.u17", 0x00001, 0x2000, CRC(0d9ddce6) SHA1(d955c0e67fc78b517cc229601ab4023cc5a644c2) )
	ROM_LOAD16_BYTE( "csd_u8c.u8",   0x04000, 0x2000, CRC(35563cd0) SHA1(5708d374dd56758194c95118f096ea51bf12bf64) )
	ROM_LOAD16_BYTE( "csd_u18d.u18", 0x04001, 0x2000, CRC(63d3f5b1) SHA1(5864a7e9b6bc3d2df6891d40965a7a0efbba6837) )

	ROM_REGION( 0x08000, "gfx1", 0 )
	ROM_LOAD( "cpu_bg0.3a",   0x00000, 0x2000, CRC(dea34fed) SHA1(cbbb2ba75e087eebdce79a0016118c327c8f0a96) )
	ROM_LOAD( "cpu_bg1.4a",   0x02000, 0x2000, CRC(8f64525f) SHA1(d457d12f31a30deb3b4e5b8189c9414aac1ad701) )
	ROM_LOAD( "cpu_bg2.5a",   0x04000, 0x2000, CRC(ba0fd626) SHA1(f39281feb3fbbbd4234fbb70ee77bab3e1a33e3b) )
	ROM_LOAD( "cpu_bg3.6a",   0x06000, 0x2000, CRC(7b482d61) SHA1(f6a46690f69a7513a7fbacd0199946f600d796dd) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "vid_1fg.a7",   0x00000, 0x4000, CRC(9fe286ec) SHA1(d72cd7e69ef78e25cf5bc599fb0a7da11bf4657f) )
	ROM_LOAD( "vid_0fg.a8",   0x04000, 0x4000, CRC(292c5466) SHA1(5abb9e2cc592adf81f12bf8ebeaf3e2931a7fa6d) )
	ROM_LOAD( "vid_3fg.a5",   0x08000, 0x4000, CRC(b894934d) SHA1(e7d6db1635d244d002054dd223a2d0713316ef77) )
	ROM_LOAD( "vid_2fg.a6",   0x0c000, 0x4000, CRC(62c8bfa5) SHA1(f245e49c178f846b647d09c32aa97d61333bdd83) )
	ROM_LOAD( "vid_5fg.a3",   0x10000, 0x4000, CRC(2d9fbcec) SHA1(d73862b974726fe50bf011ea7977f8229b8a1e24) )
	ROM_LOAD( "vid_4fg.a4",   0x14000, 0x4000, CRC(7ca4941b) SHA1(068ecd1e91ecfedba2ae542062f8f51f1329725d) )
	ROM_LOAD( "vid_7fg.a1",   0x18000, 0x4000, CRC(940fe17e) SHA1(60d07c10ef5867875d47a4edaa68934e37e2a0aa) )
	ROM_LOAD( "vid_6fg.a2",   0x1c000, 0x4000, CRC(8cb8a066) SHA1(5fa88d471ed8fd18244dd21b976c86530f57c8ac) )

	ROM_REGION( 0x01000, "gfx3", 0 )
	ROM_LOAD( "cpu_alph.10g", 0x00000, 0x1000, CRC(936dc87f) SHA1(cdf73bea82481fbc300ec5a1fbbe8d662007c56b) )

	ROM_REGION( 0x0020, "proms", 0 )
	ROM_LOAD( "82s123.12d",   0x0000, 0x0020, CRC(e1281ee9) SHA1(9ac9b01d24affc0ee9227a4364c4fd8f8290343a) )	/* from shollow, assuming it's the same */
ROM_END


ROM_START( spyhuntp )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "sh_mb_2.bin",  0x0000, 0x2000, CRC(df6cd642) SHA1(e695aa3e3dc3b9df87bf2c1bcb75edbdf03fde98) )
	ROM_LOAD( "cpu_pg1.7d",   0x2000, 0x2000, CRC(909d044f) SHA1(67237c3efde568d52e9f8b0d36df726d05a9d9e4) )
	ROM_LOAD( "cpu_pg2.8d",   0x4000, 0x2000, CRC(afeeb8bd) SHA1(fde32863d08a745dfe19f1c1382810eab6aebcec) )
	ROM_LOAD( "cpu_pg3.9d",   0x6000, 0x2000, CRC(5e744381) SHA1(5b75e4f44dfd63d6e35294c606b84231c216e57d) )
	ROM_LOAD( "sh_mb_6.bin",  0x8000, 0x2000, CRC(8cbf04d8) SHA1(e45cb36935367e46ea41de0177b3453cd8bdce85) )
	ROM_LOAD( "cpu_pg5.11d",  0xa000, 0x4000, CRC(53a4f7cd) SHA1(051b07ae993e14b329507710c0f7cadaa952f9ae) )

	ROM_REGION( 0x10000, "ssiocpu", 0 )
	ROM_LOAD( "snd_0sd.a8",   0x0000, 0x1000, CRC(c95cf31e) SHA1(d1b0e299a27e306ddbc0654fd3a9d981c92afe8c) )
	ROM_LOAD( "snd_1sd.a7",   0x1000, 0x1000, CRC(12aaa48e) SHA1(c6b835fc45e4484a4d52b682ce015caa242c8b4f) )

	ROM_REGION( 0x8000, "csdcpu", 0 )  /* 32k for the Chip Squeak Deluxe */
	ROM_LOAD16_BYTE( "csd_u7a.u7",   0x00000, 0x2000, CRC(6e689fe7) SHA1(38ad2e9f12b9d389fb2568ebcb32c8bd1ac6879e) )
	ROM_LOAD16_BYTE( "csd_u17b.u17", 0x00001, 0x2000, CRC(0d9ddce6) SHA1(d955c0e67fc78b517cc229601ab4023cc5a644c2) )
	ROM_LOAD16_BYTE( "csd_u8c.u8",   0x04000, 0x2000, CRC(35563cd0) SHA1(5708d374dd56758194c95118f096ea51bf12bf64) )
	ROM_LOAD16_BYTE( "csd_u18d.u18", 0x04001, 0x2000, CRC(63d3f5b1) SHA1(5864a7e9b6bc3d2df6891d40965a7a0efbba6837) )

	ROM_REGION( 0x08000, "gfx1", 0 )
	ROM_LOAD( "cpu_bg0.3a",   0x00000, 0x2000, CRC(dea34fed) SHA1(cbbb2ba75e087eebdce79a0016118c327c8f0a96) )
	ROM_LOAD( "cpu_bg1.4a",   0x02000, 0x2000, CRC(8f64525f) SHA1(d457d12f31a30deb3b4e5b8189c9414aac1ad701) )
	ROM_LOAD( "cpu_bg2.5a",   0x04000, 0x2000, CRC(ba0fd626) SHA1(f39281feb3fbbbd4234fbb70ee77bab3e1a33e3b) )
	ROM_LOAD( "cpu_bg3.6a",   0x06000, 0x2000, CRC(7b482d61) SHA1(f6a46690f69a7513a7fbacd0199946f600d796dd) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "vid_1fg.a7",   0x00000, 0x4000, CRC(9fe286ec) SHA1(d72cd7e69ef78e25cf5bc599fb0a7da11bf4657f) )
	ROM_LOAD( "vid_0fg.a8",   0x04000, 0x4000, CRC(292c5466) SHA1(5abb9e2cc592adf81f12bf8ebeaf3e2931a7fa6d) )
	ROM_LOAD( "vid_3fg.a5",   0x08000, 0x4000, CRC(b894934d) SHA1(e7d6db1635d244d002054dd223a2d0713316ef77) )
	ROM_LOAD( "vid_2fg.a6",   0x0c000, 0x4000, CRC(62c8bfa5) SHA1(f245e49c178f846b647d09c32aa97d61333bdd83) )
	ROM_LOAD( "vid_5fg.a3",   0x10000, 0x4000, CRC(2d9fbcec) SHA1(d73862b974726fe50bf011ea7977f8229b8a1e24) )
	ROM_LOAD( "vid_4fg.a4",   0x14000, 0x4000, CRC(7ca4941b) SHA1(068ecd1e91ecfedba2ae542062f8f51f1329725d) )
	ROM_LOAD( "vid_7fg.a1",   0x18000, 0x4000, CRC(940fe17e) SHA1(60d07c10ef5867875d47a4edaa68934e37e2a0aa) )
	ROM_LOAD( "vid_6fg.a2",   0x1c000, 0x4000, CRC(8cb8a066) SHA1(5fa88d471ed8fd18244dd21b976c86530f57c8ac) )

	ROM_REGION( 0x01000, "gfx3", 0 )
	ROM_LOAD( "sh_mb_1.bin",  0x00000, 0x0800, CRC(5c74c9f0) SHA1(e42789d14e5510d1dcf4a30f6bd40a40ab46f7f3) )

	ROM_REGION( 0x0020, "proms", 0 )
	ROM_LOAD( "82s123.12d",   0x0000, 0x0020, CRC(e1281ee9) SHA1(9ac9b01d24affc0ee9227a4364c4fd8f8290343a) )	/* from shollow, assuming it's the same */
ROM_END


ROM_START( crater )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "crcpu.6d",     0x0000, 0x2000, CRC(ad31f127) SHA1(d03506570cd08bcdb39d7c6b9de4f9628c7373e9) )
	ROM_LOAD( "crcpu.7d",     0x2000, 0x2000, CRC(3743c78f) SHA1(05605041c7a5ef2a8271cc3dd1aae4c82c8797eb) )
	ROM_LOAD( "crcpu.8d",     0x4000, 0x2000, CRC(c95f9088) SHA1(2162c4ef2cc29857b10d64ec9e09ca6d086fa9e7) )
	ROM_LOAD( "crcpu.9d",     0x6000, 0x2000, CRC(a03c4b11) SHA1(6a442a0828942dc51dbe0d3f19be855a52c12f39) )
	ROM_LOAD( "crcpu.10d",    0x8000, 0x2000, CRC(44ae4cbd) SHA1(2188bf697f1b3fcbb2ad6cbd4d9098e3b8d56a95) )

	ROM_REGION( 0x10000, "ssiocpu", 0 )
	ROM_LOAD( "crsnd4.a7",    0x0000, 0x1000, CRC(fd666cb5) SHA1(257174266e264db8ac9af5f2296fd0a22847f85f) )
	ROM_LOAD( "crsnd1.a8",    0x1000, 0x1000, CRC(90bf2c4c) SHA1(7adfbf2251b5d46043d614554320e2fe544cc570) )
	ROM_LOAD( "crsnd2.a9",    0x2000, 0x1000, CRC(3b8deef1) SHA1(a14186a33a7b5ca07086ce44fb1c8c64900654d0) )
	ROM_LOAD( "crsnd3.a10",   0x3000, 0x1000, CRC(05803453) SHA1(1bba85e6494b4f67cea82d61e6cd341337bca998) )

	ROM_REGION( 0x08000, "gfx1", 0 )
	ROM_LOAD( "crcpu.3a",     0x00000, 0x2000, CRC(9d73504a) SHA1(23633ce43745c12b59916b85ca92a98e889b884e) )
	ROM_LOAD( "crcpu.4a",     0x02000, 0x2000, CRC(42a47dff) SHA1(cb936ac7ba3dd740c626a11bd6ee3c87dad4e311) )
	ROM_LOAD( "crcpu.5a",     0x04000, 0x2000, CRC(2fe4a6e1) SHA1(329cf9f71b7cbf2a13a05f9cd3397d73c69da893) )
	ROM_LOAD( "crcpu.6a",     0x06000, 0x2000, CRC(d0659042) SHA1(8f28155eb2725eaf6d1c410f76a88969541aeef3) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "crvid.a4",     0x00000, 0x4000, CRC(579a8e36) SHA1(1053234b82877f0f8d1a2ce745b853899cfbcc22) )
	ROM_LOAD( "crvid.a3",     0x04000, 0x4000, CRC(2c2f5b29) SHA1(f7c4caabd116d3f861e70c6740cd058479477da7) )
	ROM_LOAD( "crvid.a6",     0x08000, 0x4000, CRC(5bf954e0) SHA1(8ff9549a0b5d7e6ad4671c29bf04bb9c26a995ad) )
	ROM_LOAD( "crvid.a5",     0x0c000, 0x4000, CRC(9bdec312) SHA1(3a3b44d4f73aad1f0ab2591be479f8472583f20a) )
	ROM_LOAD( "crvid.a8",     0x10000, 0x4000, CRC(4b913498) SHA1(8aa83fd6f60d6176a8781a906ba75c663ac3f3cb) )
	ROM_LOAD( "crvid.a7",     0x14000, 0x4000, CRC(9fa307d5) SHA1(ec9ba9c372d62132f02e2a87ed5f602c360755e9) )
	ROM_LOAD( "crvid.a10",    0x18000, 0x4000, CRC(7a22d6bc) SHA1(c7f97bdf13d52665e61d83b7a8f1621c6df0dbf2) )
	ROM_LOAD( "crvid.a9",     0x1c000, 0x4000, CRC(811f152d) SHA1(f9dee6f95d903a41890fe58bfc0528fecb1d6902) )

	ROM_REGION( 0x01000, "gfx3", 0 )
	ROM_LOAD( "crcpu.10g",    0x00000, 0x1000, CRC(6fe53c8d) SHA1(ceb04916af42d58f3173e5986756a0db8be11999) )

	ROM_REGION( 0x0020, "proms", 0 )
	ROM_LOAD( "82s123.12d",   0x0000, 0x0020, CRC(e1281ee9) SHA1(9ac9b01d24affc0ee9227a4364c4fd8f8290343a) )	/* from shollow, assuming it's the same */
ROM_END


ROM_START( turbotag )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "ttprog0.bin",  0x0000, 0x2000, CRC(6110fd80) SHA1(544d3cd24b047797c2006a9fac499c14140960db) )
	ROM_LOAD( "ttprog1.bin",  0x2000, 0x2000, BAD_DUMP CRC(b0505e18) SHA1(b6871484f9a0663a9bcffd58368be5ae7717d29c) )
	ROM_LOAD( "ttprog2.bin",  0x4000, 0x2000, CRC(c4141237) SHA1(c1d10da4961e94bd8c2b30a2f9e0cbd8080eb56d) )
	ROM_LOAD( "ttprog3.bin",  0x6000, 0x2000, CRC(af294c6e) SHA1(c8573dd046aa2b071bef3bd86a783ee348c8d6eb) )
	ROM_LOAD( "ttprog4.bin",  0x8000, 0x2000, CRC(8c5bc1a4) SHA1(c38d7aa2639945e705856cf1449faf51a7c82ff0) )
	ROM_LOAD( "ttprog5.bin",  0xa000, 0x2000, CRC(11e62fe4) SHA1(72897702c61486b654e4b4a3f6560c144c862e1f) )
	ROM_RELOAD(               0xc000, 0x2000 )

	ROM_REGION( 0x10000, "ssiocpu", ROMREGION_ERASE00 )

	ROM_REGION( 0x8000, "csdcpu", 0 )  /* 32k for the Chip Squeak Deluxe */
	ROM_LOAD16_BYTE( "ttu7.bin",  0x00000, 0x2000, CRC(8ebb3302) SHA1(c516abdae6eea524a6d2a039ed9bd7dff72ab986) )
	ROM_LOAD16_BYTE( "ttu17.bin", 0x00001, 0x2000, CRC(605d6c74) SHA1(a6c2bc95cca372fa823ab256c9dd1f92b6ba45fd) )
	ROM_LOAD16_BYTE( "ttu8.bin",  0x04000, 0x2000, CRC(6bfcb22a) SHA1(7b895e3ae1e99f195bb32b052f801b58c63a401c) )
	ROM_LOAD16_BYTE( "ttu18.bin", 0x04001, 0x2000, CRC(bb25852c) SHA1(7e0346b5e2fc16a87a157c868936978be6145e3e) )

	ROM_REGION( 0x08000, "gfx1", 0 )
	ROM_LOAD( "ttbg0.bin",    0x00000, 0x2000, CRC(1cd2023f) SHA1(03b4f9d047d4fb3c4889e797e9ffd5edabd8ecff) )
	ROM_LOAD( "ttbg1.bin",    0x02000, 0x2000, CRC(784e84cd) SHA1(cf4ddbd4b8dbfaee2f4e05c89355fad30e264641) )
	ROM_LOAD( "ttbg2.bin",    0x04000, 0x2000, CRC(da9d47d2) SHA1(5af9222f62d9948ec468fb66a5afb46a00f159c3) )
	ROM_LOAD( "ttbg3.bin",    0x06000, 0x2000, CRC(367e06a5) SHA1(24040e29bbe367b497675d155c3e0343399b179b) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "ttfg1.bin",    0x00000, 0x4000, CRC(9d7e6ebc) SHA1(96c658091cb12d65e41f8ac5f609eb51857cef82) )
	ROM_LOAD( "ttfg0.bin",    0x04000, 0x4000, CRC(ed69e1a8) SHA1(4fc223da52d6df3182ba5ba3db8e793e379030f2) )
	ROM_LOAD( "ttfg3.bin",    0x08000, 0x4000, CRC(74e21c1c) SHA1(42ea0c569cd443866b46bcaac2c60d197fcbc2a9) )
	ROM_LOAD( "ttfg2.bin",    0x0c000, 0x4000, CRC(037ec6fc) SHA1(7bc0bd95cc673d321de33d17764a06f48109b38e) )
	ROM_LOAD( "ttfg5.bin",    0x10000, 0x4000, CRC(8b718879) SHA1(3e8361e4423e6822c09f866a99cf9fc789c99f66) )
	ROM_LOAD( "ttfg4.bin",    0x14000, 0x4000, CRC(6fdb0c13) SHA1(b3e140a838f8deaa26dbc86315603fdcef47f157) )
	ROM_LOAD( "ttfg7.bin",    0x18000, 0x4000, CRC(212019dc) SHA1(831bb83f6301d9f2f06b31a00ea425b13421b573) )
	ROM_LOAD( "ttfg6.bin",    0x1c000, 0x4000, CRC(4094e996) SHA1(f50fa31bc311c16bcd2459668f5a942a1d9de75d) )

	ROM_REGION( 0x01000, "gfx3", 0 )
	ROM_LOAD( "ttan.bin",     0x00000, 0x1000, CRC(aa0b1471) SHA1(e3dd69f1a14926c6b709d6b19d9e90a1f0867465) )

	ROM_REGION( 0x0020, "proms", 0 )
	ROM_LOAD( "82s123.12d",   0x0000, 0x0020, CRC(e1281ee9) SHA1(9ac9b01d24affc0ee9227a4364c4fd8f8290343a) )	/* from shollow, assuming it's the same */
ROM_END



/*************************************
 *
 *  Driver initialization
 *
 *************************************/

static void mcr_common_init(running_machine *machine, int sound_board)
{
	mcr_sound_init(machine, sound_board);

	state_save_register_global(machine, input_mux);
	state_save_register_global(machine, latched_input);
	state_save_register_global(machine, last_op4);
}


static DRIVER_INIT( demoderm )
{
	mcr_common_init(machine, MCR_TURBO_CHIP_SQUEAK);
	memory_install_read8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x01, 0x01, 0, 0, demoderm_ip1_r);
	memory_install_read8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x02, 0x02, 0, 0, demoderm_ip2_r);
	memory_install_write8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x06, 0x06, 0, 0, demoderm_op6_w);
}


static DRIVER_INIT( sarge )
{
	mcr_common_init(machine, MCR_TURBO_CHIP_SQUEAK);
	memory_install_write8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x06, 0x06, 0, 0, turbocs_data_w);
}


static DRIVER_INIT( maxrpm )
{
	mcr_common_init(machine, MCR_TURBO_CHIP_SQUEAK);
	memory_install_read8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x01, 0x01, 0, 0, maxrpm_ip1_r);
	memory_install_read8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x02, 0x02, 0, 0, maxrpm_ip2_r);
	memory_install_write8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x05, 0x05, 0, 0, maxrpm_op5_w);
	memory_install_write8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x06, 0x06, 0, 0, maxrpm_op6_w);

	state_save_register_global(machine, maxrpm_adc_control);
	state_save_register_global(machine, maxrpm_adc_select);
	state_save_register_global(machine, maxrpm_last_shift);
	state_save_register_global(machine, maxrpm_p1_shift);
	state_save_register_global(machine, maxrpm_p2_shift);
}


static DRIVER_INIT( rampage )
{
	mcr_common_init(machine, MCR_SOUNDS_GOOD);
	memory_install_read8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x04, 0x04, 0, 0, rampage_ip4_r);
	memory_install_write8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x06, 0x06, 0, 0, rampage_op6_w);
}


static DRIVER_INIT( powerdrv )
{
	mcr_common_init(machine, MCR_SOUNDS_GOOD);
	memory_install_read8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x02, 0x02, 0, 0, powerdrv_ip2_r);
	memory_install_write8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x05, 0x05, 0, 0, powerdrv_op5_w);
	memory_install_write8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x06, 0x06, 0, 0, powerdrv_op6_w);
}


static DRIVER_INIT( stargrds )
{
	mcr_common_init(machine, MCR_SOUNDS_GOOD);
	memory_install_read8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x00, 0x00, 0, 0, stargrds_ip0_r);
	memory_install_write8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x05, 0x05, 0, 0, stargrds_op5_w);
	memory_install_write8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_IO), 0x06, 0x06, 0, 0, stargrds_op6_w);
}


static DRIVER_INIT( spyhunt )
{
	mcr_common_init(machine, MCR_SSIO | MCR_CHIP_SQUEAK_DELUXE);
	ssio_set_custom_input(1, 0x60, spyhunt_ip1_r);
	ssio_set_custom_input(2, 0xff, spyhunt_ip2_r);
	ssio_set_custom_output(4, 0xff, spyhunt_op4_w);

	spyhunt_sprite_color_mask = 0x00;
	spyhunt_scroll_offset = 16;
}


static DRIVER_INIT( crater )
{
	mcr_common_init(machine, MCR_SSIO);

	spyhunt_sprite_color_mask = 0x03;
	spyhunt_scroll_offset = 96;
}


static DRIVER_INIT( turbotag )
{
	mcr_common_init(machine, MCR_SSIO | MCR_CHIP_SQUEAK_DELUXE);
	ssio_set_custom_input(1, 0x60, spyhunt_ip1_r);
	ssio_set_custom_input(2, 0xff, turbotag_ip2_r);
	ssio_set_custom_output(4, 0xff, spyhunt_op4_w);

	spyhunt_sprite_color_mask = 0x00;
	spyhunt_scroll_offset = 88;

	/* the SSIO Z80 doesn't have any program to execute */
	cputag_suspend(machine, "csdcpu", SUSPEND_REASON_DISABLE, 1);

	/* kludge for bad ROM read */
	memory_install_read8_handler(cputag_get_address_space(machine, "maincpu", ADDRESS_SPACE_PROGRAM), 0x0b53, 0x0b53, 0, 0, turbotag_kludge_r);
}



/*************************************
 *
 *  Game drivers
 *
 *************************************/

/* MCR monoboard games */
GAME( 1984, demoderm, demoderb, mono_tcs,  demoderm, demoderm, ROT0,  "Bally Midway", "Demolition Derby (2-Player Mono Board Version)", GAME_SUPPORTS_SAVE )
GAME( 1985, sarge,    0,        mono_tcs,  sarge,    sarge,    ROT0,  "Bally Midway", "Sarge", GAME_SUPPORTS_SAVE )
GAME( 1986, maxrpm,   0,        mono_tcs,  maxrpm,   maxrpm,   ROT0,  "Bally Midway", "Max RPM", GAME_SUPPORTS_SAVE )
GAME( 1986, rampage,  0,        mono_sg,   rampage,  rampage,  ROT0,  "Bally Midway", "Rampage (revision 3)", GAME_SUPPORTS_SAVE )
GAME( 1986, rampage2, rampage,  mono_sg,   rampage,  rampage,  ROT0,  "Bally Midway", "Rampage (revision 2)", GAME_SUPPORTS_SAVE )
GAME( 1986, powerdrv, 0,        mono_sg,   powerdrv, powerdrv, ROT0,  "Bally Midway", "Power Drive", GAME_SUPPORTS_SAVE )
GAME( 1987, stargrds, 0,        mono_sg,   stargrds, stargrds, ROT0,  "Bally Midway", "Star Guards", GAME_SUPPORTS_SAVE )

/* MCR scrolling games */
GAME( 1983, spyhunt,  0,        mcrsc_csd, spyhunt,  spyhunt,  ROT90, "Bally Midway", "Spy Hunter", GAME_SUPPORTS_SAVE )
GAME( 1983, spyhuntp, spyhunt,  mcrsc_csd, spyhunt,  spyhunt,  ROT90, "Bally Midway", "Spy Hunter (Playtronic license)", GAME_SUPPORTS_SAVE )
GAME( 1984, crater,   0,        mcrscroll, crater,   crater,   ORIENTATION_FLIP_X, "Bally Midway", "Crater Raider", GAME_SUPPORTS_SAVE )
GAMEL(1985, turbotag, 0,        mcrsc_csd, turbotag, turbotag, ROT90, "Bally Midway", "Turbo Tag (prototype)", GAME_SUPPORTS_SAVE, layout_turbotag )
