/*
 * This code has been taken from the Linux kernel prior removal of the big
 * kernel lock.
 */

#include <linux/module.h>
#include <linux/kallsyms.h>
#include <linux/sched.h>
#include <linux/semaphore.h>

/****************************************************************************/
/*
 * The 'big kernel lock'
 *
 * This spinlock is taken and released recursively by lock_kernel()
 * and unlock_kernel().
 */

static  __cacheline_aligned_in_smp DEFINE_RAW_SPINLOCK(kernel_flag);

/* The lock_depth variable was formerly defined globally to allow scheduling
 * with the bkl held. As this is now only locally defined scheduling is not
 * allowed with the lock active */
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0))
static int lock_depth = -1;
#define LOCK_DEPTH lock_depth
#else
#define LOCK_DEPTH current->lock_depth
#endif


/*
 * These are the BKL spinlocks - we try to be polite about preemption.
 * If SMP is not on (ie UP preemption), this all goes away because the
 * do_raw_spin_trylock() will always succeed.
 */
#ifdef CONFIG_PREEMPT
static inline void __lock_kernel(void)
{
        preempt_disable();
        if (unlikely(!do_raw_spin_trylock(&kernel_flag))) {
                /*
                 * If preemption was disabled even before this
                 * was called, there's nothing we can be polite
                 * about - just spin.
                 */
                if (preempt_count() > 1) {
                        do_raw_spin_lock(&kernel_flag);
                        return;
                }

                /*
                 * Otherwise, let's wait for the kernel lock
                 * with preemption enabled..
                 */
                do {
                        preempt_enable();
                        while (raw_spin_is_locked(&kernel_flag))
                                cpu_relax();
                        preempt_disable();
                } while (!do_raw_spin_trylock(&kernel_flag));
        }
}

#else

/*
 * Non-preemption case - just get the spinlock
 */
static inline void __lock_kernel(void)
{
        do_raw_spin_lock(&kernel_flag);
}
#endif

static inline void __unlock_kernel(void)
{
        /*
         * the BKL is not covered by lockdep, so we open-code the
         * unlocking sequence (and thus avoid the dep-chain ops):
         */
        do_raw_spin_unlock(&kernel_flag);
        preempt_enable();
}

/*
 * Getting the big kernel lock.
 *
 * This cannot happen asynchronously, so we only need to
 * worry about other CPU's.
 */
void __lockfunc _qlock_kernel(const char *func, const char *file, int line)
{
        int depth = LOCK_DEPTH + 1;

        if (likely(!depth)) {
                might_sleep();
                __lock_kernel();
        }
        LOCK_DEPTH = depth;
}

void __lockfunc _qunlock_kernel(const char *func, const char *file, int line)
{
        BUG_ON(LOCK_DEPTH < 0);
        if (likely(--LOCK_DEPTH < 0))
                __unlock_kernel();
}
