/*
 * Decompiled with CFR 0.152.
 */
package phex.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import phex.common.FileHandlingException;
import phex.common.log.NLogger;
import phex.utils.IOUtil;
import phex.utils.StringUtils;

public final class FileUtils {
    private static final int BUFFER_LENGTH = 262144;

    private FileUtils() {
    }

    public static String getFileExtension(File file) {
        String name = file.getName();
        return FileUtils.getFileExtension(name);
    }

    public static String getFileExtension(String fileName) {
        int idx = fileName.lastIndexOf(46);
        if (idx == -1) {
            return "";
        }
        return fileName.substring(idx + 1);
    }

    public static String replaceFileExtension(String fileName, String newExtension) {
        int idx = fileName.lastIndexOf(46);
        if (idx == -1) {
            return fileName + "." + newExtension;
        }
        return fileName.substring(0, idx + 1) + newExtension;
    }

    public static String convertToLocalSystemFilename(String filename) {
        filename = StringUtils.replaceChars(filename, "\\/:*?\"<>|", '_');
        return filename.substring(0, Math.min(255, filename.length()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void appendFile(File destination, File fileToAppend) throws IOException {
        long destFileLength = destination.length();
        long appendFileLength = fileToAppend.length();
        FileInputStream inStream = new FileInputStream(fileToAppend);
        try {
            RandomAccessFile destFile = new RandomAccessFile(destination, "rwd");
            try {
                int length;
                destFile.setLength(destFileLength + appendFileLength);
                destFile.seek(destFileLength);
                byte[] buffer = new byte[(int)Math.min(262144L, appendFileLength)];
                while (-1 != (length = inStream.read(buffer))) {
                    long start2 = System.currentTimeMillis();
                    destFile.write(buffer, 0, length);
                    long end2 = System.currentTimeMillis();
                    try {
                        Thread.sleep((end2 - start2) * 2L);
                    }
                    catch (InterruptedException exp) {
                        Thread.currentThread().interrupt();
                        destFile.close();
                        IOUtil.closeQuietly(destFile);
                        IOUtil.closeQuietly(inStream);
                        return;
                    }
                }
            }
            finally {
                destFile.close();
                IOUtil.closeQuietly(destFile);
            }
        }
        finally {
            IOUtil.closeQuietly(inStream);
        }
        FileUtils.deleteFileMultiFallback(fileToAppend);
    }

    public static void copyFile(File source, File destination) throws IOException {
        FileUtils.copyFile(source, destination, source.length());
        if (source.length() != destination.length()) {
            String message = "Failed to copy full contents from " + source + " to " + destination + " - " + source.length() + "/" + destination.length();
            throw new IOException(message);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void copyFile(File source, File destination, long copyLength) throws IOException {
        if (!source.exists()) {
            String message = "File " + source + " does not exist";
            throw new FileNotFoundException(message);
        }
        if (destination.getParentFile() != null && !destination.getParentFile().exists()) {
            FileUtils.forceMkdir(destination.getParentFile());
        }
        if (destination.exists() && !destination.canWrite()) {
            String message = "Unable to open file " + destination + " for writing.";
            throw new IOException(message);
        }
        if (source.getCanonicalPath().equals(destination.getCanonicalPath())) {
            String message = "Unable to write file " + source + " on itself.";
            throw new IOException(message);
        }
        if (copyLength == 0L) {
            FileUtils.truncateFile(destination, 0L);
        }
        FileInputStream input = null;
        FileOutputStream output = null;
        try {
            int read;
            long lengthLeft;
            input = new FileInputStream(source);
            output = new FileOutputStream(destination);
            byte[] buffer = new byte[(int)Math.min(262144L, lengthLeft + 1L)];
            for (lengthLeft = copyLength; lengthLeft > 0L && (read = input.read(buffer)) != -1; lengthLeft -= (long)read) {
                output.write(buffer, 0, read);
            }
            output.flush();
            output.getFD().sync();
        }
        catch (Throwable throwable) {
            IOUtil.closeQuietly(input);
            IOUtil.closeQuietly(output);
            throw throwable;
        }
        IOUtil.closeQuietly(input);
        IOUtil.closeQuietly(output);
        destination.setLastModified(source.lastModified());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void splitFile(File source, File destination, long splitPoint) throws IOException {
        RandomAccessFile sourceFile = new RandomAccessFile(source, "rws");
        try {
            FileOutputStream outStream = new FileOutputStream(destination);
            try {
                int length;
                sourceFile.seek(splitPoint);
                byte[] buffer = new byte[(int)Math.min(262144L, source.length() + 1L)];
                while (-1 != (length = sourceFile.read(buffer))) {
                    outStream.write(buffer, 0, length);
                }
                sourceFile.setLength(splitPoint);
            }
            finally {
                IOUtil.closeQuietly(outStream);
            }
        }
        finally {
            IOUtil.closeQuietly(sourceFile);
        }
    }

    public static void renameFileMultiFallback(File sourceFile, File destFile) throws FileHandlingException {
        if (destFile.exists()) {
            throw new FileHandlingException(1);
        }
        if (!sourceFile.exists()) {
            return;
        }
        boolean succ = sourceFile.renameTo(destFile);
        if (succ) {
            NLogger.warn(FileUtils.class, "First renameTo operation worked!");
            return;
        }
        NLogger.warn(FileUtils.class, "First renameTo operation failed.");
        System.gc();
        Thread.yield();
        succ = sourceFile.renameTo(destFile);
        if (succ) {
            return;
        }
        NLogger.warn(FileUtils.class, "Second renameTo operation failed.");
        FileInputStream input = null;
        FileOutputStream output = null;
        try {
            int read;
            long lengthLeft;
            input = new FileInputStream(sourceFile);
            output = new FileOutputStream(destFile);
            byte[] buffer = new byte[(int)Math.min(262144L, lengthLeft + 1L)];
            for (lengthLeft = sourceFile.length(); lengthLeft > 0L && (read = input.read(buffer)) != -1; lengthLeft -= (long)read) {
                output.write(buffer, 0, read);
            }
        }
        catch (IOException exp) {
            try {
                NLogger.warn(FileUtils.class, "Third renameTo operation failed.");
                throw new FileHandlingException(0, (Throwable)exp);
            }
            catch (Throwable throwable) {
                IOUtil.closeQuietly(input);
                IOUtil.closeQuietly(output);
                throw throwable;
            }
        }
        IOUtil.closeQuietly(input);
        IOUtil.closeQuietly(output);
        destFile.setLastModified(sourceFile.lastModified());
        FileUtils.deleteFileMultiFallback(sourceFile);
    }

    public static void deleteFileMultiFallback(File file) {
        boolean succ = file.delete();
        if (succ) {
            return;
        }
        NLogger.warn(FileUtils.class, "First delete operation failed.");
        System.gc();
        Thread.yield();
        succ = file.delete();
        if (succ) {
            return;
        }
        NLogger.warn(FileUtils.class, "Second delete operation failed.");
        file.deleteOnExit();
        try {
            FileUtils.truncateFile(file, 0L);
        }
        catch (IOException exp) {
            NLogger.warn(FileUtils.class, "Delete/truncate operation failed.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void truncateFile(File file, long size) throws IOException {
        if (size < 0L) {
            throw new IllegalArgumentException("File size < 0: " + size);
        }
        if (file.exists() && file.length() > size) {
            RandomAccessFile raf = null;
            try {
                raf = new RandomAccessFile(file, "rws");
                raf.setLength(size);
            }
            catch (Throwable throwable) {
                IOUtil.closeQuietly(raf);
                throw throwable;
            }
            IOUtil.closeQuietly(raf);
        }
    }

    public static boolean isChildOfDir(File maybeChild, File maybeParentDir) {
        if (!maybeChild.getAbsolutePath().startsWith(maybeParentDir.getAbsolutePath())) {
            return false;
        }
        return FileUtils.isChildOfDirInternal(maybeChild, maybeParentDir);
    }

    private static boolean isChildOfDirInternal(File maybeChild, File maybeParentDir) {
        File parent = maybeChild.getParentFile();
        if (parent == null) {
            return false;
        }
        if (parent.equals(maybeParentDir)) {
            return true;
        }
        return FileUtils.isChildOfDirInternal(parent, maybeParentDir);
    }

    public static void copyDirectory(File srcDir, File destDir, boolean preserveFileDate) throws IOException {
        if (srcDir == null) {
            throw new NullPointerException("Source must not be null");
        }
        if (destDir == null) {
            throw new NullPointerException("Destination must not be null");
        }
        if (!srcDir.exists()) {
            throw new FileNotFoundException("Source '" + srcDir + "' does not exist");
        }
        if (!srcDir.isDirectory()) {
            throw new IOException("Source '" + srcDir + "' exists but is not a directory");
        }
        if (srcDir.getCanonicalPath().equals(destDir.getCanonicalPath())) {
            throw new IOException("Source '" + srcDir + "' and destination '" + destDir + "' are the same");
        }
        FileUtils.doCopyDirectory(srcDir, destDir, preserveFileDate);
    }

    private static void doCopyDirectory(File srcDir, File destDir, boolean preserveFileDate) throws IOException {
        if (destDir.exists()) {
            if (!destDir.isDirectory()) {
                throw new IOException("Destination '" + destDir + "' exists but is not a directory");
            }
        } else {
            FileUtils.forceMkdir(destDir);
            if (preserveFileDate) {
                destDir.setLastModified(srcDir.lastModified());
            }
        }
        if (!destDir.canWrite()) {
            throw new IOException("Destination '" + destDir + "' cannot be written to");
        }
        File[] files = srcDir.listFiles();
        if (files == null) {
            throw new IOException("Failed to list contents of " + srcDir);
        }
        for (int i = 0; i < files.length; ++i) {
            File copiedFile = new File(destDir, files[i].getName());
            if (files[i].isDirectory()) {
                FileUtils.doCopyDirectory(files[i], copiedFile, preserveFileDate);
                continue;
            }
            FileUtils.doCopyFile(files[i], copiedFile, preserveFileDate);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void doCopyFile(File srcFile, File destFile, boolean preserveFileDate) throws IOException {
        if (destFile.exists() && destFile.isDirectory()) {
            throw new IOException("Destination '" + destFile + "' exists but is a directory");
        }
        FileChannel input = new FileInputStream(srcFile).getChannel();
        try {
            FileChannel output = new FileOutputStream(destFile).getChannel();
            try {
                output.transferFrom(input, 0L, input.size());
            }
            finally {
                IOUtil.closeQuietly(output);
            }
        }
        finally {
            IOUtil.closeQuietly(input);
        }
        if (srcFile.length() != destFile.length()) {
            throw new IOException("Failed to copy full contents from '" + srcFile + "' to '" + destFile + "'");
        }
        if (preserveFileDate) {
            destFile.setLastModified(srcFile.lastModified());
        }
    }

    public static void deleteDirectory(File directory) throws IOException {
        if (!directory.exists()) {
            return;
        }
        FileUtils.cleanDirectory(directory);
        if (!directory.delete()) {
            String message = "Unable to delete directory " + directory + ".";
            throw new IOException(message);
        }
    }

    public static void cleanDirectory(File directory) throws IOException {
        if (!directory.exists()) {
            String message = directory + " does not exist";
            throw new IllegalArgumentException(message);
        }
        if (!directory.isDirectory()) {
            String message = directory + " is not a directory";
            throw new IllegalArgumentException(message);
        }
        File[] files = directory.listFiles();
        if (files == null) {
            throw new IOException("Failed to list contents of " + directory);
        }
        IOException exception = null;
        for (int i = 0; i < files.length; ++i) {
            File file = files[i];
            try {
                FileUtils.forceDelete(file);
                continue;
            }
            catch (IOException ioe) {
                exception = ioe;
            }
        }
        if (null != exception) {
            throw exception;
        }
    }

    public static void forceDelete(File file) throws IOException {
        if (file.isDirectory()) {
            FileUtils.deleteDirectory(file);
        } else {
            if (!file.exists()) {
                throw new FileNotFoundException("File does not exist: " + file);
            }
            if (!file.delete()) {
                String message = "Unable to delete file: " + file;
                throw new IOException(message);
            }
        }
    }

    public static void forceMkdir(File directory) throws IOException {
        if (directory.exists()) {
            if (directory.isFile()) {
                String message = "File " + directory + " exists and is " + "not a directory. Unable to create directory.";
                throw new IOException(message);
            }
        } else if (!directory.mkdirs()) {
            String message = "Unable to create directory " + directory;
            throw new IOException(message);
        }
    }
}

