/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2000-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at packager/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

/*
 * @(#)TransportProtocolHandler.hpp	1.3 06/26/07
 */ 

#ifndef TRANSPORTPROTOCOLHANDLER_HPP
#define TRANSPORTPROTOCOLHANDLER_HPP

#include "../debug/DebugUtils.h"
#include "../containers/Properties.hpp"
#include "../error/ErrorCodes.h"
#include "../net/IPAddress.hpp"
#include "iMQConstants.hpp"
#include "../basictypes/Object.hpp"
#include <nspr.h>


// These constants should be declared within TransportProtocolHandler,
// but C++ only allows const static INTEGRAL data members to be
// initialized.



/**
 * TransportProtocolHandler is the abstract base class interface that
 * each transport protocol (i.e. TCP or SSL) must implement.  
 */
class TransportProtocolHandler : public Object {
public:
  /**
   * The destructor.  It is declared virtual so that the destructor of
   * the sub-classes will be called.
   */
  virtual ~TransportProtocolHandler() {}

  /**
   * Connects to the broker with the connection properties specified
   * in connectionProperties.  At a minimum the properties
   * TRANSPORT_BROKER_NAME_PROPERTY and TRANSPORT_BROKER_PORT_PROPERTY
   * must be filled in.  
   *
   * @param connectionProperties the properties used to specify which broker to 
   *  connect to.
   * @return IMQ_SUCCESS if successful and an error otherwise
   */
  virtual iMQError connect(const Properties * const connectionProperties) = 0;

  /**
   * Returns the local port number of the connection.
   *
   * @param port output parameter for the port number of the local end of the 
   *  connection to the broker.  
   * @return IMQ_SUCCESS if successful and an error otherwise
   * @see getLocalIP
   */
  virtual iMQError getLocalPort(PRUint16 * const port) const = 0;

  /**
   * Returns the local IP address of the connection.
   *
   * @param ipAddr output parameter for the IP address of the local end of the 
   *  connection to the broker.  
   * @return IMQ_SUCCESS if successful and an error otherwise
   * @see getLocalPort
   */
  virtual iMQError getLocalIP(const IPAddress ** const ipAddr) const = 0;

  /**
   * Reads bytes from the connection.
   *
   * @param numBytesToRead the number of bytes to read
   * @param timeoutMicroSeconds the timeout specified in microseconds. 
   *  (The granularity of timeoutMicroSeconds may only be 1-10 MILLIseconds.
   *  A value of 0 implies no wait, and a value of 0xFFFFFFFF implies wait 
   *  forever (i.e. no timeout).  
   * @param bytesRead the output buffer where the read bytes are
   *  placed.  It must be at least numBytesToRead bytes big.  
   * @param numBytesRead output parameter for the actual number of bytes read
   * @return IMQ_SUCCESS if successful and an error otherwise
   */
  virtual iMQError read(const PRInt32          numBytesToRead,
                        const PRUint32         timeoutMicroSeconds, 
                              PRUint8 * const  bytesRead, 
                              PRInt32 * const  numBytesRead) = 0;

  /**
   * Write bytes to the connection.
   *
   * @param numBytesToWrite the number of bytes to write
   * @param bytesToWrite the bytes to write to the connection
   * @param timeoutMicroSeconds the timeout specified in microseconds. 
   *  (The granularity of timeoutMicroSeconds may only be 1-10 MILLIseconds.
   *  A value of 0 implies no wait, and a value of 0xFFFFFFFF implies wait 
   *  forever (i.e. no timeout).  
   * @param numBytesWritten output parameter for the actual number of bytes 
   *  written
   * @return IMQ_SUCCESS if successful and an error otherwise
   */
  virtual iMQError write(const PRInt32          numBytesToWrite,
                         const PRUint8 * const  bytesToWrite,
                         const PRUint32         timeoutMicroSeconds, 
                               PRInt32 * const  numBytesWritten) = 0;

  /**
   * Shuts down the connection to the broker (but does not close the 
   * socket file descriptor.)  Causes any thread currently blocked on
   * a read() to return.
   *
   * @return IMQ_SUCCESS if successful and an error otherwise
   */
  virtual iMQError shutdown() = 0;
  
  /**
   * Closes the connection to the broker.
   *
   * @return IMQ_SUCCESS if successful and an error otherwise
   */
  virtual iMQError close() = 0;


  /**
   * @return true iff the connection is closed
   */
  virtual PRBool isClosed() = 0;
};


#endif // TRANSPORTPROTOCOLHANDLER_HPP
