/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2000-2009 Sun Microsystems, Inc. All rights reserved. 
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License ("CDDL") (collectively, the "License").  You may
 * not use this file except in compliance with the License.  You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or mq/legal/LICENSE.txt.  See the License for the specific language
 * governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at mq/legal/LICENSE.txt.  Sun designates
 * this particular file as subject to the "Classpath" exception as provided by
 * Sun in the GPL Version 2 section of the License file that accompanied this
 * code.  If applicable, add the following below the License Header, with the
 * fields enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or  to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright holder. 
 */

/*
 * @(#)PropertyDAOImpl.java	1.14 06/29/07
 */ 

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.jmsserver.persist.Store;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.Globals;

import java.util.*;
import java.sql.*;
import java.io.IOException;

/**
 * This class implement a generic PropertyDAO.
 */
class PropertyDAOImpl extends  BaseDAOImpl implements PropertyDAO {

    protected String tableName;

    // SQLs
    protected String insertSQL;
    protected String updateSQL;
    protected String deleteSQL;
    protected String selectSQL;
    protected String selectAllNamesSQL;
    protected String selectAllSQL;

    /**
     * Constructor
     * @throws BrokerException
     */
    PropertyDAOImpl() throws BrokerException {

        // Initialize all SQLs
        DBManager dbMgr = DBManager.getDBManager();

        tableName = dbMgr.getTableName( TABLE_NAME_PREFIX );

        insertSQL = new StringBuffer(128)
            .append( "INSERT INTO " ).append( tableName )
            .append( " ( " )
            .append( PROPNAME_COLUMN ).append( ", " )
            .append( PROPVALUE_COLUMN )
            .append( ") VALUES ( ?, ? )" )
            .toString();

        updateSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( PROPVALUE_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( PROPNAME_COLUMN ).append( " = ?" )
            .toString();

        deleteSQL = new StringBuffer(128)
            .append( "DELETE FROM " ).append( tableName )
            .append( " WHERE " )
            .append( PROPNAME_COLUMN ).append( " = ?" )
            .toString();

        selectSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( PROPVALUE_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( PROPNAME_COLUMN ).append( " = ?" )
            .toString();

        selectAllNamesSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( PROPNAME_COLUMN )
            .append( " FROM " ).append( tableName )
            .toString();

        selectAllSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( PROPNAME_COLUMN ).append( ", " )
            .append( PROPVALUE_COLUMN )
            .append( " FROM " ).append( tableName )
            .toString();
    }

    /**
     * Get the prefix name of the table.
     * @return table name
     */
    public final String getTableNamePrefix() {
        return TABLE_NAME_PREFIX;
    }

    /**
     * Get the name of the table.
     * @return table name
     */
    public final String getTableName() {
        return tableName;
    }

    /**
     * Persist the specified property name/value pair.
     * If the property identified by name exists in the store already,
     * it's value will be updated with the new value.
     * If value is null, the property will be removed.
     * The value object needs to be serializable.
     * @param conn the database connection
     * @param name the property name
     * @param value the property value
     * @throws BrokerException
     */
    public void update( Connection conn, String name, Object value )
        throws BrokerException {

        boolean myConn = false;
        String sql = null;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            // Check to see if the property exists
            if ( hasProperty( conn, name ) ) {
                if ( value != null ) {
                    // Update
                    sql = updateSQL;
                    pstmt = conn.prepareStatement( sql );
                    Util.setObject( pstmt, 1, value );
                    pstmt.setString( 2, name );
                    pstmt.executeUpdate();
                } else {
                    // Delete
                    sql = deleteSQL;
                    pstmt = conn.prepareStatement( sql );
                    pstmt.setString( 1, name );
                    pstmt.executeUpdate();
                }
            } else if ( value != null ) {
                // Add
                sql = insertSQL;
                pstmt = conn.prepareStatement( sql );
                pstmt.setString( 1, name );
                Util.setObject( pstmt, 2, value );
                pstmt.executeUpdate();
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof IOException ) {
                ex = DBManager.wrapIOException("[" + sql + "]", (IOException)e);
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + sql + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_PERSIST_PROPERTY_FAILED, name ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Delete an entry.
     * @param conn
     * @param name name name of the property whose value is to be deleted
     * @throws BrokerException
     */
    public void delete( Connection conn, String name )
        throws BrokerException {

        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( deleteSQL );
            pstmt.setString( 1, name );
            pstmt.executeUpdate();
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + deleteSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_REMOVE_PROPERTY_FAILED, name ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Delete all entries.
     * @param conn
     * @throws BrokerException
     */
    public void deleteAll( Connection conn )
        throws BrokerException {

        if ( Globals.getHAEnabled() ) {
            return; // Share table cannot be reset    
        } else {
            String whereClause = new StringBuffer(128)
               .append( PROPNAME_COLUMN ).append( " <> '" )
               .append( DBTool.STORE_PROPERTY_SUPPORT_JMSBRIDGE ).append( "'" )
               .toString();
 
            super.deleteAll( conn, whereClause, null, 0 );
        }
    }

    /**
     * Retrieve the value for the specified property.
     * @param conn database connection
     * @param name name of the property whose value is to be retrieved
     * @return the property value; null is returned if the specified
     *		property does not exist in the store
     * @exception BrokerException if an error occurs while retrieving the data
     * @exception NullPointerException if <code>name</code> is
     *			<code>null</code>
     */
    public Object getProperty( Connection conn, String name )
        throws BrokerException {

        Object propObj = null;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectSQL );
            pstmt.setString( 1, name );
            rs = pstmt.executeQuery();

            if ( rs.next() ) {
                try {
                    propObj = Util.readObject( rs, 1 );
                } catch ( IOException e ) {
                    // fail to parse object; just log it
                    logger.logStack( Logger.ERROR,
                        BrokerResources.X_PARSE_PROPERTY_FAILED, name, e );
                }
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_PROPERTY_FAILED, name ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return propObj;
    }

    /**
     * Return the names of all persisted properties.
     * @param conn database connection
     * @return a List of property names; an empty List will be returned
     *		if no property exists in the store
     */
    public List getPropertyNames( Connection conn )
        throws BrokerException {

        ArrayList list = new ArrayList();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectAllNamesSQL );
            rs = pstmt.executeQuery();

            while ( rs.next() ) {
                String name = rs.getString( 1 );
                list.add( name );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectAllNamesSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_PROPERTIES_FAILED ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return list;
    }

    /**
     * Load all properties from DB.
     * @param conn database connection
     * @return the Properties object
     * @throws BrokerException
     */
    public Properties getProperties( Connection conn ) throws BrokerException {

        Properties props = new Properties();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectAllSQL );
            rs = pstmt.executeQuery();

            while ( rs.next() ) {
                String name = rs.getString( PROPNAME_COLUMN );
                try {
                    Object obj = Util.readObject( rs, 2 );
                    props.put( name, obj );
                } catch ( IOException e ) {
                    // fail to parse one object; just log it
                    logger.logStack( Logger.ERROR,
                        BrokerResources.X_PARSE_PROPERTY_FAILED, name, e );
                }
            }

            if ( Store.getDEBUG() ) {
                logger.log( Logger.DEBUG, "LOADED " +
                    props.size() + " PROPERTIES FROM DATABASE" );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectAllSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_PROPERTIES_FAILED ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return props;
    }

    /**
     * Check whether the specified property exists.
     * @param conn database connection
     * @param name name of the property
     * @return return true if the specified property exists
     */
    public boolean hasProperty( Connection conn, String name )
        throws BrokerException {

        boolean found = false;
        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectSQL );
            pstmt.setString( 1, name );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                found = true;
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_PROPERTY_FAILED,
                name ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return found;
    }

    /**
     * Get debug information about the store.
     * @param conn database connection
     * @return a HashMap of name value pair of information
     */
    public HashMap getDebugInfo( Connection conn ) {

        HashMap map = new HashMap();
        int count = -1;

        try {
            // Get row count
            count = getRowCount( null, null );
        } catch ( Exception e ) {
            logger.log( Logger.ERROR, e.getMessage(), e.getCause() );
        }

        map.put( "Properties(" + tableName + ")", String.valueOf( count ) );
        return map;
    }
}
