/*
 * Copyright (c) 2008 TELE-UCL, Benoit Macq, Marc Vandenbosch(MVA)
 * All rights reserved. See Copyright.txt or web site
 * (http://code.google.com/p/opencinematools/) for details.
 *
 * Created by: MVA
 *  
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *  
 */

#include <stdio.h>
#ifdef _MSC_VER
#define snprintf _snprintf
#endif
#include <KM_xml.h>
#include <KM_fileio.h>
#include <AS_DCP.h>
#include <openssl/sha.h>
#include <openssl/hmac.h>
#include <openssl/evp.h>
#include <openssl/bio.h>
#include <openssl/buffer.h>

using namespace Kumu;
using namespace ASDCP;

const char* PACKAGE = "mkmap";
const char* VERSION = "1.1.2";
const char* CREATOR = "OpenCinemaTools - mkmap 1.1.2";

char *base64(const unsigned char *input, int length)
{
  BIO *bmem, *b64;
  BUF_MEM *bptr;

  b64 = BIO_new(BIO_f_base64());
  bmem = BIO_new(BIO_s_mem());
  b64 = BIO_push(b64, bmem);
  BIO_write(b64, input, length);
  BIO_flush(b64);
  BIO_get_mem_ptr(b64, &bptr);

  char *buff = (char *)malloc(bptr->length);
  memcpy(buff, bptr->data, bptr->length-1);
  buff[bptr->length-1] = 0;

  BIO_free_all(b64);

  return buff;
}

void copyright(FILE* stream = stdout) {
  fprintf(stream, "\
%s (OpenCinemaTools %s)\n\
Copyright (c) 2008 TELE-UCL, Benoit Macq, Marc Vandenbosch\n\n", PACKAGE, VERSION);
}

void banner(FILE* stream = stdout) {
  copyright(stream);
  fprintf(stream, "\
%s is part of the OpenCinemaTools.\n\
OpenCinemaTools may be copied only under the terms of new BSD license.\n\n\
Specify the -h (help) option for further information.\n\n", PACKAGE);
}

//
void usage(FILE* stream = stdout) {
  fprintf(stream, "\
USAGE: \n\
  %s file[file...]\n\
\n\
  %s [--annotation \"annotation\"] file[file...]\n\
  %s [--issuer \"issuer\"] file[file...]\n\
\n\
  -h | --help                      - Show help\n\
  -V | --version                   - Show version information\n\
  --annotation \"annotation\"        - Annotation text\n\
  --issuer \"issuer\"                - Issuer name\n\
\n\
  NOTES: o There is no option grouping, all options must be distinct arguments.\n\
         o All option arguments must be separated from the option by whitespace.\n\
", PACKAGE, PACKAGE, PACKAGE);
}

class CommandOptions
{
  CommandOptions();

public:
  bool   error_flag;      // true if the given options are in error or not complete
  bool   help_flag;       // true if the help display option was selected
  bool   version_flag; 
  const char* annotation;
  const char* issuer;
  std::list<const char*> files;

  CommandOptions(int argc, const char** argv) :
    error_flag(true), version_flag(false), annotation(0),
    help_flag(false), issuer(0) {
    
    for (int i = 1; i < argc; i++) {

      if ((strcmp(argv[i], "--help") == 0)) {
	      help_flag = true;
	      continue;
	    }

      if ((strcmp(argv[i], "--version") == 0)) {
	      version_flag = true;
	      continue;
	    }

      if ((strcmp(argv[i], "--annotation") == 0)) {
        i++;
        if (i<argc && argv[i][0] != '-') {
          annotation = argv[i];
        } else {
          fprintf(stderr, "Missing parameter for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }

      if ((strcmp(argv[i], "--issuer") == 0)) {
        i++;
        if (i<argc && argv[i][0] != '-') {
          issuer = argv[i];
        } else {
          fprintf(stderr, "Missing parameter for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }
     
	    if (argv[i][0] == '-' && isalpha(argv[i][1]) && argv[i][2] == 0) {
	      switch (argv[i][1]) {
	        case 'h': help_flag = true; break;
          case 'V': version_flag = true; break;
	        default:
		        fprintf(stderr, "Unrecognized option: %s\n", argv[i]);
		        return;
	      }
      }	else if (argv[i][0] != '-') {
        files.push_back(argv[i]);
        continue;
      } else {
	      fprintf(stderr, "Unrecognized option: %s\n", argv[i]);
	      return;
	    }
    }

    if (help_flag) return;
    
    if ( issuer == 0 ) {
      std::string Issuer;
#ifdef _MSC_VER
      char* p = getenv("USERNAME");
#else
      char* p = getenv("USER");
#endif
      Issuer = ( p != 0 ) ? p : "nouser";
      Issuer += "@";
#ifdef _MSC_VER
      p = getenv("USERDOMAIN");
#else
      p = getenv("HOSTNAME");
#endif
      Issuer += ( p != 0 ) ? p : "localhost";
      issuer = strdup(Issuer.c_str());
    }

    if (argc>1 &&
        files.size() >= 1) {
      error_flag = false;
    }
  }
};

struct Asset
{
  UUID        AssetID;
  std::string EditRate;
  ui32_t      IntrinsicDuration;
  ui32_t      Duration;
  ui32_t      EntryPoint;
  bool        IsCiphertext;
  UUID        KeyID;
  std::string Digest;

  bool        isPackingList;
  ui64_t      FileSize;
  std::string MIMEType;
  std::string Path;
};

struct Map_info
{
  UUID        ID;
  std::string Annotation;
  Timestamp   IssueDate;
  std::string Issuer;
  std::string Creator;
  std::list<Asset> AssetList;
};

bool make_asset_map(const Map_info& Info, std::string& OutMap) {
  XMLElement    Map("AssetMap");
  char          tmp_buf[64];

  Map.SetAttr("xmlns", "http://www.smpte-ra.org/schemas/429-9/2007/AM");
  Map.AddChildWithPrefixedContent("Id", "urn:uuid:", Info.ID.EncodeHex(tmp_buf, 64));

  if ( ! Info.Annotation.empty() )
    Map.AddChildWithContent("AnnotationText", Info.Annotation.c_str());

  if ( ! Info.Creator.empty() )
    Map.AddChildWithContent("Creator", Info.Creator.c_str());

  Map.AddChildWithContent("VolumeCount", "1");
  Map.AddChildWithContent("IssueDate", Info.IssueDate.EncodeString(tmp_buf, 64));

  if ( ! Info.Issuer.empty() )
    Map.AddChildWithContent("Issuer", Info.Issuer.c_str());

  XMLElement* XMLAssetList = Map.AddChild("AssetList");

  std::list<Asset>::const_iterator ai;
  for ( ai = Info.AssetList.begin(); ai != Info.AssetList.end(); ai++ )
    {
      XMLElement* Asset = XMLAssetList->AddChild("Asset");
      Asset->AddChildWithPrefixedContent("Id", "urn:uuid:", ai->AssetID.EncodeHex(tmp_buf, 64));

      if ( ai->isPackingList )
	  //Asset->AddChild("PackingList");
	  Asset->AddChildWithContent("PackingList", "true");

      XMLElement* XMLChunkList = Asset->AddChild("ChunkList");
      ui64Printer Length_str(ai->FileSize);

      XMLElement* Chunk = XMLChunkList->AddChild("Chunk");
      Chunk->AddChildWithContent("Path", ai->Path.c_str());
      Chunk->AddChildWithContent("VolumeIndex", "1");
      Chunk->AddChildWithContent("Offset", "0");
      Chunk->AddChildWithContent("Length", Length_str.c_str());
    }

  Map.Render(OutMap);
  return true;
}

bool make_volume_index(std::string& OutVol)
{
  XMLElement    Vol("VolumeIndex");

  Vol.SetAttr("xmlns", "http://www.smpte-ra.org/schemas/429-9/2007/AM");
  //Vol.SetAttr("xmlns", "http://www.smpte-ra.org/schemas/429-9/2006/AM");
  //Vol.SetAttr("xmlns", "http://www.digicine.com/PROTO-ASDCP-VL-20040311#");

  Vol.AddChildWithContent("Index", "1");
  Vol.Render(OutVol);
  return true;
}

int main(int argc, const char** argv) {
  CommandOptions Options(argc, argv);
  std::string path;

  if (Options.version_flag) {
    banner();
    return 0;
  }

  if (Options.help_flag) {
    usage();
    return 0;
  }

  if (Options.error_flag) {
    fprintf(stderr, "There was a problem. Type %s -h for help.\n", PACKAGE);
    return 2;
  }

  Map_info Info;
  Kumu::GenRandomValue(Info.ID);
  Info.Creator = CREATOR;
  if (Options.annotation != 0) {
    Info.Annotation = Options.annotation;
  }
  if (Options.issuer != 0) {
    Info.Issuer = Options.issuer;
  } 

  std::list<const char*>::iterator iter;
  Result_t result = RESULT_OK;
  
  iter = Options.files.begin();
  while (iter!=Options.files.end()) {
    EssenceType_t EssenceType;
    result = ASDCP::EssenceType(*iter, EssenceType);

    if (ASDCP_SUCCESS(result)) {
      switch (EssenceType) {
        case ESS_MPEG2_VES:
          fprintf(stderr, "%s: ESS_MPEG2_VES not supported.\n", *iter);
          return 3;
          break;

        case ESS_JPEG_2000:
          {
            JP2K::MXFReader     JP2KReader;
            FileReader filereader;
            ui32_t size = 0;

            result = JP2KReader.OpenRead(*iter);
            if (ASDCP_SUCCESS(result)) {
              WriterInfo WInfo;
              JP2KReader.FillWriterInfo(WInfo);

              Info.AssetList.push_back(Asset());
              Info.AssetList.back().AssetID.Set(WInfo.AssetUUID);
            } else {
              fprintf(stderr, "%s: Can't open file.\n", *iter);
              return 7;
            }
            if (ASDCP_SUCCESS(result)) {
              result = filereader.OpenRead(*iter);
            } else {
              fprintf(stderr, "%s: Can't open file.\n", *iter);
              return 8;
            }
            if (ASDCP_SUCCESS(result)) {
              Info.AssetList.back().FileSize = filereader.Size();         
              Info.AssetList.back().isPackingList = false;
			  path = "file:///";
			  path += *iter;
			  Info.AssetList.back().Path = path;
            }
            if (filereader.IsOpen()) {
              filereader.Close();
            }
          }
          break;

        case ESS_PCM_24b_48k:
          {
              PCM::MXFReader     PCMReader;
              FileReader filereader;
              ui32_t size = 0;

	            result = PCMReader.OpenRead(*iter);
              if (ASDCP_SUCCESS(result)) {
                WriterInfo WInfo;
	              PCMReader.FillWriterInfo(WInfo);

                Info.AssetList.push_back(Asset());
                Info.AssetList.back().AssetID.Set(WInfo.AssetUUID);
              } else {
                fprintf(stderr, "%s: Can't open file.\n", *iter);
                return 7;
              }
              if (ASDCP_SUCCESS(result)) {
                result = filereader.OpenRead(*iter);
              } else {
                fprintf(stderr, "%s: Can't open file.\n", *iter);
                return 8;
              }
              if (ASDCP_SUCCESS(result)) {
                Info.AssetList.back().FileSize = filereader.Size();         
                Info.AssetList.back().isPackingList = false;
				path = "file:///";
				path += *iter;
                Info.AssetList.back().Path = path;
              }
              if (filereader.IsOpen()) {
                filereader.Close();
              }
            }
          break;

        case ESS_PCM_24b_96k:
          fprintf(stderr, "%s: ESS_PCM_24b_96k not supported.\n", *iter);
          return 4;
          break;

	case ESS_JPEG_2000_S:
        {
          JP2K::MXFSReader     JP2KReader;
          FileReader filereader;
          ui32_t size = 0;
          
          result = JP2KReader.OpenRead(*iter);
          if (ASDCP_SUCCESS(result))
          {
            WriterInfo WInfo;
            JP2KReader.FillWriterInfo(WInfo);
            
            Info.AssetList.push_back(Asset());
            Info.AssetList.back().AssetID.Set(WInfo.AssetUUID);
          }
          else
          {
            fprintf(stderr, "%s: Can't open file.\n", *iter);
            return 7;
          }
          if (ASDCP_SUCCESS(result))
          {
            result = filereader.OpenRead(*iter);
          }
          else
          {
            fprintf(stderr, "%s: Can't open file.\n", *iter);
            return 8;
          }
          if (ASDCP_SUCCESS(result))
          {
            Info.AssetList.back().FileSize = filereader.Size();
            Info.AssetList.back().isPackingList = false;
            path = "file:///";
			path += *iter;
            Info.AssetList.back().Path = path;
          }
          if (filereader.IsOpen())
          {
            filereader.Close();
          }
        }
          break;

        default:
          if (ASDCP_FAILURE(result)) {
            fprintf(stderr, "%s: Unknown file type, not ASDCP essence.\n", *iter);
            return 5;
          }
      }
    } else {
      FileReader filereader;
      byte_t *filebuff = 0;
      ui32_t size = 0;
      std::string document;
      XMLElement cpl("");
      XMLElement *id = 0;

      result = filereader.OpenRead(*iter);
      if (ASDCP_SUCCESS(result)) {
        if (filereader.Size() < (10*1024*1024)) {
          size = (ui32_t)filereader.Size();
          filebuff = (byte_t*)malloc(size+1);
          if (filebuff == 0) result = RESULT_ALLOC;
        }
      }
      if (ASDCP_SUCCESS(result)) {
        result = filereader.Read(filebuff, size);
      }
      if (ASDCP_SUCCESS(result)) {
        filebuff[size] = 0;
        document = (char*)filebuff;
        if (StringIsXML(document.c_str(), document.length())) {
          if (cpl.ParseString(document) && 
             (cpl.HasName("CompositionPlaylist") != 0) || (cpl.HasName("PackingList") != 0)) {
            id = cpl.GetChildWithName("Id");
            if (id == 0) result = RESULT_FAIL;
          } else {
            result = RESULT_FAIL;
          }
        } else {
          result = RESULT_FAIL;
        }
      }
      if (ASDCP_SUCCESS(result)) {
        Info.AssetList.push_back(Asset());
        Info.AssetList.back().AssetID.DecodeHex(id->GetBody().substr(9).c_str());
        Info.AssetList.back().FileSize = size;
        Info.AssetList.back().isPackingList = (cpl.HasName("PackingList") != 0) ? true : false;
        path = "file:///";
		path += *iter;
        Info.AssetList.back().Path = path;
      }

      if (filebuff != 0) {
        free(filebuff);
        filebuff = 0;
      }
      if (filereader.IsOpen()) {
        filereader.Close();
      }
      if (ASDCP_FAILURE(result)) {
        fprintf(stderr, "%s: Invalid CompositionPlaylist or PackingList file.\n", *iter);
        return 5;
      }
    }
    iter++;
  }

  std::string OutMap;
  make_asset_map(Info, OutMap);
  FILE* mapfile = fopen("ASSETMAP.xml", "w");
  if (mapfile == 0) {
    fprintf(stderr, "Can't write ASSETMAP.xml\n");
    return 6;
  }
  fputs(OutMap.c_str(), mapfile);
  fflush(mapfile);
  fclose(mapfile);

  std::string OutVol;
  make_volume_index(OutVol);
  FILE* volfile = fopen("VOLINDEX.xml", "w");
  if (volfile == 0) {
    fprintf(stderr, "Can't write VOLINDEX.xml\n");
    return 7;
  }
  fputs(OutVol.c_str(), volfile);
  fflush(volfile);
  fclose(volfile);
  return 0;
}

