/*
 * Copyright (c) 2008 TELE-UCL, Benoit Macq, Marc Vandenbosch(MVA)
 * All rights reserved. See Copyright.txt or web site
 * (http://code.google.com/p/opencinematools/) for details.
 *
 * Created by: MVA
 *  
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *  
 */

#include <stdio.h>
#ifdef _MSC_VER
#define snprintf _snprintf
#endif
#include <KM_xml.h>
#include <AS_DCP.h>
using namespace Kumu;
using namespace ASDCP;

const char* PACKAGE = "mkcpl";
const char* VERSION = "1.1.2";
const char* CREATOR = "OpenCinemaTools - mkcpl 1.1.2";


void copyright(FILE* stream = stdout) {
  fprintf(stream, "\
%s (OpenCinemaTools %s)\n\
Copyright (c) 2008 TELE-UCL, Benoit Macq, Marc Vandenbosch\n\n", PACKAGE, VERSION);
}

void banner(FILE* stream = stdout) {
  copyright(stream);
  fprintf(stream, "\
%s is part of the OpenCinemaTools.\n\
OpenCinemaTools may be copied only under the terms of new BSD license.\n\n\
Specify the -h (help) option for further information.\n\n", PACKAGE);
}

//
void usage(FILE* stream = stdout) {
  fprintf(stream, "\
USAGE: \n\
  %s --kind contentkind --title \"title\" picture.mxf sound.mxf\n\
\n\
  %s --kind ck --title \"t\" p1.mxf s1.mxf [p2.mxf s2.mxf]\n\
  %s --kind ck --title \"t\" p1.mxf [p2.mxf] s1.mxf [s2.mxf]\n\
  %s --kind ck --title \"t\" [--annotation \"annotation\"] p.mxf s.mxf\n\
  %s --kind ck --title \"t\" [--issuer \"issuer\"] p.mxf s.mxf\n\
  %s --kind ck --title \"t\" [--ratio \"aspectratio\"] p.mxf s.mxf\n\
  %s --kind ck --title \"t\" [--versionid uid] p.mxf s.mxf\n\
  %s --kind ck --title \"t\" [--versionlabel label] p.mxf s.mxf\n\
  %s --kind ck --title \"t\" [--rating agency label] p.mxf s.mxf\n\
  %s --kind ck --title \"t\" [--norating ] p.mxf s.mxf\n\
\n\
  -h | --help                      - Show help\n\
  -V | --version                   - Show version information\n\
  --kind contentkind               - Content kind. Could be one of the\n\
                                     following: feature, trailer, test,\n\
                                     teaser, rating, advertisement,\n\
                                     short, transitional, psa, policy)\n\
  --title \"title\"                  - Content title\n\
  --annotation \"annotation\"        - Annotation text\n\
  --issuer \"issuer\"                - Issuer name\n\
  --ratio \"aspectratio\"            - Overrides Aspect ratio. Default is taken from the mxf\n\
  --versionid uid                    - Version UID. Default generates a new id\n\
  --versionlabel label               - Version label. Default is the versionid\n\
  --rating agency label              - Defines a rating agency and label.\n\
                                       Default is --rating http://www.mpaa.org G\n\
  --norating                         - Removes the default rating.\n\
\n\
  NOTES: o There is no option grouping, all options must be distinct arguments.\n\
         o All option arguments must be separated from the option by whitespace.\n\
", PACKAGE, PACKAGE, PACKAGE, PACKAGE, PACKAGE, 
   PACKAGE, PACKAGE, PACKAGE, PACKAGE, PACKAGE);
}

class CommandOptions
{
  CommandOptions();

public:
  bool   error_flag;      // true if the given options are in error or not complete
  bool   help_flag;       // true if the help display option was selected
  bool   version_flag; 
  bool   no_rating_flag;
  const char* kind;
  const char* title;
  const char* annotation;
  const char* issuer;
  const char* ratio;
  const char* version_id;
  const char* version_label;
  std::list<const char*> files;
  std::list<const char*> ratings;

  CommandOptions(int argc, const char** argv) :
    error_flag(true), version_flag(false), kind(0), title(0), annotation(0),
    help_flag(false), issuer(0), ratio(0), version_id(0), version_label(0),
    no_rating_flag(false) {
    
    for (int i = 1; i < argc; i++) {

      if ((strcmp(argv[i], "--help") == 0)) {
	      help_flag = true;
	      continue;
	    }

      if ((strcmp(argv[i], "--version") == 0)) {
	      version_flag = true;
	      continue;
	    }

      if ((strcmp(argv[i], "--norating") == 0)) {
	      no_rating_flag = true;
	      continue;
	    }

      if ((strcmp(argv[i], "--kind") == 0)) {
        i++;
        if (i<argc) {
          if ((strcmp(argv[i], "feature") == 0) || 
              (strcmp(argv[i], "trailer") == 0) ||
              (strcmp(argv[i], "test") == 0) ||
              (strcmp(argv[i], "teaser") == 0) ||
              (strcmp(argv[i], "rating") == 0) ||
              (strcmp(argv[i], "advertisement") == 0) ||
              (strcmp(argv[i], "short") == 0) ||
              (strcmp(argv[i], "transitional") == 0) ||
              (strcmp(argv[i], "psa") == 0) ||
              (strcmp(argv[i], "policy") == 0)) {
            kind = argv[i];
          } else {
            fprintf(stderr, "Invalid parameter for option: %s\n", argv[i-1]);
            return;
          }
        } else {
          fprintf(stderr, "Missing parameter for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }

      if ((strcmp(argv[i], "--title") == 0)) {
        i++;
        if (i<argc && argv[i][0] != '-') {
          title = argv[i];
        } else {
          fprintf(stderr, "Missing parameter for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }

      if ((strcmp(argv[i], "--annotation") == 0)) {
        i++;
        if (i<argc && argv[i][0] != '-') {
          annotation = argv[i];
        } else {
          fprintf(stderr, "Missing parameter for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }

      if ((strcmp(argv[i], "--issuer") == 0)) {
        i++;
        if (i<argc && argv[i][0] != '-') {
          issuer = argv[i];
        } else {
          fprintf(stderr, "Missing parameter for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }

      if ((strcmp(argv[i], "--ratio") == 0)) {
        i++;
        if (i<argc && argv[i][0] != '-') {
          ratio = argv[i];
        } else {
          fprintf(stderr, "Missing parameter for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }

      if ((strcmp(argv[i], "--versionid") == 0)) {
        i++;
        if (i<argc && argv[i][0] != '-' && (strlen(argv[i])==36)) {
          version_id = argv[i];
        } else {
          fprintf(stderr, "Missing or invalid parameter for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }

      if ((strcmp(argv[i], "--versionlabel") == 0)) {
        i++;
        if (i<argc && argv[i][0] != '-') {
          version_label = argv[i];
        } else {
          fprintf(stderr, "Missing parameter for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }

      if ((strcmp(argv[i], "--rating") == 0)) {
        i++;
        if (i<argc && argv[i][0] != '-') {
          ratings.push_back(argv[i]);
          i++;
          if (i<argc && argv[i][0] != '-') {
            ratings.push_back(argv[i]);
            no_rating_flag = true;
          } else {
            fprintf(stderr, "Missing label for option: %s\n", argv[i-1]);
            return;
          }
        } else {
          fprintf(stderr, "Missing agency for option: %s\n", argv[i-1]);
          return;
        }
	      continue;
	    }
     
	    if (argv[i][0] == '-' && isalpha(argv[i][1]) && argv[i][2] == 0) {
	      switch (argv[i][1]) {
	        case 'h': help_flag = true; break;
          case 'V': version_flag = true; break;
	        default:
		        fprintf(stderr, "Unrecognized option: %s\n", argv[i]);
		        return;
	      }
      }	else if (argv[i][0] != '-') {
        files.push_back(argv[i]);
        continue;
      } else {
	      fprintf(stderr, "Unrecognized option: %s\n", argv[i]);
	      return;
	    }
    }

    if (help_flag) return;

    if ( issuer == 0 ) {
      std::string Issuer;
#ifdef _MSC_VER
      char* p = getenv("USERNAME");
#else
      char* p = getenv("USER");
#endif
      Issuer = ( p != 0 ) ? p : "nouser";
      Issuer += "@";
#ifdef _MSC_VER
      p = getenv("USERDOMAIN");
#else
      p = getenv("HOSTNAME");
#endif
      Issuer += ( p != 0 ) ? p : "localhost";
      issuer = strdup(Issuer.c_str());
    }
    
    if (argc>1 &&
        kind != 0 &&
        title != 0 &&
        files.size() >= 1) {
      error_flag = false;
    }
  }
};

struct Asset
{
  UUID        AssetID;
  std::string EditRate;
  ui32_t      IntrinsicDuration;
  ui32_t      Duration;
  ui32_t      EntryPoint;
  bool        IsCiphertext;
  UUID        KeyID;
  std::string Digest;
  std::string FrameRate;

  bool        isPackingList;
  ui64_t      FileSize;
  std::string MIMEType;
  std::string Path;
  bool        isStereo;
};

struct Reel
{
  Asset MainPicture;
  Asset MainSound;
  Asset MainSubtitle;
};

struct Rating  {
  std::string Agency;
  std::string Label;
};

struct CPL_info
{
  UUID        ID;
  UUID        VersionID;
  std::string VersionLabel;
  std::string Annotation;
  Timestamp   IssueDate;
  std::string Issuer;
  std::string Creator;
  std::string ContentTitle;
  std::string ContentKind;
  std::string AspectRatio;
  std::list<Rating> RatingList;
  std::list<Reel> ReelList;
};

bool make_cpl(const CPL_info& Info, std::string& OutCPL) {
  XMLElement    CPL("CompositionPlaylist");
  XMLElement*   XMLReelList = 0;
  char          tmp_buf[64];

  CPL.SetAttr("xmlns", "http://www.smpte-ra.org/schemas/429-7/2006/CPL");
  // else "http://www.digicine.com/PROTO-ASDCP-CPL-20040511#"

  CPL.AddChildWithPrefixedContent("Id", "urn:uuid:", Info.ID.EncodeHex(tmp_buf, 64));

  if ( ! Info.Annotation.empty() )
    CPL.AddChildWithContent("AnnotationText", Info.Annotation.c_str());

  CPL.AddChildWithContent("IssueDate", Info.IssueDate.EncodeString(tmp_buf, 64));

  if ( ! Info.Issuer.empty() )
    CPL.AddChildWithContent("Issuer", Info.Issuer.c_str());

  if ( ! Info.Creator.empty() )
    CPL.AddChildWithContent("Creator", Info.Creator.c_str());

  if ( ! Info.ContentTitle.empty() )
    CPL.AddChildWithContent("ContentTitleText", Info.ContentTitle.c_str());

  CPL.AddChildWithContent("ContentKind", Info.ContentKind.c_str());

  XMLElement* ContentVersion = CPL.AddChild("ContentVersion");
  ContentVersion->AddChildWithPrefixedContent("Id", "urn:uuid:", Info.VersionID.EncodeHex(tmp_buf, 64));
  if (Info.VersionLabel.empty()) {
    ContentVersion->AddChildWithContent("LabelText", Info.VersionID.EncodeHex(tmp_buf, 64));
  } else {
    ContentVersion->AddChildWithContent("LabelText", Info.VersionLabel);
  }

  XMLElement* XMLRatingList = CPL.AddChild("RatingList");

  std::list<Rating>::const_iterator rating_iter;
  for (rating_iter = Info.RatingList.begin(); rating_iter != Info.RatingList.end(); rating_iter++) {
      XMLElement* Rating = XMLRatingList->AddChild("Rating");
      Rating->AddChildWithContent("Agency", rating_iter->Agency);
      Rating->AddChildWithContent("Label", rating_iter->Label);
  }

  XMLReelList = CPL.AddChild("ReelList");
  ui32_t reel_counter = 1;
  std::list<Reel>::const_iterator ri;
  for ( ri = Info.ReelList.begin(); ri != Info.ReelList.end(); ri++ ) {
      XMLElement* Reel = XMLReelList->AddChild("Reel");
      Kumu::UUID TmpID;
      GenRandomValue(TmpID);

      Reel->AddChildWithPrefixedContent("Id", "urn:uuid:", TmpID.EncodeHex(tmp_buf, 64));

      XMLElement* AssetList = Reel->AddChild("AssetList");
      XMLElement* MainPicture = 0;
	  if (ri->MainPicture.isStereo) {
        MainPicture = AssetList->AddChild("MainStereoscopicPicture");
		MainPicture->SetAttr("xmlns", "http://www.smpte-ra.org/schemas/429-10/2008/Main-Stereo-Picture-CPL");
      } else {
        MainPicture = AssetList->AddChild("MainPicture");
	  }
      MainPicture->AddChildWithPrefixedContent("Id", "urn:uuid:", ri->MainPicture.AssetID.EncodeHex(tmp_buf, 64));
      MainPicture->AddChildWithContent("EditRate", ri->MainPicture.EditRate.c_str());

      ui32Printer IntrinsicDuration_text(ri->MainPicture.IntrinsicDuration);
      MainPicture->AddChildWithContent("IntrinsicDuration", IntrinsicDuration_text.c_str());

      ui32Printer EntryPoint_text(ri->MainPicture.EntryPoint);
      MainPicture->AddChildWithContent("EntryPoint", EntryPoint_text.c_str());

      ui32Printer Duration_text(ri->MainPicture.Duration);
      MainPicture->AddChildWithContent("Duration", Duration_text.c_str());

      if ( ri->MainPicture.IsCiphertext )
	{
	  MainPicture->AddChildWithPrefixedContent("KeyId", "urn:uuid:", ri->MainPicture.KeyID.EncodeHex(tmp_buf, 64));

	  if ( ! ri->MainPicture.Digest.empty() )
	    MainPicture->AddChildWithContent("Hash", ri->MainPicture.Digest.c_str());
	}

      MainPicture->AddChildWithContent("FrameRate", ri->MainPicture.FrameRate.c_str()); // NOT ALWAYS TRUE
      MainPicture->AddChildWithContent("ScreenAspectRatio", Info.AspectRatio.c_str());

      // MainSound
      if ( ri->MainSound.IntrinsicDuration > 0 ) {
		  XMLElement* MainSound = AssetList->AddChild("MainSound");
		  MainSound->AddChildWithPrefixedContent("Id", "urn:uuid:", ri->MainSound.AssetID.EncodeHex(tmp_buf, 64));
		  MainSound->AddChildWithContent("EditRate", ri->MainSound.EditRate.c_str());

		  ui32Printer IntrinsicDuration_text(ri->MainSound.IntrinsicDuration);
		  MainSound->AddChildWithContent("IntrinsicDuration", IntrinsicDuration_text.c_str());

		  ui32Printer EntryPoint_text(ri->MainSound.EntryPoint);
		  MainSound->AddChildWithContent("EntryPoint", EntryPoint_text.c_str());
	      
		  ui32Printer Duration_text(ri->MainSound.Duration);
		  MainSound->AddChildWithContent("Duration", Duration_text.c_str());
	      
		  if ( ri->MainSound.IsCiphertext )
			{
			  MainSound->AddChildWithPrefixedContent("KeyId", "urn:uuid:", ri->MainSound.KeyID.EncodeHex(tmp_buf, 64));

			  if ( ! ri->MainSound.Digest.empty() )
			MainSound->AddChildWithContent("Hash", ri->MainSound.Digest.c_str());
			}
		}

      // MainSubtitle
      if ( ri->MainSubtitle.IntrinsicDuration > 0 )
        {
	  XMLElement* MainSubtitle = AssetList->AddChild("MainSubtitle");
	  MainSubtitle->AddChildWithPrefixedContent("Id", "urn:uuid:", ri->MainSubtitle.AssetID.EncodeHex(tmp_buf, 64));
	  MainSubtitle->AddChildWithContent("EditRate", ri->MainSubtitle.EditRate.c_str());

	  ui32Printer IntrinsicDuration_text(ri->MainSubtitle.IntrinsicDuration);
	  MainSubtitle->AddChildWithContent("IntrinsicDuration", IntrinsicDuration_text.c_str());

	  ui32Printer EntryPoint_text(ri->MainSubtitle.EntryPoint);
	  MainSubtitle->AddChildWithContent("EntryPoint", EntryPoint_text.c_str());

	  ui32Printer Duration_text(ri->MainSubtitle.Duration);
	  MainSubtitle->AddChildWithContent("Duration", Duration_text.c_str());

	  if ( ri->MainSubtitle.IsCiphertext )
	    {
	      MainSubtitle->AddChildWithPrefixedContent("KeyId", "urn:uuid:", ri->MainSubtitle.KeyID.EncodeHex(tmp_buf, 64));

	      if ( ! ri->MainSubtitle.Digest.empty() )
		MainSubtitle->AddChildWithContent("Hash", ri->MainSubtitle.Digest.c_str());
	    }
	}
    }

  CPL.Render(OutCPL);
  return true;
}

int main(int argc, const char** argv) {
  CommandOptions Options(argc, argv);

  if (Options.version_flag) {
    banner();
    return 0;
  }

  if (Options.help_flag) {
    usage();
    return 0;
  }

  if (Options.error_flag) {
    fprintf(stderr, "There was a problem. Type %s -h for help.\n", PACKAGE);
    return 2;
  }


  CPL_info Info;
  char buff[256];
  Kumu::GenRandomValue(Info.ID);
  Info.Creator = CREATOR;
  Info.ContentTitle = Options.title;
  if (Options.annotation != 0) {
    Info.Annotation = Options.annotation;
  }
  if (Options.issuer != 0) {
    Info.Issuer = Options.issuer;
  }  
  Info.ContentKind = Options.kind;
  
  if (Options.ratio != 0) {
    Info.AspectRatio = Options.ratio;
  }

  if (Options.version_id != 0) {
    Info.VersionID.DecodeHex(Options.version_id);
  } else {
    Kumu::GenRandomValue(Info.VersionID);
  }
  if (Options.version_label != 0) {
    Info.VersionLabel = Options.version_label; 
  } /* else will be the version id (copied in make_cpl) */

  if (!Options.no_rating_flag) {
    Info.RatingList.push_back(Rating());
    Info.RatingList.back().Agency = "http://www.mpaa.org";
    Info.RatingList.back().Label = "G";
  }

  std::list<const char*>::iterator rtiter;
  rtiter = Options.ratings.begin();
  while (rtiter!=Options.ratings.end()) {
    Info.RatingList.push_back(Rating());
    Info.RatingList.back().Agency = *rtiter;
    rtiter++;
    Info.RatingList.back().Label = *rtiter;
    rtiter++;
  }

  std::list<const char*>::iterator iter;
  EssenceType_t EssenceType;
  Result_t result = RESULT_OK;
  std::list<Asset> PAssets;
  std::list<Asset> AAssets;
  
  iter = Options.files.begin();
  while (iter!=Options.files.end()) {
      result = ASDCP::EssenceType(*iter, EssenceType);

      if (ASDCP_SUCCESS(result)) {
	      switch (EssenceType) {
          case ESS_MPEG2_VES:
	          fprintf(stderr, "%s: ESS_MPEG2_VES not supported.\n", *iter);
	          return 3;
	          break;

	        case ESS_JPEG_2000:
            {
              JP2K::MXFReader JP2KReader;
              result = JP2KReader.OpenRead(*iter);              

              if (ASDCP_SUCCESS(result)) {
                  JP2K::PictureDescriptor PDesc;
                  JP2KReader.FillPictureDescriptor(PDesc);
                  WriterInfo WInfo;
	              JP2KReader.FillWriterInfo(WInfo);

                  PAssets.push_back(Asset());
				  PAssets.back().isStereo = false;
                  sprintf(buff, "%u %u", PDesc.EditRate.Numerator, PDesc.EditRate.Denominator);
                  PAssets.back().EditRate = buff;
                  if (PDesc.SampleRate.Numerator==0 || PDesc.SampleRate.Denominator==0) {
                    sprintf(buff, "%u %u", PDesc.EditRate.Numerator, PDesc.EditRate.Denominator);
                  } else {
                    sprintf(buff, "%u %u", PDesc.SampleRate.Numerator, PDesc.SampleRate.Denominator);
                  }
                  PAssets.back().FrameRate = buff;

                  if (Options.ratio == 0) {
                    sprintf(buff, "%u %u", PDesc.AspectRatio.Numerator, PDesc.AspectRatio.Denominator);
                    Info.AspectRatio = buff;
                  }

                  PAssets.back().IntrinsicDuration = PDesc.ContainerDuration;
                  PAssets.back().Duration = PDesc.ContainerDuration;
                  PAssets.back().EntryPoint = 0;
                  PAssets.back().IsCiphertext = WInfo.EncryptedEssence;
                  PAssets.back().AssetID.Set(WInfo.AssetUUID);
              } else {
                fprintf(stderr, "%s: Can't open file.\n", *iter);
              }
            }
            break;

	        case ESS_PCM_24b_48k:
            {
              PCM::MXFReader     PCMReader;
              PCM::AudioDescriptor ADesc;
	            result = PCMReader.OpenRead(*iter);
                if (ASDCP_SUCCESS(result)) {
                PCMReader.FillAudioDescriptor(ADesc);
                WriterInfo WInfo;
	              PCMReader.FillWriterInfo(WInfo);

                AAssets.push_back(Asset());
                sprintf(buff, "%u %u", ADesc.SampleRate.Numerator, ADesc.SampleRate.Denominator);
                AAssets.back().EditRate = buff;

                AAssets.back().IntrinsicDuration = ADesc.ContainerDuration;
                AAssets.back().Duration = ADesc.ContainerDuration;
                AAssets.back().EntryPoint = 0;
                AAssets.back().IsCiphertext = WInfo.EncryptedEssence;
                AAssets.back().AssetID.Set(WInfo.AssetUUID);
              } else {
                fprintf(stderr, "%s: Can't open file.\n", *iter);
              }
            }
	          break;

          case ESS_PCM_24b_96k:
            fprintf(stderr, "%s: ESS_PCM_24b_96k not supported.\n", *iter);
	          return 4;
            break;

		  case ESS_JPEG_2000_S:
        {
          /*fprintf(stderr, "Detected stereoscopic input: %s\n", *iter);*/
          
          JP2K::MXFSReader JP2KReader;
          result = JP2KReader.OpenRead(*iter);
          
          if (ASDCP_SUCCESS(result))
          {
            JP2K::PictureDescriptor PDesc;
            JP2KReader.FillPictureDescriptor(PDesc);
            WriterInfo WInfo;
            JP2KReader.FillWriterInfo(WInfo);
            
            PAssets.push_back(Asset());
            PAssets.back().isStereo = true;
            
            sprintf(buff, "%u %u", PDesc.EditRate.Numerator, PDesc.EditRate.Denominator);
            PAssets.back().EditRate = buff;

            if (PDesc.SampleRate.Numerator==0 || PDesc.SampleRate.Denominator==0 || PDesc.SampleRate.Numerator==PDesc.EditRate.Numerator)
            {
              sprintf(buff, "%u %u", PDesc.EditRate.Numerator*2, PDesc.EditRate.Denominator);
            }
            else
            {
              sprintf(buff, "%u %u", PDesc.SampleRate.Numerator, PDesc.SampleRate.Denominator);
            }
            PAssets.back().FrameRate = buff;
            
            if (Options.ratio == 0)
            {
              sprintf(buff, "%u %u", PDesc.AspectRatio.Numerator, PDesc.AspectRatio.Denominator);
              Info.AspectRatio = buff;
            }
            
            PAssets.back().IntrinsicDuration = PDesc.ContainerDuration;
            PAssets.back().Duration = PDesc.ContainerDuration;
            PAssets.back().EntryPoint = 0;
            PAssets.back().IsCiphertext = WInfo.EncryptedEssence;
            PAssets.back().AssetID.Set(WInfo.AssetUUID);
          }
          else
          {
            fprintf(stderr, "%s: Can't open file.\n", *iter);
          }
        }
          break;

	        default:
	          fprintf(stderr, "%s: Unknown file type, not ASDCP essence.\n", *iter);
	          return 5;
	        }
      } else {
        fprintf(stderr, "%s: File not found.\n", *iter);
        return 6;
      }
      iter++;
  }

  if (PAssets.size() != AAssets.size() && AAssets.size() != 0) {
    fprintf(stderr, "Not the same number of assets of different types.\n");
    return 7;
  }

  std::list<Asset>::iterator piter;
  std::list<Asset>::iterator aiter;
  piter = PAssets.begin();
  if (AAssets.size() != 0) aiter = AAssets.begin();
  while (piter!=PAssets.end()) {
    Info.ReelList.push_back(Reel());
    Info.ReelList.back().MainPicture = *piter;
	if (AAssets.size() != 0) {
		Info.ReelList.back().MainSound = *aiter;
	} else {
		Info.ReelList.back().MainSound.IntrinsicDuration = 0;
	}
    Info.ReelList.back().MainSubtitle.IntrinsicDuration = 0;
    piter++;
    if (AAssets.size() != 0) aiter++;
  }

  std::string OutCPL;
  make_cpl(Info, OutCPL);
  fputs(OutCPL.c_str(), stdout);

  return 0;
}
