//---------------------------------------------------------------------------
// Copyright (C) 1999 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  Win32Lnk.C - COM functions using Win32 to be used as a test
//               for DS2480 based Universal Serial Adapter 'U' 
//               functions.
//
//  Version: 1.03
//
//  History: 1.00 -> 1.01  Added function msDelay. 
//
//           1.01 -> 1.02  Changed to generic OpenCOM/CloseCOM for easier 
//                         use with other platforms.
//
//           1.02 -> 1.03  Add function msGettick()

#include <windows.h>
#include <stdio.h>
#include "ds2480.h"

// exportable functions 
void FlushCOM(void);
int  WriteCOM(int, uchar *);
int  ReadCOM(int, uchar *);
void BreakCOM(void);
void SetBaudCOM(uchar);
void msDelay(int);
int  OpenCOM(char *);
void CloseCOM(void);
long msGettick(void);


// Win32 globals needed
static HANDLE ComID;
static OVERLAPPED osRead,osWrite;   

//---------------------------------------------------------------------------
//-------- COM required functions for MLANU 
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Flush the rx and tx buffers
//
void FlushCOM(void)
{
   // purge any information in the buffer
   PurgeComm(ComID, PURGE_TXABORT | PURGE_RXABORT |
                    PURGE_TXCLEAR | PURGE_RXCLEAR );
}


//--------------------------------------------------------------------------
// Write an array of bytes to the COM port, verify that it was
// sent out.  Assume that baud rate has been set.
//
// 'outlen' - number of bytes to write to COM port
// 'outbuf' - pointer ot an array of bytes to write
//
// Returns:  1 for success 
//           0 for failure
//
int WriteCOM(int outlen, uchar *outbuf)
{
   BOOL fWriteStat;
   DWORD dwBytesWritten=0;
   DWORD ler=0,to;

   // calculate a timeout
   to = 20 * outlen + 60;

   // reset the write event 
   ResetEvent(osWrite.hEvent);

   // write the byte
   fWriteStat = WriteFile(ComID, (LPSTR) &outbuf[0], 
                outlen, &dwBytesWritten, &osWrite );

   // check for an error
   if (!fWriteStat)
      ler = GetLastError();

   // if not done writting then wait 
   if (!fWriteStat && ler == ERROR_IO_PENDING)
   {                                         
      WaitForSingleObject(osWrite.hEvent,to);

      // verify all is written correctly
      fWriteStat = GetOverlappedResult(ComID, &osWrite, 
                   &dwBytesWritten, FALSE); 

   } 

   // check results of write
   if (!fWriteStat || (dwBytesWritten != (DWORD)outlen))
      return 0;
   else
      return 1;
}


//--------------------------------------------------------------------------
// Read an array of bytes to the COM port, verify that it was
// sent out.  Assume that baud rate has been set.
//
// 'inlen' - number of bytes to read from COM port
// 'inbuf' - pointer to a buffer to hold the incomming bytes
//
// Returns: number of characters read
//
int ReadCOM(int inlen, uchar *inbuf)
{
   DWORD dwLength=0; 
   BOOL fReadStat;
   DWORD ler=0,to;

   // calculate a timeout
   to = 20 * inlen + 60;

   // reset the read event 
   ResetEvent(osRead.hEvent);

   // read
   fReadStat = ReadFile(ComID, (LPSTR) &inbuf[0],
                      inlen, &dwLength, &osRead) ;
   
   // check for an error
   if (!fReadStat)
      ler = GetLastError();

   // if not done writing then wait 
   if (!fReadStat && ler == ERROR_IO_PENDING)
   {
      // wait until everything is read
      WaitForSingleObject(osRead.hEvent,to);

      // verify all is read correctly
      fReadStat = GetOverlappedResult(ComID, &osRead, 
                   &dwLength, FALSE); 
   }

   // check results
   if (fReadStat)
      return dwLength;
   else
      return 0;
}


//--------------------------------------------------------------------------
// Send a break on the com port for at least 2 ms
// 
void BreakCOM(void)
{
   // start the reset pulse
   SetCommBreak(ComID);

   // sleep 
   Sleep(2); 
   
   // clear the break
   ClearCommBreak(ComID);
}


//--------------------------------------------------------------------------
// Set the baud rate on the com port.  
//
// 'new_baud' - new baud rate defined as
//                PARMSET_9600     0x00
//                PARMSET_19200    0x02
//                PARMSET_57600    0x04
//                PARMSET_115200   0x06
// 
void SetBaudCOM(uchar new_baud)
{
   DCB dcb;

   // get the current com port state
   GetCommState(ComID, &dcb);

   // change just the baud rate
   switch (new_baud)
   {
      case PARMSET_115200:
         dcb.BaudRate = CBR_115200;
         break;
      case PARMSET_57600:
         dcb.BaudRate = CBR_57600;
         break;
      case PARMSET_19200:
         dcb.BaudRate = CBR_19200;
         break;
      case PARMSET_9600:
      default:
         dcb.BaudRate = CBR_9600;
         break;
   }
   
   // restore to set the new baud rate
   SetCommState(ComID, &dcb);
}


//--------------------------------------------------------------------------
//  Description:
//     Delay for at least 'len' ms
// 
void msDelay(int len)
{
   Sleep(len);
}


//---------------------------------------------------------------------------
// Attempt to open a com port.  Keep the handle in ComID.
// Set the starting baud rate to 9600.
//
// 'port_zstr' - zero terminate port name.  For this platform
//               use format COMX where X is the port number.
//
// Returns: TRUE - success, COM port opened
//
int OpenCOM(char *port_zstr)
{
   char tempstr[80];
   short fRetVal;
   COMMTIMEOUTS CommTimeOuts;
   DCB dcb;

   // open COMM device
   if ((ComID =
      CreateFile( port_zstr, GENERIC_READ | GENERIC_WRITE,
                  0, 
                  NULL,                 // no security attrs
                  OPEN_EXISTING,
                  FILE_FLAG_OVERLAPPED, // overlapped I/O
                  NULL )) == (HANDLE) -1 )
   {
      ComID = 0;
      return (FALSE) ;
   }
   else
   {
      // create events for detection of reading and write to com port 
      sprintf(tempstr,"COMM_READ_OVERLAPPED_EVENT_FOR_%s",port_zstr);
      osRead.hEvent = CreateEvent(NULL,TRUE,FALSE,tempstr);  
      sprintf(tempstr,"COMM_WRITE_OVERLAPPED_EVENT_FOR_%s",port_zstr);
      osWrite.hEvent = CreateEvent(NULL,TRUE,FALSE,tempstr); 

      // get any early notifications
      SetCommMask(ComID, EV_RXCHAR | EV_TXEMPTY | EV_ERR | EV_BREAK);

      // setup device buffers
      SetupComm(ComID, 2048, 2048);

      // purge any information in the buffer
      PurgeComm(ComID, PURGE_TXABORT | PURGE_RXABORT |
                           PURGE_TXCLEAR | PURGE_RXCLEAR );

      // set up for overlapped non-blocking I/O 
      CommTimeOuts.ReadIntervalTimeout = 0; 
      CommTimeOuts.ReadTotalTimeoutMultiplier = 20; 
      CommTimeOuts.ReadTotalTimeoutConstant = 40; 
      CommTimeOuts.WriteTotalTimeoutMultiplier = 20; 
      CommTimeOuts.WriteTotalTimeoutConstant = 40; 
      SetCommTimeouts(ComID, &CommTimeOuts);

      // setup the com port
      GetCommState(ComID, &dcb);

      dcb.BaudRate = CBR_9600;               // current baud rate 
      dcb.fBinary = TRUE;                    // binary mode, no EOF check 
      dcb.fParity = FALSE;                   // enable parity checking 
      dcb.fOutxCtsFlow = FALSE;              // CTS output flow control 
      dcb.fOutxDsrFlow = FALSE;              // DSR output flow control 
      dcb.fDtrControl = DTR_CONTROL_ENABLE;  // DTR flow control type 
      dcb.fDsrSensitivity = FALSE;           // DSR sensitivity 
      dcb.fTXContinueOnXoff = TRUE;          // XOFF continues Tx 
      dcb.fOutX = FALSE;                     // XON/XOFF out flow control 
      dcb.fInX = FALSE;                      // XON/XOFF in flow control 
      dcb.fErrorChar = FALSE;                // enable error replacement 
      dcb.fNull = FALSE;                     // enable null stripping 
      dcb.fRtsControl = RTS_CONTROL_ENABLE;  // RTS flow control 
      dcb.fAbortOnError = FALSE;             // abort reads/writes on error 
      dcb.XonLim = 0;                        // transmit XON threshold 
      dcb.XoffLim = 0;                       // transmit XOFF threshold 
      dcb.ByteSize = 8;                      // number of bits/byte, 4-8 
      dcb.Parity = NOPARITY;                 // 0-4=no,odd,even,mark,space 
      dcb.StopBits = ONESTOPBIT;             // 0,1,2 = 1, 1.5, 2 
      dcb.XonChar = 0;                       // Tx and Rx XON character 
      dcb.XoffChar = 1;                      // Tx and Rx XOFF character 
      dcb.ErrorChar = 0;                     // error replacement character 
      dcb.EofChar = 0;                       // end of input character 
      dcb.EvtChar = 0;                       // received event character 

      fRetVal = SetCommState(ComID, &dcb);
   }

   // check if successfull
   if (!fRetVal)
   {
      CloseHandle(ComID);
      ComID = 0;
   }

   return (fRetVal);
}


//---------------------------------------------------------------------------
// Closes the connection to the port.
//
void CloseCOM(void)
{
   // disable event notification and wait for thread
   // to halt
   SetCommMask(ComID, 0);

   // drop DTR
   EscapeCommFunction(ComID, CLRDTR);

   // purge any outstanding reads/writes and close device handle
   PurgeComm(ComID, PURGE_TXABORT | PURGE_RXABORT |
                    PURGE_TXCLEAR | PURGE_RXCLEAR );
   CloseHandle(ComID);
   ComID = 0;
} 


//--------------------------------------------------------------------------
// Get the current millisecond tick count.  Does not have to represent
// an actual time, it just needs to be an incrementing timer.
//
long msGettick(void)
{
   return GetTickCount();
}

