//---------------------------------------------------------------------------
// Copyright (C) 1999 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  MLanNet.C - Network functions for MicroLAN 1-Wire devices.
//
//  Version: 1.03
//
//  History: 1.00 -> 1.01  Change to MLanFamilySearchSetup, LastDiscrepancy
//                         was set to 64 instead of 8 to enable devices with 
//                         early contention to go in the '0' direction first.
//           1.02 -> 1.03  Initialized goodbits in  MLanVerify

#include "MLan.h"

// external functions required
extern int MLanTouchReset(void);
extern int MLanTouchBit(int);
extern int MLanWriteByte(int);
extern int MLanReadByte(void);
extern int MLanSpeed(int);
extern int MLanLevel(int);
extern int MLanBlock(int, uchar *, int);

// exportable functions
int  MLanFirst(int,int);
int  MLanNext(int,int);
void MLanSerialNum(uchar *, int);
void MLanFamilySearchSetup(int);
void MLanSkipFamily(void);
int  MLanAccess(void);
int  MLanVerify(int);
int  MLanOverdriveAccess(void);

// local functions       
int bitacc(int, int, int, uchar *);
uchar dowcrc(uchar);

// global variables for this module to hold search state information
static int LastDiscrepancy;
static int LastFamilyDiscrepancy;
static int LastDevice;
uchar SerialNum[8];
uchar DOWCRC;


//--------------------------------------------------------------------------
// The 'MLanFirst' finds the first device on the MicroLAN  This function 
// contains one parameter 'OnlyAlarmingDevices'.  When 
// 'OnlyAlarmingDevices' is TRUE (1) the find alarm command 0xEC is 
// sent instead of the normal search command 0xF0.
// Using the find alarm command 0xEC will limit the search to only
// 1-Wire devices that are in an 'alarm' state. 
//
// 'DoReset' - TRUE (1) perform reset before search, FALSE (0) do not
//             perform reset before search. 
// 'OnlyAlarmDevices' - TRUE (1) the find alarm command 0xEC is 
//             sent instead of the normal search command 0xF0
//
// Returns:   TRUE (1) : when a 1-Wire device was found and it's 
//                        Serial Number placed in the global SerialNum
//            FALSE (0): There are no devices on the MicroLAN.
// 
int MLanFirst(int DoReset, int OnlyAlarmingDevices)
{
   // reset the search state
   LastDiscrepancy = 0;
   LastDevice = FALSE;
   LastFamilyDiscrepancy = 0; 

   return MLanNext(DoReset,OnlyAlarmingDevices);
}


//--------------------------------------------------------------------------
// The 'MLanNext' function does a general search.  This function
// continues from the previos search state. The search state
// can be reset by using the 'MLanFirst' function.
// This function contains one parameter 'OnlyAlarmingDevices'.  
// When 'OnlyAlarmingDevices' is TRUE (1) the find alarm command 
// 0xEC is sent instead of the normal search command 0xF0.
// Using the find alarm command 0xEC will limit the search to only
// 1-Wire devices that are in an 'alarm' state. 
//
// 'DoReset' - TRUE (1) perform reset before search, FALSE (0) do not
//             perform reset before search. 
// 'OnlyAlarmDevices' - TRUE (1) the find alarm command 0xEC is 
//             sent instead of the normal search command 0xF0
//
// Returns:   TRUE (1) : when a 1-Wire device was found and it's 
//                       Serial Number placed in the global SerialNum
//            FALSE (0): when no new device was found.  Either the
//                       last search was the last device or there
//                       are no devices on the MicroLAN.
// 
int MLanNext(int DoReset, int OnlyAlarmingDevices)
{
   int bit_test, search_direction, bit_number;
   int last_zero, serial_byte_number, next_result;
   uchar serial_byte_mask;

   // initialize for search 
   bit_number = 1;
   last_zero = 0;
   serial_byte_number = 0;
   serial_byte_mask = 1;
   next_result = 0;     
   DOWCRC = 0;         
   
   // if the last call was not the last one 
   if (!LastDevice)
   {
      // check if reset first is requested
      if (DoReset)
      {
         // reset the 1-wire 
         // if there are no parts on 1-wire, return FALSE
         if (!MLanTouchReset())
         {
            // reset the search
            LastDiscrepancy = 0;        
            LastFamilyDiscrepancy = 0; 
            return FALSE;
         }
      }

      // If finding alarming devices issue a different command
      if (OnlyAlarmingDevices)
         MLanWriteByte(0xEC);  // issue the alarming search command 
      else
         MLanWriteByte(0xF0);  // issue the search command 

      // loop to do the search  
      do
      {
         // read a bit and its compliment 
         bit_test = MLanTouchBit(1) << 1;
         bit_test |= MLanTouchBit(1);

         // check for no devices on 1-wire
         if (bit_test == 3)  
            break;
         else
         {
            // all devices coupled have 0 or 1
            if (bit_test > 0)            
              search_direction = !(bit_test & 0x01);  // bit write value for search 
            else
            {
               // if this discrepancy if before the Last Discrepancy
               // on a previous next then pick the same as last time 
               if (bit_number < LastDiscrepancy) 
                  search_direction = ((SerialNum[serial_byte_number] & serial_byte_mask) > 0);
               else
                  // if equal to last pick 1, if not then pick 0              
                  search_direction = (bit_number == LastDiscrepancy);       

               // if 0 was picked then record its position in LastZero 
               if (search_direction == 0) 
                  last_zero = bit_number;  

               // check for Last discrepancy in family 
               if (last_zero < 9) 
                  LastFamilyDiscrepancy = last_zero;
            }

            // set or clear the bit in the SerialNum byte serial_byte_number 
            // with mask serial_byte_mask 
            if (search_direction == 1)
              SerialNum[serial_byte_number] |= serial_byte_mask;
            else
              SerialNum[serial_byte_number] &= ~serial_byte_mask;

            // serial number search direction write bit 
            MLanTouchBit(search_direction);

            // increment the byte counter bit_number 
            // and shift the mask serial_byte_mask 
            bit_number++;
            serial_byte_mask <<= 1;

            // if the mask is 0 then go to new SerialNum byte serial_byte_number
            // and reset mask 
            if (serial_byte_mask == 0)
            {
                dowcrc(SerialNum[serial_byte_number]);  // accumulate the CRC 
                serial_byte_number++; 
                serial_byte_mask = 1;
            }
         }
      } 
      while(serial_byte_number < 8);  // loop until through all SerialNum bytes 0-7 

      // if the search was successful then 
      if (!((bit_number < 65) || DOWCRC))  
      {
         // search successful so set LastDiscrepancy,LastDevice,next_result 
         LastDiscrepancy = last_zero;
         LastDevice = (LastDiscrepancy == 0);
         next_result = TRUE;
      }
   }
   
   // if no device found then reset counters so next 'next' will be
   // like a first 
   if (!next_result || !SerialNum[0])
   {
      LastDiscrepancy = 0;
      LastDevice = FALSE;
      LastFamilyDiscrepancy = 0; 
      next_result = FALSE;
   }

   return next_result;
}


//--------------------------------------------------------------------------
// The 'MLanSerialNum' function either reads or sets the SerialNum buffer 
// that is used in the search functions 'MLanFirst' and 'MLanNext'.  
// This function contains two parameters, 'SerialNumBuf' is a pointer
// to a buffer provided by the caller.  'SerialNumBuf' should point to 
// an array of 8 unsigned chars.  The second parameter is a flag called
// 'DoRead' that is TRUE (1) if the operation is to read and FALSE
// (0) if the operation is to set the internal SerialNum buffer from 
// the data in the provided buffer.
//
// 'SerialNumBuf' - buffer to that contains the serial number to set
//                  when DoRead = FALSE (0) and buffer to get the serial
//                  number when DoRead = TRUE (1).
// 'DoRead'       - flag to indicate reading (1) or setting (0) the current
//                  serial number.
//
void MLanSerialNum(uchar *SerialNumBuf, int DoRead)
{
   int i;

   // read the internal buffer and place in 'SerialNumBuf'
   if (DoRead)
   {
      for (i = 0; i < 8; i++)
         SerialNumBuf[i] = SerialNum[i];
   }
   // set the internal buffer from the data in 'SerialNumBuf'
   else
   {
      for (i = 0; i < 8; i++)
         SerialNum[i] = SerialNumBuf[i];
   }
}


//--------------------------------------------------------------------------
// Setup the search algorithm to find a certain family of devices
// the next time a search function is called 'MLanNext'.
//
// 'SearchFamily' - family code type to set the search algorithm to find
//                  next.
// 
void MLanFamilySearchSetup(int SearchFamily)
{
   int i;

   // set the search state to find SearchFamily type devices
   SerialNum[0] = (uchar)SearchFamily;                 
   for (i = 1; i < 8; i++)
      SerialNum[i] = 0; 
   LastDiscrepancy = 64;      
   LastDevice = FALSE;          
}


//--------------------------------------------------------------------------
// Set the current search state to skip the current family code.
//
void MLanSkipFamily(void)
{
   // set the Last discrepancy to last family discrepancy
   LastDiscrepancy = LastFamilyDiscrepancy;

   // check for end of list
   if (LastDiscrepancy == 0) 
      LastDevice = TRUE;
}


//--------------------------------------------------------------------------
// The 'MLanAccess' function resets the 1-Wire and sends a MATCH Serial 
// Number command followed by the current SerialNum code. After this 
// function is complete the 1-Wire device is ready to accept device-specific
// commands. 
//
// Returns:   TRUE (1) : reset indicates present and device is ready
//                       for commands.
//            FALSE (0): reset does not indicate presence or echos 'writes'
//                       are not correct.
//
int MLanAccess(void)
{
   uchar TranBuf[9];
   int i;

   // reset the 1-wire 
   if (MLanTouchReset())
   {
      // create a buffer to use with block function      
      // match Serial Number command 0x55 
      TranBuf[0] = 0x55; 
      // Serial Number
      for (i = 1; i < 9; i++)
         TranBuf[i] = SerialNum[i-1];
      
      // send/recieve the transfer buffer   
      if (MLanBlock(FALSE,TranBuf,9))
      {
         // verify that the echo of the writes was correct
         for (i = 1; i < 9; i++)
            if (TranBuf[i] != SerialNum[i-1])
               return FALSE;
         if (TranBuf[0] != 0x55)
            return FALSE;
         else
            return TRUE;
      }
   }

   // reset or match echo failed
   return FALSE;
}


//----------------------------------------------------------------------
// The function 'MLanVerify' verifies that the current device
// is in contact with the MicroLAN.    
// Using the find alarm command 0xEC will verify that the device
// is in contact with the MicroLAN and is in an 'alarm' state. 
// 
// 'OnlyAlarmingDevices' - TRUE (1) the find alarm command 0xEC 
//                         is sent instead of the normal search 
//                         command 0xF0. 
//
// Returns:   TRUE (1) : when the 1-Wire device was verified
//                       to be on the MicroLAN 
//                       with OnlyAlarmingDevices == FALSE 
//                       or verified to be on the MicroLAN
//                       AND in an alarm state when 
//                       OnlyAlarmingDevices == TRUE. 
//            FALSE (0): the 1-Wire device was not on the 
//                       MicroLAN or if OnlyAlarmingDevices
//                       == TRUE, the device may be on the 
//                       MicroLAN but in a non-alarm state.
// 
int MLanVerify(int OnlyAlarmingDevices)
{
   int i,TranCnt=0,goodbits=0,cnt=0,s,tst;
   uchar TranBuf[50];
   
   // construct the search  
   if (OnlyAlarmingDevices)
      TranBuf[TranCnt++] = 0xEC; // issue the alarming search command 
   else
      TranBuf[TranCnt++] = 0xF0; // issue the search command 
   // set all bits at first
   for (i = 1; i <= 24; i++)
      TranBuf[TranCnt++] = 0xFF;   
   // now set or clear apropriate bits for search 
   for (i = 0; i < 64; i++)
      bitacc(WRITE_FUNCTION,bitacc(READ_FUNCTION,0,i,&SerialNum[0]),(int)((i+1)*3-1),&TranBuf[1]);

   // send/recieve the transfer buffer   
   if (MLanBlock(TRUE,TranBuf,TranCnt))
   {
      // check results to see if it was a success 
      for (i = 0; i < 192; i += 3)
      {
         tst = (bitacc(READ_FUNCTION,0,i,&TranBuf[1]) << 1) |
                bitacc(READ_FUNCTION,0,(int)(i+1),&TranBuf[1]);

         s = bitacc(READ_FUNCTION,0,cnt++,&SerialNum[0]);

         if (tst == 0x03)  // no device on line 
         {
              goodbits = 0;    // number of good bits set to zero 
              break;     // quit 
         }

         if (((s == 0x01) && (tst == 0x02)) ||
             ((s == 0x00) && (tst == 0x01))    )  // correct bit 
            goodbits++;  // count as a good bit 
      }

      // check too see if there were enough good bits to be successful 
      if (goodbits >= 8) 
         return TRUE;
   }

   // block fail or device not present
   return FALSE;
}


//----------------------------------------------------------------------
// Perform a overdrive MATCH command to select the 1-Wire device with 
// the address in the ID data register.
//
// Returns:  TRUE: If the device is present on the MicroLAN and
//                 can do overdrive then the device is selected.
//           FALSE: Device is not present or not capable of overdrive.
//
int MLanOverdriveAccess(void)
{
   uchar TranBuf[8];
   int i, EchoBad = FALSE;

   // make sure normal level
   MLanLevel(MODE_NORMAL);

   // force to normal communication speed
   MLanSpeed(MODE_NORMAL);

   // call the MicroLAN reset function 
   if (MLanTouchReset())
   {
      // send the match command 0x69
      if (MLanWriteByte(0x69))
      {
         // switch to overdrive communication speed
         MLanSpeed(MODE_OVERDRIVE);

         // create a buffer to use with block function      
         // Serial Number
         for (i = 0; i < 8; i++)
            TranBuf[i] = SerialNum[i];
      
         // send/recieve the transfer buffer   
         if (MLanBlock(FALSE,TranBuf,8))
         {
            // verify that the echo of the writes was correct
            for (i = 0; i < 8; i++)
               if (TranBuf[i] != SerialNum[i])
                  EchoBad = TRUE;
            // if echo ok then success
            if (!EchoBad)
               return TRUE;               
         }
      }
   }
   
   // failure, force back to normal communication speed
   MLanSpeed(MODE_NORMAL);

   return FALSE;
}


//--------------------------------------------------------------------------
// Update the Dallas Semiconductor One Wire CRC (DOWCRC) from the global
// variable DOWCRC and the argument.  
//
// 'x' - data byte to calculate the 8 bit crc from
//
// Returns: the updated DOWCRC.
//
uchar dscrc_table[] = {
        0, 94,188,226, 97, 63,221,131,194,156,126, 32,163,253, 31, 65,
      157,195, 33,127,252,162, 64, 30, 95,  1,227,189, 62, 96,130,220,
       35,125,159,193, 66, 28,254,160,225,191, 93,  3,128,222, 60, 98,
      190,224,  2, 92,223,129, 99, 61,124, 34,192,158, 29, 67,161,255,
       70, 24,250,164, 39,121,155,197,132,218, 56,102,229,187, 89,  7,
      219,133,103, 57,186,228,  6, 88, 25, 71,165,251,120, 38,196,154,
      101, 59,217,135,  4, 90,184,230,167,249, 27, 69,198,152,122, 36,
      248,166, 68, 26,153,199, 37,123, 58,100,134,216, 91,  5,231,185,
      140,210, 48,110,237,179, 81, 15, 78, 16,242,172, 47,113,147,205,
       17, 79,173,243,112, 46,204,146,211,141,111, 49,178,236, 14, 80,
      175,241, 19, 77,206,144,114, 44,109, 51,209,143, 12, 82,176,238,
       50,108,142,208, 83, 13,239,177,240,174, 76, 18,145,207, 45,115,
      202,148,118, 40,171,245, 23, 73,  8, 86,180,234,105, 55,213,139,
       87,  9,235,181, 54,104,138,212,149,203, 41,119,244,170, 72, 22,
      233,183, 85, 11,136,214, 52,106, 43,117,151,201, 74, 20,246,168,
      116, 42,200,150, 21, 75,169,247,182,232, 10, 84,215,137,107, 53};

uchar dowcrc(uchar x)
{
   DOWCRC = dscrc_table[DOWCRC ^ x];
   return DOWCRC;
}

//--------------------------------------------------------------------------
//  Bit utility to read and write a bit in the buffer 'buf'.
//
int bitacc(int op, int state, int loc, uchar *buf)
{
     int nbyt,nbit;

     nbyt = (loc / 8);
     nbit = loc - (nbyt * 8);

     if (op == WRITE_FUNCTION)
     {
          if (state)
             buf[nbyt] |= (0x01 << nbit);
          else
             buf[nbyt] &= ~(0x01 << nbit);

          return 1;
     }
     else
          return ((buf[nbyt] >> nbit) & 0x01);
}
